#include "Game.h"
#include "Configuration.h"

#include "Menus/Impl/MainMenu.h"
#include "Menus/Impl/OptionsMenu.h"
#include "Menus/Impl/ExitMenu.h"
#include "Menus/Impl/PlayerConfigMenu.h"
#include "Menus/Impl/SummaryMenu.h"

#include "Util/AllegroLog.h"
#include "Util/blitfill.h"

Game::Game() {
    _state = STATE_NEW;

    _dyingVehiclesLastTick = 0;

    for (int i = 0; i < MAX_PLAYERS_COUNT; i++) {
        _player[i] = NULL;
    }

    _ticksPerSecond = 80;
}

Game::~Game() {
    delete _log;
}

void Game::Init() {
    srand(time(NULL));

    _musicPath[MUSIC_MAIN] = "music/Rafael_Peligro_-_Fragile_People_(loop).mp3";
}

void Game::ShutDown() {

}

short Game::CountLivingVehicles() {
    short count = 0;
    for (short i = 0; i < MAX_PLAYERS_COUNT; i++) {
        assert(_player[i] != NULL);
        if (_player[i]->IsEnabled() && _player[i]->GetVehicle()->GetState() == Vehicle::STATE_LIVING) {
            count++;
        }
    }
    return count;
}

void Game::Resume() {
    _state = STATE_RUNNING;
}

void Game::Exit() {
    _state = STATE_QUITTING;
}

void Game::Start() {
    if(_state == STATE_NEW) {
        PlayMusic(MUSIC_MAIN);
    }

    _state = STATE_RUNNING;

    _dyingVehiclesLastTick = 0;

    _arena->DeleteFieldWeapons();
    _arena->DeleteActiveWeapons();
    _arena->Clear();

    for (int i = 0; i < _configuration->GetMaxFieldWeaponsCount(); i++) {
        _arena->SpawnWeapon();
    }

    for (int i = 0; i < MAX_PLAYERS_COUNT; i++) {
        assert(_player[i] != NULL);
        if (_player[i]->IsEnabled()) {
            _player[i]->Reset();
        }
    }
}

void Game::Summarize() {
    _state = STATE_SUMMARIZING;
}

void Game::CalculateScores() {
    int enabledCount = 0;
    int livingCount = 0;
    int dyingCount = 0;
    int deadCount = 0;
    for (short i = 0; i < MAX_PLAYERS_COUNT; i++) {
        assert(_player[i] != NULL);
        if(_player[i]->IsEnabled()) {
            if (_player[i]->GetVehicle()->GetState() == Vehicle::STATE_LIVING) {
                livingCount++;
            } else if (_player[i]->GetVehicle()->GetState() == Vehicle::STATE_DYING) {
                dyingCount++;
            } else if (_player[i]->GetVehicle()->GetState() == Vehicle::STATE_DEAD) {
                deadCount++;
            }
            enabledCount++;
        }
    }

    if (dyingCount > 0) {
        if (dyingCount == _dyingVehiclesLastTick) {
            short score = enabledCount - livingCount - 1;
            for (short i = 0; i < MAX_PLAYERS_COUNT; i++) {
                assert(_player[i] != NULL);
                if (_player[i]->IsEnabled() && _player[i]->GetVehicle()->GetState() == Vehicle::STATE_DYING) {
                    _player[i]->GetVehicle()->Die();
                    _player[i]->ApplyRoundScore(score);
                    _dyingVehiclesLastTick = 0;
                }
            }

        } else if (dyingCount > _dyingVehiclesLastTick) {
            _dyingVehiclesLastTick = dyingCount;
        } else if (dyingCount < _dyingVehiclesLastTick) {
            GetLog()->LogError("One or more dying players were ignored");
        }
    }

    int notDeadCount = livingCount + dyingCount;
    if (notDeadCount == 0) {
        Summarize();
    } else if (notDeadCount == 1) {
        short score = enabledCount - (livingCount + 1) + 1;
        for (short i = 0; i < MAX_PLAYERS_COUNT; i++) {
            assert(_player[i] != NULL);
            if (_player[i]->IsEnabled() && _player[i]->GetVehicle()->GetState() != Vehicle::STATE_DEAD) {
                _player[i]->ApplyRoundScore(score);
            }
        }
        Summarize();
    }
}

double Game::GetNormalSpeed() {
    return 100.0f/_ticksPerSecond;
}

//Allegro

volatile int ticks_to_do = 0;

void increment_ticks_to_do() {
    ticks_to_do++;
}
END_OF_FUNCTION(increment_ticks_to_do)

volatile int close_button_pressed = false;

void close_button_handler(void) {
    close_button_pressed = true;
}
END_OF_FUNCTION(close_button_handler)

AllegroGame::AllegroGame() : Game() {
    _soundSystem = NULL;
    _channel = NULL;
    _sound = NULL;
    for (int i = 0; i < MENU_TYPES_COUNT; i++) {
        _menu[i] = NULL;
    }
    for(int i = 0; i < SOUNDS_COUNT; i++) {
        _soundSample[i] = NULL;
    }
    for(int i = 0; i < SOUNDS_COUNT; i++) {
        _weaponSoundSample[i] = NULL;
    }
    _log = new AllegroLog();
}

AllegroGame::~AllegroGame() {
    
}

void AllegroGame::Init() {
    Game::Init();
    InitSystem();
    _screenBuffer = create_bitmap(SCREEN_W, SCREEN_H);
    LoadResources();

    _menu[MENU_MAIN] = new MainMenu(this);
    _menu[MENU_OPTIONS] = new OptionsMenu(this);
    _menu[MENU_EXIT] = new ExitMenu(this);
    _menu[MENU_SUMMARY] = new SummaryMenu(this);
    for(int i=0; i<MAX_PLAYERS_COUNT; i++) {
        _menu[MENU_PLAYER_CONFIG_1 + i] = new PlayerConfigMenu(this, i + 1);
    }

    const int borderSize = 28;
    Vector2D arenaPos(borderSize, borderSize);
    Vector2D infoPanelSize(GetConfiguration()->GetResX(), 48);
    Vector2D arenaSize(GetConfiguration()->GetResX() - borderSize*2, GetConfiguration()->GetResY() - borderSize*2 - infoPanelSize.y);
    Vector2D infoPanelPos(0.0, arenaPos.x + arenaSize.y + borderSize);

    _infoPanel = new AllegroInfoPanel(this, infoPanelPos, infoPanelSize);
    _arena = new AllegroArena(this, arenaPos, arenaSize);

    for (int i = 0; i < MAX_PLAYERS_COUNT; i++) {
        _player[i] = new AllegroPlayer(this, i + 1);
    }

    _menu[MENU_MAIN]->Show();
}

void AllegroGame::InitSystem() {
    allegro_init();

    _configuration = new AllegroConfiguration();
    _configuration->Load(CONFIG_FILENAME);

    int colorDepth = desktop_color_depth();
    if(colorDepth < 15) {
        colorDepth = 16;
    }
    set_color_depth(colorDepth);
    int cardMode = _configuration->IsFullscreen() ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED;
    int gfxMode = set_gfx_mode(cardMode, _configuration->GetResX(), _configuration->GetResY(), 0, 0);
    if (gfxMode != 0) {
        GetLog()->LogError(allegro_error);
        _configuration->ChangeToLowestResolution();
        _configuration->SetFullscreen(false);
        _configuration->Save(CONFIG_FILENAME);
        exit(-1);
    }

    install_timer();
    install_keyboard();
    install_mouse();
    int soundInstallationResult = install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL);
    if (soundInstallationResult != 0) {
        GetLog()->LogError(allegro_error);
    }

    if (keyboard_needs_poll()) {
        GetLog()->LogError("Keyboard needs poll");
    }

    LOCK_VARIABLE(ticks_to_do);
    LOCK_FUNCTION(increment_ticks_to_do);
    install_int_ex(increment_ticks_to_do, BPS_TO_TIMER(_ticksPerSecond));

    LOCK_FUNCTION(close_button_handler);
    set_close_button_callback(close_button_handler);

    FMOD_System_Create(&_soundSystem);
    FMOD_System_Init(_soundSystem, 1, FMOD_INIT_NORMAL, 0);
}

void AllegroGame::LoadResources() {
    _graveBitmap = load_bitmap(BITMAPS_PATH "skull.bmp", NULL);
    _slotBitmap = load_bitmap(BITMAPS_PATH "slot.bmp", NULL);
    _mouseCursorBitmap = load_bitmap(BITMAPS_PATH "cursor.bmp", NULL);

    _weaponIcon[WEAPON_WALL] = load_bitmap(WEAPON_ICONS_PATH "wall.bmp", NULL);
    _weaponIcon[WEAPON_BOMB] = load_bitmap(WEAPON_ICONS_PATH "bomb.bmp", NULL);
    _weaponIcon[WEAPON_STOP] = load_bitmap(WEAPON_ICONS_PATH "stop.bmp", NULL);
    _weaponIcon[WEAPON_CLUSTER] = load_bitmap(WEAPON_ICONS_PATH "cluster.bmp", NULL);
    _weaponIcon[WEAPON_HOMING] = load_bitmap(WEAPON_ICONS_PATH "homing.bmp", NULL);

    _arenaBgBitmap = load_bitmap(BITMAPS_PATH "arena_bg.bmp", NULL);
    _arenaBorderHBitmap = load_bitmap(BITMAPS_PATH "border_h.bmp", NULL);
    _arenaBorderVBitmap = load_bitmap(BITMAPS_PATH "border_v.bmp", NULL);
    _arenaBorderCBitmap = load_bitmap(BITMAPS_PATH "border_c.bmp", NULL);
    _infoPanelBgBitmap = load_bitmap(BITMAPS_PATH "info_panel_bg.bmp", NULL);
    _menuBgBitmap = load_bitmap(BITMAPS_PATH "menu_bg.bmp", NULL);

    _soundSample[SOUND_MOUSE_CLICK] = load_sample(SOUNDS_PATH "mouse_click.wav");
    _soundSample[SOUND_MOUSE_OVER] = load_sample(SOUNDS_PATH "mouse_over.wav");
    _soundSample[SOUND_COLLISION] = load_sample(SOUNDS_PATH "collision.wav");
    _soundSample[SOUND_ITEM_PICKUP] = load_sample(SOUNDS_PATH "item_pickup.wav");
    _soundSample[SOUND_ITEM_SELECT] = load_sample(SOUNDS_PATH "item_select.wav");
    _soundSample[SOUND_BORDER] = load_sample(SOUNDS_PATH "border.wav");
    _soundSample[SOUND_MISSILE] = load_sample(SOUNDS_PATH "missile.wav");

    _weaponSoundSample[WEAPON_WALL] = load_sample(SOUNDS_PATH "wall.wav");
    _weaponSoundSample[WEAPON_BOMB] = load_sample(SOUNDS_PATH "bomb.wav");
    _weaponSoundSample[WEAPON_STOP] = load_sample(SOUNDS_PATH "stop.wav");
    _weaponSoundSample[WEAPON_CLUSTER] = load_sample(SOUNDS_PATH "cluster.wav");
    _weaponSoundSample[WEAPON_HOMING] = load_sample(SOUNDS_PATH "homing.wav");

    _font = load_font(BITMAPS_PATH "font.pcx", NULL, NULL);
}

void AllegroGame::Run() {
    Init();

    while (_state != STATE_QUITTING) {
        while (ticks_to_do > 0) {
            Tick();
            ticks_to_do--;
        }
        Draw();
    }

    ShutDown();
}

void AllegroGame::ShutDown() {
    Game::ShutDown();

    if (_channel != NULL) {
        FMOD_Channel_Stop(_channel);
    }
    if (_sound != NULL) {
        FMOD_Sound_Release(_sound);
    }
    if (_soundSystem != NULL) {
        FMOD_System_Close(_soundSystem);
        FMOD_System_Release(_soundSystem);
    }

    for (int i = 0; i < MAX_PLAYERS_COUNT; i++) {
        assert(_player[i] != NULL);
        delete _player[i];
        _player[i] = NULL;
    }
    
    delete _arena;
    delete _infoPanel;
    delete _configuration;

    clear_keybuf();
    allegro_exit();
}

void AllegroGame::Draw() {
    clear(_screenBuffer);

    if (_state == STATE_RUNNING || _state == STATE_SUMMARIZING) {
        _arena->Draw();
        _infoPanel->Draw();
        for (int i = 0; i < MAX_PLAYERS_COUNT; i++) {
            assert(_player[i] != NULL);
            _player[i]->Draw();
        }
    }

    if (_state == STATE_PAUSED || _state == STATE_NEW) {
        blitfill(_menuBgBitmap, _screenBuffer, 0, 0, SCREEN_W, SCREEN_H);

        //Menus
        for (int i = 0; i < MENU_TYPES_COUNT; i++) {
            assert(_menu[i] != NULL);
            _menu[i]->Draw();
        }

        //Mouse Cursor
        draw_sprite(_screenBuffer, _mouseCursorBitmap, mouse_x, mouse_y);
    }

    if (_state == STATE_SUMMARIZING && !key[KEY_TAB]) {
        _menu[MENU_SUMMARY]->Draw();

        //Mouse Cursor
        draw_sprite(_screenBuffer, _mouseCursorBitmap, mouse_x, mouse_y);
    }

    blit(_screenBuffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
}

void AllegroGame::Tick() {
    if (_state == STATE_SUMMARIZING && !key[KEY_TAB]) {
        _menu[MENU_SUMMARY]->Tick();

        if (key[KEY_ENTER]) {
            this->Start();
        }
    } else if (_state == STATE_RUNNING) {
        _arena->Tick();

        for (int i = 0; i < MAX_PLAYERS_COUNT; i++) {
            assert(_player[i] != NULL);
            _player[i]->Tick();
        }

        this->CalculateScores();
    }

    if(_state == STATE_RUNNING || _state == STATE_SUMMARIZING) {
        if (key[KEY_ESC] || key[KEY_F10] || key[KEY_F11] || key[KEY_F12]) {
            if(_state == STATE_SUMMARIZING) {
                this->Start();
            }
            _state = STATE_PAUSED;
            _menu[MENU_MAIN]->Show();
        }
    }

    if (_state == STATE_PAUSED || _state == STATE_NEW) {
        for (int i = 0; i < MENU_TYPES_COUNT; i++) {
            if (_menu[i] != NULL) {
                _menu[i]->Tick();
            }
        }
    }

    if(close_button_pressed) {
        if(_state == STATE_NEW) {
            Exit();
        } else {
            _state = STATE_PAUSED;
            HideMenus();
            _menu[MENU_EXIT]->Show();
        }
        close_button_pressed = false;
    }
}

void AllegroGame::StopMusic() {
    if (_channel != NULL) {
        FMOD_Channel_Stop(_channel);
    }

    if (_sound != NULL) {
        FMOD_Sound_Release(_sound);
        _sound = NULL;
    }
}

void AllegroGame::PlayMusic(MUSIC music) {
        StopMusic();
        FMOD_System_CreateStream(_soundSystem, GetMusicPath(music), FMOD_HARDWARE | FMOD_LOOP_NORMAL | FMOD_2D, 0, &_sound);
        FMOD_System_PlaySound(_soundSystem, FMOD_CHANNEL_FREE, _sound, false, &_channel);
        FMOD_Channel_SetVolume(_channel, 0.3f);
}

void AllegroGame::SetMusicPaused(bool paused) {
    if (_channel != NULL) {
        FMOD_Channel_SetPaused(_channel, paused);
    }
}

BITMAP* AllegroGame::GetSprite(SPRITE_BITMAP sprite) {
    if(sprite == SPRITE_GRAVE) {
        return _graveBitmap;
    } else {
        GetLog()->LogError("could not get a sprite bitmap");
        return NULL;
    }
}

void AllegroGame::HideMenus() {
    for(int i=0; i<MENU_TYPES_COUNT; i++) {
        _menu[i]->Hide();
    }
}

void AllegroGame::Start() {
    if (_state == STATE_SUMMARIZING) {
        _menu[MENU_SUMMARY]->Hide();
    }
    Game::Start();
}

void AllegroGame::Summarize() {
    Game::Summarize();
    _menu[MENU_SUMMARY]->Show();
}

void AllegroGame::PlaySound(SOUND soundEffect, float pan) {
    assert(pan >= -1.0f && pan <= 1.0f);
    assert(_soundSample[soundEffect] != NULL);
    play_sample(_soundSample[soundEffect], 255, (int)(pan*127)+127, 1000, 0);
}

void AllegroGame::PlayWeaponSound(WEAPON_TYPE weapon, float pan) {
    assert(pan >= -1.0f && pan <= 1.0f);
    assert(_weaponSoundSample[weapon] != NULL);
    play_sample(_weaponSoundSample[weapon], 255, (int)(pan*127)+127, 1000, 0);
}

double AllegroGame::GetWeaponIconW(WEAPON_TYPE weaponType) {
    return _weaponIcon[weaponType]->w;
}

double AllegroGame::GetWeaponIconH(WEAPON_TYPE weaponType) {
    return _weaponIcon[weaponType]->h;
}
