#include "Arena.h"

#include "Weapons/Impl/WallWeapon.h"
#include "Weapons/Impl/BombWeapon.h"
#include "Weapons/Impl/StopWeapon.h"
#include "Weapons/Impl/ClusterWeapon.h"
#include "Weapons/Impl/HomingWeapon.h"

#include "Util/blitfill.h"

Arena::Arena(Game* game, const Vector2D& pos, const Vector2D& size) {
    _game = game;
    _pos = pos;
    _size = size;
}

Arena::~Arena() {
    
}

void Arena::Tick() {
    //ActiveWeapons
    for (int i = 0; i < _activeWeapons.size(); i++) {
        _activeWeapons.at(i)->Tick();
    }
}

void Arena::SpawnWeapon() {
    Weapon* weapon = NULL;
    short r = rand() % WEAPON_TYPES_COUNT;
    WEAPON_TYPE t = (WEAPON_TYPE) r;

    if (t == WEAPON_WALL) weapon = new WallWeapon(_game, NULL, t);
    else if (t == WEAPON_BOMB) weapon = new BombWeapon(_game, NULL, t);
    else if (t == WEAPON_STOP) weapon = new StopWeapon(_game, NULL, t);
    else if (t == WEAPON_CLUSTER) weapon = new ClusterWeapon(_game, NULL, t);
    else if (t == WEAPON_HOMING) weapon = new HomingWeapon(_game, NULL, t);
    else {
        _game->GetLog()->LogFatal("Could not spawn weapon!");
    }

    int x = rand() % ((int)(_size.x - weapon->GetW()));
    int y = rand() % ((int)(_size.y - weapon->GetH()));
    weapon->SetPos(x, y);
    _collectableWeapons.push_back(weapon);
}

void Arena::AddActiveWeapon(Weapon* weapon) {
    _activeWeapons.push_back(weapon);
}

int Arena::GetActiveWeaponsCount() {
    return _activeWeapons.size();
}

Weapon* Arena::GetActiveWeapon(int index) {
    return _activeWeapons.at(index);
}

void Arena::RemoveActiveWeapon(Weapon* weapon) {
    _activeWeapons.erase(find(_activeWeapons.begin(), _activeWeapons.end(), weapon));
}

void Arena::DeleteActiveWeapons() {
    for (int i = 0; i < _activeWeapons.size(); i++) {
        delete _activeWeapons.at(i);
    }
    _activeWeapons.clear();
}

short Arena::GetFieldWeaponCount() {
    return _collectableWeapons.size();
}

Weapon* Arena::GetFieldWeapon(short weaponIndex) {
    return _collectableWeapons.at(weaponIndex);
}

void Arena::DeleteFieldWeapons() {
    for (int i = 0; i < _collectableWeapons.size(); i++) {
        delete _collectableWeapons.at(i);
    }
    _collectableWeapons.clear();
}

void Arena::AddFieldWeapon(Weapon* weapon) {
    _collectableWeapons.push_back(weapon);
}

void Arena::RemoveFieldWeapon(Weapon* weapon) {
    _collectableWeapons.erase(find(_collectableWeapons.begin(), _collectableWeapons.end(), weapon));
}

//Allegro

AllegroArena::AllegroArena(Game* game, const Vector2D& pos, const Vector2D& size) : Arena(game, pos, size) {
    _wallsBitmap = create_bitmap((int)_size.x, (int)_size.y);
    Clear();
}

AllegroArena::~AllegroArena() {
    destroy_bitmap(_wallsBitmap);
}

void AllegroArena::Draw() {
    BITMAP* screenBuffer = GameImplementation()->GetScreenBuffer();

    //Border
    BITMAP* borderH = GameImplementation()->GetArenaBorderHBitmap();
    BITMAP* borderV = GameImplementation()->GetArenaBorderVBitmap();
    BITMAP* borderC = GameImplementation()->GetArenaBorderCBitmap();
    blitfill(borderH, screenBuffer, (int)_pos.x, (int)_pos.y - borderH->h, (int)_size.x, borderH->h);
    blitfill(borderH, screenBuffer, (int)_pos.x, (int)(_pos.y + _size.y), (int)_size.x, borderH->h);
    blitfill(borderV, screenBuffer, (int)_pos.x - borderV->w, (int)_pos.y, borderV->w, (int)_size.y);
    blitfill(borderV, screenBuffer, (int)(_pos.x + _size.x), (int)_pos.y, borderV->w, (int)_size.y);
    blit(borderC, screenBuffer, 0, 0, (int)_pos.x - borderC->w, (int)_pos.y - borderC->h, borderC->w, borderC->h);
    blit(borderC, screenBuffer, 0, 0, (int)(_pos.x + _size.x), (int)_pos.y - borderC->h, borderC->w, borderC->h);
    blit(borderC, screenBuffer, 0, 0, (int)_pos.x - borderC->w, (int)(_pos.y + _size.y), borderC->w, borderC->h);
    blit(borderC, screenBuffer, 0, 0, (int)(_pos.x + _size.x), (int)(_pos.y + _size.y), borderC->w, borderC->h);

    //Background
    blitfill(GameImplementation()->GetArenaBackgroundBitmap(), screenBuffer, (int)_pos.x, (int)_pos.y, (int)_size.x, (int)_size.y);

    //Walls
    draw_sprite(screenBuffer, _wallsBitmap, (int)_pos.x, (int)_pos.y);

    //Weapons for pick up
    for (int i = 0; i < _collectableWeapons.size(); i++) {
        Weapon* weapon = _collectableWeapons.at(i);
        blit(GameImplementation()->GetWeaponIcon(weapon->GetType()), screenBuffer, 0, 0, (int)(_pos.x + weapon->GetX()), (int)(_pos.y + weapon->GetY()), (int)weapon->GetW(), (int)weapon->GetH());
    }

    //Active Weapons
    for (int i = 0; i < _activeWeapons.size(); i++) {
        if (_activeWeapons.at(i) != NULL)
            _activeWeapons.at(i)->Draw();
        else {
            GameImplementation()->GetLog()->LogError("Active weapon is NULL during drawing");
        }
    }
}

void AllegroArena::DrawLine(const Vector2D &pos1, const Vector2D &pos2, int color) {
    BITMAP* screenBuffer = GameImplementation()->GetScreenBuffer();

    if (GameImplementation()->GetConfiguration()->IsThickLines()) {
        const double secondLineDist = 0.4;
        const double thirdLineDist = 0.6;
        const int darkColor = DarkenColor(color);

        const double angle = (pos2 - pos1).GetAngle();
        Vector2D sideShift(angle + PI / 2, secondLineDist, true);
        Vector2D backShift(angle + PI, 0.5, true);

        Vector2D lineStart;
        Vector2D lineEnd;

        lineStart = _pos + pos1 + sideShift;
        lineEnd = _pos + pos2 + sideShift + backShift;
        line(screenBuffer, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);

        lineStart = _pos + pos1 - sideShift;
        lineEnd = _pos + pos2 - sideShift + backShift;
        line(screenBuffer, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);

        sideShift *= thirdLineDist / secondLineDist;

        lineStart = _pos + pos1 + sideShift;
        lineEnd = _pos + pos2 + sideShift + backShift;
        line(screenBuffer, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);

        lineStart = _pos + pos1 - sideShift;
        lineEnd = _pos + pos2 - sideShift + backShift;
        line(screenBuffer, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);
    }
    line(screenBuffer, (int)_pos.x + (int)pos1.x, (int)_pos.y + (int)pos1.y, (int)_pos.x + (int)pos2.x, (int)_pos.y + (int)pos2.y, color);
}

void AllegroArena::DrawCircle(const Vector2D &pos, int radius, int color, bool filled) {
    if(filled) {
        circlefill(GameImplementation()->GetScreenBuffer(), (int)pos.x + (int)_pos.x, (int)pos.y + (int)_pos.y, radius, color);
    } else {
        circle(GameImplementation()->GetScreenBuffer(), (int)pos.x + (int)_pos.x, (int)pos.y + (int)_pos.y, radius, color);
    }
}

double AllegroArena::CalcShortestCollisionDist(const Vector2D &fromPos, const Vector2D &scanLine) {
    double distToScan = scanLine.GetLength();
    if(distToScan < 1.5) {
        distToScan = 1.5;
    }
    Vector2D currentScanPos = fromPos;
    Vector2D step;

    double remainingDist = distToScan;
    double nextDistStep = 0.0;

    while(remainingDist > 0.0) {
        nextDistStep = remainingDist >= 1.0 ? 1.0 : remainingDist;
        remainingDist -= nextDistStep;
        step.SetPolar(scanLine.GetAngle(), nextDistStep);
        currentScanPos += step;

        if (((int) currentScanPos.x) < 0 || ((int) currentScanPos.x) >= GetW() || ((int) currentScanPos.y) < 0 || ((int) currentScanPos.y) >= GetH()) {
            return distToScan - remainingDist;
        } else {
            if (currentScanPos == fromPos) {
                //do nothing (because it is the original point)
            } else if (getpixel(_wallsBitmap, (int) currentScanPos.x, (int) currentScanPos.y) != TRANSPARENT_COLOR) {
                return distToScan - remainingDist;
            }
        }
    }

    //no collision point found within scan distance
    return std::numeric_limits<double>::infinity();
}

void AllegroArena::PlaceLine(const Vector2D &pos1, const Vector2D &pos2, int color) {
    if (GameImplementation()->GetConfiguration()->IsThickLines()) {
        const double secondLineDist = 0.4;
        const double thirdLineDist = 0.6;
        const int darkColor = DarkenColor(color);

        const double angle = (pos2 - pos1).GetAngle();
        Vector2D sideShift(angle + PI / 2, secondLineDist, true);
        Vector2D backShift(angle + PI, 1.0, true);

        Vector2D lineStart;
        Vector2D lineEnd;

        lineStart = pos1 + sideShift;
        lineEnd = pos2 + sideShift + backShift;
        line(_wallsBitmap, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);

        lineStart = pos1 - sideShift;
        lineEnd = pos2 - sideShift + backShift;
        line(_wallsBitmap, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);

        sideShift *= thirdLineDist / secondLineDist;

        lineStart = pos1 + sideShift;
        lineEnd = pos2 + sideShift + backShift;
        line(_wallsBitmap, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);

        lineStart = pos1 - sideShift;
        lineEnd = pos2 - sideShift + backShift;
        line(_wallsBitmap, (int)lineStart.x, (int)lineStart.y, (int)lineEnd.x, (int)lineEnd.y, darkColor);
    }
    line(_wallsBitmap, (int)pos1.x, (int)pos1.y, (int)pos2.x, (int)pos2.y, color);
}

void AllegroArena::PlaceCircle(const Vector2D &pos, int radius, int color, bool filled) {
    if(filled) {
        circlefill(_wallsBitmap, (int)pos.x, (int)pos.y, radius, color);
    } else {
        circle(_wallsBitmap, (int)pos.x, (int)pos.y, radius, color);
    }
}

void AllegroArena::PlaceSprite(int spriteID, const Vector2D &pos, bool centered) {
    BITMAP* spriteBitmap = GameImplementation()->GetSprite((Game::SPRITE_BITMAP)spriteID);
    Vector2D newPos = pos;
    if(centered) {
        newPos.x -= spriteBitmap->w/2;
        newPos.y -= spriteBitmap->h/2;
    }
    draw_sprite(_wallsBitmap, spriteBitmap, (int)newPos.x, (int)newPos.y);
}

void AllegroArena::ClearCircle(const Vector2D &pos, int radius, bool filled) {
    if(filled) {
        circlefill(_wallsBitmap, (int)pos.x, (int)pos.y, radius, TRANSPARENT_COLOR);
    } else {
        circle(_wallsBitmap, (int)pos.x, (int)pos.y, radius, TRANSPARENT_COLOR);
    }
}

void AllegroArena::ClearRect(const Vector2D &pos, const Vector2D &size) {
    Vector2D pos2 = pos + size;
    rectfill(_wallsBitmap, (int)pos.x, (int)pos.y, (int)pos2.x, (int)pos2.y, TRANSPARENT_COLOR);
}

void AllegroArena::Clear() {
    clear_to_color(_wallsBitmap, TRANSPARENT_COLOR);
}
