/**
 * 2010-05-29_16.37
 */

#ifndef L_VECTOR_2D
#define L_VECTOR_2D

#include <math.h>

static const double PI = 3.141592653589793238462643383279502884197;

class Vector2D {
public:

    double x, y;

    /**
     * Constructs a 0-vector.
     */
    Vector2D() {
        x = 0.0;
        y = 0.0;
    }

    /**
     * Constructs a vector with the specified components x and y.
     */
    Vector2D(double x, double y) {
        this->x = x;
        this->y = y;
    }

    /**
     * Constructs a unit vector with the specified angle (in radians).
     * The resulting length will be 1.
     */
    Vector2D(double angle) {
        x = cos(angle);
        y = sin(angle);
    }

    /**
     * Constructs a vector by either using the first two parameters as x and y components
     * or by using them as angle (in radians) and length values.
     * If polar parameter is false, x and y is used.
     * If polar parameter is true, angle and length is used.
     */
    Vector2D(double xOrAngle, double yOrLength, bool polar) {
        if (polar) {
            x = cos(xOrAngle) * yOrLength;
            y = sin(xOrAngle) * yOrLength;
        } else {
            x = xOrAngle;
            y = yOrLength;
        }
    }

    /**
     * Destroys the vector.
     */
    virtual ~Vector2D() {
    }

    /**
     * Sets x and y components.
     */
    void Set(double x, double y) {
        this->x = x;
        this->y = y;
    }

    /**
     * Builds this vector with the specified angle (in radians) and length.
     */
    void SetPolar(double angle, double length) {
        x = cos(angle) * length;
        y = sin(angle) * length;
    }

    /**
     * Returns the length of this vector.
     */
    double GetLength() const {
        return sqrt(x * x + y * y);
    }

    /**
     * Sets length of this vector and keeps the angle unchanged.
     * TODO: vllt ist es performanter den einheitsvektor zu berechnen und dann mit length zu multiplizieren.
     */
    void SetLength(double length) {
        SetPolar(GetAngle(), length);
    }

    /**
     * Gets the angle (in radians) of this vector.
     */
    double GetAngle() const {
        return atan2(y, x);
    }

    /**
     * Sets angle (in radians) of this vector and keeps the length unchanged.
     */
    void SetAngle(double angle) {
        SetPolar(angle, GetLength());
    }

    /**
     * Bounces from a surface defined by surfaceNormal.
     */
    Vector2D Bounce(const Vector2D &surfaceNormal) const {
        const double a = surfaceNormal.GetAngle();
        Vector2D bounced = *this ^ -a;
        bounced.x *= -1.0;
        bounced ^= a;
        return bounced;
    }

    /**
     * Bounces from a surface defined by surfaceNormal if facing that surface.
     * If this vector doesn't face the surface, an unchanged copy of it is returned.
     * (Useful to avoid contineous bouncing when already in the "wall").
     */
    Vector2D BounceFacing(const Vector2D &surfaceNormal) const {
        const double a = surfaceNormal.GetAngle();
        Vector2D bounced = *this ^ -a;
        if(bounced.x < 0) {
            bounced.x *= -1.0;
        }
        bounced ^= a;
        return bounced;
    }

    /**
     * Returns this vector projected on v.
     */
    Vector2D ProjectOn(const Vector2D &v) const {
        const double vLength = v.GetLength();
        if(vLength == 0.0) {
            return Vector2D(); //0-Vector
        }
        const double len = *this * v / vLength;
        return Vector2D(v.GetAngle(), len, true);
    }

    /**
     * Returns the distance from this vector to v.
     */
    double DistanceTo(const Vector2D &v) const {
        return (*this - v).GetLength();
    }

    /**
     * Turns this vector into a unit vector (with length of 1).
     */
    Vector2D & ApplyUnification() {
        const double length = GetLength();
        x /= length;
        y /= length;
        return *this;
    }

    /**
     * Returns a unified version of this vector (with length of 1).
     */
    Vector2D Unify() {
        const double length = GetLength();
        return Vector2D(x / length, y / length);
    }

    /**
     * Compares the integer parts of this vector's components
     * with v's components respectively and returns true if they are equal.
     * (Useful for checking if two vectors represent the same pixel).
     */
    bool operator==(const Vector2D &v) const {
        return ((int) x == (int) v.x) && ((int) y == (int) v.y);
    }

    /**
     * Compares the integer parts of this vector's components
     * with v's components respectively and returns true if they are not equal.
     * (Useful for checking if two vectors represent the same pixel).
     */
    bool operator!=(const Vector2D &v) const {
        return !(*this == v);
    }

    /**
     * Returns the vector addition result of this vector and v.
     */
    Vector2D operator+(const Vector2D &v) const {
        return Vector2D(x + v.x, y + v.y);
    }

    /**
     * Returns the vector negation result of this vector.
     */
    Vector2D operator-() const {
        return Vector2D(-x, -y);
    }

    /**
     * Returns the vector subtraction result of this vector and v.
     */
    Vector2D operator-(const Vector2D &v) const {
        return Vector2D(x - v.x, y - v.y);
    }

    /**
     * Returns the vector multiplication result of this vector and s.
     */
    Vector2D operator*(double s) const {
        return Vector2D(x*s, y * s);
    }

    /**
     * Returns the vector multiplication result of this vector and 1/s.
     */
    Vector2D operator/(double s) const {
        return Vector2D(x / s, y / s);
    }

    /**
     * Returns a vector that is equal to this, but the length is enlarged by s.
     * If s is negative, the result vector's length becomes shorter.
     * If -s is greater than the length of this vector, the result will be a 0-Vector.
     * In other words: Returns a longer copy of this vector.
     */
    Vector2D operator+(double s) const {
        double newLength = GetLength() + s;
        if (newLength < 0) {
            newLength = 0.0;
        }
        return Vector2D(GetAngle(), newLength, true);
    }

    /**
     * Returns a vector that is equal to this, but the length is shorten by s.
     * If s is negative, the result vector's length becomes longer.
     * If s is greater than the length of this vector, the result will be a 0-Vector.
     * In other words: Returns a shorter copy of this vector.
     */
    Vector2D operator-(double s) const {
        double newLength = GetLength() - s;
        if (newLength < 0) {
            newLength = 0.0;
        }
        return Vector2D(GetAngle(), newLength, true);
    }

    /**
     * Returns the dot product from this vector and v.
     */
    double operator*(const Vector2D &v) const {
        return x * v.x + y * v.y;
    }

    /**
     * Adds v to this vector.
     */
    Vector2D & operator+=(const Vector2D &v) {
        this->x += v.x;
        this->y += v.y;
        return *this;
    }

    /**
     * Subtract v from this vector.
     */
    Vector2D & operator-=(const Vector2D &v) {
        //Set(this->x - v.x, this->y - v.y);
        this->x -= v.x;
        this->y -= v.y;
        return *this;
    }

    /**
     * Adds v to the length of this vector.
     * If -s is greater than the length of this vector, the result will be a 0-Vector.
     */
    Vector2D & operator+=(double s) {
        double newLength = GetLength() + s;
        if (newLength < 0.0) {
            newLength = 0.0;
        }
        SetLength(newLength);
        return *this;
    }

    /**
     * Subtract s from the length of this vector.
     * If s is greater than the length of this vector, the result will be a 0-Vector.
     */
    Vector2D & operator-=(double s) {
        return operator+=(-s);
    }

    /**
     * Multiplies this vector by s.
     */
    Vector2D & operator*=(double s) {
        Set(x*s, y * s);
        return *this;
    }

    /**
     * Multiplies this vector by 1/s.
     */
    Vector2D & operator/=(double s) {
        Set(x / s, y / s);
        return *this;
    }

    /**
     * Returns true if the length of this vector is less than the length of v.
     * In other words: Returns true if this vector is shorter than v.
     */
    bool operator<(const Vector2D &v) const {
        return (GetLength() < v.GetLength());
    }

    /**
     * Returns true if the length of this vector is greater than the length of v.
     * In other words: Returns true if this vector is longer than v.
     */
    bool operator>(const Vector2D &v) const {
        return (GetLength() > v.GetLength());
    }

    /**
     * Returns true if the length of this vector is less than s.
     */
    bool operator<(double s) const {
        return (GetLength() < s);
    }

    /**
     * Returns true if the length of this vector is greater than s.
     */
    bool operator>(double s) const {
        return (GetLength() > s);
    }

    /**
     * Returns the result of a vector rotation of this vector by s radians.
     */
    Vector2D operator^(double s) const //TODO: evtl. performantere loesung mit matrixrotation (vector rotation)
    {
        return Vector2D(GetAngle() + s, GetLength(), true);
    }

    /**
     * Rotates this vector by s radians.
     */
    Vector2D & operator^=(double s) //TODO: evtl. performantere loesung mit matrixrotation (vector rotation)
    {
        SetAngle(GetAngle() + s);
        return *this;
    }

    /**
     * Returns the angle (in radians) between this vector and v.
     */
    double AngleBetween(const Vector2D &v) const {
        //TODO: vllt ist abs(*this.angle - v.angle) effizienter
        return acos((*this * v)/(this->GetLength() * v.GetLength()));
    }

    /**
     * Returns the angle (in radians) between this vector and v.
     */
    double operator^(const Vector2D &v) const {
        //TODO: vllt ist abs(*this.angle - v.angle) effizienter
        return acos((*this * v)/(this->GetLength() * v.GetLength()));
    }
};

#endif //L_VECTOR_2D
