#ifndef PlayerConfigMenu_h
#define PlayerConfigMenu_h

#include "../BaseMenu.h"
#include "../../Util/Colors.h"

class PlayerConfigMenu : public BaseMenu {
public:

    PlayerConfigMenu(AllegroGame* game, short playerNumber) : BaseMenu(game) {
        if (playerNumber < 1 || playerNumber > MAX_PLAYERS_COUNT) {
            _game->GetLog()->LogFatal("Invalid player-number in player config menu");
        }

        _playerNumber = playerNumber;

        sprintf(_enabledStr, "enabled");
        sprintf(_disabledStr, "disabled");

        char title[32];
        sprintf(title, "Player %i", playerNumber);
        SetTitle(title);

        this->SetLayout(LAYOUT_HORIZONTALLY_CENTERED);

        for (short i = 0; i < PLAYER_COLOR_COUNT; i++) {
            Add(_colorButton[i] = new BaseItem(this, "I"));
            _colorButton[i]->SetBackground(false);
        }

        Add(_playerTypeButton = new BaseItem(this, "?"));

        _type = (MENU_TYPE)(MENU_PLAYER_CONFIG_1 + playerNumber - 1);

        Hide();
    }

    virtual ~PlayerConfigMenu() {}

    virtual void Show() {
        for (short i = 0; i < PLAYER_COLOR_COUNT; i++) {
            _colorButton[i]->SetBackgroundColor(MakePlayerColor((PLAYER_COLOR) i));
            _colorButton[i]->SetBorderLineColor(MakePlayerColor((PLAYER_COLOR) i));
        }

        UpdateStatusButtonText();
        UpdateColorSelection();
        UpdatePos();

        lmenu::Menu::Show();
    }

    virtual void EventOccurred(lmenu::Event e) {
        BaseMenu::EventOccurred(e);

        if (e.GetSource() == _playerTypeButton) {
            if (e.GetType() == lmenu::Event::CLICKED_LEFT) {
                _game->GetConfiguration()->ChangePlayerType(_playerNumber-1);
                UpdateStatusButtonText();
            } else if (e.GetType() == lmenu::Event::MOUSE_OVER) {
                e.GetSource()->SetTextColor(MakeButtonColor(BUTTON_COLOR_MOUSE_OVER));
            } else if (e.GetType() == lmenu::Event::MOUSE_LEAVES) {
                e.GetSource()->SetTextColor(MakeButtonColor(IsPlayerEnabled() ? BUTTON_COLOR_ENABLED : BUTTON_COLOR_DISABLED));
            }
        } else {
            if (e.GetType() == lmenu::Event::CLICKED_LEFT) {
                for (short i = 0; i < PLAYER_COLOR_COUNT; i++) {
                    if (e.GetSource() == _colorButton[i]) {
                        _game->GetConfiguration()->SetPlayerColor(_playerNumber-1, (PLAYER_COLOR)i);
                        break;
                    }
                }
            }
        }

        UpdatePos();
        UpdateColorSelection();
    }

private:

    bool IsPlayerEnabled() {
        return _game->GetConfiguration()->IsPlayerEnabled(_playerNumber-1);
    }

    void UpdateStatusButtonText() {
        _playerTypeButton->SetText(Player::GetPlayerTypeString(_game->GetConfiguration()->GetPlayerType(_playerNumber-1)));
        _playerTypeButton->SetTextColor(MakeButtonColor(IsPlayerEnabled() ? BUTTON_COLOR_ENABLED : BUTTON_COLOR_DISABLED));
    }

    void UpdateColorSelection() {
        for (short i = 0; i < PLAYER_COLOR_COUNT; i++) {
            if (_game->GetConfiguration()->GetPlayerColor(_playerNumber-1) == i) {
                _colorButton[i]->SetBackground(true);
                _colorButton[i]->SetTextColor(0);
            } else {
                _colorButton[i]->SetBackground(false);
                _colorButton[i]->SetTextColor(_colorButton[i]->GetBackgroundColor());
            }
        }
    }

    void UpdatePos() {
        Configuration* c = _game->GetConfiguration();
        const int margin = 1;
        if(_type == MENU_PLAYER_CONFIG_1 + 0) {
            SetPos(margin, margin);
        } else if(_type == MENU_PLAYER_CONFIG_1 + 1) {
            SetPos(c->GetResX() - margin - GetW(), margin);
        } else if(_type == MENU_PLAYER_CONFIG_1 + 2) {
            SetPos(margin, c->GetResY() - margin - GetH());
        } else if(_type == MENU_PLAYER_CONFIG_1 + 3) {
            SetPos(c->GetResX() - margin - GetW(), c->GetResY() - margin - GetH());
        } else {
            _game->GetLog()->LogError("Failed to update player config menu position");
        }
    }

    short _playerNumber;

    char _enabledStr[32];
    char _disabledStr[32];

    BaseItem* _playerTypeButton;

    BaseItem* _colorButton[PLAYER_COLOR_COUNT];
};

#endif
