#ifndef Game_h
#define Game_h

#include <allegro.h>
#include <vector>
#include <algorithm>
#include <fmod.h>

#include "Util/Vector2D.h"
#include "Util/Log.h"
#include "Player.h"
#include "Configuration.h"
#include "Arena.h"
#include "InfoPanel.h"
#include "Menus/BaseMenu.h"
#include "Menus/MenuTypes.h"
#include "Weapons/WeaponTypes.h"

#define BITMAPS_PATH "bitmaps/"
#define SOUNDS_PATH "sounds/"
#define WEAPON_ICONS_PATH BITMAPS_PATH "weapon_icons/"
#define CONFIG_FILENAME "config.ini"

class BaseMenu;
class Configuration;
class Player;
class Arena;
class InfoPanel;

class Game {
public:
    enum STATE {
        STATE_NEW,
        STATE_RUNNING,
        STATE_PAUSED,
        STATE_SUMMARIZING,
        STATE_QUITTING
    };

    enum MUSIC {
        MUSIC_MAIN,

        MUSIC_COUNT
    };

    enum SOUND {
        SOUND_MOUSE_OVER,
        SOUND_MOUSE_CLICK,
        SOUND_COLLISION,
        SOUND_ITEM_PICKUP,
        SOUND_ITEM_SELECT,
        SOUND_BORDER,
        SOUND_MISSILE,

        SOUNDS_COUNT
    };

    enum SPRITE_BITMAP {
        SPRITE_GRAVE
    };

    Game();
    virtual ~Game();

    short int GetTicksPerSecond() { return _ticksPerSecond; }
    double GetNormalSpeed();

    Log* GetLog() { return _log; }
    Configuration* GetConfiguration() { return _configuration; }
    Arena* GetArena() { return _arena; }
    InfoPanel* GetInfoPanel() { return _infoPanel; }

    short CountLivingVehicles();

    Player* GetPlayer(short i) { return _player[i]; }

    void SetState(STATE state) { _state = state; }
    STATE GetState() { return _state; }

    virtual double GetWeaponIconW(WEAPON_TYPE weaponType) = 0;
    virtual double GetWeaponIconH(WEAPON_TYPE weaponType) = 0;

    const char* GetMusicPath(MUSIC music) { return _musicPath[music]; }
    virtual void StopMusic() = 0;
    virtual void PlayMusic(MUSIC music) = 0;
    virtual void SetMusicPaused(bool paused) = 0;

    virtual void PlaySound(SOUND soundEffect, float pan) = 0;
    virtual void PlayWeaponSound(WEAPON_TYPE weapon, float pan) = 0;

    virtual void Start();
    virtual void Resume();
    virtual void CalculateScores();
    virtual void Summarize();

    virtual void Exit();

    virtual void Init();
    virtual void ShutDown();

    virtual void Run() = 0;
    
    virtual void Draw() = 0;
    virtual void Tick() = 0;
    
protected:
    Log* _log;
    Configuration* _configuration;
    Player* _player[MAX_PLAYERS_COUNT];
    Arena* _arena;
    InfoPanel* _infoPanel;

    const char* _musicPath[MUSIC_COUNT];

    short _ticksPerSecond;

    STATE _state;

    short _dyingVehiclesLastTick;
};

class AllegroGame : public Game {
public:
    AllegroGame();
    virtual ~AllegroGame();

    void InitSystem();
    void LoadResources();

    BITMAP* GetScreenBuffer() { return _screenBuffer; }
    BITMAP* GetMouseCursorBitmap() { return _mouseCursorBitmap; }
    BITMAP* GetWeaponIcon(WEAPON_TYPE weaponType) { return _weaponIcon[weaponType]; }
    BITMAP* GetArenaBackgroundBitmap() { return _arenaBgBitmap; }
    BITMAP* GetArenaBorderHBitmap() { return _arenaBorderHBitmap; }
    BITMAP* GetArenaBorderVBitmap() { return _arenaBorderVBitmap; }
    BITMAP* GetArenaBorderCBitmap() { return _arenaBorderCBitmap; }
    BITMAP* GetInfoPanelBackgroundBitmap() { return _infoPanelBgBitmap; }
    BITMAP* GetMenuBackgroundBitmap() { return _menuBgBitmap; }
    BITMAP* GetItemSlotBitmap() { return _slotBitmap; }
    BITMAP* GetGraveBitmap() { return _graveBitmap; }
    BITMAP* GetSprite(SPRITE_BITMAP sprite);

    FONT* GetFont() { return _font; }
    BaseMenu* GetMenu(MENU_TYPE menu) { return _menu[menu]; }
    
    void HideMenus();

    virtual double GetWeaponIconW(WEAPON_TYPE weaponType);
    virtual double GetWeaponIconH(WEAPON_TYPE weaponType);

    virtual void Start();
    virtual void Summarize();

    virtual void StopMusic();
    virtual void PlayMusic(MUSIC music);
    virtual void SetMusicPaused(bool paused);

    virtual void PlaySound(SOUND soundEffect, float pan);
    virtual void PlayWeaponSound(WEAPON_TYPE weapon, float pan);

    virtual void Init();
    virtual void ShutDown();
    virtual void Run();

    virtual void Draw();
    virtual void Tick();

protected:
    BaseMenu* _menu[MENU_TYPES_COUNT];

    FONT* _font;

    SAMPLE* _soundSample[SOUNDS_COUNT];
    SAMPLE* _weaponSoundSample[WEAPON_TYPES_COUNT];

    BITMAP* _screenBuffer;
    BITMAP* _mouseCursorBitmap;
    BITMAP* _graveBitmap;
    BITMAP* _menuBgBitmap;
    BITMAP* _infoPanelBgBitmap;
    BITMAP* _arenaBgBitmap;
    BITMAP* _arenaBorderHBitmap;
    BITMAP* _arenaBorderVBitmap;
    BITMAP* _arenaBorderCBitmap;
    BITMAP* _slotBitmap;
    BITMAP* _weaponIcon[WEAPON_TYPES_COUNT];

    FMOD_SYSTEM* _soundSystem;
    FMOD_SOUND* _sound;
    FMOD_CHANNEL* _channel;
};

#endif
