/***************************************************************************
 *
 * cpconfig.c
 * ChromaPlas configuration managment
 *
 * This uses an Allegro config INI file to store the configuration,
 * so the file can be used to configure Chickens's instance of Allegro
 *
 * By Andrei Ellman
 *
 **************************************************************************/



/****************************************************************************
 Includes
 */


#include <allegro.h>	// Needed for the std int types

#ifdef ALLEGRO_WINDOWS

#include <winalleg.h>	// Needed for GetModuleFileName()

#endif

#include "aeglobal.h"

#include "cpglobal.h"	// ?: Needed?

#include "cpsttngs.h"


#define AECONFIG_NO_INTCOORD2D	/*!< Used to make the int-cooord2Ds prototypes invisible to the compiler. This is to reduce dependencies. */
#define AECONFIG_NO_ANIMPARAMDESC	/*!< Used to make the animparamdesc prototypes invisible to the compiler. This is to reduce dependencies. */
#include "aeconfig.h"

#include "cpconfig.h"


/****************************************************************************
 Local Types
 */


/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */


#define CPINIFILENAME	"chromapl.ini"	/*!< The name of the INI file that \e ChromaPlas will use. */



/* Configuration defaults */

// TODO: Check that the defaults are sensible/OK in cpCheckSanityOfConfigurationSettingsAndIfInsaneSetToDefaults()

// TODO: Merge image width/height (AeCoords2d?)

// BUG/FEATURE: INI settings names cannot have spaces.



/****************************************
 *  Config Section names and Key names  *
 ****************************************/


/* * Root section * */

#define CPCONFIGSECTION_ROOT NULL

#define CPINISETTINGNAME_ROOT_SETTIMGSNAME "settingsname"


/* * GLOBALSETTINGS section * */

#define CPINISECTION_GLOBAL "GLOBALSETTINGS"

#define CPINISETTINGNAME_GLOBAL_SPEED "GlobalSpeed"



/* * PROGRAMSETTINGS section * */

#define CPINISECTION_PROGRAM "PROGRAMSETTINGS"

#define CPINISETTINGNAME_USEIDLEPRIORITYCLASS "UseIdlePriorityClass"



/* * IMAGESETTINGS section * */

#define CPINISECTION_IMAGE "IMAGESETTINGS"

#define CPINISETTINGNAME_IMAGESOURCE "ImageSource"

#define CPINISETTINGNAME_IMAGEFILENAME "ImageFileName"



/* * RENDERINGSETTINGS section * */

#define CPINISECTION_RENDERING "RENDERINGSETTINGS"

#define CPINISETTINGNAME_IMGRESSOURCE "ImageResSource"

#define CPINISETTINGNAME_IMGCUSTOMWIDTH "ImageCustomWidth"
#define CPINISETTINGNAME_IMGCUSTOMHEIGHT "ImageCustomHeight"
#define CPINISETTINGNAME_IMGCUSTOMBPP "ImageCustomBPP"

#define CPINISETTINGNAME_ZOOM "ZoomImageToScreen"
#define CPINISETTINGNAME_ZOOM_ASPECTPRESERVE "MaintainAspectWhenZooming"

#define CPINISETTINGNAME_EXPANDPLASMATOFILLSCREEN	"ExpandPlasmaToFillScreen"

#define CPINISETTINGNAME_CONFINEEVERYTHINGTOSQUARE	"ConfineEverythingToSquare"

#define CPINISETTINGNAME_RENDERINGTYPE "RenderingType"

#define CPINISETTINGNAME_DISPLAYSTATS "DisplayStats"

#define CPINISETTINGNAME_ALWAYSUSECLUTSINTCMODES "AlwaysUseColourLookupTableForRawPlasmaInTrueColourModes"

#define CPINISETTINGNAME_SETSOURCEBPPTO32IF24 "SetSourceBPPTo32If24"
#define CPINISETTINGNAME_DISABLE24BPP "Disable24BPP"
#define CPINISETTINGNAME_DISABLE8BPP "Disable8BPP"



/* * PLASMASETTINGS section * */

#define CPINISECTION_PLASMA "PLASMASETTINGS"

#define CPINISETTINGNAME_PLSAMA_GLOBAL_SPEED "GlobalPlasmaSpeed"
#define CPINISETTINGNAME_PLASMA_GLOBAL_SEEDMAGNIFICATION "GlobalPlasmaSeedMultiplier"

#define CPINISETTINGNAME_PLASMA_SEED1 "PlasmaSeed1"
#define CPINISETTINGNAME_PLASMA_SEED2 "PlasmaSeed2"

#define CPINISETTINGNAMEFORMATSTRING_LISSAJOUS_N_SETTINGS	"PlasmaLissajous%03uSettings"

#define CPINISETTINGNAME_PLASMA_ROLLSPEED "PlasmaRollSpeed"

#define CPINISETTINGNAME_PLASFX_FULLINTENSITYWRAP "FullIntensityWrap"

#define CPINISETTINGNAME_PLASFX_INTENSITY "Intensity"
#define CPINISETTINGNAME_PLASFX_INTENSITYRISETIME "IntensityRiseTime"

#define CPINISETTINGNAME_PLASMA_CONFINELISSAJOUSSESTOSQUARE "ConfineLissajoussesToSquare"



/* * PLASMACOLOURSETTINGS section * */

#define CPINISECTION_PLASMACOLOUR "PLASMACOLOURSETTINGS"

#define CPINISETTINGNAME_PLASMACOLOUR_GLOBAL_SPEED "GlobalPlasmaColourSpeed"

#define CPINISETTINGNAME_PALETTESOURCE "PaletteSource"

#define CPINISETTINGNAME_FUNKYPALETTECOLOURSPACE "FunkyPaletteEffectColourSpace"

#define CPINISETTINGNAMEFORMATSTRING_COLOURSPACESTIMULUSWAVE_N_SETTINGS	"PlasmaColourSpaceStimulationWave%1uSettings"

#define CPINISETTINGNAME_EXTERNALPALETTEFILENAME "ExternalPaletteFileName"


/* * PLASMAEFFECTS section * */

#define CPINISECTION_PLASMAFX "PLASMAEFFECTSSETTINGS"

#define CPINISETTINGNAME_PLASFX_USECOLOURSPACE "UseColorSpace"

#define CPINISETTINGNAME_PLASFX_COLOURSPACEFLAGS_RGB "ColorSpaceFlags_RGB"
#define CPINISETTINGNAME_PLASFX_COLOURSPACEFLAGS_HSV "ColorSpaceFlags_HSV"
#define CPINISETTINGNAME_PLASFX_COLOURSPACEFLAGS_HLS "ColorSpaceFlags_HLS"




/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */



/****************************************************************************
 Local (static) Global Variables
 */




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                          Accessing the ChromaPlas settings

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


static void
_cpSetConfigFile(void)
{
	/* Make sure the INI file is always in the same directory as the executable */

	// The system driver must be initialised for get_executable_name() to work.

	char szPath[CPPATHSTRINGSIZE], szINIFile[CPPATHSTRINGSIZE];
	// ?: should config file be in user's settings dir? Would a normal-user have write-permission in WINNT/SYSTEM32?


	/* Get the full name (including path) of the EXE */
	if(system_driver!=&system_none)
	{
		/* get_executable_name should be set if we have a system-driver that isn't system_none */
		get_executable_name(szPath, sizeof(szPath));
	}
	else
	{
		/* SYSTEM_NONE is the system driver */

#ifdef ALLEGRO_WINDOWS
		if(GetModuleFileName(NULL, szPath, sizeof(szPath))==0)
		{
			// WARNING: GetModuleFileName() may return a string in a differnt format than ustrzcpy() accepts.
			ustrzcpy(szPath, CPPATHSTRINGSIZE, empty_string);	/* Empty the filename string */
		}
#else
		/* We have no choice but to accept the output of get_executable_name */
		get_executable_name(szPath, sizeof(szPath));
#endif
	}


	replace_filename(szINIFile, szPath, CPINIFILENAME, CPPATHSTRINGSIZE);

	set_config_file(szINIFile);	// ?: Use override_config_file(szINIFile); instead?
}



static void
_cpSerialiseConfigLissaJousSettings(AL_CONST char *szSection, unsigned int nLJID, AeBool bWrite)
{
	char szKeyName[80];

	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	float naLissajousParamsPerDimension[3];
	float naLissajousParamDefaultsPerDimension[3];


	ASSERT(nLJID<CPMAXNUMPLASMALJS);


	/* Compose key-name */

	uszprintf(szKeyName, 80, CPINISETTINGNAMEFORMATSTRING_LISSAJOUS_N_SETTINGS, nLJID);


	/* Begin */

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}


	/* First do the X */

	if(bWrite)
	{
		/* Copy values to the array to write. */
		naLissajousParamsPerDimension[0] = G_cps.lsetPlasmaLissajousSettings[nLJID].nXAmp;
		naLissajousParamsPerDimension[1] = G_cps.lsetPlasmaLissajousSettings[nLJID].nXFreq;
		naLissajousParamsPerDimension[2] = G_cps.lsetPlasmaLissajousSettings[nLJID].nXInitialPhase;
		// We don't care about naLissajousParamDefaultsPerDimension when writing
	}
	else
	{
		/* Retrieve the defaults (in case we cant read from config settings fil) */
		naLissajousParamDefaultsPerDimension[0] = G_cpsDefaults.lsetPlasmaLissajousSettings[nLJID].nXAmp;
		naLissajousParamDefaultsPerDimension[1] = G_cpsDefaults.lsetPlasmaLissajousSettings[nLJID].nXFreq;
		naLissajousParamDefaultsPerDimension[2] = G_cpsDefaults.lsetPlasmaLissajousSettings[nLJID].nXInitialPhase;
	}

	aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault(naLissajousParamsPerDimension, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, 3, naLissajousParamDefaultsPerDimension, bWrite, szSection, szKeyName, "lissajous.x-paramater");

	if(!bWrite)
	{
		/* Set the output values to the array. */
		G_cps.lsetPlasmaLissajousSettings[nLJID].nXAmp = naLissajousParamsPerDimension[0];
		G_cps.lsetPlasmaLissajousSettings[nLJID].nXFreq = naLissajousParamsPerDimension[1];
		G_cps.lsetPlasmaLissajousSettings[nLJID].nXInitialPhase = naLissajousParamsPerDimension[2];
	}


	/* separate with a tab */

	if(bWrite)
	{
		aeAddTabToValueBuffer(&szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, szSection, szKeyName);
	}


	/* And now do the Y */

	if(bWrite)
	{
		/* Copy values to the array to write. */
		naLissajousParamsPerDimension[0] = G_cps.lsetPlasmaLissajousSettings[nLJID].nYAmp;
		naLissajousParamsPerDimension[1] = G_cps.lsetPlasmaLissajousSettings[nLJID].nYFreq;
		naLissajousParamsPerDimension[2] = G_cps.lsetPlasmaLissajousSettings[nLJID].nYInitialPhase;
		// We don't care about naLissajousParamDefaultsPerDimension when writing
	}
	else
	{
		/* Retrieve the defaults (in case we cant read from config settings fil) */
		naLissajousParamDefaultsPerDimension[0] = G_cpsDefaults.lsetPlasmaLissajousSettings[nLJID].nYAmp;
		naLissajousParamDefaultsPerDimension[1] = G_cpsDefaults.lsetPlasmaLissajousSettings[nLJID].nYFreq;
		naLissajousParamDefaultsPerDimension[2] = G_cpsDefaults.lsetPlasmaLissajousSettings[nLJID].nYInitialPhase;
	}

	aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault(naLissajousParamsPerDimension, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, 3, naLissajousParamDefaultsPerDimension, bWrite, szSection, szKeyName, "lissajous.y-paramater");

	if(!bWrite)
	{
		/* Set the output values to the array. */
		G_cps.lsetPlasmaLissajousSettings[nLJID].nYAmp = naLissajousParamsPerDimension[0];
		G_cps.lsetPlasmaLissajousSettings[nLJID].nYFreq = naLissajousParamsPerDimension[1];
		G_cps.lsetPlasmaLissajousSettings[nLJID].nYInitialPhase = naLissajousParamsPerDimension[2];
	}



	/* End */
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "lissajous-related", "", 3+3);
	}
}



static void
_cpSerialiseConfigColourspacestimulusWaveSettings(AL_CONST char *szSection, unsigned int nCssWID, AeBool bWrite)
{
	char szKeyName[80];

	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	float naColourspacestimulusWaveFloatParams[2];
	int naColourspacestimulusWaveIntParams[1];
	float naColourspacestimulusWaveFloatParams2[1];

	float naColourspacestimulusWaveFloatParamDefaults[2];
	int naColourspacestimulusWaveIntParamDefaults[1];
	float naColourspacestimulusWaveFloatParamDefaults2[1];


	ASSERT(nCssWID<CPNUMSTIMULIPERCOLOURSPACE);


	/* Compose key-name */

	uszprintf(szKeyName, 80, CPINISETTINGNAMEFORMATSTRING_COLOURSPACESTIMULUSWAVE_N_SETTINGS, nCssWID);


	/* Begin */

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}


	if(bWrite)
	{
		/* Copy values to the array to write. */
		naColourspacestimulusWaveFloatParams[0] = G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nAmp;
		naColourspacestimulusWaveFloatParams[1] = G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nFrequency;
		naColourspacestimulusWaveIntParams[0] = G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nWaveNumber;
		naColourspacestimulusWaveFloatParams2[0] = G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nInitialPhase;
		// We don't care about naColourspacestimulusWaveFloatParamDefaults when writing
	}
	else
	{
		/* Retrieve the defaults (in case we cant read from config settings file) */
		naColourspacestimulusWaveFloatParamDefaults[0] = G_cpsDefaults.csssetColourspacestimulusWaveSettings[nCssWID].nAmp;
		naColourspacestimulusWaveFloatParamDefaults[1] = G_cpsDefaults.csssetColourspacestimulusWaveSettings[nCssWID].nFrequency;
		naColourspacestimulusWaveIntParamDefaults[0] = G_cpsDefaults.csssetColourspacestimulusWaveSettings[nCssWID].nWaveNumber;
		naColourspacestimulusWaveFloatParamDefaults2[0] = G_cpsDefaults.csssetColourspacestimulusWaveSettings[nCssWID].nInitialPhase;
	}

	aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault(naColourspacestimulusWaveFloatParams, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, 2, naColourspacestimulusWaveFloatParamDefaults, bWrite, szSection, szKeyName, "colourspacestimuluswave.float-paramater");
	aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault(naColourspacestimulusWaveIntParams, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, 1, naColourspacestimulusWaveIntParamDefaults, bWrite, szSection, szKeyName, "colourspacestimuluswave.int-paramater");
	aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault(naColourspacestimulusWaveFloatParams2, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, 1, naColourspacestimulusWaveFloatParamDefaults2, bWrite, szSection, szKeyName, "colourspacestimuluswave.float-paramater");

	if(!bWrite)
	{
		/* Set the output values to the array. */
		G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nAmp = naColourspacestimulusWaveFloatParams[0];
		G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nFrequency = naColourspacestimulusWaveFloatParams[1];
		G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nWaveNumber = naColourspacestimulusWaveIntParams[0];
		G_cps.csssetColourspacestimulusWaveSettings[nCssWID].nInitialPhase = naColourspacestimulusWaveFloatParams2[0];
	}


	/* End */
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "colourspacestimuluswave-related", "", 3+3);
	}
}



/* Serialise the app-specific part of the config (data related to Allegro are serialised elsewhere) */
void
_cpSerialiseConfig(AeBool bWrite)
{
	unsigned int nI;




#define AECONFIGCURRENTSECTION CPCONFIGSECTION_ROOT



	aeSERIALISECONFIGSTRING(G_cps.szConfigSettingsName, CPCHROMAPLASCONFIGSETTINGSNAMEBUFFERLENGTH, CPINISETTINGNAME_ROOT_SETTIMGSNAME, CPDEFAULT_SETTINGSNAME);



#undef AECONFIGCURRENTSECTION



	


	/* Global settings */

#define AECONFIGCURRENTSECTION CPINISECTION_GLOBAL



	aeSERIALISECONFIGFLOAT(&G_cps.nGlobalSpeed, CPINISETTINGNAME_GLOBAL_SPEED, CPDEFAULT_GLOBAL_SPEED);



#undef AECONFIGCURRENTSECTION






	/* Program settings */

#define AECONFIGCURRENTSECTION CPINISECTION_PROGRAM



	aeSERIALISECONFIGBOOL(&G_cps.bUseIdlePriorityClass, CPINISETTINGNAME_USEIDLEPRIORITYCLASS, CPDEFAULT_USEIDLEPRIORITYCLASS);



#undef AECONFIGCURRENTSECTION






	/* Image settings */

#define AECONFIGCURRENTSECTION CPINISECTION_IMAGE



	aeSERIALISECONFIGTYPEDEFDSCALARTYPE(&G_cps.isImgSource, CPINISETTINGNAME_IMAGESOURCE, CPDEFAULT_IMAGESOURCE, CpImageSource, aeSERIALISECONFIGINT, int);

	aeSERIALISECONFIGSTRING(G_cps.szFileName, CPPATHSTRINGSIZE, CPINISETTINGNAME_IMAGEFILENAME, CPDEFAULT_IMAGEFILENAME);



#undef AECONFIGCURRENTSECTION






	/* Rendering settings */

#define AECONFIGCURRENTSECTION CPINISECTION_RENDERING



	aeSERIALISECONFIGTYPEDEFDSCALARTYPE(&G_cps.irsImgResSource, CPINISETTINGNAME_IMGRESSOURCE, CPDEFAULT_IMGRESSOURCE, CpImageResSource, aeSERIALISECONFIGINT, int);

	aeSERIALISECONFIGINT(&G_cps.nWidth, CPINISETTINGNAME_IMGCUSTOMWIDTH, CPDEFAULT_IMGCUSTOMWIDTH);
	aeSERIALISECONFIGINT(&G_cps.nHeight, CPINISETTINGNAME_IMGCUSTOMHEIGHT, CPDEFAULT_IMGCUSTOMHEIGHT);
	aeSERIALISECONFIGINT(&G_cps.nBPP, CPINISETTINGNAME_IMGCUSTOMBPP, CPDEFAULT_IMGCUSTOMBPP);

	aeSERIALISECONFIGBOOL(&G_cps.bZoom, CPINISETTINGNAME_ZOOM, CPDEFAULT_ZOOM);
	aeSERIALISECONFIGBOOL(&G_cps.bZoomAspectPreserve, CPINISETTINGNAME_ZOOM_ASPECTPRESERVE, CPDEFAULT_ZOOM_ASPECTPRESERVE);

	aeSERIALISECONFIGBOOL(&G_cps.bExpandPlasmaToFillScreen, CPINISETTINGNAME_EXPANDPLASMATOFILLSCREEN, CPDEFAULT_EXPANDPLASMATOFILLSCREEN);
	
	aeSERIALISECONFIGBOOL(&G_cps.bConfineEverythingToSquare, CPINISETTINGNAME_CONFINEEVERYTHINGTOSQUARE, CPDEFAULT_CONFINEEVERYTHINGTOSQUARE);


	aeSERIALISECONFIGTYPEDEFDSCALARTYPE(&G_cps.rtRenderingType, CPINISETTINGNAME_RENDERINGTYPE, CPDEFAULT_RENDERINGTYPE, CpRenderingType, aeSERIALISECONFIGINT, int);

	aeSERIALISECONFIGBOOL(&G_cps.bDisplayStats, CPINISETTINGNAME_DISPLAYSTATS, CPDEFAULT_DISPLAYSTATS);

	aeSERIALISECONFIGBOOL(&G_cps.bAlwaysUseCLUTsInTruecolourModes, CPINISETTINGNAME_ALWAYSUSECLUTSINTCMODES, CPDEFAULT_ALWAYSUSECLUTSINTCMODES);

	aeSERIALISECONFIGBOOL(&G_cps.bSetSourceBPPTo32If24, CPINISETTINGNAME_SETSOURCEBPPTO32IF24, CPDEFAULT_SETSOURCEBPPTO32IF24);
	aeSERIALISECONFIGBOOL(&G_cps.bDisable24bpp, CPINISETTINGNAME_DISABLE24BPP, CPDEFAULT_DISABLE24BPP);
	aeSERIALISECONFIGBOOL(&G_cps.bDisable8bpp, CPINISETTINGNAME_DISABLE8BPP, CPDEFAULT_DISABLE8BPP);


#undef AECONFIGCURRENTSECTION






	/* Plasma settings */

#define AECONFIGCURRENTSECTION CPINISECTION_PLASMA



	aeSERIALISECONFIGFLOAT(&G_cps.nPlasmaGlobalSpeed, CPINISETTINGNAME_PLSAMA_GLOBAL_SPEED, CPDEFAULT_PLASMA_GLOBAL_SPEED);
	aeSERIALISECONFIGFLOAT(&G_cps.nPlasmaGlobalSeedMultiplier, CPINISETTINGNAME_PLASMA_GLOBAL_SEEDMAGNIFICATION, CPDEFAULT_PLASMA_GLOBAL_SEEDMAGNIFICATION);

	aeSERIALISECONFIGFLOAT(&G_cps.nPlasmaSeed1, CPINISETTINGNAME_PLASMA_SEED1, CPDEFAULT_PLASMA_SEED1);
	aeSERIALISECONFIGFLOAT(&G_cps.nPlasmaSeed2, CPINISETTINGNAME_PLASMA_SEED2, CPDEFAULT_PLASMA_SEED2);
	
	for(nI=0;nI<CPMAXNUMPLASMALJS;nI++)
	{
		_cpSerialiseConfigLissaJousSettings(AECONFIGCURRENTSECTION, nI, bWrite);
	}
	aeSERIALISECONFIGFLOAT(&G_cps.nPlasmaRollSpeed, CPINISETTINGNAME_PLASMA_ROLLSPEED, CPDEFAULT_PLASMA_ROLLSPEED);


	aeSERIALISECONFIGBOOL(&G_cps.bFullIntensityHueWrap, CPINISETTINGNAME_PLASFX_FULLINTENSITYWRAP, CPDEFAULT_PLASFX_FULLINTENSITYWRAP);

	aeSERIALISECONFIGUINT8(&G_cps.cIntensity, CPINISETTINGNAME_PLASFX_INTENSITY, CPDEFAULT_PLASFX_INTENSITY);
	aeSERIALISECONFIGFLOAT(&G_cps.nIntensityRiseTime, CPINISETTINGNAME_PLASFX_INTENSITYRISETIME, CPDEFAULT_PLASFX_INTENSITYRISETIME);

	aeSERIALISECONFIGBOOL(&G_cps.bConfineLissajoussesToSquare, CPINISETTINGNAME_PLASMA_CONFINELISSAJOUSSESTOSQUARE, CPDEFAULT_PLASMA_CONFINELISSAJOUSSESTOSQUARE);


#undef AECONFIGCURRENTSECTION






	/* Plasma colour settings (when no image) */

#define AECONFIGCURRENTSECTION CPINISECTION_PLASMACOLOUR



	aeSERIALISECONFIGFLOAT(&G_cps.nPlasmaColourGlobalSpeed, CPINISETTINGNAME_PLASMACOLOUR_GLOBAL_SPEED, CPDEFAULT_PLASMACOLOUR_GLOBAL_SPEED);

	aeSERIALISECONFIGTYPEDEFDSCALARTYPE(&G_cps.psPaletteSource, CPINISETTINGNAME_PALETTESOURCE, CPDEFAULT_PALETTESOURCE, CpPaletteSource, aeSERIALISECONFIGINT, int);

	aeSERIALISECONFIGTYPEDEFDSCALARTYPE(&G_cps.csFunkyPaletteEffectColourSpace, CPINISETTINGNAME_FUNKYPALETTECOLOURSPACE, CPDEFAULT_FUNKYPALETTECOLOURSPACE, CpColourSpace, aeSERIALISECONFIGINT, int);

	for(nI=0;nI<CPNUMSTIMULIPERCOLOURSPACE;nI++)
	{
		_cpSerialiseConfigColourspacestimulusWaveSettings(AECONFIGCURRENTSECTION, nI, bWrite);
	}

	aeSERIALISECONFIGSTRING(G_cps.szPaletteSourceFileName, CPPATHSTRINGSIZE, CPINISETTINGNAME_EXTERNALPALETTEFILENAME, CPDEFAULT_PALETTESOURCEFILENAME);


#undef AECONFIGCURRENTSECTION






	/* Plasma effects (when using an image) */

#define AECONFIGCURRENTSECTION CPINISECTION_PLASMAFX



	aeSERIALISECONFIGTYPEDEFDSCALARTYPE(&G_cps.csColourSpaceToUse, CPINISETTINGNAME_PLASFX_USECOLOURSPACE, CPDEFAULT_PLASFX_USECOLOURSPACE, CpColourSpace, aeSERIALISECONFIGINT, int);

	aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault(G_cps.baaColourSpaceFlags[CPCOLOURSPACEINDEX_RGB], AECONFIGCURRENTSECTION, CPINISETTINGNAME_PLASFX_COLOURSPACEFLAGS_RGB, CPNUMSTIMULIPERCOLOURSPACE, G_cpsDefaults.baaColourSpaceFlags[CPCOLOURSPACEINDEX_RGB], bWrite, "RGB colourspace-flag");
	aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault(G_cps.baaColourSpaceFlags[CPCOLOURSPACEINDEX_HSV], AECONFIGCURRENTSECTION, CPINISETTINGNAME_PLASFX_COLOURSPACEFLAGS_HSV, CPNUMSTIMULIPERCOLOURSPACE, G_cpsDefaults.baaColourSpaceFlags[CPCOLOURSPACEINDEX_HSV], bWrite, "HSV colourspace-flag");
	aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault(G_cps.baaColourSpaceFlags[CPCOLOURSPACEINDEX_HLS], AECONFIGCURRENTSECTION, CPINISETTINGNAME_PLASFX_COLOURSPACEFLAGS_HLS, CPNUMSTIMULIPERCOLOURSPACE, G_cpsDefaults.baaColourSpaceFlags[CPCOLOURSPACEINDEX_HLS], bWrite, "HLS colourspace-flag");



#undef AECONFIGCURRENTSECTION




	// TODO: If any of the functions return read-errors or write-errors, then bring up a [platform-specific] dialog to warn that the config could not be loaded/saved and that defaults will be used (if loading).
	// but if SS, then it's best not to dispaly a warning dialog - just a text-message on the saver to say the defaults were used.


}



// If !bLoadAppSettings, only the Allegro-engine settings will be loaded.
//  \note It is recommended to call cpCheckSanityOfConfigurationSettingsAndIfInsaneSetToDefaults() after the INI has been read.
void
cpLoadConfig(AeBool bLoadAppSettings)
{
	_cpSetConfigFile();


	if(bLoadAppSettings)
	{
		_cpSerialiseConfig(FALSE);
	}



	// TODO: Check for valid ranges / sane values
	// TODO: Also, check for values not supported under the current OS/Hardware/etc.
}


// Saves both the app-settings and th3e allegro engine settings.
void
cpSaveConfig(void)
{
	_cpSetConfigFile();


	_cpSerialiseConfig(TRUE);


	flush_config_file();	/* Make sure the config file on the disk is up to date */
}




/****************************************************************************/
/*!

 \brief Checks the sanity of things we have read from the INI file.

  \note This should be called after cpLoadConfig()

 */

void
cpCheckSanityOfConfigurationSettingsAndIfInsaneSetToDefaults(void)
{
	// ?: Is there a way of alerting the user about these insane thingies apart from trace-messages?
	// Bear in mind that the GUI has not yet been initialised.

	AeBool bAtLeastOneInsaneValueConverted = FALSE;



	// TODO: Do some!
	// ?: Are any sanitychecks done elsewhere?


	// May also want to do functions for overriding. But then again, some over-riding is best done if certain things (eg. graphics modes) cannot be set.

	if(bAtLeastOneInsaneValueConverted)
	{
		TRACE("\n");
	}

	// ?: Return a BOOL to say if owt were modified? - bAtLeastOneInsaneValueConverted ?
}
