/***************************************************************************
 *
 * main.c
 * ChromaPlas main program
 *
 * By Andrei Ellman
 *
 **************************************************************************/


/****************************************************************************
 Includes
 */


#ifdef ALLEGRO_WINDOWS

#define ALLEGRO_NO_MAGIC_MAIN

#endif



#include <allegro.h>

#include "aeglobal.h"

#include "cpglobal.h"

#include "cpmthhlp.h"
#include "cpsttngs.h"

#include "cpplasma.h"

#include "cpgfx.h"

#include "cpconfig.h"

#include "cpprglp.h"

#ifdef ALLEGRO_WINDOWS

#include <winalleg.h>
#include "cpwin_.h"
#include "cpwin_ss.h"

#endif


/****************************************************************************
 Local Types
 */


/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */


/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */


// ?: -> cpprglp ?

volatile int G_nTargetTime;
volatile int G_nGameTime;

int G_nFps;



/****************************************************************************
 Local (static) Global Variables
 */

static volatile AeBool g_bGlobalExitFlag;	/*!< TRUE if a close-message has been sent to the app. Can be set to FALSE if we want to do a prolonged exit. */


#ifdef CP_IS_SCREENSAVER

static volatile AeBool g_bAppInForeground;	/* Used to detect if the app has been switched to the background */

#endif





/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                                Interupt stuff

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


/*************************
 *  Interrupt callbacks  *
 *************************/

#ifdef CP_IS_SCREENSAVER

/* display switch callback */
static
void cpDispswCallback(void)
{
   g_bAppInForeground = FALSE;	/* If ever the app gets switched to the background, then it should end. */
}
END_OF_FUNCTION(cpDispswCallback);

#endif


static void
cpCloseMessageHandler(void)
{
	TRACE("An app-close-message was recieved.\n");
	g_bGlobalExitFlag = TRUE;
}
END_OF_FUNCTION(cpCloseMessageHandler);




// ?: -> cpprglp ?

/* Timer callback for regulating the speed */
void
cpIncrementTargetTime(void)
{
	G_nTargetTime++;
}
END_OF_FUNCTION(cpIncrementTargetTime);

/* Timer callback for measuring game-time (to be used when measuring FPS. */
void
cpIncrementGameTime(void)
{
	G_nGameTime++;
}
END_OF_FUNCTION(cpIncrementGameTime);




/*********************************************************************
 *  Auxilliary functions related to variables modified by interrupts *
 *********************************************************************/

void
cpResetGlobalExitFlag(void)
{
	g_bGlobalExitFlag = FALSE;
}


#ifdef CP_IS_SCREENSAVER

void
cpResetAppInForegroundFlag(void)
{
	g_bAppInForeground = TRUE;
}

#endif


// \sa cpStopTimers
AeBool
cpStartTimers(int nRefreshRate)
{
	/* Used to keep plasma at a constant speed */

	// ?: Instead of G_nTargetTime/cpIncrementTargetTime(), how about using Allegro's retrace simulator (retrace_count)?
	// Apparently, retrace_count outside of DOS is inacccurate. Also, what if retrace_count overflows?


	// ?: Should I wait until 'retrace_count' has changed before I start this (or vsync() )
	// Alternatively, do a rest() timed so we install this just before the next vsync(). But problem with that is that we may not get the process back immediately.

	// For some strange reason, under MSVC and Allegro 4.1.1, G_nFps is 37 without cpIncrementTargetTime. With, G_nFps is a bit above the BPS value of the speed paramater.
	if(install_int_ex(cpIncrementTargetTime, BPS_TO_TIMER(nRefreshRate)))
	{
		TRACE("Error Initialising speed timer interrupt.\n");
		return FALSE;
	}

	// TODO: Don't do this one if no FPS measurement is to be made.
	// ?: If using intensity-rise-time, will we use this one, or use the one above with the refresh-rate (but refresh-rate may not always be known)

	/* Used to time seconds so we can do things like measure the FPS rate of what we're drawing */
	if(install_int_ex(cpIncrementGameTime, BPS_TO_TIMER(AEGAMETIMENUMDIVISIONSPERSECOND)))
	{
		TRACE("Error Initialising fps timer interrupt.\n");
		return FALSE;
	}

	return TRUE;
}


// \sa cpStartTimers
void
cpStopTimers(void)
{
	/* remove interrupts */
	remove_int(cpIncrementGameTime);	// Do we need to check to see if installed?
	remove_int(cpIncrementTargetTime);	// Do we need to check to see if installed?
}




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                          Various ancillary functions.

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


// ?: -> aeplabld.c or aealleg.c?

/****************************************************************************/
/*!

 \brief Given the Allegro ID, write the 4-letter Allegro ID to a string.

  \remarks
  If high byte of nID == 0, then 'auto' is returned.
  Also writes the trailing \0

  \param  szModeIDString	The string to recieve the mode-ID (must be at least 5 bytes long).
  \param  nID				The ID of the mode.

  \return	A pointer to \p szModeIDString (this is useful because then, we can use this function as a paramater).

 */

AL_CONST char *
aeGetAllegroIDStringFromAllegroID(char *szModeIDString, int nID)
{
	char *szModeIDStringPos = szModeIDString;

	*szModeIDStringPos = nID>>24;

	/* If the first byte is \0, this is an autodetected mode. */
	if(!(*szModeIDStringPos))
	{
		*(szModeIDStringPos++)='a';
		*(szModeIDStringPos++)='u';
		*(szModeIDStringPos++)='t';
		*(szModeIDStringPos++)='o';
	}
	else
	{
		szModeIDStringPos++;
		*(szModeIDStringPos++) = (nID>>16)&0x000000FF;
		*(szModeIDStringPos++) = (nID>>8)&0x000000FF;
		*(szModeIDStringPos++) = (nID)&0x000000FF;
	}
	*szModeIDStringPos=0;

	return szModeIDString;
}



// ?: -> cpprglp?

// ?: Should we pass in a flag (or use a global) that states we are running from the preview-window, in which case the return value would always be FALSE?

AeBool
cpCanExitApp(void)
{

#ifdef CP_IS_SCREENSAVER

	int nMickeyX, nMickeyY;

	get_mouse_mickeys(&nMickeyX, &nMickeyY);
	// Mouse-buttons too?

#endif


#ifdef CP_IS_SCREENSAVER
	if(keypressed() || nMickeyX || nMickeyY || !g_bAppInForeground || g_bGlobalExitFlag)
#else
	if(key[KEY_SPACE] || key[KEY_ESC] || g_bGlobalExitFlag)
#endif
	{
		g_bGlobalExitFlag = TRUE;	/* Make sure that subsequent calls to cpCanExitApp() return TRUE */
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                                  Setup/Cleanup

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


// returns nonzero on error.
// \sa cpCleanupChromaPlasProgramTiniAllegro
AeBool
cpSetupChromaPlasProgramInitAllegro(AeBool bUseSystemDriver, AeBool bLoadAllegroSettings, AeBool bLoadAppSettings)
{
	// ?: Should we allow (!bLoadAppSettings && bLoadAllegroSettings) ?

	if(bUseSystemDriver)
	{
		/* Use standard system driver */
		if(allegro_init())
		{
			TRACE("Error Initialising Allegro (standard system driver).\n");
			return FALSE;
		}
		TRACE("ChromaPlas: Set up Allegro using standard system driver.\n");
	}
	else
	{
		/* Use the SYSTEM_NONE system-driver */
		// A stripped down version of Allegro that won't even try to touch your hardware or do anything platform specific
		if(install_allegro(SYSTEM_NONE, &errno, atexit))
		{
			TRACE("Error Initialising Allegro (no system driver).\n");
			return FALSE;
		}
		TRACE("ChromaPlas: Set up Allegro using no system driver.");
	}


	/* Acquire the user configuration */

	if(bLoadAllegroSettings || bLoadAppSettings)
	{
		TRACE("ChromaPlas: Loading %s settings.\n", (bLoadAllegroSettings?(bLoadAppSettings?"Allegro and ChromaPlas":"Allegro"):"ChromaPlas"));
		cpLoadConfig(bLoadAppSettings);
		cpCheckSanityOfConfigurationSettingsAndIfInsaneSetToDefaults(); // ?: Should this be done later on?
	}
	else
	{
		// Otherwise use the settings from G_cps which have been updated by the dialog.
		TRACE("ChromaPlas: Not loading any settings.\n");
	}


	return TRUE;
}


// \sa _cpCleanupChromaPlasProgramTiniInputAndTimer
static AeBool
_cpSetupChromaPlasProgramInitInputAndTimer(void)
{
	if(install_keyboard())
	{
		TRACE("Error Initialising Keyboard.\n");
		return FALSE;
	}

	if(install_timer())
	{
		TRACE("Error Initialising Timer.\n");
		return FALSE;
	}

	if(install_mouse() == -1)
	{
		TRACE("Error Initialising Mouse.\n");
		return FALSE;
	}

	return TRUE;
}


// \sa _cpCleanupChromaPlasProgramTiniInterrupts
static AeBool
_cpSetupChromaPlasProgramInitInterrupts(void)
{

#ifdef CP_IS_SCREENSAVER

	LOCK_VARIABLE(g_bAppInForeground);
	LOCK_FUNCTION(cpDispswCallback);

#endif

	LOCK_VARIABLE(g_bGlobalExitFlag);
	LOCK_FUNCTION(cpCloseMessageHandler);

	LOCK_VARIABLE(G_nTargetTime);
	LOCK_FUNCTION(cpIncrementTargetTime);

	LOCK_VARIABLE(G_nGameTime);
	LOCK_FUNCTION(cpIncrementGameTime);


	// Actual interrupts installed in cpStartTimers()


	return TRUE;
}


/* Sets up everything needed by Allegro etc. all in one go */
// Note if Running the plasma from elsewhere (eg. test-mode from the settings dialog), we won't be calling all of these at once.
// \sa _cpCleanUpChromaPlasProgram
static AeBool
_cpSetupChromaPlasProgram(AeBool bAllegroPreviouslyInitialised, AeBool bLoadAllegroSettings, AeBool bLoadAppSettings)
{
	if(!bAllegroPreviouslyInitialised)
	{
		// Or use cpSetupChromaPlasProgramInitAllegro(FALSE, FALSE, TRUE) to try and use Windows GDI without creating an Allegro Window. (?: can we get FPScounter and vsync and constant-speed to work like so). Do we want to load the Allegro settings? If so, we'd have to do them manually.
		if(!cpSetupChromaPlasProgramInitAllegro(TRUE, bLoadAllegroSettings, bLoadAppSettings))
		{
			return FALSE;
		}
	}


	#ifdef CP_IS_SCREENSAVER
	#ifdef ALLEGRO_WINDOWS

		/* NOTE: Done after Allegro initialised so we can read G_cps.bUseIdlePriorityClass from the INI file. */
		if(G_cps.bUseIdlePriorityClass)
		{
			cpSetPriorityClassIdle_Win();
		}

	#endif
	#endif


	if(!_cpSetupChromaPlasProgramInitInputAndTimer())
	{
		return FALSE;
	}

	if(!_cpSetupChromaPlasProgramInitInterrupts())
	{
		return FALSE;
	}


	return TRUE;
}



/* Sets up Chromaplas's paramaters and data etc. */
// bLoadSettings is set to FALSE when testing fromthe settings dialog.
// \sa cpTiniChromaPlas
AeBool
cpInitChromaPlas(AeBool bSetGfxMode, int nDestWidth, int nDestHeight, int nDestBPP)
{
	int nW, nH;



	cpSetupPreCalcs();


	// TODO: Make sure that as much as possible is out the way before we change gfx modes.
	// This is so we can display something immediately without the screen flickering.
	// TODO: Figure out how to write directly to windows-screen (ie. without an Allegro Window)
	// or how to change to desktop-rez/bits ohne changing hertz. (methinx that set_gfx_mode() should be called with GFX_DIRECTX_SOFT or GFX_DIRECTX_OVL)

	// Perhaps try DB, but use blit_to_hdc with desktop as the context.


	if(!cpSetupGfx(bSetGfxMode, nDestWidth, nDestHeight, nDestBPP))
	{
		TRACE("Error Setting up gfx.\n");
		cpCleanUpGfx(bSetGfxMode);
		cpCleanUpPreCalcs();
		return FALSE;
	}

	ASSERT(G_bmpDest);


	cpGetSizeToMakePlasma(&nW, &nH);

	if(!(cpInitPlasma(nW, nH) ))
	{
		TRACE("Error Setting up plasma tabs.\n");
		cpTiniPlasma();
		cpCleanUpGfx(bSetGfxMode);
		cpCleanUpPreCalcs();
		return FALSE;
	}

	cpResetPlasmaRiseTimeTicks(G_nRefreshRate);


	return TRUE;
}







// \sa _cpSetupChromaPlasProgramInitInterrupts
static void
_cpCleanupChromaPlasProgramTiniInterrupts(void)
{

	// Actual interrupts removed in cpStopTimers()

}



// \sa _cpSetupChromaPlasProgramInitInputAndTimer
static void
_cpCleanupChromaPlasProgramTiniInputAndTimer(void)
{
	remove_mouse();	// Do we need to check to see if installed?
	remove_timer();
	remove_keyboard();
}



// \sa cpSetupChromaPlasProgramInitAllegro
void
cpCleanupChromaPlasProgramTiniAllegro(void)
{
	TRACE("ChromaPlas: Cleaning up Allegro.\n");
	allegro_exit();
}


// \sa _cpSetupChromaPlasProgram
static void
_cpCleanUpChromaPlasProgram(AeBool bAllegroPreviouslyInitialised)
{
	clear_keybuf();	// ?: Where should I call this when exiting from test-mode?

	_cpCleanupChromaPlasProgramTiniInterrupts();
	_cpCleanupChromaPlasProgramTiniInputAndTimer();


	#ifdef CP_IS_SCREENSAVER
	#ifdef ALLEGRO_WINDOWS

		if(G_cps.bUseIdlePriorityClass)
		{
			cpRestorePriorityClass_Win();
		}

	#endif
	#endif


	if(!bAllegroPreviouslyInitialised)
	{
		cpCleanupChromaPlasProgramTiniAllegro();
	}
}



/* Un-initialises what was initialised in cpInitChromaPlas */
// \sa cpInitChromaPlas
void
cpTiniChromaPlas(AeBool bSetGfxMode)
{
	cpTiniPlasma();
	cpCleanUpGfx(bSetGfxMode);

	cpCleanUpPreCalcs();

	// ?: If the config is invalid, should we make up a valid one and save it?
}





/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                              The program

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */




// return -1 on error
// bAllegroPreviouslyInitialised would be set to FALSE if we had already fully setup Allegro (complete with the system-driver) somewhere elae.
int
cpDoProgram(AeBool bAllegroPreviouslyInitialised, AeBool bLoadAllegroSettings, AeBool bLoadAppSettings)
{


	int nDestWidth, nDestHeight, nDestBPP;

	nDestWidth = nDestHeight = nDestBPP = 0;	/* Start off with 0 so cpSetupGfx() knows it has to work these out */




	TRACE("\nWe're at the start of cpDoProgram() - let's go...\n");


	if(!_cpSetupChromaPlasProgram(bAllegroPreviouslyInitialised, bLoadAllegroSettings, bLoadAppSettings))
	{
		_cpCleanUpChromaPlasProgram(bAllegroPreviouslyInitialised);
		return -1;
	}



	cpResetGlobalExitFlag();

	// Do not bother with return value. But we could still store to a flag so we can display it in the overlay-display.
	set_close_button_callback(cpCloseMessageHandler);



#ifdef CP_IS_SCREENSAVER
	/* If this is a screensaver, then make sure that if the program has the means to know when to end if it's switched into the background. */

	/* Set the display-switch mode */
	// WARNING: Allegro-docs: You must take special care when using this mode, because bad things will happen if the screen bitmap gets changed around when your program isn't expecting it.
	// ?: Should this be done after the GFX-mode has been set?
#ifdef ALLEGRO_WINDOWS
	set_display_switch_mode(SWITCH_BACKAMNESIA);
#else
	// ?: Is SWITCH_BACKAMNESIA reuired for any other platforms?
	// Alternatively, we can try setting SWITCH_BACKGROUND and if that fails, try SWITCH_BACKAMNESIA (see Allegro's exswitch.c )
	set_display_switch_mode(SWITCH_BACKGROUND);
#endif

	cpResetAppInForegroundFlag();
	set_display_switch_callback(SWITCH_OUT, cpDispswCallback);

#endif





#ifdef ALLEGRO_WINDOWS
	set_gdi_color_format();	/* Make sure we get the RGB<->BGR thing sorted out (eg. Windows GDI) */
#else
#if 0	// Note, if we get BGR<->RGB errors (red and blue swapped), getridof this #if 0 .
	// Warning: Assumes we're not using SYSTEM_NONE
	set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0);	/* Make sure we get the RGB<->BGR thing sorted out [before we load a bitmap] if we haven't already. */
	// ?: Should we be using GFX_SAFE ?
#endif
#endif




	if(!cpInitChromaPlas(TRUE, nDestWidth, nDestHeight, nDestBPP))
	{
		_cpCleanUpChromaPlasProgram(bAllegroPreviouslyInitialised);
		return -1;
	}


	cpDoPlasma();


	cpTiniChromaPlas(TRUE);



	TRACE("We're at the end of cpDoProgram() - About to clean up...\n");

	_cpCleanUpChromaPlasProgram(bAllegroPreviouslyInitialised);

	return 0;
}




#ifdef ALLEGRO_WINDOWS


int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpszCmdParam, int nCmdShow)
{
	HWND hwnd;
	char *args;

	int nRetVal = 0;

	TRACE("Entering WinMain()\n");


	args = lpszCmdParam;

	if ((args[0] == '-') || (args[0] == '/'))
		args++;

	if ((args[0]) && ((args[1] == ' ') || (args[1] == ':')))
		hwnd = (HWND)atoi(args+2);
	else
		hwnd = GetActiveWindow();


	TRACE("Param is: ");
	TRACE(&args[0]);
	TRACE("\n\n");


#ifdef CP_IS_SCREENSAVER

	cpSetGlobalInstanceHandles_Win_SS(hInstance, hPrevInstance);

#endif

	cpInitCPS(&G_cps);	// in cpDoSettings()/cpDoPassword()/cpDoPreview()/cpDoProgram()?



#ifdef CP_IS_SCREENSAVER

	/* Windows screensavers have a single-letter command passed to their command-line to select the screensaver-mode. */
	switch (utolower(args[0]))
	{
		case 'c':
		{
			TRACE("About to execute cpDoSettings_Win_SS()\n\n");
			nRetVal = cpDoSettings_Win_SS(hwnd);
		}
		break;

		case 'a':
		{

			TRACE("About to execute cpDoPassword_Win_SS()\n\n");
			nRetVal = cpDoPassword_Win_SS(hwnd);
		}
		break;

		case 'p':
		{

			TRACE("About to execute cpDoPreview_Win_SS()\n\n");
			nRetVal = cpDoPreview_Win_SS(hwnd);
		}
		break;

		case 's':
		default:	/* TODO: instead of default, should I bring up a dialog with the given option saying it's a non-SS option. */
		{

			TRACE("About to execute cpDoProgram() (screensaver-mode) after we've created a mutex.\n\n");

#else

			TRACE("About to execute cpDoProgram() (non-screensaver-mode) after we've created a mutex.\n\n");

#endif

			/* Make sure only one instance can run at a time. */
			if(!cpCreateAppMutex_Win())
			{
				TRACE("ERROR: Could not create a mutex. Perhaps another instance of the app is running.\n\n");
				nRetVal = -1;
			}
			else
			{
				nRetVal = cpDoProgram(FALSE, TRUE, TRUE);

				cpReleaseAppMutex_Win();
			}

#ifdef CP_IS_SCREENSAVER

		}
		break;
	}

#endif


	cpDestroyCPS(&G_cps);	// in cpDoSettings()/cpDoPassword()/cpDoPreview()/cpDoProgram()?


	TRACE("Leaving WinMain()\n");

	return nRetVal;
}


#else

int
main(void)
{
	int nRetVal = 0;

	TRACE("Entering main()\n");


	cpInitCPS(&G_cps);	// in cpDoProgram()?

	nRetVal = cpDoProgram(FALSE, TRUE, TRUE);

	cpDestroyCPS(&G_cps);	// in cpDoProgram()?


//	TRACE("Leaving WinMain()\n");	// Commented out coz otherwise this overwrites the allegro.log file.

	return nRetVal;
}
END_OF_MAIN()


#endif

