/***************************************************************************
 *
 * cpplasma.c
 * ChromaPlas plasma-code
 * 
 * By Andrei Ellman
 * Portions of this code were taken from plasma code by Jan Moeller & Erik Hansen.
 *
 **************************************************************************/


/****************************************************************************
 Includes
 */


#include <stdlib.h>
#include <math.h>
#include <allegro.h>	// ?: Needed?

#include "aeglobal.h"

#include "cpglobal.h"

#include "cpmthhlp.h"

#include "cpsttngs.h"

#include "cpplasma.h"


/****************************************************************************
 Local Types
 */

/* Pre-calculated values for the lissajousses that remain constant while the plasma is running */
typedef struct CpLissaJousPreCalcsTag
{
	float nXFreq, nYFreq;	/* The frequrency (in RADIANS PER 1/CPREFERENCEREFRESHRATEth SECOND). This is pre-calculated because the settings in the config are in cycles per second. */

}
CpLissaJousPreCalcs;

/* The current state of a single lissajous. This corresponds to the position a plasma-circle is drawn at any given time. */
typedef struct CpLissajousStateTag
{
	float nXPhase, nYPhase;	/* The current phase-angle (in RADIANS) */
}
CpLissajousState;


/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */


/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */


/* Memory area for generating the plasma in. 1x1 plasma pixels corespond to 2x2 screen pixels */
int G_nPlasmaBoxW;
int G_nPlasmaBoxH;

/* Used by the intensity rise time */
AeBool G_bIntensityHasChanged;


/****************************************************************************
 Local (static) Global Variables
 */

static uint8_t *g_caPlasmaBuffer;	/* buffer for the plasma body */

/* The 'box' is just a window into the tabs. We are using a tab 4* the area of the box. */
static uint8_t *g_caPlasmaTab1;	/* table one for the plasma: the distance from (x,y) to the center (rounded off to char by simple overflow) */
static uint8_t *g_caPlasmaTab2;	/* table two for the plasma: similar to 'Tab1', except we molested it a bit with sin. */


/* Pre-calculated values to help change the state of the plasma. */
static CpLissaJousPreCalcs g_ljpPlasmaLissajousPreCalcs[CPMAXNUMPLASMALJS];

/* State of the plasma */
static CpLissajousState g_ljsaPlasmaLissajousStates[CPMAXNUMPLASMALJS];
static float g_nRoll;

static uint8_t g_nCurrentIntensity;	/* The current intensity of the plasma */

/* Used by the intensity rise time */
static float g_nPlasmaRiseTimeTicksPerIntensityLevel;	/* Number of application-ticks required for the current plasma-intensity to change by one discreet level. */
float g_nCurrentIntensityError;	/* g_nCurrentIntensity is rounded to te nearest integer, so g_nCurrentIntensityError represents the current error, and once it reaches g_nPlasmaRiseTimeTicksPerIntensityLevel, g_nCurrentIntensity is updated. */




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                                   Helpers

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


uint8_t *
cpGetPlasmaBuffer(void)
{
	return g_caPlasmaBuffer;
}


/* The function below is just for debugging purpouses so we can display the plasma-tabs */
uint8_t *
cpGetPlasmaTab(int nTab)
{
	switch(nTab)
	{
		case 1: return g_caPlasmaTab1;
		case 2: return g_caPlasmaTab2;
		default: return NULL;
	}
}



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                              Pre Calculations

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


/* Calculate table 1 for plasma (distance from centre) */
static AeBool
_cpPreCalcPlasmaTab1(float nCircleSeed)
{
	/* Parts of this code taken from code by Jan Moeller & Erik Hansen. */
	
	int i=0,j=0;
	uint8_t *cpPlasmaTabOffset;

	float nCircleSeedMultiplier = 1.0f/nCircleSeed;


	/*
	The plasma tabs correspond to a width and height of two 'windows' meaning the area is 4 times that of the window.
	This is so that as long as the amplitude of the X frequency is <= 1 (0.5 screen-width) and the amplitude of the Y frequency is <= 1 (0.5 screen-height), the tabs will never run off the edge of the window.
	*/
	// ?: if(G_cps.bConfineLissajoussesToSquare), can we get away with smaller plasma-tabs (but not a smaller plasma-body) ?
	if(!(g_caPlasmaTab1 = (uint8_t *) malloc(G_nPlasmaBoxW*G_nPlasmaBoxH*4 * sizeof(uint8_t)) ))
	{
		return FALSE;
	}

	cpPlasmaTabOffset = g_caPlasmaTab1;



	while(i<G_nPlasmaBoxH<<1)
	{
		j=0;
		while(j<G_nPlasmaBoxW<<1)
		{
			/* Work out distance from center */
			/* We're tweaking this by working out the distance to the point 4 units above the plane and subtracting 4 from the result. */
			/* We're throwing away upper byte of distance to make make value wrap from 0-255 */
			*cpPlasmaTabOffset++ = (uint8_t)(( sqrt((4.0f*4.0f)+(G_nPlasmaBoxH-i)*(G_nPlasmaBoxH-i)+(G_nPlasmaBoxW-j)*(G_nPlasmaBoxW-j))-4.0f) * nCircleSeedMultiplier );
			j++;
		}
		i++;
	}

	return TRUE;
}


/* calculate table 2 for plasma (similar to table1 but molested with sin) */
static AeBool
_cpPreCalcPlasmaTab2(float nCircleSeed)
{
	/* Parts of this code taken from code by Jan Moeller & Erik Hansen. */
	
	int i=0,j=0;
	double nTemp;
	uint8_t *cpPlasmaTabOffset;

	float nCircleSeedMultiplier = 1.0f/nCircleSeed;

	
	if(!(g_caPlasmaTab2 = (uint8_t *) malloc(G_nPlasmaBoxW*G_nPlasmaBoxH*4 * sizeof(uint8_t)) ))
	{
		return FALSE;
	}

	cpPlasmaTabOffset = g_caPlasmaTab2;


	while(i<G_nPlasmaBoxH<<1)
	{
		j=0;
		while(j<G_nPlasmaBoxW<<1)
		{
			nTemp=sqrt((4.0f*4.0f)+(G_nPlasmaBoxH-i)*(G_nPlasmaBoxH-i)+(G_nPlasmaBoxW-j)*(G_nPlasmaBoxW-j))-4.0f;
			*cpPlasmaTabOffset++ = (uint8_t)((sin(nTemp*nCircleSeedMultiplier)+1.0f)*90.0f);
//			*cpPlasmaTabOffset++ = (sin(sqrt((G_nPlasmaBoxH-i)*(G_nPlasmaBoxH-i)+(G_nPlasmaBoxW-j)*(G_nPlasmaBoxW-j))*nCircleSeedMultiplier)+1.0f)*90.0f;
			j++;
		}
		i++;
	}

	return TRUE;
}



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                           Setting up and cleaning

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


// \sa cpTiniPlasma
AeBool
cpInitPlasma(int nWidth, int nHeight)
{
	unsigned int nI;
	float nZoom;

	/* Plasma box half size of image. +1 to round up instead of round down. We round up so that if there's an odd Y dimension, there will never be a shortage of plasma-rows */
	G_nPlasmaBoxW = (nWidth+1)>>1;
	G_nPlasmaBoxH = (nHeight+1)>>1;

	/* This is so that nZoom is measured in pixels, and G_cps.nPlasmaGlobalSeedMultiplier is measured in units of screen-height rather than pixels. That way, we get the same effect as we would with 320x200 regardless of the resolution (hence 100==200/2) */
	nZoom = G_cps.nPlasmaGlobalSeedMultiplier * ((float)G_nPlasmaBoxH/100.0f);

	/* Buffer for rendering the plasma body. */

	if(!(g_caPlasmaBuffer = (uint8_t *) malloc(G_nPlasmaBoxW*G_nPlasmaBoxH * sizeof(uint8_t)) ))
	{
		TRACE("Could not allocate memory for the plasma buffer\n");
		return FALSE;
	}

	/* Buffers for the Plasma tab(le)s */

	if(!(_cpPreCalcPlasmaTab1(G_cps.nPlasmaSeed1 * nZoom) ))
	{
		TRACE("Could not create plasma tab #1\n");
		return FALSE;
	}

	if(!(_cpPreCalcPlasmaTab2(G_cps.nPlasmaSeed2 * nZoom) ))
	{
		TRACE("Could not create plasma tab #2\n");
		return FALSE;
	}


	/* Initialise the plasma state */

	for(nI=0;nI<CPMAXNUMPLASMALJS;nI++)
	{
		/* Set current phase to initial phase-angles */
		/* Because the settings are in degrees (more human-readable) and we want the state to be in radians (faster to compute), convert phase-angles to radians */
		g_ljsaPlasmaLissajousStates[nI].nXPhase = CPDEG2RAD(G_cps.lsetPlasmaLissajousSettings[nI].nXInitialPhase);
		g_ljsaPlasmaLissajousStates[nI].nYPhase = CPDEG2RAD(G_cps.lsetPlasmaLissajousSettings[nI].nYInitialPhase);

	
		/* Work out the frequencies in the correct units */
		/* Because the settings are in cycles per second (more human-readable) and each tick we want to add the number of radians per 1/CPREFERENCEREFRESHRATE second, we must convert between the two units. */
		g_ljpPlasmaLissajousPreCalcs[nI].nXFreq = (G_cps.lsetPlasmaLissajousSettings[nI].nXFreq * (2.0f*pi)) / CPREFERENCEREFRESHRATE;
		g_ljpPlasmaLissajousPreCalcs[nI].nYFreq = (G_cps.lsetPlasmaLissajousSettings[nI].nYFreq * (2.0f*pi)) / CPREFERENCEREFRESHRATE;	
	}

	g_nRoll=0.0f;	// Note: If setting to owt else, remember to fmod() this with 256.0f



	return TRUE;
}



// \sa cpInitPlasma
void
cpTiniPlasma(void)
{
	safefree(g_caPlasmaTab1);
	safefree(g_caPlasmaTab2);
	safefree(g_caPlasmaBuffer);
}



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                   Calculations while program is running

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


void
cpResetPlasmaRiseTimeTicks(int nRefreshRate)
{
	if(G_cps.cIntensity!=0 && G_cps.nIntensityRiseTime!=0.0f)
	{
		g_nPlasmaRiseTimeTicksPerIntensityLevel = (float)nRefreshRate * G_cps.nIntensityRiseTime/(float)G_cps.cIntensity;
	}
	// else the intensity remains constant so we don't care what g_nPlasmaRiseTimeTicksPerIntensityLevel ends up as.


	/* Init the intensity */
	if(G_cps.nIntensityRiseTime==0.0f || G_cps.bFullIntensityHueWrap)
	{
		g_nCurrentIntensity = G_cps.cIntensity;
	}
	else
	{
		g_nCurrentIntensity = 0;
	}
	g_nCurrentIntensityError = 0.0f;


	G_bIntensityHasChanged = TRUE;	/* Make sure the palette is updated on the first iteration so it does not have to be set to it's initiial state beforehand. */

}



void
cpUpdateCurrentPlasmaIntensity(float nTicks)
{
	if(g_nCurrentIntensity<G_cps.cIntensity)
	{
		g_nCurrentIntensityError += nTicks;
		
		if(g_nCurrentIntensityError>=g_nPlasmaRiseTimeTicksPerIntensityLevel)
		{
			/* If the error in g_nCurrentIntensity is greater than the number of ticks per intensity level, keep increasing g_nCurrentIntensity until the error is less than that required for an intensity-change. */
			// TODO: Use maths instead of a loop. Rationale: if we call this with an extrelely large # of ticks, can take a while.
			do
			{
				g_nCurrentIntensity++;
				g_nCurrentIntensityError -= g_nPlasmaRiseTimeTicksPerIntensityLevel;

				if(g_nCurrentIntensity==G_cps.cIntensity)
				{
					// As g_nCurrentIntensity is a uint8_t, it could overflow if we do a (g_nCurrentIntensity>=G_cps.cIntensity) outside the loop.
					break;
				}
			}
			while(g_nCurrentIntensityError>=g_nPlasmaRiseTimeTicksPerIntensityLevel);

			G_bIntensityHasChanged = TRUE;
		}
	}
}



uint8_t
cpGetCurrentPlasmaIntensity(void)
{
	return g_nCurrentIntensity;
}



/* Just used to update the plasma params. No heavy calculations here */
// logic
void
cpCalculatePlasmaParams(float nPlasMovementDelta)
{
	/* This moves the four windows to the plasma-tabs */

	unsigned int nI;


	for(nI=0;nI<CPMAXNUMPLASMALJS;nI++)
	{
		g_ljsaPlasmaLissajousStates[nI].nXPhase += g_ljpPlasmaLissajousPreCalcs[nI].nXFreq * nPlasMovementDelta;
		g_ljsaPlasmaLissajousStates[nI].nYPhase += g_ljpPlasmaLissajousPreCalcs[nI].nYFreq * nPlasMovementDelta;
		// ?: Do we need to bother fmod'ding these with 2*pi ?
	}

	g_nRoll += G_cps.nPlasmaRollSpeed*nPlasMovementDelta;
	g_nRoll = (float)fmod(g_nRoll, 256.0f);
}



/* This is where the heavy plasma calculating stuff is done. It works out what the plasma body currently looks like */
// ?: Do we need to pass in the intensity? Could we just use g_nCurrentIntensity instead? (if using a palettte, bUseIntensityAndBounce will be false anyway)
// render (pre-calc)
void
cpCalculatePlasmaBody(AeBool bUseIntensityAndBounce, uint8_t cIntensity)
{
	/* Parts of this code taken from code by Jan Moeller & Erik Hansen. */

	/* The Plasma effect is created by taking four moving windows into the plasma-tabs
	 * and adding the values of the positions in the tabs relative to the windows.
	 * cRoll is a constant added to everything.
	 * For static images, if cRoll is incremented, this creates a simulated colour-cycle.
	 */


	unsigned int nI;




	float nXRangeMult, nYRangeMult;	/* Used to multiply the current lissajous position so it coveres the range of the target area. */
	int nXLjOffset, nYLjOffset;
	int naX[CPMAXNUMPLASMALJS], naY[CPMAXNUMPLASMALJS];	/* Used to pre-calc plasma tab offsets from top-left.  */









	uint8_t cRoll = (uint8_t)g_nRoll;	// ?: Do we need a separate local var cRoll? It is handy to pre-convert it to uint8_t. Perhaps try changing it during calculation.

	int nSourceRowCounter,nColumn;
	uint8_t *cpSourcePos1, *cpSourcePos2, *cpSourcePos3, *cpSourcePos4;
	uint8_t *cpDestPos = g_caPlasmaBuffer;




	/* Pre-calc the positions of the plasma tabs */

	// TODO: pre-calc this.
	if(G_cps.bConfineLissajoussesToSquare)
	{
		/* The lissajousses cover a square in the centre who'se side is the smallest dimension. */
		/* Make an amplitude of 1.0 the same in pixels in both dimensions. Use the smaller of the sizes as amplitude 1.0 to prevent copying off the edge of a plasma tab.  */
		nXRangeMult = nYRangeMult = MIN(G_nPlasmaBoxW, G_nPlasmaBoxH)>>1;
		if(G_nPlasmaBoxW>G_nPlasmaBoxH)
		{
			nXLjOffset = (G_nPlasmaBoxW-G_nPlasmaBoxH)>>1;
			nYLjOffset = 0;
		}
		else
		{
			nXLjOffset = 0;
			nYLjOffset = (G_nPlasmaBoxH-G_nPlasmaBoxW)>>1;
		}
	}
	else
	{
		/* The lissajousses cover the entire plasma area */
		/* Note: Amplitude of 1.0 is a screen-width in the X dimension and a screen-height in the Y dimension. */
		nXRangeMult = G_nPlasmaBoxW>>1;
		nYRangeMult = G_nPlasmaBoxH>>1;
		nXLjOffset = nYLjOffset = 0;	/* As this covers the entire range, no offset is needed */
	}


	for(nI=0;nI<CPMAXNUMPLASMALJS;nI++)
	{
		naX[nI]=(int)((1+sin(g_ljsaPlasmaLissajousStates[nI].nXPhase)*G_cps.lsetPlasmaLissajousSettings[nI].nXAmp)*nXRangeMult)+nXLjOffset;
		naY[nI]=(int)((1+sin(g_ljsaPlasmaLissajousStates[nI].nYPhase)*G_cps.lsetPlasmaLissajousSettings[nI].nYAmp)*nYRangeMult)+nYLjOffset;
	}





	/* Build the plasma */



	/* Setup the positions in the source-buffers to start reading from. Each buffer is twice as wide as the destination. */
	// Test: see what happens when g_caPlasmaTab1 swapped with g_caPlasmaTab2
	cpSourcePos1 = &g_caPlasmaTab1[(G_nPlasmaBoxW<<1)*naY[0] + naX[0]];
	cpSourcePos2 = &g_caPlasmaTab2[(G_nPlasmaBoxW<<1)*naY[1] + naX[1]];
	cpSourcePos3 = &g_caPlasmaTab2[(G_nPlasmaBoxW<<1)*naY[2] + naX[2]];
	cpSourcePos4 = &g_caPlasmaTab2[(G_nPlasmaBoxW<<1)*naY[3] + naX[3]];

	nSourceRowCounter=G_nPlasmaBoxH;

	// OPTIMIZATION idea: If intensity = 0x7F, we can not only get away by not multiplying cVal by the intensity, but we can add 128 instead of the mormalization value (or not add anything at all if(G_cps.isImgSource == cpIMGTYPE_NONE) ).

	if(bUseIntensityAndBounce)
	{
		/* Intensity and bounce mode.
		   The resulting plasma-pixel is modified so it bounces on the middle value (sawtooth maps to triangular), and subtracted so it's centre value is 0 (normalised).
		   This is so that the values wrap round without any discontinuity,
		   and so we can have a plasma-pixel centred around 0 which means that when interpreted as a signed value, it can both increase and decrease the stimulus it's applied to. */
		// Note: If using no image and just the plasma, intensity and bounce mode can also be emulated by using a palette or colour lookup-table.


		uint8_t nNormalizationVal;	/* Used to normalise value for when output being interpreted as a signed number so it's centred around 0. Otherwise we will get discontinuity from 127 to -128. */


		nNormalizationVal = -(cIntensity>>1);
		if(G_cps.isImgSource == cpIMGTYPE_NONE)
		{
			/* If we're drawing the plasma straight to the screen, we want it centred around 0x7F instead of 0x00 */
			 nNormalizationVal += 0x7F;
		}


		while(nSourceRowCounter--)
		{
			nColumn=G_nPlasmaBoxW;
			while(nColumn--)
			{
				/* This is the heart of the plasma.
				 * The tables are added together, and a wrap-around effect is created by discarding the uppermost byte.
				 */

				// ?: Cound we just apply this straight to the output without storing it in a buffer?
				// A: But if we're doing 1x1 plasma-pixels to 2x2 screen-pixels, then best to do here.

				uint8_t cTemp =
					*cpSourcePos1++
					+*cpSourcePos2++
					+*cpSourcePos3++
					+*cpSourcePos4++
					+cRoll;


				/* Plasma pixel post-processing (best done here as a plasma pixel is 2x2 screenpixels) */

				/* Get the values to 'bounce'.
				* We need to 'bounce' the plasma if we're not using a palette so as transitions from 255 to 0 are smooth
				* We do this by mapping 0 to a dark-colour, that gets lighter until 127, and then gets darker again until 255
				* This has the drawback that we only get 7 bits of greyscale.
				*/
				cTemp = (cTemp<0x80 ? cTemp<<1 : 0xFF-((cTemp & 0x7F)<<1));	/* cTemp & 0x7F subtracts 0x7F from cTemp */ // Bear in mind that as we're only dealing with 8-bit values, we cannot get away with doing 0x1FF-(nI<<1) (unless we temporarily cast them to uints ... in fact, we could try this optimization...)
				cTemp = cpMUL8(cTemp,cIntensity);
				cTemp += nNormalizationVal;
				// ?: instead of this line, could we use a "us*us=s" multiply-table in previous line if using a multiply table?

				/* That's it. Let's write our result */
				*cpDestPos++ = cTemp;
			}

			/* Advance position in source buffers to start of next source line */
			cpSourcePos1+=G_nPlasmaBoxW;
			cpSourcePos2+=G_nPlasmaBoxW;
			cpSourcePos3+=G_nPlasmaBoxW;
			cpSourcePos4+=G_nPlasmaBoxW;
		}
	}
	else
	{
		/* Pure plasma-mode.
		   The plasma straight from the plasma-tabs. */

		while(nSourceRowCounter--)
		{
			nColumn=G_nPlasmaBoxW;
			while(nColumn--)
			{
				/* This is the heart of the plasma.
				 * The tables are added together, and a wrap-around effect is created by discarding the uppermost byte.
				 */

				// ?: Cound we just apply this straight to the output without storing it in a buffer?
				// A: But if we're doing 1x1 plasma-pixels to 2x2 screen-pixels, then best to do here.

				*cpDestPos++ =
					*cpSourcePos1++
					+*cpSourcePos2++
					+*cpSourcePos3++
					+*cpSourcePos4++
					+cRoll;	
			}

			/* Advance position in source buffers to start of next source line */
			cpSourcePos1+=G_nPlasmaBoxW;
			cpSourcePos2+=G_nPlasmaBoxW;
			cpSourcePos3+=G_nPlasmaBoxW;
			cpSourcePos4+=G_nPlasmaBoxW;
		}
	}

}
