/***************************************************************************
 *
 * cpimcosp.c
 * ChromaPlas image-colour-space code
 *
 * By Andrei Ellman
 *
 **************************************************************************/



/****************************************************************************
 Includes
 */

#include <stdlib.h>
#include <allegro.h>


#include "aeglobal.h"

#include "cpglobal.h"

#include "aecolspc.h"

#include "cpsttngs.h"

#include "cpimcosp.h"


/****************************************************************************
 Local Types
 */


/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */


/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */

CpImgColSp G_icsImgColSpc;	/* Used to store pre-calculated colour-space values for the source image. */


/****************************************************************************
 Local (static) Global Variables
 */



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                          Various ancillary functions.

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */




/* CpImgColSp functions */


void
cpImgColSpInit(CpImgColSp *icspThis)
{
	icspThis->naImgColSpChannels = NULL;
	icspThis->nW = 0;
	icspThis->nH = 0;
}


AeBool
cpImgColSpIsUsed(CpImgColSp *icspThis)
{
	return (icspThis->naImgColSpChannels!=NULL);
}


AeBool
cpImgColSpAlloc(CpImgColSp *icspThis, int nW, int nH)
{
	ASSERT(!cpImgColSpIsUsed(icspThis));

	/* Allocate buffer to store the pre-calculated colourspace values for each pixel */
	icspThis->naImgColSpChannels = (uint32_t *) malloc(nW*nH * sizeof(uint32_t));
	if(!(icspThis->naImgColSpChannels))
	{
		return FALSE;
	}

	icspThis->nW = nW;
	icspThis->nH = nH;

	return TRUE;
}


void
cpImgColSpSafeFree(CpImgColSp *icspThis)
{
	if(cpImgColSpIsUsed(icspThis))
	{
		free(icspThis->naImgColSpChannels);

		cpImgColSpInit(icspThis);
	}
}





/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                               Pre Calculations

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */



/* Pre-calc the colourspace values of the source image which remains konstant */
AeBool
cpPreCalcImgColSpcVals(CpImgColSp *icspColSpcDest, BITMAP *bmpSourceImage, PALETTE pSourcePicPal)
{
	uint32_t nPixelColour;	// uint16_t for 15,16 bits, uint8_t for 8. 24-bits = ???!

	int16_t n16Stim1;
	uint8_t c8Stim2, c8Stim3;

	int x,y;	/* Loop counters */


	uint32_t *npImgColSpChannelsBufPos;


	CpColourSpaceToFunction	*fpColourSpaceFunction;


	/* Which colourspace are we going to use? */
	switch(G_cps.csColourSpaceToUse)
	{
		case cpCOLOURSPACE_RGB:	fpColourSpaceFunction = cpTurboEncRGB;	break;
		case cpCOLOURSPACE_HSV:	fpColourSpaceFunction = cpTurboRGB2HSV;	break;
		case cpCOLOURSPACE_HLS:	fpColourSpaceFunction = cpTurboRGB2HLS;	break;

		default:
		{
			TRACE("Invalid colour-space.\n");
			fpColourSpaceFunction = NULL;
			ASSERT(FALSE);
		}
	}




	// TODO: If we decide to use whole screen rather than just image, just cpImgColSpAlloc padding round the unused screen-parts, and take this padding into account in the code below (add offsets to Y and X).


	/* Allocate buffer to store the pre-calculated colourspace values for each pixel */
	if(!cpImgColSpAlloc(icspColSpcDest, bmpSourceImage->w, bmpSourceImage->h))
	{
		TRACE("EEK! Could not create the colour-space data-structure.\n");
		return FALSE;
	}




	npImgColSpChannelsBufPos = icspColSpcDest->naImgColSpChannels;

	// ?: Should we only do the switch for the X-loop to prevent too much repetition?
	// A: Beware of the type we define npSrcBmpLineAdr to be (but of course could do some casting, but beware of adding to the pointer).

	switch(bitmap_color_depth(bmpSourceImage))
	{
		case 32:
		{
			for(y=0;y<icspColSpcDest->nH;y++)
			{
				uint32_t *npSrcBmpLineAdr;



				/* If we want to escape while doing the pre-calcs, then escape. */
				/* Only do this once per line to prevent too much slowdown */
				if(cpCanExitApp())
				{
					break;
					// ?: What if we return FALSE? A: The app would think owt went wrong.
					// If we can return error-codes with severity, return one with Severity_OK.
				}



				// If we're using a memory-bitmap, is it really nesc. to do this bmpSourceImage->line[y] thing?
				npSrcBmpLineAdr = (uint32_t *)bmpSourceImage->line[y];

				x=icspColSpcDest->nW;

				while(x--)
				{
					nPixelColour = *npSrcBmpLineAdr++;
					fpColourSpaceFunction((uint8_t)getr32(nPixelColour), (uint8_t)getg32(nPixelColour), (uint8_t)getb32(nPixelColour), &n16Stim1, &c8Stim2, &c8Stim3);
					// Note: As things like _rgb_g_shift_32 etc remain unchanged, perhaps we could use a line such as the one below instead (and use an if() to determine if RGB or BGR).
					//fpColourSpaceFunction((uint8_t)(nPixelColour>>16), (uint8_t)((nPixelColour>>8)&0x000000ff), (uint8_t)(nPixelColour&0x000000ff), &n16Stim1, &c8Stim2, &c8Stim3);	// Faster but has some RGB/BGR-ness issues.

					*npImgColSpChannelsBufPos++ = (n16Stim1<<16)+(c8Stim2<<8)+c8Stim3;
				}
			}
		}
		break;

		case 24:
		{
			for(y=0;y<icspColSpcDest->nH;y++)
			{
				uint8_t *npSrcBmpLineAdr;




				/* If we want to escape while doing the pre-calcs, then escape. */
				/* Only do this once per line to prevent too much slowdown */
				if(cpCanExitApp())
				{
					break;
					// ?: What if we return FALSE? A: The app would think owt went wrong.
					// If we can return error-codes with severity, return one with Severity_OK.
				}




				// If we're using a memory-bitmap, is it really nesc. to do this bmpSourceImage->line[y] thing?
				npSrcBmpLineAdr = (uint8_t *)bmpSourceImage->line[y];

				x=icspColSpcDest->nW;

				while(x--)
				{
					uint8_t nPixelColourR, nPixelColourG, nPixelColourB;

					// TODO: Do this if() outside the loops.
					// TODO: Confirm that this is correct on big-endian machines. Also, do I need to do owt for 32-bit colour on big-endian machines?
					if(_rgb_b_shift_24==0)
					{
						// BGR
						// ?: If _rgb_b_shift_24 = 0, does that always mean _rgb_g_shift_24 = 8 and _rgb_r_shift_24 = 16?
						nPixelColourB = *npSrcBmpLineAdr++;
						nPixelColourG = *npSrcBmpLineAdr++;
						nPixelColourR = *npSrcBmpLineAdr++;
					}
					else
					{
						//RGB
						// ?: Does this always mean that If _rgb_b_shift_24 = 16, _rgb_g_shift_24 = 16 and _rgb_r_shift_24 = 0?
						nPixelColourR = *npSrcBmpLineAdr++;
						nPixelColourG = *npSrcBmpLineAdr++;
						nPixelColourB = *npSrcBmpLineAdr++;
					}


					fpColourSpaceFunction(nPixelColourR, nPixelColourG, nPixelColourB, &n16Stim1, &c8Stim2, &c8Stim3);

					*npImgColSpChannelsBufPos++ = (n16Stim1<<16)+(c8Stim2<<8)+c8Stim3;

					//TRACE("_rgb_r_shift_24=%d, _rgb_g_shift_24=%d, _rgb_b_shift_24=%d\n", _rgb_r_shift_24, _rgb_g_shift_24, _rgb_b_shift_24);
					//TRACE("R=%X, G=%X, B=%X\n", nPixelColourR, nPixelColourG, nPixelColourB);
					//TRACE("Stim1=%X, Stim2=%X, Stim3=%X\n", n16Stim1, c8Stim2, c8Stim3);
				}
			}
		}
		break;

		case 16:
		{
			// TODO: Test this.

			for(y=0;y<icspColSpcDest->nH;y++)
			{
				uint16_t *npSrcBmpLineAdr;



				/* If we want to escape while doing the pre-calcs, then escape. */
				/* Only do this once per line to prevent too much slowdown */
				if(cpCanExitApp())
				{
					break;
					// ?: What if we return FALSE? A: The app would think owt went wrong.
					// If we can return error-codes with severity, return one with Severity_OK.
				}



				// If we're using a memory-bitmap, is it really nesc. to do this bmpSourceImage->line[y] thing?
				npSrcBmpLineAdr = (uint16_t *)bmpSourceImage->line[y];

				x=icspColSpcDest->nW;

				while(x--)
				{
					nPixelColour = *npSrcBmpLineAdr++;
					// NOTE: getr16(), getg16(), getb16() make use of _rgb_r_shift_15 so when decoding in _cpRenderImage(), might want to make use of them too.
					fpColourSpaceFunction((uint8_t)getr16(nPixelColour), (uint8_t)getg16(nPixelColour), (uint8_t)getb16(nPixelColour), &n16Stim1, &c8Stim2, &c8Stim3);
					//fpColourSpaceFunction((uint8_t)_rgb_scale_5[(nPixelColour >> 11) & 0x1F], (uint8_t)_rgb_scale_6[(nPixelColour >> 5) & 0x3F], (uint8_t)_rgb_scale_5[(nPixelColour >> 0) & 0x1F], &n16Stim1, &c8Stim2, &c8Stim3);	// Faster but has some RGB/BGR-ness issues.

					// ?: Seeing that we're only using as few as 5 bits per colour-component, could we get away with using 8 bits for hue instead of 11? Ditto for reduced # of bits with S and V. Perhaps even pack into 16 bits.
					*npImgColSpChannelsBufPos++ = (n16Stim1<<16)+(c8Stim2<<8)+c8Stim3;
				}
			}
		}
		break;

		case 15:
		{
			for(y=0;y<icspColSpcDest->nH;y++)
			{
				uint16_t *npSrcBmpLineAdr;



				/* If we want to escape while doing the pre-calcs, then escape. */
				/* Only do this once per line to prevent too much slowdown */
				if(cpCanExitApp())
				{
					break;
					// ?: What if we return FALSE? A: The app would think owt went wrong.
					// If we can return error-codes with severity, return one with Severity_OK.
				}



				// If we're using a memory-bitmap, is it really nesc. to do this bmpSourceImage->line[y] thing?
				npSrcBmpLineAdr = (uint16_t *)bmpSourceImage->line[y];

				x=icspColSpcDest->nW;

				while(x--)
				{
					nPixelColour = *npSrcBmpLineAdr++;
					// NOTE: getr15(), getg15(), getb15() make use of _rgb_r_shift_15 so when decoding in _cpRenderImage(), might want to make use of them too.
					fpColourSpaceFunction((uint8_t)getr15(nPixelColour), (uint8_t)getg15(nPixelColour), (uint8_t)getb15(nPixelColour), &n16Stim1, &c8Stim2, &c8Stim3);
					//fpColourSpaceFunction((uint8_t)_rgb_scale_5[(nPixelColour >> 10) & 0x1F], (uint8_t)_rgb_scale_5[(nPixelColour >> 5) & 0x1F], (uint8_t)_rgb_scale_5[(nPixelColour >> 0) & 0x1F], &n16Stim1, &c8Stim2, &c8Stim3);	// Faster but has some RGB/BGR-ness issues.

					// ?: Seeing that we're only using as few as 5 bits per colour-component, could we get away with using 8 bits for hue instead of 11? Ditto for reduced # of bits with S and V. Perhaps even pack into 16 bits.
					*npImgColSpChannelsBufPos++ = (n16Stim1<<16)+(c8Stim2<<8)+c8Stim3;
				}
			}
		}
		break;

		case 8:
		{
			if(bitmap_color_depth(bmpSourceImage)==8)
			{
				// ?: Should we make sure pSourcePicPal is set to RGB332 if no source pic or explicit palette?
				select_palette(pSourcePicPal);	/* Converting from indexed to a higher depth */
			}

			for(y=0;y<icspColSpcDest->nH;y++)
			{
				uint8_t *npSrcBmpLineAdr;



				/* If we want to escape while doing the pre-calcs, then escape. */
				/* Only do this once per line to prevent too much slowdown */
				if(cpCanExitApp())
				{
					break;
					// ?: What if we return FALSE? A: The app would think owt went wrong.
					// If we can return error-codes with severity, return one with Severity_OK.
				}



				// If we're using a memory-bitmap, is it really nesc. to do this bmpSourceImage->line[y] thing?
				npSrcBmpLineAdr = (uint8_t *)bmpSourceImage->line[y];

				x=icspColSpcDest->nW;

				while(x--)
				{
					// If using a 332 palette, how about using the 3/2-bit RGB values instead of get_color()?
					// BEWARE: Allegro's generate_332_palette() generates 'magic pink' for colour 0 and sets colour 254 to be black. Solution: Make the palette ourselves.
					// TODO: Use a 256-entry HSV/HLS[/RGB] lookup table?

					uint8_t nPalIndex;
					RGB rgb;

					nPalIndex = *npSrcBmpLineAdr++;
					get_color(nPalIndex, &rgb);
					fpColourSpaceFunction((uint8_t)(rgb.r<<2), (uint8_t)(rgb.g<<2), (uint8_t)(rgb.b<<2), &n16Stim1, &c8Stim2, &c8Stim3);

					*npImgColSpChannelsBufPos++ = (n16Stim1<<16)+(c8Stim2<<8)+c8Stim3;
				}
			}

			if(bitmap_color_depth(bmpSourceImage)==8)
			{
				unselect_palette();	/* Converting from indexed to a higher depth */
			}
		}
		break;

		default:
		{
			ASSERT(FALSE);
		}

	}

	return TRUE;
}


