/***************************************************************************
 *
 * cpgfx.c
 * ChromaPlas graphics-code
 *
 * By Andrei Ellman
 *
 **************************************************************************/



/****************************************************************************
 Includes
 */

#include <stdlib.h>
#include <allegro.h>
#include <allegro/internal/aintern.h>	/* Used to get _wait_for_vsync */

#ifndef CP_NO_JPEG
#include <jpgalleg.h>
#endif
#ifndef CP_NO_PNG
#include <loadpng.h>
#endif



#include "aeglobal.h"

#include "cpglobal.h"

#include "cpmthhlp.h"	// used in cpCreateMonoPaletteWithBounceAndIntensity

#include "cpsttngs.h"

#include "cpfunpal.h"

#include "cpimcosp.h"

#include "cpbltiim.h"

#include "cprender.h" // Used for cpSafeModeErrorDisplay()

#ifdef ALLEGRO_WINDOWS

#include "cpwin_.h"

#endif

#include "cpgfx.h"



/****************************************************************************
 Local Types
 */


/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */

#define CP_GFX_MAXNUMPAGES	3	/* 3 for triple buffering */

/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */


int G_nRefreshRate;
AeBool G_bActualRefreshRateIsUnknown;

CpRenderingType G_rtRenderingType;	/* Are we double-buffering, page-flipping, or rendering straight to the screen? */


BITMAP *G_bmpDest;	/* Bitmap we're drawing onto (screen, memory-buffer, page-flip buffer, whatever) */

PALETTE G_pPal;	// in 8-bit, stores state of palette. In 32-bit, stores 256-entry colour-mapping table. Values ae in the number of bits per component for each mode.
AeBool G_bPalIsMucky;	/* In 8-bit paletted modes, this tells the graphics engine that the palette needs to be changed. This is ignored in truecolour modes. */

AeBool G_bApplyingColourLookupTableDirectlyToPlasmaPixels;


/****************************************************************************
 Local (static) Global Variables
 */


/* The two below used when page-flipping */
static BITMAP *g_bmpaPages[CP_GFX_MAXNUMPAGES];
static unsigned int g_nPage;	/* When page-flipping or triple-buffering, which page is currently being drawn on - the active page (this is not the page being displayed) */


/*  Allowed BPPs (on Allegro Platforms) */
// TODO: A platform-specific version? Probably not nesc as it varies according to hardware configuration anyway - ie, if a mode is in this list, it may not be available if the hardware does not support it.
static int g_naAllowedBPPs[] =
{
	/* Ordered so most desirable BPPs first */
	// ?: Should 15 come before 16? ADV: in monochrome modes, 15 is more natural than 16 coz in 15, nR==nG==nB DISADV: 16 is probably better suited to photos than 15 bit anyways.
	32, 24, 16, 15, 8
};



static BITMAP *g_bmpSourcePic;	/* Bitmap that contains the image to use if we're using an external image. */
static PALETTE g_pSourcePicPal;	/* The palette to be used by the source-pic in 256-colour mode. */

static AeBool g_bRGBMapAllocdByChromaPlas;	/* Has the RGB-map used when converting to 8-bit been allocated by ChromaPlas itself? */



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                          Various ancillary functions.

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


/* Maps the enumerated colour-space value to a colour-space index */
int
cpMapCpColourSpaceToInt(CpColourSpace csColourSpace)
{
	switch(csColourSpace)
	{
		case cpCOLOURSPACE_RGB:
		{
			return CPCOLOURSPACEINDEX_RGB;
		}

		case cpCOLOURSPACE_HSV:
		{
			return CPCOLOURSPACEINDEX_HSV;
		}

		case cpCOLOURSPACE_HLS:
		{
			return CPCOLOURSPACEINDEX_HLS;
		}

		default:
		{
			return CPCOLOURSPACEINDEX_NONE;	/* Just in case csColourSpace is invalid (shouldn't happen) */
		}
	}

}



/* Gets an allowed BPP from the list of BPPs that are allowed - returns -1 if out of range */
int
cpGetAllowedBPP(int nIndex)
{
	if(nIndex<0 || nIndex>=(int)AEGETNUMELEMENTSINSTATICARRAY(g_naAllowedBPPs))
	{
		return -1;
	}

	return g_naAllowedBPPs[nIndex];
}


/* Finds out if the current BPP is allowed */
AeBool
cpBPPIsAllowed(int nBPPToTest)
{
	unsigned int nI;

	if((nBPPToTest==24 && G_cps.bDisable24bpp) || (nBPPToTest==8 && G_cps.bDisable8bpp))
	{
		return FALSE;	/* Prevent unwanted BPPs from being selected. */
	}

	for(nI=0;nI<AEGETNUMELEMENTSINSTATICARRAY(g_naAllowedBPPs);nI++)
	{
		if(nBPPToTest==g_naAllowedBPPs[nI])
		{
			return TRUE;
		}
	}

	return FALSE;
}




/* Lword-aligns a number to next lword-aligned number, unless it's greater than the max, in which case, it's rounded down to the previous */
static void
_cpRoundUpToLwordUnlessBeyondMax(int *npNumberToRound, int nMax)
{
	*npNumberToRound = (*npNumberToRound+3)&0xFFFFFFFC;
	if(*npNumberToRound>nMax)
	{
		*npNumberToRound-=4;
	}
}

/* Used to find the offsets required for centering the prune-window when pruning images to screen-size */
// \sa _cpGetOffsetsForCenteringImageOnScreen()
static void
_cpGetOffsetsForBlittingFromCentreOfImage(int *npXOffset, int *npYOffset, int nSourceWidth, int nSourceHeight, int nDestWidth, int nDestHeight)
{
	*npXOffset = (nSourceWidth > nDestWidth ? (nSourceWidth - nDestWidth)/2 : 0);
	*npYOffset = (nSourceHeight > nDestHeight ? (nSourceHeight- nDestHeight)/2 : 0);
}



int
cpGetXOffsetForCenteringImageOnScreen(int nSourceWidth, int nDestWidth)
{
	return (nSourceWidth < nDestWidth ? (nDestWidth-nSourceWidth)/2 : 0) & 0xFFFFFFFC; /* lword-align it (to make life easier when telling plasma to move to next plasma-pixel and so that in 8-bit mode, we're always on the same position in the lword) */
}


int
cpGetYOffsetForCenteringImageOnScreen(int nSourceHeight, int nDestHeight)
{
	return (nSourceHeight < nDestHeight ? (nDestHeight-nSourceHeight)/2 : 0);
}


/* Works out the offsets needed to centre the image on screen if one or more of it's dimensions is smaller than the screen-size */
// \sa _cpGetOffsetsForBlittingFromCentreOfImage()
void
cpGetOffsetsForCenteringImageOnScreen(int *npXOffset, int *npYOffset, int nSourceWidth, int nSourceHeight, int nDestWidth, int nDestHeight)
{
	*npXOffset = cpGetXOffsetForCenteringImageOnScreen(nSourceWidth, nDestWidth);
	*npYOffset = cpGetYOffsetForCenteringImageOnScreen(nSourceHeight, nDestHeight);
}



/* If the dimensions have been confined to a suare, return these, else return the passed in dimensions */
static void
_cpGetConfinedDimensions(int *npW, int *npH, int nDestWidth, int nDestHeight)
{
	if(G_cps.bConfineEverythingToSquare)
	{
		*npW = *npH = MIN(nDestWidth, nDestHeight);
	}
	else
	{
		*npW = nDestWidth;
		*npH = nDestHeight;
	}
}


/* functions using the global CpImgColSp instance G_icsImgColSpc */

// Must be called after cpSetupGfx() has been called.
void
cpGetSizeToMakePlasma(int *npW, int *npH)
{
	if(cpImgColSpIsUsed(&G_icsImgColSpc))
	{
		*npW = G_icsImgColSpc.nW;
		*npH = G_icsImgColSpc.nH;
	}
	else
	{
		ASSERT(G_bmpDest);	// ? Should this ASSERT() be done outside the if()?

		_cpGetConfinedDimensions(npW, npH, G_bmpDest->w, G_bmpDest->h);
	}
}



/* When the source-pic does not cover the entire screen, work out which colur to clear the rest of the screen to */
static int
_cpGetSourcePicBGCol(BITMAP *bmpSourcePic)
{
	int nSourceBGCol;

	if(bitmap_color_depth(bmpSourcePic)==8)
	{
		/* Paletted */
		/* Use 1st index as background colour */
		// ?: Should we always attempt to use black instead? If so, what if palette does not contain black?
		// If doing above, would also have to pass in the palette to this function.
		// Would select_palette/unselect_palette be involved or shall we do an rgb_map?
		nSourceBGCol = 0;
	}
	else
	{
		/* Truecolour */
		// ?: Should we have a setting for colours other than black?
		// Medium grey? White? Red?
		nSourceBGCol = makecol_depth(bitmap_color_depth(bmpSourcePic), 0x0, 0x0, 0x0);
	}

	return nSourceBGCol;
}



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                          Palette functions.

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */



// In 8-bit gfx, sets the screen palette to the current state of G_pPal. Should only be called in 8-bit mode.
// cpPreCalcFunkyPalette() or some other palette-setting function must have been called first.
// render
void
cpSetPalette(void)
{
	set_palette_range(G_pPal, 0, PAL_SIZE-1, FALSE);	// retracesync param: FALSE: Do not wait for vsync but create a tear. TRUE: wait for vsunc but do not create a tear.
	G_bPalIsMucky = FALSE;
}


// When building colour-lookup tables for modes of various BPPs, select the ammount to right-shift each value by.
void
cpGetColourComponentShift(int nBPP, unsigned char *npRs, unsigned char *npGs, unsigned char *npBs)
{
	switch(nBPP)
	{
		case 8:
		{
			/* 8-bit: 6-6-6 */
			/* 6 coz Allegro's 8-bit mode only uses 6-bits per colour-component */
			/* ( 0..255 -> 0..63 ) */
			*npRs = *npGs = *npBs = 2;
		}
		break;

		case 15:
		{
			/* 15-bit: 5-5-5 */
			/* ( 0..255 -> 0..31 ) */
			*npRs = *npGs = *npBs = 3;
		}
		break;

		case 16:
		{
			/* 16-bit: 5-6-5 */
			*npRs = 3;
			*npGs = 2;
			*npBs = 3;
		}
		break;

		case 24:
		case 32:
		{
			/* 24/32-bit: 8-8-8 */
			*npRs = *npGs = *npBs = 0;
		}
		break;

		default:
		{
			ASSERT(FALSE);
		}
	}
}


/* Given a palette with 6 bits per component (a standard Allegro PLAETTE), adjust it so it has the correct number of bits per component for the given BPP. */
static void
_cpAdjust6BitPerColourComponentPaletteForGivenBpp(int nBPP)
{
	unsigned int nI;

	unsigned char nRs, nGs, nBs;

	cpGetColourComponentShift(nBPP, &nRs, &nGs, &nBs);


	switch(nBPP)
	{
		case 8:
		{
			/* Do nowt coz palette already in correct form. */
		}
		break;

		case 15:
		case 16:
		{
			/* Less resolution per colour-component, so shift right. */
			nRs-=2;
			nGs-=2;
			nBs-=2;

			for(nI=0;nI<PAL_SIZE;nI++)
			{
				G_pPal[nI].r>>=nRs;
				G_pPal[nI].g>>=nGs;
				G_pPal[nI].b>>=nBs;					
			}

		}
		break;

		case 24:
		case 32:
		{
			/* More resolution per colour-component, so shift left. */
			nRs+=2;
			nGs+=2;
			nBs+=2;

			for(nI=0;nI<PAL_SIZE;nI++)
			{
				G_pPal[nI].r<<=nRs;
				G_pPal[nI].g<<=nGs;
				G_pPal[nI].b<<=nBs;					
			}
		}
		break;


		default:
		{
			ASSERT(FALSE);
		}

	}
}




// Idea: Could use a tint (H and S). Use either HSV or HLS.
/* Create a monochrome palette with bounce */
void
cpCreateMonoPalette(int nDestBPP)
{
	unsigned int nI=0;

	unsigned char nRs, nGs, nBs;


	cpGetColourComponentShift(nDestBPP, &nRs, &nGs, &nBs);

	while(nI<256)
	{
		unsigned char cVal;

		cVal = nI;

		G_pPal[nI].r = cVal>>nRs;
		G_pPal[nI].g = cVal>>nGs;
		G_pPal[nI].b = cVal>>nBs;

		nI++;
	}

	G_bPalIsMucky = TRUE;
}



// Idea: Could use a tint (H and S). Use either HSV or HLS.
/* Create a monochrome palette with bounce */
/* Simulate a plasma in intensity and bounce mode using a colour lookup table that peaks in the middle. */
void
cpCreateMonoPaletteWithBounceAndIntensity(int nDestBPP, uint8_t nIntensity)
{
	unsigned int nI=0;


	unsigned char nRs, nGs, nBs;

	uint8_t nNormalizationVal = -(nIntensity>>1) + 0x7F;	/* Normalise so that centred around 0x7F (the middle, FF>>1 (FF = max nIntensity) ) */


	cpGetColourComponentShift(nDestBPP, &nRs, &nGs, &nBs);


	/* The entries will be in the range 0..254 going up and 255..1 going down. */

	// OPTIMIZATION idea: If intensity = 0x7F, we can not only get away by not multiplying cVal by the intensity, but we can skip adding the normalization value.

	while(nI<128)
	{
		unsigned char cVal;

		/* We're 'bouncing' the palette here (sawtooth maps to triangular). */
		cVal = nI<<1;
		cVal = cpMUL8(cVal,nIntensity);
		cVal += nNormalizationVal;	
		
		// ( 0..255 -> 0..63 )
		G_pPal[nI].r = cVal>>nRs;
		G_pPal[nI].g = cVal>>nGs;
		G_pPal[nI].b = cVal>>nBs;

		nI++;
	}

	while(nI<256)
	{
		unsigned char cVal;

		/* We're 'bouncing' the palette here (sawtooth maps to triangular). */
		//cVal = 0xFF-((nI & 0x7F)<<1);	// As we're using an unsigned int for nI, we can get away with the line below coz it does not overflow a non 8-bit type.
		cVal = 0x1FF-(nI<<1);	/* Use 0x1FE instead of 0x1FF to get the going down bit 254..0 instead of 255..1 */
		cVal = cpMUL8(cVal,nIntensity);
		cVal += nNormalizationVal;	/* Normalise so that centred around 0x7F (the middle, FF>>1 (FF = max nIntensity) ) */
		
		// ( 0..255 -> 0..63 )
		G_pPal[nI].r = cVal>>nRs;
		G_pPal[nI].g = cVal>>nGs;
		G_pPal[nI].b = cVal>>nBs;

		nI++;
	}

	G_bPalIsMucky = TRUE;
}



/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                Obtaining the image (platform independant)

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */



static AeBool
_cpLoadPicFromFile(BITMAP **bmppSourcePic, PALETTE pSourcePicPal, AL_CONST char *szFilename)
{
	// ?: Should we make use of G_cps. width,height,depth here? Do we need to?

	// TODO: Place filenames in ""'s (in case filenames have spaces in them)
	// Not nesc. if get_config_string() can parse strings with spaces, but there may be trailing/leading spaces.


	TRACE("Attempting to load \"%s\".\n", szFilename);


	set_color_conversion(COLORCONV_NONE);


	if(exists(szFilename))
	{
		/* BMP, LBM, PCX, and TGA plus any extensions we care to register. */
		if(!(*bmppSourcePic = load_bitmap(szFilename, pSourcePicPal)))
		{
			// TODO: More detailed error-message
			// + File of non-supported type
			// + File corrupt (file integrity fails)
			// + read-error (disk failiure)
			// + No read-permission
			
			
			/* Just fail with an error and an error-message on screen */
			cpSafeModeErrorDisplay("ERROR: %s: Couldn't read the file!", szFilename);
			
			return FALSE;
		}

		// Note: If the depth of the loaded image after conversion !=8 (or !=8 on no conversion), then pSourcePicPal becomes the 332 palette.

	}
	else
	{
		cpSafeModeErrorDisplay("ERROR: %s: This file does not exist!", szFilename);
		return FALSE;
	}

	TRACE("Loading \"%s\" succeeded.\n", szFilename);


	return TRUE;
}




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                       Setting and setting-up functions

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */




void
cpTRACEGFXmodelistContents(GFX_MODE_LIST *gmlpModes, int nCard)
{
	unsigned int nI;
	GFX_MODE *mpMode = gmlpModes->mode;


#ifdef _DEBUG
	{
		char szGFXCardIDString[5];	/* Gfx-mode ID-string used by chGetCurrentModeIDString() */
		TRACE("Dumping GFX_MODE_LIST for card '%s'...\n", aeGetAllegroIDStringFromAllegroID(szGFXCardIDString,nCard));
	}
#endif



	for(nI=0;nI<(unsigned int)gmlpModes->num_modes;nI++,mpMode++)
	{
		TRACE("%02u: %d * %d @ %dBPP", nI, mpMode->width, mpMode->height, mpMode->bpp);
		if(mpMode->bpp==32)
		{
			/* Assume the modes are stored sorted by rez and then BPP, and that 32BPP is the last for each rez. Then that way, all BPPS for a given rez are on the same line. */
			TRACE("\n");
		}
		else
		{
			TRACE("\t");
		}
	}

	TRACE("\n");
}


/* npWidth and npHeight are out-paramaters */
void
cpFindRezWithLargestAreaForGivenBPP(int *npWidth, int *npHeight, int nBPP, GFX_MODE_LIST *gmlpModes)
{

	// TODO: Test when being called from the Windows-Dialog procedure AND when being called from _cpFindAvailableRez

	int nI;
	int nAreaMax = 0;
	GFX_MODE *mpMode;

	mpMode = gmlpModes->mode;
	for(nI=0;nI<gmlpModes->num_modes;nI++,mpMode++)
	{
		if(mpMode->width * mpMode->height > nAreaMax && mpMode->bpp == nBPP)
		{
			*npWidth = mpMode->width;
			*npHeight = mpMode->height;
			nBPP = mpMode->bpp;

			nAreaMax = mpMode->width * mpMode->height;
		}
	}


	if(nAreaMax==0)
	{
		// TODO: The case where no modes are found (eg. the bit-depth is not supported).
		// Or should we just place an ASSERT here and assume it's only ever passed a valid bit-depth?
	}

}


/* If we can't set the specified mode, then try and find another mode to set instead */
/* We pass in the mimimum mode requirements, but if we can't meet the minimum, then look for the best smaller mode. */
static void
_cpFindAvailableRez(int nCard, int *npWidth, int *npHeight, int *npBPP)
{

	GFX_MODE_LIST *gmlpModes;
	GFX_MODE *mpMode;

	int nI;


	if(!(gmlpModes = get_gfx_mode_list(nCard) ))
	{
		// GFX_GDI has been known to do this
		// Should we try setting GFX_SAFE in this case?

#ifdef _DEBUG
		char szGFXCardIDString[5];	/* Gfx-mode ID-string used by chGetCurrentModeIDString() */
		TRACE("EEK! Could not get the graphics mode list for GFX-ID '%s'\n", aeGetAllegroIDStringFromAllegroID(szGFXCardIDString,nCard));
#endif

		*npWidth = *npHeight = 0;
		// ?: Is it possible to get the safe resolution and colourdepth for the given card
		//if (system_driver->get_gfx_safe_mode) {do it} else {eek}
		//*npWidth>>=1;
		//*npHeight>>=1;
		//*npBPP=8;

		return;	// ?: Return 'failure'?
	}

	// TODO: Test this on a machine with less VRAM

	cpTRACEGFXmodelistContents(gmlpModes, nCard);



	/* Search the list of graphics modes */
	mpMode = gmlpModes->mode;
	for(nI=0;nI<gmlpModes->num_modes;nI++,mpMode++)
	{

		#ifdef _DEBUG
		if(mpMode->width==0 || mpMode->height==0 || mpMode->bpp==0)
		{
			TRACE("Insanity: gmlpModes->num_modes inconsistent with number of modes in gmlpModes->mode\n");
			destroy_gfx_mode_list(gmlpModes);
			return;
		}
		// WARNING: We may want to set BPP to 0 if we exclude certain modes. In that case, don't ckeck the BPP above.
		#endif


		/* Find the first mode that's big enough to contain the entire image (assumes the modes are sorted so smallest is first, and lowest BPP is first (the BPP increases for each size as we iterate thru the list so we get the lowest BPP we can get)) */

		// TODO: Option to insist on using a particular bpp. Or use the highest BPP in case source-image a lower BPP.
		// TODO: Instead of the first mode larger than the image, have an option to be able to select a smaller mode.

		// Sometimes, no 16-bit may imply 15-bit possible, and vice versa.
		if(mpMode->width >= *npWidth && mpMode->height >= *npHeight && mpMode->bpp >= *npBPP)
		{
			*npWidth = mpMode->width;
			*npHeight = mpMode->height;
			*npBPP = mpMode->bpp;

			destroy_gfx_mode_list(gmlpModes);

			return;	/* We've found one */
		}

	}


	/* We still haven't found a mode. Let's now find the largest mode we can get. The image will be pruned/zoomed later on */

	// TODO: BPP: Outer-loop for each BPP. That way, BPP gets priority over screen-size.
	// But there could be an option to see if screensize or BPP gets priority. HZ could also come into the equasion.



	/* For now, just use the mode-area to determine the size */
	cpFindRezWithLargestAreaForGivenBPP(npWidth, npHeight, *npBPP, gmlpModes);
	// TODO: The mode who'se aspect most closely resembles the aspect of the source image. - Best-fit (but may be bad idea if widescreen monitors do stretching)

	if(*npWidth==0)
	{
		/* EEK: No modes with that BPP were found - let's venture into the domain of other BPPs */
		// ?: Should we have the option to fail if no supported mode found for the current BPP without searching in a different BPP?

		/* Let's find a supported BPP */
		// For now, we're more keen on higher BPPs than we are on large rez's
		int nBPPIndex=0;
		int nBPPToTry;
		do
		{
			nBPPToTry = cpGetAllowedBPP(nBPPIndex);
			if(nBPPToTry == -1)
			{
				/* No rez found at all - this should not happen, but just in case it does... */
				break;	/* *npWidth==0 implies failiure */
			}

			if((nBPPToTry==24 && G_cps.bDisable24bpp) || (nBPPToTry==8 && G_cps.bDisable8bpp))
			{
				continue;	/* Prevent unwanted BPPs from being tried. */
			}

			cpFindRezWithLargestAreaForGivenBPP(npWidth, npHeight, *npBPP, gmlpModes);
			if(*npWidth)
			{
				/* We found a width */
				break;
			}

			nBPPIndex++;
		}
		while(TRUE);
	}


	destroy_gfx_mode_list(gmlpModes);


	if(*npWidth==0)
	{
		/* No rez found - this should not happen, but just in case it does... set values to some values that set_gfx_mode() will fail on */
		TRACE("EEK! Could not find a suitable resolution\n");
		*npWidth = *npHeight = 0;
	}
}



int
cpGetRefreshRate(void)
{
	int nRefreshRate = get_refresh_rate();
	if(nRefreshRate == 0)
	{
		if(is_windowed_mode())
		{
			// TODO: Platform-specific thingies to get desktop refresh-rate

		}

		return 0;
	}

	return nRefreshRate;
}


#ifdef ALLEGRO_VRAM_SINGLE_SURFACE

/* Some platforms (eg. DOS) do page flipping by making one large screen that you can then scroll */
/* Note that this changes G_rtRenderingType if the required number of pages could not be allocated, and then attempts to allocate the reduced number of pages. It keeps going until a failed attempt to allocate a single-paged mode. */
static int
_cpDoSetModeAccordingToRenderType(int nCard, int nW, int nH)
{
	int nSetGfxModeRetVal=-1;

	int nNumPagesVRAM;


#if !defined(ALLEGRO_VRAM_SINGLE_SURFACE)
	ASSERT(FALSE);	/* Should only be called if CH_HARDCODED_SQUARESETTINGS is #defined */
#endif


	switch(G_rtRenderingType)
	{
		case cpRENDERTYPE_TRIPLE_BUFFER:
		{
			nNumPagesVRAM = 3;
		}
		break;

		case cpRENDERTYPE_PAGE_FLIP:
		{
			nNumPagesVRAM = 2;
		}
		break;

		default:
		{
			nNumPagesVRAM = 1;
		}
	}


	ASSERT(nNumPagesVRAM<=CP_GFX_MAXNUMPAGES && nNumPagesVRAM>0);

	do
	{
		nSetGfxModeRetVal = set_gfx_mode(nCard, nW, nH, nW, nH*nNumPagesVRAM);
		if(nSetGfxModeRetVal==0)
		{
			break;	/* We set a mode */
		}
		else
		{
			if(nNumPagesVRAM==3)
			{
				TRACE("Could not set up a mode with 3 pages for triple-buffering - attempting to use a mode with 2 pages for page-flipping instead.\n");
				nNumPagesVRAM=2;
				G_rtRenderingType = cpRENDERTYPE_PAGE_FLIP;
			}
			else if(nNumPagesVRAM==2)
			{
				TRACE("Could not set up a mode with 2 pages for page-flipping - attempting to use a mode with 1 page for double-buffering instead.\n");
				nNumPagesVRAM=1;
				G_rtRenderingType = cpRENDERTYPE_DOUBLE_BUFFER;	// cpRENDERTYPE_STRAIGHT ?
			}
			else
			{
				/* No more nNumPagesVRAM-modes to try. */
				break;
			}
		}
		
	}
	while(TRUE);

	return nSetGfxModeRetVal;	/* Emulate return value of set_gfx_mode */
}

#endif


static int
_cpAttemptSetGfxMode(int nCard, int nW, int nH)
{

#ifdef ALLEGRO_VRAM_SINGLE_SURFACE
	TRACE("This platform requires a single VRAM-surface when using multi-screen modes such as page-flipping or triple-buffering.\n");
	if(_cpDoSetModeAccordingToRenderType(nCard, nW, nH))
#else
	TRACE("This platform can create multiple VRAM-surfaces when using multi-screen modes such as page-flipping or triple-buffering.\n");
	if(set_gfx_mode(nCard, nW, nH, 0, 0))
#endif
	{
		return -1;	/* Failed to set this mode in any form */
	}

	/* Triple buffering requires additional prodding */
	if(G_rtRenderingType == cpRENDERTYPE_TRIPLE_BUFFER)
	{
		if(!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
		{
			TRACE("The mode cannot triple-buffer. Hang on ... trying something now...\n");

			enable_triple_buffer();	/* Try this */

			if(!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
			{
				TRACE("Attempt to enable triple buffering failed. Trying page-flipping instead.\n");

				/* If it didn't work, give up and change the rendering-type */
				G_rtRenderingType = cpRENDERTYPE_PAGE_FLIP;	// ?: Is it nesc. to undo anything that may have been done by enable_triple_buffer()?

#ifdef ALLEGRO_VRAM_SINGLE_SURFACE
				/* If using a single surface, we need to set the gfx-mode so it doesn't use up as much VRAM */ // ?: Is this nesc?
				if(_cpDoSetModeAccordingToRenderType(nCard, nW, nH))
				{
					/* If the first _cpDoSetModeAccordingToRenderType() succeeded, so should this, but just in case... */
					return -1;	/* Failed to set this mode in any form */
				}
#endif
			
			}
			else
			{
				TRACE("Mode is now set for triple buffering (after a bit of prodding).\n");
			}
		}
		else
		{
			TRACE("Mode is set for triple buffering.\n");
		}

	}


	G_nRefreshRate = cpGetRefreshRate();
	if(G_nRefreshRate == 0)
	{
		/* We cannot get the refresh-rate, so pretend it's CPREFRESHRATEVALUEWHENUNKNOWN */
		// ?: If it's a windowed mode, try getting the desktop refresh-rate.
		G_nRefreshRate = CPREFRESHRATEVALUEWHENUNKNOWN;
		G_bActualRefreshRateIsUnknown = TRUE;
	}
	else
	{
		G_bActualRefreshRateIsUnknown = FALSE;
	}


	/* Start the timers once we set our mode and sorted out the refresh-rate */
	// ?: Does set_gfx_mode() wait for the 1st vsync? And is that just for fullscreen modes or windowed mode too?
	// ?: Should we do a vsync() here?
	if(!cpStartTimers(G_nRefreshRate))
	{
		return FALSE;
	}



	if(G_rtRenderingType == cpRENDERTYPE_TRIPLE_BUFFER)	/* 3 pages */
	{
		g_bmpaPages[0] = create_video_bitmap(SCREEN_W, SCREEN_H);
		g_bmpaPages[1] = create_video_bitmap(SCREEN_W, SCREEN_H);
		g_bmpaPages[2] = create_video_bitmap(SCREEN_W, SCREEN_H);

		g_nPage = 1;	/* Draw onto page 1 first (page 0 (the first to be created) is where the screen will be at the start) */

		if (g_bmpaPages[0] && g_bmpaPages[1])
		{
			/* We have successfully set up triple buffering */
			return 0;
		}
		else
		{
			TRACE("Could not set up the video bitmaps required for triple-buffering - attempting to use page-flipping instead\n");

			if(g_bmpaPages[0])
			{
			   destroy_bitmap(g_bmpaPages[0]);
			   g_bmpaPages[0] = NULL;
			}

			if(g_bmpaPages[1])
			{
			   destroy_bitmap(g_bmpaPages[1]);
			   g_bmpaPages[1] = NULL;
			}

			if(g_bmpaPages[2])
			{
			   destroy_bitmap(g_bmpaPages[2]);
			   g_bmpaPages[2] = NULL;
			}

			G_rtRenderingType = cpRENDERTYPE_PAGE_FLIP;

#ifdef ALLEGRO_VRAM_SINGLE_SURFACE
			/* If using a single surface, we need to set the gfx-mode so it doesn't use up as much VRAM */ // ?: Is this nesc?
			if(_cpDoSetModeAccordingToRenderType(nCard, nW, nH))
			{
				/* If the first _cpDoSetModeAccordingToRenderType() succeeded, so should this, but just in case... */
				return -1;	/* Failed to set this mode in any form */
			}
#endif
			
		}


	}


	if(G_rtRenderingType == cpRENDERTYPE_PAGE_FLIP)	/* 2 pages */
	{
		/* If we're page-flipping, set up the 2 virtual screens within our graphics mode */
		g_bmpaPages[0] = create_video_bitmap(SCREEN_W, SCREEN_H);
		g_bmpaPages[1] = create_video_bitmap(SCREEN_W, SCREEN_H);

		g_nPage = 1;	/* Draw onto page 1 first (page 0 (the first to be created) is where the screen will be at the start) */

		if (g_bmpaPages[0] && g_bmpaPages[1])
		{
			/* We have successfully set up page flipping */
			return 0;
		}
		else
		{
			TRACE("Could not set up the video bitmaps required for page-flipping - attempting to use double-buffering instead\n");

			if(g_bmpaPages[0])
			{
			   destroy_bitmap(g_bmpaPages[0]);
			   g_bmpaPages[0] = NULL;
			}

			if(g_bmpaPages[1])
			{
			   destroy_bitmap(g_bmpaPages[1]);
			   g_bmpaPages[1] = NULL;
			}

			G_rtRenderingType = cpRENDERTYPE_DOUBLE_BUFFER;	// cpRENDERTYPE_STRAIGHT ?

#ifdef ALLEGRO_VRAM_SINGLE_SURFACE
			/* If using a single surface, we need to set the gfx-mode so it doesn't use up as much VRAM */ // ?: Is this nesc?
			if(_cpDoSetModeAccordingToRenderType(nCard, nW, nH))
			{
				/* If the first _cpDoSetModeAccordingToRenderType() succeeded, so should this, but just in case... */
				return -1;	/* Failed to set this mode in any form */
			}
#endif

		}

 
	}

	return 0;
}



/* width, height, and BPP are the size of the image we're trying to set. */

static AeBool
_cpSetGfxMode(int *npWidth, int *npHeight, int *npBPP, AeBool bCanTryOtherRezs)
{
	int nCard = CPGFXCARD;


	#ifdef ALLEGRO_WINDOWS
		/* When debugging, use a GDI window */
		#ifdef _DEBUG
			nCard = GFX_GDI;
			//nCard = GFX_DIRECTX_WIN;
		#endif
	#endif





	// TODO: Before we set the mode, check to see if it is on list of allowed rez's
	// But thanks to a quirk in Allegro's ability to report available modes, the list does not include 1600 * 1200 when it should (Matrox Marvel G400TV).
	// So we first attempt to set the mode we've been given regardless of whether or not it's on the list

	



	set_color_depth(*npBPP);
	if(_cpAttemptSetGfxMode(nCard, *npWidth, *npHeight))
	{
		/* Argh! couldn't set mode */

		if(bCanTryOtherRezs)
		{
			TRACE("Could not set the resolution - attempting to try to set a different resolution.\n");

			_cpFindAvailableRez(nCard, npWidth, npHeight, npBPP);	// ?: Test for failure?

			set_color_depth(*npBPP);
			if(_cpAttemptSetGfxMode(nCard, *npWidth, *npHeight))
			{
				/* We are only giving _cpFindAvailableRez() one chance to find another resolution */
				TRACE("EEK: Could not set any resolution!\n");

				/* Just fail with an error and an error-message on screen */
				cpSafeModeErrorDisplay("ERROR! couldn't set a GFX mode (including the requested %d x %d @ %dBPP)", *npWidth, *npHeight, *npBPP);

				return FALSE;
			}
		}
		else
		{
			TRACE("EEK: Could not set the resolution we want!\n");

			/* Just fail with an error and an error-message on screen */
			cpSafeModeErrorDisplay("ERROR! couldn't set the GFX mode we want - %d x %d @ %dBPP", *npWidth, *npHeight, *npBPP);

			return FALSE;
		}
	}


	return TRUE;
}



// \sa _cpSetupPicPostSetGfxMode()
static AeBool
_cpSetupPicPreSetGfxMode(int *npSourceWidth, int *npSourceHeight, int *npSourceBPP)
{
	/* if g_bmpSourcePic has been set (ie, !=NULL), it means this function has setup the pic */

	
	switch(G_cps.isImgSource)
	{
		case cpIMGTYPE_DESKTOP:
		{
			TRACE("Using the desktop as the image.\n");

			#ifdef ALLEGRO_WINDOWS
			// TODO: Other platforms with desktops.
			// WARNING: cpDoProgram() calls set_gfx_mode() if not ALLEGRO_WINDOWS and docs for desktop_color_depth() and get_desktop_resolution() state "Under some OSes, switching to a full screen graphics mode may automatically change the desktop resolution. You have, therefore, to call this function before setting any graphics mode in order to retrieve the real desktop resolution." 



			//if(get_desktop_resolution(npSourceWidth, npSourceHeight)==0)
			if(cpGetDesktopResolution_Win(npSourceWidth, npSourceHeight)==0)
			{
				/* We have a desktop */
				//*npSourceBPP = desktop_color_depth();
				*npSourceBPP = cpGetDesktopColorDepth_Win();



				if(!(g_bmpSourcePic=create_bitmap_ex(*npSourceBPP, *npSourceWidth, *npSourceHeight)))
				{
					TRACE("EEK! Could not create a bitmap to copy the desktop onto.\n");
					return FALSE;
				}

				if(!cpGrabDesktopImage_Win(g_bmpSourcePic))
				{
					TRACE("EEK! Could get the desktop wallpaper.\n");
					return FALSE;

				}

				generate_332_palette(g_pSourcePicPal);	// When in paletted modes, use the 332 palette (?: Should we use a custom/optimised palette instead?)
			}
			else
			{
				/* This platform does not have a concept of a desktop - Use the custom resolution specified in the file instead. */
				*npSourceWidth = G_cps.nWidth;
				*npSourceHeight = G_cps.nHeight;
				*npSourceBPP = G_cps.nBPP;

				// As for the image: is it somehow possible to grab screenshot of command-prompt or character-terminal?
			}



			#else
				TRACE("Cannot use desktop as image as this platform has no concept of a desktop.\n");
				TRACE("Using built-in image instead.\n");

				/* Use builtin image instead */
				G_cps.isImgSource = cpIMGTYPE_BUILTIN;

			#endif


		}
		break;

		case cpIMGTYPE_DESKTOP_WALLPAPER:
		{
			char szDesktopWallpaperPath[CPPATHSTRINGSIZE*2 /* *2 for Wide characters in cpGetDesktopWallpaperPath_Win() */];	// ?: (MAX_PATH+1) instead of CPPATHSTRINGSIZE ?


			TRACE("Using the desktop-wallpaper as the image.\n");

			// TODO: Stretch/tile/centre according to windows settings

			#ifdef ALLEGRO_WINDOWS
			// TODO: Other platforms with desktops.


			
			
			if(!cpGetDesktopWallpaperPath_Win(szDesktopWallpaperPath))
			{
				ustrzcpy(szDesktopWallpaperPath, CPPATHSTRINGSIZE, empty_string);	/* Empty the desktop wallpaper string */

				// Use GetLastError() for more detailed info.

				TRACE("EEK! Problem getting desktop wallpaper.\n");
				return FALSE;	// ?: Should we not return false and just do cpIMGTYPE_BUILTIN ?
			}

			if(szDesktopWallpaperPath[0])
			{
				/* We have a desktop wallpaper - load it. */

				// ?: Does this support all supported Windows Desktop wallpaper filetypes?

				if(!_cpLoadPicFromFile(&g_bmpSourcePic, g_pSourcePicPal, szDesktopWallpaperPath))
				{
					TRACE("Error loading the wallpaper piccy.\n");
					return FALSE;
				}

				*npSourceWidth = g_bmpSourcePic->w;
				*npSourceHeight = g_bmpSourcePic->h;
				*npSourceBPP = bitmap_color_depth(g_bmpSourcePic);


				// ?: If 256 colours, are we better off with the 332 palette?
			}
			else
			{
				// TODO: Now that we know what to do, do something!

				uint8_t nR, nG, nB;	// ?: Should we use an RGB struct instead?

				/* We cannot get a desktop wallpaper. For now, assume desktop background is a solid colour */
				cpGetDesktopSolidColour_Win(&nR, &nG, &nB);

	
				// TODO: Create the colourspace thingies with a uniform value - this colour.
				// Or alternatively, use a no-image plasma tinted with the H and S (HSL/HSV dependent on setting for no-plasma)
				// or create a CLUT for all 256 levels applied to that particular colour.

				// ?: Pattern? // In Windows, there's SPI_SETDESKPATTERN but there does not seem to be a GET equivalent.


				TRACE("ERROR: Single-colour desktop background not yet implemented.\n");
				return FALSE;

			}

			// ?: Are active-desktops supported?




			// If just a single colour, use the 332 palette.


			#else
				TRACE("Cannot use desktop-wallpaper as image as this platform has no concept of a desktop-wallpaper (either that or it hasn't been implemented on this platform).\n");
				TRACE("Using built-in image instead.\n");

				/* Use builtin image instead */
				G_cps.isImgSource = cpIMGTYPE_BUILTIN;

			#endif

		}
		break;

		case cpIMGTYPE_FILE:
		{
			TRACE("Loading the image from a file.\n");

			if(!_cpLoadPicFromFile(&g_bmpSourcePic, g_pSourcePicPal, G_cps.szFileName))
			{
				TRACE("Error loading the piccy.\n");
				return FALSE;
			}

			*npSourceWidth = g_bmpSourcePic->w;
			*npSourceHeight = g_bmpSourcePic->h;
			*npSourceBPP = bitmap_color_depth(g_bmpSourcePic);
		}
		break;

		default:
		{
			/* Note: The rest of the cases are handled in _cpSetupPicPostSetGfxMode() */
			/* !g_bmpSourcePic implies that we haven't setup a pic yet - for that, there's _cpSetupPicPostSetGfxMode() */
		}

	}



	/* If we have not managed to obtain a resolution, get the the desktop reslution if we can */
	if(*npSourceWidth == 0 || *npSourceHeight == 0 || *npSourceBPP == 0)
	{
		// cpGetDesktopResolution()?
		if(get_desktop_resolution(npSourceWidth, npSourceHeight)==0)
		{
			/* We have a desktop */
			// cpDesktopColorDepth()
			*npSourceBPP = desktop_color_depth();
			
			*npSourceBPP = 32;	// We currently only support 32-bit thingies.
		}
		else
		{
			*npSourceWidth = *npSourceHeight = 0;	/* Still not got a rez */
		}
	}
	

	return TRUE;
}



// \sa _cpSetupPicPreSetGfxMode()
AeBool
_cpSetupPicPostSetGfxMode(int nWidth, int nHeight, int nBPP)
{
	switch(G_cps.isImgSource)
	{
		case cpIMGTYPE_NONE:
		{
			TRACE("No-image - just plasma.\n");

			/* g_bmpSourcePic remains NULL */

		}
		break;

		case cpIMGTYPE_BUILTIN:
		{
			TRACE("Making the built-in image.\n");

			if(!cpMakeBuiltinPic(&G_icsImgColSpc, nWidth, nHeight))
			{
				TRACE("Error making the builtin pic.\n");
				return FALSE;
			}

			/* g_bmpSourcePic remains NULL, but G_icsImgColSpc is used. */

		}
		break;

		default:
		{
			/* Note: valid image-source cases not handled here have already been handled in _cpSetupPicPreSetGfxMode() */
			TRACE("Invalid image-source.\n");
			return FALSE;
		}
	}


	if((!g_bmpSourcePic && !cpImgColSpIsUsed(&G_icsImgColSpc)) && G_cps.isImgSource != cpIMGTYPE_NONE)
	{
		TRACE("EEK! No image was used!\n");
		return FALSE;
	}


	return TRUE;
}



// Make sure source image (and area to be plasma'd round the source image) are correct, and adjust accordingly.
// Adjusting is done by zooming, pruning (which involve creating a new bitmap to zome/prune onto). Copying is also done if we want the plasma effect to be in the borders too.
// Note that this may change g_bmpSourcePic to a new bitmap.
static AeBool
_cpPruneZoomAddBordersIfWeHaveTo(int nDestWidth, int nDestHeight)
{
	
	/* Prune/zoom image if we have to */
	
	// TODO: if !bSetGfxMode, then we will get different results unless we're doing a non-proportional zoom - especially if G_cps.irsImgResSource==cpIMGRESSOURCE_CUSTOM.
	// In that case, we will have to do a proportion-preserving zoom as well as the clipping
	
	if(g_bmpSourcePic->w != nDestWidth || g_bmpSourcePic->h != nDestHeight)
	{
		/* Source image and destination are of different sizes */
		
		int nSourceBGCol = _cpGetSourcePicBGCol(g_bmpSourcePic);	/* Background colour when empty areas on the bitmap are required (if original not lword aligned, or bit outside the image is empty if filling screen with plasma) */
		
		if(G_cps.bZoom)
		{
			/* Zoom/reduce image to fit */
			BITMAP *bmpTemp;
			
			int nNewImgX, nNewImgY;	/* New size of the image */
			int nNewBmpX, nNewBmpY;	/* New size of the bitmap to be plasma'd (will either be {nNewImgX,nNewImgY} or {nDestWidth,nDestHeight} depending on G_cps.bExpandPlasmaToFillScreen ) */
			int nImgOffsetX, nImgOffsetY;	/* Used to centre image on plasma if plasma expanded to fill screen. */
			
			/* Work out size of our new image */
			if(G_cps.bZoomAspectPreserve)
			{
				float nAspectRatio = MIN((float)nDestWidth / (float)g_bmpSourcePic->w, (float)nDestHeight / (float)g_bmpSourcePic->h);
				
				float nFX = ((float)g_bmpSourcePic->w) * nAspectRatio;
				float nFY = ((float)g_bmpSourcePic->h) * nAspectRatio;
				
				/* Result stored in a float to prevent some weird rounding-error. */
				nNewImgX = (int) (nFX+0.5f);
				nNewImgY = (int) (nFY+0.5f);
				
				/* lword-align X (round up unless value goes off edge of dest - then we round down) */
				_cpRoundUpToLwordUnlessBeyondMax(&nNewImgX, nDestWidth);
			}
			else
			{
				/* We're not bothering to preserve the aspect ratio - we're gonna use the whole screen, baby! */
				nNewImgX = nDestWidth;
				nNewImgY = nDestHeight;
			}
			
			
			/* Work out the size of the area being plasma'd (covered by the plasma) */
			if(G_cps.bExpandPlasmaToFillScreen)
			{
				/* Plasma will be size of whole screen and image blitted to centre */
				/* Because we are zooming with aspect-preserve, nNewBmp is going to be bigger than nNewImgX in 0 or 1 dimensions. */
				nNewBmpX = nDestWidth;
				nNewBmpY = nDestHeight;
				nImgOffsetX = (nNewBmpX-nNewImgX)/2;
				nImgOffsetY = (nNewBmpY-nNewImgY)/2;
			}
			else
			{
				/* Plasma will be size of image */
				nNewBmpX = nNewImgX;
				nNewBmpY = nNewImgY;
				nImgOffsetX = 0;
				nImgOffsetY = 0;
			}
			
			
			/* Create new bitmap (the size of the area to be plasma'd) */
			if(!(bmpTemp=create_bitmap_ex(bitmap_color_depth(g_bmpSourcePic), nNewBmpX, nNewBmpY)))
			{
				destroy_bitmap(g_bmpSourcePic);
				g_bmpSourcePic = NULL;
				TRACE("EEK! Could not create a bitmap to zoom/reduce onto.\n");
				return FALSE;
			}
			
			/* Zoom to the size of the new image (this may be smaller than the bitmap that will be plasma'd if G_cps.bExpandPlasmaToFillScreen) */
			// TODO: Interpolated zoom (make sure no anti-aliasing on edges - especially if plasma confined to image).
			clear_to_color(bmpTemp,nSourceBGCol);	/* Make sure the right edge is empty if original not lword aligned, or bit outside the image is empty if filling screen with plasma */ /* Note: This isn't actually needed if zooming and not perserving the aspect. */
			stretch_blit(g_bmpSourcePic, bmpTemp, 0,0, g_bmpSourcePic->w, g_bmpSourcePic->h, nImgOffsetX,nImgOffsetY, nNewImgX, nNewImgY);
			
			/* Done with the old so use the new */
			destroy_bitmap(g_bmpSourcePic);
			g_bmpSourcePic = bmpTemp;
		}
		else
		{
			/* Prune the source image if the image is too large in a dimension for the destination. */
			/* Or if the X dimension is too big, then lword-align it. */
			
			// TODO: If it's for the preview window, then zoom before we prune.
			// That way, the preview-window will accurately show the resulting image
			
			
			if(g_bmpSourcePic->w > nDestWidth || g_bmpSourcePic->h > nDestHeight || g_bmpSourcePic->w != (g_bmpSourcePic->w&0x7FFFFFFC))	// Beware if screen-X not multiple of 4.
			{
				/* At least one dimension larger than screen, or pic has X that's not a multiple of 4 */
				BITMAP *bmpTemp;
				
				/* Work out size of our new bitmap */
				int nNewImgX, nNewImgY;
				int nNewBmpX, nNewBmpY;	/* New size of the bitmap to be plasma'd (will either be {nNewImgX,nNewImgY} or {nDestWidth,nDestHeight} depending on G_cps.bExpandPlasmaToFillScreen ) */
				int nXSourceOffset, nYSourceOffset;	/* If source larger than dest, use these offsets so we blit from a centred window into the image */
				int nImgOffsetX, nImgOffsetY;	/* Used to centre image on plasma if plasma expanded to fill screen. */
				
				nNewImgX = MIN(g_bmpSourcePic->w,nDestWidth);
				nNewImgY = MIN(g_bmpSourcePic->h,nDestHeight);
				
				_cpRoundUpToLwordUnlessBeyondMax(&nNewImgX, nDestWidth);	// Beware if screen-X not multiple of 4.
				
				/* Work out the size of the area being plasma'd (covered by the plasma) */
				if(G_cps.bExpandPlasmaToFillScreen)
				{
					/* Plasma will be size of whole screen and image blitted to centre */
					/* Because we are zooming with aspect-preserve, nNewBmp is going to be bigger than nNewImgX in 0 or 1 dimensions. */
					nNewBmpX = nDestWidth;
					nNewBmpY = nDestHeight;
					nImgOffsetX = (nNewBmpX-nNewImgX)/2;
					nImgOffsetY = (nNewBmpY-nNewImgY)/2;
				}
				else
				{
					/* Plasma will be size of image */
					nNewBmpX = nNewImgX;
					nNewBmpY = nNewImgY;
					nImgOffsetX = 0;
					nImgOffsetY = 0;
				}
				
				
				/* Create new bitmap */
				if(!(bmpTemp=create_bitmap_ex(bitmap_color_depth(g_bmpSourcePic), nNewBmpX, nNewBmpY) ))
				{
					destroy_bitmap(g_bmpSourcePic);
					g_bmpSourcePic = NULL;
					TRACE("EEK! Could not create a bitmap to prune onto.\n");
					return FALSE;
				}
				
				
				/* Work out window to blit from if source larger than dest so we can blit from the centre of the source-image */
				_cpGetOffsetsForBlittingFromCentreOfImage(&nXSourceOffset, &nYSourceOffset, g_bmpSourcePic->w, g_bmpSourcePic->h, nNewImgX, nNewImgY);
				
				clear_to_color(bmpTemp,nSourceBGCol);	/* Make sure the right edge is empty if original not lword aligned */
				blit(g_bmpSourcePic, bmpTemp, nXSourceOffset,nYSourceOffset, nImgOffsetX,nImgOffsetY, nNewImgX, nNewImgY);
				
				
				destroy_bitmap(g_bmpSourcePic);	/* Done with the old */
				g_bmpSourcePic = bmpTemp;
			}
			else
			{
				/* Either one or both dimensions smaller than screen, and no dimensions larger than screen */
				
				if(G_cps.bExpandPlasmaToFillScreen)
				{
					BITMAP *bmpTemp;
					
					int nImgOffsetX, nImgOffsetY;	/* Used to centre image on plasma if plasma expanded to fill screen. */
					
					
					/* Create new bitmap (the size of the area to be plasma'd) */
					if(!(bmpTemp=create_bitmap_ex(bitmap_color_depth(g_bmpSourcePic), nDestWidth, nDestHeight)))
					{
						destroy_bitmap(g_bmpSourcePic);
						g_bmpSourcePic = NULL;
						TRACE("EEK! Could not create a bitmap to zoom/reduce onto.\n");
						return FALSE;
					}
					
					cpGetOffsetsForCenteringImageOnScreen(&nImgOffsetX, &nImgOffsetY, g_bmpSourcePic->w, g_bmpSourcePic->h, nDestWidth, nDestHeight);
					
					clear_to_color(bmpTemp,nSourceBGCol);	/* Make sure bit outside the image is empty if filling screen with plasma */
					blit(g_bmpSourcePic, bmpTemp, 0,0, nImgOffsetX,nImgOffsetY, g_bmpSourcePic->w, g_bmpSourcePic->h);
					
					
					destroy_bitmap(g_bmpSourcePic);	/* Done with the old */
					g_bmpSourcePic = bmpTemp;
				}
				// Else, just rely on _cpRenderImage() to do the centering
			}
			
		}
	}
	

	return TRUE;
}



// \sa cpCleanUpGfx
AeBool
cpSetupGfx(AeBool bSetGfxMode, int nDestWidth, int nDestHeight, int nDestBPP)
{
	/* nDestWidth, nDestHeight, nDestBPP Used to store details about the destination bitmap size and depth (will be the screen if bSetGfxMode or else the passed in size) */
	// ?: Should nDestWidth, nDestHeight, and nDestBPP be in-out params? A: No, coz G_bmpDest contains all the info.
	
	// perhaps the destWH could implicitly imply bSetGfxMode if they're ==0, but beware, as bSetGfxMode is used further down the line too.
	// currently, bSetGfxMode means the inputs can be anything, and !bSetGfxMode means the inputs has to be set to the size.

	int nSourceWidth, nSourceHeight, nSourceBPP;	/* Used to store details about the source image we're going to use */ // ?: Can g_bmpSourcePic be used to store these? A: Not if we're not using a source-pic.


	g_bmpSourcePic = NULL;	/* Make sure we know we haven't set up the source-picture yet */

	cpImgColSpInit(&G_icsImgColSpc);	/* Make sure we know we haven't set up the source-picture yet */



	nSourceWidth = nSourceHeight = nSourceBPP = 0;	/* Start off with 0 so we know that we haven't yet filled these in if they don't get filled in */


#ifndef CP_NO_JPEG
	if(jpgalleg_init()==-1)
	{
		TRACE("EEK! Could not initialise the JPEG module!\n");
		return FALSE;
	}
#endif


#ifndef CP_NO_PNG
	if(loadpng_init()==-1)
	{
		TRACE("EEK! Could not initialise the JPEG module!\n");
		return FALSE;
	}
#endif



	if(bSetGfxMode)
	{
		G_rtRenderingType = G_cps.rtRenderingType;	/* This may change if we can't set up page-flipping or double-buffering - hence the need for a seperate copy of this value */
	}
	else
	{
		G_rtRenderingType = cpRENDERTYPE_DOUBLE_BUFFER;	/* When not setting a gfx mode, we will always be drawing to a buffer. */
	}



	/* See if we can setup source-picture that is dependant on the picture size and if not, set one up later */
	if(!_cpSetupPicPreSetGfxMode(&nSourceWidth, &nSourceHeight, &nSourceBPP))	/* Note: if this did not set the source-thingies, they are set to the desktop resolution if it can be obtained. */
	{
		TRACE("EEK! Problem setting up the pic to use!\n");
		return FALSE;
	}


	if(G_cps.bDisable8bpp)
	{
		if(nSourceBPP==8)
		{
			nSourceBPP=24;	/* Chose 24 coz it can later be convertedd to 32 if needed. */
		}
	}


	if(G_cps.bSetSourceBPPTo32If24 || G_cps.bDisable24bpp)
	{
		if(nSourceBPP==24)
		{
			nSourceBPP=32;
		}
	}



	if(bSetGfxMode)
	{
		/*	+ If we're using a custom rez, then over-ride any resolution obtained from _cpSetupPicPreSetGfxMode().
			+ If _cpSetupPicPreSetGfxMode() didn't return any resolution, then use the one specified in the file	*/
		if(G_cps.irsImgResSource==cpIMGRESSOURCE_CUSTOM || nSourceWidth == 0 || nSourceHeight == 0 || nSourceBPP == 0)
		{
			nDestWidth = G_cps.nWidth;
			nDestHeight = G_cps.nHeight;
			nDestBPP = G_cps.nBPP;
		}
		else
		{
			nDestWidth = nSourceWidth;
			nDestHeight = nSourceHeight;
			nDestBPP = nSourceBPP;
		}
	}


	if(bSetGfxMode)
	{
		/* Setup screen graphics mode */
		if(!_cpSetGfxMode(&nDestWidth, &nDestHeight, &nDestBPP, (G_cps.irsImgResSource==cpIMGRESSOURCE_IMAGE)))
		{
			TRACE("Error Setting gfx mode.\n");
			return FALSE;
		}
	}
	else
	{
		// Currently, bSetGfxMode is FALSE when doing the preview, but in future, we may be drawinf direct to the desktop-HDC 
		// TODO: If we decide to draw to the desktop HDC, set timers here too.
		// ?: Is it possible to get the Windows refresh-rate of the desktop? That could also come in handy for SS-preview-mode.
		G_nRefreshRate = CPREFRESHRATEVALUEWHENUNKNOWN;
		G_bActualRefreshRateIsUnknown = TRUE;
	}



	/**************************************************
	 *  By now, the resolution and BPP are finalised  *
	 **************************************************/



	if(g_bmpSourcePic)
	{
		int nConfinedDestWidth, nConfinedDestHeight;

		_cpGetConfinedDimensions(&nConfinedDestWidth, &nConfinedDestHeight, nDestWidth, nDestHeight);


		/* Make sure source image (and area to be plasma'd round the source image) are correct and adjust accordingly. Note that this may change g_bmpSourcePic to a new bitmap. */
		if(!_cpPruneZoomAddBordersIfWeHaveTo(nConfinedDestWidth, nConfinedDestHeight))
		{
			// could also do the destroying of the old bitmap if unequal.
			return FALSE;
		}
	
	}




	/* Sort out palettes / CLUTs */


	g_bRGBMapAllocdByChromaPlas = FALSE;

	if(G_cps.isImgSource == cpIMGTYPE_NONE)
	{
		/* Set up the palette (colour lookup table) */
		cpCreateMonoPalette(nDestBPP);	/* Set up a temporary palette so we can do things like display error messages */

		if(nDestBPP==8)
		{
			/* Use the newly created palette with 8-bit gfx modes */
			cpSetPalette();

			G_bApplyingColourLookupTableDirectlyToPlasmaPixels = TRUE;	/* 8-bit paletted modes always use a lookup table for colours: the palette. */
		}
		else
		{
			G_bApplyingColourLookupTableDirectlyToPlasmaPixels = (G_cps.bAlwaysUseCLUTsInTruecolourModes || G_cps.psPaletteSource!=cpPALETTESOURCE_NONE /* ||GbUseTint */);	/* When the funky palette effect or custom palette is to be used, it implies we will be using a CLUT. */
		}


		/* Setup whatever palette effects we need */
		// ?: switch() instead?
		if(G_cps.psPaletteSource==cpPALETTESOURCE_FUNKYPALETTEEFFECT)
		{
			cpInitFunkyPaletteEffect();
		}
		else if(G_cps.psPaletteSource==cpPALETTESOURCE_EXTERNALPALETTE)
		{
			BITMAP *bmpDummy;	// TODO: Edit _cpLoadPicFromFile() so that we can pass in NULL. While we're at it, we might want to move the colour-conversion thingy out the function.

			// ?: Other palette sources other than image files (eg. PSP's .PAL file)?

			/* Load the image just so we can obtain it's palette. */
			// NB: If loading a truecolour image, Allegro will generate the 332 palette ( although it could call generate_optimized_palette() with the colour conversion set appropriately. )

			TRACE("Loading the palette-image from a file.\n");

			if(!_cpLoadPicFromFile(&bmpDummy, G_pPal, G_cps.szPaletteSourceFileName))
			{
				TRACE("Error loading the palette-image.\n");
				// TODO: delete / deinit what needs to be deleted.
				return FALSE;
			}
			destroy_bitmap(bmpDummy);

			_cpAdjust6BitPerColourComponentPaletteForGivenBpp(nDestBPP);
			G_bPalIsMucky = TRUE;

			// ?: Should we merge g_pSourcePicPal and G_pPal ?

		}
	}
	else
	{
		G_bApplyingColourLookupTableDirectlyToPlasmaPixels = FALSE;	/* If using a source-image, the plasma pixels will affect the pixels of the source image - hence a CLUT is not appropriate. */
		

		if(nDestBPP==8)
		{
			/* create a palette for use with 8-bit gfx modes when affecting an image. */
			
			/* As we're going to convert RGB to indexed, we need to set up the palette and the rgb_map table. */

			rgb_map = (RGB_MAP *) malloc(sizeof(RGB_MAP));
			if(rgb_map)
			{
				g_bRGBMapAllocdByChromaPlas = TRUE;	// methinx that when loading in an image we have an rgb_map table generated.
			}
			/* else don't use a table for colour-conversion (this is slow!) */


			if(G_cps.isImgSource == cpIMGTYPE_BUILTIN)
			{
				/* The builtin pic is best served by the 332 palette. */

				// ?: Use G_pPal instead to store the 332-palette?

				PALETTE pal332;
				generate_332_palette(pal332);

				set_palette(pal332);   // Set the palette to the best approximation of a true-color palette we can get with 8-bit color //

				create_rgb_table(rgb_map, pal332, NULL);

			}
			else
			{
				// TODO: A choice between g_pSourcePicPal and pal332

				// Note: If the depth of the loaded image after conversion !=8 (or !=8 on no conversion), then g_pSourcePicPal becomes the 332 palette.
				// in this case, g_pSourcePicPal may need to be generated here using generate_optimized_palette(bmp, pal, NULL); Actually, methinx best to use the 332-palette.

				// Note: If using g_pSourcePicPal on cp_test3_16.png, the problem with the BG turning blue might be due the 'black' actually being a dark grey, and to the limited size of sourcepicpal (16 colours).
				// The soltion would be to use the 32-palette.

				set_palette(g_pSourcePicPal);

				create_rgb_table(rgb_map, g_pSourcePicPal, NULL);

			}

			if(bSetGfxMode)
			{
				/* Now we know which is black (or the closest colour to black), clear the screen to that colour */
				// ?: Should we use palette-index 0 instead if paletted source-image or paletted mode?
				clear_to_color(screen,makecol_depth(nDestBPP, 0x0, 0x0, 0x0));
			}
		}
	}




	/* If we need to generate an image, do so here. */


	// ?: Should we put this entire if() and it's else in the else of the above if(G_cps.isImgSource == cpIMGTYPE_NONE) ?
	if(!g_bmpSourcePic)
	{
		/* If we haven't set up a source-picture already, setup a source-picture that is dependent on the rez we've set (exactly the same size as the rez) */


		if(G_cps.isImgSource != cpIMGTYPE_NONE)
		{
			/* Print a message so we don't have to wait for the pre-calculations to finish */
			if(bSetGfxMode)
			{
				int nTextColour = makecol_depth(nDestBPP, 0xFF, 0xFF, 0xFF);
				int nBGColour = makecol_depth(nDestBPP, 0x0, 0x0, 0x0);
				textout_centre_ex(screen, font, "Please wait while building and converting the image", SCREEN_W/2, SCREEN_H/2-font->height/2, nTextColour, nBGColour);
			}
			/*
			else
			// TODO: textout_ex to the Windows SS preview window. (?: Should we use a callback for this?)
			
			*/
			
		}
		// TODO: A similar message for when building the CpImgColSp and doing the plasma-tabs, but only if cpIMGTYPE_NONE.


		{
			int nConfinedDestWidth, nConfinedDestHeight;

			_cpGetConfinedDimensions(&nConfinedDestWidth, &nConfinedDestHeight, nDestWidth, nDestHeight);


			if(!_cpSetupPicPostSetGfxMode(nConfinedDestWidth, nConfinedDestHeight, nDestBPP))	/* Sets up the built-in pic if we've chosen to set it up. if(G_cps.isImgSource == cpIMGTYPE_NONE) nothing will be done. */
			{
				TRACE("EEK! Problem setting up the pic to use!\n");
				return FALSE;
			}
			else
			{
				if((!g_bmpSourcePic && !cpImgColSpIsUsed(&G_icsImgColSpc)) && G_cps.isImgSource != cpIMGTYPE_NONE)
				{
					TRACE("EEK! We didn't set up a pic to use!\n");
					return FALSE;
				}
			}

			// ?: If using the builtin pic, should we render it to the screen here?
			// A: Not worth it as we can skip the RGB->ColSpc conversion-process.
		}
	}
	else
	{
#if 0	// We are not doing any colourdepth conversions.
		if(bitmap_color_depth(g_bmpSourcePic) != nDestBPP)	// !!! do not convert the colourdepth - _cpPreCalcImgColSpcVals() should have the original values.
		{
			/* Convert source pic to different colour-depth */

			BITMAP *bmpTemp;

			// TODO: If both images 8-bit and we explicitly told it to convert to another palette (ie. not remap to), do the conversion here.


			/*

			De colour-conversion:


			See _fixup_loaded_bitmap()

			-> 8 (incl 8 -> 8)

			 generate_optimized_palette(bmp, pal, NULL);
			 create_rgb_table(rgb_map, pal, NULL); // if present the table will also vastly accelerate the makecol8() and some create_*_table() functions. RGB tables can be precalculated with the rgbmap utility, or generated at runtime with create_rgb_table(). 
			 blit(bmp, b2, 0, 0, 0, 0, bmp->w, bmp->h);

			8 ->

			 select_palette(pal);
			 blit(bmp, b2, 0, 0, 0, 0, bmp->w, bmp->h);
			 unselect_palette();

			ohne 8:

			  blit(bmp, b2, 0, 0, 0, 0, bmp->w, bmp->h);


			*/



			/* Create new bitmap */
			if(!(bmpTemp=create_bitmap_ex(32 /* nDestBPP */, nDestWidth, nDestHeight) ))
			{
				destroy_bitmap(g_bmpSourcePic);
				g_bmpSourcePic = NULL;
				TRACE("EEK! Could not create a bitmap to convert colourdepth onto.\n");
				return FALSE;
			}


			if(bitmap_color_depth(g_bmpSourcePic)==8)
			{
				select_palette(g_pSourcePicPal);	/* Converting from indexed to a higher depth */
			}

			/* Note: If converting to an indexed depth (even if from another indexed depth),  */
			blit(g_bmpSourcePic, bmpTemp,0,0, 0,0, nDestWidth, nDestHeight);

			if(bitmap_color_depth(g_bmpSourcePic)==8)
			{
				unselect_palette();	/* Converting from indexed to a higher depth */
			}


			destroy_bitmap(g_bmpSourcePic);	/* Done with the old */
			g_bmpSourcePic = bmpTemp;


		}
#endif

		/* Blit the image immediately so we don't have to wait for the pre-calculations to finish */
		cpBlitImageToScreenImmediatelyIfNeeded(g_bmpSourcePic, g_pSourcePicPal, bSetGfxMode, nDestWidth, nDestHeight, nDestBPP);


		// If it's not the desktop and there's no rise-time to full intensity, then there will be a brief period of image without effect.
		// In that case, set gfx mode at last minute. But we can't do that, as we have to find out if we can actually use that mode.

	}




	/* Setup rendering destination */

	if(!g_bmpSourcePic)	// Leave source pic as we've already blitted it. The clear is meant for doing things like clearing progress-messages.
	{
		if(bSetGfxMode)
		{
			// TODO: We might want to wait until after we've built the plasma-tabs in cpInitPlasma() to clear the screen.
			clear_to_color(screen,makecol_depth(nDestBPP, 0x0, 0x0, 0x0));
		}
	}

	if(G_rtRenderingType == cpRENDERTYPE_DOUBLE_BUFFER)
	{
		/* We are doing double-buffering here, so we are creating a memory-bitmap back-buffer to draw onto */
		if(!(G_bmpDest=create_bitmap_ex(nDestBPP, nDestWidth, nDestHeight)))
		{
			if(bSetGfxMode)
			{
				TRACE("Could not set up a back-buffer required for double-buffering - using straight-to-screen rendering instead\n");

				G_rtRenderingType = cpRENDERTYPE_STRAIGHT;
				G_bmpDest = screen;
			}
			else
			{
				TRACE("EEK: Could not set up a back-buffer. As this is required when not setting as GFX mode, we have no choice but to exit.\n");

				destroy_bitmap(g_bmpSourcePic);
				g_bmpSourcePic = NULL;

				return FALSE;
			}
		}

		clear_to_color(G_bmpDest,makecol_depth(nDestBPP, 0x0, 0x0, 0x0));
	}
	else if(G_rtRenderingType == cpRENDERTYPE_TRIPLE_BUFFER || G_rtRenderingType == cpRENDERTYPE_PAGE_FLIP)
	{
		G_bmpDest = g_bmpaPages[g_nPage];

//		clear_to_color(g_bmpaPages[0],makecol_depth(nDestBPP, 0x0, 0x0, 0x0));	// is same as 'screen' - hence commented out.
		clear_to_color(g_bmpaPages[1],makecol_depth(nDestBPP, 0x0, 0x0, 0x0));
		if(G_rtRenderingType == cpRENDERTYPE_TRIPLE_BUFFER)
		{
			clear_to_color(g_bmpaPages[2],makecol_depth(nDestBPP, 0x0, 0x0, 0x0));
		}
	}
	else
	{
		/* cpRENDERTYPE_STRAIGHT */
		// screen cleared eralier - hence no clear_to_color()
		G_bmpDest = screen;	// NOTE: if(!bSetGfxMode), this will always imply G_rtRenderingType==cpRENDERTYPE_DOUBLE_BUFFER.
	}




	/* If we've set up a source image, we'll pre-convert it to our desired colour-space. */
	/* Note that for some types of image source (the built-in pic), it is built directly to G_icsImgColSpc without using a source-image. */
	if(g_bmpSourcePic)
	{
		ASSERT(!cpImgColSpIsUsed(&G_icsImgColSpc))	/* If there's still a source-picture, it means we shouldn't have crated G_icsImgColSpc yet */

		/* Break the source-image down into it's colourspace values */
		if(!cpPreCalcImgColSpcVals(&G_icsImgColSpc, g_bmpSourcePic, g_pSourcePicPal))
		{
			if(G_rtRenderingType == cpRENDERTYPE_DOUBLE_BUFFER)
			{
				destroy_bitmap(G_bmpDest);
				G_bmpDest = NULL;
			}
			
			destroy_bitmap(g_bmpSourcePic);
			g_bmpSourcePic = NULL;
			
			return FALSE;
		}

		// g_bmpSourcePic was created either in _cpSetupPicPreSetGfxMode() or _cpSetupPicPostSetGfxMode()
		destroy_bitmap(g_bmpSourcePic);
		g_bmpSourcePic = NULL;
	}
	else
	{
		ASSERT(cpImgColSpIsUsed(&G_icsImgColSpc) || G_cps.isImgSource == cpIMGTYPE_NONE);
	}


	return TRUE;
}



// \sa cpSetupGfx
void
cpCleanUpGfx(AeBool bSetGfxMode)
{
	cpImgColSpSafeFree(&G_icsImgColSpc);

	if(g_bRGBMapAllocdByChromaPlas)
	{
		free(rgb_map);
		rgb_map = NULL;
	}


	if(G_rtRenderingType == cpRENDERTYPE_TRIPLE_BUFFER)
	{	
		if(g_bmpaPages[0])
		{
			destroy_bitmap(g_bmpaPages[0]);
			g_bmpaPages[0] = NULL;
		}
		
		if(g_bmpaPages[1])
		{
			destroy_bitmap(g_bmpaPages[1]);
			g_bmpaPages[1] = NULL;
		}
		
		if(g_bmpaPages[2])
		{
			destroy_bitmap(g_bmpaPages[2]);
			g_bmpaPages[2] = NULL;
		}
	}
	else if(G_rtRenderingType == cpRENDERTYPE_PAGE_FLIP)
	{
		if(g_bmpaPages[0])
		{
			destroy_bitmap(g_bmpaPages[0]);
			g_bmpaPages[0] = NULL;
		}
		if(g_bmpaPages[1])
		{
			destroy_bitmap(g_bmpaPages[1]);
			g_bmpaPages[1] = NULL;
		}

	}
	else if(G_rtRenderingType == cpRENDERTYPE_DOUBLE_BUFFER)
	{
		if(G_bmpDest)
		{
			destroy_bitmap(G_bmpDest);
			G_bmpDest = NULL;
		}
	}



	if(bSetGfxMode)
	{
		cpStopTimers();


		/* If exiting from the test-mode, this removes the test-'window'. */
		// This seems to cause a problem with the GDI graphics-driver where it ends up waiting for some interrupt to exit.
		// However, if it is called after _cpCleanupChromaPlasProgramTiniInputAndTimer(), there does not seem to be a problem anymore.
//		set_gfx_mode( GFX_TEXT, 0, 0, 0, 0 );
	}
}




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

                               Rendering

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


/* Copies the back-buffer to the screen */
// This won't be called if in preview mode.
static void
_cpCopyBackBufferToScreen(void)
{
	ASSERT(G_rtRenderingType==cpRENDERTYPE_DOUBLE_BUFFER);

	/* Update the palette before we copy the back-buffer to the screen. */
	if(bitmap_color_depth(screen)==8 && G_bPalIsMucky)
	{
		cpSetPalette();
	}


	/* Blit contents of the G_bmpDest buffer onto screen */
	acquire_bitmap(screen);
	blit(G_bmpDest, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	// ?: If G_cps.bConfineEverythingToSquare, can we get away with blitting a smaller area. Or for that matter, if the area being plasma'd is smaller...
	release_bitmap(screen);
}


/* Do a page-flip */
/* To e called at render-end */
// This won't be called if in preview mode.
static void
_cpPageFlip(void)
{
	ASSERT(G_rtRenderingType==cpRENDERTYPE_PAGE_FLIP);


	/* show_video_bitmap(): This function will wait for a vertical retrace (vsync), but not if 'disable_vsync' = yes in the INI file (which means _wait_for_vsync will be 0). */
	if(show_video_bitmap(g_bmpaPages[g_nPage]))
	{
		TRACE("Failed to show the requested video bitmap when page-flipping.\n");
	}

	
	/* We've done the actual page-flip. Make sure the palette is up to date */
	if(bitmap_color_depth(screen)==8 && G_bPalIsMucky)
	{
		cpSetPalette();	// set_palette_range(G_pPal, 0, PAL_SIZE-1, FALSE);
	}


	g_nPage = 1 - g_nPage;
	G_bmpDest = g_bmpaPages[g_nPage];
}


/* Do a request_video_bitmap() when triple buffering */
/* To e called at render-end */
static void
_cpTripleBuffer(void)
{
	ASSERT(G_rtRenderingType==cpRENDERTYPE_TRIPLE_BUFFER);


	do
	{
	}
	while (poll_scroll());

	if(request_video_bitmap(g_bmpaPages[g_nPage]))
	{
		TRACE("Failed to request the video bitmap when triple-buffering.\n");
	}

	if(bitmap_color_depth(screen)==8 && G_bPalIsMucky)
	{
		// WARNING: set_palette_range() with FALSE for the retracesync param causes a tear. If we only need to set the palette once (ie. we're not in funky palette mode), we won't notice this.
		// ?: Should set_palette_range()'s retracesync param be FALSE if _wait_for_vsync==0 ?
		cpSetPalette();	// set_palette_range(G_pPal, 0, PAL_SIZE-1, FALSE);
	}


	g_nPage = (g_nPage+1)%3;
	G_bmpDest = g_bmpaPages[g_nPage];
}


void
cpRenderBegin(AeBool bIsInSSPreviewWindow)
{
	if(G_rtRenderingType==cpRENDERTYPE_STRAIGHT)
	{
		/* If we're rendering straight to the screen, then we should optionally do a vsync (and optionally update the palette) before we do any drawing */
		ASSERT(!bIsInSSPreviewWindow);
		if(_wait_for_vsync)
		{
			vsync();
		}

		if(bitmap_color_depth(screen)==8 && G_bPalIsMucky)
		{
			cpSetPalette();
		}
	}


	acquire_bitmap(G_bmpDest);	// For some reason, placing this before the vsync() sometimes makes it freeze in GFX_GDI (see bit about mutex lock in windows in acquire_bitmap())
}



void
cpRenderEnd(AeBool bIsInSSPreviewWindow)
{
	release_bitmap(G_bmpDest);


	/* Copy the back-buffer to the screen if we have to. */
	if(G_rtRenderingType==cpRENDERTYPE_DOUBLE_BUFFER)
	{
		if(!bIsInSSPreviewWindow)
		{
			if(_wait_for_vsync)
			{
				vsync();
			}

			ASSERT(!bIsInSSPreviewWindow);	// Let's be explicit!
			_cpCopyBackBufferToScreen();
		}
		else
		{
			/* The SS preview window. Leave it up to the OS to copy the back-buffer to the screen. */
			if(bitmap_color_depth(G_bmpDest)==8 && G_bPalIsMucky)	// G_bmpDest instead of screen coz in preview-mode.
			{
				cpSetPalette();
			}
		}
	}


	/* Flip the page if we have to. */
	else if(G_rtRenderingType==cpRENDERTYPE_PAGE_FLIP)
	{
		ASSERT(!bIsInSSPreviewWindow);
		_cpPageFlip();	// Note: This vsync()'s if _wait_for_vsync is set.
	}

	else if(G_rtRenderingType==cpRENDERTYPE_TRIPLE_BUFFER)
	{
		ASSERT(!bIsInSSPreviewWindow);
		_cpTripleBuffer();	// Triple buffering by nature does not wait for vsync()
	}

}

