/***************************************************************************/
/*!
  \file aeconfig.c

  \brief Configuration settings managment

  This contains some high-level configuration settings managment code.
  It uses Allegro's configuration to store things both Allegro-specific and app-specific.

  \author Andrei Ellman

 **************************************************************************/


/*

?: Should szSettingsBuffer be made static-global?

*/


/****************************************************************************
 Includes
 */



/* Libs, etc. */

#ifndef AECONFIG_NO_FLOAT
#include "math.h"	// fabs
#endif

#include <allegro.h>

#include "aeglobal.h"

// These need to be #defined (or not defined) in a header. Or how about the makefile/projectsettings??
#define AECONFIG_NO_INTCOORD2D	/*!< Used to make the int-cooord2Ds prototypes invisible to the compiler. This is to reduce dependencies. */
#define AECONFIG_NO_ANIMPARAMDESC	/*!< Used to make the animparamdesc prototypes invisible to the compiler. This is to reduce dependencies. */

#if !(defined(AECONFIG_NO_INTCOORD2D) && defined(AECONFIG_NO_ANIMPARAMDESC))

#include "aegeom2d.h"	// Needed to define some things in other header files
#include "aegfx.h"	// Needed to define some things in other header files

#endif

#include "aeconfig.h"



/****************************************************************************
 Global Prototypes
 */


/****************************************************************************
 Local (Static) Prototypes
 */


/****************************************************************************
 Local Defines
 */


#define TPREFIX_I	"AECFG: INFO: "		/*!< Module-specific trace-prefix (information). Used to prefix informational trace-messages. */
#define TPREFIX_W	"AECFG: WARNING: "	/*!< Module-specific trace-prefix (warning). Used to prefix trace-messages warning about something. */
#define TPREFIX_E	"AECFG: ERROR: "	/*!< Module-specific trace-prefix (error). Used to prefix trace-messages informing that an error has occured. */




#define AENUMANIMPARAMS	2
#define AENUMCOORD2DVALUES	2




/****************************************************************************
 Local Macros
 */


/****************************************************************************
 Global Variables (across program)
 */


/****************************************************************************
 Local (static) Global Variables
 */




/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

         Helpers for reading values from the configuration settings

 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */


/****************************
 * General purpouse helpers *
 ****************************/

/****************************************************************************/
/*!

 \brief Traces the given config-settings section name.

  The name of the config-settings section will be printed surrounded by square-brackets.
  If the section is the root-section (\p szSection is NULL), the string
  "__root_section__" (without square brackets) will be printed instead.
 
  \param szSection	The config-settings section name to print (NULL means it's the root-section).
 
 */

// ?: static?

void
aeTraceConfigSectionName(AL_CONST char *szSection)
{
	if(szSection)
	{
		TRACE("[");
		TRACE("%s", szSection);
		TRACE("]");
	}
	else
	{
		TRACE("__root_section__");
	}
}



/****************************************************************************/
/*!

 \brief When reading a multi-valued key, this helper function copies the key-value-string (containing the settings) from the config to the settings-buffer.

  If the key is absent, a notification is TRACE'd. A warning is also given if the set of values is too big for the buffer.

  \remarks If a key is absent, the default values will be used instead.

  \param szSettingsBuffer		The string-buffer the key will be copied to.
  \param nSettingsBufferSize	The size (in bytes) of the string-buffer the key will be copied to.
  \param szSection		Which section of the configuration settings to search for the key \p szKeyName in.
  \param szKeyName		The name of the key containing the values (must be in \p szSection).
 
 */

// 
// ?: Should we pass in szSettingsPtr so we can do the get_config_string() outside of this function. This is just to mae things symetirical (either that or create a separate function with set_config_string()).
void
aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(char *szSettingsBuffer, size_t nSettingsBufferSize, AL_CONST char *szSection, AL_CONST char *szKeyName)
{
	char *szDefault = "";	/* Don't bother composing a string of defaults - an empty string will cause the array's values to be loaded with the defaults anyway. */
	AL_CONST char *szSettingsPtr;

	szSettingsPtr = get_config_string(szSection, szKeyName, szDefault);
	if(szSettingsPtr==szDefault)
	{
		/* If the pointers are the same, implies the string-value for the setting szName was not present in the configuration settings */
		TRACE(TPREFIX_W "Config-settings section ");
		aeTraceConfigSectionName(szSection);
		TRACE(": Key '%s' was not found so the default values for this key will be used instead.\n", szKeyName);
	}

	ustrzcpy(szSettingsBuffer, nSettingsBufferSize, szSettingsPtr);

	if(ustrsize(szSettingsBuffer) !=  ustrsize(szSettingsPtr))
	{
		TRACE(TPREFIX_W "Config-settings section ");
		aeTraceConfigSectionName(szSection);
		TRACE(": Key '%s''s value is too big for the value-buffer. Consider using a larger buffer (or less whitespace in this value).\n", szKeyName);
	}
}



/****************************************************************************/
/*!

 \brief When reading a multi-valued key, TRACEs the start of a warning-message that not all values in the key were used.

  \note This should only be called if not all expected values were present in the multi-valued key.

  \note This does not print a newline. This is so the default values can be printed on the same line.

  \param szSection		Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName		The name of the key where not all values were found.
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful)
  \param nNumValuesFoundInKey	The number of values that were found when reading the key.
 
 */

static void
_aeWarnThatNotAllValuesUsed(AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szValDesc, unsigned int nNumValuesFoundInKey, unsigned int nNumValuesToSerialise)
{
	ASSERT(nNumValuesFoundInKey!=nNumValuesToSerialise);	/* Check to see if we're falsely callling this function. */

	/* Warn if not all values used */
	TRACE(TPREFIX_I "Config-settings section ");
	aeTraceConfigSectionName(szSection);
	TRACE(": The multi-valued key '%s' only uses %u %s values when it should have %u. The following defaults are being used for the remaining values:", szKeyName, nNumValuesFoundInKey, szValDesc, nNumValuesToSerialise);
}



/****************************************************************************/
/*!

 \brief When reading a multi-valued key, if there are more values in the key than expected, TRACE a warning-message that there is some extra data.
 
  \note Currently, this also printsthe warning if the extra data is just whitespace.

  \remarks szKeyDesc or szValDesc can be the empty string, but probably not a good idea to set both to the empty string.

  \param szSettingsBufferPosPostSerialise	The current position of the settings buffer. If there are more values, the warning is TRACE'd.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName		The name of the key where there may be extra values present.
  \param szKeyDesc		The description of the key for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful)
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful)
  \param nMaxNumValuesToSerialise	The maximum number of values that are supposed to be in this key.
 
 */

void /* AeBool? */
aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(char *szSettingsBufferPosPostSerialise, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szKeyDesc, AL_CONST char *szValDesc, unsigned int nMaxNumValuesToSerialise)
{
	if(szSettingsBufferPosPostSerialise[0]!='\0')	// TODO; Check for non-witespace until \n or \0
	{
		/* Warn if toomany values */
		TRACE(TPREFIX_I "Config-settings section ");
		aeTraceConfigSectionName(szSection);
		TRACE(": Warning: The %s key '%s' has toomany %s values (or contains trailing whitespace). Only the first %d used\n", szKeyDesc, szKeyName, szValDesc, nMaxNumValuesToSerialise); // ?: Should I print the remainder of the string or howmany more values it can find?
	}
}



/****************************************************************************/
/*!

 \brief When writing, use this function to format a multi-valued settings-buffer with an extra tab.

  If the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \param szpSettingsPos[in,out]		The current position in the settings buffer (where the tab-character will be coped to). This is then advanced to the end of the string.
  \param npSettingsBufferRemainingSize[in,out]	The remaining size (in bytes) of the settings buffer. This is decremented by the number of bytes written.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in (used to print trace-messages).
  \param szKeyName		The name of the current key (used to print trace-messages).
 
 */

void
aeAddTabToValueBuffer(char **szpSettingsPos, int *npSettingsBufferRemainingSize, AL_CONST char *szSection, AL_CONST char *szKeyName)
{
	/* Add an extra tab */
	int nNumCharsThatShouldHaveBeenPrinted = uszprintf(*szpSettingsPos, *npSettingsBufferRemainingSize, "\t");

	if(nNumCharsThatShouldHaveBeenPrinted==ustrlen(*szpSettingsPos))
	{
		/* The value was written OK */
		int nSizePrinted = ustrsize(*szpSettingsPos);
		*szpSettingsPos += nSizePrinted;	/* Advance the pointer to the next position */
		*npSettingsBufferRemainingSize -= nSizePrinted;	
	}
	else
	{
		TRACE(TPREFIX_W "Config-settings section ");
		aeTraceConfigSectionName(szSection);
		TRACE(": When writing key '%s''s value and we attempted to add a tab-character, we reached the end of the value-buffer. Consider using a larger buffer (or less whitespace in this multi-valued setting).\n", szKeyName);
	}
}



/****************************************************************************/
/*!

 \brief Serialises an integer from the configuration settings.

  An integer vaule is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultInt is used instead.

  \param npIntVal[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultInt if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param nDefaultInt		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

// ?: Should it return a value for failure? Methinx this can't fail coz writing to mem.
// But it could also return a boolean to indicate if it was changed from the default (this could also be done by comparing the read value to the default.)

// If we use a string, can check for presence and do other bases by prefixing 0x to the value.

// ?: -> aeConfigAccessInt ?
void
aeSerialiseConfigInt(int *npIntVal, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST int nDefaultInt, AeBool bWrite)
{
	if(bWrite)
	{
		set_config_int(szSection, szName, *npIntVal);
	}
	else
	{
		*npIntVal = get_config_int(szSection, szName, nDefaultInt);

		/* Used to notify if a value is different from default */
		if(*npIntVal!=nDefaultInt)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%d' which is different from the default value '%d'.\n", szName, *npIntVal, nDefaultInt);
		}
	}
}



/****************************************************************************/
/*!

 \brief Serialises an unsigned integer from the configuration settings.

  An unsigned integer vaule is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultUInt is used instead.

  \param npUIntVal[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultUInt if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param nDefaultUInt		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

// ?: Should it return a value for failure? Methinx this can't fail coz writing to mem.
// But it could also return a boolean to indicate if it was changed from the default (this could also be done by comparing the read value to the default.)

// If we use a string, can check for presence and do other bases by prefixing 0x to the value.

// ?: -> aeConfigAccessInt ?
void
aeSerialiseConfigUInt(unsigned int *npUIntVal, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST unsigned int nDefaultUInt, AeBool bWrite)
{
	if(bWrite)
	{
		if(*npUIntVal>INT_MAX)
		{
			TRACE(TPREFIX_W "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Attempting to write an unsigned int value '%u' to key Key '%s' which is greater than the signed int maximum of %d (Allegro does not provide an eqivalent for set_config_int() ", *npUIntVal, szName, INT_MAX);
		}
		set_config_int(szSection, szName, *npUIntVal);
	}
	else
	{
		int nTmpVal;
		
		nTmpVal = get_config_int(szSection, szName, nDefaultUInt);
		if(nTmpVal < 0 )
		{
			TRACE(TPREFIX_W "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a negative value '%d' which cannot be used for an unsigned int value", szName, nTmpVal);
		}

		*npUIntVal = nTmpVal;

		/* Used to notify if a value is different from default */
		if(*npUIntVal!=nDefaultUInt)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%d' which is different from the default value '%d'.\n", szName, *npUIntVal, nDefaultUInt);
		}
	}
}



/****************************************************************************/
/*!

 \brief Serialises an 8-bit unsigned integer from the configuration settings.

  An 8-bit unsigned integer vaule is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultUInt8 is used instead.

  \param npUInt8Val[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultUInt8 if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param nDefaultUInt8		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

// ?: Should it return a value for failure? Methinx this can't fail coz writing to mem.
// But it could also return a boolean to indicate if it was changed from the default (this could also be done by comparing the read value to the default.)

// If we use a string, can check for presence and do other bases by prefixing 0x to the value.

void
aeSerialiseConfigUInt8(uint8_t *npUInt8Val, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST uint8_t nDefaultUInt8, AeBool bWrite)
{
	if(bWrite)
	{
		set_config_int(szSection, szName, *npUInt8Val);
	}
	else
	{
		int nTmpVal;

		nTmpVal = get_config_int(szSection, szName, nDefaultUInt8);
		if(nTmpVal > UCHAR_MAX || nTmpVal < 0 )
		{
			TRACE(TPREFIX_W "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%d' which is outside the range of an unsigned 8-bit value (%u .. %u)", szName, nTmpVal, 0, UCHAR_MAX);
		}
		*npUInt8Val = nTmpVal;

		/* Used to notify if a value is different from default */
		if(*npUInt8Val!=nDefaultUInt8)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%u' which is different from the default value '%u'.\n", szName, *npUInt8Val, nDefaultUInt8);
		}
	}
}



/****************************************************************************/
/*!

 \brief Serialises a 16-bit unsigned integer from the configuration settings.

  A 16-bit unsigned integer vaule is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultUInt16 is used instead.

  \param npUInt16Val[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultUInt16 if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection				Which section of the configuration settings to search for the key \p szName in.
  \param szName					The name of the key containing the value (must be in \p szSection).
  \param nDefaultUInt16			When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite					Wether to do a read or write.

 */

// ?: Should it return a value for failure? Methinx this can't fail coz writing to mem.
// But it could also return a boolean to indicate if it was changed from the default (this could also be done by comparing the read value to the default.)

// If we use a string, can check for presence and do other bases by prefixing 0x to the value.

void
aeSerialiseConfigUInt16(uint16_t *npUInt16Val, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST uint16_t nDefaultUInt16, AeBool bWrite)
{
	if(bWrite)
	{
		set_config_int(szSection, szName, *npUInt16Val);
	}
	else
	{
		int nTmpVal;

		nTmpVal = get_config_int(szSection, szName, nDefaultUInt16);
		if(nTmpVal > USHRT_MAX || nTmpVal < 0 )
		{
			TRACE(TPREFIX_W "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%d' which is outside the range of an unsigned 16-bit value (%u .. %u)", szName, nTmpVal, 0, USHRT_MAX);
		}
		*npUInt16Val = nTmpVal;

		/* Used to notify if a value is different from default */
		if(*npUInt16Val!=nDefaultUInt16)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%u' which is different from the default value '%u'.\n", szName, *npUInt16Val, nDefaultUInt16);
		}
	}
}



/****************************************************************************/
/*!

 \brief Serialises a boolean value from the configuration settings.

  A boolean vaule is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefault is used instead.

  The boolean value is stored in the configuration settings as an integer.
  If it's value is zero, the boolean value is \c FALSE, and
  if the value is non-sero, the boolean value is TRUE.

  \param bpBoolVal[in,out]	When reading, this points to where to copy the read value (or where to copy \p bDefaultBool if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param bDefaultBool		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

// ?: Should it return a value for failure? Methinx this can't fail coz writing to mem.
// But it could also return a boolean to indicate if it was changed from the default (this could also be done by comparing the read value to the default.)

// If we use a string, can check for presence and do other means of specifying BOOLs such as "TRUE/FALSE", "YES/NO".

void
aeSerialiseConfigBool(AeBool *bpBoolVal, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST AeBool bDefaultBool, AeBool bWrite)
{
	if(bWrite)
	{
		set_config_int(szSection, szName, (*bpBoolVal ? 1 : 0));
		// Could also do a string of "TRUE"/"FALSE", "ON"/"OFF".
		// Perhaps pass in an extra paramater 'nBoolFormat': 0=default, 1={0,1} -1={0,-1} 2={"TRUE","FALSE"} 3={"YES","NO"} 4={"ON","OFF"}. Or even two params for custom-strings (revert to setting '0' if at least one of them is NULL).
		// Of course, would need a mechanism of substituting different-language versions of the strings and allowing a means to parse these strings in case of a language-switch
		// (apart from storing all languages, could use English or just plain {0,1} as an intermediate step, or force a write on language-change. Perhaps the language-id could be part of the config-settings but then, hand-editing the language would invalidate all textual boolean settings.).
	}
	else
	{
		*bpBoolVal = (get_config_int(szSection, szName, bDefaultBool) ? TRUE : FALSE);

		/* Used to notify if a value is different from default */
		if(*bpBoolVal!=bDefaultBool)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%d' which is different from the default value '%d'.\n", szName, (*bpBoolVal ? 1 : 0), (bDefaultBool ? 1 : 0));
		}
	}
}



/****************************************************************************/
/*!

 \brief Serialises an Allegro fixed-point value from the configuration settings.

  An Allegro fixed-point vaule (16.16) (in hexadecimal format) is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultFixed is used instead.

  \param npFixedVal[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultFixed if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param nDefaultFixed		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

// ?: Should it return a value for failure? Methinx this can't fail coz writing to mem.
// But it could also return a boolean to indicate if it was changed from the default (this could also be done by comparing the read value to the default.)

// If we use a string, can check for presence and do other bases by not prefixing 0x to the value.

void
aeSerialiseConfigHexFixed(fixed *npFixedVal, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST fixed nDefaultFixed, AeBool bWrite)
{
	ASSERT(sizeof(int)==sizeof(fixed));	// Saves us from having to do an extra type-conversion.

	if(bWrite)
	{
		set_config_hex(szSection, szName, *npFixedVal);
	}
	else
	{
		*npFixedVal = get_config_hex(szSection, szName, nDefaultFixed);

		/* Used to notify if a value is different from default */
		if(*npFixedVal!=nDefaultFixed)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '0x%X' which is different from the default value '0x%X'\n", szName, *npFixedVal, nDefaultFixed);
		}
	}
}



#ifndef AECONFIG_NO_FLOAT

/****************************************************************************/
/*!

 \brief Serialises a floating-point value from the configuration settings.

  A floating-point vaule is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultInt is used instead.

  \param npFloatVal[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultFloat if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param nDefaultInt		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

void
aeSerialiseConfigFloat(float *npFloatVal, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST float nDefaultFloat, AeBool bWrite)
{
	if(bWrite)
	{
		set_config_float(szSection, szName, *npFloatVal);
	}
	else
	{
		*npFloatVal = get_config_float(szSection, szName, nDefaultFloat);

		/* Used to notify if a value is different from default */
		if(fabs(*npFloatVal - nDefaultFloat)>0.00001)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%f' which is different from the default value '%f'.\n", szName, *npFloatVal, nDefaultFloat);
		}
	}
}



/****************************************************************************/
/*!

 \brief Serialises a double-precision floating-point value from the configuration settings.

  A double-precision floating-point is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefaultInt is used instead.

  \param npDoubleVal[in,out]	When reading, this points to where to copy the read value (or where to copy \p nDefaultDouble if the key could not be found or the key's value could not be read). When writing, this points to the value to write.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param nDefaultInt		When reading, this is the default value to return if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

void
aeSerialiseConfigDouble(double *npDoubleVal, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST double nDefaultDouble, AeBool bWrite)
{
	if(bWrite)
	{
		// ?: Should we do a warning about underflow? A: Once we ditch get_config_float and deal with the string directly, we wouldn't need to (but instead, we'd have to warn in aeSerialiseConfigFloat() about over/underflow).
		set_config_float(szSection, szName, *npDoubleVal);
	}
	else
	{
		*npDoubleVal = get_config_float(szSection, szName, nDefaultDouble);

		/* Used to notify if a value is different from default */
		if(fabs(*npDoubleVal - nDefaultDouble)>0.0000000000001)
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value '%f' which is different from the default value '%f'.\n", szName, *npDoubleVal, nDefaultDouble);
		}
	}
}

#endif



/****************************************************************************/
/*!

 \brief Serialises a string from the configuration settings.

  An string (buffer) is serialised from the key specified by \p szName in the
  section specified by \p szSection.
  When reading, if the value cannot be read, a default value \p nDefault is used instead.
  When reading, warn on buffer overflow.

  \param szValueBuffer		When reading, this points to where to copy the read string to (or where to copy \p szDefaultString if the key could not be found or the key's value could not be read). When writing, this points to the buffer containing the string to write.
  \param nValueBufferSize	The size in bytes of the buffer pointed to by szValueBuffer. When reading, If the length of the string to read (including the terminating \0) exceeds this length, a truncated and 0-terminated string is copied. This paramater is ignored when writing.
  \param szSection			Which section of the configuration settings to search for the key \p szName in.
  \param szName				The name of the key containing the value (must be in \p szSection).
  \param szDefaultString	The default value to copy to the buffer if the key cannot be found or the key's value cannot be read (when writing, this paramater is ignored).
  \param bWrite				Wether to do a read or write.

 */

void
aeSerialiseConfigString(char *szValueBuffer, int nValueBufferSize, AL_CONST char *szSection, AL_CONST char *szName, AL_CONST char *szDefaultString, AeBool bWrite)
{
	if(bWrite)
	{
		set_config_string(szSection, szName, szValueBuffer);
	}
	else
	{
		AL_CONST char *szValueInConfig;

		szValueInConfig = get_config_string(szSection, szName, szDefaultString);

		if(szValueInConfig == szDefaultString)
		{
			/* If the pointers are the same, implies the string-value for the setting szName was not present in the configuration settings */
			/* NOTE: no equivalent 'not-found' for non-string datatypes */
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' was not found so the default value \"%s\" will be used.\n", szName, szDefaultString);
		}
		else if(ustrcmp(szValueInConfig,szDefaultString))
		{
			TRACE(TPREFIX_I "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s' has a value \"%s\" which is different from the default value \"%s\".\n", szName, szValueInConfig, szDefaultString);
		}

		ustrzcpy(szValueBuffer, nValueBufferSize, szValueInConfig);

		if(ustrsize(szValueBuffer) !=  ustrsize(szValueInConfig))
		{
			TRACE(TPREFIX_W "Config-settings section ");
			aeTraceConfigSectionName(szSection);
			TRACE(": Key '%s''s value is too big for the value-buffer. Consider using a larger buffer.\n", szName);
		}
	}
}



/********************
 * non-atomic types *
 ********************/


// Allegro does not define ustrtoul() so pinch the code for ustrtol() and turn into unsigned version
/* ustrtoul:
 *  Unicode-aware version of the ANSI strtoul() function. Note the
 *  nicely bodged implementation :-)
 */
// -> aeglobal.c ?
unsigned long
ustrtoul(AL_CONST char *s, char **endp, int base)
{
   char tmp[64];
   char *myendp;
   unsigned long ret;
   char *t;
   ASSERT(s);

   t = uconvert_toascii(s, tmp);

   ret = strtoul(t, &myendp, base);

   if (endp)
      *endp = (char *)s + uoffset(s, (long)myendp - (long)t);

   return ret;
}




// -> aeglobal.c

/****************************************************************************/
/*!

 \brief Serialises a list of int values from a settings-buffer (a string-buffer containing the settings) and when reading, stores result in an array of ints, and when writing, converts the array of ints into a string.

  A range of \p nNumValuesToSerialise values are serialised starting from the element at the position in the settings-buffer pointed to by \p szpSettingsPos.
  When reading, if a value cannot be serialised or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \note When reading, The \p szpSettingsPos paramater does not have to be at the first element
  of an array (which is useful if groups of values in the list are to be processed separately).
  When writing, the values are just appended to the end of the buffer.

  \sa aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfAeBoolsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfIntsFromConfigListAndNotifyIfChangesFromDefault()

  \param naIntVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szpSettingsPos[in,out]		The current position in the settings-buffer which contains stringified integers (when reading) or the position to stringify the integers to (when writing). This is then advanced to the end of the string.
  \param npSettingsBufferRemainingSize[in,out]	When writing, this is the remaining size (in bytes) of the settings buffer. This is decremented by the number of bytes written. When reading, this is ignored.
  \param nNumValuesToSerialise	The number of values to serialise in \p naIntVals. This should not be greater than the number of elements in \p naIntVals. When reading, if there aren't enough values remaining in \p szpSettingsPos, the remainder of the values are filled with their defaults in \p naDefaultIntVals.
  \param naDefaultIntVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultIntVals will be copied to the value at the same position and onwards in \p naIntVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naIntVals.
  \param bWrite			Wether to do a read or write.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in (used to print trace-messages).
  \param szKeyName		The name of the current key (used to print trace-messages).
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

// ?: Should this return the number of values read from the string array (ie. the default was not used)?
void
aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault(int *naIntVals, char **szpSettingsPos, int *npSettingsBufferRemainingSize, unsigned int nNumValuesToSerialise, AL_CONST int *naDefaultIntVals, AeBool bWrite, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szValDesc)
{
	unsigned int nI;

	if(bWrite)
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			int nNumCharsThatShouldHaveBeenPrinted = uszprintf(*szpSettingsPos, *npSettingsBufferRemainingSize, " %d", naIntVals[nI]);

			if(nNumCharsThatShouldHaveBeenPrinted==ustrlen(*szpSettingsPos))
			{
				/* The value was written OK */
				int nSizePrinted = ustrsize(*szpSettingsPos);
				*szpSettingsPos += nSizePrinted;	/* Advance the pointer to the next position */
				*npSettingsBufferRemainingSize -= nSizePrinted;
			}
			else
			{
				TRACE(TPREFIX_W "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": When writing key '%s''s value and we attempted to add a '%d', we reached the end of the value-buffer. Consider using a larger buffer (or less whitespace in this multi-valued setting).\n", szKeyName, naIntVals[nI]);
			}
		}
	}
	else
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			// ustrtol: If the value of base is zero, the syntax expected is similar to that of integer constants, which is formed by a succession of:
			//  * An optional prefix indicating octal or hexadecimal base ("0" or "0x" respectively)
			//  * A sequence of decimal digits (if no base prefix was specified) or either octal or hexadecimal digits if a specific prefix is present

			char *cpTailPtr;
			naIntVals[nI] = ustrtol(*szpSettingsPos,&cpTailPtr,0);	/* '#', newline, \0 or any other non-numeric value will fail. */
			if(*szpSettingsPos==cpTailPtr)
			{
				/* This value is not present in the string */
				// ?: Should we check for whitespace / newline / \0 vs. an un-parsable value?
				break;
			}
			*szpSettingsPos=cpTailPtr;	/* Advance the pointer to the next position */

			if(naIntVals[nI]!=naDefaultIntVals[nI])
			{
				TRACE(TPREFIX_I "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": The multi-valued key '%s' has a %s value at position %d of '%d' which is different from the default value '%d'\n", szKeyName, szValDesc, nI, naIntVals[nI], naDefaultIntVals[nI]);
			}
		}

		if(nI<nNumValuesToSerialise)
		{
			/* In case we did not read all values from the string, warn and use the defaults instead. */

			_aeWarnThatNotAllValuesUsed(szSection, szKeyName, szValDesc, nI, nNumValuesToSerialise);

			/* Copy the defaults */
			for(;nI<nNumValuesToSerialise;nI++)
			{
				naIntVals[nI] = naDefaultIntVals[nI];
				TRACE(" %d", naDefaultIntVals[nI]);
			}

			TRACE("\n");	/* Finish the line */
		}
	}

}



// -> aeglobal.c

/****************************************************************************/
/*!

 \brief Serialises a list of unsigned int values from a settings-buffer (a string-buffer containing the settings) and when reading, stores result in an array of unsigned ints, and when writing, converts the array of unsigned ints into a string.

  A range of \p nNumValuesToSerialise values are serialised starting from the element \p szaSettings points to.
  When reading, if a value cannot be serialised or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.

  \note When reading, The \p szpSettingsPos paramater does not have to be at the first element
  of an array (which is useful if groups of values in the list are to be processed separately).
  When writing, the values are just appended to the end of the buffer.

  \sa aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfAeBoolsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromConfigListAndNotifyIfChangesFromDefault()

  \param naUIntVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szpSettingsPos[in,out]		The current position in the settings-buffer which contains stringified unsigned integers (when reading) or the position to stringify the unsigned integers to (when writing). This is then advanced to the end of the string.
  \param npSettingsBufferRemainingSize[in,out]	When writing, this is the remaining size (in bytes) of the settings buffer. This is decremented by the number of bytes written. When reading, this is ignored.
  \param nNumValuesToSerialise	The number of values to serialise in \p naUIntVals. This should not be greater than the number of elements in \p naUIntVals. When reading, if there aren't enough values remaining in \p szpSettingsPos, the remainder of the values are filled with their defaults in \p naDefaultIntVals.
  \param naDefaultUIntVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultUIntVals will be copied to the value at the same position and onwards in \p naUIntVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naIntVals.
  \param bWrite			Wether to do a read or write.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in (used to print trace-messages).
  \param szKeyName		The name of the current key (used to print trace-messages).
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

// ?: Should this return the number of values read from the string array (ie. the default was not used)?
void
aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault(unsigned int *naUIntVals, char **szpSettingsPos, int *npSettingsBufferRemainingSize, unsigned int nNumValuesToSerialise, AL_CONST unsigned int *naDefaultUIntVals, AeBool bWrite, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szValDesc)
{
	unsigned int nI;

	if(bWrite)
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			int nNumCharsThatShouldHaveBeenPrinted = uszprintf(*szpSettingsPos, *npSettingsBufferRemainingSize, " %u", naUIntVals[nI]);

			if(nNumCharsThatShouldHaveBeenPrinted==ustrlen(*szpSettingsPos))
			{
				int nSizePrinted = ustrsize(*szpSettingsPos);	/* Advance the pointer to the next position */
				*szpSettingsPos += nSizePrinted;
				*npSettingsBufferRemainingSize -= nSizePrinted;
			}
			else
			{
				TRACE(TPREFIX_W "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": When writing key '%s''s value and we attempted to add a '%u', we reached the end of the value-buffer. Consider using a larger buffer (or less whitespace in this multi-valued setting).\n", szKeyName, naUIntVals[nI]);
			}
		}
	}
	else
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			// ustrtoul: If the value of base is zero, the syntax expected is similar to that of integer constants, which is formed by a succession of:
			//  * An optional prefix indicating octal or hexadecimal base ("0" or "0x" respectively)
			//  * A sequence of decimal digits (if no base prefix was specified) or either octal or hexadecimal digits if a specific prefix is present

			char *cpTailPtr;
			naUIntVals[nI] = ustrtoul(*szpSettingsPos,&cpTailPtr,0);	/* '#', newline, \0 or any other non-numeric value will fail. */
			if(*szpSettingsPos==cpTailPtr)
			{
				/* This value is not present in the string */
				// ?: Should we check for whitespace / newline / \0 vs. an un-parsable value?
				break;
			}
			*szpSettingsPos=cpTailPtr;	/* Advance the string to the next position */

			if(naUIntVals[nI]!=naDefaultUIntVals[nI])
			{
				TRACE(TPREFIX_I "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": The multi-valued key '%s' has a %s value at position %d of '%u' which is different from the default value '%u'\n", szKeyName, szValDesc, nI, naUIntVals[nI], naDefaultUIntVals[nI]);
			}
		}

		if(nI<nNumValuesToSerialise)
		{
			/* In case we did not read all values from the string, warn and use the defaults instead. */

			_aeWarnThatNotAllValuesUsed(szSection, szKeyName, szValDesc, nI, nNumValuesToSerialise);

			/* Copy the defaults */
			for(;nI<nNumValuesToSerialise;nI++)
			{
				naUIntVals[nI] = naDefaultUIntVals[nI];
				TRACE(" %u", naDefaultUIntVals[nI]);
			}

			TRACE("\n");	/* Finish the line */
		}
	}

}



// -> aeglobal.c

/****************************************************************************/
/*!

 \brief Serialises a list of boolean values from a settings-buffer (a string-buffer containing the settings) and when reading, stores result in an array of AeBools, and when writing, converts the array of AeBools into a string.

  A range of \p nNumValuesToSerialise values are serialised starting from the element at the position in the settings-buffer pointed to by \p szpSettingsPos.
  When reading, if a value cannot be serialised or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \note When reading, The \p szpSettingsPos paramater does not have to be at the first element
  of an array (which is useful if groups of values in the list are to be processed separately).
  When writing, the values are just appended to the end of the buffer.

  \sa aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault()

  \param baBoolVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szpSettingsPos[in,out]		The current position in the settings-buffer which contains stringified boolean values (when reading) or the position to stringify the boolean values to (when writing). This is then advanced to the end of the string.
  \param npSettingsBufferRemainingSize[in,out]	When writing, this is the remaining size (in bytes) of the settings buffer. This is decremented by the number of bytes written. When reading, this is ignored.
  \param nNumValuesToSerialise	The number of values to serialise in \p baBoolVals. This should not be greater than the number of elements in \p baBoolVals. When reading, if there aren't enough values remaining in \p szpSettingsPos, the remainder of the values are filled with their defaults in \p baDefaultBoolVals.
  \param baDefaultBoolVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p baDefaultBoolVals will be copied to the value at the same position and onwards in \p baBoolVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p baBoolVals.
  \param bWrite			Wether to do a read or write.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in (used to print trace-messages).
  \param szKeyName		The name of the current key (used to print trace-messages).
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

// TODO: Check for presence and do other means of specifying BOOLs such as "TRUE/FALSE", "YES/NO".

// ?: Should this return the number of values read from the string array (ie. the default was not used)?
void
aeSerialiseArrayOfBoolsFromStringAndNotifyIfChangesFromDefault(AeBool *baBoolVals, char **szpSettingsPos, int *npSettingsBufferRemainingSize, unsigned int nNumValuesToSerialise, AL_CONST AeBool *baDefaultBoolVals, AeBool bWrite, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szValDesc)
{
	unsigned int nI;

	if(bWrite)
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			int nNumCharsThatShouldHaveBeenPrinted = uszprintf(*szpSettingsPos, *npSettingsBufferRemainingSize, " %d", (baBoolVals[nI] ? 1 : 0));

			if(nNumCharsThatShouldHaveBeenPrinted==ustrlen(*szpSettingsPos))
			{
				/* The value was written OK */
				int nSizePrinted = ustrsize(*szpSettingsPos);
				*szpSettingsPos += nSizePrinted;	/* Advance the pointer to the next position */
				*npSettingsBufferRemainingSize -= nSizePrinted;
			}
			else
			{
				TRACE(TPREFIX_W "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": When writing key '%s''s value and we attempted to add a '%d', we reached the end of the value-buffer. Consider using a larger buffer (or less whitespace in this multi-valued setting).\n", szKeyName, (baBoolVals[nI] ? 1 : 0));
			}
		}
	}
	else
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			// ustrtol: If the value of base is zero, the syntax expected is similar to that of integer constants, which is formed by a succession of:
			//  * An optional prefix indicating octal or hexadecimal base ("0" or "0x" respectively)
			//  * A sequence of decimal digits (if no base prefix was specified) or either octal or hexadecimal digits if a specific prefix is present

			char *cpTailPtr;
			baBoolVals[nI] = (ustrtol(*szpSettingsPos,&cpTailPtr,0)? TRUE : FALSE);	/* '#', newline, \0 or any other non-numeric value will fail. */
			if(*szpSettingsPos==cpTailPtr)
			{
				/* This value is not present in the string */
				// ?: Should we check for whitespace / newline / \0 vs. an un-parsable value?
				break;
			}
			*szpSettingsPos=cpTailPtr;	/* Advance the pointer to the next position */

			if(baBoolVals[nI]!=baDefaultBoolVals[nI])
			{
				TRACE(TPREFIX_I "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": The multi-valued key '%s' has a %s value at position %d of '%d' which is different from the default value '%d'\n", szKeyName, szValDesc, nI, (baBoolVals[nI] ? 1 : 0), (baDefaultBoolVals[nI] ? 1 : 0));
			}
		}

		if(nI<nNumValuesToSerialise)
		{
			/* In case we did not read all values from the string, warn and use the defaults instead. */

			_aeWarnThatNotAllValuesUsed(szSection, szKeyName, szValDesc, nI, nNumValuesToSerialise);

			/* Copy the defaults */
			for(;nI<nNumValuesToSerialise;nI++)
			{
				baBoolVals[nI] = baDefaultBoolVals[nI];
				TRACE(" %d", (baDefaultBoolVals[nI] ? 1 : 0));
			}

			TRACE("\n");	/* Finish the line */
		}
	}

}



#ifndef AECONFIG_NO_FLOAT

// -> aeglobal.c

/****************************************************************************/
/*!

 \brief Serialises a list of float values from a settings-buffer (a string-buffer containing the settings) and when reading, stores result in an array of floats, and when writing, converts the array of floats into a string.

  A range of \p nNumValuesToSerialise values are serialised starting from the element at the position in the settings-buffer pointed to by \p szpSettingsPos.
  When reading, if a value cannot be serialised or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \note When reading, The \p szpSettingsPos paramater does not have to be at the first element
  of an array (which is useful if groups of values in the list are to be processed separately).
  When writing, the values are just appended to the end of the buffer.

  \sa aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfAeBoolsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromConfigListAndNotifyIfChangesFromDefault()

  \param naFloatVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szpSettingsPos[in,out]		The current position in the settings-buffer which contains stringified floats (when reading) or the position to stringify the floats to (when writing). This is then advanced to the end of the string.
  \param npSettingsBufferRemainingSize[in,out]	When writing, this is the remaining size (in bytes) of the settings buffer. This is decremented by the number of bytes written. When reading, this is ignored.
  \param nNumValuesToSerialise	The number of values to serialise in \p naFloatVals. This should not be greater than the number of elements in \p naFloatVals. When reading, if there aren't enough values remaining in \p szpSettingsPos, the remainder of the values are filled with their defaults in \p naDefaultFloatVals.
  \param naDefaultFloatVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultFloatVals will be copied to the value at the same position and onwards in \p naFloatVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naFloatVals.
  \param bWrite			Wether to do a read or write.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in (used to print trace-messages).
  \param szKeyName		The name of the current key (used to print trace-messages).
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

// ?: Should this return the number of values read from the string array (ie. the default was not used)?
void
aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault(float *naFloatVals, char **szpSettingsPos, int *npSettingsBufferRemainingSize, unsigned int nNumValuesToSerialise, AL_CONST float *naDefaultFloatVals, AeBool bWrite, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szValDesc)
{
	unsigned int nI;

	if(bWrite)
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			int nNumCharsThatShouldHaveBeenPrinted = uszprintf(*szpSettingsPos, *npSettingsBufferRemainingSize, " %f", naFloatVals[nI]);	// ?: Use '%g' instead of '%f'?

			if(nNumCharsThatShouldHaveBeenPrinted==ustrlen(*szpSettingsPos))
			{
				/* The value was written OK */
				int nSizePrinted = ustrsize(*szpSettingsPos);
				*szpSettingsPos += nSizePrinted;	/* Advance the pointer to the next position */
				*npSettingsBufferRemainingSize -= nSizePrinted;
			}
			else
			{
				TRACE(TPREFIX_W "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": When writing key '%s''s value and we attempted to add a '%f', we reached the end of the value-buffer. Consider using a larger buffer (or less whitespace in this multi-valued setting).\n", szKeyName, naFloatVals[nI]);
			}
		}
	}
	else
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			char *cpTailPtr;
			naFloatVals[nI] = (float) ustrtod(*szpSettingsPos,&cpTailPtr);	/* '#', newline, \0 or any other non-numeric value will fail. */
			if(*szpSettingsPos==cpTailPtr)
			{
				/* This value is not present in the string */
				// ?: Should we check for whitespace / newline / \0 vs. an un-parsable value?
				break;
			}
			*szpSettingsPos=cpTailPtr;	/* Advance the pointer to the next position */

			if(fabs(naFloatVals[nI] - naDefaultFloatVals[nI])>0.00001)
			{
				TRACE(TPREFIX_I "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": The multi-valued key '%s' has a %s value at position %d of '%f' which is different from the default value '%f'\n", szKeyName, szValDesc, nI, naFloatVals[nI], naDefaultFloatVals[nI]);
			}
		}

		if(nI<nNumValuesToSerialise)
		{
			/* In case we did not read all values from the string, warn and use the defaults instead. */

			_aeWarnThatNotAllValuesUsed(szSection, szKeyName, szValDesc, nI, nNumValuesToSerialise);

			/* Copy the defaults */
			for(;nI<nNumValuesToSerialise;nI++)
			{
				naFloatVals[nI] = naDefaultFloatVals[nI];
				TRACE(" %f", naDefaultFloatVals[nI]);
			}

			TRACE("\n");	/* Finish the line */
		}
	}

}



// -> aeglobal.c

/****************************************************************************/
/*!

 \brief Serialises a list of double values from a settings-buffer (a string-buffer containing the settings) and when reading, stores result in an array of doubles, and when writing, converts the array of doubles into a string.

  A range of \p nNumValuesToSerialise values are serialised starting from the element at the position in the settings-buffer pointed to by \p szpSettingsPos.
  When reading, if a value cannot be serialised or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \note When reading, The \p szpSettingsPos paramater does not have to be at the first element
  of an array (which is useful if groups of values in the list are to be processed separately).
  When writing, the values are just appended to the end of the buffer.

  \sa aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfAeBoolsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromConfigListAndNotifyIfChangesFromDefault()

  \param naDoubleVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szpSettingsPos[in,out]		The current position in the settings-buffer which contains stringified doubles (when reading) or the position to stringify the doubles to (when writing). This is then advanced to the end of the string.
  \param npSettingsBufferRemainingSize[in,out]	When writing, this is the remaining size (in bytes) of the settings buffer. This is decremented by the number of bytes written. When reading, this is ignored.
  \param nNumValuesToSerialise	The number of values to serialise in \p naDoubleVals. This should not be greater than the number of elements in \p naDoubleVals. When reading, if there aren't enough values remaining in \p szpSettingsPos, the remainder of the values are filled with their defaults in \p naDefaultDoubleVals.
  \param naDefaultDoubleVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultDoubleVals will be copied to the value at the same position and onwards in \p naDoubleVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naDoubleVals.
  \param bWrite			Wether to do a read or write.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in (used to print trace-messages).
  \param szKeyName		The name of the current key (used to print trace-messages).
  \param szValDesc		The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

// ?: Should this return the number of values read from the string array (ie. the default was not used)?
void
aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault(double *naDoubleVals, char **szpSettingsPos, int *npSettingsBufferRemainingSize, unsigned int nNumValuesToSerialise, AL_CONST double *naDefaultDoubleVals, AeBool bWrite, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST char *szValDesc)
{
	unsigned int nI;

	if(bWrite)
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			int nNumCharsThatShouldHaveBeenPrinted = uszprintf(*szpSettingsPos, *npSettingsBufferRemainingSize, " %f", naDoubleVals[nI]);	// ?: Use '%g' instead of '%f'?

			if(nNumCharsThatShouldHaveBeenPrinted==ustrlen(*szpSettingsPos))
			{
				/* The value was written OK */
				int nSizePrinted = ustrsize(*szpSettingsPos);
				*szpSettingsPos += nSizePrinted;	/* Advance the pointer to the next position */
				*npSettingsBufferRemainingSize -= nSizePrinted;
			}
			else
			{
				TRACE(TPREFIX_W "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": When writing key '%s''s value and we attempted to add a '%f', we reached the end of the value-buffer. Consider using a larger buffer (or less whitespace in this multi-valued setting).\n", szKeyName, naDoubleVals[nI]);
			}
		}
	}
	else
	{
		for(nI=0;nI<nNumValuesToSerialise;nI++)
		{
			char *cpTailPtr;
			naDoubleVals[nI] = ustrtod(*szpSettingsPos,&cpTailPtr);	/* '#', newline, \0 or any other non-numeric value will fail. */
			if(*szpSettingsPos==cpTailPtr)
			{
				/* This value is not present in the string */
				// ?: Should we check for whitespace / newline / \0 vs. an un-parsable value?
				break;
			}
			*szpSettingsPos=cpTailPtr;	/* Advance the pointer to the next position */

			if(fabs(naDoubleVals[nI] - naDefaultDoubleVals[nI])>0.0000000000001)
			{
				TRACE(TPREFIX_I "Config-settings section ");
				aeTraceConfigSectionName(szSection);
				TRACE(": The multi-valued key '%s' has a %s value at position %d of '%f' which is different from the default value '%f'\n", szKeyName, szValDesc, nI, naDoubleVals[nI], naDefaultDoubleVals[nI]);
			}
		}

		if(nI<nNumValuesToSerialise)
		{
			/* In case we did not read all values from the string, warn and use the defaults instead. */

			_aeWarnThatNotAllValuesUsed(szSection, szKeyName, szValDesc, nI, nNumValuesToSerialise);

			/* Copy the defaults */
			for(;nI<nNumValuesToSerialise;nI++)
			{
				naDoubleVals[nI] = naDefaultDoubleVals[nI];
				TRACE(" %f", naDefaultDoubleVals[nI]);
			}

			TRACE("\n");	/* Finish the line */
		}
	}

}

#endif



/****************************************************************************/
/*!

 \brief Serialises a list of int values from a multi-valued config-settings key and when reading, stores result in an array of ints, and when writing, writes out the array of ints as a string.

  A range of \p nNumValuesToSerialise values are serialised from the given key in the configuration settings.
  When reading, if a value cannot be read or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \sa aeSerialiseArrayOfUIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault()

  \param naIntVals				An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szSection				Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName				The name of the current key.
  \param nNumValuesToSerialise	The number of values to serialise in \p naIntVals. This should not be greater than the number of elements in \p naIntVals. When reading, if there aren't enough values remaining in the list, the remainder of the values are filled with their defaults in \p naDefaultIntVals.
  \param naDefaultIntVals		An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultIntVals will be copied to the value at the same position and onwards in \p naIntVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naIntVals.
  \param bWrite					Wether to do a read or write.
  \param szValDesc				The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

void
aeSerialiseArrayOfIntsFromConfigListAndNotifyIfChangesFromDefault(int *naIntVals, AL_CONST char *szSection, AL_CONST char *szKeyName, unsigned int nNumValuesToSerialise, AL_CONST int *naDefaultIntVals, AeBool bWrite, AL_CONST char *szValDesc)
{
	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}
	aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault(naIntVals, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, nNumValuesToSerialise, naDefaultIntVals, bWrite, szSection, szKeyName, szValDesc);
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "", szValDesc, nNumValuesToSerialise);
	}
}



/****************************************************************************/
/*!

 \brief Serialises a list of unsigned int values from a multi-valued config-settings key and when reading, stores result in an array of unsigned ints, and when writing, writes out the array of unsigned ints as a string.

  A range of \p nNumValuesToSerialise values are serialised from the given key in the configuration settings.
  When reading, if a value cannot be read or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \sa aeSerialiseArrayOfIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault()

  \param naUIntVals				An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szSection				Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName				The name of the current key.
  \param nNumValuesToSerialise	The number of values to serialise in \p naUIntVals. This should not be greater than the number of elements in \p naUIntVals. When reading, if there aren't enough values remaining in the list, the remainder of the values are filled with their defaults in \p naDefaultUIntVals.
  \param naDefaultUIntVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultUIntVals will be copied to the value at the same position and onwards in \p naUIntVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naIntVals.
  \param bWrite					Wether to do a read or write.
  \param szValDesc				The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

void
aeSerialiseArrayOfUIntsFromConfigListAndNotifyIfChangesFromDefault(unsigned int *naUIntVals, AL_CONST char *szSection, AL_CONST char *szKeyName, unsigned int nNumValuesToSerialise, AL_CONST unsigned int *naDefaultUIntVals, AeBool bWrite, AL_CONST char *szValDesc)
{
	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}
	aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault(naUIntVals, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, nNumValuesToSerialise, naDefaultUIntVals, bWrite, szSection, szKeyName, szValDesc);
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "", szValDesc, nNumValuesToSerialise);
	}
}



/****************************************************************************/
/*!

 \brief Serialises a list of boolean values from a multi-valued config-settings key and when reading, stores result in an array of AeBools, and when writing, writes out the array of AeBools as a string.

  A range of \p nNumValuesToSerialise values are serialised from the given key in the configuration settings.
  When reading, if a value cannot be read or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \sa aeSerialiseArrayOfIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfAeBoolsFromStringAndNotifyIfChangesFromDefault()

  \param baBoolVals				An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szSection				Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName				The name of the current key.
  \param nNumValuesToSerialise	The number of values to serialise in \p baBoolVals. This should not be greater than the number of elements in \p baBoolVals. When reading, if there aren't enough values remaining in the list, the remainder of the values are filled with their defaults in \p baDefaultBoolVals.
  \param baDefaultBoolVals		An array of default values. When reading, if a value cannot be read, the remaining values in \p baDefaultBoolVals will be copied to the value at the same position and onwards in \p baBoolVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p baBoolVals.
  \param bWrite					Wether to do a read or write.
  \param szValDesc				The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

void
aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault(AeBool *baBoolVals, AL_CONST char *szSection, AL_CONST char *szKeyName, unsigned int nNumValuesToSerialise, AL_CONST AeBool *baDefaultBoolVals, AeBool bWrite, AL_CONST char *szValDesc)
{
	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}
	aeSerialiseArrayOfBoolsFromStringAndNotifyIfChangesFromDefault(baBoolVals, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, nNumValuesToSerialise, baDefaultBoolVals, bWrite, szSection, szKeyName, szValDesc);
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "", szValDesc, nNumValuesToSerialise);
	}
}



#ifndef AECONFIG_NO_FLOAT

/****************************************************************************/
/*!

 \brief Serialises a list of float values from a multi-valued config-settings key and when reading, stores result in an array of floats, and when writing, writes out the array of floats as a string.

  A range of \p nNumValuesToSerialise values are serialised from the given key in the configuration settings.
  When reading, if a value cannot be read or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \sa aeSerialiseArrayOfIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault()

  \param naFloatVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szSection				Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName				The name of the current key.
  \param nNumValuesToSerialise	The number of values to serialise in \p naFloatVals. This should not be greater than the number of elements in \p naFloatVals. When reading, if there aren't enough values remaining in the list, the remainder of the values are filled with their defaults in \p naDefaultFloatVals.
  \param naDefaultFloatVals		An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultFloatVals will be copied to the value at the same position and onwards in \p naFloatVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naFloatVals.
  \param bWrite					Wether to do a read or write.
  \param szValDesc				The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

void
aeSerialiseArrayOfFloatsFromConfigListAndNotifyIfChangesFromDefault(float *naFloatVals, AL_CONST char *szSection, AL_CONST char *szKeyName, unsigned int nNumValuesToSerialise, AL_CONST float *naDefaultFloatVals, AeBool bWrite, AL_CONST char *szValDesc)
{
	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}
	aeSerialiseArrayOfFloatsFromStringAndNotifyIfChangesFromDefault(naFloatVals, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, nNumValuesToSerialise, naDefaultFloatVals, bWrite, szSection, szKeyName, szValDesc);
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "", szValDesc, nNumValuesToSerialise);
	}
}



/****************************************************************************/
/*!

 \brief Serialises a list of double values from a multi-valued config-settings key and when reading, stores result in an array of doubles, and when writing, writes out the array of doubles as a string.

  A range of \p nNumValuesToSerialise values are serialised from the given key in the configuration settings.
  When reading, if a value cannot be read or is absent, the default value is used instead, and all subsequent values are replaced with their defaults.
  When writing, if the settings-buffer is full, a warning will be given and the stringified-value will be truncated.

  \sa aeSerialiseArrayOfIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfUIntsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfBoolsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfFloatsFromConfigListAndNotifyIfChangesFromDefault()
  \sa aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault()

  \param naDoubleVals			An array where the read values will be copied to when reading, or when writing, an array of the values to write.
  \param szSection				Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName				The name of the current key.
  \param nNumValuesToSerialise	The number of values to serialise in \p naDoubleVals. This should not be greater than the number of elements in \p naDoubleVals. When reading, if there aren't enough values remaining in the list, the remainder of the values are filled with their defaults in \p naDefaultDoubleVals.
  \param naDefaultDoubleVals	An array of default values. When reading, if a value cannot be read, the remaining values in \p naDefaultDoubleVals will be copied to the value at the same position and onwards in \p naDoubleVals (and all subsequent values will be filled with their default values). When writing, this paramater is ignored. This array should have the same number of elements as \p naDoubleVals.
  \param bWrite					Wether to do a read or write.
  \param szValDesc				The name of the description of the list-element values for this group of \p nNumValuesToSerialise values (used to make trace messages more meaningful) (used to print trace-messages).

 */

void
aeSerialiseArrayOfDoublesFromConfigListAndNotifyIfChangesFromDefault(double *naDoubleVals, AL_CONST char *szSection, AL_CONST char *szKeyName, unsigned int nNumValuesToSerialise, AL_CONST double *naDefaultDoubleVals, AeBool bWrite, AL_CONST char *szValDesc)
{
	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;

	if(bWrite)
	{
		szSettingsBuffer[0]='\0';
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
	}
	aeSerialiseArrayOfDoublesFromStringAndNotifyIfChangesFromDefault(naDoubleVals, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, nNumValuesToSerialise, naDefaultDoubleVals, bWrite, szSection, szKeyName, szValDesc);
	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "", szValDesc, nNumValuesToSerialise);
	}
}

#endif



/*****************************
 * Helpers for specific data *
 *****************************/


#ifndef AECONFIG_NO_INTCOORD2D

/****************************************************************************/
/*!

 \brief Serialises a set of 2D integer coordiantes.

  \param icpIntCoords	Where the read in integer coordinates will be contained, or where the integer coordinates to write are contained.
  \param szSection		Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName		The name of the current key.
  \param nDefaultIX		The default value for the X value.
  \param nDefaultIY		The default value for the Y value.
  \param bWrite			Wether to do a read or write.

 */

void
aeSerialiseConfigIntCoord2D(AeIntCoords2 *icpIntCoords, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST int nDefaultIX, AL_CONST int nDefaultIY, AeBool bWrite)
{
	int naCoord2DValues[AENUMCOORD2DVALUES];
	int naCoord2DValuesDefaults[AENUMCOORD2DVALUES];

	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;


	if(bWrite)
	{
		// uszprintf(szSettingsBuffer,AECONFIGSETTINGSVALUEBUFFERLENGTH," %d %d",icpIntCoords->nIX,icpIntCoords->nIY);
		szSettingsBuffer[0]='\0';

		naCoord2DValues[0] = icpIntCoords->nIX;
		naCoord2DValues[1] = icpIntCoords->nIY;
		// We don't care about naCoord2DValuesDefaults when writing (although we could print a trace-message if different)
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);

		naCoord2DValuesDefaults[0]=nDefaultIX;
		naCoord2DValuesDefaults[1]=nDefaultIY;
	}

	aeSerialiseArrayOfIntsFromStringAndNotifyIfChangesFromDefault(naCoord2DValues, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, AENUMCOORD2DVALUES, naCoord2DValuesDefaults, bWrite, szSection, szKeyName, "integer coordinate");

	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "Int-coordinate", "", AENUMCOORD2DVALUES);

		/* Set the output values to the array. */
		icpIntCoords->nIX = naCoord2DValues[0];
		icpIntCoords->nIY = naCoord2DValues[1];
	}

}

#endif



#ifndef AECONFIG_NO_ANIMPARAMDESC

/****************************************************************************/
/*!

 \brief Serialises a set of animation paramaters.

  There are two paramaters for an animation. The number of frames it contains
  and the frame-countdown - the number of ticks to display each frame.

  \param apdpAnimParams			Where the read in animation paramaters will be contained, or where the animation paramaters to write are contained.
  \param szSection				Which section of the configuration settings the key \p szKeyName is in.
  \param szKeyName				The name of the current key.
  \param nDefaultNumFrames		The default value for the number of frames. This is the value that will be used if the number of frames could not be read.
  \param nDefaultFrameCountdown	The default value for the frame-countdown. This is the value that will be used if the frame-countdown could not be read.
  \param bWrite					Wether to do a read or write.

 */

void
aeSerialiseConfigAnimParams(ChAminParamsDesc *apdpAnimParams, AL_CONST char *szSection, AL_CONST char *szKeyName, AL_CONST unsigned int nDefaultNumFrames, AL_CONST unsigned int nDefaultFrameCountdown, AeBool bWrite)
{
	unsigned int naAnimParams[AENUMANIMPARAMS];
	unsigned int naAnimParamsDefaults[AENUMANIMPARAMS];

	char szSettingsBuffer[AECONFIGSETTINGSVALUEBUFFERLENGTH];	/* Temporary settings-buffer used to build the setting-content when writing, and when reading, used to copy the setting-content so it can be parsed. */
	char *szSettingsBufferPosPostSerialise=szSettingsBuffer;
	int nSettingsBufferRemainingSize=AECONFIGSETTINGSVALUEBUFFERLENGTH;


	if(bWrite)
	{
		// uszprintf(szSettingsBuffer,AECONFIGSETTINGSVALUEBUFFERLENGTH," %d %d",apdpAnimParams->nNumFrames,apdpAnimParams->nFrameCountdown);
		szSettingsBuffer[0]='\0';

		naAnimParams[0] = apdpAnimParams->nNumFrames;
		naAnimParams[1] = apdpAnimParams->nFrameCountdown;
		// We don't care about naAnimParamsDefaults when writing (although we could print a trace-message if different)
	}
	else
	{
		aeCopyKeyValueToSettingsBufferAndCheckForPresenceAndOverflow(szSettingsBuffer, AECONFIGSETTINGSVALUEBUFFERLENGTH, szSection, szKeyName);
		
		naAnimParamsDefaults[0]=nDefaultNumFrames;
		naAnimParamsDefaults[1]=nDefaultFrameCountdown;
	}

	aeSerialiseArrayOfUIntsFromStringAndNotifyIfChangesFromDefault(naAnimParams, &szSettingsBufferPosPostSerialise, &nSettingsBufferRemainingSize, AENUMANIMPARAMS, naAnimParamsDefaults, bWrite, szSection, szKeyName, "animation-paramater-description");

	if(bWrite)
	{
		set_config_string(szSection, szKeyName, szSettingsBuffer);
	}
	else
	{
		aeCheckToSeeIfAnyRemainingValuesInSettingsBufferPostReadAndPrintMessageIfYes(szSettingsBufferPosPostSerialise, szSection, szKeyName, "anim-params-description", "", AENUMANIMPARAMS);

		/* Set the output values to the array. */
		apdpAnimParams->nNumFrames = naAnimParams[0];
		apdpAnimParams->nFrameCountdown = naAnimParams[1];
	}

}

#endif
