#ifndef C_AUTOMATON_H_INCLUDED
#define C_AUTOMATON_H_INCLUDED

#include "includes.h"
#include "s_cell.h"
#include "c_grid.h"

#include <ctime>
#include <cstdlib>

#define CELL_WIDTH SCREEN_WIDTH/m_width
#define CELL_HEIGHT SCREEN_HEIGHT/m_height

class c_automaton
{
    private:
        c_grid grid; //[SCREEN_WIDTH][SCREEN_HEIGHT];
        c_grid old_grid; //[SCREEN_WIDTH][SCREEN_HEIGHT];
        map<string, int> m_names;
        vector<s_cell> m_cell_types;
        int m_default_cell;
        int m_outside_cell;

        int m_width;
        int m_height;

    public:
        c_automaton()
        {
            m_default_cell = 0;
            m_outside_cell = -1;
            for(int y=0; y<SCREEN_HEIGHT; y++)
            {
                for(int x=0; x<SCREEN_WIDTH; x++)
                {
                    grid.set(x,y, m_default_cell);
                    old_grid.set(x,y, m_default_cell);
                }
            }
        }

        void randomize_grid( );
        void fill_grid( int type );
        void update_old_grid( );
        int get_tile( int x, int y );
        void set_tile( int x, int y, char value );
        void set_tile_pixel( int x, int y, char value );

        void load_rules( string filename );
        void save_rules( string filename );
        void load_grid( string filename );
        void save_grid( string filename );

        void draw( BITMAP* bmp, int x, int y );
        void step( );

        bool eval_condition( s_condition cond, int x, int y );
        int count_adjacent( vector<int> types, int x, int y, int pattern, int radius );

        void debug( );
};


void c_automaton::randomize_grid( )
{
    for(int y=0; y<SCREEN_HEIGHT; y++)
    {
        for(int x=0; x<SCREEN_WIDTH; x++)
        {
            grid.set( x,y, rand()%m_cell_types.size() );
            //grid[x][y] = rand() % m_cell_types.size();
        }
    }
}

void c_automaton::fill_grid( int type )
{
    for(int y=0; y<SCREEN_HEIGHT; y++)
    {
        for(int x=0; x<SCREEN_WIDTH; x++)
        {
            grid.set(x,y, type);
            //grid[x][y] = type;
        }
    }
}

void c_automaton::update_old_grid( )
{
    for(int y=0; y<m_height; y++)
    {
        for(int x=0; x<m_width; x++)
        {
            old_grid.set( x,y, grid.get(x,y) );
            //old_grid[x][y] = grid[x][y];
        }
    }
}

int c_automaton::get_tile( int x, int y )
{
    if( x>0 and y>0 and x<m_width and y<m_height )
    {
        return old_grid.get(x,y);
    }
    else
    {
        return m_outside_cell;
    }
}

void c_automaton::set_tile( int x, int y, char value )
{
    if(value==-1) {value = m_default_cell;}
    grid.set(x,y, value);
}

void c_automaton::set_tile_pixel( int x, int y, char value )
{
    x/=CELL_WIDTH;
    y/=CELL_HEIGHT;

    if(value==-1) {value = m_default_cell;}
    grid.set(x,y, value);
}

void c_automaton::load_rules( string filename )
{
    XMLNode xMainNode=XMLNode::openFileHelper(filename.c_str(), "automaton");

    string defcell = "";
    if( xMainNode.nChildNode("default_cell")>0 )
    {
        defcell = xMainNode.getChildNode("default_cell").getText();
    }

    string outcell = "";
    if( xMainNode.nChildNode("outside_cell")>0 )
    {
        outcell = xMainNode.getChildNode("outside_cell").getText();
    }

    m_width=32;
    m_height=20;
    if( xMainNode.nChildNode("width")>0 )
    {
        m_width = atoi( xMainNode.getChildNode("width").getText() );
    }
    if( xMainNode.nChildNode("height")>0 )
    {
        m_height = atoi( xMainNode.getChildNode("height").getText() );
    }

    int cells = xMainNode.nChildNode("cell_type");
    for(int i=0; i<cells; i++)
    {
        s_cell newCell;

        XMLNode xCellNode = xMainNode.getChildNode("cell_type", i);
        XMLNode xColorNode = xCellNode.getChildNode("color");

        newCell.m_color.r = atoi( xColorNode.getChildNode("r").getText() );
        newCell.m_color.g = atoi( xColorNode.getChildNode("g").getText() );
        newCell.m_color.b = atoi( xColorNode.getChildNode("b").getText() );

        int rules = xCellNode.nChildNode("rule");
        for(int j=0; j<rules; j++)
        {
            s_rule newRule;

            XMLNode xRuleNode = xCellNode.getChildNode("rule", j);

            newRule.m_type = xRuleNode.getChildNode("change_to").getText();

            if( xRuleNode.nChildNode("chance")>0 )
            {
                newRule.m_chance = atoi( xRuleNode.getChildNode("chance").getText() );
            }
            else { newRule.m_chance = 100; }

            int conditions = xRuleNode.nChildNode("condition");
            for(int k=0; k<conditions; k++)
            {
                s_condition newCond;

                XMLNode xCondNode = xRuleNode.getChildNode("condition", k);

                int types = xCondNode.nChildNode("check_for");
                for(int t=0; t<types; t++)
                {
                    newCond.m_ctypes.push_back( xCondNode.getChildNode("check_for", t).getText() );
                }

                newCond.m_min = atoi( xCondNode.getChildNode("min").getText() );
                newCond.m_max = atoi( xCondNode.getChildNode("max").getText() );

                newCond.m_radius = atoi( xCondNode.getChildNode("radius").getText() );

                if( xCondNode.nChildNode("pattern_diamond")>0 ) { newCond.m_search_type = search_diamond; }
                else if( xCondNode.nChildNode("pattern_circle")>0 ) { newCond.m_search_type = search_circle; }
                else if( xCondNode.nChildNode("pattern_cross")>0 ) { newCond.m_search_type = search_cross; }
                else if( xCondNode.nChildNode("pattern_x")>0 ) { newCond.m_search_type = search_x; }
                else if( xCondNode.nChildNode("pattern_horizontal")>0 ) { newCond.m_search_type = search_hor; }
                else if( xCondNode.nChildNode("pattern_vertical")>0 ) { newCond.m_search_type = search_vert; }
                else { newCond.m_search_type = search_square; }

                newRule.m_conditions.push_back(newCond);
            }

            newCell.m_rules.push_back(newRule);
        }

        string name = xCellNode.getChildNode("name").getText();
        m_names.insert( pair<string,int>(name,i) );
        m_cell_types.push_back( newCell );
    }

    if(defcell!="")
    {
        m_default_cell = m_names.find( defcell )->second;
    }

    if(outcell!="")
    {
        m_outside_cell = m_names.find( outcell )->second;
    }
}

void c_automaton::save_rules( string filename )
{
    ofstream file( filename.c_str() );

    file << "Size: " << m_width << "x" << m_height << endl;

    for(int i=0; i<int(m_cell_types.size()); i++)
    {
        file << "Cell Type " << i << endl;
        file << '\t' << "Color: " << m_cell_types.at(i).m_color.r << "," << m_cell_types.at(i).m_color.g << "," << m_cell_types.at(i).m_color.b << endl;
        file << '\t' << "Rules:" << endl;
        for(int r=0; r<int(m_cell_types.at(i).m_rules.size()); r++)
        {
            s_rule thisRule = m_cell_types.at(i).m_rules.at(r);

            file << "\t\tChange-To: " << thisRule.m_type << endl;
            file << "\t\tChance: " << thisRule.m_chance << endl;
            file << "\t\tConditions:" << endl;
            for(int c=0; c<int(thisRule.m_conditions.size()); c++)
            {
                for(int t=0; t<int(thisRule.m_conditions.at(c).m_ctypes.size()); t++)
                {
                    file << "\t\t\tCond-Type: " << thisRule.m_conditions.at(c).m_ctypes.at(t) << endl;
                }
                file << "\t\t\tNumber: " << thisRule.m_conditions.at(c).m_min << "-" << thisRule.m_conditions.at(c).m_max << endl;
                file << "\t\t\tRadius: " << thisRule.m_conditions.at(c).m_radius << endl;
                file << "\t\t\tPattern: " << thisRule.m_conditions.at(c).m_search_type << endl;
            }
        }
    }
}

void c_automaton::load_grid( string filename )
{
    if( filename.substr( filename.size()-4, filename.size() ) == ".txt" )
    {
        fill_grid( m_default_cell );

        ifstream file(filename.c_str());
        if( file.is_open() )
        {
            int width, height;

            file >> width;
            file >> height;

            if( width>m_width ) {width=m_width;}
            if( height>m_height ) {height=m_height;}

            for(int y=0; y<height; y++)
            {
                for(int x=0; x<width; x++)
                {
                    int read_int;
                    file >> read_int;
                    grid.set(x,y, read_int);
                }
            }

            file.close();
        }
        else
        {
            allegro_message( string("FILE NOT FOUND: "+filename).c_str() );
        }
    }
    else if( filename.substr( filename.size()-4, filename.size() ) == ".bmp" )
    {
        fill_grid( m_default_cell );

        BITMAP* bmp = load_bitmap( filename.c_str(), NULL );

        if( bmp!=NULL )
        {
            int width = bmp->w;
            int height = bmp->h;

            if(width>m_width) { width=m_width; }
            if(height>m_height) { height=m_height; }

            for(int y=0; y<height; y++)
            {
                for(int x=0; x<width; x++)
                {
                    int color = bmp->line[y][x];
                    int cell = 0;

                    for(int i=0; i<int(m_cell_types.size()); i++)
                    {
                        int cell_color = makecol( m_cell_types.at(i).m_color.r, m_cell_types.at(i).m_color.g, m_cell_types.at(i).m_color.b );
                        if( color == cell_color )
                        {
                            cell = i;
                            break;
                        }
                    }

                    grid.set(x,y, cell);
                }
            }

            release_bitmap( bmp );
        }
        else
        {
            allegro_message( string("FILE NOT FOUND: "+filename).c_str() );
        }
    }
    else
    {
        allegro_message( "File type unsupported!" );
    }
}

void c_automaton::save_grid( string filename )
{
    BITMAP* bmp = create_bitmap( m_width, m_height );

    for(int y=0; y<m_height; y++)
    {
        for(int x=0; x<m_width; x++)
        {
            int cell = grid.get(x,y);

            if( cell==-1 or cell>=int(m_cell_types.size()) )
            {
                cell = m_default_cell;
            }

            int red = m_cell_types.at(cell).m_color.r;
            int green = m_cell_types.at(cell).m_color.g;
            int blue = m_cell_types.at(cell).m_color.b;

            int color = makecol( red, green, blue );
            bmp->line[y][x] = color;
        }
    }

    PALLETE pal;
    get_pallete( pal );

    save_bitmap( filename.c_str(), bmp, pal );
    destroy_bitmap( bmp );
}

void c_automaton::draw( BITMAP* bmp, int x, int y )
{
    for(int grid_y=0; grid_y<m_height; grid_y++)
    {
        for(int grid_x=0; grid_x<m_width; grid_x++)
        {
            int r,g,b;

            if( old_grid.get(x,y)<int(m_cell_types.size()) )
            {
                r = m_cell_types.at( grid.get(grid_x,grid_y) ).m_color.r;
                g = m_cell_types.at( grid.get(grid_x,grid_y) ).m_color.g;
                b = m_cell_types.at( grid.get(grid_x,grid_y) ).m_color.b;
            }
            else
            {
                r = 150;
                g = 150;
                b = 150;
            }

            int color = makecol(r,g,b);

            int x1, y1, x2, y2;
            x1 = x+(grid_x*CELL_WIDTH);
            y1 = y+(grid_y*CELL_HEIGHT);
            x2 = x+(grid_x*CELL_WIDTH)+CELL_WIDTH;
            y2 = y+(grid_y*CELL_HEIGHT)+CELL_HEIGHT;

            rectfill( bmp, x1,y1, x2,y2, color );
        }
    }

    if( current_type == m_default_cell )
    {
        current_type++;
    }
    if( current_type<0 )
    {
        current_type = m_cell_types.size()-1;
    }
    if( current_type>int(m_cell_types.size()-1) )
    {
        current_type = 0;
    }

    int text_color = makecol( m_cell_types.at(current_type).m_color.r, m_cell_types.at(current_type).m_color.g, m_cell_types.at(current_type).m_color.b );
    textprintf_ex( bmp, font, 0,SCREEN_HEIGHT-10, text_color,0, "Current Cell-Type: %d", current_type );
}

void c_automaton::step( )
{
    update_old_grid();

    for(int y=0; y<m_height; y++)
    {
        for(int x=0; x<m_width; x++)
        {
            if( grid.get(x,y)<int(m_cell_types.size()) )
            {
                s_cell this_cell = m_cell_types.at( grid.get(x,y) );
                for( int r=0; r<int(this_cell.m_rules.size()); r++ )
                {
                    s_rule this_rule = this_cell.m_rules.at(r);
                    int to_pass = 0;
                    for( int c=0; c<int(this_rule.m_conditions.size()); c++ )
                    {
                        if( eval_condition(this_rule.m_conditions.at(c), x,y) )
                        {
                            to_pass++;
                        }
                    }

                    if( to_pass==int(this_rule.m_conditions.size()) and rand()%100<this_rule.m_chance )
                    {
                        grid.set(x,y, m_names.find( this_rule.m_type )->second);
                        break;
                    }
                }
            }
        }
    }
}

bool c_automaton::eval_condition( s_condition cond, int x, int y )
{
    vector<int> types;
    for(int i=0;  i<int(cond.m_ctypes.size()); i++)
    {
        types.push_back( m_names.find( cond.m_ctypes.at(i) )->second );
    }

    int numb = count_adjacent( types, x, y, cond.m_search_type, cond.m_radius );

    return (numb>=cond.m_min and numb<=cond.m_max);
}

int c_automaton::count_adjacent( vector<int> types, int x, int y, int pattern, int radius )
{
    int numb = 0;

    for(int j=0; j<int(types.size()); j++)
    {
        int type = types.at(j);

        if( pattern==search_diamond )
        {
            for(int xpos = x-radius; xpos<=x+radius; xpos++)
            {
                for(int ypos = y-radius; ypos<=y+radius; ypos++)
                {
                    int xdiff = abs(x-xpos);
                    int ydiff = abs(y-ypos);

                    if( xdiff+ydiff<=radius and !(x==xpos and y==ypos) and get_tile(xpos,ypos)==type )
                    {
                        numb++;
                    }
                }
            }
        }

        else if( pattern==search_circle )
        {
            for(int xpos = x-radius; xpos<=x+radius; xpos++)
            {
                for(int ypos = y-radius; ypos<=y+radius; ypos++)
                {
                    if( !(x==xpos and y==ypos) and get_distance(x,y, xpos,ypos)<=radius and get_tile(xpos,ypos)==type )
                    {
                        numb++;
                    }
                }
            }
        }

        else if( pattern==search_cross )
        {
            for( int i=1; i<=radius; i++ )
            {
                if( get_tile(x+i, y)==type ) {numb++;}
                if( get_tile(x-i, y)==type ) {numb++;}
                if( get_tile(x, y+i)==type ) {numb++;}
                if( get_tile(x, y-i)==type ) {numb++;}
            }
        }

        else if( pattern==search_x )
        {
            for( int i=1; i<=radius; i++ )
            {
                if( get_tile(x+i, y+i)==type ) {numb++;}
                if( get_tile(x+i, y-i)==type ) {numb++;}
                if( get_tile(x-i, y+i)==type ) {numb++;}
                if( get_tile(x-i, y-i)==type ) {numb++;}
            }
        }

        else if( pattern==search_hor )
        {
            for( int i=1; i<=radius; i++ )
            {
                if( get_tile(x+i, y)==type ) {numb++;}
                if( get_tile(x-i, y)==type ) {numb++;}
            }
        }

        else if( pattern==search_vert )
        {
            for( int i=1; i<=radius; i++ )
            {
                if( get_tile(x, y+i)==type ) {numb++;}
                if( get_tile(x, y-i)==type ) {numb++;}
            }
        }

        else if( pattern==search_square )
        {
            for(int xpos = x-radius; xpos<=x+radius; xpos++)
            {
                for(int ypos = y-radius; ypos<=y+radius; ypos++)
                {
                    if( !(x==xpos and y==ypos) and get_tile(xpos,ypos)==type )
                    {
                        numb++;
                    }
                }
            }
        }
    }

    return numb;
}

void c_automaton::debug( )
{
    textprintf_ex( screen, font, 0,180, 0, 1, "DEBUG: " );
}

#endif // C_AUTOMATON_H_INCLUDED
