/*
    This file is part of Funiter,
    Real and complex function iteration software.

    parser.c - Funiter's expression evaluator.
    
    Copyright (C) 1995-2007 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
    See README for contact information.
    
    references/sources:

        http://compilers.iecc.com/crenshaw/tutor4.txt
        http://javajeff.mb.ca/aa/5/5.html
        http://www.java2s.com/Code/CSharp/Development-Class/Thismodulecontainstherecursivedescentparserthatrecognizesvariables.htm
                    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "parser.h"

/* 
**  This routine removes spaces from a string.
**  To save time it is called best before the iteration/parse-loop.
*/

char *RemoveSpaces(char *StrOut, char *StrIn) 
{
    int     si = 0, di = 0;
    
    while(StrIn[si] != '\0') {
        if((StrIn[si] == ' ') || (StrIn[si] == '\t')) {
            si++;
            continue;
        }
        else
            StrOut[di++] = StrIn[si++];
    }
    return(StrOut);
}

/*
**  Unary + or -
**  returns 'true' for - and 'false' for + (which is default).
*/

int Unary(TParseData *ParseData)
{
    int     Neg = FALSE;
    
    if(IS_TOKEN(ADD_OP)) 
        NEXT_TOKEN;
    else if(IS_TOKEN(SUB_OP)) { 
        NEXT_TOKEN;
        Neg = TRUE;
    }
    return Neg;
}

TComplex CGetNumber(TParseData *ParseData)
{
    TComplex        c;
    
    c.r = 0.0;
    c.i = 0.0;
    if(!IS_TOKEN('{')) {
        c.r = GetNumber(ParseData);
        c.i = 0.0;
    }
    else {
        NEXT_TOKEN;
        c.r = GetNumber(ParseData);
        if(!IS_TOKEN(',')) {
            ParseData->ErrorNum = SYNTAXERR;
            ParseData->ErrorPos = ParseData->TokenCounter;
        }
        else {
            NEXT_TOKEN;
            c.i = GetNumber(ParseData);
        }
        if(!IS_TOKEN('}')) {
            ParseData->ErrorNum = NORPARENTH;
            ParseData->ErrorPos = ParseData->TokenCounter;
        }
        else
            NEXT_TOKEN;
    }
    return c; 
}

/*
**  Get a number from the parse-string.
**  
**  Possible formats are:   plain int (like 3).
**                          floatingpoint (like 3.1415).
**                          scientific notation (like 1e-10, 1e10 or 1e+10).
**
**  All numbers might have one optional unary + or - sign.
*/

long double GetNumber(TParseData *ParseData)
{
    int         PowN = 0, GetInt = TRUE, Neg = FALSE;
    long double IntPart = 0.0, FracPart = 0.0, Number = 0.0;

    /* Check for unary + or - */
    
    Neg = Unary(ParseData);
    
    /* Numbers should start with a digit (0-9) or a dot */
    
    if(!ISDIGIT(TOKEN) && !IS_TOKEN('.')) {
        ParseData->ErrorNum = SYNTAXERR;
        ParseData->ErrorPos = ParseData->TokenCounter;
    }

    /* Match digit or dot */
    
    while(ISDIGIT(TOKEN) || IS_TOKEN('.')) {

        /* As soon as a dot is encountered, flag is toggled */
        
        if(IS_TOKEN('.')) {

            /* 
            **  A second dot after the frac. part is useless and causes 
            **  an error.
            */
            
            if(!GetInt) {
                ParseData->ErrorNum = SYNTAXERR;
                ParseData->ErrorPos = ParseData->TokenCounter;
            }
                        
            /* Get frac. part and set for next token */
            
            GetInt = FALSE;
            NEXT_TOKEN;
        }

        if(GetInt)
            IntPart = IntPart * 10.0 + (double) TOKEN - 48.0;
        else {
            FracPart = FracPart * 10.0 + (double) TOKEN - 48.0;
            PowN++;
        }
        NEXT_TOKEN;
    }
    Number = IntPart + (FracPart / pow(10.0, PowN));

    /* In case a unary - is encountered the value is negated */
    
    if(Neg) Number = 0.0 - Number;

    /* Process optional scientific notation */

    Neg = FALSE;
    PowN = 0;    	
    if(IS_TOKEN('e') || IS_TOKEN('E')) {
        NEXT_TOKEN;
        Neg = Unary(ParseData);
        
        if(!ISDIGIT(TOKEN)) {
            ParseData->ErrorNum = NONUM;
            ParseData->ErrorPos = ParseData->TokenCounter;
        }
                
        while(ISDIGIT(TOKEN)) {
            PowN = PowN * 10 + (TOKEN - 0x30);
            NEXT_TOKEN;
        }
        if(Neg) PowN = -PowN;
        Number = Number * pow(10, PowN);
    }
    
    return(Number);
}

/* Handle functions and variables */

TComplex CFunVar(TParseData *ParseData)
{
    TComplex        c;
    char            FnStr[10] = "";
    int             Index = 0;
    
    if(ISALPHA(TOKEN)) {
    
        while(ISALPHA(TOKEN) && Index < 10 && TOKEN != '\0') {
            FnStr[Index] = toupper(TOKEN);
            Index++;
            NEXT_TOKEN;
        }

        if(strcmp(FnStr, "SIN") == 0)           c = CSin(CFactor(ParseData));
        else if(strcmp(FnStr, "COS") == 0)      c = CCos(CFactor(ParseData));
        else if(strcmp(FnStr, "EXP") == 0)      c = CExp(CFactor(ParseData));
        else {
        
            /*
            **  Get values for the variables.
            */
            
            for(Index = 0; Index < MAX_VAR; Index++) {
                if(strchr(FnStr, toupper(ParseData->VarNames[Index])) != NULL) {
                    c.r = ParseData->CVarValues[Index].r;
                    c.i = ParseData->CVarValues[Index].i; 
                    break;
                }
            }
            if(Index >= MAX_VAR) {
                ParseData->ErrorNum = UNDEFDFN;
                ParseData->ErrorPos = ParseData->TokenCounter;
            }
        }
    }
    else
        c = CFactor(ParseData);

    return c;
}

long double FunVar(TParseData *ParseData)
{
    long double     v, w;
    char            FnStr[10] = "";
    int             Index = 0, Neg = FALSE;

    /* Check for unary + or - */
    
    Neg = Unary(ParseData);
    
    if(ISALPHA(TOKEN)) {
    
        while(ISALPHA(TOKEN) && Index < 10 && TOKEN != '\0') {
            FnStr[Index] = toupper(TOKEN);
            Index++;
            NEXT_TOKEN;
        }

        if(strcmp(FnStr, "SIN") == 0)           v = sin(Factor(ParseData));
        else if(strcmp(FnStr, "ASN") == 0)      v = asin(Factor(ParseData));
        else if(strcmp(FnStr, "COS") == 0)      v = cos(Factor(ParseData));
        else if(strcmp(FnStr, "TAN") == 0)      v = tan(Factor(ParseData));
        else if(strcmp(FnStr, "ATN") == 0)      v = atan(Factor(ParseData));
        else if(strcmp(FnStr, "EXP") == 0)      v = exp(Factor(ParseData));
        else if(strcmp(FnStr, "LOG") == 0)      v = log(Factor(ParseData));
        else if(strcmp(FnStr, "ABS") == 0)      v = fabs(Factor(ParseData));
        else if(strcmp(FnStr, "ENTIER") == 0)   v = floor(Factor(ParseData));
        else if(strcmp(FnStr, "SQR") == 0) 
        {
            v = Factor(ParseData);
            if(v < 0.0) {
                ParseData->ErrorNum = OUTRANGE; 
                ParseData->ErrorPos = ParseData->TokenCounter;
                return v;
            }            
            else 
                v = sqrt(v);
        }
        else if(strcmp(FnStr, "FRAC") == 0) 
        {
            w = Factor(ParseData);
            v = w - floor(w);
        }
        else  
        {
            /*
            **  Get values for the variables.
            */
            
            for(Index = 0; Index < MAX_VAR; Index++) {
                if(strchr(FnStr, toupper(ParseData->VarNames[Index])) != NULL) {
                    v = ParseData->VarValues[Index]; 
                    break;
                }
            }
            if(Index >= MAX_VAR) {
                ParseData->ErrorNum = UNDEFDFN;
                ParseData->ErrorPos = ParseData->TokenCounter;
            }
        }
    }
    else
        v = Factor(ParseData);

    /* If an unary - was detected the value is negated */
        
    if(Neg) v = 0.0 - v;

    return v;
}

/* Handle parenthesis */

TComplex CFactor(TParseData *ParseData)
{
    TComplex c;

    c.r = 0.0;
    c.i = 0.0;
    if(IS_TOKEN('(')) { 
        NEXT_TOKEN;
        c = CAddSub(ParseData);
        if(IS_TOKEN(')')) 
            NEXT_TOKEN; 
        else { 
            ParseData->ErrorNum = NORPARENTH;
            ParseData->ErrorPos = ParseData->TokenCounter;
        }
    }
    else if(ISALPHA(TOKEN)) c = CFunVar(ParseData);
    else if(IS_TOKEN('{') || ISDIGIT(TOKEN)) c = CGetNumber(ParseData);
    else {
        ParseData->ErrorNum = UNDEFDFN;
        ParseData->ErrorPos = ParseData->TokenCounter;
    }
    return c;
}

long double Factor(TParseData *ParseData)
{
    long double v = 0;

    if(IS_TOKEN('(')) { 
        NEXT_TOKEN;
        v = AddSub(ParseData);
        if(IS_TOKEN(')')) 
            NEXT_TOKEN; 
        else { 
            ParseData->ErrorNum = NORPARENTH;
            ParseData->ErrorPos = ParseData->TokenCounter;
        }
    }
    else if(ISADDOP(TOKEN) || ISALPHA(TOKEN)) v = FunVar(ParseData);
    else if(ISADDOP(TOKEN) || ISDIGIT(TOKEN)) v = GetNumber(ParseData);
    else {
        ParseData->ErrorNum = UNDEFDFN;
        ParseData->ErrorPos = ParseData->TokenCounter;
    }
    return v;
}

/* Handle the ^ operator */

long double Power(TParseData *ParseData)
{
    long double v = 0;

    v = Factor(ParseData);
    while(1) {
        if(IS_TOKEN(POW_OP)) { 
            NEXT_TOKEN; 
            v = pow(v, Factor(ParseData)); 
        }
        else
            break;
    }
    return v;
}

/* Handle * and / */

TComplex CMulDiv(TParseData *ParseData)
{
    TComplex c;
    
    c.r = 0.0;
    c.i = 0.0;
    c = CFactor(ParseData);
    while(1) {
        if(IS_TOKEN(MUL_OP)) {
            NEXT_TOKEN;
            c = CMul(c, CFactor(ParseData));
        }
        else if(IS_TOKEN(DIV_OP)) {
            NEXT_TOKEN;
            c = CDiv(c, CFactor(ParseData));
        }
        else {
            break;
        }
    }
    return c;
}

long double MulDiv(TParseData *ParseData)
{
    long double v = 0, w = 0;

    v = Power(ParseData);
    while(IS_TOKEN(MUL_OP) || IS_TOKEN(DIV_OP)) {
        if(IS_TOKEN(MUL_OP)) { 
            NEXT_TOKEN;
            v *= Power(ParseData);
        }
        else if(IS_TOKEN(DIV_OP)) { 
            NEXT_TOKEN; 
            w = Power(ParseData);
            if(w == 0.0) { 
                ParseData->ErrorNum = DIVBYZERO;
                ParseData->ErrorPos = ParseData->TokenCounter;
            }
            else 
                v /= w; 
        }
    }
    return v;
}

/* Handle + and - */

TComplex CAddSub(TParseData *ParseData)
{
    TComplex c;
    
    c.r = 0.0;
    c.i = 0.0;
    c = CMulDiv(ParseData);
    while(1) {
        if(IS_TOKEN(ADD_OP)) {
            NEXT_TOKEN;
            c = CAdd(c, CMulDiv(ParseData));
        }
        else if(IS_TOKEN(SUB_OP)) {
            NEXT_TOKEN;
            c = CSub(c, CMulDiv(ParseData));
        }
        else {
            break;
        }
    }
    return c;
}

long double AddSub(TParseData *ParseData)
{
    long double v = 0;

    v = MulDiv(ParseData);
    while(1) {
        if(IS_TOKEN(ADD_OP)) { 
            NEXT_TOKEN; 
            v += MulDiv(ParseData);
        } 
        else if(IS_TOKEN(SUB_OP)) { 
            NEXT_TOKEN; 
            v -= MulDiv(ParseData);
        } 
        else {
            break;
        }
    }
    return v;
}

TComplex CParse(int *ErrorNum, int *ErrorPos, char *Str, int vars, ...)
{
    int             Index;
    TComplex        c;
    va_list         args;
    TParseData      ParseData;
    
    /* Init structure with parse information */
    
    ParseData.ErrorNum = 0;
    if(ErrorNum != NULL) *ErrorNum = 0;
    ParseData.ErrorPos = 0;
    if(ErrorPos != NULL) *ErrorPos = 0;
    ParseData.TokenCounter = 0;
    ParseData.ParseStr = Str;
        
    if(vars >= MAX_VAR) ParseData.ErrorNum = MAXVAREXC;

    /* Initialize variables */
    
    va_start(args, vars);
    for(Index = 0; Index < vars; Index++) {
        ParseData.VarNames[Index] = va_arg(args, int);
        ParseData.CVarValues[Index] = va_arg(args, TComplex);
    }
    va_end(args);

    /* Call recursive functions and return result */
    
    c = CAddSub(&ParseData);
    
    /* Check for syntax error */
    
    if(ParseData.TokenCounter < strlen(Str)) {
        ParseData.ErrorNum = SYNTAXERR;
        ParseData.ErrorPos = ParseData.TokenCounter;
    }

    if(ErrorNum != NULL) *ErrorNum = ParseData.ErrorNum;
    if(ErrorPos != NULL) *ErrorPos = ParseData.ErrorPos;
    
    return c;
}

long double Parse(int *ErrorNum, int *ErrorPos, char *Str, int vars, ...)
{
    int             Index;
    long double     v;
    va_list         args;
    TParseData      ParseData;
    
    /* Init structure with parse information */
    
    ParseData.ErrorNum = 0;
    if(ErrorNum != NULL) *ErrorNum = 0;
    ParseData.ErrorPos = 0;
    if(ErrorPos != NULL) *ErrorPos = 0;
    ParseData.TokenCounter = 0;
    ParseData.ParseStr = Str;
        
    if(vars >= MAX_VAR) ParseData.ErrorNum = MAXVAREXC;

    /* Initialize variables */
    
    va_start(args, vars);
    for(Index = 0; Index < vars; Index++) {
        ParseData.VarNames[Index] = va_arg(args, int);
        ParseData.VarValues[Index] = va_arg(args, double);
    }
    va_end(args);

    /* Call recursive functions and return result */
    
    v = AddSub(&ParseData);
    
    /* Check for syntax error */
    
    if(ParseData.TokenCounter < strlen(Str)) {
        ParseData.ErrorNum = SYNTAXERR;
        ParseData.ErrorPos = ParseData.TokenCounter;
    }

    if(ErrorNum != NULL) *ErrorNum = ParseData.ErrorNum;
    if(ErrorPos != NULL) *ErrorPos = ParseData.ErrorPos;
    
    return v;
}
