/*
    This file is part of Funiter,
    Real and complex function iteration software.

    main.c - Funiter's main routine.

    Copyright (C) 1995-2009 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).

    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include <stdio.h>
#include <allegro.h>

#include "newmenus.h"
#include "clsmenus.h"
#include "realexp.h"
#include "cplexp.h"

int Progress(int Position, int MaxPosition);

int main(int argc, char *argv[])
{
    char        LngFile[2048] = "internal", MapFile[2048] = "", 
                ConfigFile[2048] = "", ConfigBakFile[2048] = "";
    int         i, card, width = 640, height = 480, color_depth,
                FractintMap = FALSE, ConfigMode = FALSE,
                Functiontype = 0, DefaultItemIndex = 0;
    TConsole    *Console;
    TRealExp    *RealExp;
    TComplexExp *ComplexExp;

    /* Initialize graphics */
    
    set_uformat(U_ASCII);
    
    allegro_init();
    install_mouse();
    install_keyboard();
    install_timer();
    
    if((color_depth = desktop_color_depth()) != 0) 
        set_color_depth(color_depth);
    
    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0) {
        if(set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0) != 0) { 
            set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
            allegro_message("Error setting a graphmode!");
            exit(1);
        }
    }

    /* Look for commandline options. */
    
    for(i = 1; i < argc; i++) {

        /* Load language file */
        
        if(strncmp(argv[i], "-l", 2) == 0) {
            if((argv[i + 1] != NULL) && (strlen(argv[i]) == 2)) {
                strcpy(LngFile, argv[i + 1]);
                i++;
            }
            else
                strcpy(LngFile, argv[i] + 2);
        }
        
        /* Select graphic-mode */
        
        else if(strncmp(argv[i], "-s", 2) == 0) {
            set_gfx_mode(GFX_SAFE, 640, 480, 0, 0);
            gfx_mode_select_ex(&card, &width, &height, &color_depth);
            set_color_depth(color_depth);
            if(set_gfx_mode(card, width, height, 0, 0) != 0) {
                set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
                allegro_message("Error setting a graphmode!");
                exit(1);
            }
        }
        
        /* Load Fractint .map file */
        
        else if(strncmp(argv[i], "-m", 2) == 0) {
            if((argv[i + 1] != NULL) && (strlen(argv[i]) == 2)) {
                strcpy(MapFile, argv[i + 1]);
                i++;
            }
            else
                strcpy(MapFile, argv[i] + 2);
            
            if(file_exists(MapFile, 0, NULL)) 
                FractintMap = TRUE;
        }
        
        /* Load configuration file */
        
        else if(strncmp(argv[i], "-c", 2) == 0) {
            if((argv[i + 1] != NULL) && (strlen(argv[i]) == 2)) {
                strcpy(ConfigFile, argv[i + 1]);
                i++;
            }
            else
                strcpy(ConfigFile, argv[i] + 2);

            if(file_exists(ConfigFile, 0, NULL)) {
                push_config_state();
                set_config_file(ConfigFile);

                /* Create backup file for auto-update */
                
                if(get_config_int("main", "auto_config_update", 0)) {
                    replace_extension(ConfigBakFile, ConfigFile, "bak", 
                        sizeof(ConfigBakFile));                    
                    if(strcmp(ConfigFile, ConfigBakFile) != 0)
                        SimpleCopyTextFile(ConfigBakFile, ConfigFile);
                }

                /* Determine functiontype */
                
                Functiontype = get_config_int("main", "functiontype", 
                    Functiontype);
                
                /* Set language file */
                
                strncpy(LngFile, get_config_string("main", "language_file", 
                    LngFile), 2047);
            }
            ConfigMode = TRUE;
            DefaultItemIndex = 4;
        }
        
        /* Help */
        
        else if((strncmp(argv[i], "-?", 2) == 0) ||
            (strncmp(argv[i], "-h", 2) == 0)) 
        {
            printf("\nUsage: %s [-s] [-l<filename>] [-? | -h]", argv[0]);
            printf("\n-l<filename> = load a language file");
            printf("\n-s           = open dialog to select a graphicmode");
            printf("\n-m<filename> = load a Fractint '.map' file");
            printf("\n-c<filename> = load a Configuration file");
            printf("\n-? or -h     = help");
            printf("\n\nCommandline parameters are lowercase only!\n");
            return 0;
        }
        else {
            printf("Unknown command, use -h or -? for help.\n");
            return 0;
        }
    }

    clear_keybuf();
    if((RealExp = CreateRealExp(0, 0, width, height - 20)) != NULL)
    {
        RealExp->PrgProc = Progress;
        
        if(ConfigMode && file_exists(ConfigFile, 0, NULL)) 
            ReadRealConfig(RealExp);
        
        if(FractintMap) 
            ReadFractintMap(MapFile, RealExp->Settings);
        
        if((ComplexExp = CreateComplexExp(0, 0, width, height - 20)) != NULL)
        {
            ComplexExp->PrgProc = Progress;
            
            if(ConfigMode && file_exists(ConfigFile, 0, NULL)) 
                ReadComplexConfig(ComplexExp);
            
            if(FractintMap) 
                ReadFractintMap(MapFile, ComplexExp->Settings);
            
            if((Console = alconio_init(80, 30, 2, LIGHTGRAY, BLACK)) != NULL)
            {
                if(InitClassicMenus(Functiontype, LngFile) == 0) 
                {
                    if(ConfigMode)
                        ClsMnuDraw(Console, RealExp, ComplexExp);
                    else 
                        ClsMnuAbout(Console);
                    ClsMnuMain(Console, RealExp, ComplexExp, 
                        DefaultItemIndex);
                }
                else
                    allegro_message("Problem loading Language file.\n");
                alconio_exit(Console);
            }
            else
                allegro_message("Not enough memory to create 'Console'.\n");

            if(ConfigMode && file_exists(ConfigFile, 0, NULL) && 
                (Functiontype == 1)) 
            {
                if(get_config_int("main", "auto_config_update", 0))
                    WriteComplexConfig(ComplexExp);
            }    
            FreeComplexExp(ComplexExp);
        }
        else
            allegro_message("Not enough memory to create Complex experiment.\n");
        
        if(ConfigMode && file_exists(ConfigFile, 0, NULL) && 
            (Functiontype == 0)) 
        {
            if(get_config_int("main", "auto_config_update", 0))
                WriteRealConfig(RealExp);
        }
        FreeRealExp(RealExp);
    }
    else
        allegro_message("Not enough memory to create Real experiment.\n");
    
    allegro_exit();

    return 0;
}
END_OF_MAIN()

int Progress(int Position, int MaxPosition)
{
    int         Percentage = 0;
    double      OnePercent = 1.0;
    BITMAP      *Bitmap;

    OnePercent = (double) MaxPosition / 100.0;
    if(OnePercent < 1.0) OnePercent = 1.0;

    if(Position % (int) OnePercent == 0) {
        Percentage = ((double) Position / OnePercent);
        Bitmap = create_bitmap(110, 30);
        clear_bitmap(Bitmap);
        acquire_bitmap(Bitmap);
        rect(Bitmap, 0, 0, 109, 29, makecol(255, 255, 255));
        if(Percentage > 0) { 
            rectfill(Bitmap, 5, 5, 5 + Percentage - 1, 25, 
                makecol(110, 110, 110));
        }
        textprintf_ex(Bitmap, font, 45, (30 - text_height(font)) / 2,
            makecol(255, 255, 255), -1, "%3d%%", Percentage);
        release_bitmap(Bitmap);
        blit(Bitmap, screen, 0, 0, (SCREEN_W - 109) / 2, 
            (SCREEN_H - 29) / 2, 110, 30);
        destroy_bitmap(Bitmap);
    }
    return 0;
}
