/*
    This file is part of Funiter,
    Real and complex function iteration software.

    settings.c - settings.

    Copyright (C) 1995-2010 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "settings.h"


TSettings *CreateSettings(void)
{
    TSettings *Settings = malloc(sizeof(*Settings));
    
    if(Settings != NULL)
        Settings = SetDefaultSettings(Settings);

    return(Settings);
}

void FreeSettings(TSettings *Settings)
{
    if(Settings != NULL) 
        free(Settings);
}

TSettings *SetDefaultSettings(TSettings *Settings)
{
    int             i;
    
    if(Settings == NULL) return NULL;
    
    /* Default colors */
    
    Settings->Palette[ 0].b = 0x00; 
    Settings->Palette[ 0].g = 0x00; 
    Settings->Palette[ 0].r = 0x00;
    
    Settings->Palette[ 1].b = 0x00; 
    Settings->Palette[ 1].g = 0x00; 
    Settings->Palette[ 1].r = 0xA8;
    
    Settings->Palette[ 2].b = 0x54; 
    Settings->Palette[ 2].g = 0x54; 
    Settings->Palette[ 2].r = 0xFC;
    
    Settings->Palette[ 3].b = 0xA8; 
    Settings->Palette[ 3].g = 0x00; 
    Settings->Palette[ 3].r = 0xA8;
    
    Settings->Palette[ 4].b = 0xFC; 
    Settings->Palette[ 4].g = 0x54; 
    Settings->Palette[ 4].r = 0xFC;
    
    Settings->Palette[ 5].b = 0x00; 
    Settings->Palette[ 5].g = 0x54; 
    Settings->Palette[ 5].r = 0xA8;
    
    Settings->Palette[ 6].b = 0x54; 
    Settings->Palette[ 6].g = 0xFC; 
    Settings->Palette[ 6].r = 0xFC;
    
    Settings->Palette[ 7].b = 0x00; 
    Settings->Palette[ 8].g = 0xA8; 
    Settings->Palette[ 8].r = 0x00;
    
    Settings->Palette[ 8].b = 0x54; 
    Settings->Palette[ 7].g = 0xFC; 
    Settings->Palette[ 7].r = 0x54;
    
    Settings->Palette[ 9].b = 0xA8; 
    Settings->Palette[ 9].g = 0xA8; 
    Settings->Palette[ 9].r = 0x00;
    
    Settings->Palette[10].b = 0xFC; 
    Settings->Palette[10].g = 0x54; 
    Settings->Palette[10].r = 0x54;
    
    Settings->Palette[11].b = 0xFC; 
    Settings->Palette[11].g = 0xFC; 
    Settings->Palette[11].r = 0x54; 
    
    Settings->Palette[12].b = 0xA8; 
    Settings->Palette[12].g = 0x00; 
    Settings->Palette[12].r = 0x00;
    
    Settings->Palette[13].b = 0xA8; 
    Settings->Palette[13].g = 0xA8; 
    Settings->Palette[13].r = 0xA8;
    
    Settings->Palette[14].b = 0x54; 
    Settings->Palette[14].g = 0x54; 
    Settings->Palette[14].r = 0x54;
    
    Settings->Palette[15].b = 0xFC; 
    Settings->Palette[15].g = 0xFC; 
    Settings->Palette[15].r = 0xFC;
    
    /* General settings */
    
    Settings->GEN_MenuStyle = TRUE;
    Settings->GEN_Fixedpoint = TRUE;
    Settings->GEN_ParConnect = TRUE;

    /* Layout settings (what you see / and how it appears on screen) */

    Settings->LAY_Statusline = TRUE;

    /* Coordinates - Sub-section in the Layout settings */

    Settings->COO_Status = TRUE;
    Settings->COO_Grid = FALSE;
    Settings->COO_Calibrate = TRUE;

    /* Step by step iteration - Sub-section in the Layout settings */

    Settings->SSI_Text = TRUE;
    Settings->SSI_Lines = TRUE;
    Settings->SSI_LineWidth = 2;
    Settings->SSI_FontSize = 4;
    Settings->SSI_Accuracy = 8;

    /* Colors */

    Settings->COL_Inside = 0;
    Settings->COL_Outside = 15;
    Settings->COL_Grid = 15;
    Settings->COL_Function = 4;
    Settings->COL_XYLine = 7;
    Settings->COL_Iter1 = 6;
    Settings->COL_Iter2 = 5;
    Settings->COL_Max = 15;

    for(i = 0; i <= 15; i++) 
        Settings->COL_List[i] = i;

    Settings->DBG_SymmetryAreaColoring = FALSE;
    
    return(Settings);
}

/*
**  Load Settings.
*/

int LoadSettings(FILE *fp, TSettings *Settings)
{
    int             i, r, g, b;
    
    fscanf(fp, "%d,%d,%d\n", &Settings->GEN_MenuStyle, 
        &Settings->GEN_Fixedpoint, &Settings->GEN_ParConnect);
        
    fscanf(fp, "%d\n", &Settings->LAY_Statusline);
    
    fscanf(fp, "%d,%d,%d\n", &Settings->COO_Status, &Settings->COO_Grid,
        &Settings->COO_Calibrate);

    fscanf(fp, "%d,%d,%d,%d,%d\n", &Settings->SSI_Text, &Settings->SSI_Lines,
        &Settings->SSI_LineWidth, &Settings->SSI_FontSize, 
        &Settings->SSI_Accuracy);
    
    fscanf(fp, "%d,%d,%d,%d,%d,%d,%d,%d\n", &Settings->COL_Inside, 
        &Settings->COL_Outside, &Settings->COL_Grid, &Settings->COL_Function,
        &Settings->COL_XYLine, &Settings->COL_Iter1, &Settings->COL_Iter2, 
        &Settings->COL_Max);
    
    /*for(i = 0; i <= 15; i++) fscanf(fp, "%d\n", &Settings->COL_List[i]);*/
    
    for(i = 0; i <= 255; i++) {
        fscanf(fp, "%3d %3d %3d\n", &b, &g, &r);
        Settings->Palette[i].b = (unsigned char) b; 
        Settings->Palette[i].g = (unsigned char) g; 
        Settings->Palette[i].r = (unsigned char) r;
    }
    
    return 0;
}

TSettings *ReadSettingsConfig(char *Filename, TSettings *Settings)
{
    char            mapfile[2048] = "", colindex[7] = "", **s;
    AL_CONST char   *p;
    int             argc;
    unsigned int    i;
    
    if(file_exists(Filename, 0, NULL)) {
        push_config_state();
        override_config_file(Filename);
    }
    
    /* General */
    
    Settings->LAY_Statusline = get_config_int("settings", "gen_statusbar", 
        Settings->LAY_Statusline) ? TRUE : FALSE;
    Settings->GEN_ParConnect = get_config_int("settings", "gen_parconnect", 
        Settings->GEN_ParConnect) ? TRUE : FALSE;
    Settings->GEN_Fixedpoint = get_config_int("settings", "gen_fixedpoint", 
        Settings->GEN_Fixedpoint) ? TRUE : FALSE;
    
    /* Coordinates */
    
    Settings->COO_Status = get_config_int("settings", "coo_visible", 
        Settings->COO_Status) ? TRUE : FALSE;
    Settings->COO_Grid = get_config_int("settings", "coo_grid", 
        Settings->COO_Grid) ? TRUE : FALSE;
    Settings->COO_Calibrate = get_config_int("settings", "coo_calibration", 
        Settings->COO_Calibrate) ? TRUE : FALSE;
         
    /*
    **  Step-by-step iteration
    */
    
    Settings->SSI_Text = get_config_int("settings", "step_coords", 
        Settings->SSI_Text);
    Settings->SSI_FontSize = get_config_int("settings", "step_textsize", 
        Settings->SSI_FontSize);
    Settings->SSI_LineWidth = get_config_int("settings", "step_linewidth", 
        Settings->SSI_LineWidth);

    /*
    **  Colors
    */

    if((p = get_config_string("settings", "mapfile", NULL)) != NULL) {
        strncpy(mapfile, p, sizeof(mapfile));
        if(file_exists(mapfile, 0, NULL))
            Settings = ReadFractintMap(mapfile, Settings);
    }
    else 
    {
        for(i = 0; i <= Settings->COL_Max; i++) {
            snprintf(colindex, sizeof(colindex), "col%.3d", i);
            s = get_config_argv("palette", colindex, &argc);
            if(argc == 3) {
                Settings->Palette[i].r = (unsigned char) atoi(s[0]);
                Settings->Palette[i].g = (unsigned char) atoi(s[1]);
                Settings->Palette[i].b = (unsigned char) atoi(s[2]);
            }
        }
    }
    
    Settings->COL_Grid = get_config_int("settings", "col_grid", 
        Settings->COL_Grid);
    Settings->COL_Function = get_config_int("settings", "col_function", 
        Settings->COL_Function);
    Settings->COL_XYLine = get_config_int("settings", "col_xyline", 
        Settings->COL_XYLine);
    Settings->COL_Iter1 = get_config_int("settings", "col_startvalue1", 
        Settings->COL_Iter1);
    Settings->COL_Iter2 = get_config_int("settings", "col_startvalue2", 
        Settings->COL_Iter2);

    Settings->DBG_SymmetryAreaColoring = get_config_int("settings",
        "dbg_symmetry_area_coloring", Settings->DBG_SymmetryAreaColoring);

    if(file_exists(Filename, 0, NULL)) {
        override_config_file(NULL);
        pop_config_state();
    }
        
    return Settings;
}

/*
**  Save Settings.
*/

int SaveSettings(FILE *fp, TSettings *Settings)
{
    unsigned int        i;
    
    fprintf(fp, "%d,%d,%d\n", Settings->GEN_MenuStyle, Settings->GEN_Fixedpoint,
        Settings->GEN_ParConnect);
        
    fprintf(fp, "%d\n", Settings->LAY_Statusline);
    
    fprintf(fp, "%d,%d,%d\n", Settings->COO_Status, Settings->COO_Grid,
        Settings->COO_Calibrate);

    fprintf(fp, "%d,%d,%d,%d,%d\n", Settings->SSI_Text, Settings->SSI_Lines,
        Settings->SSI_LineWidth, Settings->SSI_FontSize, 
        Settings->SSI_Accuracy);
    
    fprintf(fp, "%d,%d,%d,%d,%d,%d,%d,%d\n", Settings->COL_Inside, 
        Settings->COL_Outside, Settings->COL_Grid, Settings->COL_Function,
        Settings->COL_XYLine, Settings->COL_Iter1, Settings->COL_Iter2, 
        Settings->COL_Max);

    for(i = 0; i <= Settings->COL_Max; i++) { 
        fprintf(fp, "%3d %3d %3d\n", Settings->Palette[i].b, 
            Settings->Palette[i].g, Settings->Palette[i].r);
    }
    
    return 0;
}

void WriteSettingsConfig(char *Filename, TSettings *Settings)
{
    char            colindex[7] = "", s[30];
    unsigned int    i;

    if(file_exists(Filename, 0, NULL)) {
        push_config_state();
        override_config_file(Filename);
    }
    
    /* General */
    
    set_config_int("settings", "gen_statusbar", Settings->LAY_Statusline ? 1 : 0);
    set_config_int("settings", "gen_parconnect", Settings->GEN_ParConnect ? 1 : 0);
    set_config_int("settings", "gen_fixedpoint", Settings->GEN_Fixedpoint ? 1 : 0);
    
    /* Coordinates */
    
    set_config_int("settings", "coo_visible", Settings->COO_Status ? 1 : 0);
    set_config_int("settings", "coo_grid", Settings->COO_Grid ? 1 : 0);
    set_config_int("settings", "coo_calibration", Settings->COO_Calibrate ? 1 : 0);
         
    /*
    **  Step-by-step iteration
    */
    
    set_config_int("settings", "step_coords", Settings->SSI_Text ? 1 : 0);
    set_config_int("settings", "step_textsize", Settings->SSI_FontSize);
    set_config_int("settings", "step_linewidth", Settings->SSI_LineWidth);

    /*
    **  Colors
    */

    set_config_int("settings", "col_grid", Settings->COL_Grid);
    set_config_int("settings", "col_function", Settings->COL_Function);
    set_config_int("settings", "col_xyline", Settings->COL_XYLine);
    set_config_int("settings", "col_startvalue1", Settings->COL_Iter1);
    set_config_int("settings", "col_startvalue2", Settings->COL_Iter2);

    /* Write colors */
    
    for(i = 0; i <= Settings->COL_Max; i++) {
        snprintf(colindex, sizeof(colindex), "col%.3d", i);
        snprintf(s, sizeof(s), "%3d %3d %3d", 
            Settings->Palette[i].r,
            Settings->Palette[i].g,
            Settings->Palette[i].b);
        set_config_string("palette", colindex, s);
    }
    
    set_config_int("settings", "dbg_symmetry_area_coloring", 
        Settings->DBG_SymmetryAreaColoring ? 1 : 0);

    if(file_exists(Filename, 0, NULL)) {
        override_config_file(NULL);
        pop_config_state();
    }
}

TSettings *ReadFractintMap(char *Filename, TSettings *Settings)
{
    FILE                        *fp;
    char                        Buffer[81];
    int                         r[256], g[256], b[256];
    unsigned int                i;
    
    if((fp = fopen(Filename, "rt")) != NULL) 
    {
        for(i = 0; i <= 255; i++) {
            if(!feof(fp))
                fgets(Buffer, 80, fp);
            else
                break;
            sscanf(Buffer, "%3d %3d %3d\n", &r[i], &g[i], &b[i]);
        }
        Settings->COL_Max = i - 1;
        fclose(fp);
    }
    
    /* 
    **  Reverse colors since Funiter starts at the bottom of the color list for
    **  The 'Escapetime diagram'. 
    */
    
    for(i = 0; i <= Settings->COL_Max; i++) 
    {
        Settings->Palette[Settings->COL_Max - i].r = (unsigned char) r[i];
        Settings->Palette[Settings->COL_Max - i].g = (unsigned char) g[i];
        Settings->Palette[Settings->COL_Max - i].b = (unsigned char) b[i]; 
        Settings->COL_List[i] = i;
    }
    return Settings;
}
