/*
    This file is part of Funiter,
    Real and complex function iteration software.

    cplexp.c - a complex experiment (functiontype: from C to C)

    Copyright (C) 1995-2009 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "cplexp.h"

static int      JuliaVisible = FALSE;

/*
**  Set values for the GraphPar.
*/

void SetComplexGraphPar(TCGraphPar *GraphPar, double zReal, double zImag, 
    double cReal, double cImag, double EscapeValue, unsigned int IterSteps, 
    unsigned int IterSkip, unsigned int IterMax, unsigned int Mirror)
{
    if(GraphPar != NULL) {
        GraphPar->zReal = zReal;
        GraphPar->zImag = zImag;
        GraphPar->cReal = cReal;
        GraphPar->cImag = cImag;
        GraphPar->EscapeValue = EscapeValue;
        GraphPar->IterSteps = IterSteps;
        GraphPar->IterSkip = IterSkip;
        GraphPar->IterMax = IterMax;
        GraphPar->Mirror = Mirror;
    }
}

/*
**  Load ComplexGraphPar.
*/

int LoadComplexGraphPar(FILE *fp, TCGraphPar *GraphPar)
{
    int         Res = 0;
    
    Res = fscanf(fp, "%lg,%lg,%lg,%lg\n", &GraphPar->zReal, &GraphPar->zImag, 
        &GraphPar->cReal, &GraphPar->cImag);
        
    Res = fscanf(fp, "%lg,%lg,%lg,%lg\n", &GraphPar->Coords->XMin, 
        &GraphPar->Coords->XMax, &GraphPar->Coords->YMin, 
        &GraphPar->Coords->YMax);

    /* Set coordinates and make sure that width and height are correct */

    SetCoords(GraphPar->Coords, 
        GraphPar->Coords->XMin, GraphPar->Coords->YMin, 
        GraphPar->Coords->XMax, GraphPar->Coords->YMax);

    Res = fscanf(fp, "%lg\n", &GraphPar->EscapeValue);

    Res = fscanf(fp, "%u,%u,%u\n", &GraphPar->IterMax, &GraphPar->IterSteps,
        &GraphPar->IterSkip);

    Res = fscanf(fp, "%u\n", &GraphPar->EscapeValueChanged);

    return Res;
}

/*
**  Save a ComplexGraphPar.
*/

int SaveComplexGraphPar(FILE *fp, TCGraphPar *GraphPar)
{
    int         Res;
    
    Res = fprintf(fp, "%.*g,%.*g,%.*g,%.*g\n", N_DBL_DIG,  GraphPar->zReal, 
        N_DBL_DIG, GraphPar->zImag, N_DBL_DIG, GraphPar->cReal, N_DBL_DIG, 
        GraphPar->cImag);
        
    Res = fprintf(fp, "%.*g,%.*g,%.*g,%.*g\n", N_DBL_DIG, GraphPar->Coords->XMin, 
        N_DBL_DIG, GraphPar->Coords->XMax, N_DBL_DIG, GraphPar->Coords->YMin, 
        N_DBL_DIG, GraphPar->Coords->YMax);
        
    Res = fprintf(fp, "%.*g\n", N_DBL_DIG, GraphPar->EscapeValue);

    Res = fprintf(fp, "%u,%u,%u\n", GraphPar->IterMax, GraphPar->IterSteps,
        GraphPar->IterSkip);
        
    Res = fprintf(fp, "%u\n", GraphPar->EscapeValueChanged);
    
    return Res;
}

/*
**  Creates a complex experiment including the four GraphPar's.
**  This function returns NULL on error, and a pointer to a new
**  Complex experiment on success.
*/

TComplexExp *CreateComplexExp(unsigned int XStart, unsigned int YStart,
    unsigned int XEnd, unsigned int YEnd)
{
    TComplexExp     *CplExp;
    
    if((CplExp = malloc(sizeof(*CplExp))) == NULL)
        return NULL;

#ifdef DEBUGMODE
    TRACE("FUNITER - cplexp.c: Malloc()'ed 'TComplexExp' data at %p\n", CplExp);
#endif
    
    CplExp->Settings = CreateSettings();
    if(CplExp->Settings == NULL) {
        free(CplExp);
        return NULL;
    }

    CplExp->Function = 0;
    CplExp->Diagram = SH_POINT;
    CplExp->Coloring = 0;
    CplExp->EscValueChanged = FALSE;
    CplExp->InvJuliaIterMax = 50000;
    strcpy(CplExp->FunctionStr, "z*z+c");

    /* Set GraphPar: Step by step */

    SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEP], 0, 0, 0, 0, 2, 1, 0, 250,
        0);
    CplExp->GraphPar[GR_C_STEP].Coords =
        CreateCoords(-3, -2.1875, 3, 2.1875, XStart, YStart, XEnd, YEnd, 56);
    if(CplExp->GraphPar[GR_C_STEP].Coords == NULL) {
        free(CplExp);
        return NULL;
    }

    /* Set GraphPar: Step by step inverse */

    SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEPINV], 0, 0, 0, 0, 2, 1, 0,
        250, 0);
    CplExp->GraphPar[GR_C_STEPINV].Coords =
        CreateCoords(-3, -2.1875, 3, 2.1875, XStart, YStart, XEnd, YEnd, 56);
    if(CplExp->GraphPar[GR_C_STEPINV].Coords == NULL) {
        free(CplExp);
        return NULL;
    }

    /* Set GraphPar: Orbits for variable z */

    SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_Z], 0, 0, -1, 0, 2, 1, 0,
        250, (1 << MIRROR_IMAG0) + (1 << MIRROR_Y) + (1 << MIRROR_X));

    CplExp->GraphPar[GR_C_ORBITS_Z].Coords =
        CreateCoords(-2, -1.4375, 2, 1.4375, XStart, YStart, XEnd, YEnd, 56);
    if(CplExp->GraphPar[GR_C_ORBITS_Z].Coords == NULL) {
        free(CplExp);
        return NULL;
    }

    /* Set GraphPar: Orbits for variable c */

    SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_C], 0, 0, 0, 0, 2, 1, 0,
        250, (1 << MIRROR_IMAG0) + (1 << MIRROR_X));
    CplExp->GraphPar[GR_C_ORBITS_C].Coords =
        CreateCoords(-2, -1.4375, 2, 1.4375, XStart, YStart, XEnd, YEnd, 56);
    if(CplExp->GraphPar[GR_C_ORBITS_C].Coords == NULL) {
        free(CplExp);
        return NULL;
    }

    /* Calculate escape values */

    CplExp->Graph = GR_C_ORBITS_C;
    CplExp->GraphPar[GR_C_ORBITS_C].EscapeValue =
        CalculateComplexFnEscapeValue(CplExp->GraphPar, CplExp->Function,
            CplExp->Graph);
    CplExp->GraphPar[GR_C_ORBITS_C].EscapeValueChanged = FALSE;

    CplExp->Graph = GR_C_ORBITS_Z;
    CplExp->GraphPar[GR_C_ORBITS_Z].EscapeValue =
        CalculateComplexFnEscapeValue(CplExp->GraphPar, CplExp->Function,
            CplExp->Graph);
    CplExp->GraphPar[GR_C_ORBITS_Z].EscapeValueChanged = FALSE;

    CplExp->Graph = GR_C_STEPINV;
    CplExp->GraphPar[GR_C_STEPINV].EscapeValue =
        CalculateComplexFnEscapeValue(CplExp->GraphPar, CplExp->Function,
            CplExp->Graph);
    CplExp->GraphPar[GR_C_STEPINV].EscapeValueChanged = FALSE;

    CplExp->Graph = GR_C_STEP;
    CplExp->GraphPar[GR_C_STEP].EscapeValue =
        CalculateComplexFnEscapeValue(CplExp->GraphPar, CplExp->Function,
            CplExp->Graph);
    CplExp->GraphPar[GR_C_STEP].EscapeValueChanged = FALSE;

    return(CplExp);
}

void FreeComplexExp(TComplexExp *CplExp)
{
    FreeCoords(CplExp->GraphPar[GR_C_STEP].Coords);
    FreeCoords(CplExp->GraphPar[GR_C_STEPINV].Coords);
    FreeCoords(CplExp->GraphPar[GR_C_ORBITS_C].Coords);
    FreeCoords(CplExp->GraphPar[GR_C_ORBITS_Z].Coords);

#ifdef DEBUGMODE
    TRACE("FUNITER - cplexp.c: Free()'ing 'TComplexExp' data at %p...\n",
        CplExp);
#endif

    free(CplExp);
}

/*
**  Save a ComplexExperiment.
*/

int LoadComplexExp(TComplexExp *ComplexExp, char *FileName, int Mode)
{
    int             Res = 0, Functiontype = 1;
    unsigned int    i;
    FILE            *fp;
    
    if(Mode == 1) { /* read old par files */ }
        
    if((fp = fopen(FileName, "r")) == NULL) return 0;

    Res = fscanf(fp, "%d", &Functiontype);
    if(Functiontype == 1) {
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEP]);
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEPINV]);
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_Z]);
        Res = LoadComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_C]);

        Res = fscanf(fp, "%d,%d,%d\n", &ComplexExp->Graph, &ComplexExp->Diagram, 
            &ComplexExp->Function); 

        LoadSettings(fp, ComplexExp->Settings);    
    }
    fclose(fp);
    if(get_config_int("main", "auto_par_convert", 0)) {
        
        /* 
        **  Reverse palette since it was written in gbr order for .par files 
        **  Normalize r,g and b values from 0-63 -> 0-255.
        */
        
        for(i = 0; i <= ComplexExp->Settings->COL_Max; i++) {
            if(ComplexExp->Settings->Palette[i].r < 64) 
                ComplexExp->Settings->Palette[i].r <<= 2;
            if(ComplexExp->Settings->Palette[i].g < 64) 
                ComplexExp->Settings->Palette[i].g <<= 2;
            if(ComplexExp->Settings->Palette[i].b < 64) 
                ComplexExp->Settings->Palette[i].b <<= 2;
        }
        WriteComplexConfig(ComplexExp);
        ReadComplexConfig(ComplexExp);
    }    
    return Functiontype;
}

/*
**  Load a ComplexExperiment from a configuration file.
*/

int ReadComplexConfig(TComplexExp *ComplexExp)
{
    char            EscValueStr[80] = "", FnParFileTag[6] = "", 
                    FnParFile[2048] = "", **s;
    int             argc, i;
    AL_CONST char   *p;
    TCGraphPar      *Step = &ComplexExp->GraphPar[GR_C_STEP], 
                    *StepInv = &ComplexExp->GraphPar[GR_C_STEPINV], 
                    *OrbitsZ = &ComplexExp->GraphPar[GR_C_ORBITS_Z], 
                    *OrbitsC = &ComplexExp->GraphPar[GR_C_ORBITS_C];
    
    ComplexExp->Function = 
        get_config_int("complex", "function", ComplexExp->Function);
    if(ComplexExp->Function > 3) ComplexExp->Function = 0;
    
    /*
    **  Extract the 'slot' for the correct function and set the corresponding 
    **  file.
    */
    
    snprintf(FnParFileTag, 5, "fn%d", ComplexExp->Function);
    if((p = get_config_string("complex", FnParFileTag, NULL)) != NULL) {
        strncpy(FnParFile, p, 2047);
        fix_filename_slashes(FnParFile);
    }
    
    if(file_exists(FnParFile, 0, NULL)) {
        push_config_state();
        override_config_file(FnParFile);
    }
    
    strncpy(ComplexExp->FunctionStr, 
        get_config_string("complex", "userfunction", "z*z+c"), 39);
    
    ComplexExp->Graph = get_config_int("complex", "graph", 0);
    if(ComplexExp->Graph > 3) ComplexExp->Graph = 0;
    
    switch(ComplexExp->Graph)
    {
        case GR_C_STEP:
        case GR_C_STEPINV:
            ComplexExp->Diagram = get_config_int("complex", "shape", 0);
            if(ComplexExp->Diagram > 3) ComplexExp->Diagram = 0;
            break;
        case GR_C_ORBITS_Z:
            ComplexExp->Diagram = get_config_int("complex", "zdiagram", 0);
            if(ComplexExp->Diagram > 3) ComplexExp->Diagram = 0;
            break;
        case GR_C_ORBITS_C:
            ComplexExp->Diagram = get_config_int("complex", "cdiagram", 0);
            if(ComplexExp->Diagram > 2) ComplexExp->Diagram = 0;
            break;
    }
            
    /*
    **  Read graph specific parameters 
    */
    
    /*
    **  Step-by-step iteration 
    */
    
    Step->cReal = get_config_double("complex_step", "creal", Step->cReal);
    Step->cImag = get_config_double("complex_step", "cimag", Step->cImag);
    
    SetCoords(Step->Coords,
        get_config_double("complex_step", "min_real_values", Step->Coords->XMin),
        get_config_double("complex_step", "min_imag_values", Step->Coords->YMin),
        get_config_double("complex_step", "max_real_values", Step->Coords->XMax),
        get_config_double("complex_step", "max_imag_values", Step->Coords->YMax));
    
    strncpy(EscValueStr, 
        get_config_string("complex_step", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        Step->EscapeValue = CalculateComplexFnEscapeValue(Step, 
            ComplexExp->Function, GR_C_STEP);
    else
        Step->EscapeValue = ustrtod(EscValueStr, NULL);
    
    Step->IterSteps = get_config_int("complex_step", "itersteps", 
        Step->IterSteps);
    Step->IterSkip = get_config_int("complex_step", "iterskip", 
        Step->IterSkip);
    
    /*
    **  Step-by-step inverse iteration 
    */
    
    StepInv->cReal = 
        get_config_double("complex_stepinv", "creal", StepInv->cReal);
    StepInv->cImag = 
        get_config_double("complex_stepinv", "cimag", StepInv->cImag);
    
    SetCoords(StepInv->Coords,
        get_config_double("complex_stepinv", "min_real_values", 
            StepInv->Coords->XMin),
        get_config_double("complex_stepinv", "min_imag_values", 
            StepInv->Coords->YMin),
        get_config_double("complex_stepinv", "max_real_values", 
            StepInv->Coords->XMax),
        get_config_double("complex_stepinv", "max_imag_values", 
            StepInv->Coords->YMax));
    
    strncpy(EscValueStr, 
        get_config_string("complex_stepinv", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        StepInv->EscapeValue = CalculateComplexFnEscapeValue(StepInv,
            ComplexExp->Function, GR_C_STEPINV);
    else {
        StepInv->EscapeValue = ustrtod(EscValueStr, NULL);
        StepInv->EscapeValueChanged = TRUE;
    }
    StepInv->IterSteps =    
        get_config_int("complex_stepinv", "itersteps", Step->IterSteps);
    StepInv->IterSkip = 
        get_config_int("complex_stepinv", "iterskip", Step->IterSkip);
    
    /*
    **  Orbitdiagram for variable z 
    */
    
    OrbitsZ->cReal = 
        get_config_double("complex_orbits_z", "creal", OrbitsZ->cReal);
    OrbitsZ->cImag = 
        get_config_double("complex_orbits_z", "cimag", OrbitsZ->cImag);
    
    SetCoords(OrbitsZ->Coords,
        get_config_double("complex_orbits_z", "min_real_values", 
            OrbitsZ->Coords->XMin),
        get_config_double("complex_orbits_z", "min_imag_values", 
            OrbitsZ->Coords->YMin),
        get_config_double("complex_orbits_z", "max_real_values", 
            OrbitsZ->Coords->XMax),
        get_config_double("complex_orbits_z", "max_imag_values", 
            OrbitsZ->Coords->YMax));
    
    strncpy(EscValueStr, 
        get_config_string("complex_orbits_z", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        OrbitsZ->EscapeValue = CalculateComplexFnEscapeValue(OrbitsZ,
            ComplexExp->Function, GR_C_ORBITS_Z);
    else {
        OrbitsZ->EscapeValue = ustrtod(EscValueStr, NULL);
        OrbitsZ->EscapeValueChanged = TRUE;
    }
    
    OrbitsZ->IterSkip = 
        get_config_int("complex_orbits_z", "iterskip", OrbitsZ->IterSkip);
    OrbitsZ->IterMax = 
        get_config_int("complex_orbits_z", "itermax", OrbitsZ->IterMax);
    
    s = get_config_argv("complex_orbits_z", "symmetry", &argc);
    if(s != NULL) {
        for(i = 0; i < argc; i++) {
            if(strcmp(s[i], "off") == 0) { 
                OrbitsZ->Mirror = 0; 
                break; 
            } 
            else if(strcmp(s[i], "mirror_x") == 0) 
                SETBIT(OrbitsZ->Mirror, MIRROR_X); 
            else if(strcmp(s[i], "mirror_y") == 0) 
                SETBIT(OrbitsZ->Mirror, MIRROR_Y); 
            else if(strcmp(s[i], "mirror_imag0") == 0) 
                SETBIT(OrbitsZ->Mirror, MIRROR_IMAG0);
            else { 
                OrbitsZ->Mirror = 0; 
                break; 
            }
        }
    }
    
    /*
    **  Orbitdiagram for variable c
    */
    
    OrbitsC->zReal = 
        get_config_double("complex_orbits_c", "zreal", OrbitsC->zReal);
    OrbitsC->zImag = 
        get_config_double("complex_orbits_c", "zimag", OrbitsC->zImag);
    
    SetCoords(OrbitsC->Coords,
        get_config_double("complex_orbits_c", "min_real_values", 
            OrbitsC->Coords->XMin),
        get_config_double("complex_orbits_c", "min_imag_values", 
            OrbitsC->Coords->YMin),
        get_config_double("complex_orbits_c", "max_real_values", 
            OrbitsC->Coords->XMax),
        get_config_double("complex_orbits_c", "max_imag_values", 
            OrbitsC->Coords->YMax));
    
    strncpy(EscValueStr, 
        get_config_string("complex_orbits_c", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        OrbitsC->EscapeValue = CalculateComplexFnEscapeValue(OrbitsC,
            ComplexExp->Function, GR_C_ORBITS_C);
    else {
        OrbitsC->EscapeValue = ustrtod(EscValueStr, NULL);
        OrbitsC->EscapeValueChanged = TRUE;
    }
    
    OrbitsC->IterSkip = 
        get_config_int("complex_orbits_c", "iterskip", OrbitsC->IterSkip);
    OrbitsC->IterMax = 
        get_config_int("complex_orbits_c", "itermax", OrbitsC->IterMax);
    
    s = get_config_argv("complex_orbits_c", "symmetry", &argc);
    if(s != NULL) {
        for(i = 0; i < argc; i++) {
            if(strcmp(s[i], "off") == 0) { 
                OrbitsC->Mirror = 0; 
                break; 
            } 
            else if(strcmp(s[i], "mirror_x") == 0) 
                SETBIT(OrbitsC->Mirror, MIRROR_X); 
            else if(strcmp(s[i], "mirror_y") == 0) 
                SETBIT(OrbitsC->Mirror, MIRROR_Y); 
            else if(strcmp(s[i], "mirror_imag0") == 0) 
                SETBIT(OrbitsC->Mirror, MIRROR_IMAG0);
            else { 
                OrbitsC->Mirror = 0; 
                break; 
            }
        }
    }
    
    /*
    **  Settings.
    */

    ComplexExp->Settings = ReadSettingsConfig(FnParFile, ComplexExp->Settings);
    
    Step->Coords->Mode = StepInv->Coords->Mode = OrbitsZ->Coords->Mode = 
    OrbitsC->Coords->Mode = 
        SetCoordsFlags(ComplexExp->Settings->COO_Status, 
                       ComplexExp->Settings->COO_Grid,
                       TRUE,
                       ComplexExp->Settings->COO_Calibrate, 
                       TRUE, TRUE); 
    
    /*
    **  Close a specified parameter file 
    */
            
    if(file_exists(FnParFile, 0, NULL)) {
        override_config_file(NULL);
        pop_config_state();
    }
    
    return 0;
}

/*
**  Write a ComplexExperiment to a configuration file.
*/

int WriteComplexConfig(TComplexExp *ComplexExp)
{
    char            FnParFileTag[6] = "", 
                    FnParBakFile[2048] = "", FnParFile[2048] = "",
                    MirrorStr[3][15], s[100], st[25];
    int             i, Count, FirstEntry = TRUE;
    AL_CONST char   *p;
    TCGraphPar      *Step = &ComplexExp->GraphPar[GR_C_STEP], 
                    *StepInv = &ComplexExp->GraphPar[GR_C_STEPINV], 
                    *OrbitsZ = &ComplexExp->GraphPar[GR_C_ORBITS_Z], 
                    *OrbitsC = &ComplexExp->GraphPar[GR_C_ORBITS_C];
    
    /*
    **  Extract the 'slot' for the correct function and set the corresponding 
    **  file.
    */
    
    snprintf(FnParFileTag, 5, "fn%d", ComplexExp->Function);
    if((p = get_config_string("complex", FnParFileTag, NULL)) != NULL) {
        strncpy(FnParFile, p, 2047);
        fix_filename_slashes(FnParFile);
    }
    else
        set_config_int("complex", "function", ComplexExp->Function);
        
    if(file_exists(FnParFile, 0, NULL)) {
        push_config_state();
        override_config_file(FnParFile);
        replace_extension(FnParBakFile, FnParFile, "bak", sizeof(FnParBakFile));                    
        if(strcmp(FnParFile, FnParBakFile) != 0)
            SimpleCopyTextFile(FnParBakFile, FnParFile);
    }
    
    set_config_string("complex", "userfunction", ComplexExp->FunctionStr);
    
    set_config_int("complex", "graph", ComplexExp->Graph);
    
    switch(ComplexExp->Graph)
    {
        case GR_C_STEP:
        case GR_C_STEPINV:
            set_config_int("complex", "shape", ComplexExp->Diagram);
            break;
        case GR_C_ORBITS_Z:
            set_config_int("complex", "zdiagram", ComplexExp->Diagram);
            break;
        case GR_C_ORBITS_C:
            set_config_int("complex", "cdiagram", ComplexExp->Diagram);
            break;
    }    
    
    /*
    **  Set graph specific parameters 
    */
    
    /* 
    **  Step-by-step iteration 
    */
    
    set_config_double("complex_step", "creal", Step->cReal);
    set_config_double("complex_step", "cimag", Step->cImag);
    
    set_config_double("complex_step", "min_real_values", Step->Coords->XMin);
    set_config_double("complex_step", "min_imag_values", Step->Coords->YMin);
    set_config_double("complex_step", "max_real_values", Step->Coords->XMax);
    set_config_double("complex_step", "max_imag_values", Step->Coords->YMax);
    
    if(!Step->EscapeValueChanged)
        set_config_string("complex_step", "escapevalue", "auto");
    else
        set_config_double("complex_step", "escapevalue", Step->EscapeValue);
    
    set_config_int("complex_step", "itersteps", Step->IterSteps);
    set_config_int("complex_step", "iterskip", Step->IterSkip);
    
    /* 
    **  Step-by-step inverse iteration 
    */
    
    set_config_double("complex_stepinv", "creal", StepInv->cReal);
    set_config_double("complex_stepinv", "cimag", StepInv->cImag);
    
    set_config_double("complex_stepinv", "min_real_values", 
        StepInv->Coords->XMin);
    set_config_double("complex_stepinv", "min_imag_values", 
        StepInv->Coords->YMin);
    set_config_double("complex_stepinv", "max_real_values", 
        StepInv->Coords->XMax);
    set_config_double("complex_stepinv", "max_imag_values", 
        StepInv->Coords->YMax);
    
    if(!Step->EscapeValueChanged)
        set_config_string("complex_step", "escapevalue", "auto");
    else
        set_config_double("complex_step", "escapevalue", Step->EscapeValue);
    
    set_config_int("complex_step", "itersteps", Step->IterSteps);
    set_config_int("complex_step", "iterskip", Step->IterSkip);
    
    /*
    **  Orbitdiagram for variable z 
    */
    
    set_config_double("complex_orbits_z", "creal", OrbitsZ->cReal);
    set_config_double("complex_orbits_z", "cimag", OrbitsZ->cImag);
    
    set_config_double("complex_orbits_z", "min_real_values", 
        OrbitsZ->Coords->XMin);
    set_config_double("complex_orbits_z", "min_imag_values", 
        OrbitsZ->Coords->YMin);
    set_config_double("complex_orbits_z", "max_real_values", 
        OrbitsZ->Coords->XMax);
    set_config_double("complex_orbits_z", "max_imag_values", 
        OrbitsZ->Coords->YMax);
    
    if(!OrbitsZ->EscapeValueChanged)
        set_config_string("complex_orbits_z", "escapevalue", "auto");
    else
        set_config_double("complex_orbits_z", "escapevalue", 
            OrbitsZ->EscapeValue);
    
    set_config_int("complex_orbits_z", "iterskip", OrbitsZ->IterSkip);
    set_config_int("complex_orbits_z", "itermax", OrbitsZ->IterMax);
    
    if(OrbitsZ->Mirror == 0) 
        strcpy(s, "off");
    else {
        memset(s, '\0',  99);
        Count = 0;
        if(TSTBIT(OrbitsZ->Mirror, MIRROR_X))
            strcpy(MirrorStr[Count++], "mirror_x");
        if(TSTBIT(OrbitsZ->Mirror, MIRROR_Y))
            strcpy(MirrorStr[Count++], "mirror_y");
        if(TSTBIT(OrbitsZ->Mirror, MIRROR_IMAG0))
            strcpy(MirrorStr[Count++], "mirror_imag0");
        for(i = 0; i < Count; i++) {
            if(FirstEntry) {
                FirstEntry = FALSE;
                sprintf(st, "%s", MirrorStr[i]);
                strcat(s, st);
            }
            else {
                sprintf(st, " %s", MirrorStr[i]);
                strcat(s, st);
            }
        }
    }
    set_config_string("complex_orbits_z", "symmetry", s);

    /* 
    **  Orbitdiagram for variable c
    */
    
    set_config_double("complex_orbits_c", "zreal", OrbitsC->zReal);
    set_config_double("complex_orbits_c", "zimag", OrbitsC->zImag);
    
    set_config_double("complex_orbits_c", "min_real_values", 
        OrbitsC->Coords->XMin);
    set_config_double("complex_orbits_c", "min_imag_values", 
        OrbitsC->Coords->YMin);
    set_config_double("complex_orbits_c", "max_real_values", 
        OrbitsC->Coords->XMax);
    set_config_double("complex_orbits_c", "max_imag_values", 
        OrbitsC->Coords->YMax);
    
    if(!OrbitsC->EscapeValueChanged)
        set_config_string("complex_orbits_c", "escapevalue", "auto");
    else
        set_config_double("complex_orbits_c", "escapevalue", 
            OrbitsC->EscapeValue);
    
    set_config_int("complex_orbits_c", "iterskip", OrbitsC->IterSkip);
    set_config_int("complex_orbits_c", "itermax", OrbitsC->IterMax);
    
    if(OrbitsC->Mirror == 0) 
        strcpy(s, "off");
    else {
        memset(s, '\0',  99);
        Count = 0;
        if(TSTBIT(OrbitsC->Mirror, MIRROR_X))
            strcpy(MirrorStr[Count++], "mirror_x");
        if(TSTBIT(OrbitsC->Mirror, MIRROR_Y))
            strcpy(MirrorStr[Count++], "mirror_y");
        if(TSTBIT(OrbitsC->Mirror, MIRROR_IMAG0))
            strcpy(MirrorStr[Count++], "mirror_imag0");

        for(i = 0; i < Count; i++) {
            if(FirstEntry) {
                FirstEntry = FALSE;
                sprintf(st, "%s", MirrorStr[i]);
                strcat(s, st);
            }
            else {
                sprintf(st, " %s", MirrorStr[i]);
                strcat(s, st);
            }
        }
    }
    set_config_string("complex_orbits_c", "symmetry", s);

    /*
    **  Settings.
    */

    WriteSettingsConfig(FnParFile, ComplexExp->Settings);
    
    /*
    **  Close a specified parameter file 
    */
            
    if(file_exists(FnParFile, 0, NULL)) {
        override_config_file(NULL);
        pop_config_state();
    }
    
    return 0;
}

/*
**  
**  Save a ComplexExperiment.
*/

int SaveComplexExp(TComplexExp *ComplexExp, char *FileName)
{
    FILE        *fp;
    
    if((fp = fopen(FileName, "w+")) == NULL) return 0;
    
    fprintf(fp, "%d\n", 1);
    
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEP]);
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_STEPINV]);
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_Z]);
    SaveComplexGraphPar(fp, &ComplexExp->GraphPar[GR_C_ORBITS_C]);
        
    fprintf(fp, "%d,%d,%d\n", ComplexExp->Graph, ComplexExp->Diagram, 
        ComplexExp->Function); 
    
    SaveSettings(fp, ComplexExp->Settings);    
      
    fclose(fp);
    return 0;
}

void SetComplexExpDefaultCoords(TCoords *Coords, int Function, int Graph)
{
    switch(Function) {

        /*
        **  z -> z ^ 2 + c
        **  z -> z ^ 3 + c
        **  z -> z ^ 4 + c
        */

        case 0:
        case 1:
        case 2:
            switch(Graph) {
                case GR_C_STEP:
                case GR_C_STEPINV:
                    SetCoords(Coords, -3, -2.1875, 3, 2.1875); 
                    break;
                case GR_C_ORBITS_Z:
                case GR_C_ORBITS_C:
                    SetCoords(Coords, -2, -1.4375, 2, 1.4375); 
                    break;
            }
            break;

        /*
        **  z -> c * z * (1 - z)
        */
        
        case 3:
            switch(Graph) {
                case GR_C_STEP:
                case GR_C_STEPINV:
                case GR_C_ORBITS_Z:
                case GR_C_ORBITS_C:
                    SetCoords(Coords, -4, -3, 4, 3);
                    break;
                default:
                    break;
            }
            break;
        
        /*
        **  z -> -cz ^ 3 + (1 + c) z ^ 2
        */
        
        case 4:
            switch(Graph) {
                case GR_C_STEP:
                case GR_C_STEPINV:
                case GR_C_ORBITS_Z:
                case GR_C_ORBITS_C:
                    SetCoords(Coords, -4, -3, 4, 3);
                    break;
                default:
                    break;
            }
            break;
        
        /*
        **  User defined
        */
        
        case 5:
            switch(Graph) {
                case GR_C_STEP:
                case GR_C_STEPINV:
                    SetCoords(Coords, -3, -2.1875, 3, 2.1875); 
                    break;
                case GR_C_ORBITS_Z:
                case GR_C_ORBITS_C:
                    SetCoords(Coords, -2, -1.4375, 2, 1.4375); 
                    break;
            }
            break;
    }
}

void SelectComplexExperiment(TComplexExp *CplExp, int Mode, int Function)
{
    SetComplexExpDefaultCoords(CplExp->GraphPar[GR_C_STEP].Coords, Function, 
        GR_C_STEP);
    SetComplexExpDefaultCoords(CplExp->GraphPar[GR_C_STEPINV].Coords, Function, 
        GR_C_STEPINV);
    SetComplexExpDefaultCoords(CplExp->GraphPar[GR_C_ORBITS_Z].Coords, Function, 
        GR_C_ORBITS_Z);
    SetComplexExpDefaultCoords(CplExp->GraphPar[GR_C_ORBITS_C].Coords, Function, 
        GR_C_ORBITS_C);

    switch(Function) {

        /*
        **  z -> z ^ 2 + c
        */

        case 0:
            if(Mode) {
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEP], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEPINV], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_Z], 
                    0, 0, -1, 0, 2, 1, 0, 250,
                    (1 << MIRROR_IMAG0) + (1 << MIRROR_Y) + (1 << MIRROR_X));
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_C], 
                    0, 0, 0, 0, 2, 1, 0, 250,     
                    (1 << MIRROR_IMAG0) + (1 << MIRROR_X));
            }
            break;
        
        /*
        **  z -> z ^ 3 + c
        */
        
        case 1:
            if(Mode) {
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEP], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEPINV], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_Z], 
                    0, 0, -0.3, 0, 2, 1, 0, 250, 
                    (1 << MIRROR_IMAG0) + (1 << MIRROR_Y) + (1 << MIRROR_X));
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_C], 
                    0, 0, 0, 0, 2, 1, 0, 250,
                    (1 << MIRROR_IMAG0) + (1 << MIRROR_Y) + (1 << MIRROR_X));    
            }
            break;

        /*
        **  z -> z ^ 4 + c
        */

        case 2:
            if(Mode) {
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEP], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEPINV], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_Z], 
                    0, 0, -0.3, 0, 2, 1, 0, 250,
                    (1 << MIRROR_IMAG0) + (1 << MIRROR_Y) + (1 << MIRROR_X));    
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_C], 
                    0, 0, 0, 0, 2, 1, 0, 250,    
                    (1 << MIRROR_IMAG0) + (1 << MIRROR_X));
            }
            break;


        /*
        **  User defined
        */

        case 3:
            if(Mode) {
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEP], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_STEPINV], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_Z], 
                    0, 0, -1, 0, 2, 1, 0, 250, 0);
                SetComplexGraphPar(&CplExp->GraphPar[GR_C_ORBITS_C], 
                    0, 0, 0, 0, 2, 1, 0, 250, 0);    
            }
            break;

        default:
            break;
    }
}

/*
**  This function could be improved, it's inefficient but at least more
**  efficient than the original which used a cursor that walked across the
**  screen to scan for a particular color :-).
*/

int IsInList(int *XArray, int *YArray, int x, int y, int *Size)
{
    int i = 0, flag;

    flag = FALSE;
    do {
        if((XArray[i] == x) && (YArray[i] == y)) {
            flag = TRUE;
            break;
        }
        i++;
    } while(i <= *Size);

    return(flag);
}

/*
**  Since BGI doesn't support different write-modes for circles, we have to
**  draw the circle ourselves (OK, a polygon was a possibility...).
*/

void DrawCircle(BITMAP *Bitmap, TCoords *Coords, int x, int y, int r, 
    int Color)
{
    int         j, NumberOfPoints, xa, ya;
    double      xr, yr, Angle, AspectRatio;

    AspectRatio = (double) (Coords->YEnd - Coords->YStart) /
        (double) (Coords->XEnd - Coords->XStart);
    xr = (double) r;
    yr = (double) r * AspectRatio / (Coords->Height / Coords->Width);
    NumberOfPoints = (int) (2.0 * M_PI * (double) max(xr, yr));
    for(j = 0; j < NumberOfPoints; j++) {
        Angle = M_PI / 180.0 * ((double) j * (360.0 / NumberOfPoints));
        xa = cos(Angle) * xr + (double) x;
        ya = sin(Angle) * yr + (double) y;
        putpixel(Bitmap, xa, ya, Color);
    }
}

/*
**  This function draws a line and stores all (converted) points in an array.
**  Simple linedrawing is used, no Bresenham-algorithm (yet).
**  PosInLine is used to continue counting when we call this routine multiple
**  times to draw the rectangle.
*/

int StoreLine(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, int y2, 
    int Color, int PosInLine, int NumPoints, TComplex **Points)
{
    int     x, y, c, s, i, j;
    double  r, ch, sh;
        
    s = (double) (x2 - x1);
    c = (double) (y2 - y1);
    r = sqrt(fabs(c) * fabs(c) + fabs(s) * fabs(s));
    if(r == 0.0) 
        return 0;

    if(NumPoints == 0) NumPoints = r;

    sh = s / r;
    ch = c / r;
    j = PosInLine;
    for(i = 0; i < (int) r; i++) {
        x = sh * (double) i + (double) x1;
        y = ch * (double) i + (double) y1;
        if(Points != NULL) {
            (*Points)[j].r = R2M_X(*Coords, abs((int) x));
            (*Points)[j].i = R2M_Y(*Coords, abs((int) y));
            j++;
        }
        else
            putpixel(Bitmap, x, y, Color);
    }
    return j;
}

/*
**  This function draws a rectangle and stores all (converted) points in an
**  array and requires StoreLine in order to do that :-).
*/

int StoreRectangle(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, 
    int y2, int Color, int NumPoints, TComplex **Points)
{
    int     Pos1, Pos2, Pos3, Pos4;

    Pos1 = StoreLine(Bitmap, Coords, x1, y1, x2, y1, Color, 0, NumPoints, 
        Points);
    Pos2 = StoreLine(Bitmap, Coords, x2, y1, x2, y2, Color, Pos1, NumPoints, 
        Points);
    Pos3 = StoreLine(Bitmap, Coords, x2, y2, x1, y2, Color, Pos2, NumPoints, 
        Points);
    Pos4 = StoreLine(Bitmap, Coords, x1, y2, x1, y1, Color, Pos3, NumPoints, 
        Points);
    return Pos4;
}

/*
**  This function draws a circle and stores all (converted) points in an
**  array.
*/

int StoreCircle(BITMAP *Bitmap, TCoords *Coords, int x, int y, 
    int r, int Color, int NumPoints, TComplex **Points)
{
    int     j, xa, ya, NumberOfPoints;
    double  Angle, AspectRatio, xr, yr;

    AspectRatio = (double) (Coords->YEnd - Coords->YStart) /
        (double) (Coords->XEnd - Coords->XStart);
    xr = (double) r;
    yr = (double) r * AspectRatio / (Coords->Height / Coords->Width);
    NumberOfPoints = NumPoints;
    for(j = 0; j < NumberOfPoints - 1; j++) {
        Angle = M_PI / 180.0 * ((double) j * (360.0 / NumberOfPoints));
        xa = cos(Angle) * xr + (double) x;
        ya = sin(Angle) * yr + (double) y;
        (*Points)[j].r = R2M_X(*Coords, xa);
        (*Points)[j].i = R2M_Y(*Coords, ya);
    }
    return NumberOfPoints;
}

/*
**  Create a shape and return the number of points or an error.
**  If Mode is negative the user will be able to select a shape, otherwise 
**  a realloc takes place in which case NumPoints holds the new size.
*/

TComplex *CreateShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int Color, 
    int *NumPoints, int *BufferSpace, 
    char **DefPointHelpText, char **DefShapeHelpText)
{
    int         x1, y1, x2, y2;
    TComplex    *Points;
    
    *NumPoints = DefineShape(Bitmap, ComplexExp, &x1, &y1, &x2, &y2, Color, 
        DefPointHelpText, DefShapeHelpText);        

    if(*NumPoints < 0)
        return NULL;

    if(*NumPoints == 0) {
        *NumPoints = -3;
        return(NULL);
    }
    
    if(*BufferSpace < *NumPoints) *BufferSpace = *NumPoints;
    if((Points = (TComplex *) malloc(sizeof(TComplex) * (*BufferSpace))) == NULL)
        return(NULL);

#ifdef DEBUGMODE
    TRACE("* Allocating Points array: %d for %d points at %p.\n", 
        *BufferSpace, *NumPoints, Points);
#endif

    switch(ComplexExp->Diagram) {
        case SH_POINT:
            Points[0].r = 
                R2M_X(*(ComplexExp->GraphPar[ComplexExp->Graph].Coords), x1);
            Points[0].i = 
                R2M_Y(*(ComplexExp->GraphPar[ComplexExp->Graph].Coords), y1);
            break;
        case SH_LINE:
            StoreLine(Bitmap, ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                x1, y1, x2, y2, Color, 0, *NumPoints, &Points);
            break;
        case SH_RECT:
            StoreRectangle(Bitmap, ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                x1, y1, x2, y2, Color, *NumPoints, &Points);
            break;
        case SH_CIRCLE:
            StoreCircle(Bitmap, ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                x1, y1, x2, Color, *NumPoints, &Points);
            break;
    }

    return Points;
}

int DefineShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int *x1, int *y1, 
    int *x2, int *y2, int Color, char **DefPointHelpText, 
    char **DefShapeHelpText)
{
    char        v1[256];
    int         XMickeys, YMickeys, Key = 0, ExtKey = 0, Col, xnew, ynew;
    double      Perimeter = 1.0, sp, cp, r, rold;
    TCoords     Coords;

    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    Col =  makecol(ComplexExp->Settings->Palette[Color].r,
            ComplexExp->Settings->Palette[Color].g,
            ComplexExp->Settings->Palette[Color].b);

    /*
    **  Select first point which is the upperleft
    **  corner of a reactangle or the centre of a circle.
    **  In case of points, the routine will not wait for a mouse-click.
    */

    do {
        *x1 = mouse_x;
        *y1 = mouse_y;
        get_mouse_mickeys(&XMickeys, &YMickeys);

        rest(1);

        if((XMickeys != 0) || (YMickeys != 0)) {
            sprintf(v1, "(%2.2f, %2.2f)", R2M_X(Coords, *x1), 
                R2M_Y(Coords, *y1));
            if(ComplexExp->Settings->LAY_Statusline)
            {
                StatusBar(1, -1, ComplexExp->Graph, 
                    ComplexExp->Settings->Palette, 
                    ComplexExp->Settings->COL_Max, v1);
            }
        }

        if(ComplexExp->Diagram == SH_POINT) {
            *x2 = *x1;
            *y2 = *y1;
        }

        if(keypressed()) {
            Key = readkey();
            ExtKey = Key >> 8;
            if(ExtKey == KEY_F1) {
                scare_mouse();
                if(ComplexExp->Diagram == SH_POINT) {
                    if(DefPointHelpText != NULL) 
                        InfoBox(DefPointHelpText, 2);
                }
                else {
                    if(DefShapeHelpText != NULL) 
                        InfoBox(DefShapeHelpText, 3);
                }                    
                unscare_mouse();
            }
            else if(ExtKey == KEY_ESC)
                return -2;
            else if((ExtKey == KEY_ENTER) && (ComplexExp->Diagram == SH_POINT)
                && (ComplexExp->Graph == GR_C_STEP))
                return 1;
            else if(((ExtKey == KEY_ENTER) || (ExtKey == KEY_PLUS_PAD) ||
                (ExtKey == KEY_MINUS)) && (ComplexExp->Diagram == SH_POINT) &&
                (ComplexExp->Graph == GR_C_STEPINV)) 
                return 1;
        }
    } while(!(mouse_b & 1)); 

    /*
    **  Second selection: not active for single dots, lower-right corner of
    **  a rectangle and radius of the circle.
    */

    *x2 = xnew = *x1;
    *y2 = ynew = *y1;
    while(1) {
        
        get_mouse_mickeys(&XMickeys, &YMickeys);

        if((XMickeys != 0) || (YMickeys != 0)) {
            xnew = mouse_x;
            ynew = mouse_y;

            sp = fabs((double) (xnew - *x1));
            cp = fabs((double) (ynew - *y1));
            r = sqrt(sp * sp + cp * cp);

            solid_mode();
            if((ComplexExp->Diagram == SH_RECT) || 
                (ComplexExp->Diagram == SH_LINE))
            { 
                sprintf(v1, "(%2.2f, %2.2f) - (%2.2f, %2.2f)", 
                    R2M_X(Coords, *x1), R2M_Y(Coords, *y1),
                    R2M_X(Coords, xnew), R2M_Y(Coords, ynew));
            }
            else {
                sprintf(v1, "(%2.2f, %2.2f) - (%2.2f)", 
                    R2M_X(Coords, *x1), R2M_Y(Coords, *y1), 
                    R2M_Y(Coords, *y1 + r));
            }
            if(ComplexExp->Settings->LAY_Statusline)
            {
                StatusBar(1, -1, ComplexExp->Graph, 
                    ComplexExp->Settings->Palette, 
                    ComplexExp->Settings->COL_Max, v1);
            }
        }       

        if(!(mouse_b & 1)) {
            switch(ComplexExp->Diagram) {
                case 1:
                case 2:
                    *x2 = xnew;
                    *y2 = ynew;
                    break;
                case 3:
                    *x2 = (int) (r);
                    *y2 = 0;
                    break;
            }
            break;
        }

        if((XMickeys != 0) || (YMickeys != 0)) {
            scare_mouse();
            xor_mode(TRUE);
            switch(ComplexExp->Diagram) {
                case 1:
                    StoreLine(Bitmap, 
                        ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                        *x1, *y1, xnew, ynew, Col, 0, 0, NULL);
                    StoreLine(Bitmap, 
                        ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                        *x1, *y1, *x2, *y2, Col, 0, 0, NULL);
                    Perimeter = r;
                    break;
                case 2:
                    StoreRectangle(Bitmap, 
                        ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                        *x1, *y1, xnew, ynew, Col, 0, NULL);
                    StoreRectangle(Bitmap, 
                        ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
                        *x1, *y1, *x2, *y2, Col, 0, NULL);
                    Perimeter = (sp + cp) * 2.0;
                    break;
                case 3:
                    DrawCircle(Bitmap, &Coords, *x1, *y1, r, Col);
                    DrawCircle(Bitmap, &Coords, *x1, *y1, rold, Col);
                    Perimeter = 2.0 * 3.1415926535 * r;
                    break;
            }
            xor_mode(FALSE);
            unscare_mouse();

            *x2 = xnew;
            *y2 = ynew;
            rold = r;
        }
        
        if(keypressed()) {
            Key = readkey();
            ExtKey = Key >> 8;
            if(ExtKey == KEY_F1) {
                scare_mouse();
                solid_mode();
                if(DefShapeHelpText != NULL) 
                    InfoBox(DefShapeHelpText, 3);
                unscare_mouse();
            }
            else if(ExtKey == KEY_ESC)
                return -2;
        }
    }
    solid_mode();
    return (int) (Perimeter);
}

/*
**  Draw a line between two points (which are represented by a small block).
*/

void IPDDrawLine(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex *Point,
    TComplex *PointLast, int Color, unsigned int Iter, int ClearMode)
{
    char            v1[40];
    int             xz, yz, xzlast, yzlast, Col;
    TCoords         Coords;
    TCGraphPar      GraphPar;
    
    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    Col = makecol(ComplexExp->Settings->Palette[Color].r,
            ComplexExp->Settings->Palette[Color].g,
            ComplexExp->Settings->Palette[Color].b);

    xz = M2R_X(Coords, Real(*Point));
    yz = M2R_Y(Coords, Imag(*Point));

    xzlast = M2R_X(Coords, Real(*PointLast));
    yzlast = M2R_Y(Coords, Imag(*PointLast));

    scare_mouse();
    
    if(Iter == 0 /*GraphPar.IterSkip*/)
        circlefill(Bitmap, xzlast, yzlast, 4, Col);
    else {
        if(ComplexExp->Settings->SSI_Lines) 
            line(Bitmap, xzlast, yzlast, xz, yz, Col);
    
        rectfill(Bitmap, xz - 2, yz - 2, xz + 2, yz + 2, Col);
    }        
    sprintf(v1, "z(%5d) = (%2.2f + %2.2fi)", Iter/*+ 1*/, Real(*Point), 
        Imag(*Point));
    if(ComplexExp->Settings->LAY_Statusline) 
    {
        StatusBar(1, -1, ComplexExp->Graph, ComplexExp->Settings->Palette, 
            ComplexExp->Settings->COL_Max, v1);
    }
    if(ComplexExp->Settings->SSI_Text && (!ClearMode)) {
        textprintf_ex(Bitmap, font, M2R_X(Coords, Real(*Point)), M2R_Y(Coords, 
            Imag(*Point)), Col, -1, "(%1.1f,%1.1f)", Real(*Point), 
            Imag(*Point));
    }
    
    unscare_mouse();
}

TComplex *IPDPlot(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex *Points, 
    int NumRoots, unsigned int Iter, unsigned int Color, unsigned int IterColor, 
    int *NumPoints, int *BufferSpace, int KeyPressed)
{
    /*char            *CurHelpStr = "[F1]=Help.";*/
    int                 ExitValue = TRUE, EscStatus, xc, yc, XScr, YScr, 
                        i, j, k, *PixelsX, *PixelsY, 
                        PrevNumPoints, RandomRoot;
    static unsigned int TempColor = 1;
    double              zrRoots[MAX_ROOT], ziRoots[MAX_ROOT];
    TComplex            c, LastPoint, Point;
    TCoords             Coords;
    TCGraphPar          GraphPar;

    PrevNumPoints = *NumPoints;
    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    c = Complex(GraphPar.cReal, GraphPar.cImag);

    XScr = Coords.XEnd;
    YScr = Coords.YEnd;

    /*
    **  Allocate memory for the x and y pixels and set to 0.
    */

    if(ComplexExp->Graph == GR_C_STEPINV) {
        if((PixelsX = malloc(*BufferSpace * sizeof(int))) == NULL) return NULL;
        if((PixelsY = malloc(*BufferSpace * sizeof(int))) == NULL) return NULL;
        memset(PixelsX, 0, (*BufferSpace - 1) * sizeof(int));
        memset(PixelsY, 0, (*BufferSpace - 1) * sizeof(int));
    }
    
    /* Set color for dots */
    
    if(ComplexExp->Coloring == CM_START) {
        if(Iter == 0) {
            TempColor++;
            if(TempColor == ComplexExp->Settings->COL_Max) 
                TempColor = 1; 
        }
        Color = TempColor;
    }
    else if(ComplexExp->Coloring == CM_ITERSTEP) {
        Color = (int) (Iter % ComplexExp->Settings->COL_Max) + 1;
        if(Color == (ComplexExp->Settings->COL_Max - 1)) 
            Color = 1;
    }
    else
        Color = IterColor;

    /* Step by step iteration */

    if(ComplexExp->Graph == GR_C_STEP) {
        for(j = 0; j < *NumPoints; j++) {

            LastPoint.r = Points[j].r;
            LastPoint.i = Points[j].i;

            if((ComplexExp->Diagram == SH_POINT) && (Iter == 0)) 
                IPDDrawLine(Bitmap, ComplexExp, &Points[0], &LastPoint,  
                    Color, Iter, KeyPressed);

            if((Iter > 0) && (ComplexExp->Diagram == SH_POINT))
            {
                CIterateFn(ComplexExp->Function, ComplexExp->FunctionStr,
                    Points[j].r, Points[j].i, c.r, c.i, GraphPar.EscapeValue, 
                    /* 
                    ** Since periodicity-check is set to false, pixelheight and
                    ** pixelwidth are 0. MaxPeriod can be set to 1 and MaxIter 
                    ** too.
                    */
                    0, 0, 1, 1, FALSE, 
                    &Points[j].r, &Points[j].i, &EscStatus, 
                    &ComplexExp->ParseErrNum, &ComplexExp->ParseErrPos);
            }
            
            xc = M2R_X(Coords, Points[j].r);
            yc = M2R_Y(Coords, Points[j].i);
            
            if(ComplexExp->Diagram == SH_POINT) {
                IPDDrawLine(Bitmap, ComplexExp, &Points[0], &LastPoint,  
                    Color, Iter, KeyPressed);
            }
            else
                putpixel(Bitmap, xc, yc, 
                    makecol(ComplexExp->Settings->Palette[Color].r,
                    ComplexExp->Settings->Palette[Color].g,
                    ComplexExp->Settings->Palette[Color].b));
        }
    }

    /* Step by step inverse iteration */

    else if(ComplexExp->Graph == GR_C_STEPINV) {
        i = 0;
        for(j = 0; j < PrevNumPoints; j++) {
            zrRoots[0] = Points[j].r;
            ziRoots[0] = Points[j].i;

            LastPoint.r = Points[j].r;
            LastPoint.i = Points[j].i;

            if((ComplexExp->Diagram == SH_POINT) && (Iter == 0)) {
                IPDDrawLine(Bitmap, ComplexExp, &Points[0], &LastPoint,  
                    Color, Iter, KeyPressed);
                break;
            }

            /* Resize pixel arrays */

            if((i + NumRoots) >= *BufferSpace) {
                *BufferSpace *= NumRoots;

                #ifdef DEBUGMODE
                    TRACE("* realloc() the complex number array from "
                        "%d to %d points.\n", i + 1, *BufferSpace);
                #endif
                Points = realloc(Points, *BufferSpace * sizeof(TComplex));
                #ifdef DEBUGMODE
                    TRACE("* realloc'ed complex number array: %d bytes at %p.\n", 
                        (i + 1) * sizeof(TComplex), Points);
                #endif

                #ifdef DEBUGMODE
                    TRACE("* realloc() the pixel-arrays from %d to %d\n", 
                        i, *BufferSpace);
                #endif
                PixelsX = realloc(PixelsX, *BufferSpace * sizeof(int));
                PixelsY = realloc(PixelsY, *BufferSpace * sizeof(int));
                #ifdef DEBUGMODE
                    TRACE("* realloc'ed the pixel-arrays:"
                        "%d bytes at x:%p and y:%p\n", 
                        *BufferSpace * sizeof(int), 
                        PixelsX, PixelsY);
                #endif
                if((Points == NULL) || (PixelsX == NULL) || (PixelsY == NULL)) {
                    ExitValue = FALSE;
                    break;
                } 
            }

            IterateInvFn(ComplexExp->Function, zrRoots, ziRoots, c.r, c.i);

            RandomRoot = rand() % NumRoots;

            for(k = 0; k < NumRoots; k++) {

                xc = M2R_X(Coords, zrRoots[k]);
                yc = M2R_Y(Coords, ziRoots[k]);

                /* Select random, all, negative or positve root */

                if(((KeyPressed == KEY_ENTER) && (k == RandomRoot)) || 
                    (KeyPressed == KEY_EQUALS) ||
                   ((KeyPressed == KEY_MINUS) && (zrRoots[k] < 0.0)) ||
                   ((KeyPressed == KEY_PLUS_PAD) && (zrRoots[k] >= 0.0)))
                {
                   if(!IsInList(PixelsX, PixelsY, xc, yc, &i)) {
                        PixelsX[i] = xc; 
                        PixelsY[i] = yc;
                        i++;
                        if(ComplexExp->Diagram == SH_POINT) {
                            Point.r = zrRoots[k];
                            Point.i = ziRoots[k];
                            IPDDrawLine(Bitmap, ComplexExp, &Point, &LastPoint,  
                                Color, Iter, KeyPressed);
                        }
                    }
                }
            }
        }

        if(Iter > 0) {

#ifdef DEBUGMODE
    TRACE("* Number of points was %d, now it is %d. The allocated space is %d\n", 
    PrevNumPoints, i, *BufferSpace);
#endif
            
            if(ExitValue) {
#ifdef DEBUGMODE
    TRACE("* Fill points array with converted values from the "
    "pixel arrays, %d in total.\n\n", i);
#endif
                for(j = 0; j < i; j++) {
                    Points[j].r = R2M_X(Coords, PixelsX[j]);
                    Points[j].i = R2M_Y(Coords, PixelsY[j]);
                    if(ComplexExp->Diagram != SH_POINT) {
                        putpixel(Bitmap, PixelsX[j], PixelsY[j], 
                            makecol(ComplexExp->Settings->Palette[Color].r,
                            ComplexExp->Settings->Palette[Color].g,
                            ComplexExp->Settings->Palette[Color].b));
                    }
                }
                
                /* 
                **  Set new value to NumPoints, which is returned in *NumPoints 
                */

                *NumPoints = i;
            }
        }
    }

    if(ComplexExp->Graph == GR_C_STEPINV) {
        if(PixelsX != NULL) free(PixelsX);
        if(PixelsY != NULL) free(PixelsY);
    }
    
    if(ExitValue)
        return Points;
    else
        return NULL;
}

int DoComplexStep(BITMAP *Bitmap, BITMAP *JuliaBitmap, TComplexExp *ComplexExp,
    int JuliaVisible, int NumRoots, char **StepHelpText, char **StepInvHelpText, 
    char **JuliaNoMemStr, char **DefPointHelpText, char **DefShapeHelpText)
{
    int                     IterColor = 0, Key = 0, ExtKey = 0, NumPoints = 1,
                            ShapeDefined = FALSE, EscStatus, OldGraph,
                            OldDiagram, BufferSpace = 4096;
    unsigned int            l;
    double                  PixelWidth, PixelHeight;
    TComplex                c;
    TComplex                *Points;
    TCoords                 Coords;
    TCGraphPar              GraphPar;

    /* Create some shortcuts */
    
    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    
    PixelWidth = (Coords.XMax - Coords.XMin) / (Coords.XEnd - Coords.XStart);
    PixelHeight = (Coords.YMax - Coords.YMin) / (Coords.YEnd - Coords.YStart);
    
    /* 
    **  Try to copy the image of the Julia-set on the screen to the JuliaBitmap.
    **  For 'delete' there is something added:
    **
    **  In case the Julia-set should be visible, but there was not enough mem.
    **  to create the JuliaBitmap, the user will be prompted with a question
    **  whether or not the Julia-set should be recalculated. Since I made the
    **  mistake of declaring Diagram not in GraphPar, this is limited to only
    **  the Julia inverse method.
    */
    
    if(JuliaVisible) {
        if(JuliaBitmap != NULL) 
            blit(JuliaBitmap, Bitmap, 0, 0, 0, 0, Coords.XEnd, Coords.YEnd);
        else {
            if(alert(JuliaNoMemStr[0], JuliaNoMemStr[1], JuliaNoMemStr[2],
                JuliaNoMemStr[3], JuliaNoMemStr[4], 13, 27) == 1) 
            {
                rectfill(Bitmap, Coords.XStart, Coords.YStart, Coords.XEnd, 
                    Coords.YEnd, 0);
                OldGraph = ComplexExp->Graph;
                OldDiagram = ComplexExp->Diagram;
                ComplexExp->Graph = GR_C_ORBITS_Z;
                if(ComplexExp->Function < 3)
                    ComplexExp->Diagram = DM_INVERSE;
                else
                    ComplexExp->Diagram = DM_BOUNDARY;
                DrawComplexOrbits(Bitmap, ComplexExp);
                ComplexExp->Graph = OldGraph;
                ComplexExp->Diagram = OldDiagram;
            }
            else {
                rectfill(Bitmap, Coords.XStart, Coords.YStart, Coords.XEnd, 
                    Coords.YEnd, 0);
            }
        }
    }
    else {
        rectfill(screen, Coords.XStart, Coords.YStart, Coords.XEnd, 
            Coords.YEnd, 0);
    }
    
    DrawGrid(Bitmap, font, &Coords);

    if(ComplexExp->Diagram == SH_POINT)
        ExtKey = InfoBox(DefPointHelpText, 2);
    else
        ExtKey = InfoBox(DefShapeHelpText, 3);
        
    if(ExtKey == KEY_ESC) 
        return 1;

    c = Complex(GraphPar.cReal, GraphPar.cImag);
    show_mouse(Bitmap);

    if(ComplexExp->Coloring == CM_SAME) {
        IterColor =
            makecol(
            ComplexExp->Settings->Palette[ComplexExp->Settings->COL_Function].r,
            ComplexExp->Settings->Palette[ComplexExp->Settings->COL_Function].g,
            ComplexExp->Settings->Palette[ComplexExp->Settings->COL_Function].b);
    }
    
    do {
        l = 0L;
        
        rest(1);

        if(!ShapeDefined) {
            Points = CreateShape(Bitmap, ComplexExp, 15, 
                &NumPoints, &BufferSpace, 
                DefPointHelpText, DefShapeHelpText);
        }

        /* Points couldn't be allocated */
        
        if(Points == NULL) {
            if(NumPoints == -3)
                continue;
            else 
                return 0;        
        }
        
        /* Esc-key */
        
        if(NumPoints == -2) { 
            show_mouse(NULL); 
            free(Points); 
            return 1; 
        }

        if(NumPoints >= 0) {
            ShapeDefined = TRUE;
            if(ComplexExp->Diagram == SH_POINT) 
                Points = IPDPlot(Bitmap, ComplexExp, Points, NumRoots, l, 0,
                    IterColor, &NumPoints, &BufferSpace, ExtKey); 
            l = 1L;
        }
        
        /* Restart Creation of shape */
        
        if(NumPoints < 0) { 
            printf("NumPoints < 0.\n"); 
            continue; 
        }

        do {
            if(l >= GraphPar.IterSkip) {

                if((l % GraphPar.IterSteps) == 0) {
                    
                    while(TRUE) {

                        rest(1);

                        /* 
                        **  right mousebutton selected? 
                        **  A new shape can be defined. 
                        */

                        if(mouse_b & 2) {
                            scare_mouse();
                            if(ShapeDefined) 
                                free(Points); 
                            ShapeDefined = FALSE;
                            unscare_mouse();
                            break; 
                        } 
                        
                        if(keypressed()) {
                            Key = readkey();
                            ExtKey = Key >> 8;
                        }
                        else 
                            continue;

                        if((ExtKey == KEY_ENTER) || (ExtKey == KEY_PLUS_PAD) ||
                            (ExtKey == KEY_MINUS) || (ExtKey == KEY_EQUALS)) 
                        {
                            break;
                        }

                        if(ExtKey == KEY_ESC) {
                            show_mouse(NULL);
                            if(ShapeDefined) 
                                free(Points);
                            ShapeDefined = FALSE;
                            clear_keybuf();
                            return 1;
                        }

                        if((ExtKey == KEY_DEL) || (ExtKey == KEY_D)) {
                            scare_mouse();
                            if(!JuliaVisible) {
                                rectfill(Bitmap, Coords.XStart, Coords.YStart,
                                    Coords.XEnd, Coords.YEnd, 0);
                            }
                            else {
                                if(JuliaBitmap != NULL) {
                                    blit(JuliaBitmap, Bitmap, 0, 0, 0, 0, 
                                        Coords.XEnd, Coords.YEnd);
                                }
                                else {
                                    if(alert(JuliaNoMemStr[0], 
                                        JuliaNoMemStr[1], JuliaNoMemStr[2],
                                        JuliaNoMemStr[3], JuliaNoMemStr[4], 13,
                                        27) == 1) 
                                    {
                                        rectfill(Bitmap, Coords.XStart, 
                                            Coords.YStart, Coords.XEnd, 
                                            Coords.YEnd, 0);
                                        OldGraph = ComplexExp->Graph;
                                        OldDiagram = ComplexExp->Diagram;
                                        ComplexExp->Graph = GR_C_ORBITS_Z;
                                        if(ComplexExp->Function < 3)
                                            ComplexExp->Diagram = DM_INVERSE;
                                        else
                                            ComplexExp->Diagram = DM_BOUNDARY;
                                        DrawComplexOrbits(Bitmap, ComplexExp);
                                        ComplexExp->Graph = OldGraph;
                                        ComplexExp->Diagram = OldDiagram;
                                    }
                                    else {
                                        rectfill(Bitmap, Coords.XStart, 
                                            Coords.YStart, Coords.XEnd, 
                                            Coords.YEnd, 0);
                                    }
                                }
                            }
                            if(ShapeDefined) 
                                free(Points);
                            ShapeDefined = FALSE;
                            DrawGrid(Bitmap, font, &Coords);
                            unscare_mouse();
                            l = 0L;
                            clear_keybuf();
                            break;
                        }

                        if(ExtKey == KEY_F1) {
                            if(ComplexExp->Graph == GR_C_STEPINV) {
                                if(ComplexExp->Diagram == SH_POINT)  
                                    InfoBox(StepInvHelpText + 1, 5);
                                else
                                    InfoBox(StepInvHelpText, 6);
                            }
                            else 
                                InfoBox(StepHelpText, 3);
                            clear_keybuf();
                        }
                    } /* end of while(TRUE) */

                    if(!ShapeDefined) break;
                }
                
                if(ShapeDefined) {
                    Points = IPDPlot(Bitmap, ComplexExp, Points, NumRoots, l, 0,
                        IterColor, &NumPoints, &BufferSpace, ExtKey); 
                    if(Points == NULL) return 0;
                }
            }
            
            l++;

            if(EscStatus && ShapeDefined && (ComplexExp->Graph == GR_C_STEP)) 
                continue;  
            
        } while(l <= MAXINT);
    } while(TRUE);
    return 1;
}


/*
**  Draw orbits for variable z and c.
*/

int DrawComplexOrbits(BITMAP *Bitmap, TComplexExp *ComplexExp)
{
    char            TextStr[100];
    int             Res, XStart, XEnd, YStart, YEnd, XScr, YScr, 
                    TempXStart, TempXEnd, TempXScr, MXStart, MXEnd, MXScr,
                    Width, Height, MirrorState, PrevMirrorState, cf, ri;
    unsigned long   Iter;
    double          PixelWidth, PixelHeight, EscapeValue, zReal[10], zImag[10];
    clock_t         StartTime, EndTime;
    TCoords         Coords;
    TCGraphPar      GraphPar;

    /* Init */

    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];
    
    EscapeValue = ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue;
    ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue =
        EscapeValue * EscapeValue;
    
    XStart = 0;
    YStart = 0;
    XEnd = Coords.XEnd - Coords.XStart - 1;
    YEnd = Coords.YEnd - Coords.YStart - 1;
    XScr = XEnd;
    YScr = YEnd;
    Width = XEnd + 1;
    Height = YEnd + 1;
    
    PixelWidth = (Coords.XMax - Coords.XMin) / Width;
    PixelHeight = (Coords.YMax - Coords.YMin) / Height;

    cf = ComplexExp->Settings->COL_Function;
    
    /* Julia-set: Inverse Iteration Method using random-method */
    
    if(ComplexExp->Graph == GR_C_ORBITS_Z && ComplexExp->Diagram == DM_INVERSE) 
    {
        zReal[0] = GraphPar.zReal;
        zImag[0] = GraphPar.zImag;

        for(Iter = 1; Iter <= ComplexExp->InvJuliaIterMax; Iter++) {
            
            if((ComplexExp->Function >= 0) && (ComplexExp->Function < 3)) {
            
                /* Iterate function */
                        
                IterateInvFn(ComplexExp->Function, zReal, zImag, 
                    GraphPar.cReal, GraphPar.cImag);
                
                /* Choose random root */
        
                ri = rand() % (ComplexExp->Function + 2);
                
                GraphPar.zReal = zReal[ri];
                GraphPar.zImag = zImag[ri];
            }
            
            /* Plot pixel */
            
            if(Iter >= GraphPar.IterSkip) {
                putpixel(Bitmap, M2R_X(Coords, GraphPar.zReal), 
                    M2R_Y(Coords, GraphPar.zImag),
                    makecol(ComplexExp->Settings->Palette[cf].r,
                    ComplexExp->Settings->Palette[cf].g,
                    ComplexExp->Settings->Palette[cf].b));
            }
            zReal[0] = GraphPar.zReal;
            zImag[0] = GraphPar.zImag;

        }
        DrawGrid(Bitmap, font, GraphPar.Coords);
        return 0;
    }

    StartTime = clock();

    MirrorState = MS_NONE;

    /*
    ** Mirror in x.
    */

    if(Coords.YMin < 0.0 && Coords.YMax > 0.0 && 
        (TSTBIT(GraphPar.Mirror, MIRROR_X))) 
    {
        if((TSTBIT(GraphPar.Mirror, MIRROR_IMAG0)) && (GraphPar.cImag == 0.0))
        {
            if(Coords.YMin + Coords.YMax > 0.0) {
                YStart = (Coords.YMax + Coords.YMin) / PixelHeight;
                YEnd = Coords.YMax / PixelHeight;
                MirrorState |= MS_X;
            }
            if(Coords.YMin + Coords.YMax < 0.0) {
                YStart = 0;
                YEnd = Coords.YMax / PixelHeight;
                YScr = YEnd * 2;
                MirrorState |= MS_X;
            }
            if(Coords.YMin + Coords.YMax == 0.0) 
            {
                YEnd = Height / 2;
                MirrorState |= MS_X;
            }
        }
    }

    /*
    **  Mirror in y.
    */

    if(Coords.XMin < 0.0 && Coords.XMax > 0.0 &&
        (TSTBIT(GraphPar.Mirror, MIRROR_Y))) 
    {
        if((TSTBIT(GraphPar.Mirror, MIRROR_IMAG0)) && (GraphPar.cImag == 0.0))
        {
            if(Coords.XMin + Coords.XMax > 0.0) {
                XStart = 0;
                XEnd = (-Coords.XMin - Coords.XMin) / PixelWidth;
                XScr = XEnd;
                MirrorState |= MS_Y;
            }
            if(Coords.XMin + Coords.XMax < 0.0) {
                XStart = (-Coords.XMax - Coords.XMin) / PixelWidth;
                XEnd = (0.0 - Coords.XMin) / PixelWidth;
                MirrorState |= MS_Y;
            }
            if(Coords.XMin + Coords.XMax == 0.0) {
                XEnd = Width / 2;
                MirrorState |= MS_Y;
            }
        }
    }
    
    /* 
    **  Try to determine whether all symmetric area's of the image have been
    **  calculated.
    */
    
    if((XStart == 0) && (((XEnd * 2) == Width) || ((XEnd + 1) == Width))) 
        MirrorState |= MS_X_COMPLETE;
    if((YStart == 0) && (((YEnd * 2) == Height - 1) || ((YEnd + 1) == Height))) 
        MirrorState |= MS_Y_COMPLETE;

    /* Draw area */

    Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, XScr, YScr,
        Width, Height, MirrorState, ComplexExp);
#ifdef DEBUGMODE
    TRACE("Mirror pass 1 (X/Y): (%4d , %4d) - (%4d , %4d) [XScr: %4d , YScr: %4d]\n",
        XStart, YStart, XEnd, YEnd, XScr, YScr);
    blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H); 
#endif

    /* Store coordinates for the central mirror part */

    PrevMirrorState = MirrorState;
    MXStart = XStart;
    MXEnd = XEnd;
    MXScr = XScr;

    /* Try to mirror remaining parts */

    /*
    **  Mirror in x (pass 2).
    */

    MirrorState &= ~(MS_X + MS_Y);

    if(Coords.YMin < 0.0 && Coords.YMax > 0.0 && 
        (TSTBIT(GraphPar.Mirror, MIRROR_X)) && 
        (!(MirrorState & MS_X_COMPLETE)))
    {
        /* Set screen coordinates of the remaining symmetric area */
        
        if(XStart > 0) {
            XEnd = XStart; XStart = 0; XScr = XEnd;
        }
        else {
            XStart = XEnd; XEnd = XScr = Coords.XEnd - 1;
        }

        if((TSTBIT(GraphPar.Mirror, MIRROR_IMAG0)) && (GraphPar.cImag == 0.0))
            MirrorState |= MS_X;

        /* Draw area */
        
        Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, 
            XScr, YScr, Width, Height, MirrorState, ComplexExp);
#ifdef DEBUGMODE
        TRACE("Mirror pass 2 (  X): (%4d , %4d) - (%4d , %4d) [XScr: %4d , YScr: %4d]\n",
            XStart, YStart, XEnd, YEnd, XScr, YScr);
        blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H);
#endif
    }

    TempXStart = XStart;
    TempXEnd = XEnd;
    TempXScr = XScr;

    /*
    **  Mirror in y (pass 2).
    */

    MirrorState &= ~(MS_X + MS_Y);

    XStart = MXStart;
    XEnd = MXEnd;
    XScr = MXScr;

    /* 
    **  Determine screen coordinates for the 2nd pass Y-mirror, or for the 
    **  remaining area that is not symmetric.
    */
    
    if(YStart > 0) { 
        YEnd = YStart; YStart = 0; YScr = YEnd;
    }
    else {
        YStart = YScr; YEnd = YScr = Coords.YEnd - 1;
    }

    if(Coords.XMin < 0.0 && Coords.XMax > 0.0 &&
        (TSTBIT(GraphPar.Mirror, MIRROR_Y)) && 
        (!(MirrorState & MS_Y_COMPLETE)))
    {
        if((TSTBIT(GraphPar.Mirror, MIRROR_IMAG0)) && (GraphPar.cImag == 0.0))
            MirrorState |= MS_Y;

        Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd,
            XScr, YScr, Width, Height, MirrorState, ComplexExp);
#ifdef DEBUGMODE
        TRACE("Mirror pass 2 (  Y): (%4d , %4d) - (%4d , %4d) [XScr: %4d , YScr: %4d]\n",
            XStart, YStart, XEnd, YEnd, XScr, YScr);
        blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H); 
#endif
    }

    XStart = TempXStart;
    XEnd = TempXEnd;
    XScr = TempXScr;

    /* The last part where no mirroring is possible */

    MirrorState &= ~(MS_X + MS_Y);
        
    Res = DrawComplexOrbitsArea(Bitmap, XStart, YStart, XEnd, YEnd, XScr, 
        YScr, Width, Height, MirrorState, ComplexExp);
#ifdef DEBUGMODE
    TRACE("Mirror pass 3 (  N): (%4d , %4d) - (%4d , %4d) [XScr: %4d , YScr: %4d]\n",
        XStart, YStart, XEnd, YEnd, XScr, YScr);
    blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H);
#endif

    EndTime = clock();

#ifdef DEBUGMODE
    sprintf(TextStr, "Total calc. time: %4.4lfs (<ESC> = close)", 
    ((EndTime - StartTime) / (double) CLOCKS_PER_SEC));
    StatusBar(ComplexExp->Settings->LAY_Statusline, -1, ComplexExp->Graph, 
    ComplexExp->Settings->Palette, ComplexExp->Settings->COL_Max, TextStr);
    readkey();
#endif

    ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue = EscapeValue;

    MirrorState = MS_NONE;
    
    /* Wait for escape key to draw */
    
    DrawGrid(Bitmap, font, GraphPar.Coords);
    return Res;
}

int DrawComplexOrbitsArea(BITMAP *Bitmap, unsigned int XStart, 
    unsigned int YStart, unsigned int XEnd, unsigned int YEnd, 
    unsigned int XScr, unsigned int YScr, unsigned int Width, 
    unsigned int Height, unsigned int MirrorState, TComplexExp *ComplexExp)
{
    unsigned char       Col = 0;
    unsigned int        MemPrevYLine = FALSE, XPos, YPos,
                        PrevYPixel, *PrevYLinePixel = 0, LastIter, Iter,
                        MaxPeriod;
    int                 AltKey, tmpe, Color;
    double              zRealTemp, cRealTemp, zImagTemp, cImagTemp, PixelWidth,
                        PixelHeight, XPixel, YPixel, tmpr, tmpi;
    TCoords             Coords;
    TCGraphPar          GraphPar;

    /* Init */

    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    GraphPar = ComplexExp->GraphPar[ComplexExp->Graph];

    PixelWidth = (Coords.XMax - Coords.XMin) / Width;
    PixelHeight = (Coords.YMax - Coords.YMin) / Height;

    MaxPeriod = (unsigned int) sqrt(GraphPar.IterMax);

    /*
    **  Array which holds the calculated pixels for the previous line.
    **  If there's not enough memory to hold the array, the pixel will be
    **  calculated. This function is just for the 'JM_BOUNDARY'.
    */

    if((PrevYLinePixel = (unsigned int *) malloc(Width *
        sizeof(unsigned int) + 1)) != NULL) 
    {
        MemPrevYLine = TRUE;
    }
    
    if(MemPrevYLine) memset(PrevYLinePixel, 0, Width);
    
    LastIter = 0L;

    for(YPos = YStart; YPos <= YEnd; YPos++) {
         
        YPixel = Coords.YMax - YPos * PixelHeight;
        if(ComplexExp->Graph == GR_C_ORBITS_C) {
            cImagTemp = YPixel;
            zImagTemp = GraphPar.zImag;
        }
        else {
            cImagTemp = GraphPar.cImag;
            cImagTemp = YPixel;
        }

        if(ComplexExp->PrgProc != NULL) 
            ComplexExp->PrgProc(YPos, YEnd);

        cImagTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ? 
            YPixel : GraphPar.cImag;
        zImagTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ?
            GraphPar.zImag : YPixel;

        for(XPos = XStart; XPos <= XEnd; XPos++) {

            XPixel = Coords.XMin + XPos * PixelWidth;

            cRealTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ?
                XPixel : GraphPar.cReal;
            zRealTemp = (ComplexExp->Graph == GR_C_ORBITS_C) ?
                GraphPar.zReal : XPixel;
            
            /* Iterate function */
            
            Iter = CIterateFn(ComplexExp->Function, 
                ComplexExp->FunctionStr,
                zRealTemp,
                (ComplexExp->Graph == GR_C_ORBITS_C) ?
                zImagTemp : zImagTemp - PixelHeight, cRealTemp,
                (ComplexExp->Graph == GR_C_ORBITS_C) ?
                cImagTemp - PixelHeight : cImagTemp,
                GraphPar.EscapeValue, PixelWidth, PixelHeight,
                MaxPeriod, GraphPar.IterMax, TRUE, &tmpr, &tmpi, &tmpe,
                &ComplexExp->ParseErrNum, &ComplexExp->ParseErrPos);

            /*
            **  If there was not enough memory for the previous line, and the
            **  diagram is 'Boundary', then the previous y pixel is calculated,
            **  in the other case it is stored in the array of
            **  'previous y pixels'.
            */

            if(ComplexExp->Diagram == DM_BOUNDARY) {
                if(!MemPrevYLine)
                    PrevYPixel = CIterateFn(ComplexExp->Function, 
                        ComplexExp->FunctionStr, 
                        zRealTemp, (ComplexExp->Graph == GR_C_ORBITS_C) ?
                        zImagTemp : zImagTemp - PixelHeight, cRealTemp,
                        (ComplexExp->Graph == GR_C_ORBITS_C) ?
                        cImagTemp - PixelHeight : cImagTemp,
                        GraphPar.EscapeValue, PixelWidth, PixelHeight,
                        MaxPeriod, GraphPar.IterMax, TRUE, &tmpr, &tmpi, &tmpe,
                        &ComplexExp->ParseErrNum, &ComplexExp->ParseErrPos);
                else
                    PrevYPixel = PrevYLinePixel[XPos];
            }

            /* Filled set */

            if(ComplexExp->Diagram == DM_FILLED) {
                if(Iter == 0)
                    Col = ComplexExp->Settings->COL_Function;
                else
                    goto Quit;
            }

            /* Escape Time Diagram */

            if(ComplexExp->Diagram == DM_ETD) {
                if(Iter == 0)
                    Col = ComplexExp->Settings->COL_Inside;
                else if(Iter == GraphPar.IterMax)
                    Col = ComplexExp->Settings->COL_Outside;
                else {
                    Col = ComplexExp->Settings->COL_Max - ((GraphPar.IterMax -
                        Iter) % ComplexExp->Settings->COL_Max);
                    if(Col == 0)
                        Col++;
                    else if(Col == ComplexExp->Settings->COL_Max)
                        Col--;
                }
            }

            /*
            **  Boundary:
            **
            **  This is a diagram that requires some explanation, certainly
            **  since the names of the variables don't exactly represent what
            **  they do. LastIter is the iteration-value for the last calculated
            **  x pixel, and PrevYPixel the iteration value for the pixel just
            **  above the current. To determine the pixel that is closest to
            **  the M-sets boudary the current pixel is compared to the ones
            **  just mentioned. If the current pixel is not in the set and one
            **  of the others is, or the other way around, the pixel is set
            **  (which means we consider it on the boundary of the M-set).
            */

            if(ComplexExp->Diagram == DM_BOUNDARY) {
                if((!LastIter && Iter) || (LastIter && !Iter) ||
                    (!PrevYPixel && Iter) || (PrevYPixel && !Iter)) 
                {
                    Col = ComplexExp->Settings->COL_Function;
                }
                else
                    goto Quit;
            }

            /* Create selected color */

            Color = makecol(ComplexExp->Settings->Palette[Col].r,
                ComplexExp->Settings->Palette[Col].g,
                ComplexExp->Settings->Palette[Col].b);

            /* Drawing */

            putpixel(Bitmap, XPos, YPos, Color);

            if(MirrorState & MS_X) {
                if((TSTBIT(GraphPar.Mirror, MIRROR_IMAG0)) && 
                    (GraphPar.cImag == 0.0))
                {
#ifdef DEBUGMODE
                    if(ComplexExp->Settings->DBG_SymmetryAreaColoring) 
                        Color ^= 127;
#endif
                    putpixel(Bitmap, XPos, YScr - YPos + YStart, Color);
                }
                else 
                {
#ifdef DEBUGMODE
                    if(ComplexExp->Settings->DBG_SymmetryAreaColoring) 
                        Color ^= 32513;
#endif
                    putpixel(Bitmap, XScr - XPos + XStart, YScr - YPos + YStart,
                        Color);
                }
            }

            if(MirrorState & MS_Y) {
#ifdef DEBUGMODE
                    if(ComplexExp->Settings->DBG_SymmetryAreaColoring) 
                        Color ^= 8355840;
#endif
                putpixel(Bitmap, XScr - XPos + XStart, YPos, Color);
            }

            if((MirrorState & MS_X) && (MirrorState & MS_Y)) {
#ifdef DEBUGMODE
                    if(ComplexExp->Settings->DBG_SymmetryAreaColoring)
                        Color = ((Color >> 16) ^ 255) + 
                            ((Color >> 8) ^ 255) + (Color ^ 255);
#endif
                putpixel(Bitmap, XScr - XPos + XStart, YScr - YPos + YStart,
                    Color);
            }

            /* I know... but it is probably somewhat quicker :-) */

            Quit:

            /* Store iterationvalue for the y pixel in the array */

            if(MemPrevYLine) PrevYLinePixel[XPos] = Iter;

            /* Store iterationvalue for the previous x pixel */

            LastIter = Iter;
        }
        
        /* Look for a keypress */
        
        if(keypressed()) {
            AltKey = readkey() >> 8;
            if(AltKey == KEY_ESC) { 
                if(MemPrevYLine) free(PrevYLinePixel);
                return 0;
            }
        }
    }

    if(MemPrevYLine) 
        free(PrevYLinePixel);
    
    return 1;
}

/*
**  calculate complex escapevalue.
*/

double CalculateComplexFnEscapeValue(TCGraphPar *GraphPar, int Function, 
    int Graph)
{
    double      cr, ci, miny, maxy, cw, p;

    cw = 0.0;

    cr = GraphPar->cReal;
    ci = GraphPar->cImag;
    
    miny = GraphPar->Coords->YMin;
    maxy = GraphPar->Coords->YMax;

    switch(Function)
    {
        case 0:
        case 1:
        case 2:
            cw = max(2.0, sqrt(cr * cr + ci * ci));
            break;
        default:
            cw = 0.0;
            break;
    }
    if(cw == 0.0) 
        p = 2.0; 
    else 
        p = 1.0;
    
    return(max(max(fabs(p * miny), fabs(p * maxy)), cw));
}

/*
**  Iterate function.
*/

unsigned int CIterateFn(int Function, char *FunctionStr, double zReal, 
    double zImag, double cReal, double cImag,  double EscapeValue, 
    double PixelWidth, double PixelHeight, unsigned int MaxPeriod, 
    unsigned int MaxIter, int PeriodCheck, double *RealRes, double *ImagRes, 
    int *EscStatus, int *ErrorNum, int *ErrorPos)
{
    unsigned int    Iter, Period;
    double          zRealTemp, wReal = zReal, wImag = zImag, zRealMul,
                    zImagMul;
    TComplex        z, c;
        
    *EscStatus = FALSE;
    Period = 0;
    
    for(Iter = MaxIter; Iter > 0; Iter--) {
        
        zRealMul = zReal * zReal;
        zImagMul = zImag * zImag;
        
        if(zRealMul + zImagMul > EscapeValue) {
            *EscStatus = TRUE;
            break;
        }

        if(Function == 0) {
            zRealTemp = zReal;
            zReal = zRealMul - zImagMul + cReal;
            zImag = (zRealTemp + zRealTemp) * zImag + cImag;
        }
        else {
            z.r = zReal; z.i = zImag;
            c.r = cReal; c.i = cImag;
        }
                            
        switch(Function) {
            case 1:
                
                /* z -> z^3 + c */
                
                z = CAdd(CMul(CMul(z, z), z), c);            
                break;
            case 2:
                
                /* z -> z^4 + c */
                
                z = CAdd(CMul(CMul(CMul(z, z), z), z), c);            
                break;
            case 3:
                
                /* User defined */
                
                z = CParse(ErrorNum, ErrorPos, FunctionStr, 2, 
                    'z', z, 'c', c);
                break;
            default:
                break;
        }

        if(Function > 0) {
            zReal = z.r; 
            zImag = z.i;
        }
            
        /*
        **  Check if the difference between z and z' is less than the size of
        **  one pixel, in which case the iteration cycle can end.
        */
        
        if(PeriodCheck) {
            if(fabs(zReal - wReal) < PixelWidth &&
                fabs(zImag - wImag) < PixelHeight) 
            {
                Iter = 0;
                break;
            }

            if(Period > MaxPeriod) {
                Period = 0;
                wReal = zReal;
                wImag = zImag;
            }
            Period++;
        }
    }
    *RealRes = zReal;
    *ImagRes = zImag;
    return Iter;
}

/*
**  Inverse function.
*/

void IterateInvFn(int Function, double *zReal, double *zImag, double cReal, 
    double cImag)
{
    TComplex    z, c, Root[10];
    int         i;

    z = Complex(zReal[0], zImag[0]);
    c = Complex(cReal, cImag);
    
    switch(Function) {
        case 0:
        case 1:
        case 2:
            CSqrt(Root, CSub(z, c), Function + 2);
            break;
        default:
            break;
    }
        
    for(i = 0; i < (Function + 2); i++) {
        zReal[i] = Real(Root[i]);
        zImag[i] = Imag(Root[i]);
    }
}

/*
**  This routine can be used to jump from one graph to another.
**  For the current (source/original) graph the routine uses the member in 
**  TComplexExp, DestGraph is the graph to which should be jumped. 
**  InfoStr points to a text which should display the key to hit for help,
**  as HelpText should point to an array of lines for the actual help-text.
**  Mode selects whether to display a small image of the 'inverse Julia-set"
**  while a c is selected on the "Mandelbrot-set".
**  On 'cancel' the routine returns -1, on 'ok' it returns the selected graph. 
*/

int ComplexJump(TComplexExp *ComplexExp, int DestGraph, char *InfoStr, 
    char *HelpText[], int Mode)
{
    char        Str[255], *TmpInfoStr;
    int         Key = 0, AltKey = 0, XMickeys, YMickeys, tmp_xp = 0, tmp_yp = 0, 
                tmp_b = 0, JuliaPreview = TRUE;
    BITMAP      *JuliaBitmap, *MandelBkBitmap;
    TCoords     Coords;
    TComplexExp *SmallJuliaSet;
    
    Coords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);

    /* Orbit diagram for var. c  ->  Orbit diagram for var. z */

    if(ComplexExp->Graph == GR_C_ORBITS_C && DestGraph == GR_C_ORBITS_Z) {
        
        if(Mode) {
            SmallJuliaSet = CreateComplexExp(0, 0, 99, 99);
            
            SmallJuliaSet->Graph = GR_C_ORBITS_Z;
            SmallJuliaSet->Diagram = DM_INVERSE;
            SmallJuliaSet->GraphPar[GR_C_ORBITS_Z].IterMax = 500;
            
            MandelBkBitmap = create_bitmap(100, 100);
            blit(screen, MandelBkBitmap, 0, SCREEN_H - 120, 0, 0, 99, 99);
            
            JuliaBitmap = create_bitmap(100, 100);
            clear_bitmap(JuliaBitmap); 
        }

        if((TmpInfoStr = (char *) malloc(strlen(InfoStr) + 80)) == NULL) 
            return -2;
        
        strncpy(TmpInfoStr, InfoStr, strlen(InfoStr) + 1);

        if(DestGraph == GR_C_ORBITS_Z) 
            strcat(TmpInfoStr, " c = %2.*f + %2.*fi ");
        if(DestGraph == GR_C_ORBITS_C) 
            strcat(TmpInfoStr, " z = %2.*f + %2.*fi ");
        
        show_mouse(screen);
        tmp_xp = mouse_x;
        tmp_yp = mouse_y;
        tmp_b = mouse_b;
        do {
            tmp_b = mouse_b;
            get_mouse_mickeys(&XMickeys, &YMickeys);
            if((XMickeys != 0) || (YMickeys != 0)) {
                tmp_xp = mouse_x;
                tmp_yp = mouse_y;
                if(Mode && JuliaPreview && (ComplexExp->Function < 3)) {
                    SmallJuliaSet->GraphPar[SmallJuliaSet->Graph].cReal = 
                        R2M_X(Coords, mouse_x);
                    SmallJuliaSet->GraphPar[SmallJuliaSet->Graph].cImag = 
                        R2M_Y(Coords, mouse_y);
                    SmallJuliaSet->GraphPar[SmallJuliaSet->Graph].Coords->Mode =
                        0;
                    SmallJuliaSet->InvJuliaIterMax = 150;
                    SmallJuliaSet->Function = ComplexExp->Function;
                    clear_bitmap(JuliaBitmap);
                    DrawComplexOrbits(JuliaBitmap, SmallJuliaSet);
                    acquire_screen();
                    blit(JuliaBitmap, screen, 0, 0, 0, SCREEN_H - 120, 99, 99);
                    release_screen();
                }
                sprintf(Str, TmpInfoStr, 4, R2M_X(Coords, mouse_x), 
                    4, R2M_Y(Coords, mouse_y));
                if(ComplexExp->Settings->LAY_Statusline)
                {
                    StatusBar(1, -1, ComplexExp->Graph, 
                        ComplexExp->Settings->Palette, 
                        ComplexExp->Settings->COL_Max, Str);
                }
            }
            if(keypressed()) {
                Key = readkey();
                AltKey = Key >> 8;
                if(HelpText != NULL) { 
                    if(AltKey == KEY_F1) 
                        InfoBox(HelpText, 5);
                }
                if(Mode) { 
                    if(AltKey == KEY_P) { 
                        JuliaPreview = !JuliaPreview;
                        if(!JuliaPreview) {
                            blit(MandelBkBitmap, screen, 0, 0, 0, 
                                SCREEN_H - 120, 99, 99);
                        }
                        else {
                            blit(JuliaBitmap, screen, 0, 0, 0, SCREEN_H - 120, 
                                99, 99);
                        }
                    }
                }
            }
        } while(!(tmp_b & 1) && !(tmp_b & 2) && AltKey != KEY_ENTER && 
            AltKey != KEY_I && AltKey != KEY_ESC);

        show_mouse(NULL);

        free(TmpInfoStr);
        if(Mode) {
            blit(MandelBkBitmap, screen, 0, 0, 0, SCREEN_H - 120, 99, 99);
            FreeComplexExp(SmallJuliaSet);
            destroy_bitmap(JuliaBitmap);
            destroy_bitmap(MandelBkBitmap);
        }
        
        if(AltKey == KEY_ESC) 
            return(-1);
    }

    /* Step by step iteration  ->  Orbit diagram for var. z */

    if((ComplexExp->Graph == GR_C_STEP || ComplexExp->Graph == GR_C_STEPINV) &&
        DestGraph == GR_C_ORBITS_Z) 
    {
        if(ComplexExp->Graph == GR_C_STEPINV) {
            ComplexExp->GraphPar[DestGraph].cReal = 
                ComplexExp->GraphPar[ComplexExp->Graph].cReal =
                ComplexExp->GraphPar[GR_C_STEPINV].cReal;         
            ComplexExp->GraphPar[DestGraph].cImag =  
                ComplexExp->GraphPar[ComplexExp->Graph].cImag =
                ComplexExp->GraphPar[GR_C_STEPINV].cImag;         
        }
        else { 
            ComplexExp->GraphPar[DestGraph].cReal = 
                ComplexExp->GraphPar[ComplexExp->Graph].cReal =
                ComplexExp->GraphPar[GR_C_STEP].cReal;         
            ComplexExp->GraphPar[DestGraph].cImag =  
                ComplexExp->GraphPar[ComplexExp->Graph].cImag =
                ComplexExp->GraphPar[GR_C_STEP].cImag;
        }
        ComplexExp->Graph = DestGraph;        
    }

    /* Orbit diagram for var. z  ->  Orbit diagram for var. c */

    if(ComplexExp->Graph == GR_C_ORBITS_Z && DestGraph == GR_C_ORBITS_C) {
        ComplexExp->Graph = DestGraph;   
        
        /* Set coordinates to default so that the entire M-set is displayed */

        SetComplexExpDefaultCoords(
            ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
            ComplexExp->Function, DestGraph);
        
        /* Set z to 0 */
         
        ComplexExp->GraphPar[DestGraph].zReal = 0; 
        ComplexExp->GraphPar[DestGraph].zImag = 0; 
    }        

    /* Orbit diagram for var. c  ->  Orbit diagram for var. z */
    
    if(ComplexExp->Graph == GR_C_ORBITS_C && DestGraph == GR_C_ORBITS_Z) {
        ComplexExp->Graph = DestGraph;   
        
        /* If right mousebutton is clicked we switch to inverse */
        
        if((tmp_b & 2) || (AltKey == KEY_I)) 
            ComplexExp->Diagram = DM_INVERSE;

        /* Set coordinates to default so that the entire J-set is displayed */
        
        SetComplexExpDefaultCoords(
            ComplexExp->GraphPar[ComplexExp->Graph].Coords, 
            ComplexExp->Function, DestGraph);
        
        /* Convert screen coordinates and copy them to real and imag c */
         
        ComplexExp->GraphPar[DestGraph].cReal = R2M_X(Coords, tmp_xp); 
        ComplexExp->GraphPar[DestGraph].cImag = R2M_Y(Coords, tmp_yp); 
    }
    
    /* Orbit diagram for var. z  ->  Step by step iteration */
    
    if(ComplexExp->Graph == GR_C_ORBITS_Z && 
        (DestGraph == GR_C_STEP || DestGraph == GR_C_STEPINV)) 
    {
        
        /* Copy the entire GraphPar to destination graph */

        SetComplexGraphPar(&ComplexExp->GraphPar[DestGraph],
            ComplexExp->GraphPar[GR_C_ORBITS_Z].zReal,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].zImag,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].cReal,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].cImag,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].EscapeValue,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].IterSteps,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].IterSkip,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].IterMax, 0);
        SetCoords(ComplexExp->GraphPar[DestGraph].Coords,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].Coords->XMin,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].Coords->YMin,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].Coords->XMax,
            ComplexExp->GraphPar[GR_C_ORBITS_Z].Coords->YMax);
        
        /* Set new graph */
        
        ComplexExp->Graph = DestGraph;
    }        
    return(ComplexExp->Graph);    
}

int ComplexDraw(BITMAP *Bitmap, BITMAP *JuliaBitmap, TComplexExp *ComplexExp, 
    int NumRoots, 
    char **StepHelpText, char **StepInvHelpText, char **JuliaNoMemStr, 
    char **DefPointHelpText, char **DefShapeHelpText)
{
    unsigned int        Mode = 0;
    TCGraphPar          SGraphPar, SIGraphPar, JGraphPar;  
    
    if(Bitmap == NULL) Bitmap = screen;

    clear_bitmap(Bitmap);
    
    SGraphPar = ComplexExp->GraphPar[GR_C_STEP];
    SIGraphPar = ComplexExp->GraphPar[GR_C_STEPINV];
    JGraphPar = ComplexExp->GraphPar[GR_C_ORBITS_Z];
    
    switch(ComplexExp->Graph) {
        case GR_C_STEP:
            if((memcmp(&SGraphPar, &JGraphPar, sizeof(TCGraphPar)) != 0) && 
                !JuliaVisible) 
            { 
                clear_bitmap(Bitmap);
                JuliaVisible = FALSE;
            }
            DoComplexStep(screen, JuliaBitmap, ComplexExp, JuliaVisible, NumRoots, 
                StepHelpText, StepInvHelpText, JuliaNoMemStr, DefPointHelpText, 
                DefShapeHelpText);
            break;
        case GR_C_STEPINV:
            if((memcmp(&SIGraphPar, &JGraphPar, sizeof(TCGraphPar)) != 0) && 
                !JuliaVisible)
            { 
                clear_bitmap(Bitmap);
                JuliaVisible = FALSE;
            }
            DoComplexStep(screen, JuliaBitmap, ComplexExp, JuliaVisible, NumRoots,
                StepHelpText, StepInvHelpText, JuliaNoMemStr, DefPointHelpText, 
                DefShapeHelpText);
            break;
        case GR_C_ORBITS_Z:
            clear_bitmap(JuliaBitmap);
            JuliaVisible = TRUE;
            DrawComplexOrbits(JuliaBitmap, ComplexExp);
            blit(JuliaBitmap, Bitmap, 0,0, 0, 0, SCREEN_W, SCREEN_H);
            if(Bitmap != screen)
                blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H); 
            break;
        case GR_C_ORBITS_C:
            JuliaVisible = FALSE;
            DrawComplexOrbits(Bitmap, ComplexExp);
            if(Bitmap != screen)
                blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H); 
            break;
        default:
            break;            
    }            
    if(ComplexExp->Settings->LAY_Statusline) {
        Mode = 1; /* Display 'Mandelbrot' by default */
        if(((ComplexExp->GraphPar[GR_C_ORBITS_C].zReal != 0.0) ||
            (ComplexExp->GraphPar[GR_C_ORBITS_C].zImag != 0.0)) &&
            (ComplexExp->Graph == GR_C_ORBITS_C))
        {
            Mode = 3; /* Don't display 'Mandelbrot' in case z is not 0 */
        }
        if((ComplexExp->Graph == GR_C_ORBITS_C) && (ComplexExp->Function > 0))
            Mode = 3; /* Don't display 'Mandelbrot' if function is not z^2+c */
        StatusBar(Mode, -1, ComplexExp->Graph, ComplexExp->Settings->Palette, 
            ComplexExp->Settings->COL_Max, NULL);
    }
    return JuliaVisible ;
}
