/*
    This file is part of Funiter,
    Real and complex function iteration software.

    misc.c - some misc. routines.

    Copyright (C) 1995-2007 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "misc.h"

double max(double a, double b) { return((a > b) ? (a) : (b)); }
double min(double a, double b) { return((a < b) ? (a) : (b)); }

/* 
**  get_config_double:
**
**  Reads a double from the configuration file.
**
**  Adapted from the 'get_config_float' in Allegro 4.2.0's 'config.c'.
*/

double get_config_double(AL_CONST char *section, AL_CONST char *name, double def)
{
    AL_CONST char   *s = get_config_string(section, name, NULL);

    return(((s) && (ugetc(s))) ? (ustrtod(s, NULL)) : (def)); 
}

/* 
**  get_config_double:
**
**  Writes a double to the configuration file.
**
**  Adapted from the 'set_config_float' in Allegro 4.2.0's 'config.c'.
*/

void set_config_double(AL_CONST char *section, AL_CONST char *name, double val)
{
    char            buf[80] = "";
    
    uszprintf(buf, sizeof(buf), "%.*g", DBL_DIG, val);
    set_config_string(section, name, buf);
    return;
}

/*
**  Search for the highest string in an array of strings.
*/

int GetHighestStrLen(char *Str[], int First, int Last)
{
    int     i, length = 0;

    for(i = First; i <= Last; i++) {
        if(Str[i] != NULL) {
            if(gui_strlen(Str[i]) > length)
                length = gui_strlen(Str[i]);
        }
    }

    return(length);
}

int SimpleCopyTextFile(char *dest, char *src)
{
    FILE    *f_out, *f_in;

    if((f_in = fopen(src, "rt")) == NULL) return 1;
    if((f_out = fopen(dest, "wt")) == NULL) return 1;

    while(!feof(f_in)) 
        fputc(fgetc(f_in), f_out);

    fclose(f_out);
    fclose(f_in);
    
    return 0;
}

/*
**  WriteEPS
**
**  Write a bitmap to a Postscript file.
**  EPS = Encapsulated PostScript
**
**  Bitmap      - Pointer to a bitmap.  
**  w           - Bitmap width.
**  h           - Bitmap height.
**  fp          - FILE *, pointer naar bestandsinformatie.
**  Mode        - int, 0 = portrait, 1 = landscape
*/

int WriteEPS(BITMAP *Bitmap, int w, int h, FILE *fp, int Mode)
{
    int         xs, x, y, c, pp;
    //long int    colpow = pow(2.0, (double) get_color_depth);

    /* Write comment-header */

    fputs("%!PS-Adobe-3.0 EPSF\n", fp);
    fputs("%%Creator: Funiter by Stijn Wolters\n", fp);
    fputs("%%Pages: 1\n", fp);
    if(Mode == LANDSCAPE)
        fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n", 
            (int) (36.0 * ((double) w / 640.0)), 
            (int) (78.0 * ((double) h / 480.0)), 
            (int) (521.0 * ((double) w / 640.0)),
            (int) (720.0 * ((double) h / 480.0)));
    else /*if(Mode == PORTRAIT)*/
        fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n", 
            (int) (32.0 * ((double) w / 640.0)), 
            (int) (64.0 * ((double) h / 480.0)), 
            (int) (570.0 * ((double) w / 640.0)),
            (int) (472.0 * ((double) h / 480.0)));

    fputs("%%EndComments\n", fp);

    /* Create dictionary and write gfx status */

    fputs("gsave\n", fp);
    fputs("20 dict begin\n", fp);
    fputs("gsave\n", fp);
    fputs("2 setlinewidth\n", fp);

    /*
    **  Line drawing function, parameters:
    **      - graylevel (real) 0..1
    **      - xbegin, xend, y
    */

    fputs("/l { setrgbcolor moveto lineto stroke } bind def\n", fp);

    /* translation */

    if(Mode == LANDSCAPE)
        fprintf(fp, "%d %d translate\n",
            (int) (40.0 * ((double) w / 640.0)),
            (int) (80.0 * ((double) h / 480.0)));
            
    else /*if(Mode == PORTRAIT)*/ {
        fputs("0.84 0.84 scale\n", fp);
        fputs("-90 rotate\n", fp);
        fprintf(fp, "%d %d translate\n",
            (int) (-560.0 * ((double) w / 640.0)),
            (int) (40.0 * ((double) h / 480.0)));
    }

    /* Encode the image */

    xs = x = w - 1;
    for(y = h - 1; y >= 0; y--) {
        do {
            c = getpixel(Bitmap, x, y);
            pp = c;
            do {
                x--;
                c = getpixel(Bitmap, x, y);
            } while(c == pp && x > 0);

            fprintf(fp, "\n%d %d %d %d %g %g %g l",
                y, x, y, xs, 
                getr(pp) / 256.0, getg(pp) / 256.0, getb(pp) / 256.0);
            
            if(x == 0) {
                xs = x = w - 1;
                break;
            }
            else
                xs = x;
        } while(x >= 0);
    }
    fputs("\nshowpage\n", fp);
    fputs("grestore\n", fp);
    fputs("end\n", fp);
    fputs("%%Trailer", fp);

    return 0;
}
