/*
    This file is part of Funiter,
    Real and complex function iteration software.

    help.c - some routines which are part of the funiter help-function.

    Copyright (C) 1995-2009 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "help.h"

static char   *ConstGraphStr[] = 
    {    
        "Step-by-step iteration",
        "Orbit diagram for var. x",
        "Orbit diagram for var. c",
        "Step-by-step iteration",
        "Step-by-step inverse iteration",
        "Orbit diagram for var. z (Julia)",
        "Orbit diagram for var. c (Mandelbrot)"
    };

/*
**  This function draws a box on screen with information, it waits for a key
**  which is returned by the function.
**
**  Char **Text         -   Lines with the text.
**  int MaxItem         -   Number of items - 1.
*/

int InfoBox(char **Text, int MaxItem)
{
    int         x, y, w, h, i, ch;
    BITMAP      *Bitmap, *BitmapHelp;

    w = GetHighestStrLen(Text, 0, MaxItem) + 12;
    if(w > SCREEN_W) 
        w = SCREEN_W;
    h = (MaxItem + 1) * text_height(font) + 12;

    Bitmap = create_bitmap(w, h);
    if(Bitmap == NULL) return -1;
    
    BitmapHelp = create_bitmap(w, h);
    if(BitmapHelp == NULL) return -1;
    
    clear_to_color(BitmapHelp, makecol(255, 255, 255));

    x = (SCREEN_W - w) / 2;
    y = (SCREEN_H - h) / 2;

    rect(BitmapHelp, 2, 2, w - 2, h - 2, makecol(0, 0, 0));

    for(i = 0; i <= MaxItem; i++) {
        textout_ex(BitmapHelp, font, Text[i], 6, 6 + i * text_height(font), 
            makecol(0, 0, 0), -1);
    }

    blit(screen, Bitmap, x, y, 0, 0, w, h);

    blit(BitmapHelp, screen, 0, 0, x, y, w, h);
    
    ch = readkey(); 

    blit(Bitmap, screen, 0, 0, x, y, w, h);

    destroy_bitmap(BitmapHelp);
    destroy_bitmap(Bitmap);
    return(ch);
}

void StatusBar(int Mode, int RealGraph, int ComplexGraph, RGB *Palette, 
    unsigned int MaxCol, char *ValueStr)
{
    int             SLGraphMenuWidth, SLGraphMaxItemLength, TempGraph = -1,
                    uw, h;
    unsigned int    i;
    char            **GraphStr = GetItemStrings("SLGRAPH", &SLGraphMenuWidth, 
                    &SLGraphMaxItemLength, 7), *gp;
    BITMAP          *Bitmap;
    
    if(!(Mode & 1)) 
        return;
    
    if(GraphStr == NULL)
        DestroyItemStrings(GraphStr, 7); 
        
    Bitmap = create_bitmap(SCREEN_W, 20);
    if(Bitmap == NULL) return;
    
    clear_bitmap(Bitmap);

    /* draw a line between the image and the beginning of the statusline */

    hline(Bitmap, 0, 1, SCREEN_W, makecol(255, 255, 255));

    /* Draw the used colorpalette */
    
    if(MaxCol > 127) {
        h = 4; 
        uw = 1;
    }
    else {
        h = 8;
        uw = (int) (128.0 / (double) (MaxCol + 1));
    } 
    
    for(i = 0; i <= MaxCol; i++)
    {
        if(i <= 127) 
        {
            rectfill(Bitmap, 
                (SCREEN_W - 1) - i * uw, 8, (SCREEN_W - 1) - (i * uw + (uw / 2)),
                8 + h, makecol(Palette[i].r, Palette[i].g, Palette[i].b));
        }
        else 
        {
            rectfill(Bitmap, 
                (SCREEN_W - 1) - (i - 128) * uw, 8 + h + 1, 
                (SCREEN_W - 1) - ((i - 128) * uw + (uw / 2)), 8 + h * 2 + 1, 
                makecol(Palette[i].r, Palette[i].g, Palette[i].b));
        }        
    }
    
    /* Print the name of the current Graph */

    if(RealGraph >= 0 && ComplexGraph < 0)
        TempGraph = RealGraph;
    else if(RealGraph < 0 && ComplexGraph >= 0)
        TempGraph = ComplexGraph + 3;
    else
        TempGraph = -1;
    
    /*
    **  If the complex 'orbit diagram for var. c' is selected and z <> 0, it is
    **  not a true Mandelbrot-set so the text '(Mandelbrot)' is removed by
    **  selecting the 'orbit diagram for var. c' from the RealExp->Graph.
    */

    if((TempGraph == 6) && (Mode & 2)) TempGraph = 2;
    
    if(TempGraph >= 0) {
        
        if(GraphStr == NULL) 
            gp = ConstGraphStr[TempGraph]; 
        else
            gp = GraphStr[TempGraph];
        
        rect(Bitmap, SCREEN_W - 132 - gui_strlen(gp), 6, SCREEN_W - 130, 19, 
            15);
        textout_ex(Bitmap, font, gp, SCREEN_W - 130 - gui_strlen(gp), 6 + (14 - 
            text_height(font)) / 2, makecol(255, 255, 255), -1);

        /* print additional info such as iteration-step or iteration-values */

        if(ValueStr != NULL) {
            rect(Bitmap, SCREEN_W - 137 - (gui_strlen(gp) + 
                gui_strlen(ValueStr)), 6, SCREEN_W - 135 - gui_strlen(gp), 19, 
                makecol(255, 255, 255));
            textout_ex(Bitmap, font, ValueStr, SCREEN_W - 135 - 
                (gui_strlen(gp) + gui_strlen(ValueStr)), 6 + (14 - 
                text_height(font)) / 2, makecol(255, 255, 255), -1);
        }
    }

    acquire_screen();
    blit(Bitmap, screen, 0, 0, 0, SCREEN_H - 20, SCREEN_W, 20);
    release_screen();
    vsync();
    
    destroy_bitmap(Bitmap);
    DestroyItemStrings(GraphStr, 7);
}
