/*
    This file is part of Funiter,
    Real and complex function iteration software.

    cntmenu.c - Funiter's (old) central menu routine.

    Copyright (C) 1995-2009 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen). 

    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "cntmenu.h"

int MnuDraw(int Functiontype, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    BITMAP  *Bitmap, *JuliaBitmap;
    int     MSGMenuWidth, MSGMaxItemLength, CDPMenuWidth, CDPMaxItemLength,
            CDSMenuWidth, CDSMaxItemLength, CSIMenuWidth, CSIMaxItemLength, 
            RCSMenuWidth, RCSMaxItemLength, RFEMenuWidth, RFEMaxItemLength, 
            CentralMenuStatus = 0, JuliaVisible = FALSE,
            JuliaDiagram = DM_INVERSE;
    char    **MSGStr = GetItemStrings("MESSAGES", &MSGMenuWidth,
            &MSGMaxItemLength, 22),
            **CDPStr = GetItemStrings("CDEFPOINTHELP", &CDPMenuWidth,
            &CDPMaxItemLength, 3),
            **CDSStr = GetItemStrings("CDEFSHAPEHELP", &CDSMenuWidth, 
            &CDSMaxItemLength, 4),
            **CSIStr = GetItemStrings("CSTEPINVHELP", &CSIMenuWidth, 
            &CSIMaxItemLength, 7),
            **RCSStr = GetItemStrings("RCSTEPHELP", &RCSMenuWidth,
            &RCSMaxItemLength, 4),
            **RFEStr = GetItemStrings("RFERRORS", &RFEMenuWidth,
            &RFEMaxItemLength, 8),
            **AlertStr;
                
    scare_mouse();
    if(!Functiontype) {
        while(CentralMenuStatus != -1) {
            if(RealExp->Settings->LAY_Statusline) {
                StatusBar(1, RealExp->Graph, -1, RealExp->Settings->Palette,
                RealExp->Settings->COL_Max, 
                (MSGStr != NULL) ? MSGStr[6] : CalcHelpStr);
            }
            RealExp->ParseErrNum = 0;
            RealDraw(RealExp, 
                (RCSStr != NULL) ? RCSStr : RealComplexStepHelpStr);
            
            if((RealExp->ParseErrNum != 0) && (RealExp->Function == 6)) 
            {
                if((AlertStr = (char **) malloc(sizeof(char *) * 4)) != NULL) 
                {
                    AlertStr[0] = (RFEStr != NULL) ? RFEStr[0] : FnErrText[0];
                    AlertStr[1] = (RFEStr != NULL) ? 
                        RFEStr[RealExp->ParseErrNum] : 
                        FnErrText[RealExp->ParseErrNum];
                    if((AlertStr[2] = (char *) malloc(sizeof(char *) * 2)) 
                        != NULL) 
                    {
                        strcpy(AlertStr[2], " ");
                        
                        AlertStr[3] = (RCSStr != NULL) ? RCSStr[3] : 
                            RealComplexStepHelpStr[3];
                        
                        while((InfoBox(AlertStr, 3) >> 8) != KEY_ESC) ;
                        
                        free(AlertStr[2]);
                    }
                    free(AlertStr);
                }
            }
            if((RealExp->Graph == GR_R_ORBITS_X) || 
                (RealExp->Graph == GR_R_ORBITS_C))
            {
                if(RealExp->Settings->LAY_Statusline) {
                    StatusBar(1, RealExp->Graph, -1, RealExp->Settings->Palette,
                    RealExp->Settings->COL_Max,
                    (MSGStr != NULL) ? MSGStr[21] : "<Esc> = to central menu");
                }
                while((readkey() >> 8) != KEY_ESC)
                    ;
                if(RealExp->Settings->LAY_Statusline) {
                    StatusBar(1, RealExp->Graph, -1, RealExp->Settings->Palette, 
                        RealExp->Settings->COL_Max, NULL);
                }
            }

            CentralMenuStatus = DoRCentralMenu(RealExp);
        }
    }
    else {
        Bitmap = create_bitmap(SCREEN_W, SCREEN_H);
        if(Bitmap == NULL) Bitmap = screen;
        JuliaBitmap = create_bitmap(SCREEN_W, SCREEN_H);
        while(CentralMenuStatus != -1) {
            if(ComplexExp->Settings->LAY_Statusline) {
                if((ComplexExp->Graph != GR_C_ORBITS_C) ||
                    ((ComplexExp->Graph == GR_C_ORBITS_C) &&
                    (ComplexExp->GraphPar[GR_C_ORBITS_C].zReal == 0.0) &&
                    (ComplexExp->GraphPar[GR_C_ORBITS_C].zImag == 0.0) &&
                    (ComplexExp->Function == 0)))
                {
                    StatusBar(1, -1, ComplexExp->Graph, 
                    ComplexExp->Settings->Palette,
                    ComplexExp->Settings->COL_Max, 
                    (MSGStr != NULL) ? MSGStr[6] : CalcHelpStr);
                }
                else
                {
                    StatusBar(3, -1, ComplexExp->Graph,
                    ComplexExp->Settings->Palette, 
                    ComplexExp->Settings->COL_Max, 
                    (MSGStr != NULL) ? MSGStr[6] : CalcHelpStr);
                }
            }
            JuliaVisible = ComplexDraw(Bitmap, JuliaBitmap, ComplexExp, 
                ComplexExp->Function + 2,  
                (RCSStr != NULL) ? RCSStr : RealComplexStepHelpStr, 
                (CSIStr != NULL) ? CSIStr : ComplexStepInvHelpStr, 
                (MSGStr != NULL) ? MSGStr : JuliaNoMemStr,
                (CDPStr != NULL) ? CDPStr : DefPointHelpStr, 
                (CDSStr != NULL) ? CDSStr : DefShapeHelpStr);

            if(ComplexExp->Graph == GR_C_ORBITS_Z) 
                JuliaDiagram = ComplexExp->Diagram;

            if((ComplexExp->Graph == GR_C_ORBITS_Z) ||
                (ComplexExp->Graph == GR_C_ORBITS_C))
            {
                if(ComplexExp->Settings->LAY_Statusline) {
                    StatusBar(1, -1, ComplexExp->Graph,
                        ComplexExp->Settings->Palette,
                        ComplexExp->Settings->COL_Max,
                        (MSGStr != NULL) ? 
                        MSGStr[21] : 
                        "<Esc> = to central menu");
                }
                while((readkey() >> 8) != KEY_ESC)
                    ;
                if(ComplexExp->Settings->LAY_Statusline) {
                    StatusBar(1, -1, ComplexExp->Graph,
                    ComplexExp->Settings->Palette,
                    ComplexExp->Settings->COL_Max, NULL);
                }
            }

            CentralMenuStatus = DoCCentralMenu(JuliaBitmap, JuliaVisible, 
                JuliaDiagram, ComplexExp);
        }
        free(JuliaBitmap);
        if(Bitmap != NULL) free(Bitmap);
    }
    unscare_mouse();
    DestroyItemStrings(RFEStr, 8);
    DestroyItemStrings(RCSStr, 4);
    DestroyItemStrings(CSIStr, 7);
    DestroyItemStrings(CDSStr, 4);
    DestroyItemStrings(CDPStr, 3);
    DestroyItemStrings(MSGStr, 22);
    return 0;
}

static int RStepSubMenu(TRealExp *RealExp)
{
    TConsole            *Console;
    TMenu               *Menu; 
    int                 AltKey, ExitValue, RSMenuWidth, RSMaxItemLength, i = 0;
    char                **RSStr = GetItemStrings("RJSUBMENU", &RSMenuWidth, 
                        &RSMaxItemLength, 6);

    /* Diagram selection for 'Step by step iteration' */

    if(RSStr == NULL) {
        Console = alconio_init(35, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, 33, 0, BLACK, WHITE, WHITE, BLACK,
            5, 0, RJSItems);
    }
    else {
        Console = alconio_init(RSMaxItemLength + 2, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, RSMaxItemLength, 0, BLACK, WHITE,
            WHITE, BLACK, 5, 0, RJSItems);
        for(i = 0; i <= 4; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &RSStr[i]);
    }

    DrawMenu(Console, Menu);

    ExitValue = 0;
    while(TRUE) {
        AltKey = DoMenu(Console, Menu);

        if((AltKey == KEY_W) || ((Menu->ItemIndex == 0) &&
            (AltKey == KEY_ENTER))) 
        {
            RealExp->Diagram = DM_WEB;
            break;
        }
        if((AltKey == KEY_I) || ((Menu->ItemIndex == 1) &&
            (AltKey == KEY_ENTER))) 
        {
            RealExp->Diagram = DM_ITERVALUES;
            break;
        }
        if((AltKey == KEY_S) || ((Menu->ItemIndex == 2) &&
            (AltKey == KEY_ENTER))) 
        {
            RealExp->Diagram = DM_STEP;
            break;
        }
        if((AltKey == KEY_C) || ((Menu->ItemIndex == 3) &&
            (AltKey == KEY_ENTER))) 
        {
            RealExp->Diagram = DM_COMBIWEBSTEP;
            break;
        }
        if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 4) &&
            (AltKey == KEY_ENTER))) 
        {
            ExitValue = -1;
            break;
        }
        else
            continue;
    }
    
    DestroyMenu(Menu);
    DestroyItemStrings(RSStr, 6);
    a_putimage(Console, Console->ScreenContents);
    alconio_exit(Console);

    return ExitValue;               
}

static int CStepColorSubMenu(TComplexExp *ComplexExp)
{
    TConsole            *Console;
    TMenu               *Menu;
    int                 AltKey, ExitValue, CSMenuWidth, CSMaxItemLength, i = 0;
    char                **CSStr = GetItemStrings("CJSTEPCOLORMENU",
                        &CSMenuWidth, &CSMaxItemLength, 4);

    if(CSStr == NULL) {
        Console = alconio_init(42, 6, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 4, 40, 0, BLACK, WHITE, WHITE, BLACK,
            4, 0, CJSCItems);
    }
    else {
        Console = alconio_init(CSMaxItemLength + 2, 6, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 4, CSMaxItemLength, 0, BLACK, WHITE,
            WHITE, BLACK, 4, 0, CJSCItems);
        for(i = 0; i <= 3; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &CSStr[i]);
}

    DrawMenu(Console, Menu);

    ExitValue = 0;
    while(TRUE) {
        AltKey = DoMenu(Console, Menu);

        Menu->DefaultItemIndex = Menu->ItemIndex;

        if((AltKey == KEY_S) || ((Menu->ItemIndex == 0) &&
            (AltKey == KEY_ENTER))) 
        {
            ComplexExp->Coloring = CM_START;
            break;
        }
        if((AltKey == KEY_I) || ((Menu->ItemIndex == 1) &&
            (AltKey == KEY_ENTER))) 
        {
            ComplexExp->Coloring = CM_ITERSTEP;
            break;
        }
        if((AltKey == KEY_A) || ((Menu->ItemIndex == 2) &&
            (AltKey == KEY_ENTER))) 
        {  
            ComplexExp->Coloring = CM_SAME;
            break;
        }
        if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 3) &&
            (AltKey == KEY_ENTER))) 
        {  
            ExitValue = -1;
            break;
        }
        else
            continue;
    }
    DestroyMenu(Menu);
    DestroyItemStrings(CSStr, 4);
    a_putimage(Console, Console->ScreenContents);
    alconio_exit(Console);
    return ExitValue;
}

static int CStepShapeSubMenu(TComplexExp *ComplexExp)
{
    TConsole            *Console;
    TMenu               *Menu;
    int                 AltKey, ExitValue, CSMenuWidth, CSMaxItemLength, i = 0;
    char                **CSStr = GetItemStrings("CJSTEPSHAPEMENU", 
                        &CSMenuWidth, &CSMaxItemLength, 5);
    
    if(CSStr == NULL) {
        Console = alconio_init(25, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, 23, 0, BLACK, WHITE, WHITE, BLACK, 
            5, 0, CJSSItems);
    }
    else {
        Console = alconio_init(CSMaxItemLength + 2, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, CSMaxItemLength, 0, BLACK, WHITE, 
            WHITE, BLACK, 5, 0, CJSSItems);
        for(i = 0; i <= 4; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &CSStr[i]);
    }

    DrawMenu(Console, Menu);
    
    ExitValue = 0;
    while(TRUE) {
        AltKey = DoMenu(Console, Menu);

        Menu->DefaultItemIndex = Menu->ItemIndex;

        if((AltKey == KEY_P) || ((Menu->ItemIndex == 0) &&
            (AltKey == KEY_ENTER))) 
        {
            ComplexExp->Diagram = SH_POINT;
            if(CStepColorSubMenu(ComplexExp) == -1) {
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu); 
                continue;
            }
            break;
        }
        if((AltKey == KEY_L) || ((Menu->ItemIndex == 1) &&
            (AltKey == KEY_ENTER))) 
        { 
            ComplexExp->Diagram = SH_LINE;
            if(CStepColorSubMenu(ComplexExp) == -1) { 
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu); 
                continue;
            }
            break;
        }
        if((AltKey == KEY_R) || ((Menu->ItemIndex == 2) &&
            (AltKey == KEY_ENTER))) 
        {  
            ComplexExp->Diagram = SH_RECT;
            if(CStepColorSubMenu(ComplexExp) == -1) { 
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu); 
                continue;
            }
            break;
        }
        if((AltKey == KEY_C) || ((Menu->ItemIndex == 3) &&
            (AltKey == KEY_ENTER))) 
        {  
            ComplexExp->Diagram = SH_CIRCLE;
            if(CStepColorSubMenu(ComplexExp) == -1) { 
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu); 
                continue;
            }
            break;
        }
        if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 4) &&
            (AltKey == KEY_ENTER))) 
        {  
            ExitValue = -1;
            break;
        }
        else
            continue;
    }
    DestroyMenu(Menu);
    DestroyItemStrings(CSStr, 5);
    a_putimage(Console, Console->ScreenContents);
    alconio_exit(Console);
    return ExitValue;
}

static int CStepOrbitsSubMenu(TComplexExp *ComplexExp)
{
    TConsole            *Console;
    TMenu               *Menu; 
    int                 AltKey, ExitValue, CSMenuWidth, CSMaxItemLength, 
                        COMenuWidth, COMaxItemLength, i = 0, OldGraph;
    char                **CSStr = GetItemStrings("CJSTEPMENU", &CSMenuWidth, 
                        &CSMaxItemLength, 3),
                        **COStr = GetItemStrings("CJORBITMENU", &COMenuWidth,
                        &COMaxItemLength, 7);

    OldGraph = ComplexExp->Graph;

    /* Graph selection for 'Step by step iteration' */

    if((ComplexExp->Graph == GR_C_STEP || ComplexExp->Graph == GR_C_STEPINV)) {
        
        if(CSStr == NULL) {
            Console = alconio_init(29, 5, 1, BLACK, GRAY);
            Menu = CreateMenu("", 1, 1, 0, 3, 27, 0, BLACK, WHITE, WHITE, BLACK, 
                3, 0, CJSItems);
        }
        else {
            Console = alconio_init(CSMaxItemLength + 2, 5, 1, BLACK, GRAY);
            Menu = CreateMenu("", 1, 1, 0, 3, CSMaxItemLength, 0, BLACK, WHITE, 
                WHITE, BLACK, 3, 0, CJSItems);
            for(i = 0; i <= 2; i++) 
                SwapItemCaptions(&Menu->Items[i]->Caption, &CSStr[i]);
        }

        /* Disable 'inverse iteration' for functions other than z -> z ^ n + c */
        
        if(ComplexExp->Function > 2)
            RESBIT(Menu->Items[1]->Flag, I_VISIBLE);
            
        DrawMenu(Console, Menu);
        ExitValue = 0;
        while(TRUE) {
            AltKey = DoMenu(Console, Menu);

            Menu->DefaultItemIndex = Menu->ItemIndex;

            if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 2) && 
                (AltKey == KEY_ENTER))) 
            {
                ExitValue = -1;
                break;
            }   

            if(((AltKey == KEY_1) || (AltKey == KEY_1_PAD)) || 
                ((Menu->ItemIndex == 0) && (AltKey == KEY_ENTER))) 
            {
                ComplexExp->Graph = GR_C_STEP;
                if(CStepShapeSubMenu(ComplexExp) == -1) {
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu);
                    continue;
                }
                break;
            }
            if(((AltKey == KEY_2) || (AltKey == KEY_2_PAD)) || 
                ((Menu->ItemIndex == 1) && (AltKey == KEY_ENTER))) 
            {
                ComplexExp->Graph = GR_C_STEPINV;
                if(CStepShapeSubMenu(ComplexExp) == -1) {
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu);
                    continue;
                }
                break;
            }
            else 
                continue;
        }

        if(AltKey == KEY_ENTER) {

            /* Copy the entire GraphPar to destination graph */

            SetComplexGraphPar(&ComplexExp->GraphPar[ComplexExp->Graph],
                ComplexExp->GraphPar[OldGraph].zReal,
                ComplexExp->GraphPar[OldGraph].zImag,
                ComplexExp->GraphPar[OldGraph].cReal,
                ComplexExp->GraphPar[OldGraph].cImag,
                ComplexExp->GraphPar[OldGraph].EscapeValue,
                ComplexExp->GraphPar[OldGraph].IterSteps,
                ComplexExp->GraphPar[OldGraph].IterSkip,
                ComplexExp->GraphPar[OldGraph].IterMax,
                ComplexExp->GraphPar[OldGraph].Mirror);
            SetCoords(ComplexExp->GraphPar[ComplexExp->Graph].Coords,
                ComplexExp->GraphPar[OldGraph].Coords->XMin,
                ComplexExp->GraphPar[OldGraph].Coords->YMin,
                ComplexExp->GraphPar[OldGraph].Coords->XMax,
                ComplexExp->GraphPar[OldGraph].Coords->YMax);
        }

        DestroyMenu(Menu);
        a_putimage(Console, Console->ScreenContents);
        alconio_exit(Console);
    }

    /* Diagram selection for orbit-diagrams */

    else {
        if(COStr == NULL) {
            Console = alconio_init(37, 8, 1, BLACK, GRAY);
            Menu = CreateMenu("", 1, 1, 0, 6, 35, 0, BLACK, WHITE, WHITE, BLACK, 
                7, 0, CJOItems);
        }
        else {
            Console = alconio_init(COMaxItemLength + 2, 8, 1, BLACK, GRAY);
            Menu = CreateMenu("", 1, 1, 0, 6, COMaxItemLength, 0, BLACK, WHITE, 
                WHITE, BLACK, 7, 0, CJOItems);
            for(i = 0; i <= 5; i++) 
                SwapItemCaptions(&Menu->Items[i]->Caption, &COStr[i]);
        }

        /* Set correct captions */

        if(ComplexExp->Graph == GR_C_ORBITS_C) {
            SwapItemCaptions(&Menu->Items[0]->Caption, &Menu->Items[6]->Caption);
            Menu->Items[4]->Flag = 0;
        }    
        else 
            Menu->Items[4]->Flag = 96;
        
        /* Disable 'inverse iteration' for functions other than z -> z ^ n + c */
        
        if(ComplexExp->Function > 2)
            RESBIT(Menu->Items[4]->Flag, I_VISIBLE);
            
        DrawMenu(Console, Menu);

        ExitValue = 0;
        while(TRUE) {
            AltKey = DoMenu(Console, Menu);

            Menu->DefaultItemIndex = Menu->ItemIndex;

            if((AltKey == KEY_F) || ((Menu->ItemIndex == 1) &&
                (AltKey == KEY_ENTER))) 
            {
                ComplexExp->Diagram = DM_FILLED;
                break;
            }
            if((AltKey == KEY_E) || ((Menu->ItemIndex == 2) &&
                (AltKey == KEY_ENTER))) 
            { 
                ComplexExp->Diagram = DM_ETD;
                break;
            }
            if((AltKey == KEY_B) || ((Menu->ItemIndex == 3) &&
                (AltKey == KEY_ENTER))) 
            {  
                ComplexExp->Diagram = DM_BOUNDARY;
                break;
            }
            if(ComplexExp->Graph == GR_C_ORBITS_Z &&
                ((AltKey == KEY_I) || ((Menu->ItemIndex == 4) &&
                (AltKey == KEY_ENTER)))) 
            {  
                ComplexExp->Diagram = DM_INVERSE;
                break;
            }
            if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 5) &&
                (AltKey == KEY_ENTER))) 
            {  
                ExitValue = -1;
                break;
            }
            else
                continue;
        }
        DestroyMenu(Menu);
        a_putimage(Console, Console->ScreenContents);
        alconio_exit(Console);
    }
    DestroyItemStrings(COStr, 7);
    DestroyItemStrings(CSStr, 3);
    return ExitValue;               
}

int DoRCentralMenu(TRealExp *RealExp)
{
    TConsole            *Console;
    TMenu               *Menu; 
    int                 AltKey, ExitValue = 0, OldGraph, NewGraph, MSGMenuWidth, 
                        MSGMaxItemLength, ZHMenuWidth, ZHMaxItemLength, 
                        RMenuWidth, RMaxItemLength, RJMenuWidth, 
                        RJMaxItemLength, i = 0;
    char                **RStr = GetItemStrings("RJMENU", &RMenuWidth, 
                        &RMaxItemLength, 8),
                        **RJStr = GetItemStrings("RJHELP", &RJMenuWidth,
                        &RJMaxItemLength, 3),
                        **ZHStr = GetItemStrings("ZOOMHELP", &ZHMenuWidth,
                        &ZHMaxItemLength, 5),
                        **MSGStr = GetItemStrings("MESSAGES", &MSGMenuWidth,
                        &MSGMaxItemLength, 7);

    /* Allocate space for menu */
    
    if(RStr == NULL) {
        Console = alconio_init(32, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, 30, 0, BLACK, WHITE, WHITE, BLACK, 
            8, 0, RJItems);
    }
    else {
        Console = alconio_init(RMaxItemLength + 2, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, RMaxItemLength, 0, BLACK, WHITE, 
            WHITE, BLACK, 8, 0, RJItems);
        for(i = 0; i <= 7; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &RStr[i]);
    }

    /* Set correct menu-texts */
        
    if(RealExp->Graph == GR_R_ORBITS_X) {
        SwapItemCaptions(&Menu->Items[1]->Caption, &Menu->Items[5]->Caption);
        SwapItemCaptions(&Menu->Items[2]->Caption, &Menu->Items[7]->Caption);
    }
    if(RealExp->Graph == GR_R_ORBITS_C) {
        SwapItemCaptions(&Menu->Items[1]->Caption, &Menu->Items[6]->Caption);
        Menu->Items[2]->Flag = 0;
    }

    DrawMenu(Console, Menu);

    OldGraph = RealExp->Graph;
    
    while(TRUE) {
        AltKey = DoMenu(Console, Menu);
        
        Menu->DefaultItemIndex = Menu->ItemIndex;

        /* User canceled menu */
        
        if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 4) &&
            (AltKey == KEY_ENTER))) 
        {
            ExitValue = -1;
            break;
        }

        /* Zoom */

        if((AltKey == KEY_Z) || ((Menu->ItemIndex == 0) &&
            (AltKey == KEY_ENTER))) 
        {

            /* Restore original image */

            a_putimage(Console, Console->ScreenContents);

            /* Zoom function */

            if(ZoomBox(screen, font, RealExp->GraphPar[RealExp->Graph].Coords,
                (ZHStr != NULL) ? ZHStr : ZoomHelpStr, 0, 
                RealExp->GraphPar[RealExp->Graph].Coords->YEnd) != 0) 
            {
                RealExp->Changed = TRUE;
                ExitValue = 0;
                break;
            }
            else {

                /* User canceled zoom-function */
                
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu);
                continue;
            }
        }
        
        /* Change type (diagram for 'step-by-step iteration' */
        
        if((AltKey == KEY_T) || ((Menu->ItemIndex == 1) &&
            (AltKey == KEY_ENTER))) 
        {
            if(RealExp->Graph == GR_R_STEP) {
                if(RStepSubMenu(RealExp) == 0) 
                    break; 
                else 
                    continue;
                RealExp->Changed = TRUE;
            }
        }
        
        /* 
        **  Jump from 'step-by-step iteration' to 'orbitdiagram for var.x' or
        **  from 'orbitdiagram var. x' to 'orbitdiagram var. c' 
        */
        
        if((AltKey == KEY_SPACE) || ((Menu->ItemIndex == 2) &&
            (AltKey == KEY_ENTER))) 
        { 

            /*
            **  Step by step iteration    --> Orbit diagram for variable x
            */

            if(RealExp->Graph == GR_R_STEP) {

                /* Restore part of screen */
            
                a_putimage(Console, Console->ScreenContents);

                NewGraph = RealJump(RealExp, GR_R_ORBITS_X, NULL, 
                    (MSGStr != NULL) ? MSGStr[5] : F1KeyStr, 
                    (RJStr != NULL) ? RJStr : RealJumpStr);

                if(NewGraph > -1) {
                    RealExp->Graph = NewGraph;
                    ExitValue = 0;
                    break;
                }
            }
           	
            /*
            **  Orbit diagram for variable x  --> Orbit diagram for variabele c
            */
            
            else if(RealExp->Graph == GR_R_ORBITS_X) {
                
                /* Restore part of screen */
            
                a_putimage(Console, Console->ScreenContents);

                NewGraph = RealJump(RealExp, GR_R_ORBITS_C, NULL, 
                    (MSGStr != NULL) ? MSGStr[5] : F1KeyStr, 
                    (RJStr != NULL) ? RJStr : RealJumpStr);
        
                if(NewGraph > -1) {
                    RealExp->Graph = NewGraph;
                    ExitValue = 0;
                    break;
                }
                else {
                    RealExp->Graph = OldGraph;
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu); 
                    continue;
                }
            }

            else {
                clear_keybuf();
                continue;
            }
        }
        
        /* Save image/parameters */
            
        if((AltKey == KEY_S) || ((Menu->ItemIndex == 3) &&
            (AltKey == KEY_ENTER)))
        { 
            a_putimage(Console, Console->ScreenContents);
            CntMnuSave(FT_REAL, RealExp, NULL);
            Console->BkCol = GRAY;
            DrawMenu(Console, Menu);
        }
        
        /* 
        **  Jump from 'orbitdiagram var. c' to 'orbitdiagram var. x' or
        **  from 'orbitdiagram var. x' to 'step-by-step iteration'
        */
        
        if((AltKey == KEY_BACKSPACE) || ((Menu->ItemIndex == 1) && 
            (AltKey == KEY_ENTER))) 
        {
            
            /*
            **  Banendiagram variabele x  --> Step by step iteration
            */

            if(RealExp->Graph == GR_R_ORBITS_X) {

                /* Restore part of screen */
            
                a_putimage(Console, Console->ScreenContents);
                OldGraph = RealExp->Graph;
                RealExp->Graph = GR_R_STEP;
                if(RStepSubMenu(RealExp) == -1) {
                    RealExp->Graph = OldGraph;
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu); 
                    continue;
                }
                else {
                    RealExp->Graph = OldGraph;
                    NewGraph = RealJump(RealExp, GR_R_STEP, NULL,  
                        (MSGStr != NULL) ? MSGStr[5] : F1KeyStr, 
                        (RJStr != NULL) ? RJStr : RealJumpStr);
            
                    if(NewGraph > -1) {
                        RealExp->Graph = NewGraph;
                        break; 
                    }
                    else {
                        RealExp->Graph = OldGraph;
                        Console->BkCol = GRAY;
                        DrawMenu(Console, Menu); 
                        continue;
                    }
                }
            }

            /*
            **  Orbit diagram variable c  --> Banendiagram variabele x
            */
            
            else if(RealExp->Graph == GR_R_ORBITS_C) {

                /* Restore part of screen */
            
                a_putimage(Console, Console->ScreenContents);
                OldGraph = RealExp->Graph;
                NewGraph = RealJump(RealExp, GR_R_ORBITS_X, NULL,  
                    (MSGStr != NULL) ? MSGStr[5] : F1KeyStr, 
                    (RJStr != NULL) ? RJStr : RealJumpStr);
        
                if(NewGraph > -1) {
                    RealExp->Graph = NewGraph;
                    ExitValue = 0;
                    break;
                }
                else {
                    RealExp->Graph = OldGraph;
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu); 
                    continue;
                }
            }

            else {
                clear_keybuf();
                continue;
            }
        }
    }
    clear_keybuf();
    DestroyMenu(Menu);
    DestroyItemStrings(MSGStr, 7);
    DestroyItemStrings(ZHStr, 5);
    DestroyItemStrings(RJStr, 3);
    DestroyItemStrings(RStr, 8);
    a_putimage(Console, Console->ScreenContents);
    alconio_exit(Console);
    return ExitValue;
}

int DoCCentralMenu(BITMAP *JuliaBitmap, int JuliaVisible, int JuliaDiagram,  
    TComplexExp *ComplexExp)
{
    TConsole            *Console;
    TMenu               *Menu; 
    int                 AltKey, ExitValue, OldGraph, NewGraph, ZHMenuWidth,
                        ZHMaxItemLength, CMenuWidth, CMaxItemLength, 
                        CJMenuWidth, CJMaxItemLength, MSGMenuWidth, 
                        MSGMaxItemLength, i = 0, OldDiagram;
    char                **CStr = GetItemStrings("CJMENU", &CMenuWidth, 
                        &CMaxItemLength, 8),
                        **CJStr = GetItemStrings("CJHELP", &CJMenuWidth, 
                        &CJMaxItemLength, 6),
                        **ZHStr = GetItemStrings("ZOOMHELP", &ZHMenuWidth,
                        &ZHMaxItemLength, 5),
                        **MSGStr = GetItemStrings("MESSAGES", &MSGMenuWidth,
                        &MSGMaxItemLength, 7);

    /* Allocate space for menu */
    
    if(CStr == NULL) {
        Console = alconio_init(36, 8, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 6, 34, 0, BLACK, WHITE, WHITE, BLACK, 
            8, 0, CJItems);
    }
    else {
        Console = alconio_init(CMaxItemLength + 2, 8, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 6, CMaxItemLength, 0, BLACK, WHITE, 
            WHITE, BLACK, 8, 0, CJItems);
        for(i = 0; i <= 7; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &CStr[i]);
    }
    
    /*
    **  Set correct menu texts.
    */

    if(ComplexExp->Graph == GR_C_STEP || ComplexExp->Graph == GR_C_STEPINV) {
        SwapItemCaptions(&Menu->Items[3]->Caption, &Menu->Items[6]->Caption);
        Menu->Items[2]->Flag = 0;
    }
    
    if(ComplexExp->Graph == GR_C_ORBITS_C) {
        SwapItemCaptions(&Menu->Items[2]->Caption, &Menu->Items[7]->Caption);
        Menu->Items[3]->Flag = 0;
    }

    DrawMenu(Console, Menu);

    /*OldGraph = RealExp->Graph;*/
    
    while(TRUE) {
    
        AltKey = DoMenu(Console, Menu);
        
        Menu->DefaultItemIndex = Menu->ItemIndex;

        /* User canceled menu */
        
        if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 5) && 
            (AltKey == KEY_ENTER))) 
        {
            ExitValue = -1;
            break;
        }
        
        /* Zoom */
        
        if((AltKey == KEY_Z) || ((Menu->ItemIndex == 0) && 
            (AltKey == KEY_ENTER))) 
        {

            /* Restore original image */
            
            a_putimage(Console, Console->ScreenContents);
            
            /* Zoom function */
            
            if(ZoomBox(screen, font, 
                ComplexExp->GraphPar[ComplexExp->Graph].Coords,
                (ZHStr != NULL) ? ZHStr : ZoomHelpStr, 0,
                ComplexExp->GraphPar[ComplexExp->Graph].Coords->YEnd) != 0) 
            {
                ExitValue = 0;
                if(JuliaVisible && ((ComplexExp->Graph == GR_C_STEP) ||
                    (ComplexExp->Graph == GR_C_STEPINV)))
                {
                    memcpy(ComplexExp->GraphPar[GR_C_ORBITS_Z].Coords,
                        ComplexExp->GraphPar[ComplexExp->Graph].Coords,
                        sizeof(TCoords));
                    clear_bitmap(JuliaBitmap);
                    OldGraph = ComplexExp->Graph;
                    OldDiagram = ComplexExp->Diagram;
                    ComplexExp->Graph = GR_C_ORBITS_Z;
                    ComplexExp->Diagram = JuliaDiagram;
                    DrawComplexOrbits(JuliaBitmap, ComplexExp);
                    ComplexExp->Diagram = OldDiagram;
                    ComplexExp->Graph = OldGraph;
                }
                break;
            }
            else {
            
                /* User canceled zoom-function */
                
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu);
                continue;
            }
        }  
        
        /* Change type (diagram for 'step-by-step iteration' */
        
        if((AltKey == KEY_T) || ((Menu->ItemIndex == 1) &&
            (AltKey == KEY_ENTER))) 
        {
            if(CStepOrbitsSubMenu(ComplexExp) == 0) 
                break; 
            else 
                continue;
        }

        if((AltKey == KEY_BACKSPACE) || ((Menu->ItemIndex == 2) && 
            (AltKey == KEY_ENTER))) 
        {
            
            /* Restore part of screen */
            
            a_putimage(Console, Console->ScreenContents);
            
            /*
            **  Orbit diagram for variable c  --> Banendiagram for variabele z
            */
        
            if(ComplexExp->Graph == GR_C_ORBITS_C) {
                
                OldGraph = ComplexExp->Graph;
                ComplexExp->Graph = GR_C_ORBITS_Z;
                if(CStepOrbitsSubMenu(ComplexExp) == -1) {                
                    ComplexExp->Graph = OldGraph;
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu); 
                    continue;
                }
                else {
                    ComplexExp->Graph = OldGraph;
                    NewGraph = ComplexJump(ComplexExp, GR_C_ORBITS_Z, 
                        (MSGStr != NULL) ? MSGStr[5] : F1KeyStr,
                        (CJStr != NULL) ? CJStr : CplJumpHelpStr, TRUE);
                
                    if(NewGraph != -1) { 
                        ComplexExp->Graph = NewGraph; 
                        break; 
                    } 
                    else {
                        ComplexExp->Graph = OldGraph;
                        Console->BkCol = GRAY;
                        DrawMenu(Console, Menu); 
                        continue;
                    }
                }
            }
    
            /*
            **  Orbit diagram for variable z  --> Step by step iteration (inv)
            */
        
            else if(ComplexExp->Graph == GR_C_ORBITS_Z) {
                OldGraph = ComplexExp->Graph; 
                ComplexExp->Graph = GR_C_STEP; 
                if(CStepOrbitsSubMenu(ComplexExp) == 0) {
                    NewGraph = ComplexExp->Graph;
                    ComplexExp->Graph = OldGraph; 
                    ComplexJump(ComplexExp, NewGraph,  
                        (MSGStr != NULL) ? MSGStr[5] : F1KeyStr,
                        (CJStr != NULL) ? CJStr : CplJumpHelpStr, FALSE);
                    break; 
                }
                else { 
                    ComplexExp->Graph = OldGraph; 
                    Console->BkCol = GRAY;
                    DrawMenu(Console, Menu); 
                    continue;
                }
            }
        }

        if((AltKey == KEY_SPACE) || ((Menu->ItemIndex == 3) &&
            (AltKey == KEY_ENTER))) 
        { 

            /* Restore part of screen */
            
            a_putimage(Console, Console->ScreenContents);

            /*
            **  Step by step iteration (inv)  --> Orbit diagram for variable z
            */
        
            if((ComplexExp->Graph == GR_C_STEP) || 
                (ComplexExp->Graph == GR_C_STEPINV)) 
            {
                OldGraph = ComplexExp->Graph;
                NewGraph = ComplexJump(ComplexExp, GR_C_ORBITS_Z, 
                    (MSGStr != NULL) ? MSGStr[5] : F1KeyStr,
                    (CJStr != NULL) ? CJStr : CplJumpHelpStr, FALSE);

                if(NewGraph != -1) { 
                    ComplexExp->Graph = NewGraph; 
                    
                    if(CStepOrbitsSubMenu(ComplexExp) == 0) 
                        break; 
                    else {
                        ComplexExp->Graph = OldGraph; 
                        Console->BkCol = GRAY;
                        DrawMenu(Console, Menu); 
                        continue;
                    }
                }
            }
            
            /*
            **  Orbit diagram for variable z  --> Banendiagram for variabele c
            */
        
            else if(ComplexExp->Graph == GR_C_ORBITS_Z) {
            
                OldGraph = ComplexExp->Graph;
                NewGraph = ComplexJump(ComplexExp, GR_C_ORBITS_C, 
                    (MSGStr != NULL) ? MSGStr[5] : F1KeyStr,
                    (CJStr != NULL) ? CJStr : CplJumpHelpStr, FALSE);
           
                if(NewGraph != -1) { 
                    ComplexExp->Graph = NewGraph; 
                    
                    if(CStepOrbitsSubMenu(ComplexExp) == 0) 
                        break; 
                    else {
                        ComplexExp->Graph = OldGraph; 
                        Console->BkCol = GRAY;
                        DrawMenu(Console, Menu); 
                        continue;
                    }
                }
            }
        }
            
        /* Save image/parameters */
            
        if((AltKey == KEY_S) || ((Menu->ItemIndex == 4) &&
            (AltKey == KEY_ENTER))) 
        {
            a_putimage(Console, Console->ScreenContents);
            CntMnuSave(FT_COMPLEX, NULL, ComplexExp);
            Console->BkCol = GRAY;
            DrawMenu(Console, Menu);
        }
    }
    clear_keybuf();
    DestroyMenu(Menu);
    DestroyItemStrings(MSGStr, 7);
    DestroyItemStrings(ZHStr, 5);
    DestroyItemStrings(CJStr, 6);
    DestroyItemStrings(CStr, 8);
    a_putimage(Console, Console->ScreenContents);
    alconio_exit(Console);
    return ExitValue;
}

int CntMnuSave(int Functiontype, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    FILE                *fp;
    BITMAP              *TempBmp;
    PALETTE             Pal;
    TConsole            *Console;
    TMenu               *Menu; 
    int                 AltKey, CSMenuWidth, MTMenuWidth, CSMaxItemLength, 
                        MTMaxItemLength, i = 0;
    char                **CSStr = GetItemStrings("CSAVEMENU", &CSMenuWidth, 
                        &CSMaxItemLength, 5),
                        **MTStr = GetItemStrings("MISCTEXTS",
                        &MTMenuWidth, &MTMaxItemLength, 5),
                        FilePath[2048] = "default.ps";
    
    if(CSStr == NULL) {
        Console = alconio_init(33, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, 31, 0, BLACK, WHITE, WHITE, BLACK, 
            5, 0, CSItems);
    }
    else {
        Console = alconio_init(CSMaxItemLength + 2, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, CSMaxItemLength, 0, BLACK, WHITE, 
            WHITE, BLACK, 5, 0, CSItems);
        for(i = 0; i <= 4; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &CSStr[i]);
    }
#ifndef BUILD_WITH_JPGALLEG
    RESBIT(Menu->Items[3]->Flag, I_VISIBLE);
#endif
    DrawMenu(Console, Menu);
    
    while(TRUE) {
    
        AltKey = DoMenu(Console, Menu);
    
        /*
        **  Write Postscript Portrait.
        */
        
        if((AltKey == KEY_P) || ((Menu->ItemIndex == 0) &&
            (AltKey == KEY_ENTER))) 
        { 
            if(FILE_SELECT_BOX((MTStr != NULL) ? MTStr[4] : "Save: ", 
                FilePath, "PS", 2047, 300, 400) != 0) 
            {
                scare_mouse();
                a_putimage(Console, Console->ScreenContents);
                if((fp = fopen(FilePath, "w")) != NULL) {
                    WriteEPS(screen, SCREEN_W, SCREEN_H, fp, 0);
                    fclose(fp);
                }
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu);
                unscare_mouse();
                break;
            }
        }

        /*
        **  Write Postscript Landscape.
        */

        if((AltKey == KEY_L) || ((Menu->ItemIndex == 1) &&
            (AltKey == KEY_ENTER))) 
        { 
            if(FILE_SELECT_BOX((MTStr != NULL) ? MTStr[4] : "Save: ", 
                FilePath, "PS", 2047, 300, 400) != 0) 
            {
                scare_mouse();
                a_putimage(Console, Console->ScreenContents);
                if((fp = fopen(FilePath, "w")) != NULL) {
                    WriteEPS(screen, SCREEN_W, SCREEN_H, fp, 1);
                    fclose(fp);
                }
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu);
                unscare_mouse();
                break;
            }
        }

        /*
        **  Write PCX (PC Paintbrush format).
        */

        if((AltKey == KEY_X) || ((Menu->ItemIndex == 2) &&
            (AltKey == KEY_ENTER))) 
        { 
            TempBmp = create_bitmap(SCREEN_W, SCREEN_H);
            strcpy(FilePath, "default.pcx");
            if(FILE_SELECT_BOX((MTStr != NULL) ? MTStr[4] : "Save: ", 
                FilePath, "PCX", 2047, 300, 400) != 0) 
            {
                scare_mouse();
                a_putimage(Console, Console->ScreenContents);
                if(TempBmp != NULL)
                    blit(screen, TempBmp, 0, 0, 0, 0, SCREEN_W - 1, SCREEN_H - 1);
                save_bitmap(FilePath, (TempBmp != NULL) ? TempBmp : screen, 
                    NULL); 
                    
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu);
                unscare_mouse();
                break;
            }
            if(TempBmp != NULL) destroy_bitmap(TempBmp);
        }

        if((AltKey == KEY_J) || ((Menu->ItemIndex == 3) &&
            (AltKey == KEY_ENTER))) 
        {
#ifdef BUILD_WITH_JPGALLEG
            TempBmp = create_bitmap(SCREEN_W, SCREEN_H);
            strcpy(FilePath, "default.jpg");
            if(FILE_SELECT_BOX((MTStr != NULL) ? MTStr[4] : "Save: ", 
                FilePath, "JPG", 2047, 300, 400) != 0) 
            {
                scare_mouse();
                a_putimage(Console, Console->ScreenContents);
                if(TempBmp != NULL)
                    blit(screen, TempBmp, 0, 0, 0, 0, SCREEN_W - 1, SCREEN_H - 1);
                get_palette(Pal);
                save_jpg(FilePath, (TempBmp != NULL) ? TempBmp : screen, Pal); 
                
                Console->BkCol = GRAY;
                DrawMenu(Console, Menu);
                unscare_mouse();
                break;
            }
            if(TempBmp != NULL) destroy_bitmap(TempBmp);
#endif
        }

        if((AltKey == KEY_ESC) || ((Menu->ItemIndex == 4) &&
            (AltKey == KEY_ENTER))) 
        {
            DestroyMenu(Menu);
            DestroyItemStrings(MTStr, 5);
            DestroyItemStrings(CSStr, 5);
            a_putimage(Console, Console->ScreenContents);
            alconio_exit(Console);
            return 1; 
        }
    }
    DestroyMenu(Menu);
    DestroyItemStrings(MTStr, 5);
    DestroyItemStrings(CSStr, 5);
    a_putimage(Console, Console->ScreenContents);
    alconio_exit(Console);
    return 0;
}
