/*
    This file is part of Funiter,
    Real and complex function iteration software.

    parser.h - Funiter's expression evaluator.

    Copyright (C) 1995-2007 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
    See README for contact information.

    references/sources:
     
        http://compilers.iecc.com/crenshaw/
        http://javajeff.mb.ca/aa/5/5.html
        http://www.java2s.com/Code/CSharp/Development-Class/Thismodulecontainstherecursivedescentparserthatrecognizesvariables.htm
                    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#ifndef _PARSER
    #define _PARSER

    /* Includes */

    #include <math.h>
    #include <string.h>
    #include <ctype.h>
    #include <stdarg.h>
    #include "complex.h"
    
    /* 
    **  Error definitions:
    **
    **      0   No error.
    **      1   Syntax error.
    **      2   Out of range.
    **      3   Division by zero.
    **      4   Undefined symbol
    **      5   No number
    **      6   Maximum number of variables exceeded.
    **      7   No right parenthesis.
    */

    #define SYNTAXERR           1
    #define OUTRANGE            2
    #define DIVBYZERO           3
    #define UNDEFDFN            4
    #define NONUM               5
    #define MAXVAREXC           6
    #define NORPARENTH          7
    
    /*
    **  Define TRUE and FALSE.
    */

    #ifndef TRUE
        #define TRUE            (1==1)
    #endif

    #ifndef FALSE
        #define FALSE           (1==2)
    #endif

    #define MAX_VAR             5

    /*
    **  Some function-macro's to speed up things a bit and keep
    **  the code more readable :-).
    */

    #define TOKEN           (ParseData->ParseStr[ParseData->TokenCounter])
    
    #define IS_TOKEN(C)     (ParseData->ParseStr[ParseData->TokenCounter] \
                            == (C))

    #define NEXT_TOKEN      (ParseData->TokenCounter++)
    
    #define ISADDOP(S)      (((S) == '+') || ((S) == '-'))

    #define ISMULOP(S)      (((S) == '*') || ((S) == '/'))

    #define ISPOWOP(S)      (((S) == '^'))

    #define ISDIGIT(S)      (((S) >= '0') && ((S) <= '9'))

    #define ISALPHA(S)      ((((S) >= 'a') && ((S) <= 'z')) \
                            || (((S) >= 'A') && ((S) <= 'Z')))

    #define POW_OP          '^'
    #define MUL_OP          '*'
    #define DIV_OP          '/'
    #define SUB_OP          '-'
    #define ADD_OP          '+'
    
    /* Struct with parser info */
    
    typedef struct {
        char            *ParseStr, VarNames[MAX_VAR + 1];
        unsigned int    ErrorNum, ErrorPos, TokenCounter, TypeMode;
        long double     VarValues[MAX_VAR + 1];
        TComplex        CVarValues[MAX_VAR + 1];
    } TParseData;
        
    /* Datatype independant functions */

    char *RemoveSpaces(char *StrOut, char *StrIn);
    TComplex CGetNumber(TParseData *ParseData);
    long double GetNumber(TParseData *ParseData);

    /* Functions for the floating point parser */

    TComplex CFactor(TParseData *ParseData);
    long double Factor(TParseData *ParseData);
    TComplex CFunVar(TParseData *ParseData);
    long double FunVar(TParseData *ParseData);
    long double Power(TParseData *ParseData);
    TComplex CMulDiv(TParseData *ParseData);
    long double MulDiv(TParseData *ParseData);
    TComplex CAddSub(TParseData *ParseData);
    long double AddSub(TParseData *ParseData);
    TComplex CParse(int *ErrorNum, int *ErrorPos, char *Str, int vars, ...);
    long double Parse(int *ErrorNum, int *ErrorPos, char *Str, int vars, ...);
    
#endif
