/*  
    This file is part of Funiter,
    Real and complex function iteration software.

    cplexp.h - a complex experiment (functiontype: from C to C)

    Copyright (C) 1995-2009 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#ifndef __CPLEXP
    #define __CPLEXP

    #include <stdio.h>
    #include <stdlib.h>
    #include <string.h>
    #include <limits.h>
    #include <math.h>
    #include <time.h>
    #include <allegro.h>
    #include "misc.h"
    #include "coords.h"
    #include "complex.h"
    #include "parser.h"
    #include "settings.h"
    #include "help.h"

    #define FT_COMPLEX                      1

    #define MAX_ROOT                        5
    
    /* MirrorState */

    #define MS_NONE                         0
    #define MS_X                            2
    #define MS_Y                            4
    #define MS_X_COMPLETE                   8
    #define MS_Y_COMPLETE                  16

    /* Graph */

    #define GR_C_STEP                       0
    #define GR_C_STEPINV                    1
    #define GR_C_ORBITS_Z                   2
    #define GR_C_ORBITS_C                   3

    /* Shapes */

    #define SH_POINT                        0
    #define SH_LINE                         1
    #define SH_RECT                         2
    #define SH_CIRCLE                       3

    /* Diagram */

    #define DM_FILLED                       0
    #define DM_ETD                          1
    #define DM_BOUNDARY                     2
    #define DM_INVERSE                      3
    
    /* Coloring methods */
    
    #define CM_SAME                         2
    #define CM_ITERSTEP                     1
    #define CM_START                        0

    /* Mirroring */
    
    #define MIRROR_NONE                     0
    #define MIRROR_X                        1
    #define MIRROR_Y                        2
    #define MIRROR_IMAG0                    3
    
    /*
    **  Structure for complex graphpar:
    **
    **  Coords                  -   Math and screen coordinates.
    **  zReal, zImag            -   Complex parameter z (orbits for var. c).
    **  cReal, cImag            -   Complex parameter c (orbits for var. z).
    **  EscapeValue             -   Value the escape function uses.
    **  IterMax                 -   Maximum number of iterations.
    */

    typedef struct {
        TCoords             *Coords;
        double              zReal, zImag, cReal, cImag, EscapeValue;
        unsigned int        IterSteps, IterSkip, IterMax, EscapeValueChanged,
                            Mirror;
    } TCGraphPar;

    /*
    **  GraphPar                -   Parameters for each graph.
    **  Settings                -   General settings.
    **  Graph                   -   Type of orbits (see defines).
    **  Diagram                 -   Method of drawing (see defines).
    **  Coloring                -   Coloringmethod used for step-by-step.
    */

    typedef struct {
        TCGraphPar          GraphPar[4];
        TSettings           *Settings;
        char                FunctionStr[40];
        int                 Function, Graph, Diagram, Coloring, EscValueChanged,
                            (*PrgProc)(int Position, int MaxPosition),
                            ParseErrNum, ParseErrPos;
        unsigned int        InvJuliaIterMax;
    } TComplexExp;

    /* General */
    
    void SetComplexGraphPar(TCGraphPar *GraphPar, double zReal, double zImag, 
        double cReal, double cImag, double EscapeValue, unsigned int IterSteps, 
        unsigned int IterSkip, unsigned int IterMax, unsigned int Mirror);
    
    int LoadComplexGraphPar(FILE *fp, TCGraphPar *GraphPar);
    int SaveComplexGraphPar(FILE *fp, TCGraphPar *GraphPar);

    TComplexExp *CreateComplexExp(unsigned int XStart, unsigned int YStart,
        unsigned int XEnd, unsigned int YEnd);

    void FreeComplexExp(TComplexExp *CplExp);
    
    int LoadComplexExp(TComplexExp *ComplexExp, char *FileName, int Mode);
    int ReadComplexConfig(TComplexExp *ComplexExp);
    int SaveComplexExp(TComplexExp *ComplexExp, char *FileName);
    int WriteComplexConfig(TComplexExp *ComplexExp);

    void SetComplexExpDefaultCoords(TCoords *Coords, int Function, int Graph);
    void SelectComplexExperiment(TComplexExp *CplExp, int Mode, int Function);

    /* Step by step */

    int IsInList(int *XArray, int *YArray, int x, int y, int *Size);

    void DrawCircle(BITMAP *Bitmap, TCoords *Coords, int x, int y, int r,
        int Color);

    int StoreLine(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, 
        int y2, int Color, int PosInLine, int NumPoints, TComplex **Points);
    int StoreRectangle(BITMAP *Bitmap, TCoords *Coords, int x1, int y1, int x2, 
        int y2, int Color, int NumPoints, TComplex **Points);
    int StoreCircle(BITMAP *Bitmap, TCoords *Coords, int x, int y, int r, 
        int Color, int NumPoints, TComplex **Points);

    TComplex *CreateShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int Color, 
        int *NumPoints, int *BufferSpace, 
        char **DefPointHelpText, char **DefShapeHelpText);
    int DefineShape(BITMAP *Bitmap, TComplexExp *ComplexExp, int *x1, int *y1, 
        int *x2, int *y2, int Color, char **DefPointHelpText, 
        char **DefShapeHelpText);

    void IPDDrawLine(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex *Point,
        TComplex *PointLast, int Color, unsigned int Iter, int ClearMode);

    TComplex *IPDPlot(BITMAP *Bitmap, TComplexExp *ComplexExp, TComplex *Points,
        int NumRoots, unsigned int Iter, unsigned int Color, 
        unsigned int IterColor, int *Size, int *BufferSpace, int KeyPressed);

    int DoComplexStep(BITMAP *Bitmap, BITMAP *JuliaBitmap, 
        TComplexExp *ComplexExp, int JuliaVisible, int NumRoots, 
        char **StepHelpText, char **StepInvHelpText, char **JuliaNoMemStr,  
        char **DefPointHelpText, char **DefShapeHelpText);
        
    /* Orbits */

    int DrawComplexOrbits(BITMAP *Bitmap, TComplexExp *ComplexExp);

    int DrawComplexOrbitsArea(BITMAP *Bitmap, unsigned int XStart, 
        unsigned int YStart, unsigned int XEnd, unsigned int YEnd, 
        unsigned int XScr, unsigned int YScr, unsigned int Width, 
        unsigned int Height, unsigned int MirrorState, TComplexExp *ComplexExp);

    double CalculateComplexFnEscapeValue(TCGraphPar *GraphPar, int Function, 
        int Graph);

    /* Iteration */

    unsigned int CIterateFn(int Function, char *FunctionStr, double zReal, 
        double zImag, double cReal, double cImag,  double EscapeValue, 
        double PixelWidth, double PixelHeight, unsigned int MaxPeriod, 
        unsigned int MaxIter, int PeriodCheck, double *RealRes, double *ImagRes, 
        int *EscStatus, int *ErrorNum, int *ErrorPos);

    void IterateInvFn(int Function, double *zReal, double *zImag, double cReal,
        double cImag);
    
    /* 
    **  Routine to switch between several graphs with the possibilty to select
    **  values with the mouse for certain parameters.
    */
    
    int ComplexJump(TComplexExp *ComplexExp, int DestGraph, char *InfoStr, 
        char *HelpText[], int Mode);

    /* Draw complex experiment. */

    int ComplexDraw(BITMAP *Bitmap, BITMAP *JuliaBitmap, TComplexExp *ComplexExp, 
        int NumRoots, 
        char **StepHelpText, char **StepInvHelpText, char **JuliaNoMemStr, 
        char **DefPointHelpText, char **DefShapeHelpText);
    
#endif
