/*
    This file is part of Funiter,
    Real and complex function iteration software.

    realexp.c - a real experiment (functiontype: from R to R)

    Copyright (C) 1995-2009 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "realexp.h"

/*!\brief
**  Set values for GraphPar.
**
**  This routine returns nothing.
**
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
**\param    c Value for c.
**\param    xbegin1 First startingvalue for iteration.
**\param    xbegin2 Second startingvalue for iteration.
**\param    EscapeValue The escapevalue, it is a number which is used
**          in CalculateRealFnEscapeValue to determine when a value tends to go
**          towards infinity.
**\param    Iterated Represents how many times the function should be
**          iterated for a given x when drawing the function for
**          'step-by-step iteration' and the diagram 'combination
**          of web- and stepdiagram'.
**\param    IterMax Maximum number of iterations.
**\param    IterSkip Number of skipped iterations. These are calculated,
**          not displayed.
**\param    IterSteps The number of steps displayed at one, this value 
**          should ofcourse be less than the value for IterMax.
*/

void SetRealGraphPar(TRGraphPar *GraphPar,double c, double xbegin1, 
    double xbegin2, double EscapeValue, unsigned int Iterated, 
    unsigned long int IterMax, unsigned long int IterSkip, 
    unsigned long int IterSteps)
{
    if(GraphPar != NULL) {
        GraphPar->c = c;
        GraphPar->xbegin1 = xbegin1;
        GraphPar->xbegin2 = xbegin2;
        GraphPar->EscapeValue = EscapeValue;
        GraphPar->Iterated = Iterated;
        GraphPar->IterMax = IterMax;
        GraphPar->IterSkip = IterSkip;
        GraphPar->IterSteps = IterSteps;
    }
}

/*!\brief
**  Load a RealGraphPar.
**
**  This routine returns 0 on success otherwise a value < 0.
**
**\param    fp File-pointer.
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
*/

int LoadRealGraphPar(FILE *fp, TRGraphPar *GraphPar)
{
    int         Res = 0;
    
    Res = fscanf(fp, "%lg,%lg,%lg\n", &GraphPar->c, &GraphPar->xbegin1, 
        &GraphPar->xbegin2);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%lg,%lg,%lg,%lg\n", 
        &GraphPar->Coords->XMin, &GraphPar->Coords->XMax, 
        &GraphPar->Coords->YMin, &GraphPar->Coords->YMax);
    if(Res <= 0) return(Res);

    /* Set coordinates and make sure that width and height are correct */

    SetCoords(GraphPar->Coords, 
        GraphPar->Coords->XMin, GraphPar->Coords->YMin, 
        GraphPar->Coords->XMax, GraphPar->Coords->YMax);

    Res = fscanf(fp, "%lg\n", &GraphPar->EscapeValue);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%lu,%lu,%lu\n", &GraphPar->IterMax, &GraphPar->IterSteps, 
        &GraphPar->IterSkip);
    if(Res <= 0) return(Res);
        
    Res = fscanf(fp, "%u,%u\n", &GraphPar->Iterated, 
        &GraphPar->EscapeValueChanged);
    if(Res <= 0) return(Res);
    
    return Res;
}

/*!\brief
**  Save a RealGraphPar.
**
**  This routine returns always 0. 
**
**\param    fp File-pointer.
**\param    GraphPar Points to an instance of the TRGraphPar    
**          struct, see CreateGraphPar for more info.
*/

int SaveRealGraphPar(FILE *fp, TRGraphPar *GraphPar)
{
    int         Res = 0;
    
    Res = fprintf(fp, "%.*g,%.*g,%.*g\n", N_DBL_DIG, GraphPar->c, 
        N_DBL_DIG, GraphPar->xbegin1, N_DBL_DIG, GraphPar->xbegin2);
    if(Res <= 0) return(Res);
        
    Res = fprintf(fp, "%.*g,%.*g,%.*g,%.*g\n", 
        N_DBL_DIG, GraphPar->Coords->XMin, N_DBL_DIG, GraphPar->Coords->XMax, 
        N_DBL_DIG, GraphPar->Coords->YMin, N_DBL_DIG, GraphPar->Coords->YMax);
    if(Res <= 0) return(Res);
        
    Res = fprintf(fp, "%.*g\n", N_DBL_DIG, GraphPar->EscapeValue);
    if(Res <= 0) return(Res);
        
    Res = fprintf(fp, "%lu,%lu,%lu\n", GraphPar->IterMax, GraphPar->IterSteps, 
        GraphPar->IterSkip);
    if(Res <= 0) return(Res);
        
    Res = fprintf(fp, "%u,%u\n", GraphPar->Iterated, 
        GraphPar->EscapeValueChanged);
    if(Res <= 0) return(Res);
    
    return Res;
}

/*!\brief
**  Creates a RealExperiment including the three GraphPar's.
**
**  This function returns a pointer to an instance of the TRealExp structure.
**
**\param    XStart X begin of the field (screen coordinates).
**\param    YStart Y begin of the field (screen coordinates).   
**\param    XEnd X end of the field (screen coordinates). 
**\param    YEnd Y end of the field (screen coordinates).
**
**\note
**  All members are set to the default values, including the GraphPar's
*/

TRealExp *CreateRealExp(unsigned int XStart, unsigned int YStart,
    unsigned int XEnd, unsigned int YEnd)
{
    TRealExp        *RealExp;

    if((RealExp = malloc(sizeof(*RealExp))) == NULL) 
        return NULL;

#ifdef DEBUGMODE
TRACE("FUNITER - realexp.c: Malloc()'ed 'TRealExp' data at %p\n", RealExp);
#endif
    
    RealExp->Settings = CreateSettings();
    if(RealExp->Settings == NULL) {
        free(RealExp);
        return NULL;
    }
    
    /* Step by step iteration */

    SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP], -0.7, 1.4, 1.4, 2, 
        1, 100, 0, 1);
    RealExp->GraphPar[GR_R_STEP].Coords = CreateCoords(-2, -2, 2, 2, 
        XStart, YStart, XEnd, YEnd, 56);
    if(RealExp->GraphPar[GR_R_STEP].Coords == NULL) { 
        free(RealExp);
        return NULL;
    }

    /* Orbits for variable x */

    SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X], -1.41, 1.4, 1.4, 2,
        1, 250, 0, 1); 
    RealExp->GraphPar[GR_R_ORBITS_X].Coords = CreateCoords(-2.5, -2, 2.5, 2, 
        XStart, YStart, XEnd, YEnd, 56);
    if(RealExp->GraphPar[GR_R_ORBITS_X].Coords == NULL) {
        free(RealExp);
        return NULL;
    }

    /* Orbits for variable c */

    SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C], -0.7, 0, 0, 2, 
        1, 250, 50, 1);
    RealExp->GraphPar[GR_R_ORBITS_C].Coords = CreateCoords(-2.5, -2, 0.5, 2, 
        XStart, YStart, XEnd, YEnd, 56);
    if(RealExp->GraphPar[GR_R_ORBITS_C].Coords == NULL) { 
        free(RealExp);
        return NULL;
    }

    /* Set default escapevalues for the graphs */
    
    RealExp->Graph = GR_R_STEP;
    RealExp->GraphPar[GR_R_STEP].EscapeValue = 
        CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_STEP],
        0, GR_R_STEP);
    RealExp->GraphPar[GR_R_STEP].EscapeValueChanged = FALSE;

    RealExp->Graph = GR_R_ORBITS_X;
    RealExp->GraphPar[GR_R_ORBITS_X].EscapeValue =
        CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_ORBITS_X],
        0, GR_R_ORBITS_X);
    RealExp->GraphPar[GR_R_ORBITS_X].EscapeValueChanged = FALSE;

    RealExp->Graph = GR_R_ORBITS_C;
    RealExp->GraphPar[GR_R_ORBITS_C].EscapeValue =
        CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_ORBITS_C],
        0, GR_R_ORBITS_C);
    RealExp->GraphPar[GR_R_ORBITS_C].EscapeValueChanged = FALSE;
    
    /* Initial values for graph, diagram, function, etc */
    
    RealExp->Graph = GR_R_STEP;
    RealExp->Diagram = DM_WEB;
    RealExp->Function = 0;
    RealExp->Changed = FALSE;    
    strcpy(RealExp->FunctionStr, "x*x+c");
    
    return(RealExp);
}

/*!\brief
**  Destroy (free memory) for a RealExperiment.
**
**  This function returns nothing.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
*/

void FreeRealExp(TRealExp *RealExp)
{
    FreeCoords(RealExp->GraphPar[GR_R_ORBITS_C].Coords);
    FreeCoords(RealExp->GraphPar[GR_R_ORBITS_X].Coords);
    FreeCoords(RealExp->GraphPar[GR_R_STEP].Coords);
    FreeSettings(RealExp->Settings);
    
#ifdef DEBUGMODE
TRACE("FUNITER - realexp.c: Free()'ing 'TRealExp' data at %p\n", RealExp);
#endif
    
    free(RealExp);
}

/*!\brief
**  Load RealExperiment.
**
**  This function returns 0 or 1 on succes, which is the functiontype, 
**  otherwise a value less than 0.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
**\param    FileName path + filename of the 'experiment' which should be
**          loaded.
**\param    Mode FALSE: experiments should be loaded in the new style, TRUE:
**          they should be loaded in old style. 
*/

int LoadRealExp(TRealExp *RealExp, char *FileName, int Mode)
{
    char    *fm = (Mode) ? "r" : "rb";
    int     Res = 0, Index, Graph = GR_R_STEP, EscapeValueChanged, 
                Functiontype = 0;
    FILE    *fp;
        
    if((fp = fopen(FileName, fm)) == NULL) return -1;

    /* Experimental code to read the old prm format */
    
    if(!Mode) {
        for(Index = 0; Index <= 2; Index++) {
           
            switch(Index) {
                case 0:
                    Graph = GR_R_STEP;  
                    break;
                case 1:
                    Graph = GR_R_ORBITS_X;
                    break;
                case 2:
                    Graph = GR_R_ORBITS_C;
                    break;
            }
        
            fread(&RealExp->GraphPar[Graph].c,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].xbegin1,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].xbegin2,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->XMin,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->XMax,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->YMin,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].Coords->YMax,1,sizeof(double),fp);
            fread(&RealExp->GraphPar[Graph].EscapeValue,1,sizeof(double),fp);
         
            fread(&RealExp->GraphPar[Graph].IterMax,1,sizeof(int),fp);
            fread(&RealExp->GraphPar[Graph].IterSkip,1,sizeof(int),fp);
            fread(&RealExp->GraphPar[Graph].IterSteps,1,sizeof(int),fp);

            fread(&RealExp->GraphPar[Graph].EscapeValue,1,sizeof(double),fp);
        }
        
        fseek(fp, 0x30e, SEEK_SET);
        
        fread(&RealExp->Graph, 1, sizeof(int), fp);
        
        fseek(fp, 0x318, SEEK_SET);
        
        fread(&RealExp->Function, 1, sizeof(int), fp);
        
        fseek(fp, 0x320, SEEK_SET);
        fread(&EscapeValueChanged, 1, sizeof(int), fp);
        RealExp->GraphPar[GR_R_STEP].EscapeValueChanged=EscapeValueChanged;
        RealExp->GraphPar[GR_R_ORBITS_X].EscapeValueChanged=EscapeValueChanged;
        RealExp->GraphPar[GR_R_ORBITS_C].EscapeValueChanged=EscapeValueChanged;
        
        fread(RealExp->FunctionStr, 100, sizeof(char), fp);
    }
    
    /* Read new par format */

    else {
        fscanf(fp, "%d", &Functiontype);
        
        if(Functiontype == 0) {
            Res = LoadRealGraphPar(fp, &RealExp->GraphPar[GR_R_STEP]);
            Res = LoadRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_X]);
            Res = LoadRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_C]);

            Res = fscanf(fp, "%d,%d,%d,%s\n", &RealExp->Graph, 
                &RealExp->Diagram, &RealExp->Function, RealExp->FunctionStr); 
    
            LoadSettings(fp, RealExp->Settings);    
        }
    }
    
    fclose(fp);
    if(get_config_int("main", "auto_par_convert", 0))
        WriteRealConfig(RealExp);
    return Functiontype;
}

/*!\brief
**  Read RealExperiment from configuration file.
**
**  This function returns 0 on succes, otherwise a value less than 0.
**  
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
*/

int ReadRealConfig(TRealExp *RealExp)
{
    char            EscValueStr[80] = "", FnParFileTag[6] = "", 
                    FnParFile[2048] = "";
    AL_CONST char   *p;
    TRGraphPar      *Step = &RealExp->GraphPar[GR_R_STEP], 
                    *OrbitsX = &RealExp->GraphPar[GR_R_ORBITS_X], 
                    *OrbitsC = &RealExp->GraphPar[GR_R_ORBITS_C];
    
    /*! 
    **  Extract the 'slot' for the correct function and set the corresponding 
    **  file.
    */
    
    snprintf(FnParFileTag, 5, "fn%d", RealExp->Function);
    if((p = get_config_string("real", FnParFileTag, NULL)) != NULL) {
        strncpy(FnParFile, p, 2047);
        fix_filename_slashes(FnParFile);
    }
    
    if(file_exists(FnParFile, 0, NULL)) {
        push_config_state();
        override_config_file(FnParFile);
    }
    
    strncpy(RealExp->FunctionStr, 
        get_config_string("real", "userfunction", "x*x+c"), 39);
    
    RealExp->Graph = get_config_int("real", "graph", 0);
    if(RealExp->Graph > 2) RealExp->Graph = 0;
    RealExp->Diagram = get_config_int("real", "diagram", 0);
    if(RealExp->Diagram > 3) RealExp->Diagram = 0;
    
    /*!
    **  Read graph specific parameters 
    */
    
    /*! 
    **  Step-by-step iteration 
    */
    
    Step->c = get_config_double("real_step", "c", Step->c);
    Step->xbegin1 = get_config_double("real_step", "startvalue1", Step->xbegin1);
    Step->xbegin2 = get_config_double("real_step", "startvalue2", Step->xbegin2);
    
    SetCoords(Step->Coords,
        get_config_double("real_step", "low", Step->Coords->XMin),
        get_config_double("real_step", "low", Step->Coords->XMin),
        get_config_double("real_step", "high", Step->Coords->XMax),
        get_config_double("real_step", "high", Step->Coords->XMax));
    
    strncpy(EscValueStr, 
        get_config_string("real_step", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        Step->EscapeValue = 
            CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_STEP],
            RealExp->Function, GR_R_STEP);
    else {
        Step->EscapeValue = ustrtod(EscValueStr, NULL);
        Step->EscapeValueChanged = TRUE;
    }    
    
    Step->IterSteps = get_config_int("real_step", "iterstep", Step->IterSteps);
    Step->Iterated = get_config_int("real_step", "iterated", Step->Iterated);
    
    /*! 
    **  Orbitdiagram for variable x 
    */
    
    OrbitsX->c = get_config_double("real_orbit_x", "c", OrbitsX->c);
    
    SetCoords(OrbitsX->Coords,
        get_config_double("real_orbit_x", "xlow", OrbitsX->Coords->XMin),
        get_config_double("real_orbit_x", "ylow", OrbitsX->Coords->YMin),
        get_config_double("real_orbit_x", "xhigh", OrbitsX->Coords->XMax),
        get_config_double("real_orbit_x", "yhigh", OrbitsX->Coords->YMax));
    
    strncpy(EscValueStr, 
        get_config_string("real_orbit_x", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        OrbitsX->EscapeValue = 
            CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_ORBITS_X],
            RealExp->Function, GR_R_ORBITS_X);
    else {
        OrbitsX->EscapeValue = ustrtod(EscValueStr, NULL);
        OrbitsX->EscapeValueChanged = TRUE;
    }    
    
    OrbitsX->IterSkip = 
        get_config_int("real_orbit_x", "iterskip", OrbitsX->IterSkip);
    OrbitsX->IterMax = 
        get_config_int("real_orbit_x", "itermax", OrbitsX->IterMax);
    
    /*! 
    **  Orbitdiagram for variable c
    */
    
    OrbitsC->xbegin1 =  
        get_config_double("real_orbit_c", "x", OrbitsC->xbegin1);
    
    SetCoords(OrbitsC->Coords, 
        get_config_double("real_orbit_c", "clow", OrbitsC->Coords->XMin),
        get_config_double("real_orbit_c", "ylow", OrbitsC->Coords->YMin),
        get_config_double("real_orbit_c", "chigh", OrbitsC->Coords->XMax),
        get_config_double("real_orbit_c", "yhigh", OrbitsC->Coords->YMax));
    
    strncpy(EscValueStr, 
        get_config_string("real_orbit_c", "escapevalue", "auto"), 79);
    if(strncmp(EscValueStr, "auto", 4) == 0)
        OrbitsC->EscapeValue = 
            CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_ORBITS_C],
            RealExp->Function, GR_R_ORBITS_C);
    else {
        OrbitsC->EscapeValue = ustrtod(EscValueStr, NULL);
        OrbitsC->EscapeValueChanged = TRUE;
    }    
    
    OrbitsC->IterSkip = 
        get_config_int("real_orbit_c", "iterskip", OrbitsC->IterSkip);
    OrbitsC->IterMax = 
        get_config_int("real_orbit_c", "itermax", OrbitsC->IterMax);

    /*!
    **  Settings.
    */
    
    RealExp->Settings = ReadSettingsConfig(FnParFile, RealExp->Settings);
    
    Step->Coords->Mode = OrbitsX->Coords->Mode = OrbitsC->Coords->Mode = 
        SetCoordsFlags(RealExp->Settings->COO_Status, 
                       RealExp->Settings->COO_Grid,
                       TRUE,
                       RealExp->Settings->COO_Calibrate, 
                       TRUE, TRUE); 
    
    /*!
    **  Close a specified parameter file 
    */
            
    if(file_exists(FnParFile, 0, NULL)) {
        override_config_file(NULL);
        pop_config_state();
    }
    
    return 0;
}

/*!\brief
**  Save RealExperiment.
**
**  This function returns 0 on succes, otherwise a value less than 0.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info.
**\param    FileName path + filename of the 'experiment' which should be
**          loaded.
*/

int SaveRealExp(TRealExp *RealExp, char *FileName)
{
    FILE    *fp;
    
    if((fp = fopen(FileName, "w+")) == NULL) 
        return -1;

    fprintf(fp, "%d\n", 0);

    SaveRealGraphPar(fp, &RealExp->GraphPar[GR_R_STEP]);
    SaveRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_X]);
    SaveRealGraphPar(fp, &RealExp->GraphPar[GR_R_ORBITS_C]);
        
    fprintf(fp, "%d,%d,%d,%s\n", RealExp->Graph, RealExp->Diagram, 
        RealExp->Function, RealExp->FunctionStr); 
    
    SaveSettings(fp, RealExp->Settings);    
      
    fclose(fp);
    return 0;
}

int WriteRealConfig(TRealExp *RealExp)
{
    char            FnParFileTag[6] = "", 
                    FnParBakFile[2048] = "", FnParFile[2048] = "";
    AL_CONST char   *p;
    TRGraphPar      *Step = &RealExp->GraphPar[GR_R_STEP], 
                    *OrbitsX = &RealExp->GraphPar[GR_R_ORBITS_X], 
                    *OrbitsC = &RealExp->GraphPar[GR_R_ORBITS_C];
    
    RealExp->Function = get_config_int("real", "function", RealExp->Function);
    if(RealExp->Function > 6) RealExp->Function = 0;
    
    /*! 
    **  Extract the 'slot' for the correct function and set the corresponding 
    **  file.
    */
    
    snprintf(FnParFileTag, 5, "fn%d", RealExp->Function);
    if((p = get_config_string("real", FnParFileTag, NULL)) != NULL) {
        strncpy(FnParFile, p, 2047);
        fix_filename_slashes(FnParFile);
    }
    else
        set_config_int("real", "function", RealExp->Function);

    if(file_exists(FnParFile, 0, NULL)) {
        push_config_state();
        override_config_file(FnParFile);
        replace_extension(FnParBakFile, FnParFile, "bak", sizeof(FnParBakFile));                    
        if(strcmp(FnParFile, FnParBakFile) != 0)
            SimpleCopyTextFile(FnParBakFile, FnParFile);
    }
    
    set_config_string("real", "userfunction", RealExp->FunctionStr); 
    
    set_config_int("real", "graph", RealExp->Graph);
    set_config_int("real", "diagram", RealExp->Diagram);
            
    /*!
    **  Set graph specific parameters 
    */
    
    /*! 
    **  Step-by-step iteration 
    */
    
    set_config_double("real_step", "c", Step->c);
    set_config_double("real_step", "startvalue1", Step->xbegin1);
    set_config_double("real_step", "startvalue2", Step->xbegin2);
    
    set_config_double("real_step", "low", Step->Coords->XMin);
    set_config_double("real_step", "high", Step->Coords->XMax);
    
    if(!Step->EscapeValueChanged)
        set_config_string("real_step", "escapevalue", "auto");
    else
        set_config_double("real_step", "escapevalue", Step->EscapeValue);
        
    set_config_int("real_step", "iterstep", Step->IterSteps);
    set_config_int("real_step", "iterated", Step->Iterated);
    
    /*! 
    **  Orbitdiagram for variable x 
    */
    
    set_config_double("real_orbit_x", "c", OrbitsX->c);
    
    set_config_double("real_orbit_x", "xlow", OrbitsX->Coords->XMin);
    set_config_double("real_orbit_x", "xhigh", OrbitsX->Coords->XMax);
    set_config_double("real_orbit_x", "ylow", OrbitsX->Coords->YMin);
    set_config_double("real_orbit_x", "yhigh", OrbitsX->Coords->YMax);
    
    if(!OrbitsX->EscapeValueChanged)
        set_config_string("real_orbit_x", "escapevalue", "auto");
    else
        set_config_double("real_orbit_x", "escapevalue", OrbitsX->EscapeValue);
    
    set_config_int("real_orbit_x", "iterskip", OrbitsX->IterSkip);
    set_config_int("real_orbit_x", "itermax", OrbitsX->IterMax);
    
    /*! 
    **  Orbitdiagram for variable c
    */
    
    set_config_double("real_orbit_c", "x", OrbitsC->xbegin1);
    
    set_config_double("real_orbit_c", "clow", OrbitsC->Coords->XMin),
    set_config_double("real_orbit_c", "chigh", OrbitsC->Coords->XMax),
    set_config_double("real_orbit_c", "ylow", OrbitsC->Coords->YMin),
    set_config_double("real_orbit_c", "yhigh", OrbitsC->Coords->YMax);

    if(!OrbitsC->EscapeValueChanged)
        set_config_string("real_orbit_c", "escapevalue", "auto");
    else
        set_config_double("real_orbit_c", "escapevalue", OrbitsC->EscapeValue);
        
    set_config_int("real_orbit_c", "iterskip", OrbitsC->IterSkip);
    set_config_int("real_orbit_c", "itermax", OrbitsC->IterMax);

    /*!
    **  Settings
    */
    
    WriteSettingsConfig(FnParFile, RealExp->Settings);
    
    /*!
    **  Close a specified parameter file 
    */
            
    if(file_exists(FnParFile, 0, NULL)) {
        override_config_file(NULL);
        pop_config_state();
    }
        
    return 0;
}


/*!\brief
**  Calculate escapevalue for real experiments.
**
**  This function returns the escapevalue.
**
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info. 
**
*/

double CalculateRealFnEscapeValue(TRGraphPar *GraphPar, int Function, int Graph)
{
    double      cvalue, maxc, minc, miny, maxy, cw, p;
    
    /*! 
    **  User changed escapevalue, unless the flag is reset it aborts this 
    **  function.
    */
    
    if(GraphPar->EscapeValueChanged) 
        return(GraphPar->EscapeValue);

    cw = 0.0;

    cvalue = GraphPar->c;
    minc = maxc = cvalue;
    miny = GraphPar->Coords->YMin;
    maxy = GraphPar->Coords->YMax;

    /*!
    **  When the graph is 'Orbits for variable c', for each c
    **  the escape-value is calculated.
    */

    if(Graph == GR_R_ORBITS_C) {
        minc = GraphPar->Coords->XMin;
        maxc = GraphPar->Coords->XMax;
    }

    switch(Function) {

        /*!
        **  x->x*x+c
        */

        case 0:
            if(cvalue <= 0.25)
                cw = 1.0 / 2.0 * (1.0 + sqrt(1.0 - 4.0 * cvalue));
            else
                cw = 0.5;
            break;
        
        /*!
        **  x->frac(cx)
        */

        case 1:
            cw = 1.0;
            break;

        /*!
        **  x->c * cos(x)
        */

        case 2:
            if(Graph == GR_R_STEP || Graph == GR_R_ORBITS_X)
                cw = fabs(cvalue);
            else
                cw = max(fabs(minc), fabs(maxc));
            break;
            
        /*!
        **  x->cx(1-x)
        */

        case 3:
            if(minc > 0.0)
                cw = 1.0;
            else if(maxc < 0.0) 
                cw = 0.5 * (1.0 + sqrt(1.0 - 4.0 / maxc));
            else
                cw = 1000000000000000.0;
            break;
            
        /*!
        **  x->-cx^3+(c+1)x^2
        */

        case 4:
            if(cvalue > 0.0)
                cw = max(1.0, fabs(2.0 / cvalue) + 1.0);
            else if((cvalue > -1.0) && (cvalue < 0))
                cw = -1.0 / cvalue;
            else if(cvalue <= -1.0)
                cw = 1.0;
            break;
            
        /*!
        **  x->c(1-|2x-1|)
        */

        case 5:
            cw = 1.0;
            break;
            
        /*
        **  userdefined function
        */

        case 6:
            cw = 0.0;
            break;

            /*
                if(RealExp->GraphPar[RealExp->Graph].EscapeFunctionStr != NULL)
                    cw = evalx(RealExp->GraphPar[RealExp->Graph].
                    EscapeFunctionStr, 3, 'C', cvalue, 'M', maxc, 'N', minc);
                else
                    cw = 0.0;
            */
    }

    if(cw == 0.0) 
        p = 2.0; 
    else 
        p = 1.0;
    
    return(max(max(fabs(p * miny), fabs(p * maxy)), cw));
}

/*!\brief
**  Draw the function for 'step-by-step iteration' web and combi web, step.
**
**  This function returns nothing.
**
**\param    Bitmap Points to an instance of the Allegro BITMAP structure.
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info. 
**
*/

void DrawFunction(BITMAP *Bitmap, TRealExp *RealExp)
{
    unsigned int    XPos;
    int             XLast, YLast, rfc, rxy;
    double          x, y, XEnd = 0, YStart = 0;     
    TCoords         Coords;
    TRGraphPar      GraphPar;
    
    Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
    GraphPar = RealExp->GraphPar[RealExp->Graph];
    rfc = RealExp->Settings->COL_Function;
    rxy = RealExp->Settings->COL_XYLine;
    
    /* 
    **  Use only the lower half of the bitmap in case the diagram is
    **  Combination of web and step.
    */
    
    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        XEnd = Coords.XEnd;
        YStart = Coords.YStart;
        Coords.XEnd = Coords.XEnd / 2;
        Coords.YStart = Coords.YStart + Coords.YEnd / 2;
    }
    
    acquire_bitmap(Bitmap);
        
    if(RealExp->Diagram == DM_WEB || RealExp->Diagram == DM_COMBIWEBSTEP) {

        /* Draw the function */
        
        for(XPos = 0; XPos < Coords.XEnd; XPos++) {
            x = R2M_X(Coords, XPos);
            y = IterateFn(RealExp->FunctionStr, RealExp->Function,
                GraphPar.Iterated, x, GraphPar.c,
                &RealExp->ParseErrNum, &RealExp->ParseErrPos);
            
            if((RealExp->Function == 6) && (RealExp->ParseErrNum != 0)) {
                break;
            }
            
            if(XPos == 0) {
                XLast = x;
                YLast = M2R_Y(Coords, y);
                continue;
            }            
            
            if(y <= Coords.YMax && y >= Coords.YMin) {
                line(Bitmap, XPos, M2R_Y(Coords, y), XLast, YLast,
                    makecol(RealExp->Settings->Palette[rfc].r,
                        RealExp->Settings->Palette[rfc].g,
                        RealExp->Settings->Palette[rfc].b));
            }
            XLast = XPos; YLast = M2R_Y(Coords, y);
        }

        /* Draw the line x=y */
        
        for(XPos = 0; XPos < Coords.XEnd; XPos++) {
            y = R2M_X(Coords, XPos);
            putpixel(Bitmap, XPos, (int) M2R_Y(Coords, y),
                makecol(RealExp->Settings->Palette[rxy].r,
                    RealExp->Settings->Palette[rxy].g,
                    RealExp->Settings->Palette[rxy].b));
        }
    }

    release_bitmap(Bitmap);
    vsync();
    
    /* Draw grid */
    
    DrawGrid(Bitmap, font, &Coords);

    /* Restore old values. */
    
    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        Coords.XEnd = XEnd;
        Coords.YStart = YStart;
    }
}

/*!\brief
**  Draw one iteration step ('step-by-step iteration, all diagrams).
**
**  This function returns the current step's line position.
**
**\param    Bitmap Points to an instance of the Allegro BITMAP structure.
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info. 
**\param    CurStepLinePos The line position of the current step.
**\param    startx The startingvalue (like xbegin1 and xbegin2).
**\param    x 
**\param    y
**\param    Col Drawingcolor for the current step.
*/

int DrawIterStep(BITMAP *Bitmap, TRealExp *RealExp, int CurStepLinePos,
    double startx, double x, double y, int Col)
{
    int         x1, y1, x2, y2, XEnd = 0, YStart = 0;
    TCoords     Coords;
        
    /* Some shortcuts */
    
    Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);

    /* In case of combination web/step we use the lower half of the screen */
    
    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        XEnd = Coords.XEnd;
        YStart = Coords.YStart;
        Coords.XEnd = Coords.XEnd / 2;
        Coords.YStart = Coords.YStart + (Coords.YEnd / 2);
    }

    /* Convert coordinates */

    x1 = (int) M2R_X(Coords, x);
    y1 = (int) M2R_Y(Coords, x);
    x2 = (int) M2R_X(Coords, y);
    y2 = (int) M2R_Y(Coords, y);

    /* Draw lines */

    acquire_bitmap(Bitmap);
    if(RealExp->Diagram == DM_WEB || RealExp->Diagram == DM_COMBIWEBSTEP) {
        vline(Bitmap, x1, y1, y2, Col);
        hline(Bitmap, x1, y2, x2, Col);
        if(RealExp->Diagram == DM_WEB)
            hline(Bitmap, 0, y2, 10, Col);
        if(RealExp->Diagram == DM_COMBIWEBSTEP) {
            hline(Bitmap, Coords.XEnd + CurStepLinePos, y2, Coords.XEnd +
                CurStepLinePos + RealExp->Settings->SSI_LineWidth, Col);
            CurStepLinePos+=RealExp->Settings->SSI_LineWidth;
        }
    }
    else if(RealExp->Diagram == DM_ITERVALUES)
        hline(Bitmap, M2R_X(Coords, startx), y2, M2R_X(Coords, startx) +
            RealExp->Settings->SSI_LineWidth, Col);
    else if(RealExp->Diagram == DM_STEP || RealExp->Diagram == DM_COMBIWEBSTEP)
    {
        hline(Bitmap, 50 + CurStepLinePos, y2, 50 + CurStepLinePos +
        RealExp->Settings->SSI_LineWidth, Col);
        CurStepLinePos+=RealExp->Settings->SSI_LineWidth;
    }
    release_bitmap(Bitmap);
    vsync();

    /* Restore old values. */

    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
        Coords.XEnd = XEnd;
        Coords.YStart = YStart;
    }
    return(CurStepLinePos);
}

/*!\brief
**  Step-by-step iteration, all diagrams.
**
**  This function returns the current step's line position.
**
**\param    Bitmap Points to an instance of the Allegro BITMAP structure.
**\param    RealExp Points to an instance of the TRealExp    
**          struct, see CreateGraphPar and CreateRealExp for more info. 
**\param    ValueMode      
**\param    HelpText A pointer to strings with helptexts for the current graph.
*/

int DoRealStep(BITMAP *Bitmap, TRealExp *RealExp, int ValueMode, 
    char **HelpText)
{
    char                v1[512], v2[512];
    int                 XWidth, XMickeys, YMickeys;
    unsigned int        xa = 0, ya = 0, CurStepLinePos = 0, Col = 0, Col2 = 0, 
                        CurCol, TempValueMode, Key, ExtKey = 0, Values = 1, 
                        XEnd = 0, YStart = 0;
    unsigned long int   Iter;
    double              x = 0.0, y = 0.0, xx = 0.0, yy = 0.0, xr = 0.0, 
                        yr = 0.0, TempWidth;
    TCoords             Coords;
    TRGraphPar          GraphPar;

    /* Some shortcuts */

    Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
    GraphPar = RealExp->GraphPar[RealExp->Graph];

    /* Determine the number of startingvalues */

    Values = (GraphPar.xbegin1 != GraphPar.xbegin2) + 1;

    /* Set the number of digits after the dot */

    TempWidth = (Coords.XMax - Coords.XMin);
    XWidth = 0;
    while(((int) TempWidth) == 0) { 
        TempWidth *= 10.0; 
        XWidth++; 
    }
    XWidth+=2;

    /*
    **  For 'Webdiagram' and 'Combination: web- and stepdiagram'
    **  the selected function is drawn and the x=y line.
    */

    DrawFunction(Bitmap, RealExp);
    if((RealExp->Function == 6) && (RealExp->ParseErrNum != 0)) {
        return 0;
    }

    /* Display help-text, user has to hit a key to continue */

    if(HelpText != NULL) 
        InfoBox(HelpText, 3);

    /* Store the valuemode */

    TempValueMode = ValueMode;

    /*
    **  In case Values is set to 0 (selecting startingvalue with the mouse),
    **  the mouse is set to the location of the 1st startingvalue in the
    **  parametermenu.
    */

    if(Values == 0) {
        rest(1);
        position_mouse((int) M2R_X(Coords, 
            RealExp->GraphPar[RealExp->Graph].xbegin1),
            (int) M2R_Y(Coords, RealExp->GraphPar[RealExp->Graph].xbegin1));
    }
    
    /* Start main-loop */

    CurCol = 1;
    do {

        /*
        **  In case 1 or 2 startingvalues are selected the initial values are
        **  set, otherwise the mouse-position is the new initial value.
        */

        if(Values == 1 || Values == 2) {
            ValueMode = 2;
            x = y = RealExp->GraphPar[RealExp->Graph].xbegin1;
            xx = yy = RealExp->GraphPar[RealExp->Graph].xbegin2;
        }
        if(Values == 0) {
            //rest(1);
            x = xx = y = yy = R2M_X(Coords, mouse_x);
        }

        /*
        **  Set starting iteration number,
        **  line position  for step-diagram is 0.
        */

        Iter = 0L;
        CurStepLinePos = 0;

        /* Start iteration-loop */

        while(Iter <= MAXINT) {

            if(Values == 0) {
                Values = 0;
                unscare_mouse();
                while(!keypressed()) {
                    rest(1);

                    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
                        XEnd = Coords.XEnd;
                        YStart = Coords.YStart;
                        Coords.XEnd = Coords.XEnd / 2;
                        Coords.YStart = Coords.YStart + (Coords.YEnd / 2);
                    }

                    xr = R2M_X(Coords, mouse_x);
                    if(RealExp->Diagram == DM_WEB ||
                        RealExp->Diagram == DM_COMBIWEBSTEP)
                    {
                        yr = xr;
                    }
                    xa = (int) M2R_X(Coords, xr);
                    if(RealExp->Diagram == DM_WEB ||
                        RealExp->Diagram == DM_COMBIWEBSTEP)
                    {
                        ya = (int) M2R_Y(Coords, yr);
                    }
                    if(RealExp->Diagram == DM_COMBIWEBSTEP) {
                        Coords.XEnd = XEnd;
                        Coords.YStart = YStart;
                    }

                    get_mouse_mickeys(&XMickeys, &YMickeys);
                    if((XMickeys != 0) || (YMickeys != 0)) {
#ifndef ALLEGRO_MACOSX
                        if(RealExp->Diagram == DM_WEB)
                            position_mouse(xa, ya);
                        else if(RealExp->Diagram == DM_COMBIWEBSTEP) {
                            position_mouse(xa, ya);
                        }
                        else
                            position_mouse(xa, Coords.YStart + 20);
#endif

                        sprintf(v1, "x = %2.*f", XWidth, xr);
                        if(RealExp->Settings->LAY_Statusline) 
                        {
                            StatusBar(1, RealExp->Graph, -1, 
                                RealExp->Settings->Palette, 
                                RealExp->Settings->COL_Max,v1);
                        }
                        unscare_mouse();

                        /*
                        **  Since a new startingvalue is selected, the
                        **  starting-iteration-number and current line position
                        **  for stepdiagram are reset to 0 again.
                        */

                        Iter = 0L;
                        CurStepLinePos = 0;

                        /*
                        **  Since this is the select-with-mouse mode,
                        **  Only one startingvalue is used (xbegin1 and xbegin2
                        **  will be made equal).
                        */

                        x = y = RealExp->GraphPar[RealExp->Graph].xbegin1 =
                            RealExp->GraphPar[RealExp->Graph].xbegin2 = xr;
                    }
                }
                scare_mouse();
            }
            else {
                do {
                    rest(1);
                    
                    /* 
                    **  Switch to select-with-mouse-mode (Values = 0) when the
                    **  right mousebutton is pressed. 
                    */
                    
                    if(mouse_b & 2) {
                        Values = 0;
                        ValueMode = TempValueMode;
                        clear_keybuf();
                        show_mouse(screen);
                        break;
                    }
                } while(!keypressed());
            }
            if(keypressed() && (Iter % (long int) GraphPar.IterSteps) == 0) {
                Key = readkey();
                ExtKey= Key >> 8;
            }

            if(Iter == 0 && ValueMode == 0) CurCol++;

            if(ExtKey == KEY_ESC) {
                scare_mouse();
                return 0;
            }

            if((ExtKey == KEY_DEL) || (ExtKey == KEY_D)) {
                if(Values == 0) 
                    scare_mouse();
                rectfill(Bitmap, Coords.XStart, Coords.YStart,
                    Coords.XEnd, Coords.YEnd, 0);
                DrawFunction(Bitmap, RealExp);
                if(Values == 0) 
                    unscare_mouse();
                break;
            }

            /* Show help-text */

            if(ExtKey == KEY_F1) {
                if(HelpText != NULL) 
                    InfoBox(HelpText, 3);
                continue;
            }

            /* Draw an iterationstep (or multiple at once) */

            if(ExtKey == KEY_ENTER && Iter >= GraphPar.IterSkip) {

                /*
                **  Print text next to each iterationstep
                */

                /*
                if((Iter == 0L || l == GraphPar.IterSkip) &&
                    (RealExp->Diagram == DM_ITERVALUES ||
                    RealExp->Diagram == DM_STEP) &&
                    (RealExp->Settings.SSI_Text)) 
                {
                    settextstyle(SMALL_FONT, HORIZ_DIR,
                        RealExp->Settings->SSI_FontSize);
                    sprintf(v1, "%2.*f", XWidth, x);
                        outtextxy(M2R_X(*(GraphPar.Coords), x),
                        M2R_Y(*(GraphPar.Coords), x), v1);
                    if(Values == 2) {
                        sprintf(v1, "%2.*f", XWidth, xx);
                        outtextxy(M2R_X(*(GraphPar.Coords), xx),
                            M2R_Y(*(GraphPar.Coords), xx), v1);
                    }
                }
                */

                /*
                **  Select the colors which are used for drawing the
                **  iterationstep(s).
                */

                switch(ValueMode) {
                    case 0:
                        if(CurCol == 0 ||
                            CurCol == RealExp->Settings->COL_Function || 
                            CurCol == RealExp->Settings->COL_XYLine)
                        {
                            CurCol++;
                        }
                        if(CurCol == RealExp->Settings->COL_Max)
                            CurCol = 1;
                        Col = RealExp->Settings->COL_List[CurCol %
                            RealExp->Settings->COL_Max];
                        break;
                    case 2:
                        Col =
                        RealExp->Settings->COL_List[RealExp->Settings->COL_Iter1];
                        Col2 = 
                        RealExp->Settings->COL_List[RealExp->Settings->COL_Iter2];
                        break;
                }

                /* Draw the lines */

                CurStepLinePos = DrawIterStep(Bitmap, RealExp, CurStepLinePos,
                    RealExp->GraphPar[RealExp->Graph].xbegin1, x, y, 
                    makecol(RealExp->Settings->Palette[Col].r,
                        RealExp->Settings->Palette[Col].g,
                        RealExp->Settings->Palette[Col].b));

                sprintf(v1, "n=%-4ld x=%.*f", Iter, 6, y);
                
                if(Values != 2) { 
                    if(RealExp->Settings->LAY_Statusline)
                    {
                        StatusBar(1, RealExp->Graph, -1, 
                            RealExp->Settings->Palette, 
                            RealExp->Settings->COL_Max, v1);
                    }
                }
                if(Values == 2) {
                    CurStepLinePos = DrawIterStep(Bitmap, RealExp, 
                        CurStepLinePos, 
                        RealExp->GraphPar[RealExp->Graph].xbegin2,
                        xx, yy, 
                        makecol(RealExp->Settings->Palette[Col2].r,
                            RealExp->Settings->Palette[Col2].g,
                            RealExp->Settings->Palette[Col2].b));

                    sprintf(v2, " x=%.*f", 6, yy);
                    strcat(v1, v2);
                    if(RealExp->Settings->LAY_Statusline)
                    {
                        StatusBar(1, RealExp->Graph, -1, 
                            RealExp->Settings->Palette, 
                            RealExp->Settings->COL_Max, v1);
                    }
                }
            }
            x = y;
            if(Values == 2)
                xx = yy;

            if(abs(x) <= GraphPar.EscapeValue) {
                y = IterateFn(RealExp->FunctionStr, RealExp->Function, 
                    GraphPar.Iterated, x, GraphPar.c,
                    &RealExp->ParseErrNum, &RealExp->ParseErrPos);
            }
            else if(Values < 2) {
                /*sound(1000); delay(dt); nosound(); dt = 20;*/
                continue;
            }
            else { 
                /*sound(1000); delay(dt); nosound(); dt = 20;*/ 
            }

            if(Values == 2) {
                if(abs(xx) <= GraphPar.EscapeValue) {
                    yy = IterateFn(RealExp->FunctionStr, RealExp->Function,
                        GraphPar.Iterated, xx, GraphPar.c,
                        &RealExp->ParseErrNum, &RealExp->ParseErrPos);
                }
                else { 
                    /*sound(1000); delay(dt); nosound(); dt = 20;*/ 
                }
            }

            if((RealExp->Function == 6) && (RealExp->ParseErrNum != 0)) {
                return 1;
            }
            
            Iter++;
        }
    } while(TRUE);
}

/*
**  DrawRealOrbits  -   Draw an orbitdiagram for variable c or variable x.
**
**  This function requires iterfun.c which requires parser.c.
**
**  DrawRealOrbits returns 0 when completed, and 1 when calculations are
**  interrupted by the user due to pressing the escape key.
*/

int DrawRealOrbits(BITMAP *Bitmap, TRealExp *RealExp)
{
    unsigned int        ExitValue = 0, Col, XPos, YPos, y0, rfc;
    unsigned long int   i;
    double              ca = 0.0, xa = 0.0, EscapeValue = 0.0;
    TCoords             Coords;
    TRGraphPar          GraphPar;

    /* Shortcuts for some very often used parameters */

    Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);
    GraphPar = RealExp->GraphPar[RealExp->Graph];
    rfc = RealExp->Settings->COL_Function;
    
    if(RealExp->Graph == GR_R_ORBITS_X) EscapeValue = GraphPar.EscapeValue;

    /*
    **  Store the original value for c, since that is used for the
    **  EscapeValue calculation for each c.
    */

    ca = RealExp->GraphPar[RealExp->Graph].c;

    /* Convert the 0.0 position on the Y-axe to a screen coordinate */

    y0 = M2R_Y(Coords, 0.0);

    /* Start a loop for each X position */

    for(XPos = 0; XPos < Coords.XEnd; XPos++) {

        if(RealExp->PrgProc != NULL) 
            RealExp->PrgProc(XPos, Coords.XEnd);
        
        /*
        **  Orbitdiagram for variable x.
        */

        if(RealExp->Graph == GR_R_ORBITS_X)
            xa = R2M_X(*(GraphPar.Coords), XPos);

        /*
        **  Orbitdiagram for variable c.
        **
        **  For 'Orbitdiagram for variable c' we calculate the escape-
        **  value for each c.
        */

        if(RealExp->Graph == GR_R_ORBITS_C) {
            RealExp->GraphPar[RealExp->Graph].c = R2M_X(*(GraphPar.Coords), 
                XPos);
            EscapeValue = 
                CalculateRealFnEscapeValue(&RealExp->GraphPar[RealExp->Graph], 
                RealExp->Function, RealExp->Graph);
            xa = GraphPar.xbegin1;
        }

        /* Iterationloop */

        for(i = 0; i < GraphPar.IterMax; i++) {

            /*
            **  Calculate Y position only when i is larger than the
            **  number of skipped iterations.
            */

            if(i >= GraphPar.IterSkip) {
                YPos = M2R_Y(Coords, xa);
                if(YPos <= Coords.YEnd)
                    putpixel(Bitmap, XPos, YPos,
                    makecol(RealExp->Settings->Palette[rfc].r,
                        RealExp->Settings->Palette[rfc].g,
                        RealExp->Settings->Palette[rfc].b));
            }

            /*
            **  Call routine which iterates the function.
            **  For the orbitdiagrams only the 1st iterated is calculated.
            */

            xa = IterateFn(RealExp->FunctionStr, RealExp->Function, 1,
                xa, RealExp->GraphPar[RealExp->Graph].c,
                &RealExp->ParseErrNum, &RealExp->ParseErrPos);

            if((RealExp->Function == 6) && (RealExp->ParseErrNum != 0)) {
                break;
            }
            
            /*
            **  Plot the escape colors when the absolute value of the
            **  function result is larger than the escape value.
            */

            if(fabs(xa) > EscapeValue) {
                if(i == 0)
                    Col = RealExp->Settings->COL_Max;
                else if(i == GraphPar.IterMax)
                    Col = 0;
                else {
                    Col = RealExp->Settings->COL_Max -
                        (i % RealExp->Settings->COL_Max);
                    if(Col == 0)
                        Col++;
                    else if(Col == RealExp->Settings->COL_Max)
                        Col--;
                }
                vline(Bitmap, XPos, y0 - 3, y0 + 3,
                    makecol(RealExp->Settings->Palette[Col].r,
                        RealExp->Settings->Palette[Col].g,
                        RealExp->Settings->Palette[Col].b));
                break;
            }
        }
        
        /* Check for keypress */
            
        if(keypressed()) {
            if((readkey() >> 8) == KEY_ESC) {
                ExitValue = 1;
                break;
            }
        }
    }

    /* Restore original value for c */

    RealExp->GraphPar[RealExp->Graph].c = ca;

    /* Draw grid */

    DrawGrid(Bitmap, font, &Coords);
    return ExitValue;
}

/*
**  IterateFn
**
**  Iterate a function.
**
**  FunctionStr             -   char, Function string for the parser.
**
**  Function                -   int, Function-number.
**
**  n                       -   int, n-th iterated of function.
**
**  x                       -   double, x-parameter.
**
**  c                       -   double, c-parameter.
**
**  ErrNum                  -   int *, Number of the error: 0 = no error.
**                                     see parser.h for the complete list.
**
**  ErrPos                  -   int *, Position of the error in the string.
**
**  The routine returns the function-result.
*/

double IterateFn(char *FunctionStr, int Function, int n, double x, double c,
    int *ErrNum, int *ErrPos)
{
    int        i;
    double     y = 0.0;

    switch(Function) {

        /*
        **  x->x*x+c
        */

        case 0:
            for(i = 1; i <= n; i++) {
                y = x * x + c;
                x = y;
            }
            break;

        /*
        **  x->frac(cx)
        */

        case 1:
            for(i = 1; i <= n; i++) {
                y = c * x - floor(c*x);
                x = y;
            }
            break;

        /*
        **  x->c * cos(x)
        */

        case 2:
            for(i = 1; i <= n; i++) {
                y = c * cos(x);
                x = y;
            }
            break;

        /*
        **  x->cx(1-x)
        */

        case 3:
            for(i = 1; i <= n; i++) {
                y = c * x * (1.0 - x);
                x = y;
            }
            break;

        /*
        **  x->-cx^3+(c+1)x^2
        */

        case 4:
            for(i = 1; i <= n; i++) {
                y = -c * x * x * x + (c + 1.0) * x * x;
                x = y;
            }
            break;

        /*
        **  x->c(1-|2x-1|)
        */

        case 5:
            for(i = 1; i <= n; i++) {
                y = c * (1.0 - fabs(2.0 * x - 1.0));
                x = y;
            }
            break;

            /*
            **  userdefined function
            */

        case 6:
            for(i = 1; i <= n; i++) {
                y = Parse(ErrNum, ErrPos, FunctionStr, 2, 'x', x, 'c', c);
                x = y;
            }
            break;
    }
    return(y);
}

/*
**  This routine can be used to jump from one graph to another.
**  For the current (source/original) graph the routine uses the member in 
**  TRealExp, DestGraph is the graph to which should be jumped. 
**  Dialog points to a DIALOG structure which contains fields to enter the
**  value for x or c manually (not yet implemented).
**  InfoStr points to a text which should display the key to hit for help,
**  as HelpText should point to an array of lines for the actual help-text.
**  On 'cancel' the routine returns -1, on 'ok' it returns the selected graph. 
*/

int RealJump(TRealExp *RealExp, int DestGraph, DIALOG *Dialog, 
    const char *InfoStr, char *HelpText[])
{
    char        Str[1024], *TmpInfoStr;
    int         Key, AltKey, XMickeys, YMickeys, tmp_xp = 0, tmp_yp = 0;
    TCoords     Coords;
        
    Coords = *(RealExp->GraphPar[RealExp->Graph].Coords);

    if((RealExp->Graph == GR_R_ORBITS_X || RealExp->Graph == GR_R_ORBITS_C)) {
        
        if((TmpInfoStr = (char *) malloc(1024)) == NULL) 
            return -2;

        if((strlen(InfoStr) < 512) && (InfoStr != NULL)) 
            strcpy(TmpInfoStr, InfoStr);
            
        if(DestGraph == GR_R_STEP) 
            strcat(TmpInfoStr, " [ x = %2.*f ]");
        if(DestGraph == GR_R_ORBITS_X) 
            strcat(TmpInfoStr, " [ c = %2.*f ]");
        if(DestGraph == GR_R_ORBITS_C) 
            strcat(TmpInfoStr, " [ x = %2.*f ]");
        
        show_mouse(screen);
        tmp_xp = mouse_x;
        tmp_yp = mouse_y;
        do {
            rest(1);
            get_mouse_mickeys(&XMickeys, &YMickeys);
            if((XMickeys != 0) || (YMickeys != 0)) {
                if(DestGraph == GR_R_STEP) {
                    tmp_yp = mouse_y;
                    sprintf(Str, TmpInfoStr, 4, R2M_Y(Coords, mouse_y));
                }
                else {
                    tmp_xp = mouse_x;
                    sprintf(Str, TmpInfoStr, 4, R2M_X(Coords, mouse_x));
                }
                if(RealExp->Settings->LAY_Statusline)
                {
                    StatusBar(1, RealExp->Graph, -1, RealExp->Settings->Palette, 
                        RealExp->Settings->COL_Max, Str);
                }
            }
            if(keypressed()) {
                Key = readkey();
                AltKey = Key >> 8;
                if(HelpText != NULL) {
                    if(AltKey == KEY_F1) 
                        InfoBox(HelpText, 2);
                }
                if(Dialog != NULL) {
                    if(AltKey == KEY_F2) 
                        popup_dialog(Dialog, 0);
                }
            }
        } while(!(mouse_b & 1) && AltKey != KEY_ENTER && AltKey != KEY_ESC);
        show_mouse(NULL);
        
        free(TmpInfoStr);
        
        if(AltKey == KEY_ESC) 
            return(-1);
    }
  
    /* Step-by-step  ->  Orbit diagram for var. x */
    
    if(RealExp->Graph == GR_R_STEP && DestGraph == GR_R_ORBITS_X) {
        RealExp->GraphPar[DestGraph].c = RealExp->GraphPar[RealExp->Graph].c;
        RealExp->Graph = DestGraph;
    }
    
    /* Orbit diagram for var. x  ->  Orbit diagram for var. c */
    
    if(RealExp->Graph == GR_R_ORBITS_X && DestGraph == GR_R_ORBITS_C) {
        RealExp->Graph = DestGraph;
        RealExp->GraphPar[RealExp->Graph].xbegin1 = R2M_X(Coords, tmp_xp);
    }
    
    /* Orbit diagram for var. x  ->  Step-by-step */
    
    if(RealExp->Graph == GR_R_ORBITS_X && DestGraph == GR_R_STEP) {
        RealExp->GraphPar[DestGraph].c = RealExp->GraphPar[RealExp->Graph].c; 
        RealExp->GraphPar[DestGraph].xbegin1 = 
        RealExp->GraphPar[DestGraph].xbegin2 = R2M_Y(Coords, tmp_yp);
        RealExp->Graph = DestGraph;
    }
    
    /* Orbit diagram for var. c  ->  Orbit diagram for var. x */
    
    if(RealExp->Graph == GR_R_ORBITS_C && DestGraph == GR_R_ORBITS_X) {
        RealExp->Graph = DestGraph;
        RealExp->GraphPar[DestGraph].c = R2M_X(Coords, tmp_xp); 
    }
    
    /* Recalculate escapevalue */
        
    RealExp->GraphPar[RealExp->Graph].EscapeValueChanged = FALSE;
    RealExp->GraphPar[RealExp->Graph].EscapeValue =
        CalculateRealFnEscapeValue(&RealExp->GraphPar[RealExp->Graph],
        RealExp->Function, RealExp->Graph);

    return(RealExp->Graph);
}

int RealDraw(TRealExp *RealExp, char **RealStepHelpStr) 
{
    BITMAP              *Bitmap;
    
    Bitmap = create_bitmap(SCREEN_W, SCREEN_H);
    clear_bitmap(Bitmap);

    if(RealExp->Graph < GR_R_ORBITS_X) {
        clear_bitmap(screen);
        DoRealStep(screen, RealExp, 0, RealStepHelpStr);
    }
    else {
        if(Bitmap != NULL) {
            DrawRealOrbits(Bitmap, RealExp);
            blit(Bitmap, screen, 0,0, 0, 0, SCREEN_W, SCREEN_H);
        }
        else
            DrawRealOrbits(screen, RealExp);
    }
    if(RealExp->Settings->LAY_Statusline) {
        StatusBar(1, RealExp->Graph, -1, RealExp->Settings->Palette, 
            RealExp->Settings->COL_Max, NULL);
    }
    destroy_bitmap(Bitmap);
    return 0;
}

void SelectRealStdCoords(TCoords *Coords, int Function, int Graph)
{
    switch(Function) {

        /*
        **  x -> x * x + c
        */

        case 0:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -2, -2, 2, 2);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -2.5, -2, 2.5, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -2.5, -2, 0.5, 2);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> frac(cx)
        */

        case 1:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, 0, 0, 1, 1);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -1, 0, 2, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -2, 0, 2, 1);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> c * cos(x)
        */

        case 2:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -5, -5, 5, 5);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -4, -5, 4, 5);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -4, -5, 4, 5);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> cx(1 - x)
        */

        case 3:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, 0, 0, 1, 1);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -5, -1, 1.5, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, 0, 0, 5, 1.25);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> -cx ^ 3 + (c + 1) x ^ 2
        */

        case 4:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -0.5, -0.5, 1.5, 1.5);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -0.3, -1.5, 1.3, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, 0, -0.5, 10, 2);
                    break;
                default:
                    break;
            }
            break;

        /*
        **  x -> c(1 - |2x - 1|)
        */

        case 5:
            switch(Graph) {
                case GR_R_STEP:
                    SetCoords(Coords, -2, -2, 1.5, 1.5);
                    break;
                case GR_R_ORBITS_X:
                    SetCoords(Coords, -0.2, -2, 1.2, 2);
                    break;
                case GR_R_ORBITS_C:
                    SetCoords(Coords, -2, -2, 2, 2);
                    break;
                default:
                    break;
            }
            break;

        default:
            break;
    }
}

void SelectExperiment(TRealExp *RealExp, int Mode, int n)
{
    int             OldGraph = RealExp->Graph;

    SelectRealStdCoords(RealExp->GraphPar[GR_R_STEP].Coords, n, 
        GR_R_STEP);
    SelectRealStdCoords(RealExp->GraphPar[GR_R_ORBITS_X].Coords, n, 
        GR_R_ORBITS_X);
    SelectRealStdCoords(RealExp->GraphPar[GR_R_ORBITS_C].Coords, n, 
        GR_R_ORBITS_C);

    switch(n) {

        /*
        **  x -> x * x + c
        */

        case 0:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    -0.7, 1.4, 1.4, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    -1.41, -2.5, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    -2.5, 0, 0, 0, 1, 250, 50, 1);
            }
            break;

        /*
        **  x -> frac(cx)
        */

        case 1:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    3, 0.1, 0.1, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    1.3, -1, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    -1, 0.2, 0, 0, 1, 250, 0, 1);
            }
            break;

        /*
        **  x -> c * cos(x)
        */

        case 2:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    1.5, 4, 4, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    3, 0.5, 0.5, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                0, 0, 0.5, 0, 1, 250, 0, 1);
            }
            break;

        /*
        **  x -> cx(1 - x)
        */

        case 3:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    3.5, 0.015, 0.015, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    3.5, -0.5, 0, 0, 1, 250, 50, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    0, 0.5, 0.5, 0, 1, 250, 50, 1);
            }
            break;

        /*
        **  x -> -cx ^ 3 + (c + 1) x ^ 2
        */

        case 4:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    5, 0.3, 0.3, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    5, -0.3, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    0, 0.3, 0, 0, 1, 250, 0, 1);
            }
            break;

        /*
        **  x -> c(1 - |2x - 1|)
        */

        case 5:
            if(Mode) {
                SetRealGraphPar(&RealExp->GraphPar[GR_R_STEP],
                    1.5, 0.075441, 0.075441, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_X],
                    1.5, -1, 0, 0, 1, 250, 0, 1);
                SetRealGraphPar(&RealExp->GraphPar[GR_R_ORBITS_C],
                    -2, 0.11111111, 0.3, 0, 1, 250, 0, 1);
            }
            break;

        default:
            break;
    }

    /* Set default escapevalues for the graphs */

    RealExp->Graph = GR_R_STEP;
    RealExp->GraphPar[GR_R_STEP].EscapeValue =
        CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_STEP],
        n, GR_R_STEP);
    RealExp->GraphPar[GR_R_STEP].EscapeValueChanged = FALSE;

    RealExp->Graph = GR_R_ORBITS_X;
    RealExp->GraphPar[GR_R_ORBITS_X].EscapeValue =
        CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_ORBITS_X],
        n, GR_R_ORBITS_X);
    RealExp->GraphPar[GR_R_ORBITS_X].EscapeValueChanged = FALSE;

    RealExp->Graph = GR_R_ORBITS_C;
    RealExp->GraphPar[GR_R_ORBITS_C].EscapeValue =
        CalculateRealFnEscapeValue(&RealExp->GraphPar[GR_R_ORBITS_C],
        n, GR_R_ORBITS_C);
    RealExp->GraphPar[GR_R_ORBITS_C].EscapeValueChanged = FALSE;

    RealExp->Graph = OldGraph;
}
