/*
    This file is part of Funiter,
    Real and complex function iteration software.

    complex.c - complex math routines.

    Copyright (C) 1995-2007 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen)

    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "complex.h"

/*
**  Constructor: create a complex number
*/

TComplex Complex(double RealPart, double ImagPart)
{
    TComplex c;

    c.r = RealPart;
    c.i = ImagPart;

    return(c);
}

/*
**  Return the imaginary-part of a complex-number.
*/

double Imag(TComplex c)
{
    return(c.i);
}

/*
**  Return the real-part of a complex-number.
*/

double Real(TComplex c)
{
    return(c.r);
}

/*
**  Complex negation.
*/

TComplex CNeg(TComplex c)
{
    TComplex    Res;

    Res.r = -c.r;
    Res.i = -c.i;

    return(Res);
}

/*
**  Complex addition (c1 + c2).
*/

TComplex CAdd(TComplex c1, TComplex c2)
{
    TComplex    Res;

    Res.r = c1.r + c2.r;
    Res.i = c1.i + c2.i;

    return(Res);
}

/*
**  Complex subtraction (c1 - c2).
*/

TComplex CSub(TComplex c1, TComplex c2)
{
    TComplex    Res;

    Res.r = c1.r - c2.r;
    Res.i = c1.i - c2.i;

    return(Res);
}

/*
**  Multiply two complex-numbers (c1 * c2).
*/

TComplex CMul(TComplex c1, TComplex c2)
{
    TComplex    Res;

    Res.r = c1.r * c2.r - c1.i * c2.i;
    Res.i = c2.r * c1.i + c2.i * c1.r;

    return(Res);
}

/*
**  Divide two complex numbers (c1 / c2).
*/

TComplex CDiv(TComplex c1, TComplex c2)
{
    TComplex    Res;
    double      Absc2;

    Absc2 = c2.r * c2.r + c2.i * c2.i;

    Res.r = c1.r * c2.r + c1.i * c2.i / Absc2;
    Res.i = c1.i * c2.r - c1.r * c2.i / Absc2;

    return(Res);
}

/*
**  Return the absolute value of c.
*/

double CAbs(TComplex c)
{
    return(sqrt(c.r * c.r + c.i * c.i));
}

/*
**  Complex root (calculates the nth root of value c).
**
**  Root is a pointer to a list with the roots.
*/
 
void CSqrt(TComplex *Root, TComplex c, int n)
{
    int         i;
    double      r, p, tpn;

    if(Root == NULL) return;
 
    if(c.r == 0.0 && c.i == 0.0) return;

    p = atan2(c.i, c.r) / (double) n;
    r = pow(sqrt(c.r * c.r + c.i * c.i), 1.0 / (double) n);
    tpn = (2.0 * M_PI) / (double) n;

    for(i = 0; i < n; i++) {
        Root[i].r = r * cos(p);
        Root[i].i = r * sin(p);
        p += tpn;
    }
}

/*
**  Complex sine.
*/

TComplex CSin(TComplex c)
{
    return(Complex(sin(c.r) * cosh(c.i), cos(c.r) * sinh(c.i)));
}

/*
**  Complex cosine.
*/

TComplex CCos(TComplex c)
{
    return(Complex(cos(c.r) * cosh(c.i), -sin(c.r) * sinh(c.i)));
}

/*
**  Complex exp.
*/

TComplex CExp(TComplex c)
{
    return(Complex(exp(c.r) * cos(c.i), exp(c.r) * sin(c.i)));
}

/*
**  Complex log.
*/

TComplex CLog(TComplex c)
{
    return(Complex(log(CAbs(c)), atan2(c.i, c.r)));
}
