/*
    This file is part of Funiter,
    Real and complex function iteration software.

    clsmenus.c - Funiter's old menu-style routine.

    Copyright (C) 1995-2010 Stijn Wolters.
    Original idea: Ernic Kamerich (University of Nijmegen).
    
    See README for contact information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    See COPYING for more information.
*/

#include "clsmenus.h"

static int          Functiontype;

int InitClassicMenus(int FnType, char *LngFilename)
{
    Functiontype = FnType;
    return(OpenLanguageFile(LngFilename));
}

int ClsMnuMain(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp,
    int DefaultItemIndex)
{
    TMenu       *Menu;
    int         i, MenuXPos, MenuWidth, MaxItemLength, Key, FunctionFlag = TRUE,
                OldFunctiontype, ISet = 0, ILay = 0, ICoo = 0, IStep = 0, 
                ICol = 0, IFile = 0;
    char        **Str = GetItemStrings("MAINMENU", &MenuWidth, &MaxItemLength, 
                14);

    OldFunctiontype = Functiontype;
    
    if(Str == NULL) {
        MaxItemLength = 27;
        Menu = CreateMenu(MenuTitleStrings[0], 26, 7, 0, 11, MaxItemLength, 0,
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 13, 
            (1 << M_HEADER) + (1 << M_STATUSBAR), MainItems);
    }
    else {
        MenuXPos = (80 - MaxItemLength) / 2;
        Menu = CreateMenu(Str[0], MenuXPos, 7, 0, 11, MenuWidth, 0,
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 13, 
            (1 << M_HEADER) + (1 << M_STATUSBAR), MainItems);
        for(i = 1; i <= 13; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &Str[i]);
    }

    /* Set correct value for the first item (Real/Complex functions) */

    if(Functiontype == FT_COMPLEX)
        SwapItemCaptions(&Menu->Items[11]->Caption, &Menu->Items[0]->Caption);

    /* For Standard menu's, Settings and Files are not vissible */

    if(SA(GEN_MenuStyle) == MS_ADVANCED) {
        SwapItemCaptions(&Menu->Items[12]->Caption, &Menu->Items[8]->Caption);
        Menu->Items[7]->Flag = 96;
        Menu->Items[6]->Flag = 96;
    }
    else {
        SwapItemCaptions(&Menu->Items[8]->Caption, &Menu->Items[12]->Caption);
        Menu->Items[7]->Flag = 0;
        Menu->Items[6]->Flag = 0;
    }

    Console->CursorMode = _NOCURSOR;
    Menu->DefaultItemIndex = DefaultItemIndex;
    if(Menu->DefaultItemIndex != 0) FunctionFlag = FALSE;
    
    DrawMenu(Console, Menu);

    while(TRUE) {

        Key = DoMenu(Console, Menu);

        switch(Menu->ItemIndex) {
            case 0:
                if(Key == KEY_ENTER) {
                    Menu->DefaultItemIndex = 0;
                    Functiontype = !Functiontype;
                    
                    /*
                    **  Don't skip the item 'Function' when 'functiontype' is
                    **  toggled.
                    */

                    FunctionFlag = TRUE;

                    if(Functiontype == FT_REAL)
                        SwapItemCaptions(&Menu->Items[11]->Caption,
                            &Menu->Items[0]->Caption);
                    else
                        SwapItemCaptions(&Menu->Items[0]->Caption,
                            &Menu->Items[11]->Caption);

                    DrawItem(Console, Menu, Menu->Items[0]);
                }
                break;
            case 1:
                if(Key == KEY_ENTER) {
                    ClsMnuGraph(Console, RealExp, ComplexExp);

                    /* To 'Function' or 'Parameters' */

                    if(FunctionFlag) {
                        FunctionFlag = FALSE;
                        Menu->DefaultItemIndex = 2;
                    }
                    else
                        Menu->DefaultItemIndex = 3;

                    DrawMenu(Console, Menu);
                }
                break;
            case 2:
                if(Key == KEY_ENTER) {
                    ClsMnuFunction(Console, RealExp, ComplexExp);
                    Menu->DefaultItemIndex = 3;
                    DrawMenu(Console, Menu);
                }
                break;
            case 3:
                if(Key == KEY_ENTER) {
                    ClsMnuPar(Console, RealExp, ComplexExp);
                    Menu->DefaultItemIndex = 4;
                    DrawMenu(Console, Menu);
                }
                break;
            case 4:
                if(Key == KEY_ENTER) {
                    ClsMnuDraw(Console, RealExp, ComplexExp);
                    DrawMenu(Console, Menu);
                }
                break;
            case 5:
                if(Key == KEY_ENTER) {
                }
                break;
            case 6:
                if(Key == KEY_ENTER) {
                    ClsMnuSettings(Console, RealExp, ComplexExp, &ISet, &ILay,
                        &ICoo, &IStep, &ICol);
                    Menu->DefaultItemIndex = 4;
                    DrawMenu(Console, Menu);
                }
                break;
            case 7:
                if(Key == KEY_ENTER) {
                    OldFunctiontype = Functiontype;
                    ClsMnuFiles(Console, RealExp, ComplexExp, &IFile);
                    Menu->DefaultItemIndex = 4;
                    if(Functiontype != OldFunctiontype) {
                        SwapItemCaptions(&Menu->Items[11]->Caption,
                            &Menu->Items[0]->Caption);
                    }
                    DrawMenu(Console, Menu);
                }
                break;
            case 8:
                if(Key == KEY_ENTER) {
                    RealExp->Settings->GEN_MenuStyle = 
                        !RealExp->Settings->GEN_MenuStyle;
                    ComplexExp->Settings->GEN_MenuStyle = 
                        !ComplexExp->Settings->GEN_MenuStyle;

                    if(SA(GEN_MenuStyle) == MS_ADVANCED) {
                        SwapItemCaptions(&Menu->Items[8]->Caption,
                            &Menu->Items[12]->Caption);
                        Menu->Items[7]->Flag = 96;
                        Menu->Items[6]->Flag = 96;
                    }
                    else {
                        SwapItemCaptions(&Menu->Items[12]->Caption,
                            &Menu->Items[8]->Caption);
                        Menu->Items[7]->Flag = 0;
                        Menu->Items[6]->Flag = 0;
                    }
                    DrawItem(Console, Menu, Menu->Items[8]);
                    DrawItem(Console, Menu, Menu->Items[7]);
                    DrawItem(Console, Menu, Menu->Items[6]);
                }
                break;
            case 9:
                if(Key == KEY_ENTER) {
                    ClsMnuAbout(Console);
                    Menu->DefaultItemIndex = 9;
                    DrawMenu(Console, Menu);
                }
                break;
            case 10:
                if(Key == KEY_ENTER) {
                    DestroyItemStrings(Str, 14);
                    DestroyMenu(Menu);
                    CloseLanguageFile();
                    return 0;
                }
                break;
        }
    }
    return 0;
}

int ClsMnuGraph(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    TMenu       *Menu;
    int         Key, OldGraph, MenuXPos, RMenuWidth, CMenuWidth,
                RMaxItemLength, CMaxItemLength, i;
    char        **GRStr = GetItemStrings("RGRAPHMENU", &RMenuWidth, 
                &RMaxItemLength, 4),
                **GCStr = GetItemStrings("CGRAPHMENU", &CMenuWidth, 
                &CMaxItemLength, 6);

    if(Functiontype == FT_REAL)
        OldGraph = RealExp->Graph;
    else
        OldGraph = ComplexExp->Graph;

    /* Create menu */

    if(Functiontype == FT_COMPLEX) {
        if(GCStr == NULL) {
            CMaxItemLength = 43; 
            Menu = CreateMenu(MenuTitleStrings[1], 18, 10, ComplexExp->Graph, 4,
                CMaxItemLength, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 5,
                (1 << M_HEADER) + (1 << M_STATUSBAR), ComplexGraphItems);
        }
        else {
            MenuXPos = (80 - CMenuWidth) / 2;
            Menu = CreateMenu(GCStr[0], MenuXPos, 10, ComplexExp->Graph, 4,
                CMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 5,
                (1 << M_HEADER) + (1 << M_STATUSBAR), ComplexGraphItems);
            for(i = 1; i <= 5; i++) 
                SwapItemCaptions(&Menu->Items[i - 1]->Caption, &GCStr[i]);
        }

        /* Only function 0-2 (z -> z ^ n + c) has the 'inverse' option so far */
        
        if(ComplexExp->Function > 2) 
            Menu->Items[1]->Flag = 0;
        else 
            Menu->Items[1]->Flag = 97;

        /*
        **  For 'Orbit diagram for var. c and z not equal to 0, it isn't
        **  a real Mandelbrot-set anymore, so we delete that from the caption.
        */

        if((ComplexExp->GraphPar[GR_C_ORBITS_C].zReal != 0) ||
            (ComplexExp->GraphPar[GR_C_ORBITS_C].zImag != 0) ||
            (ComplexExp->Function > 0)) 
        {
                SwapItemCaptions(&Menu->Items[3]->Caption,
                    &Menu->Items[4]->Caption);
        }
    }
    else {
        if(GRStr == NULL) {
            RMenuWidth = 31; 
            Menu = CreateMenu(MenuTitleStrings[1], 24, 11, RealExp->Graph, 3,
                RMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 3,
                (1 << M_HEADER) + (1 << M_STATUSBAR), RealGraphItems);
        }
        else {
            MenuXPos = (80 - RMenuWidth) / 2;
            Menu = CreateMenu(GRStr[0], MenuXPos, 11, RealExp->Graph, 3,
                RMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 3,
                (1 << M_HEADER) + (1 << M_STATUSBAR), RealGraphItems);
            for(i = 1; i <= 3; i++) 
                SwapItemCaptions(&Menu->Items[i - 1]->Caption, &GRStr[i]);
        }
    }

    if(Functiontype == FT_COMPLEX)
        Menu->DefaultItemIndex = ComplexExp->Graph;
    else
        Menu->DefaultItemIndex = RealExp->Graph;

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    while(TRUE) {
        Key = DoMenu(Console, Menu);

        if(Functiontype == FT_COMPLEX) {
            ComplexExp->Graph = Menu->ItemIndex;
            if(Key == KEY_ENTER) {
                if(ClsMnuDiagram(Console, RealExp, ComplexExp) == KEY_ESC) {
                    DrawMenu(Console, Menu);
                    continue;
                }
                break;
            }
        }
        else {
            RealExp->Graph = Menu->ItemIndex;
            if(RealExp->Graph == GR_R_STEP) {
                if(Key == KEY_ENTER) { 
                    if(ClsMnuDiagram(Console, RealExp, ComplexExp) == KEY_ESC) {
                        DrawMenu(Console, Menu);
                        continue;
                    }
                    break;
                }
            }
        }

        if(Key == KEY_ESC) 
            break;
        if(Key == KEY_ENTER) 
            break;
    }

    /* Process values */

    if(Key == KEY_ESC) {
        if(Functiontype == FT_REAL) {
            RealExp->Graph = OldGraph;
            if(RealExp->Graph != Menu->ItemIndex) RealExp->Changed = TRUE;
        }
        else 
            ComplexExp->Graph = OldGraph;
    }

    if(Functiontype == FT_REAL) {
        if(RealExp->Settings->GEN_ParConnect) {

            /* 
            **  From 'Step-by-step iteration'  ->  'Orbitdiagram for var. x' 
            **  From 'Orbitdiagram for var. x' ->  'Step-by-step iteration'
            */

            if(((RealExp->Graph == GR_R_ORBITS_X) && (OldGraph == GR_R_STEP)) ||
               ((RealExp->Graph == GR_R_STEP) && (OldGraph == GR_R_ORBITS_X))) 
            {
                RealExp->GraphPar[RealExp->Graph].c = 
                    RealExp->GraphPar[OldGraph].c;
                RealExp->GraphPar[RealExp->Graph].Coords->XMin =
                    RealExp->GraphPar[OldGraph].Coords->XMin;
                RealExp->GraphPar[RealExp->Graph].Coords->XMax =
                    RealExp->GraphPar[OldGraph].Coords->XMax;
                RealExp->GraphPar[RealExp->Graph].Coords->YMin =
                    RealExp->GraphPar[OldGraph].Coords->YMin;
                RealExp->GraphPar[RealExp->Graph].Coords->YMax =
                    RealExp->GraphPar[OldGraph].Coords->YMax;
                RealExp->GraphPar[RealExp->Graph].EscapeValue = 
                    RealExp->GraphPar[OldGraph].EscapeValue;
            }

            /* 
            **  From 'Step-by-step iteration'  ->  'Orbitdiagram for var. c' 
            **  From 'Orbitdiagram for var. c' ->  'Step-by-step iteration'
            */

            if(((RealExp->Graph == GR_R_ORBITS_C) && (OldGraph == GR_R_STEP)) ||
               ((RealExp->Graph == GR_R_STEP) && (OldGraph == GR_R_ORBITS_C))) 
            {
                RealExp->GraphPar[RealExp->Graph].Coords->YMin =
                    RealExp->GraphPar[OldGraph].Coords->YMin;
                RealExp->GraphPar[RealExp->Graph].Coords->YMax =
                    RealExp->GraphPar[OldGraph].Coords->YMax;
            }

            /* 
            **  From 'Orbitdiagram for var. c'  ->  'Orbitdiagram for var. x' 
            **  From 'Orbitdiagram for var. x' ->  'Orbitdiagram for var. c'
            */

            if(((RealExp->Graph == GR_R_ORBITS_C) && 
                (OldGraph == GR_R_ORBITS_X)) ||
               ((RealExp->Graph == GR_R_ORBITS_X) && 
                (OldGraph == GR_R_ORBITS_C))) 
            {
                RealExp->GraphPar[RealExp->Graph].Coords->YMin =
                    RealExp->GraphPar[OldGraph].Coords->YMin;
                RealExp->GraphPar[RealExp->Graph].Coords->YMax =
                    RealExp->GraphPar[OldGraph].Coords->YMax;
            }

        }
    }

    DestroyItemStrings(GCStr, 6);
    DestroyItemStrings(GRStr, 4);
    DestroyMenu(Menu);
    return Key;
}

int ClsMnuDiagram(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    TMenu       *Menu;
    int         Key, MenuXPos, RMenuWidth, CSMenuWidth, CMenuWidth, 
                RMaxItemLength, CSMaxItemLength, CMaxItemLength, i;
    char        **DRStr = GetItemStrings("RDIAGRAMMENU", &RMenuWidth, 
                &RMaxItemLength, 5),
                **DCSStr = GetItemStrings("CSDIAGRAMMENU", &CSMenuWidth, 
                &CSMaxItemLength, 5),
                **DCStr = GetItemStrings("CDIAGRAMMENU", &CMenuWidth, 
                &CMaxItemLength, 5);

    /* Select menu */

    if(Functiontype == FT_REAL) {
        if(DRStr == NULL) {
            RMenuWidth = 40;
            Menu = CreateMenu("TYPE OF DIAGRAM - MENU:", 20, 10,
                RealExp->Diagram, 4, RMenuWidth, 0, LIGHTGRAY, BLACK, BLACK,
                LIGHTGRAY, 4, (1 << M_HEADER) + (1 << M_STATUSBAR), 
                RealDiagramItems);
        }
        else {
            MenuXPos = (80 - RMenuWidth) / 2;
            Menu = CreateMenu(DRStr[0], MenuXPos, 10, RealExp->Diagram, 4,
                RMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
                (1 << M_HEADER) + (1 << M_STATUSBAR), RealDiagramItems);
            for(i = 1; i <= 4; i++) 
                SwapItemCaptions(&Menu->Items[i - 1]->Caption, &DRStr[i]);
        }
    }
    else {
        if((ComplexExp->Graph == GR_C_STEP) ||
            (ComplexExp->Graph == GR_C_STEPINV)) 
        {
            if(DCSStr == NULL) {
                CSMenuWidth = 11;
                Menu = CreateMenu(MenuTitleStrings[3], 34, 10,
                    ComplexExp->Diagram, 4, CSMenuWidth, 0, LIGHTGRAY, BLACK,
                    BLACK, LIGHTGRAY, 4, 
                    (1 << M_HEADER) + (1 << M_STATUSBAR), 
                    ComplexStepShapeItems);
            }
            else {
                MenuXPos = (80 - CSMenuWidth) / 2;
                Menu = CreateMenu(DCSStr[0], MenuXPos, 10, ComplexExp->Diagram, 
                    4, CSMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
                    (1 << M_HEADER) + (1 << M_STATUSBAR), 
                    ComplexStepShapeItems);
                for(i = 1; i <= 4; i++)
                    SwapItemCaptions(&Menu->Items[i - 1]->Caption, &DCSStr[i]);
            }
        }
        else {
            if(DCStr == NULL) {
                CMenuWidth = 11;
                Menu = CreateMenu(MenuTitleStrings[2], 26, 10,
                    ComplexExp->Diagram, 4, CMenuWidth, 0, LIGHTGRAY, BLACK,
                    BLACK, LIGHTGRAY, 4, (1 << M_HEADER) + (1 << M_STATUSBAR),
                    ComplexDiagramItems);
            }
            else {
                MenuXPos = (80 - CMenuWidth) / 2;
                Menu = CreateMenu(DCStr[0], MenuXPos, 10, ComplexExp->Diagram, 
                    4, CMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
                    (1 << M_HEADER) + (1 << M_STATUSBAR), ComplexDiagramItems);
                for(i = 1; i <= 4; i++) 
                    SwapItemCaptions(&Menu->Items[i - 1]->Caption, &DCStr[i]);
            }
            
            /* 
            **  For 'Orbit diagram for variable c', the last diagram is not 
            **  available, and for functions other than z -> z ^ n + c it 
            **  doesn't make sense either (in this case).
            */

            if((ComplexExp->Graph == GR_C_ORBITS_Z) && 
                ((ComplexExp->Function >= 0) && (ComplexExp->Function <= 3)))
            {
                Menu->Items[3]->Flag = 96;
            }
            else
            {
                Menu->Items[3]->Flag = 0;
            }
        }
    }
    
    a_clrscr(Console);
    DrawMenu(Console, Menu);

    while(TRUE) {
        Key = DoMenu(Console, Menu);

        if(Key == KEY_ESC) 
            break;
        
        if(Key == KEY_ENTER) {
            if((Functiontype == FT_COMPLEX) && 
                (ComplexExp->Graph == GR_C_STEP || 
                ComplexExp->Graph == GR_C_STEPINV)) 
            { 
                if(ClsSubDiagram(Console, ComplexExp) == KEY_ESC) {
                    DrawMenu(Console, Menu);
                    continue;
                }
            }
            break;
        } 

    }

    /* Process data */

    if(Key == KEY_ENTER) {
        if(Functiontype == FT_REAL) {
            RealExp->Diagram = Menu->ItemIndex;
            if(RealExp->Diagram != Menu->ItemIndex) 
                RealExp->Changed = TRUE;
        }
        else
            ComplexExp->Diagram = Menu->ItemIndex;
    }
    
    DestroyItemStrings(DCStr, 5);
    DestroyItemStrings(DCSStr, 5);
    DestroyItemStrings(DRStr, 5);
    DestroyMenu(Menu);
    return Key;
}

int ClsSubDiagram(TConsole *Console, TComplexExp *ComplexExp)
{
    TMenu       *Menu;
    int         Key, MenuXPos, CSCMenuWidth, CSCMaxItemLength, i = 0,
                DefaultItemIndex = 0;
    char        **CSCStr = GetItemStrings("CSTEPCOLORMENU",
                &CSCMenuWidth, &CSCMaxItemLength, 4);
    
    switch(ComplexExp->Coloring) {
        case CM_START: 
            DefaultItemIndex = 0; 
            break;
        case CM_ITERSTEP: 
            DefaultItemIndex = 1; 
            break;
        case CM_SAME: 
            DefaultItemIndex = 2; 
            break;
    }

    if(CSCStr == NULL) {
        Menu = CreateMenu("SELECT A COLORING-METHOD:", 24, 10,
        DefaultItemIndex, 3, 31, 0, 
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 3, (1 << M_HEADER) + 
            (1 << M_STATUSBAR), ComplexStepColorItems);
    }
    else {
        MenuXPos = (80 - CSCMenuWidth) / 2;
        Menu = CreateMenu(CSCStr[0], MenuXPos, 10, DefaultItemIndex, 
            3, CSCMaxItemLength, 0, 
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 3, (1 << M_HEADER) + 
            (1 << M_STATUSBAR), ComplexStepColorItems);
        for(i = 1; i <= 3; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &CSCStr[i]);
    }

    DrawMenu(Console, Menu);

    while(TRUE) {
        Key = DoMenu(Console, Menu);
        if((Key == KEY_ESC) || (Key == KEY_ENTER)) 
            break;
    }
    
    switch(Menu->ItemIndex) {
        case 0: 
            ComplexExp->Coloring = CM_START; 
            break;
        case 1: 
            ComplexExp->Coloring = CM_ITERSTEP; 
            break;
        case 2: 
            ComplexExp->Coloring = CM_SAME; 
            break;
    }

    DestroyItemStrings(CSCStr, 4);    
    DestroyMenu(Menu);
    return Key;

}

int ClsMnuFunction(TConsole *Console, TRealExp *RealExp,
    TComplexExp *ComplexExp)
{
    TMenu       *Menu;
    int         Res, Key, MenuXPos, RMenuWidth, RMaxItemLength, 
                CMenuWidth, CMaxItemLength,
                MTMenuWidth, MTMaxItemLength, MEMenuWidth, MEMaxItemLength, i;
    char        **FRStr = GetItemStrings("RFUNCTIONMENU", &RMenuWidth, 
                &RMaxItemLength, 8),
                **FCStr = GetItemStrings("CFUNCTIONMENU", &CMenuWidth,
                &CMaxItemLength, 5),
                **MTStr = GetItemStrings("MISCTEXT", &MTMenuWidth, 
                &MTMaxItemLength, 6), 
                **MEStr = GetItemStrings("MESSAGES", &MEMenuWidth,
                &MEMaxItemLength, 10),
                /*
                *MsgStr[] = {   "This experiment has not been saved yet",
                                "Do you want to save it now?\n",
                                " ",
                                "Yes", "No"
                },
                */
                FilePath[2048] = "default.par";

    clear_keybuf();

    if(Functiontype == FT_REAL) {
        if(FRStr == NULL)
            Menu = CreateMenu(MenuTitleStrings[4], 29, 4, RealExp->Function,
                7, 22, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 7,
                (1 << M_HEADER) + (1 << M_STATUSBAR), RealFnItems);
        else {
            MenuXPos = (80 - RMenuWidth) / 2;
            Menu = CreateMenu(FRStr[0], MenuXPos, 4, RealExp->Function,
                7, RMaxItemLength, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 7, 
                (1 << M_HEADER) + (1 << M_STATUSBAR), RealFnItems);
            for(i = 1; i <= 7; i++)
                SwapItemCaptions(&Menu->Items[i - 1]->Caption, &FRStr[i]);
        }
    }
    else {
        if(FCStr == NULL)
            Menu = CreateMenu(MenuTitleStrings[4], 32, 4, ComplexExp->Function,
                4, 16, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4, 
                (1 << M_HEADER) + (1 << M_STATUSBAR), ComplexFnItems);
        else {
            MenuXPos = (80 - CMenuWidth) / 2;
            Menu = CreateMenu(FCStr[0], 32, 4, ComplexExp->Function,
                4, CMaxItemLength, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
                (1 << M_HEADER) + (1 << M_STATUSBAR), ComplexFnItems);
            for(i = 1; i <= 4; i++)
                SwapItemCaptions(&Menu->Items[i - 1]->Caption, &FCStr[i]);
        }
    }
    
    a_clrscr(Console);
    DrawMenu(Console, Menu);

    while(TRUE) {
        Key = DoMenu(Console, Menu);
        if(Functiontype == FT_REAL && Menu->ItemIndex == 6 && Key == KEY_ENTER) 
            UserDefFnMenu(Console, RealExp, ComplexExp);
        if(Functiontype == FT_COMPLEX && Menu->ItemIndex == 3 && Key == KEY_ENTER) 
            UserDefFnMenu(Console, RealExp, ComplexExp);
        if((Key == KEY_ESC) || (Key == KEY_ENTER)) 
            break;
    }

    if(Key == KEY_ENTER) {
        if(Functiontype == FT_REAL) {
            if((Menu->ItemIndex != RealExp->Function) && RealExp->Changed) {
                if(Res == 1) {
                    if(FILE_SELECT_BOX((MTStr == NULL) ? MiscTexts[5] : 
                        MTStr[5], FilePath, "PAR", 2047, 300, 400) != 0) 
                    {
                        scare_mouse();
                        SaveRealExp(RealExp, FilePath);
                        unscare_mouse();
                    }                    
                }
            }
            RealExp->Function = Menu->ItemIndex;
            SelectExperiment(RealExp, TRUE, Menu->ItemIndex);    
        }
        else {
            ComplexExp->Function = Menu->ItemIndex;
            SelectComplexExperiment(ComplexExp, TRUE, Menu->ItemIndex);
            
            /* 
            **  Inverse isn't available for functions other than z -> z ^ n + c
            */
            
            if(ComplexExp->Function > 2) {
                if(ComplexExp->Graph == GR_C_STEPINV)
                    ComplexExp->Graph = GR_C_STEP;
                if(ComplexExp->Diagram == DM_INVERSE)
                    ComplexExp->Diagram = DM_BOUNDARY;
                /* Notification that diagram has been modified by Funiter */
            }
        }
    }

    DestroyItemStrings(MEStr, 10);    
    DestroyItemStrings(MTStr, 6);    
    DestroyItemStrings(FCStr, 5);    
    DestroyItemStrings(FRStr, 8);    
    DestroyMenu(Menu);
    return Key;
}

void UserDefFnMenu(TConsole *Console, TRealExp *RealExp,
    TComplexExp *ComplexExp)
{
    TMenu       *Menu;
    int         OldX, OldY, ErrNo = 0, ErrPos = 0, Key, MenuXPos, FHMenuWidth, 
                FEMenuWidth, xtmp, ytmp,
                FHMaxItemLength, FEMaxItemLength;
    char        **FHStr = GetItemStrings("RFHELPMENU", &FHMenuWidth, 
                &FHMaxItemLength, 16),
                **CFHStr = GetItemStrings("CFHELPMENU", &FHMenuWidth, 
                &FHMaxItemLength, 4),
                **FEStr = GetItemStrings("RFERRORS", &FEMenuWidth,
                &FEMaxItemLength, 8);

    
    if(FHStr == 0)
        MenuXPos = 5; 
    else 
        MenuXPos = (80 - FHMenuWidth) / 2;

    if(Functiontype == FT_REAL) 
        Menu = CreateMenu("", 15, 21, 0, 1, 1, 0, 
            LIGHTGRAY, BLACK, LIGHTGRAY, BLACK, 1, (1 << M_STATUSBAR), 
            &UserDefFnItems[0]);
    else
        Menu = CreateMenu("", 15, 21, 0, 1, 1, 0, 
            LIGHTGRAY, BLACK, LIGHTGRAY, BLACK, 1, (1 << M_STATUSBAR), 
            &UserDefFnItems[1]);

    a_clrscr(Console);
    
    if(Functiontype == FT_REAL)
        strcpy(Menu->Items[0]->Contents, RealExp->FunctionStr);
    else
        strcpy(Menu->Items[0]->Contents, ComplexExp->FunctionStr);
    
    _a_setcursortype(Console, _NORMALCURSOR);
    DrawMenu(Console, Menu);
    
    xtmp = a_wherex(Console);
    ytmp = a_wherey(Console);

    if(Functiontype == FT_REAL) {
        a_gotoxy(Console, MenuXPos, 3);    
        a_cprintf(Console, "%s", (FHStr == NULL) ? FnHelpText[0] : FHStr[0]);
        a_gotoxy(Console, MenuXPos, 5);    
        a_cprintf(Console, "%s", (FHStr == NULL) ? FnHelpText[1] : FHStr[1]);
        a_gotoxy(Console, MenuXPos, 6);
        a_cprintf(Console, "%s", (FHStr == NULL) ? FnHelpText[2] : FHStr[2]);
        a_gotoxy(Console, MenuXPos, 7);    
        a_cprintf(Console, "%s", (FHStr == NULL) ? FnHelpText[3] : FHStr[3]);
    }
    else {
        a_gotoxy(Console, MenuXPos, 3);    
        a_cprintf(Console, "%s", (FHStr == NULL) ? CFnHelpText[0] : CFHStr[0]);
        a_gotoxy(Console, MenuXPos, 5);    
        a_cprintf(Console, "%s", (FHStr == NULL) ? CFnHelpText[1] : CFHStr[1]);
        a_gotoxy(Console, MenuXPos, 6);
        a_cprintf(Console, "%s", (FHStr == NULL) ? CFnHelpText[2] : CFHStr[2]);
        a_gotoxy(Console, MenuXPos, 7);    
        a_cprintf(Console, "%s", (FHStr == NULL) ? CFnHelpText[3] : CFHStr[3]);
    }
    
    a_gotoxy(Console, MenuXPos + 5, 10);
    a_cprintf(Console, "+ %s - %s * %s / %s ^ %s", 
        (FHStr == NULL) ? FnHelpText[4] : FHStr[4],
        (FHStr == NULL) ? FnHelpText[5] : FHStr[5],
        (FHStr == NULL) ? FnHelpText[6] : FHStr[6],
        (FHStr == NULL) ? FnHelpText[7] : FHStr[7],
        (FHStr == NULL) ? FnHelpText[8] : FHStr[8]);

    a_gotoxy(Console, MenuXPos + 5, 12);    
    if(Functiontype == FT_REAL) {
        a_cprintf(Console, "sin  cos  tan  asn (%s)  atn (%s)",
            (FHStr == NULL) ? FnHelpText[9] : FHStr[9],
            (FHStr == NULL) ? FnHelpText[10] : FHStr[10]);
    }
    else
        a_cprintf(Console, "sin  cos");
    
    if(Functiontype == FT_REAL) {
        a_gotoxy(Console, MenuXPos + 5, 13);    
        a_cprintf(Console, "exp  log (%s)  sqr (%s)",
            (FHStr == NULL) ? FnHelpText[11] : FHStr[11],
            (FHStr == NULL) ? FnHelpText[12] : FHStr[12]);

        a_gotoxy(Console, MenuXPos + 5, 15);    
        a_cprintf(Console, "abs (%s)  entier  frac, %s:",
            (FHStr == NULL) ? FnHelpText[13] : FHStr[13],
            (FHStr == NULL) ? FnHelpText[14] : FHStr[14]);

        a_gotoxy(Console, MenuXPos + 5, 17);    
        a_cprintf(Console, "entier(x) = [x] := %s",
            (FHStr == NULL) ? FnHelpText[15] : FHStr[15]);

        a_gotoxy(Console, MenuXPos + 5, 18);
        a_cprintf(Console, "frac(x) := x - [x]");
    }
    
    DrawItemContents(Console, Menu, Menu->Items[0]);
        
    a_gotoxy(Console, xtmp, ytmp);
    
    while(TRUE) {
        Key = DoMenu(Console, Menu);
        if((Key == KEY_ESC) || (Key == KEY_ENTER)) {

            if(Functiontype == FT_REAL) 
                Parse(&ErrNo, &ErrPos, Menu->Items[0]->Contents, 2,
                    'x', RealExp->GraphPar[RealExp->Graph].xbegin1, 
                    'c', RealExp->GraphPar[RealExp->Graph].c);
            else
                CParse(&ErrNo, &ErrPos, Menu->Items[0]->Contents,2,
                    'z', 
                    Complex(ComplexExp->GraphPar[ComplexExp->Graph].zReal,
                            ComplexExp->GraphPar[ComplexExp->Graph].zImag),
                    'c', 
                    Complex(ComplexExp->GraphPar[ComplexExp->Graph].cReal,
                            ComplexExp->GraphPar[ComplexExp->Graph].cImag));
            
            if(ErrNo) {
                OldX = a_wherex(Console); 
                OldY = a_wherey(Console);
                
                a_gotoxy(Console, MenuXPos + 10, 22);
                a_cprintf(Console, "%s", (FEStr == NULL) ? FnErrText[0] : 
                    FEStr[0]);
                a_gotoxy(Console, MenuXPos + 12, 23);
                a_cprintf(Console, "%s [%d]", (FEStr == NULL) ? FnErrText[ErrNo]
                    : FEStr[ErrNo], ErrPos);

                a_gotoxy(Console, OldX, OldY);
                continue;
            }
            else {
                if(Functiontype == FT_REAL) 
                    strcpy(RealExp->FunctionStr, Menu->Items[0]->Contents);
                else
                    strcpy(ComplexExp->FunctionStr, Menu->Items[0]->Contents);
                
                break;
            }
        }
    }

    _a_setcursortype(Console, _NOCURSOR);
    
    DestroyItemStrings(FEStr, 8);    
    DestroyItemStrings(CFHStr, 4);
    DestroyItemStrings(FHStr, 16);    
    DestroyMenu(Menu);
    return;
}

TMenu *SelectRealParItems(TMenu *Menu, TRealExp *RealExp)
{
    TCoords     RCoords;
    TRGraphPar  RGraphPar;

    RCoords = *(RealExp->GraphPar[RealExp->Graph].Coords);
    RGraphPar = RealExp->GraphPar[RealExp->Graph];

    switch(RealExp->Graph) {
        case GR_R_STEP:

            /* Y-Coordinate (lowest)    */

            RESBIT(Menu->Items[5]->Flag, I_VISIBLE);

            /* Y-Coordinate (highest)   */

            RESBIT(Menu->Items[6]->Flag, I_VISIBLE);

            /* Total number of iter...  */

            RESBIT(Menu->Items[9]->Flag, I_VISIBLE);

            strcpy(Menu->Items[10]->Caption, Menu->Items[13]->Caption);

            break;
        case GR_R_ORBITS_X:
          
            /* 1st starting value...    */

            RESBIT(Menu->Items[1]->Flag, I_VISIBLE);

            /* 2nd starting value...    */

            RESBIT(Menu->Items[2]->Flag, I_VISIBLE);

            strcpy(Menu->Items[3]->Caption, Menu->Items[14]->Caption);
            strcpy(Menu->Items[4]->Caption, Menu->Items[15]->Caption);
               
            /* Number of steps/iter...  */

            RESBIT(Menu->Items[8]->Flag, I_VISIBLE);
            break;
        case GR_R_ORBITS_C:

            /* Value for parameter c    */

            RESBIT(Menu->Items[0]->Flag, I_VISIBLE);

            /* 2nd starting value...    */

            RESBIT(Menu->Items[2]->Flag, I_VISIBLE);

            strcpy(Menu->Items[3]->Caption, Menu->Items[11]->Caption);
            strcpy(Menu->Items[4]->Caption, Menu->Items[12]->Caption);
               
            /* Number of steps/iter...  */

            RESBIT(Menu->Items[8]->Flag, I_VISIBLE);
            break;
    }

    /* Copy values to the input strings */

    sprintf(Menu->Items[ 0]->Contents, "%.*g", N_DBL_DIG, RGraphPar.c);
    sprintf(Menu->Items[ 1]->Contents, "%.*g", N_DBL_DIG, RGraphPar.xbegin1);
    sprintf(Menu->Items[ 2]->Contents, "%.*g", N_DBL_DIG, RGraphPar.xbegin2);
    sprintf(Menu->Items[ 3]->Contents, "%.*g", N_DBL_DIG, RCoords.XMin);
    sprintf(Menu->Items[ 4]->Contents, "%.*g", N_DBL_DIG, RCoords.XMax);
    sprintf(Menu->Items[ 5]->Contents, "%.*g", N_DBL_DIG, RCoords.YMin);
    sprintf(Menu->Items[ 6]->Contents, "%.*g", N_DBL_DIG, RCoords.YMax);
    sprintf(Menu->Items[ 7]->Contents, "%.*g", N_DBL_DIG, RGraphPar.EscapeValue);
    sprintf(Menu->Items[ 8]->Contents, "%ld", RGraphPar.IterSteps);
    sprintf(Menu->Items[ 9]->Contents, "%ld", RGraphPar.IterMax);
    if(RealExp->Graph == GR_R_STEP)
        sprintf(Menu->Items[10]->Contents, "%d", RGraphPar.Iterated);
    else
        sprintf(Menu->Items[10]->Contents, "%ld", RGraphPar.IterSkip);
    
    return(Menu);
}

TMenu *SelectComplexParItems(TMenu *Menu, TComplexExp *ComplexExp)
{
    TCoords     CCoords;
    TCGraphPar  CGraphPar;

    CCoords = *(ComplexExp->GraphPar[ComplexExp->Graph].Coords);
    CGraphPar = ComplexExp->GraphPar[ComplexExp->Graph];

    RESBIT(Menu->Items[0]->Flag, I_VISIBLE);

    strcpy(Menu->Items[1]->Caption, Menu->Items[18]->Caption);
    strcpy(Menu->Items[2]->Caption, Menu->Items[19]->Caption);
    strcpy(Menu->Items[3]->Caption, Menu->Items[20]->Caption);
    strcpy(Menu->Items[4]->Caption, Menu->Items[21]->Caption);
    strcpy(Menu->Items[5]->Caption, Menu->Items[22]->Caption);
    strcpy(Menu->Items[6]->Caption, Menu->Items[23]->Caption);

    switch(ComplexExp->Graph) {
        case GR_C_STEP:
        case GR_C_STEPINV:
            RESBIT(Menu->Items[9]->Flag, I_VISIBLE);
            break;
        case GR_C_ORBITS_Z:
            RESBIT(Menu->Items[8]->Flag, I_VISIBLE);
            break;
        case GR_C_ORBITS_C:
            SwapItemCaptions(&Menu->Items[1]->Caption,
                &Menu->Items[24]->Caption);
            SwapItemCaptions(&Menu->Items[2]->Caption,
               &Menu->Items[25]->Caption);
            RESBIT(Menu->Items[8]->Flag, I_VISIBLE);
            break;
    }

    /* Copy values to the input strings */

    if(ComplexExp->Graph == GR_C_ORBITS_C) {
        sprintf(Menu->Items[ 1]->Contents, "%.*g", N_DBL_DIG, CGraphPar.zReal);
        sprintf(Menu->Items[ 2]->Contents, "%.*g", N_DBL_DIG, CGraphPar.zImag);
    }
    else {
        sprintf(Menu->Items[ 1]->Contents, "%.*g", N_DBL_DIG, CGraphPar.cReal);
        sprintf(Menu->Items[ 2]->Contents, "%.*g", N_DBL_DIG, CGraphPar.cImag);
    }
    sprintf(Menu->Items[ 3]->Contents, "%.*g", N_DBL_DIG, CCoords.XMin);
    sprintf(Menu->Items[ 4]->Contents, "%.*g", N_DBL_DIG, CCoords.XMax);
    sprintf(Menu->Items[ 5]->Contents, "%.*g", N_DBL_DIG, CCoords.YMin);
    sprintf(Menu->Items[ 6]->Contents, "%.*g", N_DBL_DIG, CCoords.YMax);
    sprintf(Menu->Items[ 7]->Contents, "%.*g", N_DBL_DIG, CGraphPar.EscapeValue);
    sprintf(Menu->Items[ 8]->Contents, "%d", CGraphPar.IterSteps);
    if((ComplexExp->Graph == GR_C_ORBITS_Z) && 
        (ComplexExp->Diagram == DM_INVERSE))
    {
        sprintf(Menu->Items[ 9]->Contents, "%d", ComplexExp->InvJuliaIterMax);
    }
    else
        sprintf(Menu->Items[ 9]->Contents, "%d", CGraphPar.IterMax);
    sprintf(Menu->Items[10]->Contents, "%d", CGraphPar.IterSkip);

    return(Menu);
}

int ClsMnuPar(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    TCoords     *RCoords;
    TRGraphPar  RGraphPar;
    TCoords     *CCoords;
    TCGraphPar  CGraphPar;
    TMenu       *Menu;
    char        *n, *m;
    int         i, tmp_x, tmp_y, Key, MenuXPos, MenuYPos, MenuWidth, 
                MaxItemLength, MEMenuWidth, MEMaxItemLength, 
                RGraph = RealExp->Graph, CGraph = ComplexExp->Graph;
    char        **PRStr = GetItemStrings("PARAMETERMENU", &MenuWidth, 
                &MaxItemLength, 27),
                **MEStr = GetItemStrings("MESSAGES", &MEMenuWidth,
                &MEMaxItemLength, 20);
                
    /* Create shortcuts for coordinates and graph parameters */

    RCoords = RealExp->GraphPar[RealExp->Graph].Coords;
    memcpy(&RGraphPar, &RealExp->GraphPar[RealExp->Graph], sizeof(TRGraphPar));
    RGraphPar.Coords = CreateCoords(
        RCoords->XMin, RCoords->YMin, 
        RCoords->XMax, RCoords->YMax, 
        RCoords->XStart, RCoords->YStart, 
        RCoords->XEnd, RCoords->YEnd, 
        RCoords->Mode);

    CCoords = ComplexExp->GraphPar[ComplexExp->Graph].Coords;
    memcpy(&CGraphPar, &ComplexExp->GraphPar[ComplexExp->Graph], 
        sizeof(TCGraphPar));
    CGraphPar.Coords = CreateCoords(
        CCoords->XMin, CCoords->YMin, 
        CCoords->XMax, CCoords->YMax, 
        CCoords->XStart, CCoords->YStart, 
        CCoords->XEnd, CCoords->YEnd, 
        CCoords->Mode);

    /* Recalculate escape value in case the user didn't change it */
   
    if(!RealExp->GraphPar[RGraph].EscapeValueChanged && 
    Functiontype == FT_REAL) {
        RGraphPar.EscapeValue = CalculateRealFnEscapeValue(&RGraphPar,
            RealExp->Function, RealExp->Graph);
        RealExp->GraphPar[RealExp->Graph].EscapeValue = RGraphPar.EscapeValue;
    }
    if(!ComplexExp->GraphPar[CGraph].EscapeValueChanged &&
    Functiontype == FT_COMPLEX) {
        CGraphPar.EscapeValue = CalculateComplexFnEscapeValue(&CGraphPar, 
            ComplexExp->Function, ComplexExp->Graph);
        ComplexExp->GraphPar[ComplexExp->Graph].EscapeValue = 
            CGraphPar.EscapeValue;
    }
    
    /* Create menu */
    
    MenuXPos = (80 - (MenuWidth + 12)) / 2;
    MenuYPos = (Console->CharHeight - 12) / 2;
    if(PRStr == NULL)
        Menu = CreateMenu(MenuTitleStrings[5], 16, MenuYPos, 0, 11, 34, 0,
        LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 26,
        (1 << M_HEADER) + (1 << M_STATUSBAR), ParameterItems);
    else {
        Menu = CreateMenu(PRStr[0], MenuXPos, MenuYPos, 0, 11, 
        MenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 26, 
        (1 << M_HEADER) + (1 << M_STATUSBAR), ParameterItems);
        for(i = 1; i <= 26; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &PRStr[i]);
    }
    
    /* R -> R */
    
    if(Functiontype == FT_REAL) 

        /*
        **  Hide fields that are not required/make no sense for the selected
        **  graph, and change names for some fields.
        */

        Menu = SelectRealParItems(Menu, RealExp);
    
    else 

        /*
        **  Hide fields that are not required/make no sense for the selected
        **  graph, and change names for some fields.
        */

        Menu = SelectComplexParItems(Menu, ComplexExp);

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    /* Print available keys */

    tmp_x = a_wherex(Console); tmp_y = a_wherey(Console);
    a_gotoxy(Console, 1, 1);
    if(MEStr != NULL) 
        a_cputs(Console, MEStr[16]);
    else
        a_cputs(Console, "F3 = Set standard coordinates.");
    a_gotoxy(Console, 1, 2);
    if(MEStr != NULL) 
        a_cputs(Console, MEStr[17]);
    else
        a_cputs(Console, "F5 = Set standard escapevalue.");
    
    a_textcolor(Console, GREEN);
    a_gotoxy(Console, 1, 4);
    if(MEStr != NULL) {
        n = strchr(MEStr[18], '*');
        m = strchr(MEStr[19], '*');
        a_cputs(Console, MEStr[18]);
        a_gotoxy(Console, 1, 5);
        a_cputs(Console, MEStr[19]);
        
        a_textcolor(Console, RED);
        if(n != NULL) {
             a_gotoxy(Console, 1 + (int) (n - MEStr[18]), 4); 
             a_cputs(Console, "*");
        }
        if(m != NULL) {
             a_gotoxy(Console, 1 + (int) (m - MEStr[19]), 5); 
             a_cputs(Console, "*");
        }
        a_textcolor(Console, RED);
    }
    else {
        a_cputs(Console, "A ");
        a_textcolor(Console, RED);
        a_cputs(Console, "* ");
        a_textcolor(Console, GREEN);
        a_cputs(Console, "in front of the escapevalue line indicates the " 
            "escapevalue is your own");
        a_gotoxy(Console, 1, 5);
        a_cputs(Console, "responsibility now.");
    }
    a_gotoxy(Console, tmp_x, tmp_y);
    a_textcolor(Console, LIGHTGRAY);
    
    Console->CursorMode = _SOLIDCURSOR;

    while(TRUE) {
        
        tmp_x = a_wherex(Console); 
        tmp_y = a_wherey(Console);
            
        a_gotoxy(Console, (PRStr == NULL) ? 14 : (MenuXPos - 2), 18); 

        if(Functiontype == FT_REAL) {
            if(RGraphPar.EscapeValueChanged || RealExp->Function == 6) {
                a_textcolor(Console, RED); 
                a_cputs(Console, "*");
            }
            else {
                a_cputs(Console, " ");
            }
        }
        else if(Functiontype == FT_COMPLEX) {
            if(CGraphPar.EscapeValueChanged) {
                a_textcolor(Console, RED); 
                a_cputs(Console, "*");
            }
            else {
                a_cputs(Console, " ");
            }
        }        
        a_gotoxy(Console, tmp_x, tmp_y);
        a_textcolor(Console, LIGHTGRAY);
                
        Key = DoMenu(Console, Menu);

        /* Set coordinates to default values */
        
        if(Key == KEY_F3) {
            if(Functiontype == FT_REAL) {
                SelectRealStdCoords(RGraphPar.Coords, 
                    RealExp->Function, RGraph);
                sprintf(Menu->Items[ 3]->Contents, "%.*g", N_DBL_DIG,
                    RGraphPar.Coords->XMin);
                sprintf(Menu->Items[ 4]->Contents, "%.*g", N_DBL_DIG,
                    RGraphPar.Coords->XMax);
                sprintf(Menu->Items[ 5]->Contents, "%.*g", N_DBL_DIG,
                    RGraphPar.Coords->YMin);
                sprintf(Menu->Items[ 6]->Contents, "%.*g", N_DBL_DIG,
                    RGraphPar.Coords->YMax);
                Menu->DefaultItemIndex = 0;

                if(!RGraphPar.EscapeValueChanged) {
                    sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                        CalculateRealFnEscapeValue(&RGraphPar,
                        RealExp->Function, RealExp->Graph));
                }
            }
            else if(Functiontype == FT_COMPLEX) {
                SetComplexExpDefaultCoords(CGraphPar.Coords, 
                ComplexExp->Function, CGraph);
                sprintf(Menu->Items[ 3]->Contents, "%.*g", N_DBL_DIG,
                    CGraphPar.Coords->XMin);
                sprintf(Menu->Items[ 4]->Contents, "%.*g", N_DBL_DIG,
                    CGraphPar.Coords->XMax);
                sprintf(Menu->Items[ 5]->Contents, "%.*g", N_DBL_DIG,
                    CGraphPar.Coords->YMin);
                sprintf(Menu->Items[ 6]->Contents, "%.*g", N_DBL_DIG,
                    CGraphPar.Coords->YMax);
                Menu->DefaultItemIndex = 0;

                if(!ComplexExp->GraphPar[CGraph].EscapeValueChanged) {
                    sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                        CalculateComplexFnEscapeValue(&CGraphPar, 
                        ComplexExp->Function, ComplexExp->Graph));
                }
            }
            tmp_x = a_wherex(Console); 
            tmp_y = a_wherey(Console);
            DrawItem(Console, Menu, Menu->Items[3]);
            DrawItem(Console, Menu, Menu->Items[4]);
            DrawItem(Console, Menu, Menu->Items[5]);
            DrawItem(Console, Menu, Menu->Items[6]);
            DrawItem(Console, Menu, Menu->Items[7]);
            clear_keybuf();
            a_gotoxy(Console, tmp_x, tmp_y);
            continue;
        }
        
        /* Recalculate escape value */
        
        if(Key == KEY_F5) {
            if(Functiontype == FT_REAL) {
                RGraphPar.EscapeValueChanged = FALSE;
                RGraphPar.EscapeValue = 
                    CalculateRealFnEscapeValue(&RGraphPar,
                    RealExp->Function, RealExp->Graph);
                sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                    RGraphPar.EscapeValue);
                Menu->DefaultItemIndex = 0;
            }   
            else if(Functiontype == FT_COMPLEX) {
                CGraphPar.EscapeValueChanged = 
                    CGraphPar.EscapeValueChanged = FALSE;                
                CGraphPar.EscapeValue = 
                    CalculateComplexFnEscapeValue(&CGraphPar, 
                    ComplexExp->Function, ComplexExp->Graph);
                sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                    CGraphPar.EscapeValue);            
                Menu->DefaultItemIndex = 0;
            }
            tmp_x = a_wherex(Console); 
            tmp_y = a_wherey(Console);
            DrawItem(Console, Menu, Menu->Items[7]);
            clear_keybuf();
            a_gotoxy(Console, tmp_x, tmp_y);
            continue;
        }

        if(Functiontype == FT_REAL) {
            switch(Menu->ItemIndex) {
                case 0:
                    if(RGraph == GR_R_STEP || RGraph ==  GR_R_ORBITS_X) {
                        if(!RGraphPar.EscapeValueChanged) {
                            sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                                CalculateRealFnEscapeValue(&RGraphPar,
                                RealExp->Function, RealExp->Graph));
                        }
                    }
                    break;
                case 1:
                    if(RGraph == GR_R_STEP) {
                        if(TSTBIT(Menu->Items[1]->Flag, I_CHANGED)) {
                            strcpy(Menu->Items[2]->Contents,
                                Menu->Items[1]->Contents);
                        }
                    }
                    else if(RGraph == GR_R_ORBITS_C) {
                        if(!RGraphPar.EscapeValueChanged) {
                            sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                                CalculateRealFnEscapeValue(&RGraphPar,
                                RealExp->Function, RealExp->Graph));
                        }
                    }
                    break;
                case 3:
                    if(RGraph == GR_R_STEP) {
                        strcpy(Menu->Items[5]->Contents,
                            Menu->Items[3]->Contents);
                        if(!RGraphPar.EscapeValueChanged) {
                            sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                                CalculateRealFnEscapeValue(&RGraphPar,
                                RealExp->Function, RealExp->Graph));
                        }
                    }
                    break;
                case 4:
                    if(RGraph == GR_R_STEP) {
                        strcpy(Menu->Items[6]->Contents,
                        Menu->Items[4]->Contents);
                        if(!RGraphPar.EscapeValueChanged) {
                            sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                                CalculateRealFnEscapeValue(&RGraphPar,
                                RealExp->Function, RealExp->Graph));
                        }
                    }
                    break;
                case 5:
                case 6:
                    if(!RGraphPar.EscapeValueChanged) {
                        sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                            CalculateRealFnEscapeValue(&RGraphPar,
                            RealExp->Function, RealExp->Graph));
                    }
                    break;
                case 7:

                    /*
                    **  If the calculated value isn't equal to that which the
                    **  user entered, the editfield/text is painted red and
                    **  the 'changed' flag is set to TRUE (user changed value).
                    */

                    if(TSTBIT(Menu->Items[7]->Flag, I_CHANGED)) {

                        /* Action which indicates user has changed value */

                        RGraphPar.EscapeValueChanged = TRUE;
                    }
                    break;    
                default:
                    break;
            }
        }
        else if(Functiontype == FT_COMPLEX) {
            switch(Menu->ItemIndex) {
                case 1:
                case 2:
                case 5:
                case 6:
                    if(!CGraphPar.EscapeValueChanged) {
                        sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                            CalculateComplexFnEscapeValue(&CGraphPar, 
                            ComplexExp->Function, ComplexExp->Graph));
                    }
                    break;
                case 7:

                    /*
                    **  If the calculated value isn't equal to that which the
                    **  user entered, the editfield/text is painted red and
                    **  the 'changed' flag is set to TRUE (user changed value).
                    */

                    if(TSTBIT(Menu->Items[7]->Flag, I_CHANGED)) {

                        /* Action which indicates user has changed value */

                        CGraphPar.EscapeValueChanged = TRUE;
                    }
                    break;
                default:
                    break;
            }
        }

        /* Check for equal values in min/max coordinates */

        if(((Key == KEY_UP) || (Key == KEY_DOWN) || Key == KEY_ESC) && 
            (Menu->ItemIndex <= 7)) 
        {
            if(strcmp(Menu->Items[3]->Contents, Menu->Items[4]->Contents) == 0)
            {
                if((TSTBIT(Menu->Items[3]->Flag, I_LEFT_EDIT)) || 
                    (Key == KEY_ESC))
                {
                    sprintf(Menu->Items[3]->Contents, "%.*g", N_DBL_DIG,
                    (Functiontype == FT_REAL) ? RCoords->XMin : CCoords->XMin);

                    tmp_x = a_wherex(Console);
                    tmp_y = a_wherey(Console);
                    DrawItem(Console, Menu, Menu->Items[3]);
                    DrawItem(Console, Menu, Menu->Items[7]);
                    a_gotoxy(Console, tmp_x, tmp_y);
                }
                if((TSTBIT(Menu->Items[4]->Flag, I_LEFT_EDIT)) ||
                    (Key == KEY_ESC))
                {
                    sprintf(Menu->Items[4]->Contents, "%.*g", N_DBL_DIG,
                    (Functiontype == FT_REAL) ? RCoords->XMax : CCoords->XMax);

                    tmp_x = a_wherex(Console);
                    tmp_y = a_wherey(Console);
                    DrawItem(Console, Menu, Menu->Items[4]);
                    DrawItem(Console, Menu, Menu->Items[7]);
                    a_gotoxy(Console, tmp_x, tmp_y);
                }
            }
            if(strcmp(Menu->Items[5]->Contents, Menu->Items[6]->Contents) == 0)
            {
                if((TSTBIT(Menu->Items[5]->Flag, I_LEFT_EDIT)) ||
                    (Key == KEY_ESC))
                {
                    sprintf(Menu->Items[5]->Contents, "%.*g", N_DBL_DIG,
                    (Functiontype == FT_REAL) ? RCoords->YMin : CCoords->YMin);

                    tmp_x = a_wherex(Console);
                    tmp_y = a_wherey(Console);
                    DrawItem(Console, Menu, Menu->Items[5]);
                    DrawItem(Console, Menu, Menu->Items[7]);
                    a_gotoxy(Console, tmp_x, tmp_y);
                }
                if((TSTBIT(Menu->Items[6]->Flag, I_LEFT_EDIT)) ||
                    (Key == KEY_ESC))
                {
                    sprintf(Menu->Items[6]->Contents, "%.*g", N_DBL_DIG,
                    (Functiontype == FT_REAL) ? RCoords->YMax : CCoords->YMax);

                    tmp_x = a_wherex(Console);
                    tmp_y = a_wherey(Console);
                    DrawItem(Console, Menu, Menu->Items[6]);
                    DrawItem(Console, Menu, Menu->Items[7]);
                    a_gotoxy(Console, tmp_x, tmp_y);
                }
            }
        }

        /* Copy strings into values */

        if(Functiontype == FT_REAL) {
            RGraphPar.c = strtod(Menu->Items[0]->Contents, NULL);
            RGraphPar.xbegin1 = strtod(Menu->Items[1]->Contents, NULL);
            RGraphPar.xbegin2 = strtod(Menu->Items[2]->Contents, NULL);

            SetCoords(RGraphPar.Coords, 
                strtod(Menu->Items[3]->Contents, NULL),
                strtod(Menu->Items[5]->Contents, NULL), 
                strtod(Menu->Items[4]->Contents, NULL),
                strtod(Menu->Items[6]->Contents, NULL));

            RGraphPar.EscapeValue = strtod(Menu->Items[7]->Contents, NULL);
            RGraphPar.IterSteps = atoi(Menu->Items[8]->Contents);
            RGraphPar.IterMax = atoi(Menu->Items[9]->Contents);

            if(RealExp->Graph == GR_R_STEP)
                RGraphPar.Iterated = atoi(Menu->Items[10]->Contents);
            else
                RGraphPar.IterSkip = atoi(Menu->Items[10]->Contents);

            /* Recalculate escapevalue */
            
            if(!RGraphPar.EscapeValueChanged) {
                sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                    CalculateRealFnEscapeValue(&RGraphPar,
                    RealExp->Function, RealExp->Graph));
            }
        }
        if(Functiontype == FT_COMPLEX) {    

            if(ComplexExp->Graph == GR_C_ORBITS_C) {
                CGraphPar.zReal = strtod(Menu->Items[1]->Contents, NULL);
                CGraphPar.zImag = strtod(Menu->Items[2]->Contents, NULL);
            }
            else {
                CGraphPar.cReal = strtod(Menu->Items[1]->Contents, NULL);
                CGraphPar.cImag = strtod(Menu->Items[2]->Contents, NULL);
            }

            SetCoords(CGraphPar.Coords, 
                strtod(Menu->Items[3]->Contents, NULL),
                strtod(Menu->Items[5]->Contents, NULL), 
                strtod(Menu->Items[4]->Contents, NULL),
                strtod(Menu->Items[6]->Contents, NULL));

            CGraphPar.EscapeValue = strtod(Menu->Items[7]->Contents, NULL);
            CGraphPar.IterSteps = atoi(Menu->Items[8]->Contents);

            if((ComplexExp->Graph == GR_C_ORBITS_Z) &&
                (ComplexExp->Diagram == DM_INVERSE))
            {
                ComplexExp->InvJuliaIterMax = atoi(Menu->Items[9]->Contents);
            }
            else
                CGraphPar.IterMax = atoi(Menu->Items[9]->Contents);

            CGraphPar.IterSkip = atoi(Menu->Items[10]->Contents);
            
            /* Recalculate escapevalue */
            
            if(!ComplexExp->GraphPar[CGraph].EscapeValueChanged) {
                sprintf(Menu->Items[7]->Contents, "%.*g", N_DBL_DIG,
                    CalculateComplexFnEscapeValue(&CGraphPar, 
                        ComplexExp->Function, ComplexExp->Graph));
            }
        }

        /* Quit this menu */

        if(Key == KEY_ESC) { 
            Console->CursorMode = _NOCURSOR; 
            clear_keybuf(); 
            break; 
        }
    }
    
    /* Check for changes */
        
    for(i = 0; i <= 10; i++) {
        if(TSTBIT(Menu->Items[i]->Flag, I_CHANGED)) {
            if(Functiontype == FT_REAL)
                RealExp->Changed = TRUE;
            break;
       }
    }

    if(Functiontype == FT_REAL) {
        memcpy(&RealExp->GraphPar[RealExp->Graph], &RGraphPar, 
            sizeof(TRGraphPar));
        memcpy(RCoords, RGraphPar.Coords, sizeof(TCoords));
        RealExp->GraphPar[RealExp->Graph].Coords = RCoords;
    }

    if(Functiontype == FT_COMPLEX) {
        memcpy(&ComplexExp->GraphPar[ComplexExp->Graph], &CGraphPar, 
            sizeof(TCGraphPar));
        memcpy(CCoords, CGraphPar.Coords, sizeof(TCoords));
        ComplexExp->GraphPar[ComplexExp->Graph].Coords = CCoords;
    }

    FreeCoords(RGraphPar.Coords);
    FreeCoords(CGraphPar.Coords);

    Console->CursorMode = _NOCURSOR;

    DestroyItemStrings(MEStr, 20);
    DestroyItemStrings(PRStr, 27);
    DestroyMenu(Menu);
    return 0;
}

int ClsMnuSummary(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    TMenu           *Menu;
    int             Key, AltKey, n, XPos, MenuWidth[11], MaxItemLength[11], 
                    YPos = 3;
    unsigned int    i;
    char            **MStr = GetItemStrings("MAINMENU", &MenuWidth[0], 
                        &MaxItemLength[0], 14),
                    **GRStr = GetItemStrings("RGRAPHMENU", &MenuWidth[1],  
                        &MaxItemLength[1], 4),
                    **GCStr = GetItemStrings("CGRAPHMENU", &MenuWidth[2], 
                        &MaxItemLength[2], 6),
                    **DRStr = GetItemStrings("RDIAGRAMMENU", &MenuWidth[3], 
                        &MaxItemLength[3], 5),
                    **DCSStr = GetItemStrings("CSDIAGRAMMENU", &MenuWidth[4],
                        &MaxItemLength[4], 5),
                    **DCStr = GetItemStrings("CDIAGRAMMENU", &MenuWidth[5],
                        &MaxItemLength[5], 5),
                    **CSCStr = GetItemStrings("CSTEPCOLORMENU", &MenuWidth[6],
                        &MaxItemLength[6], 4),
                    **FRStr = GetItemStrings("RFUNCTIONMENU", &MenuWidth[7], 
                        &MaxItemLength[7], 8),
                    **FCStr = GetItemStrings("CFUNCTIONMENU", &MenuWidth[10], 
                        &MaxItemLength[10], 7),
                    **PStr = GetItemStrings("PARAMETERMENU", &MenuWidth[8],
                        &MaxItemLength[8], 27),
                    **MTStr = GetItemStrings("MISCTEXTS", &MenuWidth[9],
                    &MaxItemLength[9], 7);
                    
    if(MStr == NULL) MaxItemLength[0] = 27;
    if(GRStr == NULL) MaxItemLength[1] = 31;
    if(GCStr == NULL) MaxItemLength[2] = 43;
    if(DRStr == NULL) MaxItemLength[3] = 40;
    if(DCSStr == NULL) MaxItemLength[4] = 11;
    if(DCSStr == NULL) MaxItemLength[5] = 11;
    if(CSCStr == NULL) MaxItemLength[6] = 31;
    if(FRStr == NULL) MaxItemLength[7] = 22;
    if(FCStr == NULL) MaxItemLength[10] = 22;
    if(PStr == NULL) MaxItemLength[8] = 34;
    if(MTStr == NULL) MaxItemLength[8] = 52;

    n = MaxItemLength[0];
    for(i = 1; i <= 9; i++) {
        if(MaxItemLength[i] > n) 
            n = MaxItemLength[i];
    }
    
    XPos = (80 - n) / 2;

    /* Create menu */
    
    if(PStr == NULL)
        Menu = CreateMenu("", XPos + 1, (Functiontype == FT_REAL) ? 14 : 15,
            0, 11, MenuWidth[8], 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 26, 0, 
            ParameterItems);
    else {
        Menu = CreateMenu("", XPos + 1, (Functiontype == FT_REAL) ? 14 : 15,
            0, 11, MenuWidth[8], 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 26, 0,
            ParameterItems);
        for(i = 1; i <= 26; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &PStr[i]);
    }
    
    a_clrscr(Console);
    
    if(Functiontype == FT_REAL) {
    
        /* Functiontype */
        
        a_gotoxy(Console, XPos, YPos); 
        a_textcolor(Console, WHITE);
        if(MStr == NULL) 
            a_cputs(Console, MainItems[0].Caption); 
        else 
            a_cputs(Console, MStr[1]);
        YPos+=2;
        
        /* Graph */

        a_gotoxy(Console, XPos, YPos); 
        a_textcolor(Console, WHITE); 
        a_cputs(Console, (MStr == NULL) ? MainItems[1].Caption : MStr[2]);
        YPos++;
                
        a_gotoxy(Console, XPos + 1, YPos); a_textcolor(Console, LIGHTGRAY);
        if(GRStr == NULL)
            a_cputs(Console, RealGraphItems[RealExp->Graph].Caption);
        else
            a_cputs(Console, GRStr[RealExp->Graph + 1]);
        YPos++;
        
        /* Diagram */
        
        if(RealExp->Graph == GR_R_STEP) {        
            a_gotoxy(Console, XPos + 2, YPos); 
            a_textcolor(Console, LIGHTGRAY);
            if(DRStr == NULL)
                a_cputs(Console, RealDiagramItems[RealExp->Diagram].Caption);
            else
                a_cputs(Console, DRStr[RealExp->Diagram + 1]);
            YPos++;
        }
        
        YPos++;
        
        /* Function */
        
        a_gotoxy(Console, XPos, YPos); 
        a_textcolor(Console, WHITE); 
        a_cputs(Console, (MStr == NULL) ? MainItems[2].Caption : MStr[3]);
        YPos++;

        a_textcolor(Console, LIGHTGRAY);
        if(RealExp->Function < 6) { 
            DrawCaption(Console, XPos + 1, YPos,
                RealFnItems[RealExp->Function].Caption);        
        }
        else {
            a_gotoxy(Console, XPos + 2, YPos);  
            if(FRStr == NULL)
                a_cputs(Console, RealFnItems[RealExp->Function].Caption);    
            else
                a_cputs(Console, FRStr[1]);
            
            YPos++;
            a_gotoxy(Console, XPos + 3, YPos);  
            a_cputs(Console, RealExp->FunctionStr);
        }
         
        YPos++;
         
        /* Parameters */

        SelectRealParItems(Menu, RealExp);
    }
    else {            

        /* Functiontype */
        
        a_gotoxy(Console, XPos, YPos); a_textcolor(Console, WHITE);
        if(MStr == NULL) 
            a_cputs(Console, MainItems[11].Caption); 
        else 
            a_cputs(Console, MStr[12]);
        YPos+=2;
        
        /* Graph */
        
        a_gotoxy(Console, XPos, YPos); 
        a_textcolor(Console, WHITE);
        a_cputs(Console, (MStr == NULL) ? MainItems[1].Caption : MStr[2]);
        YPos++;

        /*
        **  For 'Orbit diagram for var. c and z not equal to 0, it isn't
        **  a real Mandelbrot-set anymore, so we delete that from the caption.
        */

        if((ComplexExp->GraphPar[GR_C_ORBITS_C].zReal != 0) ||
            (ComplexExp->GraphPar[GR_C_ORBITS_C].zImag != 0) ||
            (ComplexExp->Function > 0)) 
        {
                SwapItemCaptions(&ComplexGraphItems[GR_C_ORBITS_C].Caption,
                    &ComplexGraphItems[GR_C_ORBITS_C + 1].Caption);
        }
        
        a_gotoxy(Console, XPos + 1, YPos); 
        a_textcolor(Console, LIGHTGRAY);
        if(GCStr == NULL)
            a_cputs(Console, ComplexGraphItems[ComplexExp->Graph].Caption);
        else
            a_cputs(Console, GCStr[ComplexExp->Graph + 1]);
        YPos++;
        
        if((ComplexExp->GraphPar[GR_C_ORBITS_C].zReal != 0) ||
            (ComplexExp->GraphPar[GR_C_ORBITS_C].zImag != 0) ||
            (ComplexExp->Function > 0)) 
        {
                SwapItemCaptions(&ComplexGraphItems[GR_C_ORBITS_C].Caption,
                    &ComplexGraphItems[GR_C_ORBITS_C + 1].Caption);
        }
        
        /* Diagram */

        if((ComplexExp->Graph == GR_C_STEP) || 
        (ComplexExp->Graph == GR_C_STEPINV)) {        
            a_gotoxy(Console, XPos + 2, YPos); 
            a_textcolor(Console, LIGHTGRAY);
            if(DCSStr == NULL)
                a_cputs(Console, 
                ComplexStepShapeItems[ComplexExp->Diagram].Caption);
            else
                a_cputs(Console, DCSStr[ComplexExp->Diagram + 1]);

            YPos++;
            
            a_gotoxy(Console, XPos + 3, YPos); a_textcolor(Console, LIGHTGRAY);
            if(CSCStr == NULL)
                a_cputs(Console, 
                ComplexStepColorItems[ComplexExp->Coloring].Caption);
            else
                a_cputs(Console, CSCStr[ComplexExp->Coloring + 1]);
            
            YPos++;
        }
        else {
            a_gotoxy(Console, XPos + 1, YPos); 
            a_textcolor(Console, LIGHTGRAY);
            if(DCStr == NULL)
                a_cputs(Console, 
                ComplexDiagramItems[ComplexExp->Diagram].Caption);
            else
                a_cputs(Console, DCStr[ComplexExp->Diagram + 1]);
            
            YPos++;
        }
        
        YPos++;
        
        /* Function */

        a_gotoxy(Console, XPos, YPos); a_textcolor(Console, WHITE);
        a_cputs(Console, (MStr == NULL) ? MainItems[2].Caption : MStr[3]);
        YPos++;

        a_textcolor(Console, LIGHTGRAY);
        if(Functiontype == FT_REAL) {
            DrawCaption(Console, XPos + 1, YPos,
                (FRStr == NULL) ? RealFnItems[RealExp->Function].Caption :
                FRStr[RealExp->Function + 1]);        
            if(RealExp->Function == 6) 
                DrawCaption(Console, XPos + 2, YPos + 1, RealExp->FunctionStr); 
        }
        else {
            DrawCaption(Console, XPos + 1, YPos,
                (FCStr == NULL) ? ComplexFnItems[ComplexExp->Function].Caption :
                FCStr[ComplexExp->Function + 1]);        
            if(ComplexExp->Function == 5) {
                DrawCaption(Console, XPos + 2, YPos + 1, 
                    ComplexExp->FunctionStr); 
            }
        }
        YPos++;
        
        /* Parameters */
        
        SelectComplexParItems(Menu, ComplexExp);
    }

    YPos+=2;
    
    /* Parameters */

    a_gotoxy(Console, XPos, YPos); a_textcolor(Console, WHITE);
    a_cputs(Console, (MStr == NULL) ? MainItems[3].Caption : MStr[4]);
    YPos++;

    for(i = 0; i < Menu->MaxItem; i++) 
        DrawItem(Console, Menu, Menu->Items[i]);

    YPos+=13;
    
    a_gotoxy(Console, XPos, YPos); a_textcolor(Console, WHITE);
    a_cputs(Console, (MTStr != NULL) ? MTStr[6] :
        "<ENTER> - Draw  / <Escape> - Back to parameter-menu.");
        
    DestroyItemStrings(MTStr, 7);
    DestroyItemStrings(PStr, 27);
    DestroyItemStrings(FCStr, 7);
    DestroyItemStrings(FRStr, 8);
    DestroyItemStrings(CSCStr, 4);
    DestroyItemStrings(DCStr, 5);
    DestroyItemStrings(DCSStr, 5);
    DestroyItemStrings(DRStr, 5);
    DestroyItemStrings(GCStr, 6);
    DestroyItemStrings(GRStr, 4);
    DestroyItemStrings(MStr, 14);
    DestroyMenu(Menu);
    
    Key = readkey();
    AltKey = Key >> 8;

    if(AltKey == KEY_ENTER) 
        return TRUE; 
    else if(AltKey == KEY_ESC)
        return FALSE;
    return FALSE;
}

int ClsMnuDraw(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    if(ClsMnuSummary(Console, RealExp, ComplexExp)) 
        MnuDraw(Functiontype, RealExp, ComplexExp);

    ClsMnuPar(Console, RealExp, ComplexExp);
    return 0;
}

void ClsMnuSettings(TConsole *Console, TRealExp *RealExp, 
    TComplexExp *ComplexExp, int *ISet, int *ILay, int *ICoo, int *IStep, 
    int *ICol)
{
    TMenu           *Menu;
    unsigned int    Value;
    int             i, Key, MenuXPos, SMenuWidth, MTMenuWidth, SMaxItemLength,
                    MTMaxItemLength;
    char            **SStr = GetItemStrings("SETTINGMENU", &SMenuWidth,
                    &SMaxItemLength, 6),
                    **MTStr = GetItemStrings("MISCTEXTS", &MTMenuWidth,
                    &MTMaxItemLength, 4);

    if(MTStr == NULL) {
        SMaxItemLength = 33;
        SMenuWidth = 33;
    }

    if(SStr == NULL) {
        SMenuWidth = 33;
        Menu = CreateMenu("SETTINGS - MENU:", 23, 10, *ISet, 5,
            SMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 5,
            (1 << M_HEADER) + (1 << M_STATUSBAR), SettingsItems);
    }
    else {
        MenuXPos = (80 - SMenuWidth) / 2;
        Menu = CreateMenu(SStr[0], MenuXPos, 10, *ISet, 5, SMenuWidth, 0,
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 5,
            (1 << M_HEADER) + (1 << M_STATUSBAR), SettingsItems);
        for(i = 1; i <= 5; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &SStr[i]);
    }

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    /* Parameter connect */
    
    a_gotoxy(Console, Menu->XPos + SMaxItemLength + 1, Menu->Items[4]->YPos);
    if(MTStr == NULL) 
        a_cputs(Console, (SA(GEN_ParConnect)) ? MiscTexts[0] : MiscTexts[1]);
    else
        a_cputs(Console, (SA(GEN_ParConnect)) ? MTStr[0] : MTStr[1]);
    
    /* Menu loop */
    
    while(TRUE) {

        Key = DoMenu(Console, Menu);

        if(Key == KEY_ESC) { 
            clear_keybuf(); 
            break; 
        }

        switch(Menu->ItemIndex) {
            case 0:
                if(Key == KEY_ENTER) {
                    ClsMnuLayout(Console, RealExp, ComplexExp, ILay, ICoo, 
                        IStep, ICol);
                    DrawMenu(Console, Menu);

                    /* Parameter connect */

                    a_gotoxy(Console, Menu->XPos + SMaxItemLength + 1, 
                        Menu->Items[4]->YPos);
                    if(MTStr == NULL) 
                        a_cputs(Console, (SA(GEN_ParConnect)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else
                        a_cputs(Console, (SA(GEN_ParConnect)) ? MTStr[0] : 
                            MTStr[1]);
                }
                break;
            case 1:
                break;
            case 3:
                break;
            case 4:
                if(Key == KEY_ENTER) {
                    Value = !SA(GEN_ParConnect);
                    if((SA(GEN_ParConnect)) != Value) 
                        RealExp->Changed = TRUE;
                    SLA(GEN_ParConnect, Value);
                    a_gotoxy(Console, Menu->XPos + SMaxItemLength + 1,
                        Menu->Items[4]->YPos);
                    if(MTStr == NULL) { 
                        a_cputs(Console, (SA(GEN_ParConnect)) ? (MiscTexts[0]) : 
                            (MiscTexts[1]));
                    }
                    else {
                        a_cputs(Console, 
                            (SA(GEN_ParConnect)) ? (MTStr[0]) : (MTStr[1]));
                    }
                    RealExp->Changed = TRUE;
                }
                break;
        }
    }
    *ISet = Menu->ItemIndex;
    
    DestroyItemStrings(MTStr, 4);
    DestroyItemStrings(SStr, 6);
    DestroyMenu(Menu);
    return;
}

void ClsMnuLayout(TConsole *Console, TRealExp *RealExp, 
    TComplexExp *ComplexExp, int *ILay, int *ICoo, int *IStep, int *ICol)
{
    TMenu           *Menu;
    unsigned int    Value;
    int             i, Key, MenuXPos, LMenuWidth, MTMenuWidth,
                    LMaxItemLength, MTMaxItemLength;
    char            **LStr = GetItemStrings("LAYOUTMENU", &LMenuWidth,
                    &LMaxItemLength, 6), 
                    **MTStr = GetItemStrings("MISCTEXTS", &MTMenuWidth, 
                    &MTMaxItemLength, 4);

    clear_keybuf();

    if(MTStr == NULL) {
        LMaxItemLength = 24;
        LMenuWidth = 24;
    }

    if(LStr == NULL) {
        LMenuWidth = 24;
        Menu = CreateMenu("LAYOUT - MENU:", 23, 10, *ILay, 5, LMenuWidth, 0,
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 5, 
            (1 << M_HEADER) + (1 << M_STATUSBAR), LayoutItems);
    }
    else {
        MenuXPos = (80 - LMaxItemLength) / 2;
        Menu = CreateMenu(LStr[0], MenuXPos, 10, *ILay, 5, LMenuWidth, 0,
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 5, 
            (1 << M_HEADER) + (1 << M_STATUSBAR), LayoutItems);
        for(i = 1; i <= 5; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &LStr[i]);
    }

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    a_gotoxy(Console, Menu->XPos + LMaxItemLength + 1, Menu->Items[4]->YPos);
    if(MTStr == NULL)
        a_cputs(Console, (SA(LAY_Statusline)) ? MiscTexts[0] : MiscTexts[1]);
    else        
        a_cputs(Console, (SA(LAY_Statusline)) ? (MTStr[0]) : (MTStr[1]));

    while(TRUE) {

        Key = DoMenu(Console, Menu);

        if(Key == KEY_ESC) { 
            clear_keybuf(); 
            break; 
        }

        switch(Menu->ItemIndex) {
            case 0:
                if(Key == KEY_ENTER) {
                    *ICoo = ClsMnuCoords(Console, RealExp, ComplexExp, *ICoo);
                    Menu->DefaultItemIndex = Menu->ItemIndex;
                    DrawMenu(Console, Menu);
                    a_gotoxy(Console, Menu->XPos + LMaxItemLength + 1, 
                        Menu->Items[4]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(LAY_Statusline)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else        
                        a_cputs(Console, 
                            (SA(LAY_Statusline)) ? (MTStr[0]) : (MTStr[1]));
                }
                break;
            case 1:
                if(Key == KEY_ENTER) {
                    *IStep = ClsMnuStep(Console, RealExp, ComplexExp, *IStep);
                    Menu->DefaultItemIndex = Menu->ItemIndex;
                    DrawMenu(Console, Menu);
                    a_gotoxy(Console, Menu->XPos + LMaxItemLength + 1, 
                        Menu->Items[4]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(LAY_Statusline)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else        
                        a_cputs(Console,
                            (SA(LAY_Statusline)) ? (MTStr[0]) : (MTStr[1]));
                }
                break;
            case 2:
                if(Key == KEY_ENTER) {
                    ClsEditColorPalette(Console, RealExp, ComplexExp);
                    Menu->DefaultItemIndex = Menu->ItemIndex;
                    DrawMenu(Console, Menu);
                    a_gotoxy(Console, Menu->XPos + LMaxItemLength + 1, 
                        Menu->Items[4]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(LAY_Statusline)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else        
                        a_cputs(Console, 
                            (SA(LAY_Statusline)) ? (MTStr[0]) : (MTStr[1]));
                }
                break;
            case 3:
                break;
            case 4:
                if(Key == KEY_ENTER) {
                    Value = !SA(LAY_Statusline);
                    if((SA(LAY_Statusline)) != Value) 
                        RealExp->Changed = TRUE;
                    SLA(LAY_Statusline, Value); 
                    a_gotoxy(Console, Menu->XPos + LMaxItemLength + 1, 
                        Menu->Items[4]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(LAY_Statusline)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else        
                        a_cputs(Console, 
                            (SA(LAY_Statusline)) ? (MTStr[0]) : (MTStr[1]));
                }
                break;
            default:
                break;
        }
    }
    *ILay = Menu->ItemIndex;
    DestroyItemStrings(MTStr, 4);
    DestroyItemStrings(LStr, 6);
    DestroyMenu(Menu);
    return;
}

void SetCoo(int Functiontype, TRealExp *RealExp, TComplexExp *ComplexExp)
{
    unsigned int    NewMode = 96;   /* X and Y Coords + Calibration (text) */
    
    NewMode = SetCoordsFlags((SA(COO_Status)),
                             (SA(COO_Grid)),
                             TRUE,
                             (SA(COO_Calibrate)),
                             TRUE, TRUE);
    
    if(Functiontype == FT_REAL) {
        RealExp->GraphPar[GR_R_STEP].Coords->Mode = NewMode;
        RealExp->GraphPar[GR_R_ORBITS_X].Coords->Mode = NewMode;
        RealExp->GraphPar[GR_R_ORBITS_C].Coords->Mode = NewMode;
    }
    else {
        ComplexExp->GraphPar[GR_C_STEP].Coords->Mode = NewMode;
        ComplexExp->GraphPar[GR_C_STEPINV].Coords->Mode = NewMode;
        ComplexExp->GraphPar[GR_C_ORBITS_Z].Coords->Mode = NewMode;
        ComplexExp->GraphPar[GR_C_ORBITS_C].Coords->Mode = NewMode;
    }
}
   
int ClsMnuCoords(TConsole *Console, TRealExp *RealExp, 
    TComplexExp *ComplexExp, int ICoo)
{
    TMenu           *Menu;
    unsigned int    Value;
    int             i, Key, MenuXPos, CMenuWidth, MTMenuWidth,
                    CMaxItemLength, MTMaxItemLength;
    char            **CStr = GetItemStrings("COORDINATEMENU", &CMenuWidth,
                    &CMaxItemLength, 4),
                    **MTStr = GetItemStrings("MISCTEXTS", &MTMenuWidth, 
                    &MTMaxItemLength, 4);

    if(MTStr == NULL) {
        CMaxItemLength = 28;
        CMenuWidth = 28;
    }

    if(CStr == NULL) {
        Menu = CreateMenu("SETTINGS FOR COORDINATES:", 23, 10, ICoo,
            3, 28, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 3,
            (1 << M_HEADER) + (1 << M_STATUSBAR), CoordItems);
    }
    else {
        MenuXPos = (80 - CMenuWidth) / 2;
        Menu = CreateMenu(CStr[0], MenuXPos, 10, ICoo, 3, CMaxItemLength, 
            0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 3, 
            (1 << M_HEADER) + (1 << M_STATUSBAR), CoordItems);
        for(i = 1; i <= 3; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &CStr[i]);
    }
    
    if(!SA(COO_Status)) {
        Menu->Items[1]->Flag = 0;
        Menu->Items[2]->Flag = 0;
    }
    else {
        Menu->Items[1]->Flag = 96;
        Menu->Items[2]->Flag = 96;
    }

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1, Menu->Items[0]->YPos);
    if(MTStr == NULL)
        a_cputs(Console, (SA(COO_Status)) ? MiscTexts[0] : MiscTexts[1]);
    else
        a_cputs(Console, (SA(COO_Status)) ? (MTStr[0]) : (MTStr[1]));

    if(SA(COO_Status)) {
        a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1, 
            Menu->Items[1]->YPos);
        if(MTStr == NULL)
            a_cputs(Console, (SA(COO_Grid)) ? (MiscTexts[0]) : (MiscTexts[1]));
        else
            a_cputs(Console, (SA(COO_Grid)) ? (MTStr[0]) : (MTStr[1]));

        a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1,
            Menu->Items[2]->YPos);
        if(MTStr == NULL)
            a_cputs(Console, (SA(COO_Calibrate)) ? MiscTexts[0] : MiscTexts[1]);
        else
            a_cputs(Console, (SA(COO_Calibrate)) ? (MTStr[0]) : (MTStr[1]));
    }
    
    while(TRUE) {

        Key = DoMenu(Console, Menu);

        if(Key == KEY_ESC) break;

        switch(Menu->ItemIndex) {
            case 0:
                if(Key == KEY_ENTER) {
                    Value = !SA(COO_Status);
                    if((SA(COO_Status)) != Value) 
                        RealExp->Changed = TRUE;
                    SLA(COO_Status, Value);
                    if(!SA(COO_Status)) {
                        Menu->Items[1]->Flag = 0;
                        Menu->Items[2]->Flag = 0;
                    }
                    else {
                        Menu->Items[1]->Flag = 96;
                        Menu->Items[2]->Flag = 96;
                    }
                    DrawMenu(Console, Menu);

                    a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1, 
                        Menu->Items[0]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(COO_Status)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else
                        a_cputs(Console,
                            (SA(COO_Status)) ? (MTStr[0]) : (MTStr[1]));

                    if(SA(COO_Status)) {
                        a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1, 
                            Menu->Items[1]->YPos);
                        if(MTStr == NULL)
                            a_cputs(Console, (SA(COO_Grid)) ? (MiscTexts[0]) : 
                                (MiscTexts[1]));
                        else
                            a_cputs(Console, 
                                (SA(COO_Grid)) ? (MTStr[0]) : (MTStr[1]));
                    
                        a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1, 
                            Menu->Items[2]->YPos);
                        if(MTStr == NULL)
                            a_cputs(Console, (SA(COO_Calibrate)) ? 
                                MiscTexts[0] : MiscTexts[1]);
                        else
                            a_cputs(Console, 
                                (SA(COO_Calibrate)) ? (MTStr[0]) : (MTStr[1]));
                    }
                }
                break;
            case 1:
                if(Key == KEY_ENTER) {
                    Value = !SA(COO_Grid);
                    if((SA(COO_Grid)) != Value) 
                        RealExp->Changed = TRUE;
                    SLA(COO_Grid, Value);
                    a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1,
                        Menu->Items[1]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(COO_Grid)) ? (MiscTexts[0]) : 
                            (MiscTexts[1]));
                    else
                        a_cputs(Console, (SA(COO_Grid)) ? (MTStr[0]) : 
                            (MTStr[1]));
                }
                break;
            case 2:
                if(Key == KEY_ENTER) {
                    Value = !SA(COO_Calibrate);
                    if((SA(COO_Calibrate)) != Value) 
                        RealExp->Changed = TRUE;
                    SLA(COO_Calibrate, Value);
                    a_gotoxy(Console, Menu->XPos + CMaxItemLength + 1, 
                        Menu->Items[2]->YPos);
                    if(MTStr == NULL)
                        a_cputs(Console, (SA(COO_Calibrate)) ? MiscTexts[0] : 
                            MiscTexts[1]);
                    else
                        a_cputs(Console,
                            (SA(COO_Calibrate)) ? (MTStr[0]) : (MTStr[1]));
                }
                break;
        }
    }
    SetCoo(Functiontype, RealExp, ComplexExp);
    ICoo = Menu->ItemIndex;
    DestroyItemStrings(MTStr, 4);
    DestroyItemStrings(CStr, 4);
    DestroyMenu(Menu);
    return(ICoo);
}

int ClsMnuStep(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp, 
    int IStep)
{
    TMenu           *Menu;
    unsigned int    Value;
    int             i, Key, MenuXPos, SMenuWidth, MTMenuWidth,
                    SMaxItemLength, MTMaxItemLength;
    char            **SStr = GetItemStrings("STEPMENU",
                    &SMenuWidth, &SMaxItemLength, 5), 
                    **MTStr = GetItemStrings("MISCTEXTS",
                    &MTMenuWidth, &MTMaxItemLength, 5);
    
    if(MTStr == NULL) {
        SMaxItemLength = 29;
        SMenuWidth = 29;
    }

    if(SStr == NULL) {
        SMenuWidth = 29;
        Menu = CreateMenu("SETTINGS FOR STEP-BY-STEP ITERATION:", 23, 10, IStep,
            4, SMaxItemLength, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4, 
            (1 << M_HEADER) + (1 << M_STATUSBAR), StepItems);
    }
    else {
        MenuXPos = (80 - SMenuWidth) / 2;
        Menu = CreateMenu(SStr[0], MenuXPos, 10, IStep, 4, SMaxItemLength, 0,
            LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
            (1 << M_HEADER) + (1 << M_STATUSBAR), StepItems);
        for(i = 1; i <= 4; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &SStr[i]);
    }


    if(Functiontype == FT_REAL) {
        Menu->Items[2]->Flag = 0;
        Menu->Items[3]->Flag = 2146;
        sprintf(Menu->Items[3]->Contents, "%d", SA(SSI_LineWidth));
    }
    else {
        Menu->Items[2]->Flag = 96;
        Menu->Items[3]->Flag = 0;
    }

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    if(Functiontype == FT_COMPLEX) { 
        a_gotoxy(Console, Menu->XPos + SMaxItemLength + 1, 
            Menu->Items[2]->YPos);
        if(MTStr == NULL)
            a_cputs(Console, (SA(SSI_Lines)) ? (MiscTexts[0]) : (MiscTexts[1]));
        else
            a_cputs(Console, (SA(SSI_Lines)) ? (MTStr[0]) : (MTStr[1]));
    }

    Console->CursorMode = _SOLIDCURSOR;

    while(TRUE) {

        if(Functiontype == FT_REAL)
            SLA(SSI_LineWidth, atoi(Menu->Items[3]->Contents));

        Key = DoMenu(Console, Menu);

        if(Key == KEY_ESC) {
            Console->CursorMode = _NOCURSOR;
            break;
        }
        
        switch(Menu->ItemIndex) {
            case 0:
                break;
            case 1:
                break;
            case 2:
                if(Functiontype == FT_COMPLEX) {
                    if(Key == KEY_ENTER) {
                        Value = !SA(SSI_Lines);
                        SLA(SSI_Lines, Value);
                        a_gotoxy(Console, Menu->XPos + SMaxItemLength + 1, 
                            Menu->Items[2]->YPos);
                        if(MTStr == NULL)
                            a_cputs(Console, (SA(SSI_Lines)) ? (MiscTexts[0]) : 
                                (MiscTexts[1]));
                        else
                            a_cputs(Console, (SA(SSI_Lines)) ? (MTStr[0]) : 
                                (MTStr[1]));
                    }
                }
                break;
            case 3:
                break;
        }
    }
    IStep = Menu->ItemIndex;
    DestroyItemStrings(MTStr, 5);
    DestroyItemStrings(SStr, 5);
    DestroyMenu(Menu);
    return(IStep);
}

int ClsEditColorPalette(TConsole *Console, TRealExp *RealExp, 
    TComplexExp *ComplexExp)
{
    TConsole            *CSConsole;
    TMenu               *Menu;
    int                 OldIdxCol, c, x, y, xp, yp, oxp, oyp, 
                        BlockSize, AltKey = 0, Key = 0, 
                        CPEMenuWidth, MTMenuWidth,
                        CPEMaxItemLength, MTMaxItemLength, 
                        CSMenuWidth, CSMaxItemLength,
                        ColorFlag, ExitFlag, XOffset, YOffset;
    unsigned int        MaxCol, Col = 0;
    unsigned long int   i;
    char                **CPEStr = GetItemStrings("CPEHELP",
                        &CPEMenuWidth, &CPEMaxItemLength, 7), 
                        **MTStr = GetItemStrings("MISCTEXTS",
                        &MTMenuWidth, &MTMaxItemLength, 5),
                        **CSStr = GetItemStrings("SETCOLORMENU", 
                        &CSMenuWidth, &CSMaxItemLength, 7);

    a_clrscr(Console);

    /* Allocate space for menu */
   
    if(CSStr == NULL) {
        CSConsole = alconio_init(30, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, 28, 0, BLACK, WHITE, WHITE, BLACK, 
            7, 0, ColorItems);
    }
    else {
        CSConsole = alconio_init(CSMaxItemLength + 2, 7, 1, BLACK, GRAY);
        Menu = CreateMenu("", 1, 1, 0, 5, CSMaxItemLength, 0, BLACK, WHITE, 
            WHITE, BLACK, 7, 0, ColorItems);
        for(i = 0; i <= 4; i++) 
            SwapItemCaptions(&Menu->Items[i]->Caption, &CSStr[i + 1]);
    }
    
    XOffset = (SCREEN_W - 640) / 2;
    YOffset = (SCREEN_H - 480) / 2;

    if(Functiontype == FT_REAL) 
        MaxCol = RealExp->Settings->COL_Max;
    else
        MaxCol = ComplexExp->Settings->COL_Max;
        
    if(MTStr == NULL) {
        CPEMaxItemLength = 22;
        CPEMenuWidth = 22;
    }
    
    /* Draw Palette */

    BlockSize = 20;
    x = 0;
    y = 0;
    for(i = 0; i <= MaxCol; i++) {
        rectfill(screen, XOffset + x * BlockSize + 6, 
            YOffset + y * BlockSize + 6,
            XOffset + x * BlockSize + BlockSize, 
            YOffset + y * BlockSize + BlockSize, 
            makecol(SA(Palette[i].r), SA(Palette[i].g), SA(Palette[i].b)));
        
        c = makecol(255 - SA(Palette[i].r), 255 - SA(Palette[i].g),
            255 - SA(Palette[i].b));

        if(i == SA(COL_Grid)) 
            textprintf_ex(screen, font,  
                XOffset + x * BlockSize + 6, 
                YOffset + y * BlockSize + (BlockSize - text_height(font)) / 2, 
                c, -1, "0");
        else if(i == SA(COL_Function))
            textprintf_ex(screen, font,  
                XOffset + x * BlockSize + 6, 
                YOffset + y * BlockSize + (BlockSize - text_height(font)) / 2, 
                c, -1, "1");
        else if(i == SA(COL_XYLine))
            textprintf_ex(screen, font,  
                XOffset + x * BlockSize + 6, 
                YOffset + y * BlockSize + (BlockSize - text_height(font)) / 2, 
                c, -1, "2");
        else if(i == SA(COL_Iter1))
            textprintf_ex(screen, font,  
                XOffset + x * BlockSize + 6, 
                YOffset + y * BlockSize + (BlockSize - text_height(font)) / 2, 
                c, -1, "3");
        else if(i == SA(COL_Iter2))
            textprintf_ex(screen, font,  
                XOffset + x * BlockSize + 6, 
                YOffset + y * BlockSize + (BlockSize - text_height(font)) / 2, 
                c, -1, "4");

        x++;
        if(((i + 1) % 16) == 0) {
            x = 0;
            y++;
        }
    }
    
    rect(screen, XOffset + Col * BlockSize + 3, YOffset + 0 * BlockSize + 3,
        XOffset + Col * BlockSize + BlockSize + 3, YOffset + BlockSize + 3, 
        makecol(255, 255, 255));
    
    /* Print help text */

    textprintf_ex(screen, font, XOffset + 5, YOffset + 360, 
        makecol(255, 255, 255), 0, "%s", 
        (CPEStr == NULL) ? EditColPalHelp[3] : CPEStr[3]);
    textprintf_ex(screen, font, XOffset + 5, 
        YOffset + 360 + text_height(font) + 3, 
        makecol(255, 255, 255), 0, "%s", 
        (CPEStr == NULL) ? EditColPalHelp[4] : CPEStr[4]);
    textprintf_ex(screen, font, XOffset + 5, 
        YOffset + 360 + 2 * (text_height(font) + 3), 
        makecol(255, 255, 255), 0, "%s", 
        (CPEStr == NULL) ? EditColPalHelp[5] : CPEStr[5]);
    textprintf_ex(screen, font, XOffset + 5, 
        YOffset + 360 + 3 * (text_height(font) + 3), 
        makecol(255, 255, 255), 0, "%s", 
        (CPEStr == NULL) ? EditColPalHelp[6] : CPEStr[6]);

    for(i = 0; i <= 4; i++) {
        textprintf_ex(screen, font, XOffset + 375, 
            YOffset + (20 + i * 2) * text_height(font), 
            makecol(255, 255, 255), 0, "%s %1d", 
            Menu->Items[i]->Caption, (int) i);
    }

    Console->CursorMode = _NOCURSOR;
    BlockSize = 20;
    ColorFlag = -1;
    ExitFlag = FALSE;
    xp = oxp = 0;
    yp = oyp = 0;
    while(!ExitFlag) {
        if((Key == KEY_LEFT || Key == KEY_RIGHT || Key == KEY_UP || 
            Key == KEY_DOWN) && !(key_shifts & KB_SHIFT_FLAG)) 
        {
            rect(screen, XOffset + oxp * BlockSize + 3, 
                YOffset + oyp * BlockSize + 3,
                XOffset + oxp * BlockSize + BlockSize + 3, 
                YOffset + oyp * BlockSize + BlockSize + 3, 
                makecol(0, 0, 0));
            rect(screen, XOffset + xp * BlockSize + 3, 
                YOffset + yp * BlockSize + 3,
                XOffset + xp * BlockSize + BlockSize + 3, 
                YOffset + yp * BlockSize + BlockSize + 3, 
                makecol(255, 255, 255));
        }
        textprintf_ex(screen, font, XOffset + 400, 
            YOffset + 10 * text_height(font), 
            makecol(255, 255, 255), 0, "%s %3d", 
            (CPEStr == NULL) ? EditColPalHelp[0] : CPEStr[0], 
            SA(Palette[Col].r));
        textprintf_ex(screen, font, XOffset + 400, 
            YOffset + 12 * text_height(font), 
            makecol(255, 255, 255), 0, "%s %3d", 
            (CPEStr == NULL) ? EditColPalHelp[1] : CPEStr[1], 
            SA(Palette[Col].g));
        textprintf_ex(screen, font, XOffset + 400, 
            YOffset + 14 * text_height(font), 
            makecol(255, 255, 255), 0, "%s %3d", 
            (CPEStr == NULL) ? EditColPalHelp[2] : CPEStr[2], 
            SA(Palette[Col].b));
        
        clear_keybuf();
        Key = readkey() >> 8;
        switch(Key) {
            case KEY_TAB:
                ColorFlag = -1; 
                a_getimage(CSConsole, CSConsole->ScreenContents);
                DrawMenu(CSConsole, Menu);
                while(TRUE) {
                    AltKey = DoMenu(CSConsole, Menu);
                    if(AltKey == KEY_ENTER) {
            
                        switch(Menu->ItemIndex) {
                            case 0:
                                OldIdxCol = SA(COL_Grid);
                                break;
                            case 1:
                                OldIdxCol = SA(COL_Function);
                                break;
                            case 2:
                                OldIdxCol = SA(COL_XYLine);
                                break;
                            case 3:
                                OldIdxCol = SA(COL_Iter1);
                                break;
                            case 4:
                                OldIdxCol = SA(COL_Iter2);
                                break;
                        }    
                        
                        if((Col == SA(COL_Grid)) || (Col == SA(COL_Function)) ||
                            (Col == SA(COL_XYLine)) ||
                            (Col == SA(COL_Iter1)) || (Col == SA(COL_Iter2)))
                                break;

                        /* Set new value */
                        
                        switch(Menu->ItemIndex) {
                            case 0:
                                SLA(COL_Grid, Col);
                                break;
                            case 1:
                                SLA(COL_Function, Col);
                                break;
                            case 2:
                                SLA(COL_XYLine, Col);
                                break;
                            case 3:
                                SLA(COL_Iter1, Col);
                                break;
                            case 4:
                                SLA(COL_Iter2, Col);
                                break;
                        }
    
                        /* Erase text in the previously selected color */

                        x = XOffset + (OldIdxCol % 16) * BlockSize;
                        y = YOffset + (OldIdxCol / 16) * BlockSize;
                        c = makecol(SA(Palette[OldIdxCol].r),
                            SA(Palette[OldIdxCol].g),
                            SA(Palette[OldIdxCol].b));
                        
                        rectfill(screen, x + 6, y + 6, x + BlockSize,
                                y + BlockSize, c);

                        x = XOffset + xp * BlockSize;
                        y = YOffset + yp * BlockSize;
                        c = makecol(255 - SA(Palette[Col].r), 
                            255 - SA(Palette[Col].g),
                            255 - SA(Palette[Col].b));
                        
                        textprintf_ex(screen, font,  
                            x + 6, y + (BlockSize - text_height(font)) / 2,
                            c, -1, "%d", Menu->ItemIndex);

                        Menu->DefaultItemIndex = Menu->ItemIndex;
                        break;
                    }
                }
                
                a_putimage(CSConsole, CSConsole->ScreenContents);
                break;
            case KEY_RIGHT:
                if(Col < MaxCol) {
                    ColorFlag = -1; 
                    Col++; 
                    oxp = xp;
                    oyp = yp;
                    if(xp < 15) 
                        xp++; 
                    else 
                    { 
                        xp = 0; 
                        if(yp < ((int) (MaxCol / 16))) 
                            yp++; 
                    }
                }
                break;
            case KEY_LEFT:
                if(Col > 0) { 
                    ColorFlag = -1; 
                    Col--; 
                    oxp = xp;
                    oyp = yp;
                    if(xp > 0) 
                        xp--; 
                    else { 
                        xp = 15; 
                        if(yp > 0) 
                            yp--; 
                    }
                }
                break;
            case KEY_UP:
                if(Col > 15 && !(key_shifts & KB_SHIFT_FLAG)) { 
                    ColorFlag = -1; 
                    Col-=16; 
                    oxp = xp;
                    oyp = yp;
                    if(yp > 0) 
                        yp--; 
                }
                break;
            case KEY_DOWN: 
                if(Col < (MaxCol - 16) && !(key_shifts & KB_SHIFT_FLAG)) { 
                    ColorFlag = -1; 
                    Col+=16; 
                    oxp = xp;
                    oyp = yp;
                    if(yp < ((int) (MaxCol / 16)))
                        yp++;
                }
                break;
            case KEY_R:
                ColorFlag = 0;
                break;
            case KEY_G:
                ColorFlag = 1;
                break;
            case KEY_B:
                ColorFlag = 2;
                break;
            case KEY_ESC:
                ExitFlag = TRUE;
                break;
            default:
                break;            
        }
        if(ColorFlag == 0) {
            switch(Key) {
                case KEY_UP:
                    if(SA(Palette[Col].r) < 255 && (key_shifts & KB_SHIFT_FLAG)) 
                        SA(Palette[Col].r++);
                    break;
                case KEY_DOWN: 
                    if(SA(Palette[Col].r) > 0 && (key_shifts & KB_SHIFT_FLAG)) 
                        SA(Palette[Col].r--);
                    break;
                case KEY_PGUP:
                    if(SA(Palette[Col].r) <= 234) SA(Palette[Col].r+=21);
                    break;
                case KEY_PGDN: 
                    if(SA(Palette[Col].r) >= 21) SA(Palette[Col].r-=21);
                    break;
                default:
                    break;
            }
        }
        else if(ColorFlag == 1) {
            switch(Key) {
                case KEY_UP:
                    if(SA(Palette[Col].g) < 255 && (key_shifts & KB_SHIFT_FLAG)) 
                        SA(Palette[Col].g++);
                    break;
                case KEY_DOWN: 
                    if(SA(Palette[Col].g) > 0 && (key_shifts & KB_SHIFT_FLAG)) 
                        SA(Palette[Col].g--);
                    break;
                case KEY_PGUP:
                    if(SA(Palette[Col].g) <= 234) 
                        SA(Palette[Col].g+=21);
                    break;
                case KEY_PGDN: 
                    if(SA(Palette[Col].g) >= 21) 
                        SA(Palette[Col].g-=21);
                    break;
                default:
                    break;
            }
        }
        else if(ColorFlag == 2) {
            switch(Key) {
                case KEY_UP:
                    if(SA(Palette[Col].b) < 255 && (key_shifts & KB_SHIFT_FLAG)) 
                        SA(Palette[Col].b++);
                    break;
                case KEY_DOWN: 
                    if(SA(Palette[Col].b) > 0  && (key_shifts & KB_SHIFT_FLAG)) 
                        SA(Palette[Col].b--);
                    break;
                case KEY_PGUP:
                    if(SA(Palette[Col].b) <= 234) 
                        SA(Palette[Col].b+=21);
                    break;
                case KEY_PGDN: 
                    if(SA(Palette[Col].b) >= 21) 
                        SA(Palette[Col].b-=21);
                    break;
                default:
                    break;
            }
        }
        
        if(ColorFlag > -1) {
            x = XOffset + xp * BlockSize;
            y = YOffset + yp * BlockSize;
            c = makecol(SA(Palette[Col].r), SA(Palette[Col].g),
                SA(Palette[Col].b));

            rectfill(screen, x + 6, y + 6, x + BlockSize, y + BlockSize, c);

            c = makecol(255 - SA(Palette[Col].r), 255 - SA(Palette[Col].g),
                255 - SA(Palette[Col].b));

            if(Col == SA(COL_Grid)) 
                textprintf_ex(screen, font,  
                    x + 6, y + (BlockSize - text_height(font)) / 2, c, -1, "0");
            else if(Col == SA(COL_Function))
                textprintf_ex(screen, font,  
                    x + 6, y + (BlockSize - text_height(font)) / 2, c, -1, "1");
            else if(Col == SA(COL_XYLine))
                textprintf_ex(screen, font,  
                    x + 6, y + (BlockSize - text_height(font)) / 2, c, -1, "2");
            else if(Col == SA(COL_Iter1))
                textprintf_ex(screen, font,  
                    x + 6, y + (BlockSize - text_height(font)) / 2, c, -1, "3");
            else if(Col == SA(COL_Iter2))
                textprintf_ex(screen, font,  
                    x + 6, y + (BlockSize - text_height(font)) / 2, c, -1, "4");
        }
    }

    clear_keybuf();
    DestroyMenu(Menu);
    DestroyItemStrings(MTStr, 5);
    DestroyItemStrings(CPEStr, 7);
    DestroyItemStrings(CSStr, 7);
    alconio_exit(CSConsole);
    Console->CursorMode = _SOLIDCURSOR;
    return 0;
}

int ClsMnuFiles(TConsole *Console, TRealExp *RealExp, TComplexExp *ComplexExp,
int *ItemF)
{
    TMenu       *Menu;
    int         i, Key, MenuXPos, FMenuWidth, MTMenuWidth,
                FMaxItemLength, MTMenuItemLength, MEMenuWidth, MEMaxItemLength,
                TempFT;
    char        *FTMsg[] = {    
                    
                    "You are trying to read a parameter-file for functiontype ",
                    "'From R to R'",
                    "'From C to C'",
                    "yet the current functiontype is ",
                    "Do you want to change the functiontype?",
                    "Yes", "No"
                
                }, *Str1, *Str2,
                **FStr = GetItemStrings("FILEMENU", &FMenuWidth,
                &FMaxItemLength, 5),
                **MTStr = GetItemStrings("MISCTEXT", &MTMenuWidth,
                &MTMenuItemLength, 6),
                **MEStr = GetItemStrings("MESSAGES", &MEMenuWidth,
                &MEMaxItemLength, 16),
                FilePath[2048] = "default.par";

    if(MEStr != NULL) {
        if(Functiontype == FT_REAL) {
            Str1 = (char *) malloc(10 + strlen(MEStr[10]) + strlen(MEStr[12]));
            Str2 = (char *) malloc(10 + strlen(MEStr[13]) + strlen(MEStr[11]));
        }
        else {
            Str1 = (char *) malloc(10 + strlen(MEStr[10]) + strlen(MEStr[11]));
            Str2 = (char *) malloc(10 + strlen(MEStr[13]) + strlen(MEStr[12]));
        }        
        strcpy(Str1, MEStr[10]);
        strcat(Str1, (Functiontype == FT_REAL) ? MEStr[12] : MEStr[11]);
        strcpy(Str2, MEStr[13]);
        strcat(Str2, (Functiontype == FT_REAL) ? MEStr[11] : MEStr[12]);
    }
    else {
        if(Functiontype == FT_REAL) {
            Str1 = (char *) malloc(10 + strlen(FTMsg[0]) + strlen(FTMsg[2]));
            Str2 = (char *) malloc(10 + strlen(FTMsg[3]) + strlen(FTMsg[1]));
        }
        else {
            Str1 = (char *) malloc(10 + strlen(FTMsg[0]) + strlen(FTMsg[1]));
            Str2 = (char *) malloc(10 + strlen(FTMsg[3]) + strlen(FTMsg[2]));
        }        
        strcpy(Str1, FTMsg[0]);
        strcat(Str1, (Functiontype == FT_REAL) ? FTMsg[2] : FTMsg[1]);
        strcpy(Str2, FTMsg[3]);
        strcat(Str2, (Functiontype == FT_REAL) ? FTMsg[1] : FTMsg[2]);
    }

    if(MTStr == NULL) {
        FMaxItemLength = 28;
        FMenuWidth = 28;
    }

    if(FStr == NULL) {
        FMenuWidth = 28;
        Menu = CreateMenu("FILE - MENU:", 23, 10, *ItemF, 4, FMenuWidth,
            0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
            (1 << M_HEADER) + (1 << M_STATUSBAR), FilesItems);
    }
    else {
        MenuXPos = (80 - FMenuWidth) / 2;
        Menu = CreateMenu(FStr[0], MenuXPos, 10, *ItemF,
            4, FMenuWidth, 0, LIGHTGRAY, BLACK, BLACK, LIGHTGRAY, 4,
            (1 << M_HEADER) + (1 << M_STATUSBAR), FilesItems);
        for(i = 1; i <= 4; i++) 
            SwapItemCaptions(&Menu->Items[i - 1]->Caption, &FStr[i]);
    }

    a_clrscr(Console);
    DrawMenu(Console, Menu);

    while(TRUE) {

        Key = DoMenu(Console, Menu);

        if(Key == KEY_ESC) 
            break;

        switch(Menu->ItemIndex) {
            case 0:
                if(Key == KEY_ENTER) {
                    a_clrscr(Console);
                    if(FILE_SELECT_BOX((MTStr == NULL) ? MiscTexts[4] : 
                        MTStr[4], FilePath, "PAR", 2047, 300, 400) != 0) 
                    {
                        scare_mouse();
                        if(Functiontype == FT_REAL)
                            TempFT = LoadRealExp(RealExp, FilePath, TRUE);
                        else
                            TempFT = LoadComplexExp(ComplexExp, FilePath, TRUE);
                        unscare_mouse();
                        SetCoo(Functiontype, RealExp, ComplexExp);
                    }
                    DrawMenu(Console, Menu);
                }
                break;
            case 1:
                if(Key == KEY_ENTER) {
                    strcpy(FilePath, "default.prm");
                }
                break;
            case 2:
                if(Key == KEY_ENTER) {
                    scare_mouse();
                    if(Functiontype == FT_REAL)
                        WriteRealConfig(RealExp);
                    else
                        WriteComplexConfig(ComplexExp);
                    unscare_mouse();
                    Menu->DefaultItemIndex = 2;
                    DrawMenu(Console, Menu);
                }
                break;
            case 3:
                if(Key == KEY_ENTER) {
                    a_clrscr(Console);
                    strcpy(FilePath, "default.par");
                    if(FILE_SELECT_BOX((MTStr == NULL) ? MiscTexts[5] : 
                        MTStr[5], FilePath, "PAR", 2047, 300, 400) != 0) 
                    {
                        scare_mouse();
                        if(Functiontype == FT_REAL) {
                            SaveRealExp(RealExp, FilePath);
                            RealExp->Changed = FALSE;
                        }
                        else
                            SaveComplexExp(ComplexExp, FilePath);
                        unscare_mouse();
                    }
                    DrawMenu(Console, Menu);
                }
                break;
        }
    }

    free(Str2);
    free(Str1);

    *ItemF = Menu->ItemIndex;
    
    DestroyItemStrings(MEStr, 16);
    DestroyItemStrings(MTStr, 6);
    DestroyItemStrings(FStr, 5);
    DestroyMenu(Menu);

    return Functiontype;
}

int ClsMnuAbout(TConsole *Console)
{
    a_clrscr(Console);
    a_textcolor(Console, GREEN);
    a_gotoxy(Console, 6, 1);
    a_cputs(Console,
        "Please read this document carefully before using this software:");
    a_textcolor(Console, YELLOW);
    a_gotoxy(Console, 6, 3);
    a_cputs(Console,
        "Funiter V2.4.0 Copyright (C) 1995-2010 Stijn Wolters.");
    a_gotoxy(Console, 6, 4);
    a_cputs(Console,
        "Original idea: Ernic Kamerich (University of Nijmegen).");
    a_textcolor(Console, GREEN);
    a_gotoxy(Console, 6, 6);
    a_cputs(Console, 
        "See README for contact information.");
    a_gotoxy(Console, 6, 8);
    a_textcolor(Console, RED);
    a_cputs(Console,
        "This program is free software; you can redistribute it and/or modify ");
    a_gotoxy(Console, 6, 9);
    a_cputs(Console, 
        "it under the terms of the GNU General Public License as published by");
    a_gotoxy(Console, 6, 10);
    a_cputs(Console,
        "the Free Software Foundation; either version 2 of the License, or");
    a_gotoxy(Console, 6, 11);
    a_cputs(Console, 
        "(at your option) any later version.");
    a_gotoxy(Console, 6, 13);
    a_cputs(Console, 
        "This program is distributed in the hope that it will be useful,");
    a_gotoxy(Console, 6, 14);
    a_cputs(Console, 
        "but WITHOUT ANY WARRANTY; without even the implied warranty of");
    a_gotoxy(Console, 6, 15);
    a_cputs(Console,
        "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
    a_gotoxy(Console, 6, 16);
    a_cputs(Console, 
        "GNU General Public License for more details.");
    a_gotoxy(Console, 6, 18);
    a_cputs(Console, 
        "You should have received a copy of the GNU General Public License");
    a_gotoxy(Console, 6, 19);
    a_cputs(Console, 
        "along with this program; if not, write to the Free Software");
    a_gotoxy(Console, 6, 20);
    a_cputs(Console, 
        "Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307");  
    a_gotoxy(Console, 6, 21);
    a_cputs(Console, "USA");
    a_textcolor(Console, LIGHTGREEN);
    a_gotoxy(Console, 6, 23); 
    a_cprintf(Console,
        "Funiter was last compiled on %s at %s", __DATE__, __TIME__);
    a_textcolor(Console, GREEN);
    a_gotoxy(Console, 6, 25);
    a_cputs(Console, 
        "See COPYING for more information.");               
    a_textcolor(Console, WHITE);
    a_gotoxy(Console, 51, 25);
    a_cputs(Console, "[Hit a key to continue]");

    a_textcolor(Console, LIGHTGRAY);
    a_gotoxy(Console, 6, 27);
    a_cputs(Console, "Build with: Allegro");
#ifdef BUILD_WITH_JPGALLEG
    a_cputs(Console, ", JPGalleg");
#endif
    a_gotoxy(Console, 6, 29);
    a_cputs(Console, "See README for more info on the libraries.");

    clear_keybuf();    
    readkey();
    clear_keybuf();    
    return 0;
}
