#include <time.h>

#include <allegro.h>

#include "dprimitives.h"	// additional drawing primitives
#include "tetgui.h"	// the game's GUI

#include "tetdat.h"	// tetris datafile header

DATAFILE* tetdat; // game datafile

// different palettes for changing background

int bgpal = 3;

const int NBG_PALETTES = 5;

PALETTE background_palettes[NBG_PALETTES];

BITMAP* dbuf;

/* basic block image. 20x20 pixels */

int block20x20_image[20][20] =
{
   {14, 14, 14, 14, 13, 13, 13, 13, 12, 12, 11, 11, 10, 10,  9,  9,  8,  8,  8,  8},
   {14, 14, 14, 14, 13, 13, 13, 13, 12, 12, 11, 11, 10, 10,  9,  9,  8,  8,  8,  8},
   {14, 14,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  7,  7},
   {14, 14,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  7,  7},
   {13, 13,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  6,  6},
   {13, 13,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  6,  6},
   {13, 13,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5},
   {13, 13,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5},
   {12, 12,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  4,  4},
   {12, 12,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  4,  4},
   {11, 11,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  3,  3},
   {11, 11,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  3,  3},
   {10, 10,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  2,  2},
   {10, 10,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  2,  2},
   { 9,  9,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  1,  1},
   { 9,  9,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  1,  1},
   { 8,  8,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  0,  0},
   { 8,  8,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  5,  0,  0},
   { 8,  8,  7,  7,  6,  6,  5,  5,  4,  4,  3,  3,  2,  2,  1,  1,  0,  0,  0,  0},
   { 8,  8,  7,  7,  6,  6,  5,  5,  4,  4,  3,  3,  2,  2,  1,  1,  0,  0,  0,  0}
};

const int NBLOCKS = 19;

/* rotation lookup table
   block with block.nr X will become blockrotation[X] */
   
int blockrotation[NBLOCKS] = {1,0,5,4,2,3,11,12,6,8,7,9,13,10,14,18,17,16,15};

/* blockheight lookup table
   block with block.nr X has height blockheight[X] */

int blockheight[NBLOCKS] = {1,4,3,3,2,2,3,3,2,3,2,2,2,3,2,3,3,2,2};

/* blockwidth lookup table
   block with block.nr X has width blockwidth[X] */

int blockwidth[NBLOCKS] = {4,1,2,2,3,3,2,2,3,2,3,3,3,2,2,2,2,3,3};

/* block shapes */
 
bool blockshapes[NBLOCKS][4][4]=

/*0*/ { 
        {{1,0,0,0}, /* # */
         {1,0,0,0}, /* # */
         {1,0,0,0}, /* # */
         {1,0,0,0}},/* # */

/*1*/   {{1,1,1,1}, /* # # # # */
         {0,0,0,0},
         {0,0,0,0},
         {0,0,0,0}},

/*2*/   {{0,1,0,0}, /*   #   */
         {1,1,1,0}, /* # # # */
         {0,0,0,0},
         {0,0,0,0}},

/*3*/   {{1,1,1,0}, /* # # # */
         {0,1,0,0}, /*   #   */
         {0,0,0,0},
         {0,0,0,0}},

/*4*/   {{0,1,0,0}, /*   #  */
         {1,1,0,0}, /* # #  */
         {0,1,0,0}, /*   #  */
         {0,0,0,0}},

/*5*/   {{1,0,0,0}, /* #   */
         {1,1,0,0}, /* # # */
         {1,0,0,0}, /* #   */
         {0,0,0,0}} ,

/*6*/   {{1,1,1,0}, /* # # # */
         {1,0,0,0}, /* #     */
         {0,0,0,0},
         {0,0,0,0}} ,

/*7*/   {{1,0,0,0}, /* #     */
         {1,1,1,0}, /* # # # */
         {0,0,0,0},
         {0,0,0,0}} ,

/*8*/   {{1,0,0,0}, /* #   */   
         {1,0,0,0}, /* #   */
         {1,1,0,0}, /* # # */
         {0,0,0,0}} ,

/*9*/   {{0,0,1,0}, /*     # */
         {1,1,1,0}, /* # # # */
         {0,0,0,0},
         {0,0,0,0}} ,

/*10*/  {{0,1,0,0}, /*   # */
         {0,1,0,0}, /*   # */
         {1,1,0,0}, /* # # */
         {0,0,0,0}} ,

/*11*/  {{1,1,0,0}, /* # # */
         {0,1,0,0}, /*   # */
         {0,1,0,0}, /*   # */
         {0,0,0,0}} ,

/*12*/  {{1,1,0,0}, /* # # */
         {1,0,0,0}, /* #   */
         {1,0,0,0}, /* #   */
         {0,0,0,0}} ,

/*13*/  {{1,1,1,0}, /* # # # */
         {0,0,1,0}, /*     # */
         {0,0,0,0},
         {0,0,0,0}} ,

/*14*/  {{1,1,0,0}, /* # # */
         {1,1,0,0}, /* # # */
         {0,0,0,0},
         {0,0,0,0}} ,

/*15*/  {{0,1,1,0}, /*   # # */
         {1,1,0,0}, /* # #   */
         {0,0,0,0},
         {0,0,0,0}} ,

/*16*/  {{1,1,0,0}, /* # #   */
         {0,1,1,0}, /*   # # */
         {0,0,0,0},
         {0,0,0,0}} ,

/*17*/  {{0,1,0,0}, /*   # */
         {1,1,0,0}, /* # # */
         {1,0,0,0}, /* #   */
         {0,0,0,0}} ,

/*18*/  {{1,0,0,0}, /* #   */
         {1,1,0,0}, /* # # */
         {0,1,0,0}, /*   # */
         {0,0,0,0}} 
};

const int NCOLORS = 10; // N colors for blocks

BITMAP* block_20x20_images[NCOLORS];

const int MIN_SPEED_LEVEL = 1;
const int MAX_SPEED_LEVEL = 80;
const int N_PREFILL_LINES = 6;

// creates block sprites (NCOLORS 20x20 bitmaps)

void create_block_bitmaps()
{
   for(int c=0; c<NCOLORS; c++) 
   {
      block_20x20_images[c] = create_bitmap(20,20);
      for(int xc=0; xc<20; xc++)
      for(int yc=0; yc<20; yc++)
      putpixel(block_20x20_images[c], xc, yc, 1 + block20x20_image[yc][xc]+c*15);
   }
}

/* free block-sprite memory */

void destroy_block_bitmaps()
{
   for(int c=0; c<NCOLORS; c++) destroy_bitmap(block_20x20_images[c]);
}

// generates palettes for changing background
// - makes copies of original palette, and changes the background part

void generate_background_palettes()
{
	for (int cc=0; cc<NBG_PALETTES; cc++)
	{
		for (int c=0; c<256; c++)
		{
			get_color(c, &background_palettes[cc][c]);
		}
	}

	for(int c=0; c<31; c++)
	{
		background_palettes[0][151+c].r = 10;
		background_palettes[0][151+c].g = 63-2*c;
		background_palettes[0][151+c].b = 2*c;

		background_palettes[1][151+c].r = 63-2*c;
		background_palettes[1][151+c].g = 10;
		background_palettes[1][151+c].b = 2*c;

		background_palettes[2][151+c].r = 63-c;
		background_palettes[2][151+c].g = 2*c;
		background_palettes[2][151+c].b = c;

		background_palettes[3][151+c].r = 10+c;
		background_palettes[3][151+c].g = 2*c;
		background_palettes[3][151+c].b = 40;

		background_palettes[4][151+c].r = 63-c;
		background_palettes[4][151+c].g = 2*c;
		background_palettes[4][151+c].b = 20+c;
	}
}

// change background, by changing the palette

void change_background()
{
	int newbgpal = bgpal;
   while(newbgpal == bgpal) newbgpal = rand()%NBG_PALETTES;
   bgpal = newbgpal;
	PALETTE p;
	get_palette(p);
	fade_from(p, background_palettes[bgpal], 10);
   set_mouse_sprite(NULL);   
}

void generate_tetris_game_palette(PALETTE &pal)
{
   // 213-218
   RGB black = {0,0,0};
   RGB white = {63,63,63};
   RGB red1 = {63,10,10};
   RGB green1 = {10,63,10};
   RGB grey1 = {15,15,15};
   RGB grey2 = {40,40,40};

   set_color(213, &black);
   set_color(214, &white);
	set_color(215, &green1);
	set_color(216, &red1);
	set_color(217, &grey1);
	set_color(218, &grey2);

   // create palette part for background

   RGB cc;
   for(int c=0; c<31; c++)
   {
      cc.r = 63-2*c;
      cc.g = 5;
      cc.b = 2*c;
      set_color(151+c, &cc);
   }

	generate_background_palettes();
	set_palette(background_palettes[bgpal]);
   set_mouse_sprite(NULL);
}

/* field struct, base struct fot the tetfield (20x20 pixels) */

struct field_struct
{
	field_struct() { occupied = false; color = 0; }	// basic constructor
	void operator=(const field_struct &s) { color = s.color; occupied = s.occupied; }	// assignment operator
	bool occupied;	// is this part of the tetfield occupied?
	int color;	// block color
};

/* tetfield constants */

const int FIELD_W = 12;       // tetfield width in blocks
const int FIELD_H = 20;       // tetfield height in blocks
const int NLINES = FIELD_H;   // number of lines in blocks

/* tetfield class
   contains FIELD_W * FIELD_H field_structs
	+ functions for drawing, clearing, inserting blocks etc.	*/

class tetfield
{
	friend class game;	// the game class is your pal

   public:
    tetfield(int _x, int _y);		// constructor
    ~tetfield();				// destructor
    void draw(BITMAP *bmp);	// draw on bitmap
    void clear();					// clear entire field
    void clear_line(int n);	// clear line N
    bool check_line(int n);	// check line N, return true on full
    void fill_random(int uptoline);	// random fill, from botton to X
    bool check_block(int bnr, int xpos, int ypos);	// check if a block will fit
    void putblock(int blocknr, int xpos, int ypos, int color);	// 'merge' a block with the field
    int check_for_full_lines();	// check for full lines, returns N full lines
    void lower_field(int index);	// lowers the field 1 line
        
   private:
    int x, y;	// draw position
    field_struct field[FIELD_H][FIELD_W];	// field structs
};

/* constructor */

tetfield::tetfield(int _x, int _y)
{
   x=_x;
   y=_y;
   clear();
}

/* destructor */

tetfield::~tetfield()
{
}

/* draws the field on bitmap */

void tetfield::draw(BITMAP *bmp)
{
   for(int xc=0; xc<FIELD_W; xc++)
   for(int yc=0; yc<FIELD_H; yc++)
   {
      if(field[yc][xc].occupied) 
      blit(block_20x20_images[(field[yc][xc].color)], bmp, 0, 0, x+2+xc*20, y+2+yc*20, 20, 20);
   }    
}

/* clears the entire field */

void tetfield::clear()
{
   for(int xc=0; xc<FIELD_W; xc++)
   for(int yc=0; yc<FIELD_H; yc++)
   {
      field[yc][xc].occupied = false;
      field[yc][xc].color = 0;
   }      
}

/* fills X lines randomly, starting at the bottom of the field. */

void tetfield::fill_random(int uptoline)
{
   for(int xc=0; xc<FIELD_W; xc++)
   for(int yc=FIELD_H-uptoline; yc<FIELD_H; yc++)
   {
      if(rand()%5 == 0)
      {   
         field[yc][xc].occupied = true;
         field[yc][xc].color = rand()%NCOLORS;
      }
   }      
}

/* clears line 'n' */

void tetfield::clear_line(int n)
{
   for(int xc=0; xc<FIELD_W; xc++) 
   {
		field[n][xc].occupied = false;
		field[n][xc].color = 0;
   }
}

/* returns true if line 'n' is full */

bool tetfield::check_line(int n)
{
   for(int c=0; c<FIELD_W; c++) 
      if (field[n][c].occupied == false) return false; // if not all occupied return false
   return true; // otherwise return true
}

/* checks if a block will fit in the tefield at given position */

bool tetfield::check_block(int bnr, int xpos, int ypos)
{
	// field-border check
   if(xpos<0 || xpos+blockwidth[bnr] > FIELD_W) return(false);
      if(ypos<0 || ypos+blockheight[bnr] > FIELD_H) return(false);

	// position check
   for(int xc=0; xc<4; xc++)      
   for(int yc=0; yc<4; yc++)   
   {      
      if((field[ypos+xc][xpos+yc].occupied == true) && (blockshapes[bnr][yc][xc] == 1)) return(false);
   }
   return(true);	// returns true if OK
}

/* 'merges a block with the field' */

void tetfield::putblock(int blocknr, int xpos, int ypos, int color)
{
   for(int xc=0; xc<4; xc++)      
   for(int yc=0; yc<4; yc++)   
   {      
      if(blockshapes[blocknr][yc][xc] == 1)
      {
         field[ypos+xc][xpos+yc].occupied = true;
         field[ypos+xc][xpos+yc].color = color;
      }
   }
}

/* lowers the field 1 line, using 'index' as the bottom line of moving part */

void tetfield::lower_field(int index)
{
   for(int yc=index; yc>=0; yc--)
   for(int xc=0; xc<FIELD_W; xc++) 
		field[yc+1][xc] = field[yc][xc];
   clear_line(0);
}

/* checks full lines, clears them, lowers field and then returns N lines
	line-numbers of full lines are stored in array i, then these lines
	are cleared and the field is lowered */

int tetfield::check_for_full_lines()
{
   int i[4] = {-1,-1,-1,-1};	// .. -1 = no line
   int ii=0;
  
   for(int c=0; c<FIELD_H; c++) if(check_line(c) == true) { i[ii]=c; ii++; }

   for(int c=0; c<4; c++)
	if(i[c] != -1)
	{
		clear_line(i[c]);
		lower_field(i[c]-1);
	}

   return(ii);
}

/* block class
	contains position info,
	number (usefull for lookup in blockshapes), color
	and reference to the tetfield.
	+ functions for drawing, moving, rotating the block etc.

	note: xpos and ypos are tetfield-cordinates (private)
*/

class block 
{
   friend class tetfield;	// let the tetfield be your pal

   public:
    block(tetfield &f);	// basic constructor
    ~block();	// destructor
    void operator=(const block &b);	// assign operator
    void init(int _xpos, int _ypos, int _nr, int _color);
    void random_block(int _xpos, int _ypos);	// make this a random block at x,y
    void draw(BITMAP *bmp, int x, int y);	
    void draw_onpos(BITMAP *bmp, int xoffs, int yoffs);
    void rotate();
    void move_left();
    void move_right();
    bool move_down();
    void drop();    
    bool check();
    
   private:   
    int xpos, ypos, nr, color;
    tetfield &myfield;
};

/* constructor */

block::block(tetfield &f)
: xpos(0), ypos(0), nr(0), color(0), myfield(f)
{
}

/* destructor */

block::~block()
{
}

/* block operator = */

void block::operator=(const block &b)
{
   xpos = b.xpos;
   ypos = b.ypos;
   nr = b.nr;
   color = b.color;
   myfield = b.myfield;
}

/* initialize block */

void block::init(int _xpos, int _ypos, int _nr, int _color)
{
   xpos = _xpos;
   ypos = _ypos;
   nr = _nr;
   color = _color;
}

/* block gets random shape + color */

void block::random_block(int _xpos, int _ypos)
{
   xpos = _xpos;
   ypos = _ypos;
   nr = rand()%NBLOCKS;
   color = rand()%NCOLORS;
}

/* draws block on bitmap at (x,y)
	(x and y are screen coordinates)
   - uses blockshape_array for lookup	*/

void block::draw(BITMAP *bmp, int x, int y)
{
   for(int xc=0; xc<4; xc++)
   for(int yc=0; yc<4; yc++)   
      if(blockshapes[nr][xc][yc] == 1) 
         blit(block_20x20_images[color], bmp, 0, 0, x+xc*20, y+yc*20, 20, 20);
}

/* draws block on bitmap at (x,y), starting from xoffs,yoffs
	xoffs and yoffs are screen cordinates,
	xpos and ypos are tetfield cordinates */

void block::draw_onpos(BITMAP *bmp, int xoffs, int yoffs)
{
   for(int xc=0; xc<4; xc++)
   for(int yc=0; yc<4; yc++)   
      if(blockshapes[nr][xc][yc] == 1) 
         blit(block_20x20_images[color], bmp, 0, 0, xoffs+(xpos+xc)*20, yoffs+(ypos+yc)*20, 20, 20);
}

/* moves block left, if possible */

void block::move_left()
{
   if(myfield.check_block(nr, xpos-1, ypos) == true) xpos--;
}

/* moves block right, if possible */

void block::move_right()
{
   if(myfield.check_block(nr, xpos+1, ypos) == true) xpos++;
}

/* moves block down, if possible returns true  */

bool block::move_down()
{
   if(myfield.check_block(nr, xpos, ypos+1) == true) { ypos++; return(true); }
      else return(false);
}

/* rotates the block, if possible */

void block::rotate()
{
   int newnr = blockrotation[nr]; 
   if(myfield.check_block(newnr, xpos, ypos) == true) nr = newnr;
}

/* drop the block */

void block::drop()
{
   while(move_down()){}
   myfield.putblock(nr, xpos, ypos, color);
}

bool block::check()
{
   return(myfield.check_block(nr, xpos, ypos));
}

/* game class
	*	everything is combined to a game.
	- tetfield + blocks
	- user keyboard input
	- score/statistics
*/

class game
{
   public:
    game(int initial_level, int randomfill, bool _auto_adjust);
    ~game();
    
    void update();
    void draw(BITMAP *bmp);
    void draw_interface(BITMAP *bmp);
    void keyb_input();
    bool quit() {  if(_quit == true) { _quit = false; return(true); } return(false); }
	 bool gameover() { return(_gameover); }
    void clear();
	 void newgame(int initial_level, int randomfill, bool _auto_adjust);

   private:
    bool _quit, _gameover;
    int score;
    int block_count;
    int score_inc;
    int lines;
    int initial_level, level, lcounter;
    tetfield field;
    block curblock, nextblock;
    bool auto_adjust_level;
    
    void new_block();
    void flush_block();
};

/* constructor
	initialize score, field, gameover/quit, and blocks	*/

game::game(int _initial_level, int randomfill, bool _auto_adjust):
_quit(false), _gameover(false),
score(0), block_count(0), score_inc(FIELD_H*2), lines(0), initial_level(_initial_level),
level(initial_level), lcounter(0),
field(40,40),
curblock(field), nextblock(field),
auto_adjust_level(_auto_adjust)
{
   if(randomfill > 0) field.fill_random(randomfill);
   nextblock.random_block(4,0);      
   curblock.random_block(4,0);   
}

game::~game()	// destructor, bla!
{
}

void game::new_block()
{
   if(nextblock.check() == false)
   {
		nextblock.drop();
   	_gameover = true;
   }
	   else
	   {
   	   curblock = nextblock;
	      nextblock.random_block(4,0);
	      block_count++;
	   }
}

/* game update
	move block down, handle score */

void game::update()
{
   if(lcounter == (MAX_SPEED_LEVEL+1-level))
   {
		if(curblock.move_down() == false) flush_block();
			else score_inc -= 2;
		lcounter = 0;
   }   
   lcounter++;

	// auto adjust the speed leve, when enabled, and then change background color
   
	if(auto_adjust_level)
   {
   	int orig_level = level;
   	level = initial_level + (lines/10);
      if(level > MAX_SPEED_LEVEL) level = MAX_SPEED_LEVEL;
		if(level != orig_level)	change_background();
	}
}

/* game draw function: draws field, current block, score & nextblock */

void game::draw(BITMAP *bmp)
{
   rectfill(bmp, 42, 42, 41+FIELD_W*20, 41+FIELD_H*20, 0);
   field.draw(bmp);
   curblock.draw_onpos(bmp,42,42);
	text_mode(0);
	FONT *F = (FONT*)tetdat[FONT2].dat;
   textprintf(bmp, F, 320+text_length(F,"lines: "), 290, 10, "%d ", lines);
   textprintf(bmp, F, 320+text_length(F,"score: "), 320, 10, "%d ", score);
   textprintf(bmp, F, 320+text_length(F,"played blocks: "), 350, 10, "%d ", block_count);
   textprintf(bmp, F, 320+text_length(F,"current: "), 380, 10, "%d ", score_inc);
   textprintf(bmp, F, 320+text_length(F,"level: "), 410, 10, "%d ", level);
   rectfill(bmp, 350, 180, 430, 260, 0);
   nextblock.draw(bmp,350,180);
   text_mode(-1);
}

void game::draw_interface(BITMAP *bmp)
{
   for(int c=0; c<7; c++)
		for(int cc=0; cc<5; cc++)
		{
	   	blit((BITMAP*)tetdat[BACKGR1].dat, bmp, 0, 0, c*100, cc*100, 100,100);
	   }

   rounded_box(bmp, 310, 15, 310+305, 15+145, 255, 0);
   masked_blit((BITMAP*)tetdat[BWLOGO].dat, bmp, 0, 0, 315, 20, 300, 140);
   rect(bmp, field.x, field.y, field.x+FIELD_W*20+3, field.y+FIELD_H*20+3, 255);
   rect(bmp, field.x+1, field.y+1, field.x+FIELD_W*20+2, field.y+FIELD_H*20+2, 255);
   rounded_box(bmp, 340, 170, 440, 270, 255, 0);
   rounded_box(bmp, 310, 280, 625, 440, 255, 0);
   text_mode(0);
   textprintf(bmp, (FONT*)tetdat[FONT2].dat, 320, 290, 255, "lines: ");
   textprintf(bmp, (FONT*)tetdat[FONT2].dat, 320, 320, 255, "score: ");
   textprintf(bmp, (FONT*)tetdat[FONT2].dat, 320, 350, 255, "played blocks: ");
   textprintf(bmp, (FONT*)tetdat[FONT2].dat, 320, 380, 255, "current:");
   textprintf(bmp, (FONT*)tetdat[FONT2].dat, 320, 410, 255, "level: ");
   text_mode(-1);
}

void game::keyb_input()
{
	if(key[KEY_ESC] || key[KEY_Q]) _quit = true;
	if(key[KEY_UP]) { curblock.rotate(); key[KEY_UP] = false; }
  	if(key[KEY_DOWN]) if(!curblock.move_down()) flush_block();
	if(key[KEY_LEFT]) curblock.move_left();
	if(key[KEY_RIGHT]) curblock.move_right();

	if(key[KEY_SPACE])
   {
		flush_block();
		key[KEY_SPACE] = false;
	}
}

void game::flush_block()
{
   curblock.drop();
   int N = field.check_for_full_lines();               
   lines += N;
   score += N*N*10;
   score += score_inc;
   new_block();
   lcounter = 0;
   score_inc = FIELD_H*2;
}

void game::clear()
{
	_gameover = false;
	_quit = false;
	score = block_count = lines = lcounter = 0;
   score_inc = FIELD_H*2;
   level = 10;
   field.clear();
}

void game::newgame(int _initial_level, int randomfill, bool _auto_adjust)
{
	clear();
	level = _initial_level;
   initial_level = _initial_level;
   auto_adjust_level = _auto_adjust;
   if(randomfill > 0) field.fill_random(randomfill);
   nextblock.random_block(4,0);      
   curblock.random_block(4,0);   
}

game tetgame(1,0, false);
bool quit = false;
bool playing = false;

/* GUI */

appearance blackwhite =
//{214, 213, 214, 214,   213, 214, 213, 213,   217, 218, 218, 217,  217, font };
{214, 213, 214, 214,   213, 214, 213, 213, 213, 214, 213, 215,  214, font, 217, 218, 218, 217};

appearance blackwhite2 =
{214, 213, 214, 214,   213, 214, 213, 213, 213, 214, 213, 216,  214, font, 217, 213, 218, 217};

/* game start dialog */

dialog start_dialog;

static_box bgbox(0, 0, 390, 230, &blackwhite, &start_dialog);
static_textbox newgametxt(45, 5, 300, 30, "Start new game", &blackwhite2,  &start_dialog);
checkbox randomfill(135, 70, 15, 50, &blackwhite, false, &start_dialog);
static_text randomfilltxt(55, 55, "Pre-fill randomly", &blackwhite,  &start_dialog);
checkbox adjust_level(135, 100, 15, 80, &blackwhite, false, &start_dialog);
static_text adjusttxt(55, 85, "Auto adjust level", &blackwhite,  &start_dialog);
static_text speedleveltxt(25, 125, "speedlevel:", &blackwhite,  &start_dialog);
hslider speedlevel_slider(255, 140, 135, 120, 230, 35, &blackwhite, MIN_SPEED_LEVEL, MAX_SPEED_LEVEL, 30, &start_dialog);
button startbutton(135, 180, 15, 160, 110, 30, "Start!", &blackwhite2,  &start_dialog);
button continuebutton(255, 180, 135, 160, 110, 30, "Continue", &blackwhite2,  &start_dialog);
button quitbutton(375, 180, 255, 160, 110, 30, "Quit", &blackwhite2,  &start_dialog);
static_text infotxt(40, 200, "Tetris version 1.1 by Steven Bouma", &blackwhite,  &start_dialog);

void startbutton_callback(button *b)
{
	playing = true;
	start_dialog.deactivate();
   if(continuebutton.disabled()) continuebutton.enable();
   int prefill = 0;
   if(randomfill.toggled()) prefill = N_PREFILL_LINES;
   bool a_adj = false;
   if(adjust_level.toggled()) a_adj = true;
	tetgame.newgame(speedlevel_slider.get_value(), prefill, a_adj);
}

void continuebutton_callback(button *b)
{
	playing = true;
	start_dialog.deactivate();
}

void quitbutton_callback(button *b)
{
	quit = true;
	start_dialog.deactivate();
	fade_out_range(5, 0, 150);
	fade_out_range(10, 150, 181);
	fade_out_range(5, 181, 255);
}

/* gameover dialog */

dialog gameover_dialog;
static_box gameoverbgbox(50, 80, 220, 70, &blackwhite, &gameover_dialog);
static_text gameovertxt(100, 90, "GAME OVER!", &blackwhite,  &gameover_dialog);
button OKbutton(100, 115, 120, 25, "OK", &blackwhite2, &gameover_dialog);

void OKbutton_callback(button *b)
{
	gameover_dialog.deactivate();
   PALETTE back_pal;
  	get_palette(back_pal);
	fade_out_range(10, 1, 150);
   rest(200);
   fade_in_range(back_pal, 10, 1, 150);
}

void init_gui()
{
	randomfill.untoggle();
	continuebutton.disable();
	OKbutton.set_callback(OKbutton_callback);
   startbutton.set_callback(startbutton_callback);
   continuebutton.set_callback(continuebutton_callback);
   quitbutton.set_callback(quitbutton_callback);
	blackwhite.f = (FONT*)tetdat[FONT1].dat;
	blackwhite2.f = (FONT*)tetdat[FONT2].dat;
   start_dialog.activate();
}

// timer

volatile int speed_counter = 0;

void inc_speed_counter()
{
   if(speed_counter>180) speed_counter = 0;
   speed_counter++;

} END_OF_FUNCTION(inc_speed_counter)

// keyboard timer

volatile int keyboard_counter = 0;

void keyboard_timer()
{
	keyboard_counter++;
   
} END_OF_FUNCTION(keyboard_timer)

// fps counter

volatile int fps = 0;      // frames per second
volatile int fpscount = 0; // frames per second counter

void fpscounter()    // frames per second counter
{
   fps = fpscount;   // fps = counted fps
   fpscount = 0;     // reset counter

} END_OF_FUNCTION(fpscounter);

// main

int main()
{
	// init

   srand(time(0));
   allegro_init();

	set_color_depth(8);
   if(set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0) != 0)
   {
      allegro_message("error setting graphics mode 640x480 8bpp %s", allegro_error);
      exit(1);
   }
   
   install_keyboard();
   install_mouse();
   install_timer();

   // timers

   LOCK_VARIABLE(speed_counter);
   LOCK_FUNCTION(inc_speed_counter);
   install_int_ex(inc_speed_counter, BPS_TO_TIMER(60));

   LOCK_VARIABLE(keyboard_counter);
   LOCK_FUNCTION(keyboard_timer);
   install_int_ex(keyboard_timer, BPS_TO_TIMER(25));

   LOCK_VARIABLE(fps);
   LOCK_VARIABLE(fpscount);
   LOCK_FUNCTION(fpscounter);
   install_int_ex(fpscounter, SECS_TO_TIMER(1));

   dbuf = create_bitmap(SCREEN_W, SCREEN_H);
   create_block_bitmaps();
	BITMAP* guibuf = create_sub_bitmap(dbuf,120, 20, 390, 230);
   text_mode(-1);

   tetdat = load_datafile("tetris.dat");
   if(tetdat == NULL)
   {
      allegro_message("error loading datafile: tetris.dat");
      exit(1);
   }

   PALLETE pal;
	set_palette((PALETTE)tetdat[TETGAMEPAL].dat);
   generate_tetris_game_palette(pal);
   init_gui();

   clear_to_color(dbuf, 255);
   tetgame.draw_interface(dbuf);

	/* main game loop */

   while(!quit)
   {

		// not playing game -> start new game dialog is now active
		while(start_dialog.active())
	   {
			start_dialog.update();
			start_dialog.draw(guibuf);
         show_mouse(dbuf);
	      blit(dbuf, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);  
         show_mouse(0);
		}

	   clear_to_color(dbuf, 255);
	   tetgame.draw_interface(dbuf);

      speed_counter = 0;

      // playing
      while(playing && !quit)
      {
			// game update
	      if(speed_counter>0)
	      {
	         tetgame.update();
	         speed_counter--;
      	}

         // keyboard input
	      if(keyboard_counter>0)
	      {
				tetgame.keyb_input();
				if(key[KEY_TAB]) save_bmp("tetrisshot.bmp", dbuf, background_palettes[bgpal]);
				keyboard_counter--;
      	}

			// graphs
			tetgame.draw(dbuf);
			// textprintf(dbuf, font, 10, 10, 255, "fps: %d", fps);
         
      	blit(dbuf, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
         fpscount++;

	      if(tetgame.quit())
	      {
	      	playing = false;
	      	start_dialog.activate();
            if(continuebutton.disabled()) continuebutton.enable();
	      	start_dialog.refresh();
	      }

	   	if(tetgame.gameover())
	   	{
	      	gameover_dialog.activate();
            continuebutton.disable();	// if game is over, no continue possible
	      	gameover_dialog.refresh();

	         while(gameover_dialog.active())
	         {
	     			gameover_dialog.update();
 					gameover_dialog.draw(dbuf);
		         show_mouse(dbuf);
			      blit(dbuf, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
		         show_mouse(0);
				}

			   clear_to_color(dbuf, 255);
			   tetgame.draw_interface(dbuf);
			   tetgame.draw(dbuf);
		      blit(dbuf, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
	       	playing = false;
	      	start_dialog.activate();
	      	start_dialog.refresh();
   	   	tetgame.clear();
	      }
	  	}
	}

	// clean-up

   destroy_block_bitmaps();
   destroy_bitmap(dbuf);
   destroy_bitmap(guibuf);
   unload_datafile(tetdat);

   return(0);
   
} END_OF_MAIN()



