#include "Character.h"

Character::Character(Rectangle hitBox, int type) : Actor(hitBox,true)
{
	jumpSpeed = 0.791f;
	readyToJumpTime = 100;
	readyToJumpCounter = readyToJumpTime;

	readyToThrowTime = 500;
	readyToThrowCounter = readyToThrowTime;

	walkSpeed = 1.7f;
	maxWalkSpeed = 0.22f;
	dir = FACING_LEFT;

	currentThrowPower=0;
	maxThrowPower = 0.87f;
	startingThrowPower = 0.4f;

	this->type = type;
	walking = false;
	currentlyThrowing = false;
	throwPowerIncreasing = true;
}

Character::~Character(void)
{
}

void Character::Reset(){
	Actor::Reset();
	currentThrowPower=0;
	walking = false;
	currentlyThrowing = false;
	throwPowerIncreasing = true;
}


void Character::Draw(BITMAP *dest, int screenx, int screeny){

	
	ModifyAnimation();

	if(dir==FACING_RIGHT)
		Actor::Draw(dest,screenx,screeny);
	else
		Actor::Draw(dest,screenx,screeny,true);
}

void Character::Update(int msecPassed, Map *map){
	Actor::Update(msecPassed,map);
	EnforceMaxXSpeed(maxWalkSpeed);

	if(IsOnGround()){
		if(readyToJumpCounter>0){
			readyToJumpCounter -= msecPassed;
			if(readyToJumpCounter<0)
				readyToJumpCounter=0;
		}
	}
	else{
		readyToJumpCounter = readyToJumpTime;
	}

	if(readyToThrowCounter>0){
		readyToThrowCounter-=msecPassed;
		if(readyToThrowCounter<0)
			readyToThrowCounter=0;
	}
}

void Character::ClearForces(){
	Actor::ClearForces();
	walking = false;		//reset walking to false here, so that it can be set back to true if WalkLeft or WalkRight is called
}

void Character::Jump(int msecPassed){
	if(readyToJumpCounter==0){
		//ModifyVel(0,-jumpSpeed);
		ApplyForce(0,-jumpSpeed/msecPassed);
		readyToJumpCounter = readyToJumpTime;
	}
}

void Character::WalkLeft(){
	if(!currentlyThrowing){	//can't walk while charging a throw
		ApplyForce( - walkSpeed*GetCurrentSurfaceFriction() , 0);
		dir = FACING_LEFT;
		walking = true;

	}
}

void Character::WalkRight(){
	if(!currentlyThrowing){	//can't walk while charging a throw
		ApplyForce(  walkSpeed*GetCurrentSurfaceFriction() , 0);
		dir=FACING_RIGHT;
		walking = true;
	}
}

void Character::SetPosition(float x, float y, int dir) { 
	Actor::SetPosition(x,y); 
	if(dir!=-1)
		this->dir = dir; 
}

void Character::IncreaseThrowPower(int msecPassed){
      if(readyToThrowCounter==0){
		currentlyThrowing = true;
		if(currentThrowPower==0)
			currentThrowPower=startingThrowPower;

		if(throwPowerIncreasing){
			currentThrowPower += 0.0010 * msecPassed;
			if(currentThrowPower > maxThrowPower){
				currentThrowPower = maxThrowPower;
				throwPowerIncreasing = false;
			}
		}
		else{
			currentThrowPower -= 0.0010 * msecPassed;
			if(currentThrowPower <=startingThrowPower){
				currentThrowPower = startingThrowPower;
				throwPowerIncreasing = true;
			}
		}
	}
}

void Character::StartThrowingProjectile(int msecPassed, ProjectileList *projList){
	ThrowProjectile(msecPassed,projList);
	throwPowerIncreasing = true;
}

void Character::ThrowProjectile(int msecPassed, ProjectileList *projList){
	if(readyToThrowCounter==0){
		currentlyThrowing = false;
		readyToThrowCounter = readyToThrowTime;
		

		Projectile *p = new Projectile(MakeRectangle(-4,-4,26,22),type,Projectile::PRESENT);
		p->AddAnimation((AnimDef*)ResourceManager::inst()->GetResource("presentAnim"));
		projList->AddProjectile( p );

		if(dir==FACING_RIGHT)
			p->Shoot(msecPassed, x+ThrowLocationX(), y+ThrowLocationY(), 27*M_PI/16, currentThrowPower);
		else
			p->Shoot(msecPassed, x+ThrowLocationX(), y+ThrowLocationY(), 21*M_PI/16, currentThrowPower);

		currentThrowPower=0;
	}
}

float Character::GetCameraFocusX(){
	if(dir==FACING_LEFT)
		return Actor::GetCameraFocusX() - Constants::BACKGROUND_WIDTH*1/4;
	else
		return Actor::GetCameraFocusX() + Constants::BACKGROUND_WIDTH*1/4;
}

float Character::GetCameraFocusY(){
	return Actor::GetCameraFocusY();
}

void Character::CheckCollisionWithProjectiles(ProjectileList *projList, int msecPassed){
	if(projList){
		std::list<Projectile*> *theList = projList->GetList();

		std::list<Projectile*>::iterator it;

		for(it = theList->begin() ; it!=theList->end(); it++){
			Projectile *proj = (*it);
			if(proj && proj->GetOwner() != this->type){
				if(this->CheckCollision(proj, msecPassed)){
					this->HitByProj(proj, msecPassed);
				}
			}
		}
	}
}
