#include <allegro.h>
#include <math.h>
#include <cstdlib>

#include "Constants.h"
#include "Log.h"
#include "ResourceManager.h"
#include "Image.h"
#include "Song.h"
#include "Font.h"
#include "AnimDef.h"
#include "Anim.h"
#include "TileDef.h"
#include "Tile.h"
#include "TileSet.h"
#include "Map.h"
#include "Camera.h"
#include "Actor.h"
#include "MapDecorationDef.h"
#include "MapDecoration.h"
#include "Projectile.h"
#include "ProjectileList.h"
#include "Character.h"
#include "MapEditor.h"
#include "EnimDef.h"
#include "Enemy.h"
#include "Player.h"


void Tick();
bool InitAllegro();
void Cleanup();
bool InitGraphics();
bool LoadResources();
void InitializeAnimationDefinitions();
void InitializeTileDefinitions();
void InitializeTileSet();
void InitializeMap();
void InitializeEnemies();
void InitializeMapDecorations();
void InitializeGame();
void Update(int msecPassed);
void Draw(int msecPassed);
void CheckInput(int msecPassed);
void StartLevel();

volatile int gameTimeInMsec=0;
bool quitGame = false;
BITMAP *backgr = 0;
float xpos=30;
TileSet *tileSet;
Map *mainMap;
Camera *camera;
Player *player;
ProjectileList *projList;

MapEditor *mapEditor;
bool editingMap = false;
int level=1;
bool victory = false;

int frameCounter=0;
float fpsDisplay=0;
int lastFpsDisplayUpdateTime=0;



void Tick(){
	gameTimeInMsec++;
}
END_OF_FUNCTION(Tick);

int main(void)
{
        InitAllegro();
		InitGraphics();
		LoadResources();

		InitializeGame();
		int lastUpdateTime=gameTimeInMsec;
		int msecPassed = 0;

		Log::inst()->logLine("=====DONE PRE-GAME INITIALIZATION. STARTING GAME=====");

		Song *song = (Song*)ResourceManager::inst()->GetResource("comingToTownSong");
		if(song) song->Play(true);

	
		level=1;
		StartLevel();
		player->StartLevel(mainMap,level);
		//player->SetAmmo(5);

		while(quitGame==false)
		{
			//while(gameTimeInMsec==lastUpdateTime);	//wait here until at least 1 milisecond has passed
			//if(gameTimeInMsec > lastUpdateTime){
				msecPassed = gameTimeInMsec - lastUpdateTime;
				if(msecPassed<16){
					continue;
				}

				lastUpdateTime=gameTimeInMsec;


				Update(msecPassed);
				Draw(msecPassed);
				

				frameCounter++;
				if(gameTimeInMsec >= lastFpsDisplayUpdateTime+1000){
					fpsDisplay = frameCounter;
					frameCounter = 0;
					lastFpsDisplayUpdateTime = gameTimeInMsec;
				}
			//}

	
		}
		Log::inst()->logLine("=====DONE GAME. STARTING CLEANUP=====");

		Cleanup();

        return 0;
}
END_OF_MAIN()



bool InitAllegro(){
	Log::inst()->logLine("Entering initAllegro",1);
	allegro_init();

	set_config_file("settings.ini");

	Log::inst()->logLine("installing keyboard");
    install_keyboard();
	Log::inst()->logLine("installing timer");
    install_timer();
	install_int_ex(Tick,MSEC_TO_TIMER(1));

	Log::inst()->logLine("installing mouse");
    install_mouse();

	Log::inst()->logLine("setting graphics mode");
	int depth;
	if ((depth = desktop_color_depth()) != 0) {
		set_color_depth(depth);
     }

	
	int windowed;
	const char* val = get_config_string("graphics", "windowed", "TRUE");
	if (ustricmp(val, "FALSE") == 0)
		windowed = GFX_AUTODETECT_FULLSCREEN;
	else
		windowed = GFX_AUTODETECT_WINDOWED;

	if(set_gfx_mode(windowed,Constants::SCREEN_WIDTH,Constants::SCREEN_HEIGHT,0,0)<0)
    {
		Log::inst()->logLine("Exiting initAllegro. Error setting up graphics mode!",-1);
		return false;      
    }    

	Log::inst()->logLine("installing sound");
    if(install_sound(DIGI_AUTODETECT,MIDI_AUTODETECT,NULL)<0)
    {
		Log::inst()->logLine("Exiting initAllegro. Error setting up sound!",-1);
		return false;
	}
      
	Log::inst()->logLine("Exiting initAllegro successfully",-1);
	return true;
}

void Cleanup(){
	delete tileSet;
	delete mainMap;
	delete camera;
	delete mapEditor;
	delete projList;
}

bool InitGraphics(){
	Log::inst()->logLine("Entering initGraphics",1);
	
	Log::inst()->logLine("creating background bitmap");

#ifdef USE_VIDEO_BITMAPS
	backgr = create_video_bitmap(Constants::BACKGROUND_WIDTH, Constants::BACKGROUND_HEIGHT);
#else
	backgr = create_bitmap(Constants::BACKGROUND_WIDTH, Constants::BACKGROUND_HEIGHT);
#endif

	Image *img = new Image("name","resources/images/tomato1/frame1.bmp");
	ResourceManager::inst()->AddResource(img);

	Log::inst()->logLine("Exiting initGraphics successfully",-1);
	return true;
}



bool LoadResources(){
	ResourceManager::inst()->LoadResourceFile("resources/res.txt");

	InitializeAnimationDefinitions();
	InitializeTileDefinitions();
	

	

	return true;
}

void InitializeAnimationDefinitions(){
	Log::inst()->logLine("Entering InitializeAnimationDefinitions",1);

	AnimDef *animDef = new AnimDef("elf1Anim");
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf1-1Image"),100,0,0) );
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf1-2Image"),100,0,0) );
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf1-3Image"),100,0,0) );
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf1-4Image"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("elf1HatlessAnim");
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("elf1-hatlessImage"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("elf2Anim");
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf2-1Image"),100,0,0) );
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf2-2Image"),100,0,0) );
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf2-3Image"),100,0,0) );
	animDef->AddFrame( MakeFrame((Image*)ResourceManager::inst()->GetResource("elf2-4Image"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("elf2HatlessAnim");
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("elf2-hatlessImage"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("hatAnim");
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("elf1-hatImage"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("mainWalking");
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainWalking1Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainWalking2Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainWalking3Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainWalking4Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainWalking5Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainWalking6Image"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("mainThrowing");
	//animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainThrowing1Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainThrowing2Image"),100,0,0) );
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("mainThrowing3Image"),50,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("presentAnim");
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("presentImage"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	animDef = new AnimDef("treeAnim");
	animDef->AddFrame( MakeFrame( (Image*)ResourceManager::inst()->GetResource("tree1Image"),100,0,0) );
	ResourceManager::inst()->AddResource(animDef);

	Log::inst()->logLine("Exiting InitializeAnimationDefinitions successfully",-1);
}

void InitializeTileDefinitions(){
	TileDef *tileDef;

	tileDef = new TileDef("tile1-1Def", "tile1-1Image", MakeRectangle(0,0,32,16),true,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-2Def", "tile1-2Image", MakeRectangle(0,0,32,32),false,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-3Def", "tile1-3Image", MakeRectangle(0,0,32,32),false,false,false,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-4Def", "tile1-4Image", MakeRectangle(0,0,20,32),false,false,true,false,true);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-5Def", "tile1-5Image", MakeRectangle(12,0,20,32),false,false,true,true,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-6Def", "tile1-6Image", MakeRectangle(0,0,16,32),false,false,false,false,true);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-7Def", "tile1-7Image", MakeRectangle(16,0,16,32),false,false,false,true,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-8Def", "tile1-8Image", MakeRectangle(0,0,32,32),false,false,false,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-9Def", "tile1-9Image", MakeRectangle(0,0,32,32),false,false,false,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-10Def", "tile1-10Image", MakeRectangle(0,0,32,32),false,false,false,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-11Def", "tile1-11Image", MakeRectangle(0,0,32,32),false,false,false,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-12Def", "tile1-12Image", MakeRectangle(0,0,32,32),false,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-13Def", "tile1-13Image", MakeRectangle(0,0,32,32),false,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-14Def", "tile1-14Image", MakeRectangle(0,0,32,32),false,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-15Def", "tile1-15Image", MakeRectangle(0,0,32,32),false,true,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-17Def", "tile1-17Image", MakeRectangle(0,0,22,16),true,false,true,false,true);
	ResourceManager::inst()->AddResource(tileDef);
	
	tileDef = new TileDef("tile1-18Def", "tile1-18Image", MakeRectangle(10,0,22,16),true,false,true,true,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-19Def", "tile1-19Image", MakeRectangle(0,0,32,32),false,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-20Def", "tile1-20Image", MakeRectangle(0,0,32,32),false,false,true,false,false);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-21Def", "tile1-21Image", MakeRectangle(16,0,16,32),false,true,true,false,true);
	ResourceManager::inst()->AddResource(tileDef);

	tileDef = new TileDef("tile1-22Def", "tile1-22Image", MakeRectangle(0,0,16,32),false,true,true,true,false);
	ResourceManager::inst()->AddResource(tileDef);


}

void InitializeGame(){
	InitializeEnemies();
	InitializeMapDecorations();
	InitializeTileSet();
	InitializeMap();

	mapEditor = new MapEditor(mainMap);
	camera = new Camera();
	player = new Player(MakeRectangle(13,5,60-13*2,100-5-2));
	player->SetPosition(50,20,Character::FACING_RIGHT);
	player->AddAnimation( (AnimDef*)ResourceManager::inst()->GetResource("mainWalking") );
	player->AddAnimation( (AnimDef*)ResourceManager::inst()->GetResource("mainThrowing") );
	

	projList = new ProjectileList();
}

void InitializeTileSet(){
		//testTile = new Tile((TileDef*)ResourceManager::inst()->GetResource("tile1-2Def"));
	tileSet = new TileSet();
	tileSet->AddTile(Tile("1",(TileDef*)ResourceManager::inst()->GetResource("tile1-1Def")));
	tileSet->AddTile(Tile("2",(TileDef*)ResourceManager::inst()->GetResource("tile1-2Def")));
	tileSet->AddTile(Tile("3",(TileDef*)ResourceManager::inst()->GetResource("tile1-3Def")));
	tileSet->AddTile(Tile("4",(TileDef*)ResourceManager::inst()->GetResource("tile1-4Def")));
	tileSet->AddTile(Tile("5",(TileDef*)ResourceManager::inst()->GetResource("tile1-5Def")));
	tileSet->AddTile(Tile("6",(TileDef*)ResourceManager::inst()->GetResource("tile1-6Def")));
	tileSet->AddTile(Tile("7",(TileDef*)ResourceManager::inst()->GetResource("tile1-7Def")));
	tileSet->AddTile(Tile("8",(TileDef*)ResourceManager::inst()->GetResource("tile1-8Def")));
	tileSet->AddTile(Tile("9",(TileDef*)ResourceManager::inst()->GetResource("tile1-9Def")));
	tileSet->AddTile(Tile("10",(TileDef*)ResourceManager::inst()->GetResource("tile1-10Def")));
	tileSet->AddTile(Tile("11",(TileDef*)ResourceManager::inst()->GetResource("tile1-11Def")));
	tileSet->AddTile(Tile("12",(TileDef*)ResourceManager::inst()->GetResource("tile1-12Def")));
	tileSet->AddTile(Tile("13",(TileDef*)ResourceManager::inst()->GetResource("tile1-13Def")));
	tileSet->AddTile(Tile("14",(TileDef*)ResourceManager::inst()->GetResource("tile1-14Def")));
	tileSet->AddTile(Tile("15",(TileDef*)ResourceManager::inst()->GetResource("tile1-15Def")));
	tileSet->AddTile(Tile("17",(TileDef*)ResourceManager::inst()->GetResource("tile1-17Def")));
	tileSet->AddTile(Tile("18",(TileDef*)ResourceManager::inst()->GetResource("tile1-18Def")));
	tileSet->AddTile(Tile("19",(TileDef*)ResourceManager::inst()->GetResource("tile1-19Def")));
	tileSet->AddTile(Tile("20",(TileDef*)ResourceManager::inst()->GetResource("tile1-20Def")));
	tileSet->AddTile(Tile("21",(TileDef*)ResourceManager::inst()->GetResource("tile1-21Def")));
	tileSet->AddTile(Tile("22",(TileDef*)ResourceManager::inst()->GetResource("tile1-22Def")));
	
	tileSet->AddEnimDef((EnimDef*)ResourceManager::inst()->GetResource("JumpingElfEnimDef"));
	tileSet->AddEnimDef((EnimDef*)ResourceManager::inst()->GetResource("WalkingElfEnimDef"));
	tileSet->AddEnimDef((EnimDef*)ResourceManager::inst()->GetResource("SuperJumpingElfEnimDef"));
	tileSet->AddDecorationDef((MapDecorationDef*)ResourceManager::inst()->GetResource("treeDecoration"));
}

void InitializeEnemies(){
	EnimDef *def;

	def = new EnimDef("JumpingElfEnimDef");
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("elf1Anim"));
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("elf1HatlessAnim"));
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("hatAnim"));
	def->hitBox = MakeRectangle(10,3,38,94);
	def->type = EnimDef::ELF;
	def->jumps = true;
	def->throws = false;
	def->walks = true;
	def->jumpSpeed = 0.45f;
	def->walkSpeed = 1.7f;
	def->maxWalkSpeed = 0.17f;
	def->readyToJumpTime = 	140;
	def->readyToThrowTime = 500;
	def->maxThrowPower = 0.8f;
	ResourceManager::inst()->AddResource(def);

	def = new EnimDef("WalkingElfEnimDef");
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("elf2Anim"));
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("elf2HatlessAnim"));
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("hatAnim"));
	def->hitBox = MakeRectangle(10,3,38,94);
	def->type = EnimDef::ELF;
	def->jumps = false;
	def->throws = false;
	def->walks = true;
	def->jumpSpeed = 0.0f;
	def->walkSpeed = 1.7f;
	def->maxWalkSpeed = 0.20f;
	def->readyToJumpTime = 	140;
	def->readyToThrowTime = 500;
	def->maxThrowPower = 0.8f;
	ResourceManager::inst()->AddResource(def);

	def = new EnimDef("SuperJumpingElfEnimDef");
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("elf1Anim"));
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("elf1HatlessAnim"));
	def->AddAdnimDef((AnimDef*)ResourceManager::inst()->GetResource("hatAnim"));
	def->hitBox = MakeRectangle(10,3,38,94);
	def->type = EnimDef::ELF;
	def->jumps = true;
	def->throws = false;
	def->walks = true;
	def->jumpSpeed = 0.7f;
	def->walkSpeed = 1.7f;
	def->maxWalkSpeed = 0.20f;
	def->readyToJumpTime = 	140;
	def->readyToThrowTime = 500;
	def->maxThrowPower = 0.8f;
	ResourceManager::inst()->AddResource(def);
	

}

void InitializeMapDecorations(){
	MapDecorationDef *def;
		
	def = new MapDecorationDef("treeDecoration",
												 MakeRectangle(0,0,150,200),
												 (AnimDef*)ResourceManager::inst()->GetResource("treeAnim"),
												 false,
												 false);
	ResourceManager::inst()->AddResource(def);

	def = new MapDecorationDef("hatDecoration",
							   MakeRectangle(0,0,0,0),
							   (AnimDef*)ResourceManager::inst()->GetResource("hatAnim"),
							   true,
							   true);
	ResourceManager::inst()->AddResource(def);

}

void InitializeMap(){
	mainMap = new Map();
	mainMap->SetTileSet(tileSet);
	
}

void Update(int msecPassed)
{
	//Log::inst()->logLine("Entering Update",1);
	if(editingMap){
		bool done = mapEditor->Update(msecPassed,camera);
		camera->CentreOn(mapEditor->GetCameraFocusX(),mapEditor->GetCameraFocusY());

		if(done)
			editingMap=false;
	}
	else if(victory){
		if(key[KEY_ENTER]){
			quitGame = true;
		}
	}
	else{
		

		//CLEAR FORCES FOR ALL ACTORS
		//Log::inst()->logLine("Clearing Forces");
		player->ClearForces();
		mainMap->ClearForces();
		projList->ClearForces();
		

		//Log::inst()->logLine("Checking Input");
		CheckInput(msecPassed);

		//Log::inst()->logLine("Updateing Dudes");
		player->ApplyForce(0,GRAVITY);
		player->Update(msecPassed,mainMap);
		mainMap->CheckCollsisionWithPickups(player, msecPassed);
		mainMap->CheckEnemyCollisionWithProjectiles(projList, msecPassed);
		mainMap->Update(msecPassed);
		projList->Update(msecPassed,mainMap);

		//Log::inst()->logLine("Checking Collision");

		player->CheckCollisionWithProjectiles(projList, msecPassed);
		Enemy *enHittingPlayer = mainMap->CheckCollisionWithEnemy(player,msecPassed);
		if(enHittingPlayer!=0){
			player->HitByEnemy(enHittingPlayer,msecPassed);
		}

		if(player->ShouldStartNextLevel()){
			projList->DeleteAll();
			level++;
			
			if(level>3){
				victory = true;
			}
			else{
				StartLevel();
				player->StartLevel(mainMap,level);
			}
		}
		

		//Log::inst()->logLine("Moving Camera");
		camera->CentreOn(player->GetCameraFocusX(), player->GetCameraFocusY());

		xpos += 0.2f * msecPassed;
	}
	camera->Update(msecPassed);
	//Log::inst()->logLine("Exiting Update",-1);
}

void Draw(int msecPassed)
{
	//Log::inst()->logLine("Entering Draw",1);
	//clear_bitmap(backgr);
	clear_to_color(backgr,makecol(171, 205, 239));

	//img->Draw(backgr,200,200);
	//((Image*)ResourceManager::inst()->GetResource("image1Name"))->Draw(backgr,200,200);
	//Log::inst()->logLine("Drawing Map");
	mainMap->Draw(backgr, camera->GetX(),camera->GetY());

	player->DrawHUD(backgr);

	//Log::inst()->logLine("Drawing Player");
	player->Draw(backgr,camera->GetX(),camera->GetY());
	//Log::inst()->logLine("Drawing Projectiles");
	projList->Draw(backgr,camera->GetX(),camera->GetY());

	mainMap->DrawForeground(backgr, camera->GetX(),camera->GetY());

	if(victory){
		FONT *f1 = ((Font*)ResourceManager::inst()->GetResource("goudy14Font"))->GetFont();
		FONT *f2 = ((Font*)ResourceManager::inst()->GetResource("goudy10Font"))->GetFont();
		
		rectfill(backgr,40,120,Constants::BACKGROUND_WIDTH-40,313,makecol(40,40,40));
		if(f1){
			textprintf_centre_ex(backgr,f1,Constants::BACKGROUND_WIDTH/2,150,makecol(100,200,200),-1,"Merry Christmas!");
			textprintf_centre_ex(backgr,f1,Constants::BACKGROUND_WIDTH/2,200,makecol(100,200,200),-1,"You Have Saved All the Elfs!");
		}
		if(f2)
			textprintf_centre_ex(backgr,f2,Constants::BACKGROUND_WIDTH/2,260,makecol(60,130,230),-1,"Hit ENTER to quit");		
		
	}
	

	if(editingMap)
		mapEditor->Draw(backgr,camera->GetX(),camera->GetY());

	//textprintf_ex(backgr,font,10,10,makecol(200,200,200),0,"FPS: %.0f",fpsDisplay);
	

	//Log::inst()->logLine("Blitting to Screen");
	if(Constants::SCREEN_HEIGHT == Constants::BACKGROUND_HEIGHT && Constants::SCREEN_WIDTH==Constants::BACKGROUND_WIDTH)
		blit(backgr,screen,0,0,0,0,backgr->w,backgr->h);
	else
		stretch_blit(backgr,screen,0,0,backgr->w,backgr->h,0,0,SCREEN_W,SCREEN_H);

	//Log::inst()->logLine("Exiting Draw",-1);
}

void CheckInput(int msecPassed)
{
	static bool spaceWasPressed=false;
	if(key[KEY_ESC]){
		quitGame = true;
	}

	if(key[KEY_LEFT]){
		//player->ApplyForce(-0.0005,0);
		player->WalkLeft();
		//camera->MoveBy( - 0.1f * msecPassed, 0);
	}
	else if(key[KEY_RIGHT]){
		player->WalkRight();
		//player->ApplyForce(0.0005,0);
		//camera->MoveBy( 0.1f * msecPassed, 0);
	}

	if(key[KEY_UP]){
		player->Jump(msecPassed);
	}


	if(key[KEY_SPACE]){
		if(player->HasAmmo()){
			spaceWasPressed=true;
			player->IncreaseThrowPower(msecPassed);
		}
	}
	else if(spaceWasPressed){
		player->StartThrowingProjectile(msecPassed, projList);
		spaceWasPressed=false;
	}

	if(key[KEY_E] && !editingMap){
		while(key[KEY_E]);
		editingMap = true;
		mainMap->ResetEnemies();
	}
}

void StartLevel(){
	switch(level){
		case 1:
			mainMap->Load("resources/maps/map1.txt");
			break;
		case 2:
			mainMap->Load("resources/maps/map2.txt");
			break;
		case 3:
			mainMap->Load("resources/maps/map3.txt");
			break;
			
	}
}