#include "Game.h"
#include "InGame.h"
#include "LightningArc.h"
#include "CommonFunctions.h"
#include "music.h"
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

extern int WIDTH,HEIGHT;
extern BITMAP* buffer;
extern int TILE;
extern int RATE;

extern GameResults gameResults;

static struct {
  int x, y;
  int framestep;
  int frame;
  int dead;
  int talking;
} boy;

static struct {
  int x, y;
  int direction;
  int framestep;
  int frame;
  int talking;
  int health;
  int damage;
  int dead;
  int score;
  int cross_x, cross_y;
  int light_x, light_y;
  int energy;
  int num_plane_hits; //Hits with 1 lightning strike
} girl;

static BITMAP *girl_bmp, *girl_bmp_f, *girld_bmp, *girld_bmp_f;
static BITMAP *boy_bmp, *boyd_bmp;
static BITMAP *portraits;
static BITMAP *tiles1;
static BITMAP *plane_bmp, *plane_drop, *plane_fall, *plane_explode;
static BITMAP *plane_bmp_f, *plane_drop_f, *plane_fall_f, *plane_explode_f;
static BITMAP *bomb_bmp;
static BITMAP *symbols, *stats_bmp;
static BITMAP *gameover_bmp;
static BITMAP *black;
static FONT *fnt_sans, *fnt_chris;

static struct {
  SAMPLE* planehit;
  SAMPLE* planecrash;
  SAMPLE* bomb;
  SAMPLE* lightready;
  SAMPLE* lightfire;
  SAMPLE* bombdrop;
  SAMPLE* ouch;
  SAMPLE* ultracombo;
} sounds;

static void InGame_cleanup()
{
  //music_stop();
  destroy_bitmap(girl_bmp);
  destroy_bitmap(girl_bmp_f);
  destroy_bitmap(girld_bmp);
  destroy_bitmap(girld_bmp_f);
  destroy_bitmap(boy_bmp);
  destroy_bitmap(boyd_bmp);
  destroy_bitmap(tiles1);
  destroy_bitmap(portraits);
  destroy_bitmap(plane_bmp);
  destroy_bitmap(plane_drop);
  destroy_bitmap(plane_fall);
  destroy_bitmap(plane_explode);
  destroy_bitmap(bomb_bmp);
  destroy_bitmap(symbols);
  destroy_bitmap(stats_bmp);
  destroy_bitmap(gameover_bmp);
  destroy_bitmap(black);
  destroy_font(fnt_sans);
  destroy_font(fnt_chris);
  destroy_sample(sounds.planehit);
  destroy_sample(sounds.planecrash);
  destroy_sample(sounds.bomb);
  destroy_sample(sounds.lightready);
  destroy_sample(sounds.lightfire);
  destroy_sample(sounds.bombdrop);
  destroy_sample(sounds.ouch);
  destroy_sample(sounds.ultracombo);
}

typedef struct {
  char line[81];
  int type;
} DialogueSelect;

static struct {
  int active;
  int part;
  int step;
  DialogueSelect selections[5];
  int who;
  int type;
  const char* text;
  int selected;
} dialogue;

static int armageddon;

static struct {
  int counter;
  int triggerTrigger;
  int trigger;
} planesTrigger;    

typedef struct {
  int x, y;
  int initial_y;
  int direction;
  int framestep;
  int frame;
  int dead;
  int bombready;
} Plane;

static const int MAX_PLANES = 6;
static Plane planes[6];
static int num_planes;

typedef struct {
  int x, y;
  int initial_y;
  int explode;
  int framestep;
  int frame;
} Bomb;

static const int MAX_BOMBS = 50;
static Bomb bombs[50];
static int num_bombs;

typedef struct {
  int color;
  int x, y; //fixed point
  int dx, dy; //fixed point
  int size;
} Particle;

#define PARTICLES 30
typedef struct {
  Particle particles[PARTICLES];
} Blood;

static const int MAX_BLOODS = 20;
static Blood bloods[20];
static int num_bloods;

static const int MAX_ENERGY = 100;

static int gameover;
static int gameover_step;

void InGame_init()
{
  int i;
  BITMAP* tmp;

  DUH* sfx = music_load("media/sfx/sounds.xm"); //Generate Sounds from XM
  sounds.planehit = music_create_sample(sfx,0);
  sounds.planecrash = music_create_sample(sfx,1);
  sounds.lightready = music_create_sample(sfx,2);
  sounds.lightfire = music_create_sample(sfx,3);
  sounds.bomb = music_create_sample(sfx,4);
  sounds.bombdrop = music_create_sample(sfx,5);
  sounds.ouch = music_create_sample(sfx,6);
  sounds.ultracombo = music_create_sample(sfx,7);
  music_unload(sfx);

  fnt_sans = load_font("media/font/sans.pcx",NULL,NULL);
  fnt_chris = load_font("media/font/chris.pcx",NULL,NULL);
  tiles1 = load_pcx("media/gfx/tiles1.pcx",NULL);
  boy_bmp = load_pcx("media/gfx/boy1.pcx",NULL);
  boyd_bmp = load_pcx("media/gfx/boy1_die.pcx",NULL);
  girl_bmp = load_pcx("media/gfx/girl.pcx",NULL);
  girl_bmp_f = create_bitmap(girl_bmp->w,girl_bmp->h);
  girld_bmp = load_pcx("media/gfx/girl_die.pcx",NULL);
  girld_bmp_f = create_bitmap(girld_bmp->w,girld_bmp->h);
  portraits = load_pcx("media/gfx/portraits.pcx",NULL);
  plane_bmp = load_pcx("media/gfx/plane.pcx",NULL);
  plane_bmp_f = create_bitmap(plane_bmp->w,plane_bmp->h);
  plane_drop = load_pcx("media/gfx/plane_drop.pcx",NULL);
  plane_drop_f = create_bitmap(plane_drop->w,plane_drop->h);
  plane_fall = load_pcx("media/gfx/plane_fall.pcx",NULL);
  plane_fall_f = create_bitmap(plane_fall->w,plane_fall->h);
  plane_explode = load_pcx("media/gfx/plane_explode.pcx",NULL);
  plane_explode_f = create_bitmap(plane_explode->w,plane_explode->h);
  bomb_bmp = load_pcx("media/gfx/bomb.pcx",NULL);
  symbols = load_pcx("media/gfx/symbols.pcx",NULL);
  stats_bmp = load_pcx("media/gfx/stats.pcx",NULL);
  gameover_bmp = load_pcx("media/gfx/gameover.pcx",NULL);

  black = create_bitmap(WIDTH,HEIGHT);
  clear(black);
  tmp = create_bitmap(TILE,TILE);
  clear_to_color(girl_bmp_f,makecol16(255,0,255));
  //32x32 transformations
  for (i=0; i<girl_bmp->w; i+=TILE)
  {
    blit(girl_bmp,tmp,i,0,0,0,TILE,TILE);
    draw_sprite_h_flip(girl_bmp_f,tmp,i,0);
  }
  clear_to_color(girld_bmp_f,makecol16(255,0,255));
  for (i=0; i<girld_bmp->w; i+=TILE)
  {
    blit(girld_bmp,tmp,i,0,0,0,TILE,TILE);
    draw_sprite_h_flip(girld_bmp_f,tmp,i,0);
  }
  destroy_bitmap(tmp);
  tmp = create_bitmap(TILE*2,TILE);
  //64x32 transformations
  clear_to_color(plane_bmp_f,makecol16(255,0,255));
  for (i=0; i<plane_bmp->w; i+=TILE*2)
  {
    blit(plane_bmp,tmp,i,0,0,0,TILE*2,TILE);
    draw_sprite_h_flip(plane_bmp_f,tmp,i,0);
  }
  destroy_bitmap(tmp);
  tmp = create_bitmap(TILE,TILE*2);
  //32x64 transformations
  clear_to_color(plane_fall_f,makecol16(255,0,255));
  for (i=0; i<plane_fall->w; i+=TILE)
  {
    blit(plane_fall,tmp,i,0,0,0,TILE,TILE*2);
    draw_sprite_h_flip(plane_fall_f,tmp,i,0);
  }
  destroy_bitmap(tmp);
  tmp = create_bitmap(TILE*2,TILE*2);
  //64x64 transformations
  clear_to_color(plane_drop_f,makecol16(255,0,255));
  for (i=0; i<plane_drop->w; i+=TILE*2)
  {
    blit(plane_drop,tmp,i,0,0,0,TILE*2,TILE*2);
    draw_sprite_h_flip(plane_drop_f,tmp,i,0);
  }
  clear_to_color(plane_explode_f,makecol16(255,0,255));
  for (i=0; i<plane_explode->w; i+=TILE*2)
  {
    blit(plane_explode,tmp,i,0,0,0,TILE*2,TILE*2);
    draw_sprite_h_flip(plane_explode_f,tmp,i,0);
  }
  destroy_bitmap(tmp);

  girl.x = 250;
  girl.y = 352;
  girl.framestep = 0;
  girl.frame = 0;
  girl.talking = 0;
  girl.score = 0;
  girl.health = 100;
  girl.damage = 0;
  girl.dead = 0;
  girl.cross_x = 0;
  girl.cross_y = 0;
  girl.energy = 0;
  girl.light_x = -1;
  girl.light_y = -1;
  girl.num_plane_hits = 0;

  boy.x = 540;
  boy.y = 352;
  boy.framestep = 0;
  boy.frame = 0;
  boy.talking = 0;
  boy.dead = 0;

  dialogue.active = 0;
  dialogue.part = 0;
  dialogue.step = 0;
  dialogue.who = 0;
  dialogue.type = 0;
  dialogue.text = "";
  dialogue.selected = -1;

  armageddon = 0;

  num_planes = 0;
  num_bombs = 0;
  num_bloods = 0;
  
  planesTrigger.counter = 0;
  planesTrigger.trigger = 100;
  planesTrigger.triggerTrigger = 300;

  gameover = 0;
  gameover_step = 0;

  gameResults.finalScore = -1;
	gameResults.secondsSurvived = 0;
	gameResults.planesMissed = 0;
	gameResults.planesKilled = 0;
	gameResults.planesDodged = 0;
	gameResults.bombsDodged = 0;
	gameResults.bombsHit = 0;
	gameResults.comboCount = 0;
	gameResults.lightningHit = 0;
	gameResults.lightningMissed = 0;

  key[KEY_ESC] = 0;

  music_play("media/music/1.xm",1);
}

static void gameDTalk(int who, const char* text, int type, int next)
{
  static int step = 0;
  const int delay = (type == 0) ? RATE/16 : RATE/8;

  dialogue.active = 1;

  dialogue.who = who;
  dialogue.text = text;
  dialogue.type = type;

  step++;
  if (step > delay)
  {
    step=0;
    dialogue.step++;
  }
  else if (key[KEY_SPACE] || key[KEY_BACKSPACE])
  {
    dialogue.step++; // NOT +2 or +3 because it would skip some key points
    return;
  }
  if (dialogue.step > (int)strlen(dialogue.text)+RATE/(delay*2))
  {
    dialogue.step = 0;
    dialogue.part = next;
  }
}

static void gameDSelect(int offset)
{
  int end;
  dialogue.active = 2;
  dialogue.step = 1;
  for (end=0; end < 5 && dialogue.selections[end].type >= 0; ++end) {}
  end--;
  if (key[KEY_UP])
  {
    key[KEY_UP] = 0;
    dialogue.selected--;
    if (dialogue.selected<0) dialogue.selected = end;
  }
  if (key[KEY_DOWN])
  {
    key[KEY_DOWN] = 0;
    dialogue.selected++;
    if (dialogue.selected>end) dialogue.selected = 0;
  }
  if (key[KEY_ENTER])
  {
    dialogue.part = offset + dialogue.selected;
    dialogue.step = 0;
  }
}

static inline void clearSelections(DialogueSelect* selections)
{
  int i;
  for (i=0; i<5; ++i)
  {
    selections[i].type = -1;
  }
  dialogue.selected = 0;
}

static inline void addSelection(DialogueSelect* selections, const char* string, int type)
{
  int i;
  for (i=0; i<5; ++i)
  {
    if (selections[i].type == -1)
    {
      strcpy(selections[i].line,string);
      selections[i].type = type;
      return;
    }
  }
}

static void gameDTalkAnim()
{
  girl.talking = 0;
  boy.talking = 0;
  if (dialogue.active == 1)
  {
    if (dialogue.who == 0 || dialogue.who == 2)
    {
      girl.talking = 1;
    }
    else if (dialogue.who == 1)
    {
      boy.talking = 1;
    }
  }
  if (girl.talking)
  {
    girl.framestep++;
    if (girl.framestep > 9)
    {
      girl.framestep = 0;
      girl.frame++;
      if (girl.frame > 7 || girl.frame < 5) girl.frame = 5;
    }
  }
  else if (!girl.dead)
  {
    girl.frame = 0;
  }

  if (boy.talking && !boy.dead)
  {
    boy.framestep++;
    if (boy.framestep > 9)
    {
      boy.framestep = 0;
      boy.frame++;
      if (boy.frame > 3) boy.frame = 0;
    }
  }
  else if (!boy.dead)
  {
    boy.frame = 0;
  }
}

static void boyAnim()
{
  if (!boy.dead) return;
  boy.framestep++;
  if (boy.framestep > 10)
  {
    boy.framestep = 0;
    if (boy.frame < 11) boy.frame++;
  }
}

static void armageddonCounter()
{
  static int step;
  int delay = RATE/2;
  if (armageddon > 0)
  {
    step++;
    if (step > delay)
    {
      step = 0;
      if (armageddon < 3) armageddon++;
    }
  }
}

static void gameDialogue()
{
  gameDTalkAnim();
  switch (dialogue.part)
  {
    case 0:
      gameDTalk(1,"Hi Lily!",0,1);
      break;
    case 1:
      gameDTalk(0,"Hi John!",0,2);
      break;
    case 2:
      gameDTalk(1,"Isn't it a nice day?",0,3);
      break;
    case 3:
      if (dialogue.step == 0)
      {
	clearSelections(dialogue.selections);
	addSelection(dialogue.selections,"Yes, it's very beautiful today!",0);
	addSelection(dialogue.selections,"Not as nice as you are, John.",0);
	addSelection(dialogue.selections,"Devastatio regulor faraday krematorium!",1); //Makes no sense :)
	dialogue.step++;
      }
      gameDSelect(4);
      break;
    case 4:
      if (dialogue.step == 0)
      {
	girl.score -= 70;
	dialogue.step++;
      }
      gameDTalk(0,dialogue.selections[0].line,0,7);
      break;
    case 5:
      if (dialogue.step == 0) 
      {
	girl.score -= 120;
	dialogue.step++;
      }
      gameDTalk(0,dialogue.selections[1].line,0,7);
      break;
    case 6:
      if (dialogue.step == 0)
      {
	music_play("media/music/5.xm",1);
      }
      if (dialogue.step == 35)
      {
	armageddon = 1;
	girl.score += rand()%1000 + 100;
	dialogue.step++;
      }
      gameDTalk(2,dialogue.selections[2].line,1,7);
      break;
    case 7:
      if (armageddon) {
	if (dialogue.step == 0)
	{
	  boy.dead = 1;
	  dialogue.step++;
	}
	gameDTalk(-1,"A medium length pause",0,9);
      }
      else
      {
	dialogue.part = 0;
	dialogue.active = 0;
      }
      break;
    case 9:
      gameDTalk(2,"Arghahhahaharharhhaah!!",0,10);
      break;
    case 10:
      if (dialogue.step == 0)
      {
	music_play("media/music/6.xm",1);
      }
      gameDTalk(0,"That was fun. I just hope I get away with it...",0,11);
      break;
    case 11:
      gameDTalk(-1,"short pause",0,15);
      break;
    case 15:
      gameDTalk(3,"We got her. Drop the bombs!",0,16);
      break;
    case 16:
      if (dialogue.step == 0)
      {
	music_stop();
      }
      gameDTalk(0,"Huh?",0,17);
      break;
    case 17:
      music_play("media/music/7.it",1);
      dialogue.part = 18;
      dialogue.active = 0;
      dialogue.who = 3; //workaround for graphical glitch ("jumping" dialogue window)
      break;
    case 18:
      if (dialogue.step == 0)
      {
	music_play("media/music/8.xm",0);
      }
      gameDTalk(3,"Mission accomplished! Return to base.",0,19);
      break;
    case 19:
      dialogue.active = 0;
      break;
  }
}

static int add_blood()
{
  Blood b;
  int i;
  if (num_bloods < MAX_BLOODS)
  {
    for (i=0;i<PARTICLES;++i)
    {
      Particle* p = &b.particles[i];
      p->color = makecol16(rand()%75+180,0,0);
      p->x = girl.x + 5 + rand()%16;
      p->x<<=10;
      p->y = girl.y + rand()%28;
      p->y<<=10;
      p->dx = rand()%1024 - 512;
      p->dy = -rand()%2048 - 1024;
      p->size = 1 + rand()%2;
    }
    bloods[num_bloods] = b;
    num_bloods++;
    return 1;
  }
  return 0;
}

/*
static int remove_blood(int n) //TODO: Use this function
{
  if (n < num_bloods)
  {
    num_bloods--;
    if (n < num_bloods)
    {
      bloods[n] = bloods[num_bloods];
    }
    //TODO: Preserve w/ keeper bitmap
    return 1;
  }
  return 0;
}*/

static void do_blood()
{
  int i,j;
  int y;
  for (i=0;i<num_bloods;++i)
  {
    Blood* b = &bloods[i];
    for (j=0;j<PARTICLES;++j)
    {
      Particle* p = &b->particles[j];
      p->x += p->dx;
      p->y += p->dy;
      p->dy += 64;
      y = p->y>>10;
      if (y > girl.y + 28) 
      {
				p->dy -= 2048;
				if (p->dy <= 0) 
				{
					p->dy = 0;
					p->dx = 0;
				}
      }
    }
  }
}

static void draw_blood()
{
  int i,j;
  int x,y;
  for (i=0;i<num_bloods;++i)
  {
    Blood* b = &bloods[i];
    for (j=0;j<PARTICLES;++j)
    {
      Particle* p = &b->particles[j];
      x = p->x>>10;
      y = p->y>>10;
      if (p->size <= 1)
	putpixel(buffer,x,y,p->color);
      else
	circlefill(buffer,x,y,p->size-1,p->color);
    }
  }
}

static inline int plane_collides(Plane* p, int x, int y)
{
  if (p->dead) return 0;
  if (p->x <= x && p->x+TILE*2 >= x &&
      p->y <= y && p->y+TILE >= y)
  {
    return 1;
  }
  return 0;
}

static void shooting_collision()
{
  // A very hacky solution, but no time for a better one
  int x, y;
  int dx, dy;
  int i;
  const int minus = rand()%3;
  dx = (girl.light_x - girl.cross_x)/(5-minus);
  dy = (girl.light_y - girl.cross_y)/(5-minus);
  for (x = girl.light_x, y = girl.light_y; x>0 && y>0 && x < WIDTH && y < HEIGHT; x+=dx, y+=dy)
  { 
    for (i=0; i<num_planes; ++i)
    {
      if (plane_collides(&planes[i],x,y))
      {
				girl.score += 200;
				planes[i].dead++;
				planes[i].frame = 0;
				girl.num_plane_hits++;
      }
    }
  }
}

static void shoot()
{
  int gx, gy;
  int dgx, dgy;
  gx = girl.x + TILE/2;
  gy = girl.y + TILE;
  dgx = girl.cross_x - gx;
  dgy = girl.cross_y - gy;
  girl.light_x = girl.cross_x + dgx*2/3;
  girl.light_y = girl.cross_y + dgy*2/3;
}

static void do_shooting()
{
  static int shoot_c = 0;
  int mx, my; //Mouse x,y
  int gx, gy; //Girl x,y (center)
  int dx, dy; //difference mouse->girl
  float d;
  int factor;

  gx = girl.x + TILE/2;
  gy = girl.y + TILE;
  poll_mouse(); //Switch to polling mode
  mx = mouse_x;
  my = mouse_y;
  dx = mx - SCREEN_W/2;
  dy = my - SCREEN_H;

  d = sqrt(dx*dx + dy*dy);
  factor = (int)(50*1024/d);
  girl.cross_x = gx + dx*factor/1024;
  girl.cross_y = gy + dy*factor*SCREEN_W/1024/SCREEN_H;

  if ((mouse_b&1 && girl.energy >= MAX_ENERGY) || shoot_c)
  {
    if (shoot_c == 0)
    {
      shoot();
      shooting_collision();
      girl.energy = 0;
      play_sample(sounds.lightfire,255,127,1000,0);
    }
    if (shoot_c < 3)
    {
      shoot_c++;
      shooting_collision();
    }
    else //End of lightning strike
    {
			gameResults.planesKilled += girl.num_plane_hits;
      if (girl.num_plane_hits >= 3) //ULTRACOMBO
      {
				gameResults.comboCount++;
				girl.score += 250 * girl.num_plane_hits;
				play_sample(sounds.ultracombo,255,127,1000,0);
      }
			if (girl.num_plane_hits > 0)
			{
      	girl.num_plane_hits = 0;
				gameResults.lightningHit++;
			}
			else
			{
				gameResults.lightningMissed++;
			}
      shoot_c = 0;
      girl.light_x = -1;
      girl.light_y = -1;
    }
  }
  else
  {
    shoot_c = 0;
    girl.light_x = -1;
    girl.light_y = -1;
  }
}

static int damage_girl(int x, int y, int radius)
{
  int gx = girl.x + TILE/2;
  int gy = girl.y + TILE;
  const int distance = abs(gx-x);
  const int damage = (radius - distance) * 2;
  if (y != gy) return 0; //Dummy check, they always match :)
  if (damage > 0)
  {
    girl.damage += damage;
    add_blood();
    if (girl.damage > girl.health) girl.damage = girl.health;
    if (!girl.dead) play_sample(sounds.ouch,255,127,1000,0);
		return 1;
  }
	return 0;
}

static int remove_bomb(int n)
{
  if (n < num_bombs)
  {
    num_bombs--;
    if (n < num_bombs)
    {
      bombs[n] = bombs[num_bombs];
    }
    return 1;
  }
  return 0;
}

static int add_bomb(int x, int y)
{
  Bomb b;
  if (num_bombs < MAX_BOMBS)
  {
    b.x = x;
    b.y = y;
    b.initial_y = y;
    b.framestep = 0;
    b.frame = 0;
    b.explode = 0;
    bombs[num_bombs] = b;
    num_bombs++;
    return 1;
  }
  return 0;
}

static void do_bombs()
{
  int i;
  for (i=0; i<num_bombs; ++i)
  {
    Bomb* b = &bombs[i];
    if (b->explode)
    { //Explode
      b->framestep++;
      if (b->framestep > 2)
      {
				b->framestep = 0;
				b->frame++;
				if (b->frame == 1)
				{
					play_sample(sounds.bomb,150,b->x*100/WIDTH+155,1000,0);
				}
				if (b->frame == 2)
				{
					girl.score += 10;
					if (damage_girl(b->x+TILE/2,b->y+TILE,TILE/2))
					{
						gameResults.bombsDodged--;
						gameResults.bombsHit++;
					}
				}
				if (b->frame > 7)
				{
					remove_bomb(i--);
					continue;
				}
      }
    }
    else
    {
      b->y += 3 + (b->y-b->initial_y)/32;
      if (b->y > HEIGHT-TILE*4)
      {
				gameResults.bombsDodged++;
				b->y = HEIGHT - TILE*4;
				b->explode++;
      }
    }
  }
}

static int add_plane()
{
  Plane p;
  if (girl.dead)
  {
    return 1;
  }
  if (num_planes < MAX_PLANES)
  {
    p.direction = rand()%2;
    switch (p.direction)
    {
      case 0: p.x = -TILE*2-1; break;
      case 1: p.x = WIDTH; break;
      default: p.x = WIDTH/2; break;
    }
    p.y = rand()%100;
    p.initial_y = p.y + rand()%20;
    p.framestep = 0;
    p.frame = 0;
    p.dead = 0;
    p.bombready = rand()%5+10;
    planes[num_planes] = p;
    num_planes++;
    return 1;
  }
  return 0;
}

static int remove_plane(int n)
{
  if (n < num_planes)
  {
    num_planes--;
    if (n < num_planes)
    {
      planes[n] = planes[num_planes];
    }
    return 1;
  }
  return 0;
}

static void do_planes()
{
  int i;

	planesTrigger.counter++;
	if (planesTrigger.counter > planesTrigger.trigger)
	{
		if (add_plane())
		{
			planesTrigger.triggerTrigger-=7;
			if (planesTrigger.triggerTrigger < 0) planesTrigger.triggerTrigger = 30;
			else if (planesTrigger.triggerTrigger == 0) planesTrigger.triggerTrigger = 1;
			planesTrigger.trigger = rand()%planesTrigger.triggerTrigger + planesTrigger.triggerTrigger/2;
			planesTrigger.counter = 0;
		}
	}
	for (i=0; i<num_planes; ++i)
	{
		Plane* p = &planes[i];
		if (!p->dead)
		{
			switch (p->direction)
			{
				case 0: p->x += 3; 
								if (p->x > WIDTH) 
								{
									girl.score += 20;
									remove_plane(i--);
									gameResults.planesMissed++;
									continue;
								}
								break;
				case 1: p->x -= 3; 
								if (p->x < -TILE*2)
								{
									girl.score += 20;
									remove_plane(i--);
									gameResults.planesMissed++;
									continue;
								}
								break;
			}
			p->bombready--;
			if (p->bombready <= 0)
			{
				if (add_bomb(p->x+TILE/2,p->y))
				{
					play_sample(sounds.bombdrop,100,127,1000,0);
					p->bombready = rand()%40+20;
				}
      }
      p->framestep++;
			if (p->framestep > 2)
			{
				p->framestep = 0;
				p->frame++;
				if (p->frame > 3) p->frame = 0;
      }
    }
    else if (p->dead == 1)
    { //Drop
			switch (p->direction)
			{
				case 0: p->x += 2; break;
				case 1: p->x -= 2; break;
			}
			p->framestep++;
			if (p->framestep > 3)
			{
				p->framestep = 0;
				p->frame++;
				if (p->frame == 1)
				{
					play_sample(sounds.planehit,255,127,1000,0);
					girl.energy += 30; //Recharge
				}
				if (p->frame > 3)
				{
					p->frame = 0;
					p->dead++;
					p->x += 8; //Animation correction
					if (p->direction == 1) p->x += 16; //Another correction
				}
      }
    }
    else if (p->dead == 2)
    { //Fall
      p->framestep++;
      if (p->framestep > 2)
      {
				p->framestep = 0;
				p->frame++;
				if (p->frame > 3)
				{
					p->frame = 0;
				}
      }
      switch (p->direction)
      {
				case 0: p->x++; break;
				case 1: p->x--; break;
      }
      p->y+= 2 + (p->y-p->initial_y)/30;
      if (p->y > HEIGHT-TILE*5)
      {
				p->y = HEIGHT-TILE*5;
				p->dead++;
				p->frame = 0;
				p->x -= 16; //Animation correction
      }
    }
    else if (p->dead >= 3)
    { //Explode
      p->framestep++;
      if (p->framestep > 6)
			{
				p->framestep = 0;
				p->frame++;
				if (p->frame == 1)
				{
					play_sample(sounds.planecrash,255,127,1000,0);
				}
				if (p->frame == 2)
				{
					if (!damage_girl(p->x+TILE,p->y+TILE*2,TILE))
					{
						gameResults.planesDodged++;
					}
				}
				if (p->frame > 7)
				{
					remove_plane(i--);
					continue;
				}
			}
		}
  }
}

static void girl_movement()
{
  girl.framestep++;
  if (girl.framestep > 9) 
  {
    girl.framestep = 0;
    girl.frame++;
    if (girl.frame > 4) girl.frame = 1;
  }
  if (key[KEY_RIGHT] || key[KEY_D])
  {
    girl.x++;
    if (girl.x > WIDTH-TILE) girl.x = WIDTH - TILE;
    girl.direction = 0;
  }
  else if (key[KEY_LEFT] || key[KEY_A])
  {
    girl.x--;
    if (girl.x < 0) girl.x = 0;
    girl.direction = 1;
  }
  else
  {
    girl.frame = 0;
  }
}

static void do_girl()
{
  static int scorestep = 0;
  if (girl.dead)
  {
		girl.framestep++;
		if (girl.framestep > 5)
		{
			girl.framestep = 0;
			girl.frame++;
			if (girl.frame > 7)
			{
				girl.frame = 7;
				if (dialogue.part < 19)
				{
					dialogue.active = 1;
				}
      }
    }
  }
  else
  {
    if (girl.damage > 0)
    {
      girl.damage--;
      girl.health--;
      if (girl.health <= 0)
      {
				girl.health = 0;
				girl.damage = 0;
				girl.dead = 1;
				girl.frame = 0;
				girl.framestep = 0;
				girl.energy = 0;
      }
    }
    if (armageddon)
    {
      if (girl.energy < MAX_ENERGY)
      {
	girl.energy++;
	if (girl.energy == MAX_ENERGY)
	{
	  play_sample(sounds.lightready,255,127,1000,0);
	}
      }
      if (!dialogue.active)
      {
	scorestep++;
	if (scorestep > 10)
	{
	  scorestep = 0;
	  girl.score++;
	}
      }
    }
    girl_movement();
  }
}

static int gameoverCheck()
{
  if (gameover > 0)
  {
    switch (gameover)
    {
      case 1: //Fly in
	gameover_step+=6;
	if (gameover_step > WIDTH/2 + gameover_bmp->w/2)
	{
	  gameover_step = 0;
	  gameover++;
	}
	break;
      case 2: //Stay
	gameover_step++;
	if (gameover_step > RATE*1)
	{
	  gameover_step = 0;
	  gameover++;
	}
	break;
      case 3: //Fly out
	gameover_step+=8;
	if (gameover_step > WIDTH/2 + gameover_bmp->w/2)
	{
	  gameover_step = 0;
	  gameover++;
	}
	break;
      case 4: //Fade out
	gameover_step++;
	if (gameover_step > RATE*3)
	{
	  gameover_step = 0;
	  gameover++;
	}
	break;
      case 5: //Quit
	return 1;
	break;
    }
  }
  else if (girl.dead && !dialogue.active)
  {
    gameover_step++;
    if (gameover_step > RATE*2)
    {
      gameover_step = 0;
      gameover++;
    }
  }
  return 0;
}

static void do_survivecounter()
{
	static int count = 0;
	count++;
	if (count >= RATE)
	{
		count = 0;
		gameResults.secondsSurvived++;
	}
}

LogicReturn InGame_logic()
{
  static int d1_trigger = 0;
  if (!dialogue.active)
  {
    do_girl(); //Movement, damage and walk/death animation
    if (armageddon)
    {
			do_survivecounter();
      do_planes();
      do_bombs();
      do_blood();
      do_shooting();
    }
    else
    {
      if (girl.x > boy.x-30 && !boy.dead)
      {
				if (!d1_trigger)
				{
					dialogue.active = 1;
					d1_trigger = 1;
				}
      }
      else
      {
				d1_trigger = 0;
      }
    }
  }
  else
  {
    gameDialogue(); //Dialogue logic and talk animation
  }
  boyAnim();
  armageddonCounter();
  if (gameoverCheck())
  {
    InGame_cleanup();
    gameResults.finalScore = girl.score;
    Game_ChangeState(STATE_HIGHSCORE);
    return LR_EXIT;
  }
  if (key[KEY_ESC])
  {
    InGame_cleanup();
    Game_ChangeState(STATE_TITLE);
    return LR_EXIT;
  }
  return LR_NOOP;
}

static void draw_diabox(int b)
{
  int light, med, dark, bg;
  int t = HEIGHT-b-1;
  if (dialogue.who == 3)
  {
    t = 0;
  }
  else
  {
    b = HEIGHT-1;
  }
  light = makecol16(0,255,0);
  med = makecol16(0,200,0);
  dark = makecol16(0,170,0);
  bg = makecol16(0,100,0);

  //TOP
  hline(buffer,0,t,639,dark);
  hline(buffer,1,t+1,638,med);
  hline(buffer,2,t+2,637,light);
  //LEFT
  vline(buffer,0,t,b,dark);
  vline(buffer,1,t+1,b-1,med);
  vline(buffer,2,t+2,b-2,light);
  //RIGHT
  vline(buffer,637,t+2,b-2,light);
  vline(buffer,638,t+1,b-1,med);
  vline(buffer,639,t,b,dark);
  //BOTTOM
  hline(buffer,2,b-2,637,light);
  hline(buffer,1,b-1,638,med);
  hline(buffer,0,b,639,dark);
  //BG
  rectfill(buffer,3,t+3,636,b-3,bg);
}

static void draw_dialine()
{
  const FONT* fnt = (dialogue.type == 0) ? fnt_sans : fnt_chris;
  const int color = (dialogue.type == 0) ? makecol16(255,255,255) : makecol16(255,0,0);
  //const int MID = 62/2 + WIDTH/2;
  //const int width = text_length(fnt,dialogue.text);
  //const int start = MID - width/2;
  const int start = 72;
  int t = HEIGHT-65-1;
  if (dialogue.who == 3) 
  {
    t = 0;
  }
  char* text = strdup(dialogue.text);

  draw_diabox(65);
  blit(portraits,buffer,dialogue.who*58,0,4,t+4,58,58);
  if (dialogue.step < (int)strlen(text))
  {
    text[dialogue.step] = 0;
  }
  if (dialogue.type == 1)
  {
    textout_ex(buffer,fnt,text,start+1,t+16,makecol16(123,0,0),-1);
  }
  textout_ex(buffer,fnt,text,start,t+15,color,-1);
  
  free(text);
}

static void draw_diasels()
{
  int i;
  int pos, circlepos;
  const FONT* fnt;

  draw_diabox(80);
  //pos = 10;
  pos = HEIGHT-72;
  for (i=0; i<5 && dialogue.selections[i].type != -1; ++i)
  {
    fnt = (dialogue.selections[i].type==0)?fnt_sans:fnt_chris;
    textout_ex(buffer,fnt,dialogue.selections[i].line,30,pos,makecol16(255,255,255),-1);
    if (dialogue.selected == i)
    {
      circlepos = pos + text_height(fnt)/2;
      circlefill(buffer,15,circlepos,7,makecol16(255,255,255));
      circle(buffer,15,circlepos,7,makecol16(200,200,200));
    }
    pos += text_height(fnt) + 2;
  }
}

static void draw_dialogue()
{
  if (dialogue.who == -1) return;
  //if (dialogue.step == 0) return;
  switch (dialogue.active)
  {
    case 1: 
      draw_dialine();
      break;
    case 2:
      draw_diasels();
      break;
    default: break;
  }
}

static void draw_tile(int tile, int x, int y)
{
  masked_blit(tiles1,buffer,tile*TILE,armageddon*TILE,x*TILE,y*TILE,TILE,TILE);
}

static void draw_tilemap()
{
  static int base_map[] = {
    1,1,3,2,1,1,2,6,1,2,1,1,1,2,1,2,4,4,2,1,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5
  };
  const int starth = HEIGHT/TILE-4;
  int y,x;
  for (y=0;y<4;++y)
  {
    for (x=0;x<20;++x)
    {
      draw_tile(base_map[y*20+x],x,starth+y);
    }
  }
}

static void draw_bg()
{
  int color;
  switch (armageddon)
  {
    case 0: color = makecol16(128,128,255); break;
    case 1: color = makecol16(255,rand()%100+155,0); break;
    case 2: color = makecol16(255,rand()%50+155,50); break;
    case 3: color = makecol16(100,100,100); break;
    default: color = makecol16(255,255,255); break;
  }
  clear_to_color(buffer,color);
}

static void draw_cross()
{
  const int addc = rand()%100+100;
  int color1, color2;
  if (girl.energy < MAX_ENERGY)
  {
    color1 = makecol16(150,150,200);
    color2 = makecol16(100,100,150);
  }
  else
  {
    color1 = makecol16(addc,addc,255);
    color2 = makecol16(addc/2,addc/2,255);
  }
  circlefill(buffer,girl.cross_x,girl.cross_y,2,color1);
  circle(buffer,girl.cross_x,girl.cross_y,3,color2);
  if (girl.light_x >= 0 && girl.light_y >= 0)
  {
    lightningArc2(girl.cross_x,girl.cross_y,girl.light_x,girl.light_y);
  }
}

static void draw_planes()
{
  int i;
  int tw;
  BITMAP* m_planebmp;
  for (i=0; i<num_planes; ++i)
  {
    Plane* p = &planes[i];
    tw = TILE*2; // Most common case
    switch (p->direction)
    {
      case 0: if (p->dead > 2) { m_planebmp = plane_explode; }
	      else if (p->dead > 1) { m_planebmp = plane_fall; tw = TILE; }
	      else if (p->dead > 0) { m_planebmp = plane_drop; }
	      else { m_planebmp = plane_bmp; }
	      break;
      case 1: if (p->dead > 2) { m_planebmp = plane_explode_f; }
	      else if (p->dead > 1) { m_planebmp = plane_fall_f; tw = TILE; }
	      else if (p->dead > 0) { m_planebmp = plane_drop_f; }
	      else { m_planebmp = plane_bmp_f; }
	      break;
      default: m_planebmp = NULL; fprintf(stderr,"Plane has invalid direction!\n"); break;
    }
    masked_blit(m_planebmp,buffer,p->frame*tw,0,p->x,p->y,tw,m_planebmp->h);
  }
}

static void draw_bombs()
{
  int i;
  for (i=0; i<num_bombs; ++i)
  {
    Bomb* b = &bombs[i];
    masked_blit(bomb_bmp,buffer,b->frame*TILE,0,b->x,b->y,TILE,TILE);
  }
}

static void draw_score()
{
  //Approx. height: 20
  char text[50];
  int top = 20;
  int width, height;
  sprintf(text,"Score: %d",girl.score);
  width = text_length(fnt_sans,text) + 6;
  height = text_height(fnt_sans) + 6;
  rect(buffer,WIDTH-width-19,top-4,WIDTH-20+4,top+height-3,makecol(200,200,200));
  rectfill(buffer,WIDTH-width-18,top-3,WIDTH-20+3,top+height-4,makecol(128,128,128));
  textout_right_ex(buffer,fnt_sans,text,WIDTH-19,top+1,makecol16(100,100,100),-1);
  textout_right_ex(buffer,fnt_sans,text,WIDTH-20,top,makecol16(255,255,255),-1);
}

static void draw_stats()
{
  static unsigned char energy_blink = 0;
  //Approx. height: 35
  const int top = 15;
  const int top2 = top + 1+16+1;
  const int cur_energy = girl.energy*64/MAX_ENERGY;
  const int cur_damage = girl.damage*64/100;
  const int cur_health = girl.health*64/100 - cur_damage;
  //HEALTH
  masked_blit(symbols,buffer,0,0,20+1,top+1,16,16);
  masked_blit(stats_bmp,buffer,0,16,20+1+16+1,top+1,cur_health,16); //Health
  if (cur_damage > 0)
    masked_blit(stats_bmp,buffer,cur_health,48,20+1+16+1+cur_health,top+1,cur_damage,16); //Damage
  //ENERGY
  masked_blit(symbols,buffer,16,0,20+1,top2+1,16,16);
  if (girl.energy >= MAX_ENERGY)
    energy_blink^=1;
  else
    energy_blink = 0;

  if (energy_blink)
    masked_blit(stats_bmp,buffer,0,32,20+1+16+1,top2+1,cur_energy,16);
  else
    masked_blit(stats_bmp,buffer,0,0,20+1+16+1,top2+1,cur_energy,16);
}

static void draw_boy()
{
  BITMAP* m_boybmp = boy_bmp;
  if (boy.dead)
  {
    m_boybmp = boyd_bmp;
  }
  masked_blit(m_boybmp,buffer,boy.frame*TILE,0,boy.x,boy.y,TILE,TILE);
}

static void draw_girl()
{
  BITMAP* m_girlbmp = girl_bmp;
  if (girl.dead)
  {
    m_girlbmp = girld_bmp;
    if (girl.direction == 1)
    {
      m_girlbmp = girld_bmp_f;
    }
  }
  else
  {
    if (girl.direction == 1)
    {
      m_girlbmp = girl_bmp_f;
    }
  }
  masked_blit(m_girlbmp,buffer,girl.frame*TILE,0,girl.x,girl.y,TILE,TILE);
}

void draw_gameover()
{
  if (!gameover) return;
  int x;
  const int y = (HEIGHT - gameover_bmp->h)/2;
  switch (gameover)
  {
    case 1:
      x = gameover_step - gameover_bmp->w;
      draw_sprite(buffer,gameover_bmp,x,y);
      break;
    case 2:
      x = WIDTH/2 - gameover_bmp->w/2;
      draw_sprite(buffer,gameover_bmp,x,y);
      break;
    case 3:
      x = gameover_step + WIDTH/2 - gameover_bmp->w/2;
      draw_sprite(buffer,gameover_bmp,x,y);
      break;
    case 4:
      set_trans_blender(255,255,255,(gameover_step*2<255)?gameover_step*2:255);
      draw_trans_sprite(buffer,black,0,0);
      break;
    case 5:
      draw_sprite(buffer,black,0,0);
      break;
    default:break;
  }
}

void InGame_draw()
{
  draw_bg();
  if (!girl.dead) draw_girl();
  if (!boy.dead) draw_boy();
  if (armageddon)
  {
    draw_planes();
    draw_bombs();
    if (!dialogue.active && !girl.dead)
    {
      draw_cross();
    }
  }
  draw_tilemap();
  if (armageddon) draw_blood();
  if (girl.dead) draw_girl();
  if (boy.dead) draw_boy();
  draw_score();
  if (armageddon && !dialogue.active)
  {
    draw_stats();
  }
  if (dialogue.active)
  {
    draw_dialogue();
  }
  draw_gameover();
}

