#define DEBUGMODE

#include "Play.h"
#include "Resources.h"
#include "Game.h"
#include "GameScreen.h"
#include "MainCharacter.h"
#include "Particles.h"
#include "RandomSounds.h"
#include "PauseMenu.h"

#include "Pip.h"

#include "Messages.h"

Play::Play(Game *owner):
	game(owner), player(NULL),
	mode(MODE_PLAY),
	framecounter(0),
	prev_gs(0),
	cur_gs(1),
	cursor(NULL),
	cursor_mode(0),
	cursor_objid(0),
	cursor_fc(0),
	cursor_dir(1),
	rescued(0)
{
	temp = create_bitmap(320, 240);
	cursor = owner->GetResources()->GetImage("Cursor");
	
TRACE("Creating ambient music...\n");
	CreateAmbient();
TRACE("Creating particles...\n");
	CreateParticles();
TRACE("Creating gamescreens...\n");
	CreateGamescreens();
TRACE("Creating main character...\n");
	CreatePlayer();
TRACE("Creating random sounds...\n");
	CreateRandomSounds();
TRACE("Creating pause menu\n");
	CreatePauseMenu();
	PlayAmbient();
	
TRACE("Play successfully created...\n");	
}

Play::~Play()
{
	DestroyParticles();
	DestroyGamescreens();
	DestroyPlayer();
	DestroyRandomSounds();
	DestroyAmbient();
	DestroyPauseMenu();
}

void Play::CreatePauseMenu()
{
	pausemenu = new PauseMenu(game);
	pausemenu->AddOption("Back to game");
	pausemenu->AddOption("Load game");
	pausemenu->AddOption("Save game");
	pausemenu->AddOption("Quit");
}

void Play::DestroyPauseMenu()
{
	delete pausemenu;
}

void Play::CreateAmbient()
{
	ambient = new AudioSampler(GetResources()->GetSound());
}

void Play::DestroyAmbient()
{
	delete ambient;
}

void Play::CreateRandomSounds()
{
	rndsounds = new RandomSounds(GetResources()->GetSound(), "data\\rndsounds.txt");
}

void Play::DestroyRandomSounds()
{
	delete rndsounds;
}

void Play::CreateGamescreens()
{
	for( int i=0; i < GetResources()->maps.size(); i++ )
	{
		gs.push_back(new GameScreen(
						this,
						GetResources()->maps[i],
						GetResources()->mapex[i],
						i+1));
	}
}

void Play::DestroyGamescreens()
{
	for( int i=0; i<gs.size(); i++ )
	{
		gs[i]->ResetSwitches();
		//delete gs[i];
	}
	gs.clear();
}

void Play::ResetGamescreens()
{
	DestroyGamescreens();
	//gs.clear();
	CreateGamescreens();
}

void Play::Die()
{
	mode = MODE_DIE;
}

void Play::UpdatePlay()
{
	player->LogicalUpdate();
	particles->LogicalUpdate();
	for( int i=0; i<gs.size(); i++ )
		gs[i]->LogicalUpdate();
	rndsounds->Frametick();
	if (mode == MODE_PLAY)
		if (key[KEY_ESC])
			mode = MODE_PAUSE;
}

void Play::UpdateCursor()
{
	switch( cursor_mode )
	{
		case 0:
			if (GetButtons()->KeyPress(BTN_ACT2))
			{
				if (gs[cur_gs-1]->EnemiesCount())
				{
					cursor_mode = 1;
					cursor_objid = 0;
				}
			}
			break;
		case 1:
			cursor_fc += cursor_dir;
			if ((cursor_fc > 16) || (cursor_fc < 0))
			{
				cursor_fc = 0;
				cursor_dir = -cursor_dir;
			}
			
			if (GetButtons()->KeyPress(BTN_ACT2))
				cursor_objid++;
			
			if (cursor_objid >= gs[cur_gs-1]->EnemiesCount())
			{
				//if (!gs[cur_gs-1]->EnemiesCount())
					cursor_mode = 0;
				//else
				//	cursor_objid = 0;
			}	
			break;
		case 2:
			break;
	}
}

void Play::UpdateScreenTransition()
{
	framecounter += 8;
	if (framecounter >= 320)
	{
		mode = MODE_PLAY;
		framecounter = 0;
	}
}

void Play::UpdateDeathFadeout()
{
	UpdatePlay();
	framecounter++;
	if (framecounter > 500)
	{
		mode = MODE_RESPAWN;
		framecounter = 0;
		ResetParticles();
		ResetGamescreens();
		cur_gs = 1;
	}
}

void Play::UpdateRespawn()
{
	//UpdatePlay();
	framecounter++;
	if (framecounter > 200)
	{
		ResetPlayer();
		player->SetGamescreen(1);
		mode = MODE_PLAY;
		framecounter = 0;
	}
}

void Play::UpdatePauseMenu()
{
	pausemenu->LogicalUpdate();
	if (saves == 0)
		pausemenu->Disable(2);
	else
		pausemenu->Enable(2);
	switch(pausemenu->ReturnValue())
	{
		case 0:
			mode = MODE_PLAY;
			break;
		case 1:
			break;
		case 2:
			break;
		case 3:
			game->Quit();
			break;
	}
}

void Play::LogicalUpdate()
{
	switch( mode )
	{
		case MODE_PLAY:
			UpdatePlay();
			UpdateCursor();
			break;
		case MODE_SCRTRANSITION:
			UpdateScreenTransition();
			break;
		case MODE_DIE:
			UpdateDeathFadeout();
			break;
		case MODE_RESPAWN:
			UpdateRespawn();
			break;
		case MODE_PAUSE:
			UpdatePauseMenu();
			break;
	}
}

void Play::DrawPlay(BITMAP *dest)
{
	gs[cur_gs-1]->DrawBackground(dest);
	player->Draw(dest);
	particles->Draw(dest);
	gs[cur_gs-1]->DrawForeground(dest);
//	textout_ex(dest, font, "Uneksians", 0, 0, makecol(255,255,255), -1);
//	textprintf_ex(dest, font, 0, 0, makecol(255, 255 ,255), -1, "Voices: %d", get_mixer_voices());
//	textprintf_ex(dest, font, 0, 10, makecol(255,255,255), -1,
//	"Delta=%d", rndsounds->GetDelta());
//	textprintf_ex(dest, font, 0, 20, makecol(255,255,255), -1,
//	"Heartrate = %d Sb = %d St = %d", player->Heartrate(), player->SolidBottom(), player->SolidTop());
}

void Play::DrawCursor(BITMAP *dest)
{
	switch(cursor_mode)
	{
		case 1:
			if (CursorValid())
			{
				Object *obj = gs[cur_gs-1]->GetObject(cursor_objid);
				draw_sprite(dest, cursor, obj->CenterX()-cursor->w/2, obj->YPos()-(cursor_fc>>2)-4);
			}
			break;
	}
}

void Play::DrawScreenTransition(BITMAP *dest)
{
	gs[prev_gs-1]->DrawBackground(dest);
	gs[prev_gs-1]->DrawForeground(dest);
	clear(temp);
	gs[cur_gs-1]->DrawBackground(temp);
	player->Draw(temp);
	particles->Draw(temp);
	gs[cur_gs-1]->DrawForeground(temp);
	blit(temp, dest, 0, 0, 0, 0, framecounter, 240);
}

void Play::DrawDeathFadeout(BITMAP *dest)
{
	DrawPlay(dest);
	if (framecounter > 200)
	{
		BITMAP *alpha = create_bitmap(dest->w, dest->h);
		
		clear(alpha);
		set_trans_blender(0,0,0,(framecounter-200)*255/300);
		draw_trans_sprite(dest, alpha, 0, 0);
		destroy_bitmap(alpha);
	}
}

void Play::DrawRespawn(BITMAP *dest)
{
	DrawPlay(dest);
	BITMAP *alpha = create_bitmap(dest->w, dest->h);
	
	clear(alpha);
	
	set_trans_blender(0,0,0,255-framecounter*255/200);
	draw_trans_sprite(dest, alpha, 0, 0);
	destroy_bitmap(alpha);
}

void Play::DrawPauseMenu(BITMAP *dest)
{
	DrawPlay(dest);
	BITMAP *alpha = create_bitmap(dest->w, dest->h);
	
	clear(alpha);
	
	set_trans_blender(0,0,0,128);
	draw_trans_sprite(dest, alpha, 0, 0);
	destroy_bitmap(alpha);
	
	pausemenu->Draw(dest);
}

void Play::Draw(BITMAP *dest)
{
	switch( mode )
	{
		case MODE_PLAY:
			DrawPlay(dest);
			DrawCursor(dest);
			break;
		case MODE_SCRTRANSITION:
			DrawScreenTransition(dest);
			break;
		case MODE_DIE:
			DrawDeathFadeout(dest);
			break;
		case MODE_RESPAWN:
			DrawRespawn(dest);
		case MODE_PAUSE:
			DrawPauseMenu(dest);
			break;
	}
	
//	textprintf_ex(dest, font, 0, 10, makecol(255,255,255), -1,
//	"CUR_GS=%d for object=%d fc=%d", cur_gs, player->Curgs(), framecounter);
}

void Play::PlayAmbient()
{
	ambient->Stop();
	ambient->SetObjectID(gs[cur_gs-1]->GetMapinfo().music);
	ambient->SetVolume(255);
	ambient->PlayLooped();
}

void Play::StopAmbient()
{
	ambient->Stop();
}

void Play::GoToScreen( int next_gs )
{
	if (cur_gs == next_gs) return;
	cursor_mode = 0;
	prev_gs = cur_gs;
	cur_gs = next_gs;
	mode = MODE_SCRTRANSITION;
	
	if (gs[cur_gs-1]->GetMapinfo().plays_music)
		if (strcmp(ambient->AudioSample(), gs[cur_gs-1]->GetMapinfo().music))
			PlayAmbient();
}

Tegmap *Play::GetMap(int id)
{
	return gs[id-1]->GetMap();
}

Tegmap *Play::GetActualMap()
{
	return gs[cur_gs-1]->GetMap();
}

Resources *Play::GetResources()
{
	return game->GetResources();
}

Buttons *Play::GetButtons()
{
	return game->GetButtons();
}

Object *Play::Cursor()
{
	if (CursorValid())
		return gs[cur_gs-1]->GetObject(cursor_objid);
	else
		return NULL;
}

bool Play::CursorValid()
{
	return (cursor_objid < gs[cur_gs-1]->EnemiesCount()) && cursor_mode;
}
