#ifndef		TIGCPB_OBJECT_H
#define		TIGCPB_OBJECT_H

#include <allegro.h>
#include <vector>
#include "ObjectStates.h"
#include "Messages.h"

class Object;
class GameScreen;
class Particles;
class AnimPlayer;
class Sound;
class AudioSampler;
class Possession;
class MyINI;

typedef
	struct _VLEAP
	{
		long
			vlsize,
			vlspeed,
			leapboost,
			velocity,
			acceleration;
	} VLEAP;

typedef
	struct _RESPAWN
	{
		Object* obj;
		int
			rs,
			gs,
			x, y,
			state,
			stp1, stp2,
			health,
			time;
	} RESPAWN;
	
class Object
{
	protected:
		GameScreen *gamescreen;
		
		long
			xpos, ypos; // Position of object relative to its parent game screen (8-bytes precision)
		int
			w, h, // Dimensions of object (sprite)
			hx, hy, hw, hh; // Position and shape of object's hard surface region
		
		int type;
		
		// Movement deltas
		int
			deltax,
			deltay;
		bool
			dxvalid,
			dyvalid;
		
		int
			health,
			fullhealth,
			dangerous,
			heartrate,
			walkspeed,
			runspeed,
			state,
			stparam1, stparam2;
		
		bool
			main_character,
			possessable,
			interactable,
			is_character,
			kill_after_possess,
			uses_anims;
		
		Object *callingparty_obj;
		int callingparty_id;
		
		// Healthbar data
		bool show_healthbar;
		int healthbar_life;
		
		// Blood particles data
		int
			bp_mincolor,
			bp_maxcolor;
		
		RESPAWN respawn; // respawn data
		VLEAP vleap; // Vertical leap data
		
		// Gamespeak data
		BITMAP *msgbox;
		MESSAGE message;
		
		bool
			wboost,
			facing,
			user_controlled,
			visible,
			hidden,
			should_respawn;
		
		// Possession source
		Possession *possession;
		
		Object *hiddenby;
		
		// Audio data
		AudioSampler
			*afootstep,
			*adie,
			*ajump,
			*ahurt,
			*acollide;
		
		int current_animation;
		std::vector<AnimPlayer*> anims;
	protected:
		// Physical interaction
		int IsEndOfScreen(int x, int y);
		int VEdgeCol(int y, int side);
		int HEdgeCol(int x, int side);
		int VCollision(int x, int y, int side);
		int HCollision(int x, int y, int side);
		int IsMissingGround(int x, int y);
		virtual int MoveX(int dx);
		virtual int MoveY(int dy);
		
		// Gravitation
		void PrepareLeap(int boost);
		void PrepareFall(int boost);
		int LeapFrametick();
		int FallFrametick();
		
		// Facing
		void TurnLeft() { facing = true; }
		void TurnRight() { facing = false; }
		
		// Gamespeak
		void SetMessage(int type, int rtype, Object *reciever, int d);
		void CreateMsgbox(int w, int h, const char *text);
		void MsgboxUpdate();
		void MsgboxDraw(BITMAP *dest);
		void ReadMessages();
		virtual void ProcessMessage(MESSAGE msg) = 0;
		
		// Healthbar
		void HealthbarDraw(BITMAP *dest);
		
		// Movement deltas
		void InvalidateDeltas() { dxvalid = false; dyvalid = false; }
		void SetDeltaX(int dx) { deltax = dx; dxvalid = true; }
		void SetDeltaY(int dy) { deltay = dy; dyvalid = true; }
	public:
		Object(GameScreen *owner, bool is_main);
		~Object();
		
		virtual void Init();
		
		virtual void LogicalUpdate();
		
		virtual void Draw(BITMAP *dest);
		
		//virtual void ProcessShared();
		virtual void ProcessInput();
		virtual void ProcessAI();
		
		void LoadProperties(char *path);
		virtual void ReadProperties(MyINI *ini);
		
		void Damage(int hit);
		void DamageWD(int hit);
		void Kill();
		void Death();
		void Destroy();
		virtual void Respawn();
		void SetupRespawn();
		bool IsOnScreen();
		//bool IsVisible() { return visible; }
		bool IsHidden() { return hidden; }
		bool IsDead() { return health<=0; }
		bool IsPossessable() { return possessable; }
		bool IsInteractable() { return interactable; }
		bool FacingLeft() { return facing; }
		//bool IsMainCharacter() { return main_character; }
		bool WalkBoost() { return wboost && dxvalid; }
		bool KillAfterPossess() { return kill_after_possess; }
		bool IsUserControlled() { return user_controlled; }
		void SetUserControl(bool control) { user_controlled = control; }
		
		// Gamescreen
		void SetGamescreen(int gs_id);
		void SetGamescreen(GameScreen *gs);
		void ExitGamescreen( int exitcode );
		
		// Collision and stuff
		bool Intersect(Object *obj);
		
		// Screen position
		void SetLeft() { xpos = -16 << 8; }
		void SetRight() { xpos = 319-16 << 8; }
		void SetTop() { ypos = -16 << 8; }
		void SetBottom() { ypos = 239-16 << 8; }
		void SetX(int x) { xpos = x; }
		void SetY(int y) { ypos = y; }
		int XPos() { return xpos >> 8; }
		int YPos() { return ypos >> 8; }
		int X() { return xpos; }
		int Y() { return ypos; }
		int DX() { return deltax; }
		int DY() { return deltay; }
		int CenterX() { return (xpos>>8)+hx+hw/2; }
		int CenterY() { return (ypos>>8)+hy+hh/2; }
		int BCenterX() { return (xpos>>8)+w/2; }
		int BCenterY() { return (xpos>>8)+h/2; }
		int SolidX() { return hx; }
		int SolidY() { return hy; }
		int SolidLeft() { return (xpos>>8) + hx; }
		int SolidRight() { return (xpos>>8) + hx + hw - 1; }
		int SolidTop() { return (ypos>>8) + hy; }
		int SolidBottom() { return (ypos>>8) + hh + hy - 1; }
		
		int SetShape(int x, int y, int wd, int hg) { xpos = x<<8; ypos = y<<8; w = wd; h = hg; }
		
		// Dimensions
		int Width() { return w; }
		int Height() { return h; }
		int SolidWidth() { return hw; }
		int SolidHeight() { return hh; }
		
		// Vertical leap properties
		int VLSize() { return vleap.vlsize; }
		
		bool DXValid() { return dxvalid; }
		bool DYValid() { return dyvalid; }
		
		// Message
		MESSAGE Message() { return message; }
		
		int Type() { return type; }
		RESPAWN RespawnData() { return respawn; }
		bool ShouldRespawn() { return should_respawn; }
		
		void CheckForMissingGround()
		{
			if ((state != STATE_FALLING) && (state != STATE_JUMPING))
				if (IsMissingGround(xpos, ypos))
					SwitchToState(STATE_FALLING, 0, 0);
		}
		
		void ShowHealthbar() { show_healthbar = true; }
		void HideHealthbar() { show_healthbar = false; }
		void SetHealthbarLife(int value) { healthbar_life = value; }
		void Show() { visible = true; }
		void Hide() { visible = false; }
		
		Sound *GetSound();
		int Curgs();
		GameScreen *GS() { return gamescreen; }
		
		int Dangerous() { return dangerous; }
		int Heartrate() { return heartrate; }
		void SetHeartrate(int hr) { heartrate = hr; }
		void SetPossession(Possession *p) { possession = p; }
		Possession *GetPossession() { return possession; }
		void SetCallingParty(Object *obj) { callingparty_obj = obj; }
		void SetHidden(bool value, Object *by) { hidden = value; hiddenby = by; }
		Object *HiddenBy() { return hiddenby; }
		
		void AddHealth(int add) { health += add; if (health > fullhealth) health = fullhealth; }
		
		int State() { return state; }
		
		virtual void InitState(int st);
		virtual void SwitchToState(int st, int param1, int param2);
		virtual void ChangeParams(int param1, int param2);
};

#endif		// TIGCPB_OBJECT_H