#define DEBUGMODE

#include "Play.h"
#include "GameScreen.h"
#include "Object.h"
#include "Particles.h"
#include "Animation.h"
#include "MyINI.h"
#include "Buttons.h"
#include "AudioSampler.h"
#include "Possession.h"

Object::Object(GameScreen *owner, bool is_main):
	gamescreen(owner),
	xpos(0), ypos(0),
	w(0), h(0),
	hx(0), hy(0), hw(0), hh(0), type(0),
	deltax(0), deltay(0), dxvalid(false), dyvalid(false),
	health(0), fullhealth(0),
	dangerous(0), heartrate(150),
	walkspeed(0), runspeed(0), state(0),
	stparam1(0), stparam2(0),
	main_character(is_main),
	possessable(false), interactable(false),
	is_character(true),
	kill_after_possess(true), uses_anims(true),
	callingparty_obj(NULL), callingparty_id(0),
	show_healthbar(false), healthbar_life(0),
	bp_mincolor(0), bp_maxcolor(0),
	msgbox(NULL), wboost(false),
	facing(true),
	user_controlled(false),
	visible(true), hidden(false), should_respawn(false),
	possession(NULL), hiddenby(NULL),
	current_animation(0)
{
	message.duration = 0;
	
	respawn.x = 0;
	respawn.y = 0;
	respawn.state = 0;
	respawn.stp1 = 0;
	respawn.stp2 = 0;
	respawn.health = fullhealth;
	respawn.gs = 1;	
}

Object::~Object()
{	
	delete afootstep;
	delete adie;
	delete ajump;
	delete ahurt;
	delete acollide;
	
	for( std::vector<AnimPlayer*>::iterator i = anims.begin(); i != anims.end(); i++ )
	{
		delete *i;
	}
	
	destroy_bitmap(msgbox);
}

void Object::Init()
{
	afootstep = new AudioSampler(GetSound());
	adie = new AudioSampler(GetSound());
	ajump = new AudioSampler(GetSound());
	ahurt = new AudioSampler(GetSound());
	acollide = new AudioSampler(GetSound());
}

void Object::ReadProperties(MyINI *ini)
{
	int r, g, b;
	
	if (!strcmp(ini->Variable(), "state"))
	{
		state = ini->ReadInteger();
		stparam1 = ini->ReadInteger();
		stparam2 = ini->ReadInteger();
		SwitchToState(state, stparam1, stparam2);
	}
	if (!strcmp(ini->Variable(), "stateid"))
		state = ini->ReadInteger();
	if (!strcmp(ini->Variable(), "stateparams"))
	{
		stparam1 = ini->ReadInteger();
		stparam2 = ini->ReadInteger();
	}
	if (!strcmp(ini->Variable(), "position"))
	{
		xpos = ini->ReadInteger() << 8;
		ypos = ini->ReadInteger() << 8;
	}
	if (!strcmp(ini->Variable(), "tileposition"))
	{
		xpos = ini->ReadInteger() << 12;
		ypos = ini->ReadInteger() << 12;
	}
	if (!strcmp(ini->Variable(), "shape"))
	{
		w = ini->ReadInteger();
		h = ini->ReadInteger();
	}
	if (!strcmp(ini->Variable(), "solidshape"))
	{
		hx = ini->ReadInteger();
		hy = ini->ReadInteger();
		hw = ini->ReadInteger();
		hh = ini->ReadInteger();
	}
	if (!strcmp(ini->Variable(), "leap"))
	{
		vleap.vlsize = ini->ReadInteger();
		vleap.vlspeed = ini->ReadInteger();
		vleap.leapboost = ini->ReadInteger();
	}
	if (!strcmp(ini->Variable(), "movementspeed"))
	{
		walkspeed = ini->ReadInteger();
		runspeed = ini->ReadInteger();
	}
	if (!strcmp(ini->Variable(), "usercontrolled"))
	{
		user_controlled = ini->ReadBool();
	}
	if (!strcmp(ini->Variable(), "killafterpossess"))
	{
		kill_after_possess = ini->ReadBool();
	}
	if (!strcmp(ini->Variable(), "bpmincolor"))
	{
		r = ini->ReadInteger();
		g = ini->ReadInteger();
		b = ini->ReadInteger();
		bp_mincolor = makecol32(r,g,b);
	}
	if (!strcmp(ini->Variable(), "bpmaxcolor"))
	{
		r = ini->ReadInteger();
		g = ini->ReadInteger();
		b = ini->ReadInteger();
		bp_maxcolor = makecol32(r,g,b);				
	}
	if (!strcmp(ini->Variable(), "respawn"))
		should_respawn = ini->ReadBool();
	if (!strcmp(ini->Variable(), "possessable"))
		possessable = ini->ReadBool();
	if (!strcmp(ini->Variable(), "interactable"))
		interactable = ini->ReadBool();
	if (!strcmp(ini->Variable(), "fullhealth"))
		fullhealth = ini->ReadInteger();
	if (!strcmp(ini->Variable(), "dangerous"))
		dangerous = ini->ReadInteger();
	if (!strcmp(ini->Variable(), "afootstep"))
		afootstep->LoadFromINI(ini);
	if (!strcmp(ini->Variable(), "adie"))
		adie->LoadFromINI(ini);
	if (!strcmp(ini->Variable(), "ajump"))
		ajump->LoadFromINI(ini);
	if (!strcmp(ini->Variable(), "ahurt"))
		ahurt->LoadFromINI(ini);
	if (!strcmp(ini->Variable(), "acollide"))
		acollide->LoadFromINI(ini);
	if (!strcmp(ini->Variable(), "rsposition"))
	{
		respawn.x = ini->ReadInteger() << 8;
		respawn.y = ini->ReadInteger() << 8;
	}
	if (!strcmp(ini->Variable(), "rstileposition"))
	{
		respawn.x = ini->ReadInteger() << 12;
		respawn.y = ini->ReadInteger() << 12;
	}
	if (!strcmp(ini->Variable(), "rsstate"))
	{
		respawn.state = ini->ReadInteger();
		respawn.stp1 = ini->ReadInteger();
		respawn.stp2 = ini->ReadInteger();	
	}
	if (!strcmp(ini->Variable(), "rshealth"))
		respawn.health = ini->ReadInteger() / 100 * fullhealth;
	if (!strcmp(ini->Variable(), "rsgamescreen"))
		respawn.gs = ini->ReadInteger();
	if (!strcmp(ini->Variable(), "rstime"))
		respawn.time = ini->ReadInteger();
}

void Object::LoadProperties(char *path)
{
	MyINI ini;
	char *s;

	ini.LoadINI(path);

	for( int i=0; i<ini.Count(); i++ )
	{
		if (ini.IsAssignment())
			ReadProperties(&ini);
		
		ini.NextLine();
	}
}

void Object::SetGamescreen(int gs_id)
{
	gamescreen = gamescreen->GetPlay()->GetGS(gs_id);
}

void Object::SetGamescreen(GameScreen *gs)
{
	gamescreen = gs;
}

void Object::Draw(BITMAP *dest)
{
	if (!IsOnScreen()) return;
	if (health <= 0) return;
	if (!visible) return;
	
	if (facing)
		draw_sprite(dest, anims[current_animation]->GetCurrentFrame(), xpos>>8, ypos>>8);
	else
		draw_sprite_h_flip(dest, anims[current_animation]->GetCurrentFrame(), xpos>>8, ypos>>8);
	
//	textprintf_ex(dest, font, XPos(), YPos()-10, makecol(255,255,255), -1,
//	"%d", hidden);
//	textprintf_ex(dest, font, XPos(), YPos()-10, makecol(255,255,255), -1,
//	"%d %d %d", state, stparam1, stparam2);
//	textprintf_ex(dest, font, XPos(), YPos()-10, makecol(255,255,255), -1,
//	"%d", xpos>>8);
	if (message.duration)
		MsgboxDraw(dest);
	
	if (show_healthbar || healthbar_life)
		HealthbarDraw(dest);
}

void Object::LogicalUpdate()
{
	if (state == STATE_ENDPOSSESS)
		if (possession->Phase() == 4)
			possession->ChantOut();
	
	if (health <= 0) return;
	
	if (uses_anims)
		anims[current_animation]->FrameTick();
	
	MsgboxUpdate();
	
	if (healthbar_life > 0)
		healthbar_life--;
	
	InvalidateDeltas();
	wboost = false;
	
	if (user_controlled)
		ProcessInput();
	else
		ProcessAI();
}

int Object::VCollision(int x, int y, int side)
{
	x >>= 8; y >>= 8;
	
	int
		collision = 0,
		x1 = (x+hx)>>4,
		x2 = (x+hx+hw)>>4;
	
	y += side > 0 ? hy + hh : hy;
	y >>= 4;
	
	if ((y > 14) || (y < 0)) return 0;
	
	if (x2 > 19) x2 = 19;
	if (x1 < 0) x1 = 0;
	
	for( int i=x1; (i<=x2) && !collision; i++ )
	{
		collision = gamescreen->GetMap()->MapGet(1, i, y);
	}
	
	if ( (side<0) && collision )
	{
		if (IsOnScreen())
			gamescreen->GetParticles()->SpawnParticles(
				CenterX(), y+1<<4,
				rand()%8+2, gamescreen->Identifier(),
				makecol32(255,128,0), makecol32(255,255,0));
	}
	
	return collision;
	
}

int Object::HCollision(int x, int y, int side)
{
	x >>= 8; y >>= 8;
	
	int
		collision = 0,
		y1 = (y+hy)>>4,
		y2 = (y+hy+hh)>>4;
	
	x += side > 0 ? hx + hw : hx;
	x >>= 4;

	if ((x > 19) || (x < 0)) return 0;
	
	if (y2 > 14) y2 = 14;
	if (y1 < 0) y1 = 0;
	
	for( int i=y1; (i<=y2) && !collision; i++ )
	{
		collision = gamescreen->GetMap()->MapGet(1, x, i);
	}
	
	return collision;
}

int Object::IsMissingGround(int x, int y)
{
	x >>= 8; y >>= 8;
	
	if ((x+hx+hw > 319) || (x < 0)) return 0;
	if ((y+hy+hh > 223)) return 1;
	if (y<-1000) return 1;
	
	int
		lp = gamescreen->GetMap()->MapGet(1, (x+hx)/16, (y+hy+hh+1)/16),
		rp = gamescreen->GetMap()->MapGet(1, (x+hx+hw)/16, (y+hy+hh+1)/16);

	return !lp && !rp;
}

int Object::IsEndOfScreen(int x, int y)
{
	x >>= 8; y >>= 8;
	
	if (x+hx <= 0) return 1;
	if (x+hx+hw >= 320) return 1;
	if (y+hy <= 0) return 1;
	if (y+hy+hh >= 240) return 1;
	
	return 0;
}

int Object::VEdgeCol(int y, int side)
{
	y >>= 8;
	
	if (side > 0)
	{
		if (y+hy >= 240) return 2;
		if (y+hy+hh >= 240) return 1;
	}
	else
	{
		if (y+hy+hh <= 0) return 2;
		if (y+hy <= 0) return 1;
	}
	
	return 0;
}

int Object::HEdgeCol(int x, int side)
{
	x >>= 8;
	
	if (side > 0)
	{
		//if (x+hx >= 320) return 2;
		//if (x+hx+hw >= 320) return 1;
		if (x+8 >= 320) return 2;
		if (x+16 >= 320) return 1;
	}
	else
	{
		//if (x+hx+hw <= 0) return 2;
		//if (x+hx <= 0) return 1;
		if (x+16 <= 0) return 2;
		if (x+8 <= 0) return 1;
	}

	return 0;
}

int Object::MoveX(int dx)
{
	bool move_possible = 1;
	
	int eos = HEdgeCol(xpos+dx, dx);
	
	if (eos)
	{
		if (gamescreen->CanExit(dx>0 ? 1 : 0))
		{
			if (eos == 2)
			{
				if (dx>0)
					ExitGamescreen(1);
				else
					ExitGamescreen(0);
				return 1;
			}
			else
			{
				SetDeltaX(dx);
				xpos += dx;
				return 1;
			}
		}
		else return 0;
	}
	
	move_possible = !HCollision(xpos+dx, ypos, dx) || (HCollision(xpos, ypos, +dx) && HCollision(xpos, ypos, -dx));
	
	
	if (move_possible)
	{
		SetDeltaX(dx);
		xpos += dx;
		return 1;
	}
	else return 0;
}

int Object::MoveY(int dy)
{
	bool move_possible = 1;
	
	int eos = VEdgeCol(ypos+dy, dy);
	
	if (eos)
	{
		if (gamescreen->CanExit(dy>0 ? 3 : 2))
		{
			if (eos == 2)
			{
				if (dy>0)
					ExitGamescreen(3);
				else
					ExitGamescreen(2);
				return 1;
			}
			else
			{
				SetDeltaY(dy);
				ypos += dy;
				return 1;
			}
		}
		else return 0;
	}
	
	move_possible = !VCollision(xpos, ypos+dy, dy) || (VCollision(xpos, ypos, +dy) && VCollision(xpos, ypos, -dy));
	
	if ((dy<0) && (!move_possible) && IsOnScreen())
	{
		acollide->Play();
		ajump->Stop();
	}
	
	if (move_possible)
	{
		SetDeltaY(dy);
		ypos += dy;
		return 1;
	}
	else return 0;
}

void Object::PrepareLeap(int boost)
{
	vleap.velocity = (vleap.vlsize + boost << 8) / vleap.vlspeed;
	vleap.acceleration = -vleap.velocity / vleap.vlspeed;
}

void Object::PrepareFall(int boost)
{
	vleap.velocity = 0;
	vleap.acceleration = - (vleap.vlsize + boost << 8) / vleap.vlspeed / vleap.vlspeed;
}

int Object::LeapFrametick()
{
	if (MoveY(-vleap.velocity))
	{
		vleap.velocity += vleap.acceleration;
		if (vleap.velocity < 0)
			return 0;
		else
			return 1;
		
	}
	else
	{
		PrepareFall(0);
		return 0;
	}
}

int Object::FallFrametick()
{
	if (MoveY(-vleap.velocity))
	{
		vleap.velocity += vleap.acceleration;
		return 1;
	}
	else
	{
		ypos = (((((ypos-vleap.velocity)>>8)+hy+hh)/16)*16-hh-hy-1)<<8;
		return 0;
	}
}

void Object::InitState(int st)
{
	switch( st )
	{
		case STATE_WALKL:
			afootstep->Reset();
			TurnLeft();
			break;
		case STATE_WALKR:
			afootstep->Reset();
			TurnRight();
			break;
		case STATE_FALLING:
			PrepareFall(0);
			break;
		case STATE_JUMPING:
			if (IsOnScreen())
				ajump->Play();
			if (wboost)
				PrepareLeap(vleap.leapboost);
			else
				PrepareLeap(0);
			break;
		case STATE_ENDPOSSESS:
			possession->PrepareChantOut();
			break;
		case STATE_KILL:
			if (should_respawn)
			{
				if (is_character)
					gamescreen->RemoveCharacter(this);
				else
					gamescreen->RemoveObject(this);
			}
			else
				{
					if (is_character)
						gamescreen->KillCharacter(this);
					else
						gamescreen->KillObject(this);
				}
			break;
	}	
}

void Object::SwitchToState(int st, int param1, int param2)
{
	state = st;
	InitState(state);
	ChangeParams(param1, param2);
}

void Object::ChangeParams(int param1, int param2)
{
	stparam1 = param1;
	stparam2 = param2;
}

void Object::ProcessInput()
{
	Buttons *input = gamescreen->GetButtons();
	char buf[64];
	
	wboost = key[KEY_LSHIFT];
	
	switch( state )
	{
		case STATE_IDLE:
			if (key[KEY_LEFT])
				SwitchToState(STATE_WALKL, 0, 0);
			else if (key[KEY_RIGHT])
					SwitchToState(STATE_WALKR, 0, 0);
			break;
		case STATE_WALKL:
			MoveX(wboost ? -runspeed : -walkspeed);
			if (key[KEY_RIGHT] && !key[KEY_LEFT])
				SwitchToState(STATE_WALKR, 0, 0);
			else
				if (!key[KEY_LEFT]) SwitchToState(STATE_IDLE, 0, 0);
			if (IsOnScreen())
			{
				afootstep->Frametick();
			}
			if (IsMissingGround(xpos, ypos))
				SwitchToState(STATE_FALLING, 0, 0);
			break;
		case STATE_WALKR:
			MoveX(wboost ? +runspeed : +walkspeed);
			if (key[KEY_LEFT] && !key[KEY_RIGHT])
				SwitchToState(STATE_WALKL, 0, 0);
			else
				if (!key[KEY_RIGHT])
					SwitchToState(STATE_IDLE, 0, 0);
			if (IsOnScreen())
			{
				afootstep->Frametick();
			}
			if (IsMissingGround(xpos, ypos))
				SwitchToState(STATE_FALLING, 0, 0);
			break;
		case STATE_JUMPING:
			if (!LeapFrametick()) state = STATE_FALLING;
			if (key[KEY_LEFT]) { MoveX(wboost ? -runspeed : -walkspeed); TurnLeft(); }
			if (key[KEY_RIGHT]) { MoveX(wboost ? +runspeed : +walkspeed); TurnRight(); }
			break;
		case STATE_FALLING:
			if (!FallFrametick())
			{
				SwitchToState(STATE_IDLE, 0, 0);
				if (IsOnScreen())
					afootstep->Play();
			}
			else
				{
					if (key[KEY_LEFT]) { MoveX(wboost ? -runspeed : -walkspeed); TurnLeft(); }
					if (key[KEY_RIGHT]) { MoveX(wboost ? +runspeed : +walkspeed); TurnRight(); }
				}
			break;
		case STATE_ENDPOSSESS:
			if (!input->KeyDown(BTN_ACT1) && (possession->Phase() != 4))
				SwitchToState(STATE_IDLE, 0, 0);
			else
				possession->ChantOut();
			break;
	}
	
	if (input->KeyPress(BTN_UP) && (state != STATE_FALLING) && (state != STATE_JUMPING))
		SwitchToState(STATE_JUMPING, 0, 0);
	
	if (possessable && possession)
		if (input->KeyPress(BTN_ACT1) && ((state == STATE_IDLE) || (state == STATE_WALKL) || (state == STATE_WALKR)))
			SwitchToState(STATE_ENDPOSSESS, 0, 0);
	
	if (key[KEY_SPACE])
		ShowHealthbar();
	else
		HideHealthbar();
}

void Object::ExitGamescreen( int exitcode )
{
	int new_gs;
	MAPEX mi = gamescreen->GetMapinfo();

	switch( exitcode )
	{
		case 0: // Left screen edge
			new_gs = mi.left;
			SetRight();
			break;
		case 1: // Right screen edge
			new_gs = mi.right;
			SetLeft();
			break;
		case 2: // Top screen edge
			new_gs = mi.top;
			SetBottom();
			break;
		case 3: // Bottom screen edge
			new_gs = mi.bottom;
			SetTop();
			break;
	}
	
	if (!main_character)
		gamescreen->RemoveCharacter(this);
	SetGamescreen(new_gs);
	if (!main_character)
		gamescreen->AddCharacter(this);
	
	if (user_controlled)
		gamescreen->GetPlay()->GoToScreen(new_gs);
}

void Object::Kill()
{
	if (IsOnScreen())
	{
		adie->Play();
		gamescreen->GetParticles()->SpawnParticles(
			CenterX(), CenterY(),
			0, gamescreen->Identifier(),
			bp_mincolor, bp_maxcolor);
	}
	if (main_character)
		gamescreen->GetPlay()->Die();
	else
		{
			if (should_respawn)
				gamescreen->SetupRespawn(this);
		}
}

void Object::Destroy()
{
	Kill();
	if (!main_character)
		SwitchToState(STATE_KILL, 0, 0);
}

void Object::Death()
{
	health = 0;
	if (IsOnScreen())
	{
		ahurt->Play();
		healthbar_life = 100;
	}
	if (!main_character)
		if (should_respawn)
			gamescreen->SetupRespawn(this);
	if (possession && !main_character)
	{
		possession->Deactivate(STATE_KILL);
		gamescreen->GetPlay()->GoToScreen(gamescreen->GetPlay()->GetMainCharacter()->GS()->Identifier());
	}
	else
		Destroy();
}

void Object::DamageWD( int hit )
{
	if (health>0)
	{
		health-=hit;
		if (IsOnScreen())
		{
			for (int i=0; i<hit; i++)
				ahurt->Frametick();
			gamescreen->GetParticles()->SpawnParticles(
				CenterX(), CenterY(),
				hit, gamescreen->Identifier(),
				bp_mincolor, bp_maxcolor);
			healthbar_life = 100;
		}
	}
	if (health <= 0)
	{
		if (possession && !main_character)
		{
			if (should_respawn)
				gamescreen->SetupRespawn(this);
			possession->Deactivate(STATE_KILL);
			gamescreen->GetPlay()->GoToScreen(gamescreen->GetPlay()->GetMainCharacter()->GS()->Identifier());
		}
		else
			Destroy();
	}
}

void Object::Damage( int hit )
{
	if (health>0)
	{
		health-=hit;
		if (IsOnScreen())
		{
			for (int i=0; i<hit; i++)
				ahurt->Frametick();
			gamescreen->GetParticles()->SpawnParticles(
				CenterX(), CenterY(),
				hit, gamescreen->Identifier(),
				bp_mincolor, bp_maxcolor);
			healthbar_life = 100;
		}
	}
	if (health <= 0)
		Kill();
//		Destroy();
}

bool Object::IsOnScreen()
{
	return gamescreen == gamescreen->GetPlay()->GetActualGS();
}

void Object::SetMessage(int type, int rtype, Object *reciever, int d)
{
	message.sender = this;
	message.msgtype = type;
	message.reftype = rtype;
	message.reciever = reciever;
	message.duration = d;
	
	if ((type == MSG_CALL) && (!rtype))
		gamescreen->ResetInteraction();
	
	destroy_bitmap(msgbox);
	
	const char *text = gamescreen->GetResources()->GetRndMsgtext(type, rtype);
	int
		w = text_length(font, text)+8,
		h = text_height(font)+8;
	
	msgbox = create_bitmap(w, h);
	
	// Constructing message-box bitmap
	
	// Transparent corners
	rectfill(msgbox, 0, 0, 1, 1, makecol(255, 0, 255));
	rectfill(msgbox, w-2, h-2, w-1, h-1, makecol(255, 0, 255));
	rectfill(msgbox, 0, h-2, 1, h-1, makecol(255, 0, 255));
	rectfill(msgbox, w-2, 0, w-1, 1, makecol(255, 0, 255));
	// Horizontal sides
	rectfill(msgbox, 2, 0, w-3, 1, makecol(0, 0, 0));
	rectfill(msgbox, 2, h-1, w-3, h-2, makecol(0, 0, 0));
	// Vertical sides
	rectfill(msgbox, 0, 2, 1, h-3, makecol(0, 0, 0));
	rectfill(msgbox, w-2, 2, w-1, h-3, makecol(0, 0, 0));
	// Text-space
	rectfill(msgbox, 2, 2, w-3, h-3, makecol(0, (rand()%95)+160, (rand()%95)+160));
	// Drawing text
	textout_ex(msgbox, font, text, 4, 4, makecol(0,0,0), -1);
}

void Object::MsgboxUpdate()
{
	if (message.duration > 0)
		message.duration--;
}

void Object::MsgboxDraw(BITMAP *dest)
{
	int
		x = xpos>>8,
		y = ypos>>8;
	
	x = x+w+msgbox->w < dest->w ? x+w : x-msgbox->w;
	y = y-msgbox->h > 0 ? y-msgbox->h : y;
	
	draw_sprite(dest, msgbox, x, y);
}

void Object::HealthbarDraw(BITMAP *dest)
{
	int
		x = xpos>>8,
		y = ypos>>8;
	
	if (y-8 > 0)
	{
		rectfill(dest, x, y-8, x+w, y-4, makecol(62,0,0));
		rectfill(dest, x+1, y-7, x+health*(w-2)/fullhealth+1, y-5, makecol(128,0,0));
		rect(dest, x, y-8, x+w, y-4, makecol(0,0,0));
	}
	else
		{
			rectfill(dest, x, y+h+8, x+w, y+h+4, makecol(62,0,0));
			rectfill(dest, x+1, y+h+7, x+health*(w-2)/fullhealth+1, y+h+5, makecol(128,0,0));
			rect(dest, x, y+h+8, x+w, y+h+4, makecol(0,0,0));
		}
}

Sound *Object::GetSound()
{
	return gamescreen->GetResources()->GetSound();
}

int Object::Curgs()
{ return gamescreen->Identifier(); }

void Object::ReadMessages()
{
	int i;
	MESSAGE msg;
	
	for(i=0; i < gamescreen->MsgCount(); i++)
	{
		msg = gamescreen->Message(i);
		
		if (msg.sender == this) continue;
		
		if  ((msg.reftype==1) || ((msg.msgtype==MSG_CALL) && (msg.reciever==this)) ||
			((msg.msgtype!=MSG_CALL) && (msg.sender==callingparty_obj)))
				ProcessMessage(msg);
	}
}

void Object::ProcessAI()
{
	ReadMessages();
}

void Object::Respawn()
{
	xpos = respawn.x;
	ypos = respawn.y;
	state = respawn.state;
	stparam1 = respawn.stp1;
	stparam2 = respawn.stp2;
	health = respawn.health;
	
	if (is_character)
		gamescreen->GetPlay()->GetGS(respawn.gs)->AddCharacter(this);
	else
		gamescreen->GetPlay()->GetGS(respawn.gs)->AddObject(this);
}

bool Object::Intersect(Object *obj)
{
	bool
		gscond = gamescreen == obj->GS(),
		/*
		xcond = ((obj->SolidLeft() >= SolidLeft()) && (obj->SolidLeft() <= SolidRight())) ||
			((obj->SolidRight() >= SolidLeft()) && (obj->SolidRight() <= SolidRight())),
		ycond = ((obj->SolidTop() >= SolidTop()) && (obj->SolidTop() <= SolidBottom())) ||
			((obj->SolidBottom() >= SolidTop()) && (obj->SolidBottom() <= SolidBottom()));*/
		icond =
			!(obj->SolidLeft() > SolidRight()
			|| obj->SolidRight() < SolidLeft()
			|| obj->SolidTop() > SolidBottom()
			|| obj->SolidBottom() < SolidTop());
	
	
	return gscond && icond;
}