//This file is part of Future's End
//Copyright 2006-2008 SiegeLord
//See license.txt for distribution information
//
//game.cpp
//Handles the game logic etc

#include "mouse.h"
#include "types.h"
#include "input_functions.h"
#include "galaxy.h"
#include "gfx.h"
#include "wave.h"
#include "menu.h"
#include "planet.h"
#include "ai.h"
#include "sound.h"
#include "palette.h"
#include "game.h"
#include "bomb.h"
#include "npc.h"

int g_nChatSize = 0;

extern bool g_bQuit;

extern int g_nPlayerRace;
extern int g_nPlayerNumber;
extern string g_sPlayerName;

extern int g_nGameType;

extern int g_nMaxPoints;

float g_fDamageMultiplier = 1.0f;

vector<string> g_sChatBuffer;//buffer for the chat

void DrawGrid(int view_x, int view_y)//draws the grid
{
#define _SPACING 128

	int num_x = SCREEN_W / _SPACING + 2;
	int num_y = SCREEN_H / _SPACING + 2;

	int dx = -view_x + int(view_x / _SPACING) * _SPACING + 0;
	int dy = -view_y + int(view_y / _SPACING) * _SPACING + 0;

	int x = 0;
	int y = 0;
	for(int Y = 0; Y < num_y; Y++)
	{
		y = dy + _SPACING * Y;
		for(int X = 0; X < num_x; X++)
		{
			x = dx + _SPACING * X;
			//draws the crosses
			
			vline(GetDrawPage(), x, y - 10, y + 10, 101);
			hline(GetDrawPage(), x - 10, y, x + 10, 101);

			//draws the little dots
			for(int ii = 0; ii < 16; ii++)
			{
				putpixel(GetDrawPage(), x, y + ii * _SPACING / 16, 101);
				putpixel(GetDrawPage(), x + ii * _SPACING / 16, y, 101);
			}
		}
	}
}

//harvests the star matter around _x, _y within the radius
//also returns the ammount of energy left
int HarvestStarMatter(BITMAP* galaxy, int _x, int _y, int radius, int* energy_left)
{
#define _HARVEST(a,b)\
	{																		\
	__x = _x + a; __y = _y + b;												\
	val = getpixel(galaxy, __x, __y);										\
	if(val > 0)																\
	{																		\
		val -= amt;															\
		if(val < 0)															\
		{																	\
			total += amt + val;												\
			val = 0;														\
		}																	\
		else																\
		{																	\
			total += amt;													\
			total_left += val;												\
		}																	\
		_putpixel(galaxy, __x, __y, val);									\
	}																		\
	}																		\
	
	float r2 = radius;
	r2 *= r2;

	float factor = -10.1f / (float(radius));

	int total = 0;
	int total_left = 0;

	for(int x = 0; x < radius; x++)
	{
		float x2 = float(x);
		x2 *= x2;
		int dy = sqrt(r2 - x2);
		for(int y = 0; y <= dy; y++)
		{
			int __x, __y, val;
			float y2 = y;
			y2 *= y2;
			int amt = sqrtf(x2 + y2) * factor + 10.0f;
			_HARVEST(x,y);
			if(y != 0)
				_HARVEST(x,-y);
			if(x != 0)
			{
				_HARVEST(-x,y);
				if(y != 0)
					_HARVEST(-x, -y);
			}
		}
	}
	if(energy_left)
		*energy_left = total_left;
	return total;
}

//current turn in the action, ranges from 0 to 100
int turn = 0;

//a little function I made to clip n to cap... no idea why this is used only here though
inline int clip(int n, int cap)
{
	return n > cap ? cap : n;
}

//the action function
//this performs most of the game logic during the action phase
//it is called 100 times before it returns false, when the calling should be stopped
bool Action(ActionParam &p)
{
	vector<SPlanet> *planets = p.planets;
	list<SWave> *waves = p.waves;
	list<SParticle> *particles = p.particles;
	list<SBomb> *bombs = p.bombs;
	list<SNPC> *npcs = p.npcs;
	BITMAP* galaxy = p.galaxy;
	BITMAP* subspace = p.subspace;

	int* total_turn = p.total_turn;	//used for total time
	int player = p.player;			//the player who is on this computer(used for multiplayer)

	*total_turn = *total_turn + 1;	//advance the timer

	if(turn == 0)//the start of the action
		//we fire the weapons
	{
		//loop through the planets to see if they fired anything
		for(unsigned int ii = 0; ii < planets->size(); ii++)
		{
			SPlanet* planet = &((*planets)[ii]);

			if(planet->dead)//no undead allowed
				continue;

			if(planet->state == 1 || planet->state == 2)//state 1 is the standard weapon
														//state 2 is the subspace
			{
				float energy = planet->state_val3;//the chosen energy setting...

				if(energy > planet->energy)//...which can't be bigger than the planets actual energy reserves
					energy = planet->energy;

				if(planet->race == 1)//anarchists
				{
					if(planet->state == 1)
						PlaySound((SAMPLE*)g_pData[_Cannon1].dat, planet->x, planet->y, false);
					else
						PlaySound((SAMPLE*)g_pData[_Cannon2].dat, planet->x, planet->y, false);

					float theta = planet->state_val1;
					SBomb bomb;
					bomb.x = planet->x;
					bomb.y = planet->y;
					bomb.radius = 0;
					bomb.parent = true;

					bomb.life = planet->state_val2;

					bomb.energy = energy;
					bomb.source = ii;
					bomb.wait = 0;
					bomb.theta = theta;
					bomb.subspace = planet->state == 2;
					bombs->push_back(bomb);
					
					planet->state = -1;
				}
				else if(planet->race == 0)//communists
				{
					SWave wav;
					
					if(planet->state == 1)
						PlaySound((SAMPLE*)g_pData[_Wave].dat, planet->x, planet->y, false);
					else
						PlaySound((SAMPLE*)g_pData[_Wave2].dat, planet->x, planet->y, false);

					wav.x = planet->x;
					wav.y = planet->y;

					wav.angle2 = planet->state_val2;
					wav.angle1 = planet->state_val1;

					wav.source = ii;

					float span = GetSpan(wav.angle1, wav.angle2);

					wav.energy = energy * 10 / span;
					wav.radius = 5;	//This makes some weird bugs associated with the wave algorithm
									//to go away
					wav.subspace = planet->state == 2;
					for(int ii = 0; ii < 10; ii++)
					{
						if(ii == 0)
							wav.energies.resize(10);
						wav.energies[ii] = wav.energy / 10;
					}

					waves->push_back(wav);

					planet->state = -1;
				}
				else //capitalists
				{
					if(planet->state == 1)//we fire the pod
					{
						SNPC npc;
						npc.x = planet->x;
						npc.y = planet->y;
						npc.type = 0;
						npc.val1 = 30;
						npc.val2 = planet->state_val2;
						npc.val3 = energy;
						npc.val4 = ii;
						npc.theta = planet->state_val1;

						PlaySound((SAMPLE*)g_pData[_Npc_launch].dat, planet->x, planet->y, false);

						npcs->push_back(npc);
					}
					else//we fire the burst
					{
						SBomb bomb;
						bomb.x = planet->x;
						bomb.y = planet->y;
						bomb.subspace = true;
						bomb.parent = false;
						bomb.source = ii;
						bomb.energy = energy;
						bomb.life = energy;
						bomb.radius = 0;
						
						PlaySound((SAMPLE*)g_pData[_Burst].dat, planet->x, planet->y, false);

						for(int jj = 0; jj < _BOMB_NUMBER; jj++)
						{
							bomb.wait = rande()*20;
							bomb.theta = planet->state_val1 + float(rande()* 200 - 100) / 100.0f * planet->state_val2;
							bombs->push_back(bomb);
						}
					}
					planet->state = -1;
				}

				planet->energy -= energy;

				//spawn the flash particle
				SParticle part;
				part.bmp = p.flash[clip(energy / 20000, 4)];
				part.life = 10;
				part.x = planet->x;
				part.y = planet->y;

				particles->push_back(part);
			}
		}		
	}

	if(turn == 50)//the halfway point, we move and build generators or shields
	{
		for(unsigned int ii = 0; ii < planets->size(); ii++)
		{
			SPlanet* planet = &((*planets)[ii]);
			
			if(planet->dead)
				continue;

			if(planet->state == 0)//move state
			{
				SParticle part;
				part.bmp = p.flash[clip(planet->state_val4 / 20000, 4)];
				part.life = 10;
				part.x = planet->x;
				part.y = planet->y;

				particles->push_back(part);

				planet->x = planet->state_x;
				planet->y = planet->state_y;

				PlaySound((SAMPLE*)g_pData[_Teleport].dat, planet->x, planet->y, false);

				planet->energy -= planet->state_val4;

				part.x = planet->x;
				part.y = planet->y;

				particles->push_back(part);

				planet->state = -1;
			}
			if(planet->state == 3)//build generator
			{
				if(planet->energy >= planet->state_val4)
				{
					if(ii == (unsigned)player)
						PlaySound((SAMPLE*)g_pData[_Generator].dat, planet->x, planet->y, false);
					planet->energy -= planet->state_val4;
					planet->num_generators++;
				} 
				planet->state = -1;
			}
			if(planet->state == 4)//build shield
			{
				if(planet->energy >= planet->state_val4)
				{
					if(ii == (unsigned)player)
						PlaySound((SAMPLE*)g_pData[_Shield_up].dat, planet->x, planet->y, false);
					planet->energy -= planet->state_val4;
					planet->num_shields++;
					planet->shield += 1000;
				}
				planet->state = -1;
			}
		}
	}

	if(turn % 33 == 0)//the third of an action phase, we harvest energy
	{
		for(unsigned int ii = 0; ii < planets->size(); ii++)
		{
			SPlanet* planet = &((*planets)[ii]);
			if(planet->dead)
				continue;
			planet->energy += HarvestStarMatter(galaxy, planet->x, planet->y, 
				planet->num_generators * 4 + 10, &planet->energy_left);
			planet->energy += 250 * planet->num_generators;
		}
	}

	//the following happens every turn
	
	//handle the waves
	for(list<SWave>::iterator it = waves->begin(); it != waves->end(); it++)
	{
		SWave* wav = &(*it);

		MoveWave(planets, npcs, wav, galaxy, subspace);

		if(wav->energy < 10)//kill the waves(not 0, since some can magically retain some energy
							//almost forever)
			it = waves->erase(it);
		else if(wav->energies.size() == 0)
			it = waves->erase(it);
	}
	
	//handle the bombs
	for(list<SBomb>::iterator it = bombs->begin(); it != bombs->end(); it++)
	{
		SBomb* bomb = &(*it);

		MoveBomb(planets, npcs, bomb, galaxy, subspace);

		if(bomb->life < 0)//the bomb has run out of life
		{
			if(bomb->parent)//we detonate
			{
				SBomb nbomb;//initialize the template

				int bx = bomb->radius * cosf(bomb->theta);//the bomb's starting positions
				int by = bomb->radius * sinf(bomb->theta);

				nbomb.x = bomb->x + bx;
				nbomb.y = bomb->y + by;
				nbomb.radius = 0;
				nbomb.parent = false;
				nbomb.life = _BOMB_LIFE;
				nbomb.energy = bomb->energy / 5 / 3;
				nbomb.subspace = bomb->subspace;
				nbomb.source = bomb->source;
				
				//spawn particle
				SParticle part;
				part.bmp = p.flash[clip(bomb->energy / 20000, 4)];
				part.life = 10;
				part.x = bomb->x + bx;
				part.y = bomb->y + by;

				particles->push_back(part);

				PlaySound((SAMPLE*)g_pData[_Cannon3].dat, nbomb.x, nbomb.y, false);

				for(int ii = 0; ii < _BOMB_NUMBER; ii++)//spawn particles
				{
					nbomb.wait = rande()*20;
					nbomb.theta = float(rande()* 360) / 360.0f * 2 * AL_PI;
					bombs->push_back(nbomb);
				}
			}
			it = bombs->erase(it);
		}
	}

	//handle npcs
	for(list<SNPC>::iterator it = npcs->begin(); it != npcs->end(); it++)
	{
		SNPC* npc = &(*it);

		int ret = MoveNPC(planets, npcs, bombs, npc, galaxy, subspace);

		if(ret == 0)//we died
		{
			PlaySound((SAMPLE*)g_pData[_Npc_death].dat, npc->x, npc->y, false);
			SParticle part;
			part.bmp = p.flash[0];
			part.life = 10;
			part.x = npc->x;
			part.y = npc->y;
			particles->push_back(part);
			it = npcs->erase(it);
		}
	}

	//handle particles
	for(list<SParticle>::iterator it = particles->begin(); it != particles->end(); it++)
	{
		SParticle* part = &(*it);

		part->life--;
	
		if(part->life < 0)
			it = particles->erase(it);
	}

	//check if planet is alive, handle shield regeneration
	for(unsigned int ii = 0; ii < planets->size(); ii++)
	{
		SPlanet* planet = &((*planets)[ii]);

		if(planet->dead)
			continue;
		if(planet->shield < 0)
		{
			if(g_nGameType == 0)
			{
				planet->dead = true;
				if((unsigned)planet->last_hit_by == ii)//killed self
				{
					string entry = planet->name + string(" comitted suicide...");
					PutInChat(entry.c_str());
					planet->score--;
				}
				else if(planet->last_hit_by != -1)//someone else killed it
				{
					string entry = planet->name + string(" was destroyed by ") + (*planets)[planet->last_hit_by].name;
					PutInChat(entry.c_str());

					(*planets)[planet->last_hit_by].score++;
				}
					
				PlaySound((SAMPLE*)g_pData[_Death].dat, planet->x, planet->y, false);
				SParticle part;
				part.bmp = p.flash[4];
				part.life = 10;
				part.x = planet->x;
				part.y = planet->y;

				particles->push_back(part);
			}
			else
			{
				planet->shield = 0;
			}
		}
		if(planet->shield < planet->num_shields * 1000)
		{
			if(planet->energy > 1000)
			{
				planet->energy -= 50 * planet->num_shields;		
				if(planet->energy < 1000)	//we do not want to deprive the planet of all of its energy
											//due to the shield regeneration
				{
					planet->energy += 50 * planet->num_shields;
				}
				else
				{
					planet->shield += 0.5 * planet->num_shields + 1;
					if(planet->shield > 1000 * planet->num_shields)
						planet->shield = 1000 * planet->num_shields;
				}
			}
		}

		if(planet->energy < 0)
			planet->energy = 0;
	}

	turn++;//advance the turn

	if(turn >= 100)//end of turn, if we are in bounty we handle the circles etc, otherwise we end
	{
		turn = 0;

		if(g_nGameType == 1)
		{
			SNPC bounty;
			bounty.type = 1;
			bounty.val1 = 100;
			bounty.val2 = 12 * 16;

			for(unsigned int ii = 0; ii < planets->size(); ii++)
			{
				bounty.x = int(rande()* (2 * 600) - 600 + (*planets)[ii].x) % (2 * _GALAXY_RADIUS);
				bounty.y = int(rande()* (2 * 600) - 600 + (*planets)[ii].y) % (2 * _GALAXY_RADIUS);

				SParticle part;
				part.bmp = p.flash[0];
				part.life = 10;
				part.x = bounty.x;
				part.y = bounty.y;
				particles->push_back(part);

				npcs->push_back(bounty);
			}
		}
		return false;
	}
	return true;
}

//#pragma optimize( "", on )//turn the optimization back on

//a simple point in rectangle check
bool InRect(int x, int y, int x1, int y1, int x2, int y2)
{
	return x > x1 && x < x2 && y > y1 && y < y2;
}

//handle the input for the player
void HandlePlanetInput(BITMAP* subspace, vector<SPlanet> *planets, int player, int view_x, int view_y)
{
	SPlanet* planet = &((*planets)[player]);

	if(planet->dead)
		return;

	//handle energy setting for weapons
	if(IsKeyJustPressed(KEY_PGUP))
	{
		IncreaseWavePower(planet);
		PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	}

	//handle energy setting for weapons
	if(IsKeyJustPressed(KEY_PGDN))
	{
		DecreaseWavePower(planet);
		PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	}

	//sets the move state
	if(IsMKeyDown(0))
	{
		int x = 0;
		int y = 0;
		GetMousePos(&x, &y);

		//clips it to the gui areas
		if(!InRect(x, y, SCREEN_W - 100, 0, SCREEN_W - 20, 15) && 
			!InRect(x, y, SCREEN_W / 2 - 100, SCREEN_H - 30, SCREEN_W / 2 + 100, SCREEN_H) && 
			!InRect(x, y, SCREEN_W - 100, SCREEN_H - 15, SCREEN_W - 20, SCREEN_H))
		{
			x += view_x;
			y += view_y;
			
			SetMoveState(planet, x, y, subspace);
		}
	}
	//sets the generator state
	if(IsKeyJustPressed(KEY_G))
	{
		SetGeneratorState(planet);
		PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	}
	//sets the shield state
	if(IsKeyJustPressed(KEY_S))
	{
		SetShieldState(planet);
		PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	}
	//clears all of the states
	if(IsKeyJustPressed(KEY_ESC))
	{
		if(planet->state != -1)
			planet->state = -1;
		PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	}
	//sets the weapon stuff
	if(IsMKeyJustPressed(1))
	{
		int x = 0;
		int y = 0;
		GetMousePos(&x, &y);
		x += view_x;
		y += view_y;
		if(planet->race == 0)//for communists we initialize the wave parameters
		{
			SetIniWaveState(planet, x, y, IsShiftDown());
		}
		else if(planet->race == 2)	//since the burst action of the capitalists is wavelike
									//it needs the initial parameters too
		{
			if(IsShiftDown())
				SetIniBurstState(planet, x, y, true);
		}
	}

	//further sets the weapon parameters
	if(IsMKeyDown(1))
	{
		int x = 0;
		int y = 0;
		GetMousePos(&x, &y);
		x += view_x;
		y += view_y;

		if(planet->race == 0)//handles the angular size of the wave stuff
		{
			SetWaveState(planet, x, y);
		}
		else if(planet->race == 1)//simply handles the direction the bomb will go
		{
			SetBombState(planet, x, y, IsShiftDown(), subspace);
		}
		else
		{
			if(IsShiftDown())//handles the angular size of the burst
			{
				SetBurstState(planet, x, y);
			}
			else//handles the pod direction etc
			{
				SetPodState(planet, x, y, subspace);
			}
		}
	}
}

//draws planets
void DrawPlanets(vector<SPlanet> *planets, int player, int view_x, int view_y)
{
	for(unsigned int ii = 0; ii < planets->size(); ii++)
	{
		SPlanet* planet = &((*planets)[ii]);
		if(planet->dead)
			continue;
		int x = planet->x - view_x;
		int y = planet->y - view_y;

		if(ii == (unsigned)player)//we draw the generator radius for the player
						//also, we draw stuff associated with the orders
		{
			int radius = planet->num_generators * 4 + 10;
			aa_circle(GetDrawPage(), x, y, radius, 127);

			if(planet->state == 0)
				aa_line(GetDrawPage(), x, y, planet->state_x - view_x,
					planet->state_y - view_y, 112);

			putpixel(GetDrawPage(), x, y, 127);//hehe, the planet itself is only a pixel wide!

			if(planet->state == 1 || planet->state == 2)
			{
				if(planet->race == 0)//communists have two lines designating the wave extent
				{
					aa_line(GetDrawPage(), x, y, cosf(planet->state_val1)
						* 3000.0f + x, 
						sinf(planet->state_val1) * 3000.0f + y, 112);

					aa_line(GetDrawPage(), x, y, cosf(planet->state_val2)
						* 3000.0f + x, 
						sinf(planet->state_val2) * 3000.0f + y, 112);
				}
				else if(planet->race == 1)//anarchists have a line with markers
				{
					aa_line(GetDrawPage(), x, y, planet->state_x - view_x,
						planet->state_y - view_y, 112);
					float distance = sqrtf((planet->x - planet->state_x) * (planet->x - planet->state_x)
						+ (planet->y - planet->state_y) * (planet->y - planet->state_y));
					float d = 100;
					float cs = cosf(planet->state_val1);
					float sn = sinf(planet->state_val1);
					float dx = 5 * sn;
					float dy = -5 * cs;
					while(d < distance)//draw the markers
					{
						int X = d * cs + x;
						int Y = d * sn + y;
						aa_line(GetDrawPage(), X - dx, Y - dy, X + dx, Y + dy, 112);
												
						d += 100;
					}
				}
				else//capitalists either have the two lines, or the marked line
				{	
					if(planet->state == 1)//this needs the line
					{
						aa_line(GetDrawPage(), x, y, planet->state_x - view_x,
							planet->state_y - view_y, 112);
						float distance = sqrtf((planet->x - planet->state_x) * (planet->x - planet->state_x)
							+ (planet->y - planet->state_y) * (planet->y - planet->state_y));
						float d = 100;
						float cs = cosf(planet->state_val1);
						float sn = sinf(planet->state_val1);
						float dx = 5 * sn;
						float dy = -5 * cs;
						while(d < distance)
						{
							int X = d * cs + x;
							int Y = d * sn + y;
							aa_line(GetDrawPage(), X - dx, Y - dy, X + dx, Y + dy, 112);
													
							d += 100;
						}
					}
					else//this needs the two lines
					{
						aa_line(GetDrawPage(), x, y, cosf(planet->state_val1 - planet->state_val2)
							* 3000.0f + x, 
							sinf(planet->state_val1 - planet->state_val2) * 3000.0f + y, 112);

						aa_line(GetDrawPage(), x, y, cosf(planet->state_val1 + planet->state_val2)
							* 3000.0f + x, 
							sinf(planet->state_val1 + planet->state_val2) * 3000.0f + y, 112);
					}
				}
			}
		}
		else//non player planets just get the pixel
		{
			putpixel(GetDrawPage(), x, y, 127);
		}

		int color = 60 + 60 * sinf(float(g_nTime) / 120 * 2 * AL_PI);//for cycling of the colour

		if(planet->race == 0)//communists have 8 radial lines
		{
			aa_line(GetDrawPage(), x + 5, y, x + 10, y, color);
			aa_line(GetDrawPage(), x - 5, y, x - 10, y, color);
			aa_line(GetDrawPage(), x, y - 5, x, y - 10, color);
			aa_line(GetDrawPage(), x, y + 5, x, y + 10, color);

			aa_line(GetDrawPage(), x + 4, y + 4, x + 7, y + 7, color);
			aa_line(GetDrawPage(), x - 4, y - 4, x - 7, y - 7, color);
			aa_line(GetDrawPage(), x - 4, y + 4, x - 7, y + 7, color);
			aa_line(GetDrawPage(), x + 4, y - 4, x + 7, y - 7, color);
		}
		else if(planet->race == 1)//anarchists have 4 lines
		{
			aa_line(GetDrawPage(), x + 10, y - 3, x + 10, y + 3, color);
			aa_line(GetDrawPage(), x - 10, y - 3, x - 10, y + 3, color);
			aa_line(GetDrawPage(), x - 3, y - 10, x + 3, y - 10, color);
			aa_line(GetDrawPage(), x - 3, y + 10, x + 3, y + 10, color);
		}
		else//capitalists have 3 lines
		{
			aa_line(GetDrawPage(), x, y + 5, x, y + 10, color);
			aa_line(GetDrawPage(), x - 4, y - 3, x - 9, y - 6, color);
			aa_line(GetDrawPage(), x + 4, y - 3, x + 9, y - 6, color);
		}
	}
}

//draws the planet names
void DrawPlanetNames(vector<SPlanet> *planets, int player, int view_x, int view_y)
{
	int height = text_height(font) / 2;
	int mx,my;
	GetMousePos(&mx, &my);

	for(unsigned int ii = 0; ii < planets->size(); ii++)
	{
		SPlanet* planet = &((*planets)[ii]);

		if(planet->dead)
			continue;

		int x = planet->x - view_x;
		int y = planet->y - view_y;
			
		char buf[40];

		if(abs(x - mx) < 10 && abs(y - my) < 10)//if mouse close to the planet we draw its name
		{
			sprintf(buf, planet->name.c_str());
			textout_centre_ex(GetDrawPage(), font, buf, x, y + 22 - height, 127, -1);
		}
		else if(ii != (unsigned)player)
		{
			sprintf(buf, "%d", ii);
			textout_centre_ex(GetDrawPage(), font, buf, x, y + 22 - height, 127, -1);
		}
	}
	solid_mode();
}

//draws the waves
void DrawWaves(list<SWave> *waves, int view_x, int view_y)
{
	for(list<SWave>::iterator it = waves->begin(); it != waves->end(); it++)
	{
		SWave* wav = &(*it);
		DrawWave(wav, view_x, view_y);
	}
}

//draw the bombs
void DrawBombs(list<SBomb> *bombs, int view_x, int view_y)
{
	for(list<SBomb>::iterator it = bombs->begin(); it != bombs->end(); it++)
	{
		SBomb* bomb = &(*it);
		DrawBomb(bomb, view_x, view_y);
	}
}

//draw the npcs
void DrawNPCs(list<SNPC> *npcs, int view_x, int view_y)
{
	for(list<SNPC>::iterator it = npcs->begin(); it != npcs->end(); it++)
	{
		SNPC* npc = &(*it);
		DrawNPC(npc, view_x, view_y);
	}
}

//draw the particles
void DrawParticles(list<SParticle> *particles, int view_x, int view_y)
{
	for(list<SParticle>::iterator it = particles->begin(); it != particles->end(); it++)
	{
		SParticle* part = &(*it);
		int x = part->x - view_x;
		int y = part->y - view_y;
			draw_trans_sprite(GetDrawPage(), part->bmp, x - part->bmp->w / 2, 
				y  - part->bmp->h / 2);
	}
}

//draw the gui
void DrawGUI(SPlanet* planet, int total_turn, int enemies_left)
{
	drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);
	//first we draw the semi-transparent panels
	rectfill(GetDrawPage(), 0, 0, SCREEN_W, 15, 30);
	rectfill(GetDrawPage(), 0, SCREEN_H - 15, SCREEN_W / 2 - 101, SCREEN_H, 30);
	rectfill(GetDrawPage(), SCREEN_W / 2 + 101, SCREEN_H - 15, SCREEN_W, SCREEN_H, 30);
	rectfill(GetDrawPage(), SCREEN_W / 2 - 100, SCREEN_H - 30, SCREEN_W / 2 + 100, SCREEN_H, 30);


	//then we draw the text stuff
	solid_mode();
	char buf[256];

	int n = total_turn % 100;
	int h = total_turn / 100;
	sprintf(buf, "Time: %03d.%02d", h, n);
	textout_ex(GetDrawPage(), font, buf, 2, SCREEN_H - 12, 110, -1);

	sprintf(buf, "Enemies: %d", enemies_left);
	textout_ex(GetDrawPage(), font, buf, 120, SCREEN_H - 12, 110, -1);

	sprintf(buf, "Score: %d", planet->score);
	textout_ex(GetDrawPage(), font, buf, SCREEN_W - 150, SCREEN_H - 12, 110, -1);

	sprintf(buf, "E: %d", planet->energy);
	textout_ex(GetDrawPage(), font, buf, 2, 3, 110, -1);

	sprintf(buf, "| %d", planet->energy_left);
	textout_ex(GetDrawPage(), font, buf, 92, 3, 110, -1);

	sprintf(buf, "G: %d", planet->num_generators);
	textout_ex(GetDrawPage(), font, buf, 272, 3, 110, -1);

	sprintf(buf, "S: %d", planet->num_shields);
	textout_ex(GetDrawPage(), font, buf, 322, 3, 110, -1);

	sprintf(buf, "Sl: %d", planet->shield);
	textout_ex(GetDrawPage(), font, buf, 182, 3, 110, -1);

	int val = 0;
	//select the action phrase
	switch(planet->state)
	{
	case 0:
		val = int(planet->state_val4);
		sprintf(buf, "Action: Opening Wormhole");
		break;
	case 1:
		val = int(planet->state_val3);
		sprintf(buf, "Action: Destroying Space");
		break;
	case 2:
		val = int(planet->state_val3);
		sprintf(buf, "Action: Destroying Subspace");
		break;
	case 3:
		val = int(planet->state_val4);
		sprintf(buf, "Action: Building Generator");
		break;
	case 4:
		val = int(planet->state_val4);
		sprintf(buf, "Action: Building Shield");
		break;
	case 5:
		val = int(planet->energy / 2);
		sprintf(buf, "Action: Building Probe");
		break;
	default:
		sprintf(buf, "Action: Collecting Energy");
		break;
	}

	textout_ex(GetDrawPage(), font, buf, 372, 3, 110, -1);

	if(planet->state != -1)
	{
		sprintf(buf, "Cost: %d", val);
	}
	else
	{
		sprintf(buf, "Cost: None");
	}
	textout_ex(GetDrawPage(), font, buf, 600, 3, 110, -1);
}

//draws the help
void DrawHelp(SPlanet* planet, int view_x, int view_y)
{	
	//yeah... I know that making an automated function to draw these things would be
	//more portable... if I ever get to improve my gui, I will certainly do that
	//until then, enjoy the mess
	int x = planet->x - view_x;
	int y = planet->y - view_y;
	
	//draw the lines
	vline(GetDrawPage(), 40, 15, 100, 110);
	vline(GetDrawPage(), 140, 15, 70, 110);
	vline(GetDrawPage(), 230, 15, 40, 110);
	vline(GetDrawPage(), 300, 15, 70, 110);
	vline(GetDrawPage(), 350, 15, 40, 110);
	vline(GetDrawPage(), 470, 15, 100, 110);
	vline(GetDrawPage(), 670, 15, 70, 110);

	vline(GetDrawPage(), SCREEN_W - 90, 15, 100, 110);
	vline(GetDrawPage(), SCREEN_W - 70, 15, 130, 110);
	vline(GetDrawPage(), SCREEN_W - 50, 15, 160, 110);
	vline(GetDrawPage(), SCREEN_W - 30, 15, 190, 110);

	vline(GetDrawPage(), 70, SCREEN_H - 85, SCREEN_H - 15, 110);
	vline(GetDrawPage(), 195, SCREEN_H - 55, SCREEN_H - 15, 110);
	vline(GetDrawPage(), SCREEN_W - 90, SCREEN_H - 55, SCREEN_H - 15, 110);

	//draw the line to the planet
	aa_line(GetDrawPage(), 190, 208, x, y, 110);

	
	//draw the line to the generator circle
	if(270 > x)
	{
		aa_line(GetDrawPage(), 270, 408, x + planet->num_generators * 4 + 10, y, 110);
	}
	else
	{
		aa_line(GetDrawPage(), 270, 408, x - planet->num_generators * 4 - 10, y, 110);
	}

	drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);

	//now draw the half transparent boxes
	rectfill(GetDrawPage(), 10, 100, 200, 115, 30);
	rectfill(GetDrawPage(), 50, 70, 200, 85, 30);
	rectfill(GetDrawPage(), 160, 40, 270, 55, 30);
	rectfill(GetDrawPage(), 240, 70, 405, 85, 30);
	rectfill(GetDrawPage(), 310, 40, 450, 55, 30);
	rectfill(GetDrawPage(), 360, 100, 560, 115, 30);
	rectfill(GetDrawPage(), 480, 70, 705, 85, 30);

	rectfill(GetDrawPage(), SCREEN_W - 210, 100, SCREEN_W - 80, 115, 30);
	rectfill(GetDrawPage(), SCREEN_W - 160, 130, SCREEN_W - 60, 145, 30);
	rectfill(GetDrawPage(), SCREEN_W - 140, 160, SCREEN_W - 40, 175, 30);
	rectfill(GetDrawPage(), SCREEN_W - 180, 190, SCREEN_W - 20, 205, 30);

	rectfill(GetDrawPage(), 10, SCREEN_H - 100, 160, SCREEN_H - 85, 30);
	rectfill(GetDrawPage(), 80, SCREEN_H - 70, 230, SCREEN_H - 55, 30);

	rectfill(GetDrawPage(), SCREEN_W - 150, SCREEN_H - 70, SCREEN_W - 65, SCREEN_H - 55, 30);

	rectfill(GetDrawPage(), 100, 200, 190, 215, 30);
	rectfill(GetDrawPage(), 100, 400, 270, 415, 30);

	solid_mode();

	int height = text_height(font) / 2;

	//and now draw the actual text
	textout_ex(GetDrawPage(), font, "Current Energy Reserves", 12, 108 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Harvestable Energy", 52, 78 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Shield Charge", 162, 48 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Number of Generators", 242, 78 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Number of Shields", 312, 48 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Action for the Next Turn", 362, 108 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Energy Cost to do the Action", 482, 78 - height, 110, -1);

	textout_ex(GetDrawPage(), font, "Build Generator", SCREEN_W - 208, 108 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Build Shield", SCREEN_W - 158, 138 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Toggle Grid", SCREEN_W - 138, 168 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Cycle Subspace View", SCREEN_W - 178, 198 - height, 110, -1);

	textout_ex(GetDrawPage(), font, "Game Time(seconds)", 12, SCREEN_H - 93 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Enemy Planets Left", 82, SCREEN_H - 63 - height, 110, -1);
	
	textout_ex(GetDrawPage(), font, "Your Score", SCREEN_W - 148, SCREEN_H - 63 - height, 110, -1);

	textout_ex(GetDrawPage(), font, "Your Planet", 102, 208 - height, 110, -1);
	textout_ex(GetDrawPage(), font, "Mass Absorption Zone", 102, 408 - height, 110, -1);
}

//generator button of the gui
int b_gn()
{
	PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	return 2;
}

//shield button of the gui
int b_sh()
{
	PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	return 4;
}

//grid button of the gui
int b_gr()
{
	PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	return 8;
}

//view button of the gui
int b_vw()
{
	PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	return 16;
}

//end turn button of the gui
int b_et()
{
	PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	return 32;
}

//menu button of the gui
int b_me()
{
	PlaySound((SAMPLE*)g_pData[_Button].dat, false);
	return 64;
}

//returns the nummber of players in a mission
int GetNumPlayers(const char* mis_name)
{
	set_config_file(mis_name);

	int n = 0;
	char buf[50];
	while(true)
	{
		sprintf(buf, "start%d", n);

//		float coords[2];
		string params(get_config_string("", buf, ""));
		if(params == "")
			break;
		n++;
	}
	return n;
}

//the score struct
//I have no idea why I used this instead of the planet struct directly
struct SScore
{
	SPlanet* planet;
};

//sorts the scores of planets
int score_compare(const void *a, const void *b)
{
	SPlanet* A = ((SScore*)a)->planet;
	SPlanet* B = ((SScore*)b)->planet;

	int score1 = A->score;
	if(A->dead)//dead ones go last, but we still want to preserve their ordering
		score1 -= 100000;

	int score2 = B->score;
	if(B->dead)
		score2 -= 100000;

	if(score1 > score2)
		return -1;
	else if(score1 < score2)
		return 1;
	else
		return 0;
}

//draws scores
//passing player = -1 resets the array
void DrawScores(vector<SPlanet> *planets, int player)
{
	static vector<SScore> scores;//static array of scores

	//reset the array
	if(player == -1 || scores.size() == 0)
	{
		scores.resize(planets->size());
		for(unsigned int ii = 0; ii < scores.size(); ii++)
		{
			scores[ii].planet = &((*planets)[ii]);
		}
		return;
	}

	SPlanet* player_planet = &((*planets)[player]);

	//it pains my heart to have to sort this array every frame, so I'd rather check
	//on its integrity instead, and when only when it needs sorting does it get sorted
	bool sorted = true;
	for(unsigned int ii = 1; ii < scores.size(); ii++)//check if sorted
	{
		if(score_compare(&scores[ii], &scores[ii - 1]) < 0)
		{
			sorted = false;
			break;
		}
	}
	if(!sorted)
	{
		qsort(&scores[0], scores.size(), sizeof(SScore), score_compare);
	}

	int total_height = scores.size() * 15;
	int height = text_height(font) / 2;

	drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);

	//draw the half transparent box
	rectfill(GetDrawPage(), SCREEN_W / 2 - 100, (SCREEN_H - total_height) / 2 - 20, 
		SCREEN_W / 2 + 100, (SCREEN_H + total_height) / 2 + 20, 30);

	solid_mode();

	int y = (SCREEN_H - total_height) / 2;
	char buf[256];

	//draw the scores
	for(unsigned int ii = 0; ii < scores.size(); ii++)
	{
		sprintf(buf, "%s", scores[ii].planet->name.c_str());

		int color = 110;

		if(player_planet == scores[ii].planet)//highlight the player
			color = 128;

		textout_ex(GetDrawPage(), font, buf, SCREEN_W / 2 - 80, y + height, color, -1);

		sprintf(buf, "%d", scores[ii].planet->score);

		color = 110;

		if(scores[ii].planet->dead)//darken the dead guys
			color = 50;

		textout_right_ex(GetDrawPage(), font, buf, SCREEN_W / 2 + 80, y + height, color, -1);

		y += 15;
	}
}

//draws the chat
void DrawChat(int num_entries, int start_entry)
{
	int height = text_height(font) / 2;
	int y = SCREEN_H - 50;
	int lower_num = g_sChatBuffer.size() - num_entries - start_entry;
	if(lower_num < 0)
		lower_num = 0;
	int color = 110;
	bool first = true;
	for(int ii = g_sChatBuffer.size() - start_entry - 1; ii >= lower_num; ii--)
	{
		if(first && start_entry != 0)//show that there is other stuff
			textout_ex(GetDrawPage(), font, "V V V", 10, y + height, color, -1);
		else
			textout_ex(GetDrawPage(), font, g_sChatBuffer[ii].c_str(), 10, y + height, color, -1);

		first = false;

		color -= 2;//some colour gradation
		y -= 15;
	}
}

//puts some text into the chat
void PutInChat(const char* txt)
{
	PlaySound((SAMPLE*)g_pData[_Chat].dat, false);
	g_sChatBuffer.push_back(string(txt));
}

//main function for the game
//runs the game according to the profile that is filled out by the menu functions
//
//"Yo P-Dawg, I Love you!"
//						-Alissa, 6/4/2007
//"I love you too, Alissa."
//						-SiegeLord, 6/5/2007
int RunGame(SMissionProfile* profile)
{
	int ret = 0;

	const char* player_name = g_sPlayerName.c_str();
	int num_bots = profile->num_bots;
	const char* misname = profile->mission_name.c_str();

	int player = g_nPlayerNumber;

	ShowPage();

	UpdateInput();

	g_sChatBuffer.clear();

	if(num_bots)
		InitAI();

	BITMAP* galaxy;
	BITMAP* subspace;

	int view_x = 400;
	int view_y = 400;

	int subspace_mode = 0;

	int grid_mode = 0;

	int game_mode = 0;//used for switching the phases: 0 - planning phase, 1 - action phase

	turn = 0;

	srand(time(0));

	//colour maps for various things
	COLOR_MAP sub_map;
	COLOR_MAP add_map;
	COLOR_MAP minus_map;
	COLOR_MAP invert_map;

	//the gui menu stuff
	SButton buttons[] = 
	{
		{1,		SCREEN_W - 100,		0,				20,		15,			0, 0, 0,	b_gn, "+G"		,1},
		{1,		SCREEN_W - 80,		0,				20,		15,			0, 0, 0,	b_sh, "+S"		,1},
		{1,		SCREEN_W - 60,		0,				20,		15,			0, 0, 0,	b_gr, "#"		,1},
		{1,		SCREEN_W - 40,		0,				20,		15,			0, 0, 0,	b_vw, "V"		,1},
		{1,		SCREEN_W / 2 - 100,	SCREEN_H - 30,	200,	30,			0, 0, 0,	b_et, "End Turn",1},
		{1,		SCREEN_W - 80,		SCREEN_H - 15,	80,		15,			0, 0, 0,	b_me, "Quit"	,1},
		{0,		0, 0, 0, 0,			0,				0,		0,			0, 0, 0}
	};

	//initialize the colour maps
	CreateSubSpaceMap(sub_map, .5, 10, 100);
	CreateAdditionMap(add_map, 128);
	CreateSubtractionMap(minus_map);
	CreateInverseMap(invert_map);

	//various data structures
	vector<SPlanet> aPlanets;
	list<SWave> lWaves;
	list<SParticle> lParticles;
	list<SBomb> lBombs;
	list<SNPC> lNPCs;
	vector<SPoint> spots;

	aPlanets.clear();
	lWaves.clear();
	lParticles.clear();
	lBombs.clear();
	lNPCs.clear();
	spots.clear();

	//load the mission
	LoadMission(misname, &spots, &galaxy, &subspace);

	int num_players = spots.size();

	//now we initialize all of the players
	int offset = rand() % num_players;//pick a random spot to start from

	for(int ii = 0; ii < num_bots + 1; ii++)//and then iterate across the spots
	{
		SPlanet planet;
		planet.x = spots[(ii + offset) % num_players].x;
		planet.y = spots[(ii + offset) % num_players].y;
		if(ii == player)
		{
			planet.name.assign(player_name);
			planet.race = g_nPlayerRace;
			planet.ai = false;
		}
		else
		{
			planet.name.assign(GetAIName());
			planet.race = rand()%3;
			planet.ai = true;
		}
		aPlanets.push_back(planet);
	}

	bool game_over = false;//used in the main loop to denote whether the game is over
							//i.e. we display the score screen or something

	view_x = aPlanets[player].x - SCREEN_W / 2;//we start the view centered on the player
	view_y = aPlanets[player].y - SCREEN_H / 2;

	int total_turn = 0;//used for time purposes

	bool ended = false;//a little hack to fix some sound issues, this is tied to the game_over variable
						//in fascinating ways


	bool ai_done = false;//a fascinating hack for the ai players... I may need to homogenize this, so
							//ai players are treated on an equal footing with the humans
	
	turn = 0;//we reset the turn(fixes some bugs)

	DrawScores(&aPlanets, -1);//reset the scores

	bool chat_mode = false;	//chat stuff
	int chat_offset = 0;

	int chat_sizes[] = {10, 5, 2, 0}; //size(in lines) of the chat box

	//a struct used for the action function
	ActionParam a_param;
	a_param.planets = &aPlanets;
	a_param.waves = &lWaves;
	a_param.particles = &lParticles;
	a_param.galaxy = galaxy;
	a_param.subspace = subspace;
	a_param.total_turn = &total_turn;
	a_param.flash[0] = CreateAstroBrush(40, 127, 0, 2);
	a_param.flash[1] = CreateAstroBrush(50, 127, 0, 2);
	a_param.flash[2] = CreateAstroBrush(60, 127, 0, 2);
	a_param.flash[3] = CreateAstroBrush(70, 127, 0, 2);
	a_param.flash[4] = CreateAstroBrush(80, 127, 0, 2);
	a_param.bombs = &lBombs;
	a_param.npcs = &lNPCs;
	a_param.player = player;

	while(!g_bQuit)
	{
		srand(g_nTime);
		int snap = g_nTime;//used for fps cap

		if(!chat_mode)//when we aren't in the chat, we can use the keyboard to do stuff
		{
			if(IsKeyJustPressed(KEY_EQUALS))//moves the chat cursor around
			{
				chat_offset++;
			}
			if(IsKeyJustPressed(KEY_MINUS))
			{
				chat_offset--;
			}
			if(chat_offset > int(g_sChatBuffer.size()) - chat_sizes[g_nChatSize])
				chat_offset = int(g_sChatBuffer.size()) - chat_sizes[g_nChatSize];
			if(chat_offset < 0)
				chat_offset = 0;

			if(IsKeyDown(KEY_RIGHT))//view movement
				view_x += 5;

			if(IsKeyDown(KEY_LEFT))
				view_x -= 5;

			if(IsKeyDown(KEY_UP))
				view_y -= 5;

			if(IsKeyDown(KEY_DOWN))
				view_y += 5;

			if(IsKeyJustPressed(KEY_F2))//switches the subspace mode
			{
				PlaySound((SAMPLE*)g_pData[_Button].dat, false);
				subspace_mode++;
				subspace_mode = subspace_mode % 3;
			}

			if(IsKeyJustPressed(KEY_F9))//toggles the chat size along its values
			{
				g_nChatSize++;
				g_nChatSize %= 4;
			}

			if(IsKeyJustPressed(KEY_F3))//toggles the grid mode
			{
				PlaySound((SAMPLE*)g_pData[_Button].dat, false);
				grid_mode++;
				grid_mode = grid_mode % 2;
			}

			if(IsKeyJustPressed(KEY_TAB))//next turn
			{
				if(game_mode == 0)
				{
					PlaySound((SAMPLE*)g_pData[_Button].dat, false);
					aPlanets[0].done = true;
				}
			}

			if(IsKeyJustPressed(KEY_ENTER))//evokes the chat
			{
				UpdateInput();//prevent the dialog from closing itself
				clear_keybuf();
				SetEditBoxString("");
				chat_mode = true;
			}
		}

		//view clamping things
		if(view_x < 0)
			view_x = 0;
		if(view_y < 0)
			view_y = 0;

		if(view_x > _GALAXY_RADIUS * 2 - SCREEN_W)
			view_x = _GALAXY_RADIUS * 2 - SCREEN_W;

		if(view_y > _GALAXY_RADIUS * 2 - SCREEN_H)
			view_y = _GALAXY_RADIUS * 2 - SCREEN_H;

		if(game_mode == 0)//we are in the planning phase
		{
			if(!game_over && !chat_mode)//game is not done and we are not chatting
				HandlePlanetInput(subspace, &aPlanets, player, view_x, view_y);
		}
		else//we are in the action phase
		{
			if(!Action(a_param))//Action will return false once it is done
			{
				PlaySound((SAMPLE*)g_pData[_Start_turn].dat, false);
				game_mode = 0;//back to planning
				ai_done = false;//ai is no longer done
				if(!game_over)//check to see who won
				{
					game_over = true;
					for(unsigned int ii = 0; ii < aPlanets.size(); ii++)
					{
						if(g_nGameType == 0)//in this game type people can die, so check for this
						{
							if(ii != 0)
								game_over &= aPlanets[ii].dead;
						}
						else
							game_over = false;

						if(aPlanets[ii].score >= g_nMaxPoints)//someone reached the max points
						{
							game_over = true;
							break;
						}
					}
				}
			}
		}

		if(aPlanets[player].dead)//if player is dead, game is over
		{
			game_over = true;
		}

		if(game_over && !ended)//ended is needed here, to make sure we only play this sound once
		{
			PlaySound((SAMPLE*)g_pData[_End].dat, false);
			ended = true;
		}

		//turn advancement section
		bool done = true;
		int enemies_left = -1;
		//first we check if everyone is done
		//we also count the enemies
		for(unsigned int ii = 0; ii < aPlanets.size(); ii++)
		{
			if(!aPlanets[ii].dead)
			{
				done &= aPlanets[ii].done;
				enemies_left++;
			}
		}

		if(enemies_left == -1)
			//if there is noone around, we cannot be done
			//this is to prevent turn advancement if everyone is dead

			done = false;

		if(done)//everyone is done, start the action phase
		{
			for(unsigned int ii = 0; ii < aPlanets.size(); ii++)
			{
				aPlanets[ii].done = false;
			}
			g_nSeed = rand() % _NUM_SEEDS;
			g_nStartPoint = rand();
			game_mode = 1;
		}

		//make the ai do stuff
		if(game_mode == 0 && !ai_done)
		{
			for(unsigned int ii = 0; ii < aPlanets.size(); ii++)
			{
				if(aPlanets[ii].ai)
					HandleAI(&aPlanets[ii], &lNPCs, &aPlanets, subspace);
			}
			ai_done = true;
		}

		//draw stuff
		//first we draw the galaxy
		switch(subspace_mode)
		{
		case 0:
			blit(galaxy, GetDrawPage(), view_x, view_y, 0, 0, SCREEN_W, SCREEN_H);//just the galaxy
			break;
		case 1:
			blit(subspace, GetDrawPage(), view_x, view_y, 0, 0, SCREEN_W, SCREEN_H);//just the subspace
			break;
		case 2:
			color_map = &sub_map;
			blit(galaxy, GetDrawPage(), view_x, view_y, 0, 0, SCREEN_W, SCREEN_H);//galaxy then
			draw_trans_sprite(GetDrawPage(), subspace, -view_x, -view_y);//overlay the subspace
			break;
		}
		
		//draw grid
		if(grid_mode == 0)
			DrawGrid(view_x, view_y);

		DrawPlanets(&aPlanets, player, view_x, view_y);
		
		color_map = &invert_map;
		DrawPlanetNames(&aPlanets, player, view_x, view_y);
		DrawWaves(&lWaves, view_x, view_y);
		DrawBombs(&lBombs, view_x, view_y);
		DrawNPCs(&lNPCs, view_x, view_y);

		color_map = &add_map;
		DrawParticles(&lParticles, view_x, view_y);

		color_map = &minus_map;

		if(!IsKeyDown(KEY_F1))
		{
			DrawChat(chat_sizes[g_nChatSize], chat_offset);
		}

		if(!game_over)//the following handles the gui buttons
		{
			DrawGUI(&aPlanets[player], total_turn, enemies_left);

			int ret = DoButtons(buttons);

			if(game_mode == 0)
			{
				if(ret & 2)
				{
					SetGeneratorState(&aPlanets[player]);
				}
				if(ret & 4)
				{
					SetShieldState(&aPlanets[player]);
				}
				if(ret & 8)
				{
					grid_mode++;
					grid_mode = grid_mode % 2;
				}
				if(ret & 16)
				{
					subspace_mode++;
					subspace_mode = subspace_mode % 3;
				}
				if(ret & 32)
				{
					if(game_mode == 0)
						aPlanets[player].done = true;
				}
				if(ret & 64)
				{
					break;
				}
			}
		}

		if(!chat_mode)
			DrawMouse();

		if(chat_mode)
		{
			color_map = &minus_map;
			int ret = HandleEditBox(false);
			if(ret != 0)
			{
				chat_mode = false;
				if(ret == 1)
				{
					string entry(GetEditBoxString());
					if(entry != "")
					{
						entry = aPlanets[player].name + string(": ") + entry;
						PutInChat(entry.c_str());
					}
				}
			}
		}
		
		//game over stuff
		if(game_over)
		{
			if(IsKeyJustPressed(KEY_ESC))//quit
			{
				PlaySound((SAMPLE*)g_pData[_Button].dat, false);
				break;
			}
			if(IsKeyJustPressed(KEY_R))//quit and restart
			{
				ret = 1;
				break;
			}

			//draw the game over message
			int x = SCREEN_W / 2;
			int y = SCREEN_H / 2;
			int height = text_height(font) / 2;
					
			color_map = &minus_map;
			drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);

			bool lost = false;//determine whether we have lost or not
			if(g_nGameType == 0)
			{
				lost = aPlanets[player].dead;//we lose if we die
			}
			else
			{
				lost = aPlanets[player].score < g_nMaxPoints;//we lose if we fail to achieve the goal
			}

			//draw the actual message
			if(lost)
			{
				rectfill(GetDrawPage(), x - 170, y - 15, x + 170, y + 40, 30);
				solid_mode();

				textout_centre_ex(GetDrawPage(), font, "You Have Lost...", x, y - height, 110, -1);
				textout_centre_ex(GetDrawPage(), font, "Press Esc to Quit, or R to Restart...", x, y + 5 * height, 110, -1);
			}
			else
			{
				rectfill(GetDrawPage(), x - 170, y - 15, x + 170, y + 40, 30);
				solid_mode();

				textout_centre_ex(GetDrawPage(), font, "You Have Won!", x, y - height, 110, -1);
				textout_centre_ex(GetDrawPage(), font, "Press Esc to Quit, or R to Restart...", x, y + 5 * height, 110, -1);
			}
		}

		//draw help
		if(IsKeyDown(KEY_F1))
		{
			if(!game_over)
			{
				color_map = &minus_map;
				DrawHelp(&aPlanets[player], view_x, view_y);
			}
		}
		
		//draw scores
		if(IsKeyDown(KEY_F10))
		{
			color_map = &minus_map;
			DrawScores(&aPlanets, player);
		}

		//handle the particles
		for(list<SParticle>::iterator it = lParticles.begin(); it != lParticles.end(); it++)
		{
			SParticle* part = &(*it);
			part->life--;
		
			if(part->life < 0)
				it = lParticles.erase(it);
		}

		ShowPage();//flip the page

		UpdatePalettes();
		UpdateInput();
		UpdateSound(view_x, view_y);
		
		while(g_nTime - snap == 0)//fps cap mechanism
			rest(1);
	}

	//clean up
	destroy_bitmap(a_param.flash[0]);
	destroy_bitmap(a_param.flash[1]);
	destroy_bitmap(a_param.flash[2]);
	destroy_bitmap(a_param.flash[3]);
	destroy_bitmap(a_param.flash[4]);

	destroy_bitmap(galaxy);
	destroy_bitmap(subspace);

	return ret;
}
