//This file is part of Future's End
//Copyright 2006-2008 SiegeLord
//See license.txt for distribution information
//
//wave.cpp
//handles the wave stuff

#include "wave.h"
#include "gfx.h"
#include "sound.h"

extern int g_nGameType;
extern float g_fDamageMultiplier;

//the param struct that we feed into the proc function
struct SWaveProcParam
{
	SWave* wav;//the wave in question
	vector<float>* buffer;//a new buffer
	BITMAP* subspace;//pointer to the subspace
	float* total_energy;//total energy of the wave
	float factor;//a factor
	int circumf;//the circumpherence of the circle

	vector<SPlanet>* planets;//planets

	int idx1;
	int idx2;

	bool is_subspace;//figures out whether this is the subspace
};

//similar to the get correct index function from the gfx
//converts the discontinous angle space angle to the continous angle space
float GetCorrectAngle(float a, float a1, float a2)
{
	if(a1 > a2)//thus we pass through the discontinuity at 0 degrees
	{
		if(a >= a1 && a <= 2 * AL_PI)
			return a - a1;
		if(a < a1 && a > a2)
			return -1;
		return a + (2 * AL_PI - a1);
	}
	else
	{
		if(a < a1)
			return -1;
		if(a > a2)
			return -1;
		return a - a1;
	}
}

//similar to the above, figures out the maximum angle in the continous angle space
float GetSpan(float a1, float a2)
{
	if(a1 > a2)//thus we pass through the discontinuity at 0 degrees
	{
		return 2 * AL_PI - a1 + a2;
	}
	else
	{
		return a2 - a1;
	}
}

//the proc that handles each pixel of the wave
void WaveProc(BITMAP* bmp, int x, int y, int idx, void* ptr, bool on_arc)
{
	SWaveProcParam* param = (SWaveProcParam*)ptr;

	if(on_arc)//only the stuff on arc gets attenuated
	{
		int index = get_correct_index(idx, param->idx2, param->idx1, param->circumf);//get the index

		//clamp the index
		if(index < 0)
			index = 0;
		if(index >= param->buffer->size())
			index = param->buffer->size() - 1;

		//attenuate this pixel by subspace
		int attenuate = .2 * (100 - _getpixel(param->subspace, x, y));
		if(attenuate < 0)
			attenuate = 0;
		
		float val = param->wav->energies[index] - attenuate;//attenuate the energy

		if(val < 0)
			val = 0.0;

		*param->total_energy += val;//keep track of the total energy
		
		//and now we actually destroy the bitmap in question
		int dens = _getpixel(bmp, x, y);

		float factor = 10;

		dens -= val / factor;

		if(dens < 0)
			dens = 0;

		_putpixel(bmp, x, y, dens);

		if(!param->is_subspace)//if this isn't subspace, we attack planets
		{
			for(unsigned int ii = 0; ii < param->planets->size(); ii++)
			{
				SPlanet* planet = &((*param->planets)[ii]);
				if(planet->x == x && planet->y == y)
				{
					if(!planet->dead)
					{
						planet->last_hit_by = param->wav->source;
						planet->shield -= g_fDamageMultiplier * val / 5;
						PlaySound((SAMPLE*)g_pData[_Shield_hit].dat, x, y, false);
					}
				}
			}
		}

		param->wav->energies[index] = val;
	}
	else//not on arc, but we still destroy space
	{
		int dens = _getpixel(bmp, x, y);
		int index = get_correct_index(idx, param->idx2, param->idx1, param->circumf);

		if(index < 0)
			index = 0;
		if(index >= param->buffer->size())
			index = param->buffer->size() - 1;

		float factor = 10;

		dens -= param->wav->energies[index] / factor;
		if(dens < 0)
			dens = 0;

		_putpixel(bmp, x, y, dens);

		if(!param->is_subspace)
		{
			for(unsigned int ii = 0; ii < param->planets->size(); ii++)
			{
				SPlanet* planet = &((*param->planets)[ii]);
				if(planet->x == x && planet->y == y)
				{
					if(!planet->dead)
					{
						planet->last_hit_by = param->wav->source;
						planet->shield -= g_fDamageMultiplier * param->wav->energies[index] / 5;
						PlaySound((SAMPLE*)g_pData[_Shield_hit].dat, x, y, false);
					}
				}
			}
		}
	}
}


//simple cubic interpolation
float Interpolate(float a, float b, float x)
{
	float f = x * x * (3 - 2 * x);//3x^2-2x^3

	return a + (b - a) * f;
}

//collides a circle with a wave
bool CollideCircleWithWave(int x, int y, int r, SWave* wave)
{
	float dx = float(x - wave->x);
	float dy = float(y - wave->y);

	float dist = dx * dx + dy * dy;

	//first we make sure that the circle collides with the circle that contains the arc
	if(dist > float(r + wave->radius) * float(r + wave->radius))
		return false;
	if(dist < float(-r + wave->radius) * float(-r + wave->radius))
		return false;

	//now we check whether the circle collides with the arc within the arc's angle
	float a = atan2(dy, dx);

	float ret = GetCorrectAngle(a, wave->angle1, wave->angle2);

	if(ret < 0)	//if ret is positive, then it is clearly colliding, but if it is negative, more checks
				//are needed
	{
		//what we do, is we simply get the positions of the end points of the wave
		//and then do simple distance tests to them with the circle
		float X = float(wave->radius) * cosf(wave->angle1) + wave->x;
		float Y = float(wave->radius) * sinf(wave->angle1) + wave->y;

		dx = float(x - X);
		dy = float(y - Y);

		if(dx * dx + dy * dy < float(r) * float(r))
		{
			return true;
		}
		else
		{
			X = float(wave->radius) * cosf(wave->angle2) + wave->x;
			Y = float(wave->radius) * sinf(wave->angle2) + wave->y;

			dx = float(x - X);
			dy = float(y - Y);
			if(dx * dx + dy * dy < float(r) * float(r))
			{
				return true;
			}
		}
		return false;
	}
	return true;
}

//the move wave function
//the general algorithm is as follows:
//
//first we interpolate a higher radius wave
//while keeping track of the total energy
//then we attenuate the new high wave by it's
//underlaying subspace while also scaling down
//so that the total un-attenuated energy is equal
//to the previous energy

//"I am a pirate and I'm here for your booty!"
//										-Alissa
void MoveWave(vector<SPlanet>* planets, list<SNPC>* npcs, SWave* wave, BITMAP* galaxy, BITMAP* subspace)
{
	static vector<float> buffer;
	
	int new_radius = wave->radius + 1;

	float new_total_energy = 0.0f;

	//interpolate the new wave

	int idx1, idx2;

	get_arc_indicies(new_radius, wave->angle1, wave->angle2, &idx1, &idx2);

	int buffer_size = get_correct_index(idx1, idx2, idx1, get_circle_circumference(new_radius)) + 1;
	buffer.resize(buffer_size);//resize the energy buffer

	ASSERT(wave->energies.size() != 0);

	float factor = float(new_radius - 1) / float(new_radius);//attenuation due to the bigger size
	factor *= factor;

	for(int ii = 0; ii < buffer_size; ii++)
	{
		float index = float(ii) / float(buffer_size) * float(wave->energies.size());
		int index1 = int(index);
		int index2 = index1 + 1;
		
		if(index2 >= wave->energies.size())//clamp the index if needed
		{
			index1 = wave->energies.size() - 1;
			index2 = index1;
		}

		float frac = index - index1;
		float val = Interpolate(wave->energies[index1], wave->energies[index2], frac);

		if(val < 0)
			val = 0;

		new_total_energy += val;
		
		buffer[ii] = val * factor;
	}

	//a hack, destroys some annoying waves
	if(new_total_energy / GetSpan(wave->angle1, wave->angle2) / float(new_radius) < 1)
		new_total_energy = 0;
	wave->energies.resize(buffer_size);
	wave->radius = new_radius;

	for(int ii = 0; ii < buffer_size; ii++)
	{
		wave->energies[ii] = buffer[ii];
	}

	//attenuate and propagate

	SWaveProcParam param;
	param.buffer = &buffer;
	param.subspace = subspace;
	param.total_energy = &new_total_energy;
	param.wav = wave;
	param.factor = factor;
	param.circumf = get_circle_circumference(new_radius);
	param.planets = planets;
	param.is_subspace = wave->subspace;

	param.idx1 = idx1;
	param.idx2 = idx2;

	BITMAP* dest;
	if(wave->subspace)
		dest = subspace;
	else
		dest = galaxy;

	cont_arc(dest, wave->x, wave->y, wave->radius, wave->angle1, wave->angle2, &param, WaveProc);

	wave->energy = new_total_energy * factor - 1;//-1 is for some persistent waves

	//now we collide it with the bounty circles, if needed
	if(g_nGameType == 1 && !wave->subspace)
	{
		for(list<SNPC>::iterator it = npcs->begin(); it != npcs->end(); it++)
		{
			SNPC* npc = &(*it);
			if(npc->type != 1)
				continue;

			if(CollideCircleWithWave(npc->x, npc->y, npc->val2 / 16, wave))
			{
				(*planets)[wave->source].score += npc->val1 / 100;
				npc->val1 = -999;
			}
		}
	}
}

//this function is used for drawing of the wave
void proc_fn(BITMAP* bmp, int x, int y, int idx, void* ptr, bool on_arc)
{
	SWaveProcParam* param = (SWaveProcParam*)ptr;

	int index = get_correct_index(idx, param->idx2, param->idx1, param->circumf);

	index = float(index) / (float)get_correct_index(param->idx2, param->idx2, param->idx1, param->circumf)
		* (float)param->wav->energies.size();

	if(index < 0)
		index = 0;
	if(index >= param->wav->energies.size())
		index = param->wav->energies.size() - 1;

	int val = param->wav->energies[index] * param->factor;

	if(val < 0)
		val = 0;
	if(val > 100)
		val = 100;

	val += _getpixel(bmp, x, y);

	if(val < 0)
		val = 0;
	if(val > 100)
		val = 100;

	_putpixel(bmp, x, y, val);
}

//draws the wave
void DrawWave(SWave* wave, int view_x, int view_y)
{	
//	float angle_delta = 1 / float(wave->radius);
//	float angle = wave->angle1;

	SWaveProcParam param;
	param.wav = wave;

	get_arc_indicies(wave->radius, wave->angle1, wave->angle2, &param.idx1, &param.idx2);

	param.factor = 1.0f;

	//draw the back arc, which shows the wave power
	for(int ii = 0; ii < 1; ii++)//can be used to make it thicker
	{
		param.circumf = get_circle_circumference(wave->radius - ii);
		param.factor -= .25f;
		cont_arc(GetDrawPage(), wave->x - view_x, wave->y - view_y, wave->radius - ii, wave->angle1, wave->angle2,
			&param, proc_fn);
	}

	//draw the wavefront
	aa_arc(GetDrawPage(), wave->x - view_x, wave->y - view_y, wave->radius + 1, wave->angle1, wave->angle2,
		120);
}
