//This file is part of Future's End
//Copyright 2006-2008 SiegeLord
//See license.txt for distribution information
//
//galaxy.cpp
//Handles the galaxy creation routines

#include "types.h"
#include "gfx.h"

extern int g_nGameType;

//seed primes, used for the pseudo rand
int primes[][3] = 
{
	{15731, 789221, 1376312589},
	{15271, 789533, 1376312489},
	{15277, 789557, 1376312491},
	{15287, 789571, 1376312501},
	{15289, 789577, 1376312507},
	{15299, 789587, 1376312513},
	{15307, 789589, 1376312527},
	{15313, 789611, 1376312543},
	{15319, 789623, 1376312551},
	{15329, 789631, 1376312579},
	{15331, 789653, 1376312627},
	{15349, 789671, 1376312629},
	{15359, 789673, 1376312657},
	{15361, 789683, 1376312687},
	{15373, 789689, 1376312689},
	{15377, 789709, 1376312753},
	{15383, 789713, 1376312783},
	{15391, 789721, 1376312789},
	{15401, 789731, 1376312813},
	{15413, 789739, 1376312857},
	{15427, 789749, 1376312879}
};

int g_nSeed = 0; //current seed

int g_nStartPoint = 0; //used for the predictable rand calls in the game itself(useful for multiplier)

//size of the rand table(sized to be large enough for the user not to notice repetitions)
#define _RAND_TABLE_SIZE (0x000fffff) 

//the rand table itself, used for the perturb functions
float g_nRandTable[_RAND_TABLE_SIZE][2];

//the magical psedo random generator, uses the seed primes
//the range is -1 to 1
float prand2(int x, int y)
{
	int n = x + y * 57;//this line means that in theory this function can be used as a one parameter function
	n = (n<<13) ^ n;
	return ( 1.0 - ( (n * (n * n * primes[g_nSeed][0] + primes[g_nSeed][1]) + primes[g_nSeed][2]) & 0x7fffffff) / 1073741824.0);    
}

//the predictable rand function used in game
//range 0 to 1
float rande()
{
	return (prand2(g_nStartPoint++, 0) + 1.0f) / 2.0f;
}

//this initializes the rand table
void psrand(int seed)
{
	float *temp = new float[_RAND_TABLE_SIZE * 2];//this is created for the smoothing step later on

	//this just fills the rand table with the prand's
	for(int ii = 0; ii < _RAND_TABLE_SIZE; ii++)
	{
		//the values are interleaved
		g_nSeed = seed % _NUM_SEEDS;
		temp[2 * ii] = prand2(ii, 0);
		
		g_nSeed = (seed + 1) % _NUM_SEEDS;
		temp[2 * ii + 1] = prand2(ii, 0);
	}

	//the following smoothes the rand values using a simple convolution matrix
	for(int ii = 0; ii < _RAND_TABLE_SIZE; ii++)
	{
		//as I look at this again, I note that I used the original prand's x-y relationship
		//here... I am unsure how necessary this is, but since it works, it stays
		//this explains the weird 57's that pop in in places
		//57 means a change of 1 unit in the y direction

		float sub_total1 = 0.0f;
		float sub_total2 = 0.0f;
		float total1 = 0.0f;
		float total2 = 0.0f;

		//sides
		int index = (unsigned int)(ii - 1) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		index = (unsigned int)(ii - 57) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		index = (unsigned int)(ii + 57) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		index = (unsigned int)(ii + 1) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];
		
		total1 += sub_total1 * 0.125f;
		total2 += sub_total2 * 0.125f;

		sub_total1 = 0;
		sub_total2 = 0;

		//corners

		index = (unsigned int)(ii - 1 - 57) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		index = (unsigned int)(ii - 1 + 57) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		index = (unsigned int)(ii + 1 + 57) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		index = (unsigned int)(ii + 1 - 57) % _RAND_TABLE_SIZE;
		sub_total1 += temp[index * 2];
		sub_total2 += temp[index * 2 + 1];

		total1 += sub_total1 * 0.0625f;
		total2 += sub_total2 * 0.0625f;

		total1 += temp[index * 2] * 0.25f;
		total2 += temp[index * 2 + 1] * 0.25f;

		g_nRandTable[index][0] = total1;
		g_nRandTable[index][1] = total2;
	}
	delete[] temp;
}

//generates some smooth noise at x,y
float InterpolatedNoise(float x, float y)
{
	float v1;
	float v2;
	float v3;
	float v4;

	int integer_X    = int(x);
	float fractional_X = x - integer_X;

	int integer_Y    = int(y);
	float fractional_Y = y - integer_Y;

	int n = integer_X + integer_Y * 57;

	//as before, 57 means a change of 1 in the y direction
	v1 = g_nRandTable[(n + 0 ) & _RAND_TABLE_SIZE][g_nSeed];
	v2 = g_nRandTable[(n + 1 ) & _RAND_TABLE_SIZE][g_nSeed];
	v3 = g_nRandTable[(n + 57) & _RAND_TABLE_SIZE][g_nSeed];
	v4 = g_nRandTable[(n + 58) & _RAND_TABLE_SIZE][g_nSeed];

	//simple linear interpolation
	float i1 = v1 + (v2 - v1) * fractional_X;
	float i2 = v3 + (v4 - v3) * fractional_X;

	return i1 + (i2 - i1) * fractional_Y;
}

//this is not perlin noise actually...
//rather this is more commonly called value noise
float PerlinNoise_2D(float x, float y, float persistence = 1.0f, int Number_Of_Octaves = 3)
{
	float total = 0.0f;
	float p = persistence;
	int n = Number_Of_Octaves - 1;

	float frequency = 1;
	float amplitude = 1;

	for(int ii = 0; ii < n; ii++)
	{
		frequency *= 2;
		amplitude *= p;

		total += InterpolatedNoise(x * frequency, y * frequency) * amplitude;
	}
	return total;
}

//this perturbs an input bitmap using the perlin noise function
BITMAP* PerturbGalaxy(BITMAP* galaxy, int seed = 0, float factor = 40.0f, int tile_size = 128, 
					  float persistence = .9, int octaves = 10)
{
	BITMAP* ret = create_bitmap_ex(8, galaxy->w, galaxy->h);	//return bitmap

	psrand(seed);
	
	int eigth = ret->h / 20;	//we don't want to update the loading screen too often, so we do it
								//once every 1/8 of the task is complete

	acquire_bitmap(galaxy);
	acquire_bitmap(ret);

	char buf[100];

	//used for the loading screen
	if(g_nGameType == 0)
	{
		sprintf(buf, "DeathMatch: Destroy all of your opponents to win!");
	}
	else
	{
		sprintf(buf, "Bounty: Destroy as many of the bounty circles as you can. Planet that reaches the score limit wins.");
	}

	for(int y = 0; y < ret->h; y++)
	{
		if(y % eigth == 0)
			DrawProgress(float(y) / float(galaxy->h), "Applying Turbulence...", buf);
		for(int x = 0; x < ret->w; x++)
		{
			float X = float(x) / tile_size;
			float Y = float(y) / tile_size;

			g_nSeed = 0;//a misnomer, this switches between the two entries in the rand table
			int dispX = factor * PerlinNoise_2D(X, Y, persistence, octaves) + x;
			g_nSeed = 1;
			int dispY = factor * PerlinNoise_2D(X, Y, persistence, octaves) + y;

			if(dispX < 0)
				dispX = 0;
			if(dispY < 0)
				dispY = 0;

			if(dispX >= ret->w)
				dispX = ret->w - 1;
			if(dispY >= ret->h)
				dispY = ret->h - 1;

			ret->line[y][x] = galaxy->line[dispY][dispX];
		}
	}

	release_bitmap(ret);
	release_bitmap(galaxy);

	DrawProgress(1.1, "Done!", buf);
	rest(50);
	destroy_bitmap(galaxy);
	return ret;
}

BITMAP* CreateAstroBrush(int radius, int center_col, int outer_col, float power)
{
	BITMAP* ret = create_bitmap_ex(8, radius * 2, radius * 2);
	clear_bitmap(ret);
	for(int ii = radius - 1; ii >= 0; ii--)
	{
		float factor = 1 - float(ii) / float(radius);
		factor = powf((1 - powf(factor, power)), 1.0f / power);
		int color = center_col * (1 - factor) + outer_col * factor;
		circlefill(ret, radius, radius, ii, color);
	}
	return ret;
}

//this parses a string of numbers into a provided array
void ParseParams(string* str, float* params, int max_num)
{
	int a = 0, b;
	for(int ii = 0; ii < max_num; ii++)
	{
		b = str->find_first_of(" ", a);
		string token = str->substr(a, b - a);
		params[ii] = atof(token.c_str());
		a = b + 1;
	}
}

//converts the gray scale pcx images (0 - 255) into our scale (0 - 100)
void ConvertGrayToIdx(BITMAP* in)
{
	for(int y = 0; y < in->h; y++)
	{
		for(int x = 0; x < in->w; x++)
		{
			_putpixel(in, x, y, int(float(_getpixel(in, x, y)) / 256.0f * 100.0f));
		}
	}
}

//creates a galaxy by loading a gray scale pcx image
//and then scaling it up to 2048 using linear interpolation
BITMAP* CreateGalaxy(const char* file_name)
{
	int diam = _GALAXY_RADIUS * 2;
	int square_size = diam / (_GALAXY_THUMB_SIZE - 1);

	BITMAP* ret = create_bitmap_ex(8, diam, diam);

	BITMAP* source = load_bitmap(file_name, 0);
	ConvertGrayToIdx(source);

	char buf[100];

	//used for the loading screen
	if(g_nGameType == 0)
	{
		sprintf(buf, "DeathMatch: Destroy all of your opponents to win!");
	}
	else
	{
		sprintf(buf, "Bounty: Destroy as many of the bounty circles as you can. Planet that reaches the score limit wins.");
	}

	//we skip the last pixel for ease of implementation and speed
	for(int Y = 0; Y < _GALAXY_THUMB_SIZE - 1; Y++)
	{
		for(int X = 0; X < _GALAXY_THUMB_SIZE - 1; X++)
		{
			float v1,v2,v3,v4;
			v1 = _getpixel(source, X, Y);
			v2 = _getpixel(source, X + 1, Y);
			v3 = _getpixel(source, X, Y + 1);
			v4 = _getpixel(source, X + 1, Y + 1);

			for(int y = 0; y < square_size; y++)
			{
				float f1 = float(y) / float(square_size);
				for(int x = 0; x < square_size; x++)
				{
					float f2 = float(x) / float(square_size);
					float val1 = v1 + (v2 - v1) * f2;
					float val2 = v3 + (v4 - v3) * f2;
					float val = val1 + (val2 - val1) * f1;
					_putpixel(ret, x + X * square_size, y + Y * square_size, int(val));
				}
			}
		}
		if(Y % (_GALAXY_THUMB_SIZE / 8) == 0)
			DrawProgress(float(Y) / float(_GALAXY_THUMB_SIZE), "Generating Shape...", buf);
	}

	DrawProgress(1.1, "Done!", buf);

	return ret;
}

//loads a mission and creates the galaxy
void LoadMission(const char* name, vector<SPoint>* spots, BITMAP** galaxy, BITMAP** subspace)
{
	set_config_file(name);

	int n = 0;
	char buf[50];

	//first we load the starting points
	while(true)
	{
		sprintf(buf, "start%d", n);

		float coords[2];
		string params(get_config_string("", buf, ""));
		if(params == "")
			break;

		ParseParams(&params, coords, 2);

		SPoint point;
		point.x = coords[0];
		point.y = coords[1];

		spots->push_back(point);

		n++;
	}

	//now we load the space and subspace files
	string name1("missions/");
	name1.append(get_config_string("", "space_file", ""));
	string name2("missions/");
	name2.append(get_config_string("", "subspace_file", ""));

	float param[9];

	*galaxy = CreateGalaxy(name1.c_str());
	
	//and now we load the perturb parameters, and perturb the generated galaxy/subspace maps
	string str(get_config_string("", "space_perturb", ""));//seed, factor, size, persist, octave
	if(str != "")
	{
		ParseParams(&str, param, 9);

		*galaxy = PerturbGalaxy(*galaxy, param[0], param[1], param[2], param[3], param[4]);
	}
	*subspace = CreateGalaxy(name2.c_str());
	
	str.assign(get_config_string("", "subspace_perturb", ""));//seed, factor, size, persist, octave
	if(str != "")
	{
		ParseParams(&str, param, 9);

		*subspace = PerturbGalaxy(*subspace, param[0], param[1], param[2], param[3], param[4]);
	}
}

//loads a gray scale pcx image to use as thumbnail
BITMAP* CreateGalaxyThumb(const char* file_name)
{
	BITMAP* source = load_bitmap(file_name, 0);
	ConvertGrayToIdx(source);
	return source;
}
