//This file is part of Future's End
//Copyright 2006-2008 SiegeLord
//See license.txt for distribution information
//
//ai.cpp
//Handles the bot's interaction with the game
//
//The general idea behind this is the following:
//For each game type there is a number of actions that a planet can do in a turn.
//By analysing the situation, different probabilities are assigned to those actions.
//Then, an action is chosen randomly according to those probabilities

#include "ai.h"
#include "planet.h"
#include "sound.h"
#include "game.h"

extern int g_nGameType;

//a struct used to store the names of the AI bots
struct SAIName
{
	string name;	//the actual name
	bool used;		//has it been used or not by someone
	SAIName()
	{
		used = false;
	}
};


vector<SAIName> g_sAINames;		//an array of bot names
vector<string> g_sAISayings;	//an array of bot sayings
float g_fTalkRate;				//measures how often the bots say stuff

//initializes the AI
//loads the names and sayings from the ai.ini
void InitAI()
{
	g_sAINames.clear();

	set_config_file("ai.ini");
	
	int n = 0;
	char token[50];

	//load AI names
	while(true)
	{
		sprintf(token, "%d", n);

		const char* name = get_config_string("NAMES", token, "");

		if(0 == strcmp(name, ""))
			break;

		SAIName entry;
		entry.used = false;
		entry.name.assign(name);
		
		g_sAINames.push_back(entry);

		n++;
	}

	n = 0;

	g_fTalkRate = get_config_float("SPEECH", "rate", 0.02);
	//load AI sayings
	while(true)
	{
		sprintf(token, "%d", n);

		const char* str = get_config_string("SPEECH", token, "");

		if(0 == strcmp(str, ""))
			break;

		string entry(str);
		
		g_sAISayings.push_back(entry);

		n++;
	}
}

//returns a random, unique name for a Bot
//if there isn't a unique name, it returns 'Bot'
const char* GetAIName()
{
	int n = rand() % g_sAINames.size();				//picks a random start
	for(unsigned int ii = 0; ii < g_sAINames.size(); ii++)	
	{
		int idx = (ii + n) % g_sAINames.size();		//loop through the array from the start point
													//and stop at the first unused one
		if(!g_sAINames[idx].used)
		{
			g_sAINames[idx].used = true;			//mark it as used
			return g_sAINames[idx].name.c_str();
		}
	}
	return "Bot";	//we found nothing... :(
}

//handle the talking of a planet
void AITalk(SPlanet* planet)
{
	if(planet->done)	//planet is done, so we roll the chance twice
	{
		if(rand()%1000 < 2000 * g_fTalkRate)
		{
			if(rand()%1000 < 1000 * g_fTalkRate)
			{
				string entry = planet->name + string(": ") + g_sAISayings[rand() % g_sAISayings.size()];
				PutInChat(entry.c_str());
			}
		}
		return;
	}
	if(planet->dead)	//if the planet is dead, we make it done
	{
		planet->done = true;
		return;
	}

	if(rand()%1000 < 1000 * g_fTalkRate)	//the planet is not dead, roll the chance normally
	{
		string entry = planet->name + string(": ") + g_sAISayings[rand() % g_sAISayings.size()];
		PutInChat(entry.c_str());
	}
}

//handle the Bounty Style game
void HandleBounty(SPlanet* planet, list<SNPC> *npcs, vector<SPlanet> *planets, BITMAP* subspace)
{
	int scores[7];
	scores[0] = 10;		//random motion to energy
	scores[1] = 30;		//gndn
	scores[2] = 20;		//build generator
	scores[3] = 100;	//fire death beam
	scores[4] = 0;		//attack maneuver
	scores[5] = 0;		//build shield
	scores[6] = 0;		//fire subspace at target

	//we don't have enough energy for a generator, let's move around
	if(planet->energy_left < planet->num_generators * planet->num_generators * 20000)
		scores[0] += 30;

	//we have enough energy to build a generator, build it
	if(planet->energy > planet->num_generators * 20000)
		scores[2] += 90;

	//now we loop through the planets and the npc's to find the closest target
	float closest_dist = 0;
	bool first = true;

	bool have_something = false;	//this shall be used when we actually chose where to fire
	int dest_x = 0, dest_y = 0;	//these hold the target location

	//first we look through the planets
	for(unsigned int ii = 0; ii < planets->size(); ii++)
	{
		SPlanet* planet2 = &((*planets)[ii]);
		
		if(planet == planet2)
			continue;
		
		float dx = planet2->x - planet->x;
		float dy = planet2->y - planet->y;

		float d = dx * dx + dy * dy;
		
		if(d < closest_dist || first)
		{
			first = false;
			closest_dist = d;
			
			dest_x = planet2->x;
			dest_y = planet2->y;
			have_something = true;	//we now have a target	
		}
	}

	if(!have_something)
		closest_dist = 100000000;	//large

	//now loop through the npc's
	for(list<SNPC>::iterator it = npcs->begin(); it != npcs->end(); it++)
	{
		SNPC* cnpc = &(*it);
		if(cnpc->type != 1)		//the bounty circle
			continue;

		float dx = (cnpc->x - planet->x);
		float dy = (cnpc->y - planet->y);

		float d = dx * dx + dy * dy;

		d = d / 2 / sqrtf(cnpc->val1 / 100);	//we'd much rather hit this, so we factor in the
												//circle's power, and also we divide the distance by
												//root 2
		if(d < closest_dist)
		{
			have_something = true;
			dest_x = cnpc->x;
			dest_y = cnpc->y;
			closest_dist = d;
		}
	}
	
	if(!have_something)	//we have no target, so it no longer matters and the planet does nothing
		return;

	closest_dist = (dest_x - planet->x) * (dest_x - planet->x)
		+ (dest_y - planet->y) * (dest_y - planet->y); //now we calculate the unweighted distance

	closest_dist = sqrtf(closest_dist);

	//calculate how much it would take to fire that far
	int cost = closest_dist * GetMoveCost(planet->x, planet->y, dest_x, dest_y, 10000000, 0, 0, subspace) / 5;

	if(cost > planet->energy)//not enough energy to fire that far
	{
		scores[3] /= 2;		//reduce the chance to fire
		scores[4] += 40;	//increase the chance to do the attack maneuver
	}

	//now, decide what action to do
	int total = 0;					//total ammount of points
	for(int ii = 0; ii < 7; ii++)	//we sum up the points for all of the actions
	{
		total += scores[ii];
	}
	int decision = rand()%total;	//now we pick a random number in the range of the total points
	int num = 0;
	for(int ii = 0; ii < 7; ii++)	//now we go through each of the actions
	{
		num += scores[ii];			//we add the action's number of points to the number
		if(decision < num)			//and if it is larger that the decision number, we choose it
		{
			decision = ii;
			break;
		}
	}

	//now we switch through the actions
	switch(decision)
	{
	case 0://random movement
		{
		SetMoveState(planet, planet->x - 100 + rand() % 200, planet->y - 100 + rand() % 200, subspace);
		}
		break;
	case 1://do nothing
		planet->state = -1;
		break;
	case 2://build generator
		{
		SetGeneratorState(planet);
		}
		break;
	case 3://attack something
		{
		planet->state = 1;
		while(cost > GetWavePower(planet))	//we just need to get there, nothing special
			IncreaseWavePower(planet);		//so we increase the power until it matches the cost
		if(planet->race == 0)		//communists
		{
			SetIniWaveState(planet, dest_x, dest_y, false);
			SetWaveState(planet, AL_PI / 64);
		}
		else if(planet->race == 1)	//anarchists
		{
			SetBombState(planet, dest_x, dest_y, false, subspace);
		}
		else						//capitalists
		{
			float theta = atan2f(dest_y - planet->y, dest_x - planet->x);
			SetPodState(planet, planet->x + cosf(theta) * 1000, planet->y + sinf(theta) * 1000, subspace);
		}
		}
		break;
	case 4://attack maneouvre
		{
		int x,y;
		//move within 50 of the target
		x = float(dest_x - planet->x) / closest_dist * 50 + dest_x;
		y = float(dest_y - planet->y) / closest_dist * 50 + dest_y;
		SetMoveState(planet, x , y, subspace);
		}
	case 5://build shield
		{
			SetShieldState(planet);
		}
		break;
	case 6://fire subspace
		{
			if(planet->race == 0) //communists
			{
				SetIniWaveState(planet, dest_x, dest_y, true);
				SetWaveState(planet, AL_PI / 64);
			}
			else if(planet->race == 1)	//anarchists
			{
				SetBombState(planet, dest_x, dest_y, true, subspace);
			}
			else	//capitalists
			{
				SetIniBurstState(planet, dest_x, dest_y, true);
				SetBurstState(planet, AL_PI / 64);
			}
		}
		break;
	}
	planet->done = true;	//we are done
}

//handle the death match game mode
void HandleDeathMatch(SPlanet* planet, vector<SPlanet> *planets, BITMAP* subspace)
{
	int scores[7];

	scores[0] = 10;		//random motion to energy
	scores[1] = 30;		//gndn
	scores[2] = 10;		//build generator
	scores[3] = 0;		//fire death beam
	scores[4] = 10;		//attack maneuver
	scores[5] = 10;		//build shield
	scores[6] = 0;		//fire subspace at target

	//we don't have enough energy for a generator, let's move around
	if(planet->energy_left < planet->num_generators * planet->num_generators * 20000)
		scores[0] += 60;

	//can we build a generator? if yes, build it
	if(planet->energy > planet->num_generators * 20000)
		scores[2] += 100;
	
	//can we build a shield? if yes, build it
	if(planet->energy > planet->num_shields * 20000)
		scores[5] += 150;

	//if we have more generators than shields, we build some shields
	if(planet->num_generators > planet->num_shields)
		scores[5] += 20;
	else	//otherwise, we move around
		scores[0] += 20;

	if(planet->shield < planet->num_shields * 800)	//if we are low on shields, we are motivated to build
													//a new shield
		scores[5] += 10;


	//now we loop through the planets to find the closest one
	SPlanet* target = 0;
	bool first = true;
	float smallest_distance = 0;
	for(unsigned int ii = 0; ii < planets->size(); ii++)
	{
		SPlanet* pPlanet = &((*planets)[ii]);
		float dx = pPlanet->x - planet->x;
		float dy = pPlanet->y - planet->y;

		if(pPlanet == planet)
			continue;
		if(pPlanet->dead)
			continue;

		float distance = dx * dx + dy * dy;
		if(first)
		{
			first = false;


			smallest_distance = distance;
			target = pPlanet;
		}
		if(distance < smallest_distance)
		{
			smallest_distance = distance;
			target = pPlanet;
		}
	}	

	if(target == 0)//if we have no target, we probably have won, so we do nothing
		return;

	smallest_distance = sqrtf(smallest_distance);

	if(smallest_distance < 500)//if we are close, do the attack maneouvre
		scores[4] += 40;

	if(smallest_distance > 1000)//if we are far, do the attack maneouvre
		scores[4] += 40;

	if(smallest_distance < 300)//if we are very close, fire the weapon
		scores[3] += 3000 / smallest_distance;

	if(smallest_distance < 50)//if we are very very close we start to think
	{
		scores[3] += 10;	//make it more likely to fire
		if(planet->shield < planet->num_shields * 1000.0 / 2.0f)
			scores[6] += 60; //also, if we are weak, fire a subspace beam
	}

	if(smallest_distance < 10)//if we are at point black... we move out
		scores[0] += 80;

	//now we check the cost to move towards the target
	int cost = smallest_distance * GetMoveCost(planet->x, planet->y, target->x, target->y, 10000000, 0, 0, subspace) / 5;
	if(cost > planet->energy) //not enough energy to do that, so we make firing less likely
		scores[3] /= 2;

	//now, decide what action to do
	int total = 0;					//total ammount of points
	for(int ii = 0; ii < 7; ii++)	//we sum up the points for all of the actions
	{
		total += scores[ii];
	}
	int decision = rand()%total;	//now we pick a random number in the range of the total points
	int num = 0;
	for(int ii = 0; ii < 7; ii++)	//now we go through each of the actions
	{
		num += scores[ii];			//we add the action's number of points to the number
		if(decision < num)			//and if it is larger that the decision number, we choose it
		{
			decision = ii;
			break;
		}
	}
	
	//implement the chosen action
	switch(decision)
	{
	case 0://move randomly
		{
		SetMoveState(planet, planet->x - 100 + rand() % 200, planet->y - 100 + rand() % 200, subspace);
		}
		break;
	case 1://do nothing
		planet->state = -1;
		break;
	case 2://build generator
		{
		SetGeneratorState(planet);
		}
		break;
	case 3://fire weapon
		{
		planet->state = 1;
		while(planet->energy > GetWavePower(planet))//maximize the weapon power according to the max energy
			IncreaseWavePower(planet);
		if(planet->race == 0)//communists
		{
			SetIniWaveState(planet, target->x, target->y, false);
			if(smallest_distance < 50)
				SetWaveState(planet, AL_PI / 64);//fire a concentrated burst upclose
			else
				SetWaveState(planet, AL_PI / 32);
		}
		else if(planet->race == 1)//anarchists
		{
			SetBombState(planet, target->x, target->y, false, subspace);
		}
		else//capitalists
		{
			float theta = atan2f(target->y - planet->y, target->x - planet->x);
			SetPodState(planet, planet->x + cosf(theta) * 1000, planet->y + sinf(theta) * 1000, subspace);
		}
		}
		break;
	case 4://attack maneouvre
		{
		int x,y;
		//move within 50 to the target
		x = float(target->x - planet->x) / smallest_distance * 50 + target->x;
		y = float(target->y - planet->y) / smallest_distance * 50 + target->y;
		SetMoveState(planet, x , y, subspace);
		}
		break;
	case 5://build shield
		{
			SetShieldState(planet);
		}
		break;
	case 6://fire subspace
		{
			if(planet->race == 0)//communists
			{
				SetIniWaveState(planet, target->x, target->y, true);

				SetWaveState(planet, AL_PI / 64);
			}
			else if(planet->race == 1)//anarchists
			{
				SetBombState(planet, target->x, target->y, true, subspace);
			}
			else//capitalists
			{
				SetIniBurstState(planet, target->x, target->y, true);

				SetBurstState(planet, AL_PI / 64);
			}
		}
		break;
	}
	planet->done = true;
}	

//handles the AI
void HandleAI(SPlanet* planet, list<SNPC> *npcs, vector<SPlanet> *planets, BITMAP* subspace)
{
	AITalk(planet);
	switch(g_nGameType)
	{
	case 0:
		HandleDeathMatch(planet, planets, subspace);
		break;
	case 1:
		HandleBounty(planet, npcs, planets, subspace);
		break;
	}
}
