/* player.h


   This handles all aspects of the players that must be stored outside the
   GAME struct (see game.h). This includes players' controls, strength and
   weapons, for example, which would be carried over to secret levels if we
   ever implemented such a feature. However, players' coordinates are stored
   in the form of sprites inside the GAME struct (see s_player.h).

   During the game, players are handled whenever the sprite engine (see
   sprite.h) comes across their sprites. For this reason, each player sprite
   has a pointer to its PLAYER struct, but not the other way round.

   This struct also contains any information required to draw the player's
   view. Where necessary (as with the camera), this is updated when the
   sprite is processed, but it can be used in the drawing code where we do
   not have easy access to the player's sprite.
*/

#ifndef INCLUDED_PLAYER_H
#define INCLUDED_PLAYER_H

#include "zcontrol.h"

/* This struct holds a camera position. This is a linked list so that old
   camera positions may be remembered if we ever recurse into games within
   games. During a game, only the first entry is ever used.
*/
typedef struct CAMERA {

/* struct CAMERA *next; - it's not a linked list in this version! */

 /* Camera coordinates and theta (horizontal angle). Height and phi (vertical
    angle) are constant - see draw.h for their values. These are attracted
    towards the target camera position when processing the PLAYER structs.
 */

/* These have moved into the camera sprite
 float x, y, z;
 float th;
*/

 /* Target for the camera position. These are set when processing the player
    sprites, and will remain constant when the player dies. */
 float xt, yt, zt;
 float tht;

} CAMERA;

#define PLAYER_STAT_INVULNERABLE 1
#define PLAYER_STAT_SPEED_BOOST  2

#define PLAYER_STAT_INVULNERABLE_N 0
#define PLAYER_STAT_SPEED_BOOST_N  1


/* This struct holds everything about the player except his coordinates.
*/
typedef struct PLAYER {

 /* Pointer to next player */
 struct PLAYER *next, *prev;

 /* The player's ID. From 0 to N_PLAYERS - 1 to match the level data */
 int id;

 /* The player's control device. */
 CONTROL *control;

 /* This goes down when the player gets hurt. */
 int hp, maxhp;

 /* Set when the player fires, and decremented every frame. When it reaches
    0, the player can fire again! :-)
 */
 int fire_delay;

 int stat_timer[32];
 int stat;

 int won;

 int points;

 int held_powerup;
 int powerup_time;
 int num_bombs;
 int num_missiles;

 /* Used to determine how long has the player been dead */
 int time_of_death;

 /* Camera position. */
 SPRITE *camspr;

 /* Data to be used by the AI. Use it however you need to. */
 void *data;

 /* Player sprite */
 struct SPRITE *sprite;

} PLAYER;



/* create_player(): this creates a player and initialises all its fields to
   their initial values. 'camera' should be set to 0; a camera will be added
   when 'control' may be left
   uninitialised, and will be subsequently initialised by the caller;
   'camera' will be initialised by initialise_game() in game.h.
*/
PLAYER *create_player(int id);


void player_deduct_hp(PLAYER *player, int hp);

void destroy_player(PLAYER *p);
void destroy_players(PLAYER *p);

#endif /* INCLUDED_PLAYER_H */
