/*

*************************************************************************

lightmap.cpp - Iron Army cheap alpha blending effects.
Copyright (C) 2002 Mike Farrell (gccdragoonkain@yahoo.com)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

***************************************************************************/

//Includes///////////////////////////////////////////////////////////////////
#include <allegro.h>
#include "lightmap.h"
#include "main.h"

void colorize(BITMAP *bmp, float hue, float sat)
{
  int x, y, c, r, g, b;
  float h, s, v;

  acquire_bitmap(bmp);
  for(y = 0; y < bmp->h; y++) for(x = 0; x < bmp->w; x++)
  {
    c = getpixel(bmp, x, y);
    if(c == makecol(255, 0, 255)) continue;
    r = getr(c);
    g = getg(c);
    b = getb(c);
    rgb_to_hsv(r, g, b, &h, &s, &v);
    hsv_to_rgb(hue != -1 ? hue : h, sat != -1 ? sat : s, v, &r, &g, &b);
    putpixel(bmp, x, y, makecol(r, g, b));
  }
  release_bitmap(bmp);
}

void monocrome_draw_lightmap(BITMAP *dest, BITMAP *bmp, int x, int y, int c)
{
  monocrome_draw_lightmap_ex(dest, bmp, x, y, c, 0);
}

void monocrome_draw_lightmap_ex(BITMAP *dest, BITMAP *bmp, int x, int y, int c, int a_offset)
{
  int cx, cy;

  if(x > SCREEN_W || y > SCREEN_H || x+bmp->w < 0 || y+bmp->h < 0)
    return;
  drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
  acquire_bitmap(dest);
  for(cy = 0; cy < bmp->h; cy++) for(cx = 0; cx < bmp->w; cx++)
  {
    int a = getr(getpixel(bmp, cx, cy))+a_offset;

    if(a < 0) a = 0;
    if(a > 255) a = 255;
    set_trans_blender(0, 0, 0, a);
    putpixel(dest, cx+x, cy+y, c);
  }
  release_bitmap(dest);
  solid_mode(); 
}

LIGHTMAP *create_lightmap(BITMAP *bmp)
{
  int x, y;

  LIGHTMAP *lmp = (LIGHTMAP *)malloc(sizeof(struct LIGHTMAP));
  if(!lmp) return NULL;

  lmp->wt = bmp->w;
  lmp->ht = bmp->h;
  lmp->a_offset = 0;

  lmp->a = (unsigned char **)malloc(sizeof(struct LIGHTMAP[lmp->wt]));
  if(!lmp->a) return NULL;
  lmp->c = (int **)malloc(sizeof(struct LIGHTMAP[lmp->wt]));
  if(!lmp->c) return NULL;
  for(y = 0; y < lmp->wt; y++)
  {
    lmp->a[y] = (unsigned char *)malloc(sizeof(struct LIGHTMAP[lmp->ht]));
    if(!lmp->a[y]) return NULL;
    lmp->c[y] = (int *)malloc(sizeof(struct LIGHTMAP[lmp->ht]));
    if(!lmp->c[y]) return NULL;
  }

  for(y = 0; y < lmp->ht; y++) for(x = 0; x < lmp->wt; x++)
  {
    float h, s, v;
    int r, g, b, c = getpixel(bmp, x, y);

    rgb_to_hsv(getr(c), getg(c), getb(c), &h, &s, &v);
    lmp->a[x][y] = (unsigned char)(v*255);
    hsv_to_rgb(h, s, 1.0, &r, &g, &b);
    lmp->c[x][y] = makecol(r, g, b);
  }

  return lmp;
}

void draw_lightmap(BITMAP *bmp, LIGHTMAP *lmp, int x, int y)
{
  int _x, _y;

  drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
  acquire_bitmap(bmp);
  for(_y = 0; _y < lmp->ht; _y++) for(_x = 0; _x < lmp->wt; _x++)
  {
    set_trans_blender(0, 0, 0, MAX(lmp->a[_x][_y]+lmp->a_offset, 0));
    putpixel(bmp, x+_x, y+_y, lmp->c[_x][_y]);
  }
  release_bitmap(bmp);
  solid_mode();
}

void destroy_lightmap(LIGHTMAP *lmp)
{
  int y;

  for(y = 0; y < lmp->wt; y++)
  {
    free(lmp->c[y]);
    free(lmp->a[y]);
  }
  free(lmp->c);
  free(lmp->a);
  free(lmp);
}

void grad_line(BITMAP *dest, int x1, int y1, int x2, int y2, int ln, int c)
{
  int i;

  drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
  for(i = -ln/2; i < ln/2; i++)
  {
    float a = 1.0-(abs(i)/((float)ln/2));

    set_trans_blender(0, 0, 0, (int)((255 * ((i == 0) ? 1.0 : a))+0.5));

    line(dest, x1+((i+(ln/2))-(ln/2)), y1, x2+((i+(ln/2))-(ln/2)), y2, c);
  }
  solid_mode();
}

void electric_line(BITMAP *dest, int x1, int y1, int x2, int y2, int line_w, int vary)
{
  int mx = MIN(x1, x2)+abs(x1-x2)/2, my = MIN(y1, y2)+abs(y1-y2)/2, dx, dy;

  dx = random()%vary;
  dy = random()%vary;
  if(random()%2 == 0) dx *= -1;
  if(random()%2 == 0) dy *= -1;
  mx += dx;
  my += dy;

  grad_line(dest, x1, y1, mx, my, line_w, makecol(255, 255, 100));
  grad_line(dest, mx, my, x2, y2, line_w, makecol(255, 255, 100));
}
