#include <stdio.h>
#include <string.h>
#include <time.h>
#include <allegro.h>

#include "stax.h"
#include "palette.h"

int tick = 0;

static void timer_callback(void)
{
	++tick;
}
END_OF_STATIC_FUNCTION(timer_callback);

BITMAP *vs = NULL;       /* virtual screen for double buffering */
DATAFILE *dat = NULL;
MIDI *intro_midi;
char *argv0;
int driver;
int big_screen;

Config cfg = 
{
	GAME_STYLE_SUCKA,    /* game style */
	"space.dat",         /* theme */
	INPUT_KB2,           /* player 1 input */
	INPUT_NONE,          /* player 2 input */
	1                    /* sound */
};

static int my_icon_proc(int, DIALOG *, int);

DIALOG intro_dialog[] =
{
	{ my_icon_proc,130, 95,60,15,0,0,0,D_EXIT,0,0,NULL,NULL,NULL },
	{ my_icon_proc,130,110,60,15,0,0,0,D_EXIT,0,0,NULL,NULL,NULL },
	{ my_icon_proc,130,125,60,15,0,0,0,D_EXIT,0,0,NULL,NULL,NULL },
	{ my_icon_proc,130,140,60,15,0,0,0,D_EXIT,0,0,NULL,NULL,NULL },
	{         NULL,  0,  0, 0, 0,0,0,0,     0,0,0,NULL,NULL,NULL }
};

static int (*game_loops[NUM_GAME_STYLES])(void) =
{
	sucka_loop,
	clobber_loop,
	shifty_loop,
	coop_loop,
	puzzle_loop
};

static int (*single_game_loops[NUM_GAME_STYLES])(void) = {
    sucka_single_loop,
    clobber_single_loop,
    shifty_single_loop,
    NULL, // no single player co-op
    puzzle_loop
};

static int init(void);
static void cleanup(void);

#define INTRO_START    0
#define INTRO_CONFIG   1
#define INTRO_SCORES   2
#define INTRO_EXIT     3
static int intro_screen(void);

static void show_scores(void);

int main(int argc, char **argv)
{
	int quit = 0;
	int i;

	argv0 = argv[0];
	read_config();

	driver = GFX_AUTODETECT_FULLSCREEN;
	big_screen = 1;

	for (i = 1; i < argc; i++) {
		if (!stricmp(argv[i], "-windowed"))
		    driver = GFX_AUTODETECT_WINDOWED;
		else if (!stricmp(argv[i], "-big")) {
		    big_screen = 1;
		}
                else if (!stricmp(argv[i], "-small")) {
                    big_screen = 0;
                }
		else {
			printf("Available options are:\n");
			printf("\t-windowed    Run in a window\n");
			printf("\t-big         Run at 640x400 instead of 320x200\n");
                        printf("\t-small       Run at 320x200\n");
			exit(1);
		}
	}

	if (!init()) {
		cleanup();
		return 1;
	}

	while (!quit) {
		switch (intro_screen()) {
			case INTRO_START:
				stop_midi();
                                if (cfg.input2 != INPUT_NONE) {
    				    while ((*game_loops[cfg.style])())
                                        ;
                                }
                                else {
                                    while ((*single_game_loops[cfg.style])())
                                        ;
                                }
				play_midi(intro_midi, 1);
				break;
			case INTRO_CONFIG:
				config_loop();
				break;
			case INTRO_SCORES:
				show_scores();
				break;
			case INTRO_EXIT:
				quit = 1;
				break;
		}
	}

	cleanup();

	return 0;
}
END_OF_MAIN();

static int init(void)
{
	srand(time(NULL));

	allegro_init();
	install_timer();
	install_keyboard();
	install_joystick(JOY_TYPE_AUTODETECT);

	install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL);

	if (cfg.sound)
		set_volume(255, MIDI_VOLUME);
	else
		set_volume(0, 0);

	if ((dat = load_datafile(datapath("stax.dat"))) == NULL) {
		allegro_message("Error loading stax.dat\n");
		return 0;
	}

	if ((vs = create_bitmap(320, 200)) == NULL) {
		allegro_message("Error creating virtual screen\n");
		return 0;
	}

	if (!big_screen) {
		if (set_gfx_mode(driver, 320, 200, 0, 0) < 0) {
			printf("Setting mode 320x200 failed, trying 640x400...\n");
			big_screen = 1;
			if (set_gfx_mode(driver, 640, 400, 0, 0) < 0) {
				allegro_message("Error: %s\n", allegro_error);
				return 0;
			}
		}
	}
	else {
		if (set_gfx_mode(driver, 640, 400, 0, 0) < 0) {
			printf("Setting mode 640x400 failed, trying 320x200...\n");
			big_screen = 0;
			if (set_gfx_mode(driver, 320, 200, 0, 0) < 0) {
				allegro_message("Error: %s\n", allegro_error);
				return 0;
			}
		}
	}

	set_palette(palette);

	LOCK_VARIABLE(tick);
	LOCK_FUNCTION(timer_callback);
	install_int_ex(timer_callback, BPS_TO_TIMER(TIMER_FREQ));

	font = find_object_data(dat, "FONT");
	intro_midi = find_object_data(dat, "INTRO_MUSIC");

	read_high_scores(get_high_score_filename());

	return 1;
}

static void cleanup(void)
{
	if (write_high_scores(get_high_score_filename()) < 0) {
		error("Couldn't write the high score file!",
				"You might not have permission to write to that file",
				"Or your disk may be full");
	}

	remove_sound();
	remove_keyboard();
	remove_joystick();
	remove_timer();
	set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);

	if (vs) destroy_bitmap(vs);
	if (dat) unload_datafile(dat);
}

static int my_icon_proc(int msg, DIALOG *d, int c)
{
	switch (msg) {
	case MSG_WANTFOCUS:
		return D_WANTFOCUS;
	case MSG_LOSTFOCUS:
		if (d->dp3)
			play_sample(d->dp3, 255, 128, 1000, 0);
		break;
	case MSG_DRAW:
		if (d->flags & D_GOTFOCUS) {
			if (d->dp2)
				stretch_sprite(vs, d->dp2, d->x, d->y, d->w, d->h);
		}
		else
			stretch_sprite(vs, d->dp, d->x, d->y, d->w, d->h);
		if (big_screen)
			stretch_blit(vs, screen, d->x, d->y, d->w, d->h, d->x*2, d->y*2, d->w*2, d->h*2);
		else
			blit(vs, screen, d->x, d->y, d->x, d->y, d->w, d->h);
		break;
	case MSG_KEY:
		play_sample(d->dp3, 255, 128, 800, 0);
		if (d->flags & D_EXIT)
			return D_CLOSE;
		break;
	}

	return D_O_K;
}

static int intro_screen(void)
{
	BITMAP *bg;
	BITMAP *logo;
	int ret = INTRO_EXIT;
	static int first_call = 1;

	bg = find_object_data(dat, "INTRO_BG");
	logo = find_object_data(dat, "STAX_LOGO");

	intro_dialog[0].dp = find_object_data(dat, "START");
	intro_dialog[0].dp2 = find_object_data(dat, "START_H");
	intro_dialog[1].dp = find_object_data(dat, "CONFIG");
	intro_dialog[1].dp2 = find_object_data(dat, "CONFIG_H");
	intro_dialog[2].dp = find_object_data(dat, "SCORES");
	intro_dialog[2].dp2 = find_object_data(dat, "SCORES_H");
	intro_dialog[3].dp = find_object_data(dat, "EXIT");
	intro_dialog[3].dp2 = find_object_data(dat, "EXIT_H");
	intro_dialog[0].dp3 = find_object_data(dat, "BUTTON");
	intro_dialog[1].dp3 = intro_dialog[2].dp3 = intro_dialog[3].dp3 = intro_dialog[0].dp3;

	/*
	 * Rise the background and spin in a logo.
	 * This enhances the "coolness" of the game, or something like that.
	 */
	if (first_call) {
		SAMPLE *spin = find_object_data(dat, "SPIN");
		BITMAP *tmp;
		fixed y = itofix(200/2);
		fixed angle = 0;
		fixed scale;
		int w, h;
		clear(screen);
		set_palette(palette);
		tmp = create_bitmap(logo->w, logo->h);
		if (tmp) {
			play_sample(spin, 255, 128, 1000, 0);
			for (scale = 0; scale <= itofix(1); scale += fmul(ftofix(0.0005), itofix(w))) {
				tick = 0;
				clear(tmp);
				w = logo->w * fixtof(scale);
				h = logo->h * fixtof(scale);
				stretch_sprite(tmp, logo, 0, 0, w, h);
				clear(vs);
				blit(bg, vs, 0, 0, 0, y>>16, 320, 320-(y>>16));
				pivot_sprite(vs, tmp, 320/2, logo->h/2+5, w/2, h/2, angle);
				do_blit(vs);
				w = tick;
				y -= fmul(ftofix(0.05), itofix(w));
				angle += fmul(ftofix(0.256), itofix(w));
			}
		}
		blit(bg, vs, 0, 0, 0, 0, bg->w, bg->h);
		draw_sprite(vs, logo, (320-logo->w)/2, 5);
		do_blit(vs);
		play_midi(intro_midi, 1);
		first_call = 0;
	}
	else {
		blit(bg, vs, 0, 0, 0, 0, bg->w, bg->h);
		draw_sprite(vs, logo, (320-logo->w)/2, 5);
		do_blit(vs);
	}

	ret = do_dialog(intro_dialog, 0);

	return ret;
}

void show_scores(void)
{
	int style = cfg.style;
	SAMPLE *samp = find_object_data(dat, "BUTTON");
	if (style == GAME_STYLE_PUZZLE)
		style = GAME_STYLE_SUCKA;

	do 
		poll_joystick();
	while (joy_b1 || joy_b2 || joy_b3 || joy_b4);

	while (1) {
		display_high_scores(style);
		if ((tick % 2000) < 1000) {
			textprintf_shadow(vs, font, 5, 100, C_YELLOW, C_BLACK, "<");
			textprintf_shadow(vs, font, 320-5-8, 100, C_YELLOW, C_BLACK, ">");
		}
		do_blit(vs);
		if (keyboard_needs_poll())
			poll_keyboard();
		poll_joystick();
		if (key[KEY_ESC] || joy_b1 || joy_b2 || joy_b3 || joy_b4) {
			while (key[KEY_ESC] || joy_b1 || joy_b2 || joy_b3 || joy_b4) {
				if (keyboard_needs_poll())
					poll_keyboard();
				poll_joystick();
			}
			break;
		}
		if (key[KEY_LEFT] || joy_left) {
			play_sample(samp, 255, 128, 1000, 0);
			while (key[KEY_LEFT] || joy_left) {
				if (keyboard_needs_poll())
					poll_keyboard();
				poll_joystick();
			}
			style--;
			if (style < 0)
				style = GAME_STYLE_PUZZLE-1;
		}
		if (key[KEY_RIGHT] || joy_right) {
			play_sample(samp, 255, 128, 1000, 0);
			while (key[KEY_RIGHT] || joy_right) {
				if (keyboard_needs_poll())
					poll_keyboard();
				poll_joystick();
			}
			style++;
			if (style == GAME_STYLE_PUZZLE)
				style = GAME_STYLE_SUCKA;
		}
	}
	clear_keybuf();
}

