/**\file network.h
*
*  network declaration
*
*  Network Engine
*
*\author Castagnier Mickal
*
*\version 1.0
*
*\date 10/05/2005
*
*/



#ifndef N_NETWORK

    #define N_NETWORK

    #ifdef __cplusplus
        extern "C"
        {
    #endif



    /*!
     * Size of tcp receive buffer
     */

    #define NETW_SOCKET_RECV_BUF 2048



    /*!
     * Size of tcp send buffer
     */

    #define NETW_SOCKET_SEND_BUF 2048



    /*!
     * NETWORK -> mode value for a CLIENT (meaning connecting)
     */

    #define CLIENT 2



    /*!
     * NETWORK -> mode value for a SERVER (meaning listening)
     */

    #define SERVER 4



    /*!
     * Flag of TIMING system
     */

    #define NETW_TIME       8



    /*!
     * Flage of send BUFFERING system
     */

    #define NETW_SENDBUF    16



    /*!
     * Flage of recv BUFFERING system
     */

    #define NETW_RECVBUF    32



    /*!
     * NETW_QUIT State for a NETWORK who want to stop (if stopped NETWORK is uninitialized )
     */

    #define NETW_QUIT        -64



    /*!
     * NETW_RUN State for a running NETWORK
     */

    #define NETW_RUN         -128



    /*!
     * NETW_ERROR State for an error in a NETWORK
     */

    #define NETW_ERROR       -256



    /*!
     * NETW_STOP State for a stopped NETWORK
     */

    #define NETW_STOP       -512



    /*!
     * Number of item who can be queued on the sending side
     */

    #define SEND_SIZE 100



    /*!
     * Number of item who can be queued on the receiving side
     */

    #define RECV_SIZE 100



    /*!
     * Network Message is new identification: (double)type , (double)id , (N_STR *)name , (N_STR *)password
     */

    #define NETMSG_IDENT_NEW 0



    /*!
     * Network Message is identification request: (double)type , (double)id , (N_STR *)name , (N_STR *)password
     */

    #define NETMSG_IDENT_REQUEST 1



    /*!
     * Network Message is identification reply OK : (double)type , (double)id , (N_STR *)name , (N_STR *)password
     */

    #define NETMSG_IDENT_REPLY_OK 2



    /*!
     * Network Message is identification reply NON OK: (double)type , (double)id , (N_STR *)name , (N_STR *)password
     */

    #define NETMSG_IDENT_REPLY_NOK 3



    /*!
     * Network Message is string: (double)type , (double)id , (N_STR *)name , (N_STR *)chan , (N_STR *)text , (double)color
     */

    #define NETMSG_STRING 4



    /*!
     * Network Message is position: (double)type , (double)id , (double)X , (double)Y , (double)x_shift , (double)y_shift ,(double)vx ,(double)vy , (double)speed , (double)acceleration , (double)time_stamp
     */

    #define NETMSG_POSITION 5



    /*!
     * Network Message is ping request: (double)type , (double)id_from , (double)id_to
     */

    #define NETMSG_PING_REQUEST 6



    /*!
     * Network Message is ping reply: (double)type , (double)id_from , (double)id_to
     */

    #define NETMSG_PING_REPLY 7



    /*!
     *Structure of a N_SOCKET
     */

    typedef struct N_SOCKET
    {
        /*!port of socket*/
        int port;

        /*!a normal socket*/
        SOCKET sock;

        /*!ip of the connected socket*/
        char *ip;

        /*!address of local machine*/
        struct sockaddr_in haddr;

        /*!address of remote machine*/
        struct sockaddr_in raddr;

    } N_SOCKET;



    /*!
     *Structure of a NETWORK
     */

    typedef struct NETWORK
    {
        /*! Nb pending connection,if listening */
        int nb_pending,
        /*! NETWORK mode , 1 listening, 0 connecting */
        mode,
        /*! Nb octet sended */
        sended,
        /*! Send limit in octet */
        sendlimit,
        /*! state of the connection */
        state;

        /*! start time */
        N_TIME timer;
        /*! start time */
        N_TIME send_timer;
        /*! start time */
        N_TIME recv_timer;

        /*!networking socket*/
        N_SOCKET link;

        /*!sending buffer (for outgoing queuing )*/
        N_STRLIST *send_buf;

        /*!reveicing buffer (for incomming usage)*/
        N_STRLIST *recv_buf;

        /*!sending thread*/
        pthread_t send_thr;
        /*!receiving thread*/
        pthread_t recv_thr;
        /*!mutex for threaded use of Do_..._Network*/
        pthread_mutex_t sendbolt;
        /*!mutex for threaded use of Do_..._Network*/
        pthread_mutex_t recvbolt;
        /*!mutex for state event */
        pthread_mutex_t eventbolt;

    } NETWORK;



    /*!
     * Structure for cell of array of int inside NETW_MSG
     */

    typedef struct CELL_NB
    {
        /*!value*/
        double var;

        /*!next int*/
        struct CELL_NB *next,
        /*!previous int*/
                       *prev;

    } CELL_NB;



    /*!
     * Structure for array of int inside NETW_MSG
     */

    typedef struct TAB_NB
    {
        CELL_NB
        /*!start of list*/
        *start,
        /*!end of list*/
        *end;

        int
        /*!number of item*/
        nb_item;

    } TAB_NB;


    /*!network message, array of char and int*/
    typedef struct NETW_MSG
    {
        /*! array of int */
        TAB_NB    *tabint;
        /*! array of char */
        N_STRLIST *tabstr;

    }NETW_MSG;



    /*
     * Create a NETW_MSG *object
     */

    int create_msg( NETW_MSG **msg );



    /*
     * Delete a NETW_MSG *object
     */

    int delete_msg( NETW_MSG **msg );



    /*
     * Empty a NETW_MSG *object
     */

    int empty_msg( NETW_MSG **msg );



    /*
     * Add an int to the int list int the message
     */

    int add_nb_to_msg( NETW_MSG *msg , double value );



    /*
     * Add a string to the string list in the message
     */

    int add_str_to_msg( NETW_MSG *msg , N_STR *str );



    /*
     * Get an int from the message
     */

    int get_nb_from_msg( NETW_MSG *msg , double *value );



    /*
     * Get a string from the message
     */

    N_STR *get_str_from_msg( NETW_MSG *msg );



    /*
     * Make a single string of the message
     */

    N_STR *make_str_from_msg( NETW_MSG *msg );



    /*
     * Make a message from a single string
     */

    NETW_MSG *make_msg_from_str( N_STR *str );



    /*
     * Used by Init & Close network
     */

    int Handle_Wsa( int mode , int v1 , int v2 );



    /*
     * Initialize network, do nothing under linux
     */

    int Init_All_Network( int version1 , int version2 );



    /*
     * Close network, do nothing under linux
     */

    int Close_All_Network();



    /*
     * Connecting
     */

    char *Connect_Network( NETWORK **netw , char *ip , int port );



    /*
     * Closing
     */

    int Close_Network( NETWORK **netw );



    /*
     * Listening network
     */

    int Make_Listening_Network( NETWORK **netw , int PORT , int nbpending );



    /*
     * Accepting routine
     */

    NETWORK *Accept_From_Network( NETWORK *from );


    /*
     * Add a message to send in aimed NETWORK
     */

    int Add_Msg( NETWORK *netw , char *msg , int nboct );



    /*
     * Get a message from aimed NETWORK
     */

    int Get_Msg( NETWORK *netw , char **msg , int *nboct );



    /*
     * Create the receiving thread of a NETWORK
     */

    int Create_Recv_Thread( NETWORK *netw );



    /*
     * Create the sending thread of a NETWORK
     */

    int Create_Send_Thread( NETWORK *netw );



    /*
     * To Thread Sending function
     */

    void *send_thread( void *NET );



    /*
     * To Thread Receiving function
     */

    void *recv_thread( void *NET );



    /*
     * Stop a NETWORK connection sending and receing thread
     */

    int Stop_Network( NETWORK **netw );



    /*
     * Writting to a socket
     */
    int send_data( SOCKET s,   /* connected socket */
                   char *buf,  /* pointer to the buffer */
                   int n      /* number of characters (bytes) we want */
                 );



    /*
     * Reading from a socket
     */
    int recv_data( SOCKET s,   /* connected socket */
                   char *buf,  /* pointer to the buffer */
                   int n );    /* number of characters (bytes) we want */



    /*
     * Load the server config file, create it on error
     */

    int load_server_config( char *file ,
                            int *MAX_CLIENTS , int *RESERVED , int *PORT ,
                            int *ADMPORT , int *LOGGING , int *sendlimit );



    /*
     * Load the server config file, create it on error
     */

    int save_server_config( char *file ,
                            int MAX_CLIENTS , int RESERVED , int PORT ,
                            int ADMPORT , int LOGGING ,
                            int sendlimit );



    /*
     * Load the client config file, create it on error
     */

    int load_client_config( char *file ,
                            char **ip , int *PORT , int *sendlimit , int *LOGGING );



    /*
     * Load the client config file, create it on error
     */

    int save_client_config( char *file ,
                            char *ip , int PORT , int sendlimit , int LOGGING );


    /*
     * Set the send limit of a network
     */

    int set_network_rate( NETWORK *netw , int sendlimit );



    /*
     * Restart or reset the specified network ability
     */

    int network_reset( NETWORK *netw , int flag );




    /*
     * Get the type of a message without modifying it
     */

	double netw_msg_get_type( char *msg );



    /*
     * Add a formatted NETWMSG_IDENT message to the specified network
     */

    int netw_send_ident( NETWORK *netw , double type , double id , char *name , char *passwd );



    /*
     * Add a formatted NETWMSG_IDENT message to the specified network
     */

    int netw_send_position( NETWORK *netw , double id , double X , double Y , double vx , double vy , double acc_x , double acc_y , double time_stamp );


    /*
     * Add a string to the network
     */

    int netw_send_string_to_all( NETWORK *netw , double id , char *name , char *chan , char *txt , double color );



    /*
     * Add a string to the network, aiming a specific user
     */

    int netw_send_string_to( NETWORK *netw , double id_from , double id_to , char *name , char *txt , double color );



    /*
     * Add a ping reply to the network
     */

    int netw_send_ping( NETWORK *netw , double type , double id_from , double id_to , double time );



    /*
     * Retrieves identification from netwmsg
     */

    int netw_get_ident( N_STR *msg , double *type , double *ident );



    /*
     * Retrieves position from netwmsg
     */

    int netw_get_position( N_STR *msg , double *id , double *X , double *Y , double *vx , double *vy , double *acc_x , double *acc_y , double *time_stamp );



    /*
     * Retrieves string from netwmsg
     */

    int netw_get_string( N_STR *msg , double *id , char **name , char **chan , char **txt , double *color );



    /*
     * Retrieves a ping travel elapsed time
     */

    int netw_get_ping( N_STR *msg , double *type , double *from , double *to , double *time );



    #ifdef __cplusplus
        }
    #endif



#endif /*#ifndef N_NETWORK*/

