/* KIP-FRACTAL - a Julia set explorer using Allegro
 * Version beta-1
 * (c) 2007 Jurre "Kipmans" Hanema, <kipmans@gmail.com>
 *
 * Released under "THE BEER-WARE LICENSE" (Revision 42):
 * <kipmans@gmail.com> wrote this file. As long as you retain this notice you can 
 * do whatever you want with this stuff. If we meet some day, and you think this 
 * stuff is worth it, you can buy me a beer in return.
 */
 
#include <complex>
#include <list>
#include <utility>
#include <cstdio>
#include <cstdlib>
#include <allegro.h>

#include "logo.h"


typedef	std::complex<double>		complex_t;	
typedef	std::list<int>			output_list_t;


const int	screen_width	= 800;
const int	screen_height	= 600;
const int	fractal_width	= 570;
const int	fractal_height	= 570;
const int	offset_left	= 220;
const int	offset_right	= 10;
const int	offset_top	= 10;
const int	offset_bottom	= 20;


#define	COL_BLACK	makecol(0, 0, 0)
#define	COL_WHITE	makecol(255, 255, 255)
#define COL_GREY	makecol(128, 128, 128)
#define COL_LIGHTGREY	makecol(200, 200, 200)


enum GuiAction
{
	GUI_ACT_RESTORE_DOMAIN,
	GUI_ACT_SET_DOMAIN,
	GUI_ACT_REGENERATE,
	GUI_ACT_EXIT,
	GUI_ACT_PRESET,
	GUI_ACT_NONE
};


enum GuiPreset
{
	GUI_PRE_4_06_055I,
	GUI_PRE_2_M1,
	GUI_PRE_5_08_06I,
	GUI_PRE_2_03_06I,
	N_GUI_PRESETS
};


enum ColourScheme
{
	CS_HSV_VAR_H,
	CS_HSV_VAR_SV,
	CS_SHADES_GREY,
	CS_BW,
	CS_SHADES_BROWN,
	CS_SHADES_BLUE,
	N_COLOURSCHEMES
};


class Domain
{
	// Simple class to implement a mathematical domain [a, b].
public:
	const double a() const
	{
		return va;
	}
	
	const double b() const
	{
		return vb;
	}
	
	void set(const double a, const double b)
	{
		va = a;
		vb = b;
		if(va > vb) std::swap(va, vb);
	}
	
	Domain(const double a, const double b)
	{
		set(a, b);
	}
	
	Domain(): va(0), vb(0) {}
private:
	double		va;
	double		vb;
};


struct set_properties_t
{
	Domain		domain_x;
	Domain		domain_y;
	double		max_i;
	double		infinity;
	complex_t	c;
	double		a;
	ColourScheme	colour_scheme;
};


void generate_set(BITMAP *bmp, const set_properties_t& s)
{
	// This is the most important function as it generates and draws the actual Julia set.
	// Not the most efficient algorithm probably...
	
	double		x, y;
	complex_t	z;
	
	int		r, g, b;
	
	double		stepsize_x = (s.domain_x.b() - s.domain_x.a()) / fractal_width;
	double		stepsize_y = (s.domain_y.a() - s.domain_y.b()) / fractal_height;
	
	for(double ix = 0; ix < fractal_width; ix++)
	{
		x = ix * stepsize_x + s.domain_x.a();

		for(double iy = 0; iy < fractal_height; iy++)
		{
			y = iy * stepsize_y + s.domain_y.b();
			z = complex_t(x, y);
			
			for(int i = 0; i < s.max_i; i++)
			{
				z = std::pow(z, s.a) + s.c;

				if(std::abs(z) >= s.infinity)
				{
					// Select colour for the pixels that diverge to infinity
					
					switch(s.colour_scheme)
					{
					case CS_HSV_VAR_H:
						hsv_to_rgb(i / s.max_i * 360.0, 50.0, 75.0, &r, &g, &b);
						break;
					case CS_HSV_VAR_SV:
						hsv_to_rgb(360.0, i / s.max_i *100.0, i / s.max_i *100.0, &r, &g, &b);
						break;
					case CS_SHADES_GREY:
						r = g = b = int(i / s.max_i * 255);
					case CS_SHADES_BROWN:
						r = int(i / s.max_i * 255);
						b = int(i / s.max_i * 64);
						g = int(i / s.max_i * 128);
						break;
					case CS_SHADES_BLUE:
						r = g = 0;
						b = int(i / s.max_i * 255);
						break;
					case CS_BW:
						r = g = b = 255;
						break;
					default:
						g = b = 0;
						r = 255;
						break;
					}

					putpixel(bmp, int(ix), int(iy),	makecol(r, g, b));
					break;
				} else if(i == s.max_i - 1)
					putpixel(bmp, int(ix), int(iy), COL_BLACK);
			}
		}
	}
}


int d_my_button_proc(int msg, DIALOG *d, int c)
{
	d_button_proc(msg, d, c);
	
	if(msg == MSG_CLICK && (d->flags & D_SELECTED))
	{
		d->flags = 0;
		scare_mouse();
		d_button_proc(MSG_DRAW, d, c);
		unscare_mouse();

		*static_cast<GuiAction*>(d->dp2) = static_cast<GuiAction>(d->d1);
	}
	
	return D_O_K;
}


int d_preset_list_proc(int msg, DIALOG *d, int c)
{
	int rv = d_list_proc(msg, d, c);
	
	if(msg == MSG_CLICK)
		*static_cast<GuiAction*>(d->dp3) = GUI_ACT_PRESET;
	
	return rv;
}


char* colour_options_getter(int index, int *list_size)
{
	static char *strings[N_COLOURSCHEMES];
	
	strings[CS_HSV_VAR_H]		= "HSV: Variable H";
	strings[CS_HSV_VAR_SV]		= "HSV: Variable S & V";
	strings[CS_SHADES_GREY]		= "Shades of grey";
	strings[CS_BW]			= "Black & white";
	strings[CS_SHADES_BROWN]	= "Shades of brown";
	strings[CS_SHADES_BLUE]		= "Shades of blue";
	
	if(index < 0)
	{
		*list_size = N_COLOURSCHEMES;
		return 0;
	} else
		return strings[index]; 
}


char* presets_getter(int index, int *list_size)
{
	static char *strings[N_GUI_PRESETS];
	
	strings[GUI_PRE_4_06_055I]	= "a = 4, c = 0.6 + 0.55i";
	strings[GUI_PRE_2_M1]		= "a = 2, c = -1";
	strings[GUI_PRE_5_08_06I]	= "a = 5, c = 0.8 + 0.6i";
	strings[GUI_PRE_2_03_06I]	= "a = 2, c = 0.3 + 0.6i";
		
	if(index < 0)
	{
		*list_size = N_GUI_PRESETS;
		return 0;
	} else
		return strings[index];
}


void redraw_dialog(DIALOG *dialog, DIALOG_PLAYER *player)
{
	shutdown_dialog(player);
	player = init_dialog(dialog, -1);
	update_dialog(player);
}


int main(int argc, char **argv)
{
	// Initialize fractal properties
	
	set_properties_t	s;
	Domain			default_domain_x(-1, 1);
	Domain			default_domain_y(-1, 1);

	s.domain_x	= default_domain_x;
	s.domain_y	= default_domain_y;
	s.max_i		= 60;
	s.infinity	= 4;
	s.c		= complex_t(0.6, 0.55);
	s.a		= 4;
	s.colour_scheme	= CS_HSV_VAR_H;
	
	// Initialize graphics
	
	BITMAP		*buffer;
	BITMAP		*fractal;
	BITMAP		*logo;
	
	allegro_init();
	install_keyboard();
	install_mouse();
	
	set_color_depth(16);
	set_gfx_mode(GFX_AUTODETECT_WINDOWED, screen_width, screen_height, 0, 0);
	clear_to_color(screen, COL_WHITE);
	
	fixup_datafile(logo_data);
	
	logo = static_cast<BITMAP*>(logo_data[LOGO_LOGO].dat);
	buffer = create_bitmap(fractal_width, fractal_height);
	fractal = create_bitmap(fractal_width, fractal_height);
	
	clear_to_color(screen, COL_GREY);
	clear_to_color(buffer, COL_GREY);
	clear_to_color(fractal, COL_GREY);
	
	rect(fractal, 0, 0, fractal_width - 1, fractal_height - 1, COL_WHITE);
		
	show_mouse(screen);
	
	// GUI stuff
	
	const char	default_notice[] = "Kip-Fractal beta-1";
	
	char		gui_a[8];
	char		gui_c_real[8];
	char		gui_c_imag[8];
	char		gui_infinity[8];
	char		gui_xmin[12], gui_xmax[12], gui_ymin[12], gui_ymax[12];
	char		gui_max_i[8];
	char		gui_cur_domain_x[128];
	char		gui_cur_domain_y[128];
	char		gui_notice[128];
	
	GuiAction	gui_action = GUI_ACT_NONE;
	GuiPreset	gui_selected_preset = GUI_PRE_4_06_055I;
	
	std::snprintf(gui_xmin, 12, "%f", s.domain_x.a());
	std::snprintf(gui_xmax, 12, "%f", s.domain_x.b());
	std::snprintf(gui_ymin, 12, "%f", s.domain_y.a());
	std::snprintf(gui_ymax, 12, "%f", s.domain_y.b());
	std::snprintf(gui_max_i, 6, "%i", int(s.max_i));	
	std::snprintf(gui_a, 6, "%.3f", s.a);
	std::snprintf(gui_c_real, 6, "%.3f", s.c.real());
	std::snprintf(gui_c_imag, 6, "%.3f", s.c.imag());
	std::snprintf(gui_infinity, 6, "%.3f", s.infinity);
	std::snprintf(gui_cur_domain_x, 128, "x:[%f, %f]", s.domain_x.a(), s.domain_x.b());
	std::snprintf(gui_cur_domain_y, 128, "y:[%f, %f]", s.domain_y.a(), s.domain_y.b());
	
	std::strncpy(gui_notice, default_notice, 128);
	
	DIALOG main_dialog[] = {
	  	/* (dialog proc) (x) (y) (w) (h) (fg) (bg) (key) (flags) (d1) (d2) (dp) (dp2) (dp3) */
		{d_shadow_box_proc, 10, 10, 200, 447, COL_BLACK, COL_WHITE, 0, 0, 0, 0, 0, 0, 0},
		{d_shadow_box_proc, 10, 467, 200, 81, COL_BLACK, COL_WHITE, 0, 0, 0, 0, 0, 0, 0},
		
		{d_text_proc, 45, 20, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"= KIP-FRACTAL =", 0, 0},
		{d_text_proc, 26, 35, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"A Julia set explorer", 0, 0},
		
		{d_text_proc, 34, 60, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"z(n+1) = z(n)^a + c", 0, 0},
		{d_text_proc, 15, 85, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"a = ", 0, 0},
		{d_text_proc, 15, 100, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"c =         +        i", 0, 0},
		{d_text_proc, 15, 115, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"Bailout value = ", 0, 0},
		{d_text_proc, 15, 130, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"Max. iterations = ", 0, 0},
		{d_text_proc, 82, 147, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"Presets", 0, 0},
		{d_text_proc, 42, 214, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"Colouring schemes", 0, 0},
		{d_text_proc, 48, 286, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"Current domain", 0, 0},
		{d_text_proc, 82, 346, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"Actions", 0, 0},
		
		{d_edit_proc, 50, 85, 60, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 6, 0, gui_a, 0, 0},
		{d_edit_proc, 50, 100, 60, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 6, 0, gui_c_real, 0, 0},
		{d_edit_proc, 120, 100, 60, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 6, 0, gui_c_imag, 0, 0},
		{d_edit_proc, 140, 115, 60, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 6, 0, gui_infinity, 0, 0},
		{d_edit_proc, 159, 130, 42, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 6, 0, gui_max_i, 0, 0},
		
		{d_my_button_proc, 15, 361, 190, 20, COL_BLACK, -1, 0, 0, GUI_ACT_RESTORE_DOMAIN, 0, (void*)"Restore domain", 
			&gui_action, 0},
 		{d_my_button_proc, 15, 383, 190, 20, COL_BLACK, -1, 0, 0, GUI_ACT_SET_DOMAIN, 0, (void*)"Set default domain...",
 			&gui_action, 0},
 		{d_my_button_proc, 15, 405, 190, 20, COL_BLACK, -1, 0, 0, GUI_ACT_REGENERATE, 0, (void*)"(Re)generate", 
 			&gui_action, 0},
 		{d_my_button_proc, 15, 427, 190, 20, COL_BLACK, -1, 0, 0, GUI_ACT_EXIT, 0, (void*)"Exit", &gui_action, 0},
  				
 		{d_preset_list_proc, 15, 162, 190, 42, COL_BLACK, COL_WHITE, 0, 0, 0, 0, (void*)presets_getter,0, &gui_action},	
 		{d_list_proc, 15, 229, 190, 48, COL_BLACK, COL_WHITE, 0, 0, CS_HSV_VAR_H, 0, (void*)colour_options_getter, 0, 0},
 		
 		{d_bitmap_proc, 11, 468, 197, 78, 0, 0, 0, 0, 0, 0, (void*)logo, 0, 0},

		{d_yield_proc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
		{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
	};
	
	
	DIALOG status_dialog[] = {
		{d_shadow_box_proc, 10, 558, 200, 22, COL_BLACK, COL_WHITE, 0, 0, 0, 0, 0, 0, 0},
		{d_box_proc, 15, 306, 190, 25, COL_WHITE, COL_WHITE, 0, 0, 0, 0, 0, 0, 0},
		
		{d_text_proc, 15, 306, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)gui_cur_domain_x, 0, 0},
 		{d_text_proc, 15, 321, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)gui_cur_domain_y, 0, 0},
		{d_text_proc, 15, 565, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)gui_notice, 0, 0},
			
		{d_yield_proc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
		{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
	};
	
	
	DIALOG set_domain_dialog[] = {
		{d_shadow_box_proc, 215, 383, 240, 74, COL_BLACK, COL_WHITE, 0, 0, 0, 0, 0, 0},

		{d_text_proc, 220, 393, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"x: [           ,           ]", 0, 0},
		{d_text_proc, 220, 408, 0, 0, COL_BLACK, -1, 0, 0, 0, 0, (void*)"y: [           ,           ]", 0, 0},
		
		{d_edit_proc, 251, 393, 88, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 10, 0, gui_xmin, 0, 0},
		{d_edit_proc, 348, 393, 88, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 10, 0, gui_xmax, 0, 0},
		{d_edit_proc, 251, 408, 88, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 10, 0, gui_ymin, 0, 0},
		{d_edit_proc, 348, 408, 88, 12, COL_BLACK, COL_LIGHTGREY, 0, 0, 10, 0, gui_ymax, 0, 0},
		
		{d_button_proc, 220, 427, 114, 20, COL_BLACK, -1, 0, D_EXIT, 0, 0, (void*)"Ok", 0, 0},
		{d_button_proc, 336, 427, 114, 20, COL_BLACK, -1, 0, D_EXIT, 0, 0, (void*)"Cancel", 0, 0},
		
		{d_yield_proc, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
		{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}		
	};
	
	// Main loop
	
	bool		need_redraw = true;
	bool		need_regen = false;
	bool		need_dialog_redraw = true;
	bool		stop = false;
	
	bool		dragging = false;
	double		drag_x_start, drag_x_current;
	double		drag_y_start, drag_y_current;
	
	DIALOG_PLAYER	*main_dialog_player = init_dialog(main_dialog, -1);
	DIALOG_PLAYER	*status_dialog_player = init_dialog(status_dialog, -1);
	
	update_dialog(status_dialog_player);
	
	while(/* !key[KEY_ESC] &&*/ !stop)
	{
		update_dialog(main_dialog_player);
			
		switch(gui_action)
		{
		case GUI_ACT_NONE:
			break;
		case GUI_ACT_RESTORE_DOMAIN:
			s.domain_x = default_domain_x;
			s.domain_y = default_domain_y;
			
			std::snprintf(gui_cur_domain_x, 128, "x:[%f, %f]", s.domain_x.a(), s.domain_x.b());
			std::snprintf(gui_cur_domain_y, 128, "y:[%f, %f]", s.domain_y.a(), s.domain_y.b());
			
			gui_action = GUI_ACT_NONE;
			need_dialog_redraw = true;
			break;
		case GUI_ACT_REGENERATE:
			s.a = std::atof(gui_a);
			s.c = complex_t(std::atof(gui_c_real), std::atof(gui_c_imag));
			s.infinity = std::atof(gui_infinity);
			s.max_i = std::atof(gui_max_i);
			
			// Read selected colour scheme from dialog
			s.colour_scheme = static_cast<ColourScheme>(main_dialog[23].d1);	
				
			gui_action = GUI_ACT_NONE;
			need_regen = true;
			break;
		case GUI_ACT_SET_DOMAIN:
			if(popup_dialog(set_domain_dialog, -1) == 7)
			{
				default_domain_x.set(std::atof(gui_xmin), std::atof(gui_xmax));
				default_domain_y.set(std::atof(gui_ymin), std::atof(gui_ymax));
			}			
			
			gui_action = GUI_ACT_RESTORE_DOMAIN;
			break;
		case GUI_ACT_PRESET:
			if(gui_selected_preset != static_cast<GuiPreset>(main_dialog[22].d1))
			{
				gui_selected_preset = static_cast<GuiPreset>(main_dialog[22].d1);
				
				switch(gui_selected_preset)
				{
				case GUI_PRE_4_06_055I:
					std::strcpy(gui_a, "4");
					std::strcpy(gui_c_real, "0.6");
					std::strcpy(gui_c_imag, "0.55");
					break;
				case GUI_PRE_2_M1:
					std::strcpy(gui_a, "2");
					std::strcpy(gui_c_real, "-1");
					std::strcpy(gui_c_imag, "0");
					break;
				case GUI_PRE_5_08_06I:
					std::strcpy(gui_a, "5");
					std::strcpy(gui_c_real, "0.8");
					std::strcpy(gui_c_imag, "0.6");
					break;
				case GUI_PRE_2_03_06I:
					std::strcpy(gui_a, "2");
					std::strcpy(gui_c_real, "0.3");
					std::strcpy(gui_c_imag, "0.6");
					break;
				default:
					break;
				}
				
				// Update gui_a, gui_c_real and gui_c_imag input boxes
				main_dialog[13].proc(MSG_DRAW, &main_dialog[12], 0);
				main_dialog[14].proc(MSG_DRAW, &main_dialog[13], 0);
				main_dialog[15].proc(MSG_DRAW, &main_dialog[14], 0);
			}
			
			gui_action = GUI_ACT_NONE;
			break;
		default:
			stop = true;
		}
						
		if(need_dialog_redraw)
		{
			redraw_dialog(status_dialog, status_dialog_player);
			need_dialog_redraw = false;
		}
			
		if(need_regen)
		{
			scare_mouse();
			rectfill(screen, offset_left, offset_top, screen_width - offset_right, screen_height - offset_bottom,
				COL_GREY);
			rect(screen, offset_left, offset_top, screen_width - offset_right - 1, screen_height - offset_bottom -1,
				COL_WHITE);
			unscare_mouse();
			
			std::snprintf(gui_cur_domain_x, 128, "x:[%f, %f]", s.domain_x.a(), s.domain_x.b());
			std::snprintf(gui_cur_domain_y, 128, "y:[%f, %f]", s.domain_y.a(), s.domain_y.b());
			std::sprintf(gui_notice, "Generating the fractal");
			
			redraw_dialog(status_dialog, status_dialog_player);
			generate_set(fractal, s);
			
			std::sprintf(gui_notice, default_notice);
			
			need_regen = false;
			need_redraw = true;
			need_dialog_redraw = true;
		}
		
		if(need_redraw)
		{
			clear_to_color(buffer, COL_GREY);
			draw_sprite(buffer, fractal, 0, 0);
			
			if(dragging)
				rect(buffer, int(drag_x_start), int(drag_y_start), int(drag_x_current), int(drag_y_current),
					COL_WHITE);
			
			vsync();
			scare_mouse();
			draw_sprite(screen, buffer, offset_left, offset_top);
			unscare_mouse();
			
			need_redraw = false;
		}
		
		if(mouse_b & 1 && !dragging && mouse_x >= offset_left && mouse_x <= screen_width - offset_right && 
			mouse_y >= offset_top && mouse_y <= screen_height - offset_bottom)
		{
			drag_x_start = drag_x_current = mouse_x - offset_left;
			drag_y_start = drag_y_current = mouse_y - offset_top;
			
			std::sprintf(gui_notice, "Rightclick to cancel");
			
			set_mouse_range(offset_left, offset_top, screen_width - offset_right, screen_height - offset_bottom);
			
			dragging = true;
			need_dialog_redraw = true;
		} else if((mouse_b & 1) && dragging)
		{
			if(mouse_x != int(drag_x_current) && mouse_y != int(drag_y_current))
			{
// 				if(drag_x_current < 0)
// 					drag_x_current = 0;
// 				else if(drag_x_current > fractal_width)
// 					drag_x_current = fractal_width;
// 				
// 				if(drag_y_current < 0)
// 					drag_y_current = 0;
// 				else if(drag_y_current > fractal_height)
// 					drag_y_current = fractal_height;

				drag_x_current = mouse_x - offset_left;
				drag_y_current = mouse_y - offset_top;
				
				if(std::abs(drag_x_start - drag_x_current) > std::abs(drag_y_start - drag_y_current))
					drag_y_current = (drag_y_current > drag_y_start) ? drag_y_start + 
						std::abs(drag_x_start - drag_x_current) : 
						drag_y_start - std::abs(drag_x_start - drag_x_current);
				else
					drag_x_current = (drag_x_current > drag_x_start) ? drag_x_start +
						std::abs(drag_y_start - drag_y_current) :
						drag_x_start - std::abs(drag_y_start - drag_y_current);
						
//				position_mouse(drag_x_current + offset_left, drag_y_current + offset_top);

				need_redraw = true;
			}
		} else if((mouse_b & 2) && dragging)
		{
			std::sprintf(gui_notice, default_notice);
			
			set_mouse_range(0, 0, screen_width-1, screen_height-1);
			
			dragging = false;
			need_dialog_redraw = true;
			need_redraw = true;
		} else if(dragging)
		{
			dragging = false;
			
			double	stepsize_x = (s.domain_x.b() - s.domain_x.a()) / fractal_width;
			double	stepsize_y = (s.domain_y.a() - s.domain_y.b()) / fractal_height;
			
			drag_x_start = (drag_x_start * stepsize_x + s.domain_x.a());
			drag_x_current = (drag_x_current * stepsize_x + s.domain_x.a());
			drag_y_start = (drag_y_start * stepsize_y + s.domain_y.b());			
			drag_y_current = (drag_y_current * stepsize_y + s.domain_y.b());

			s.domain_x.set(drag_x_start, drag_x_current);
			s.domain_y.set(drag_y_start, drag_y_current);
			
			std::sprintf(gui_notice, default_notice);
			
			set_mouse_range(0, 0, screen_width-1, screen_height-1);
			
			need_dialog_redraw = true;
			need_regen = true;
		}
	}
	
	shutdown_dialog(main_dialog_player);
	shutdown_dialog(status_dialog_player);
					
	return 0;
}
END_OF_MAIN();
