/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__WRAPPER__
#define EME__WRAPPER__
/*----------------------------------------------------------------------------
  Wrapper class for creating and accessing a Map

- Constructor/destructor
- Misc
- Template access to tiles and properties
- Composite::Property
- Out of order access
- Raw access to internals (unless you know what you are doing, you should not
  use them)
----------------------------------------------------------------------------*/


#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "mapdlg.h" /* For MAP_SHAPE and TILE_SHAPE */

#include "prop.h"

#include "tiles.h"
#include "creator.h"
#include "ccomposi.h"

#include "template.h" // For casts

#include "emepopup.h"
#include <typeinfo>
#include "config.inc"
#include <vector>

typedef enum {
  FULL_LAYER,
  SPARSE_LAYER,
  LAYER_TYPE_COUNT
} LAYER_TYPE;


class Wrapper;
class Tiles;


class CompositeHandle {
  Composite::Creator *cc;
  CompositeHandle(Composite::Creator *cc): cc(cc)
    {}
  friend class Wrapper;
};


class Wrapper {

public:
  /* Iterators could be simple int, but if they are a struct, the user
   * won't try to do ++ on them */
  struct iterator_ {
    int it;
    iterator_(int i): it(i) {}
    iterator_ &operator=(const iterator_ &other) {
      it = other.it;
      return *this;
    }
    friend bool operator==(const iterator_&, const iterator_&);
  };
  typedef iterator_ iterator;
  typedef iterator_ const_iterator;

  /* Constructor/Destructor
     ---------------------------------*/
  /* Wrapper constructor for creating a map */
  Wrapper(
    int width, int height,             /* Size (in tiles) of the created map */
    int num_layers,
    int num_map_props
  );

  Wrapper(
    int width, int height              /* Size (in tiles) of the created map */
  );

  /* Wrapper constructor for accessing a map values */
  Wrapper(const Map *map);

  /* Wrapper constructor for modifying a map */
  Wrapper(Map *map);

  ~Wrapper(void);

  void SetNumLayers(int layer_count);
  void SetNumProperties(int property_count);


  /* Misc
     ---------------------------------*/
  /* Sets the map shape */
  void SetMapShape(MAP_SHAPE shape);

  /* Sets the tile shape */
  void SetTileShape(TILE_SHAPE shape, int w, int h);
  void SetTileOffset(int dx, int dy, ODD odd=ODD_NONE);


  int GetMapWidth(void) const;
  int GetMapHeight(void) const;


  /* Tests if a tile exists in a Layer */
  inline bool TileExists(int l, int i, int j) const
    {return RawGetTileProperty(l, i, j) != 0;}

  bool IsIn(int i, int j) const;

  /* Tells the program the map creation has ended */
  void SetMap(void);

  /* Returns the index of the layer with name 'name',
   * or the number of layers if not found */
  int FindLayerIndex(const char *name) const;

  template <typename Type>
  bool LayerIs(int l) const;


  /* User data
     ---------------------------------*/
  void SetUserdata(void *userdata, void (*destroyer)(void *userdata)=0);
  void *GetUserdata() const;


  /* Template access to tiles and properties
     ---------------------------------*/
  /* Sets a map property */
  template <typename Type>
  void SetProperty(int n, const char *name, typename Type::Base value);

  /* Sets a map property */
  template <typename Type>
  void SetProperty(
    int n, const char *name, typename Type::Base value,
    typename Type::Param param
  );

  /* Sets a layer type */
  template <typename Type> void SetLayer(
    int l, const char *name, LAYER_TYPE ltype, int i, int j, int w, int h,
    typename Type::Base default_value
  );

  template <typename Type> void SetLayer(
    int l, const char *name, LAYER_TYPE ltype, int i, int j, int w, int h,
    typename Type::Base default_value, typename Type::Param param
  );

  template <typename Type> void SetLayer(
    int l, const char *name, LAYER_TYPE ltype,
    typename Type::Base default_value
  ) { SetLayer<Type>(l, name, ltype, 0, 0, GetMapWidth(), GetMapHeight(), default_value); }

  template <typename Type> void SetLayer(
    int l, const char *name, LAYER_TYPE ltype,
    typename Type::Base default_value, typename Type::Param param
  ) { SetLayer<Type>(l, name, ltype, 0, 0, GetMapWidth(), GetMapHeight(), default_value, param); }

  /* Fill the layer with the default tile if it is a FULL_LAYER, else
     do nothing */
  void FillLayer(int l);

  /* Sets a tile in a layer */
  template <typename Type>
  void SetTileValue(int l, int i, int j, typename Type::Base value);



  /* Composite::Property
     ---------------------------------*/
  /* Creates a composite, either for a map prop or for a layer */
  CompositeHandle CreateComposite(const char *name, int num_properties);

  /* Sets a composite map property: Must be called *after* all the 'Add...' */
  void SetCompositeProperty(int n, CompositeHandle handle);

  /* Sets a composite layer: Must be called *after* all the 'Add...' */
  void SetCompositeLayer(
    int l, LAYER_TYPE ltype, CompositeHandle handle, int i, int j, int w, int h
  );

  void SetCompositeLayer(
    int l, LAYER_TYPE ltype, CompositeHandle handle
  ) { SetCompositeLayer(l, ltype, handle, 0, 0, GetMapWidth(), GetMapHeight()); }

  /* Creates a composite tile */
  Composite::Property *SetCompositeTile(int l, int i, int j);

  /* Sets the value (or default value) of a part of a composite map property
     (or of a composite tile) */
  template <typename Type>
  void Add(
    CompositeHandle handle, int index, const char *name,
    typename Type::Base default_value
  );

  /* Sets the value (or default value) of a part of a composite map property
     (or of a composite tile) */
  template <typename Type>
  void Add(
    CompositeHandle handle, int index, const char *name,
    typename Type::Base default_value, typename Type::Param param
  );

  /* Set the value of a part of a composite tile */
  template <typename Type>
  void AddToTile(
    Composite::Property *prop, int index, typename Type::Base value
  );

  const Composite::Property *GetCompositeProperty(int n) const;
  Composite::Property *GetCompositeProperty(int n);

  const Composite::Property *GetCompositeTile(int l, int i, int j) const;
  Composite::Property *GetCompositeTile(int l, int i, int j);

  template <typename Type>
  typename Type::Base Get(const Composite::Property *prop, int index) const;


  /* Map property access
     ---------------------------------*/
  /* Get a map property value */
  template <typename Type>
  typename Type::Base GetProperty(int index) const;


  /* Ordered access
     ---------------------------------*/
  int BeginI(int l) const
    { return GetLayer(l)->begini(); }
  int BeginJ(int l) const
    { return GetLayer(l)->beginj(); }
  int EndI(int l) const
    { return GetLayer(l)->endi(); }
  int EndJ(int l) const
    { return GetLayer(l)->endj(); }

  /* Get a tile value */
  template <typename Type>
  const typename Type::Base GetTileValue(int l, int i, int j) const;

  template <typename Type>
  typename Type::Base GetTileValue(int l, int i, int j);


  /* Out of order access
     ---------------------------------*/
  int Count(int l) const;

  /* Note that these are now *very* slow */
  const_iterator Begin(int l) const;
  const_iterator End(int l) const;
  const_iterator Next(int l, const_iterator id) const;

  iterator Begin(int l);
  iterator End(int l);
  iterator Next(int l, iterator id);

  template <typename Type>
  const typename Type::Base GetTileValue(int l, const_iterator id) const;

  template <typename Type>
  typename Type::Base GetTileValue(int l, iterator id);

  template <typename Type>
  const typename Type::Property *GetTileProperty(
    int l, const_iterator id
  ) const;

  template <typename Type>
  typename Type::Property *GetTileProperty(int l, iterator id);

  int GetTileI(int l, const_iterator id) const;
  int GetTileJ(int l, const_iterator id) const;
  const BaseProperty *RawGetTileProperty(int l, const_iterator id) const;
  BaseProperty *RawGetTileProperty(int l, iterator id);


  /* Raw access to internals
     ---------------------------------*/
  const Map *GetMap() const { return const_map_; }
  Map *GetMap() {
    USER_CHECK(
      if (!map_) popup_quit(UString("Error"), UString("Trying to modify a const map"), 0);
    );
    return map_;
  }

  int GetNumProperties(void) const;
  int GetNumLayers(void) const;

  const class ViewedLayers &GetViewedLayers() const;

  const class SelectedTiles *GetSelectedTiles() const;

  void Execute(class Command *cmd);
  void ClearUndoStack();

  const class TilesStack *GetLayers() const;
  class TilesStack *GetLayers();

  const Tiles *GetLayer(int l) const;
  Tiles *GetLayer(int l);

  template <typename Type>
  const typename Type::Creator *GetLayerCreator(int l) const;

  const char *GetLayerName(int l) const;
  const BaseProperty *GetLayerReference(int l) const;
  int GetActiveLayerIndex() const;

  const BaseCreator *RawGetCreator(int l) const;

  template <typename Type>
  const typename Type::Property *GetTileProperty(int l, int i, int j) const;

  template <typename Type>
  typename Type::Property *GetTileProperty(int l, int i, int j);

  void RawSetProperty(int n, BaseCreator *creator);

  void RawSetLayer(int l, LAYER_TYPE ltype, BaseCreator *creator, int i, int j, int w, int h);
  void RawSetLayer(int l, LAYER_TYPE ltype, BaseCreator *creator)
    { RawSetLayer(l, ltype, creator, 0, 0, GetMapWidth(), GetMapHeight()); }

  void RawSetTileProperty(int l, int i, int j, BaseProperty *prop);

  const BaseProperty *RawGetProperty(int n) const;

  const BaseProperty *RawGetTileProperty(int l, int i, int j) const;
  BaseProperty *RawGetTileProperty(int l, int i, int j);

  inline void RawAdd(CompositeHandle handle, int n, BaseCreator *creator)
    {handle.cc->Add(n, creator);}


private:
  Wrapper(const Wrapper&);
  Wrapper &operator=(const Wrapper&);

  void SetLayerFullness(LAYER_TYPE type, int where);
  /* Helpers to create error messages */
  const char *HackStrLayer(int index) const;
  const char *HackStrLayerAt(int index, int x, int y) const;
  const char *HackStrProperty(int index) const;

  Map *map_;
  const Map *const_map_;
  int w_, h_;
  std::vector<LAYER_TYPE> layers_fullness_;
  bool owns_map_;
  mutable char hackstr[256*6]; /* Used as a tmp string for error messages */
};


inline bool operator==(const Wrapper::iterator_ &i1, const Wrapper::iterator_ &i2)
{
  return i1.it == i2.it;
}



/*------------------------------------------------------------------------------
  Templates
------------------------------------------------------------------------------*/
#include <typeinfo>


#ifdef USER_DEBUG

#define TEST_LAYER_INDEX_POSITIVE(index, func_name, layer_name)             \
  if(index<0) {                                                             \
    popup_quit(func_name, Translation("Negative layer index"), layer_name); \
  }

#define TEST_LAYER_INDEX(index, func_name, layer_name)                      \
  TEST_LAYER_INDEX_POSITIVE(index, func_name, layer_name);                  \
  if(index>=GetNumLayers()) {                                               \
    popup_quit(func_name, Translation("Layer index too high"), layer_name); \
  }

#define TEST_MAP_PROP_INDEX(index, func_name, prop_name)                     \
  if(index<0) {                                                              \
    popup_quit(                                                              \
      func_name, Translation("Negative map property index"), prop_name);     \
  }                                                                          \
  if(index>=GetNumProperties()) {                                            \
    popup_quit(                                                              \
      func_name, Translation("Map property index too high"), prop_name);     \
  }

#define TEST_TYPE(type, object, func_name, name)           \
  if(!(typeid(type)==typeid(object))) {                    \
    popup_quit(func_name, Translation("Bad type"), name);  \
  }

#define TEST_EXISTS(object, func_name, name)                    \
  if(!object) {                                                 \
    popup_quit(func_name, Translation("Does not exist"), name); \
  }

#else /* !USER_DEBUG */

#define TEST_LAYER_INDEX_POSITIVE(index, func_name, layer_name)
#define TEST_LAYER_INDEX(index, func_name, layer_name)
#define TEST_MAP_PROP_INDEX(index, func_name, prop_name)
#define TEST_TYPE(type, object, func_name, name)
#define TEST_EXISTS(object, func_name, name)

#endif /* !USER_DEBUG */


template <typename Type>
const typename Type::Base Wrapper::GetTileValue(
  int layer, const_iterator id
) const
{
  TEST_LAYER_INDEX(layer, UString("GetTileValue"), HackStrLayer(layer));
  USER_CHECK(
    if(!GetLayer(layer)) {
      popup_quit(
        UString("GetTileValue"), Translation("Layer is not set"),
        HackStrLayer(layer)
      );
    }
  );
  return GetTileProperty<Type>(layer, id)->Get();
}


template <typename Type>
typename Type::Base Wrapper::GetTileValue(int layer, iterator id)
{
  TEST_LAYER_INDEX(layer, UString("GetTileValue"), HackStrLayer(layer));
  USER_CHECK(
    if(!GetLayer(layer)) {
      popup_quit(
        UString("GetTileValue"), Translation("Layer is not set"),
        HackStrLayer(layer)
      );
    }
  );
  return GetTileProperty<Type>(layer, id)->Get();
}


template <typename Type>
const typename Type::Property *Wrapper::GetTileProperty(
  int layer, const_iterator id
) const
{
  typedef typename Type::Property PropertyType;
  TEST_LAYER_INDEX(layer, UString("GetTileProperty"), HackStrLayer(layer));
  USER_CHECK(
    if(!GetLayer(layer)) {
      popup_quit(
        UString("GetTileProperty"), Translation("Layer is not set"),
        HackStrLayer(layer)
      );
    }
  );
  const BaseProperty *p = RawGetTileProperty(layer, id);
  TEST_TYPE(PropertyType, *p, UString("GetTileProperty"),HackStrLayer(layer));
  return Cast<Type>(p);
}


template <typename Type>
typename Type::Property *Wrapper::GetTileProperty(int layer, iterator id)
{
  typedef typename Type::Property PropertyType;
  TEST_LAYER_INDEX(layer, UString("GetTileProperty"), HackStrLayer(layer));
  USER_CHECK(
    if(!GetLayer(layer)) {
      popup_quit(
        UString("GetTileProperty"), Translation("Layer is not set"),
        HackStrLayer(layer)
      );
    }
  );
  BaseProperty *p = RawGetTileProperty(layer, id);
  TEST_TYPE(PropertyType, *p, UString("GetTileProperty"),HackStrLayer(layer));
  return Cast<Type>(p);
}


template <typename Type>
const typename Type::Creator *Wrapper::GetLayerCreator(int n) const
{
  typedef typename Type::Creator CreatorType;
  TEST_LAYER_INDEX(n, UString("GetLayerCreator"), HackStrLayer(n));
  USER_CHECK(
    if(!GetLayer(n)) {
      popup_quit(
        UString("GetLayerCreator"), Translation("Layer is not set"),
        HackStrLayer(n)
      );
    }
  );
  const BaseCreator *creator = RawGetCreator(n);
  TEST_TYPE(
    CreatorType, *creator, UString("GetLayerCreator"), HackStrLayer(n)
  );
  return Cast<Type>(creator);
}


template <typename Type>
const typename Type::Property *Wrapper::GetTileProperty(
  int n, int x, int y
) const
{
  typedef typename Type::Property PropertyType;
  TEST_LAYER_INDEX(n, UString("GetTileProperty"), HackStrLayer(n));
  USER_CHECK(
    if(!GetLayer(n)) {
      popup_quit(
        UString("GetTileProperty"), Translation("Layer is not set"),
        HackStrLayer(n)
      );
    }
  );
  const BaseProperty *prop = RawGetTileProperty(n, x, y);
  TEST_TYPE(
    PropertyType, *prop, UString("GetTileProperty"), HackStrLayerAt(n, x, y)
  );
  return Cast<Type>(prop);
}


template <typename Type>
typename Type::Property *Wrapper::GetTileProperty(int n, int x, int y)
{
  typedef typename Type::Property PropertyType;
  TEST_LAYER_INDEX(n, UString("GetTileProperty"), HackStrLayer(n));
  USER_CHECK(
    if(!GetLayer(n)) {
      popup_quit(
        UString("GetTileProperty"), Translation("Layer is not set"),
        HackStrLayer(n)
      );
    }
  );
  BaseProperty *prop = RawGetTileProperty(n, x, y);
  TEST_TYPE(
    PropertyType, *prop, UString("GetTileProperty"), HackStrLayerAt(n, x, y)
  );
  return Cast<Type>(prop);
}


template <typename Type>
void Wrapper::SetProperty(
  int index, const char *name, typename Type::Base value
)
{
  typedef typename Type::Creator CreatorType;
  TEST_MAP_PROP_INDEX(index, UString("SetProperty"), name);
  CreatorType *creator;
  creator = new CreatorType(new StaticString(name), value);
  RawSetProperty(index, creator);
}


template <typename Type>
void Wrapper::SetProperty(
  int index, const char *name, typename Type::Base value,
  typename Type::Param param
)
{
  typedef typename Type::Creator CreatorType;
  TEST_MAP_PROP_INDEX(index, UString("SetProperty"), name);
  CreatorType *creator;
  creator = new CreatorType(new StaticString(name), param, value);
  RawSetProperty(index, creator);
}


template <typename Type>
void Wrapper::SetLayer(
  int index, const char *name, LAYER_TYPE ltype, int x, int y, int w, int h,
  typename Type::Base default_value
)
{
  typedef typename Type::Creator CreatorType;
  TEST_LAYER_INDEX(index, UString("SetLayer"), name);
  CreatorType *creator;
  creator = new CreatorType(new StaticString(name), default_value);
  RawSetLayer(index, ltype, creator, x, y, w, h);
}


template <typename Type>
void Wrapper::SetLayer(
  int index, const char *name, LAYER_TYPE ltype, int x, int y, int w, int h,
  typename Type::Base default_value, typename Type::Param param
)
{
  typedef typename Type::Creator CreatorType;
  TEST_LAYER_INDEX(index, UString("SetLayer"), name);
  CreatorType *creator;
  creator = new CreatorType(new StaticString(name), param, default_value);
  RawSetLayer(index, ltype, creator, x, y, w, h);
}


template <typename Type>
void Wrapper::SetTileValue(int index, int x, int y, typename Type::Base value)
{
  typedef typename Type::Creator CreatorType;
  typedef typename Type::Property PropertyType;
  Tiles *layer = GetLayer(index);
  USER_CHECK(
    if(!layer) {
      popup_quit(
        UString("SetTileValue"), Translation("Layer is not set"),
        HackStrLayer(index)
      );
    }
  );
  const BaseCreator *creator = layer->creator();
  TEST_TYPE(
    CreatorType, *creator,
    UString("SetTileValue"), HackStrLayerAt(index, x, y)
  );
  BaseProperty *prop = Cast<Type>(creator)->Create(value);
  RawSetTileProperty(index, x, y, prop);
}


template <typename Type>
typename Type::Base Wrapper::GetProperty(int index) const
{
  typedef typename Type::Property PropertyType;
  const BaseProperty *prop = RawGetProperty(index);
  TEST_TYPE(
    PropertyType, *prop, UString("GetProperty"), HackStrProperty(index)
  );
  return Cast<Type>(prop)->Get();
}

template <typename Type>
const typename Type::Base Wrapper::GetTileValue(int index, int x, int y) const
{
  typedef typename Type::Property PropertyType;
  const BaseProperty *prop = RawGetTileProperty(index, x, y);
  TEST_TYPE(
    PropertyType, *prop, UString("GetTileValue"), HackStrLayerAt(index, x, y)
  );
  return Cast<Type>(prop)->Get();
}


template <typename Type>
typename Type::Base Wrapper::GetTileValue(int index, int x, int y)
{
  typedef typename Type::Property PropertyType;
  BaseProperty *prop = RawGetTileProperty(index, x, y);
  TEST_TYPE(
    PropertyType, *prop, UString("GetTileValue"), HackStrLayerAt(index, x, y)
  );
  return Cast<Type>(prop)->Get();
}



template <typename Type>
void Wrapper::Add(
  CompositeHandle handle, int index, const char *name,
  typename Type::Base default_value
)
{
  typedef typename Type::Creator CreatorType;
  // ...tests
  CreatorType *creator;
  creator = new CreatorType(new StaticString(name), default_value);
  handle.cc->Add(index, creator);
}


template <typename Type>
void Wrapper::Add(
  CompositeHandle handle, int index, const char *name,
  typename Type::Base default_value, typename Type::Param param
)
{
  typedef typename Type::Creator CreatorType;
  // ...tests
  CreatorType *creator;
  creator = new CreatorType(new StaticString(name), param, default_value);
  handle.cc->Add(index, creator);
}


template <typename Type>
void Wrapper::AddToTile(
  Composite::Property *prop, int index, typename Type::Base value
)
{
  //...tests
  Cast<Type>(prop->Get(index))->Set(value);
}


template <typename Type>
typename Type::Base Wrapper::Get(
  const Composite::Property *prop, int index
) const
{
  //...tests
  return Cast<Type>(prop->Get(index))->Get();
}


template <typename Type>
bool Wrapper::LayerIs(int l) const
{
  typedef typename Type::Creator CreatorType;
  const BaseCreator *creator = RawGetCreator(l);
  return typeid(*creator)==typeid(CreatorType);
}

#endif /* EME__WRAPPER__ */

