/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "tiles.h"

#include "prop.h"
#include "creator.h"
#include "size.h"

#include "debug.h"


/*
 * Tiles
 */
Tiles::Tiles(BaseCreator *c, int i, int j, int w, int h):
  size_(i, j, w, h), creator_(c)
{
  DBG_ASSERT(creator_);
}


Tiles::Tiles(BaseCreator *c):
  size_(), creator_(c)
{
  DBG_ASSERT(creator_);
}


Tiles::Tiles(const Tiles &other):
  size_(other.size_), creator_(other.creator_->Clone())
{
  DBG_ASSERT(creator_);
}


Tiles::~Tiles()
{
  delete creator_;
}


Tiles &Tiles::operator=(const Tiles &other)
{
  size_=other.size_;
  delete creator_;
  DBG_ASSERT(other.creator_);
  creator_=other.creator_->Clone();
  return *this;
}


void Tiles::move(int di, int dj)
{
  move_(di, dj);
  size_ = Size(begini()+di, beginj()+dj, width(), height());
}


void Tiles::resize(int i, int j, int w, int h)
{
  int firsti = i;
  int firstj = j;
  int lasti = i+w;
  int lastj = j+h;

  int bi = begini();
  int bj = beginj();
  int ei = endi();
  int ej = endj();

  if (lasti!=ei) {
    int di = ei-lasti;
    if (di<0) insert_col(ei, -di);
    else remove_col(lasti, di);
    move(0, 0);
  }
  if (lastj!=ej) {
    int dj = ej-lastj;
    if (dj<0) insert_row(ej, -dj);
    else remove_row(lastj, dj);
    move(0, 0);
  }
  if (firsti!=bi) {
    int di = firsti-bi;
    if (di<0) insert_col(bi, -di);
    else remove_col(bi, di);
    move(di, 0);
  }
  if (firstj!=bj) {
    int dj = firstj-bj;
    if (dj<0) insert_row(bj, -dj);
    else remove_row(bj, dj);
    move(0, dj);
  }
}


void Tiles::insert_col(int i, int count)
{
  DBG_ASSERT(count>=0);
  if (i<begini()) {
    move_(count, 0);
  }
  else if (i<=endi()) {
    insert_col_(i, count);
    size_ = Size(begini(), beginj(), width()+count, height());
  }
}


void Tiles::insert_row(int j, int count)
{
  DBG_ASSERT(count>=0);
  if (j<beginj()) {
    move_(0, count);
  }
  else if (j<=endj()) {
    insert_row_(j, count);
    size_ = Size(begini(), beginj(), width(), height()+count);
  }
}


void Tiles::remove_col(int i, int count)
{
  DBG_ASSERT(count>=0);
  if (i+count<begini()) {
    move_(-count, 0);
  }
  else if (i<begini()) {
    int move_count = begini()-i;
    move_(-move_count, 0);
    remove_col_(i, count-move_count);
    size_ = Size(begini(), beginj(), width()-count+move_count, height());
  }
  else if (i+count<endi()) {
    remove_col_(i, count);
    size_ = Size(begini(), beginj(), width()-count, height());
  }
  else if (i<endi()) {
    int remove_count = endi()-i;
    remove_col_(i, remove_count);
    size_ = Size(begini(), beginj(), width()-remove_count, height());
    move_(count-remove_count, 0);
  }
}


void Tiles::remove_row(int j, int count)
{
  DBG_ASSERT(count>=0);
  if (j+count<beginj()) {
    move_(0, -count);
  }
  else if (j<beginj()) {
    int move_count = beginj()-j;
    move_(0, -move_count);
    remove_row_(j, count-move_count);
    size_ = Size(begini(), beginj(), width(), height()-count+move_count);
  }
  else if (j+count<endj()) {
    remove_row_(j, count);
    size_ = Size(begini(), beginj(), width(), height()-count);
  }
  else if (j<endj()) {
    int remove_count = endj()-j;
    remove_row_(j, remove_count);
    size_ = Size(begini(), beginj(), width(), height()-remove_count);
    move_(0, count-remove_count);
  }
}

