/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__SPARSE_TILES__
#define EME__SPARSE_TILES__

class BaseProperty;
class BaseCreator;

#include "tiles.h"
#include "prop.h"
#include "config.inc"
#include <map>

class SparseTiles: public Tiles {
  int pack(int i, int j) const { return (j<<16)+i; }
  int unpacki(int packed) const { return packed&0xffff; }
  int unpackj(int packed) const { return (packed>>16)&0xffff; }
public:
  typedef std::pair<int, BaseProperty*> Data;
  typedef std::map<int, BaseProperty*> Imp;
  typedef Imp::iterator iterator;
  typedef Imp::const_iterator const_iterator;

  SparseTiles(BaseCreator *c, int i, int j, int w, int h);
  SparseTiles(BaseCreator *c);
  ~SparseTiles();

  SparseTiles *clone() const;

  SparseTiles &operator=(const SparseTiles &other);

  int count() const { return tiles_.size(); }

  /* SparseTiles are also used themselves */
  const_iterator begin() const { return tiles_.begin(); }
  const_iterator end() const { return tiles_.end(); }
  iterator begin() { return tiles_.begin(); }
  iterator end() { return tiles_.end(); }
  const BaseProperty *get_tile(const_iterator it) const { return (*it).second; }
  BaseProperty *get_tile(iterator it) { return (*it).second; }
  int get_i(const_iterator it) const { return unpacki((*it).first); }
  int get_j(const_iterator it) const { return unpackj((*it).first); }

  /* Deletes the prop at i, j */
  void clear(int i, int j) { BaseProperty *p=clear_(i, j); delete p; }

protected:
  /* Doesn't delete the prop at i, j, but returns it */
  BaseProperty *clear_(int i, int j);
  BaseProperty *get_(int i, int j) const;
  void set_(int i, int j, BaseProperty *p);
  void copy_(int i, int j, const BaseProperty *p);

  void move_(int di, int dj);
  void insert_col_(int i, int count);
  void remove_col_(int i, int count);
  void insert_row_(int j, int count);
  void remove_row_(int j, int count);

private:
  SparseTiles(const SparseTiles &other);
  Imp tiles_;

#ifdef EME__COMPATIBILITY
public:
  /* Compatibility with both SparseLayer and TileGroup */
  SparseTiles *Clone() const { return clone(); }

  /* Compatibility with TileGroup only */
  void RawAdd(int x, int y, BaseProperty *p) { DBG_ASSERT(!get(x, y)); set(x, y, p); }
  void Add(int x, int y, BaseProperty *p) { if (!get(x, y)) set(x, y, p); }
  //BaseProperty *Remove(int x, int y) { BaseProperty *p=get(x, y); clear(x, y); return p; }
  BaseProperty *Remove(int x, int y) { BaseProperty *p=clear_(x, y); return p; }
  BaseProperty *Remove(iterator it) { return Remove(get_i(it), get_j(it)); }

  int Num() const { return count(); }
  int MinX() const { return begini(); }
  int MinY() const { return beginj(); }
  int MaxX() const { return endi()-1; }
  int MaxY() const { return endj()-1; }

  const BaseCreator *GetCreator() const { return creator(); }

  const_iterator Begin() const { return begin(); }
  const_iterator End() const { return end(); }
  iterator Begin() { return begin(); }
  iterator End() { return end(); }
  int GetX(const_iterator it) const { return get_i(it); }
  int GetY(const_iterator it) const { return get_j(it); }
  const BaseProperty *GetTile(const_iterator it) const { return get_tile(it); }
  BaseProperty *GetTile(iterator it) { return get_tile(it); }

  void Sort() {}
  const_iterator Find(int x, int y) const {
    for (const_iterator id=begin(); id!=end(); ++id) {
      if (get_i(id)==x && get_j(id)==y) {
        return id;
      }
    }
    return end();
  }
  iterator Find(int x, int y) {
    for (iterator id=begin(); id!=end(); ++id) {
      if (get_i(id)==x && get_j(id)==y) {
        return id;
      }
    }
    return end();
  }
#endif
};

#endif /* EME__SPARSE_TILES__ */

