/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__SELECTED_TILES__
#define EME__SELECTED_TILES__
/*----------------------------------------------------------------------------
  Keep track of the selected tiles position
  Contains only the tiles position, not their value

- Constructor/destructor
- Adding/removing
- Iterator
- Queries
- Helpers (private) and internal data
----------------------------------------------------------------------------*/

#include "debug.h"

#include "place.h"
#include "config.inc"
#include <set>


#ifdef EME__COMPATIBILITY
#include "lutils.h"
#endif


class SelectedTiles {

public:
  typedef class std::set<Place> set2d;
  typedef set2d::iterator iterator;
  typedef set2d::const_iterator const_iterator;

  /* Constructor/destructor
     --------------------*/
  /* Creates an empty SelectedTiles object */
  SelectedTiles(void);

  ~SelectedTiles(void);

  /* Returns a newly created SelectedTiles object, containing the same
     positions than 'this' */
  SelectedTiles *Clone(void) const;

#ifdef EME__COMPATIBILITY
  SelectedTiles *Intersection(const SelectedTiles *s) const
    { return LayerUtils::Intersection(this, s); }
#endif


  /* Adding/removing
     --------------------*/
  /* Add to 'this' the tile position (i,j) */
  void Add( /* If it does not exist already */
    int i, int j
  )
    { tiles_.insert(MakePlace(i, j)); }
#ifdef EME__COMPATIBILITY
  void RawAdd( /* Must not exist already */
    int i, int j
  )
    { tiles_.insert(MakePlace(i, j)); }
#endif

  /* Add to 'this' the tile positions in 'to_add' */
  void Add(
    const SelectedTiles *to_add
  )
    { tiles_.insert(to_add->Begin(), to_add->End()); }

  /* Replace the content of 'this' */
  void Replace( /* By the position (i, j) */
    int i, int j
  )
    { Clear(); Add(i, j); }
  void Replace( /* By 'to_replace' content */
    const SelectedTiles *to_replace
  )
    { Clear(); Add(to_replace); }

  /* Remove from 'this' the tile position (i,j) */
  void Remove(
    int i, int j
  )
    { tiles_.erase(MakePlace(i, j)); }

  /* Remove from 'this' the tile positions in 'to_remove' */
  void Remove(
    const SelectedTiles *to_remove
  );

  /* Remove all the tile positions in 'this' */
  void Clear(void)
    { tiles_.clear(); }


  /* Iterator
     Usage:
       SelectedTiles sel;
       for(SelectedTiles::const_iterator id=sel.Begin(); id!=sel.End(); ++id)
         {...}
     --------------------*/
  const_iterator Begin() const { return tiles_.begin(); }
  const_iterator End() const { return tiles_.end(); }
  iterator Begin() { return tiles_.begin(); }
  iterator End() { return tiles_.end(); }


  /* Queries
     --------------------*/
  /* Returns the number of tiles position */
  int Num() const { return tiles_.size(); }

  /* Returns true if there is no position in the selection */
  bool Empty() const { return Num()==0; }

  /* Returns true if 'sel' and 'this' contains the same tile positions */
  bool IsSame(const SelectedTiles *sel) const { return tiles_==sel->tiles_; }

  /* Returns the I index of the position "pointed to" by 'id' */
  int GetX(const_iterator id) const { return PlaceI(*id); }

  /* Returns the J index of the position "pointed to" by 'id' */
  int GetY(const_iterator id) const { return PlaceJ(*id); }

  /* Returns true if the position (i, j) is in 'this' */
  bool IsIn(int i, int j) const { return tiles_.find(MakePlace(i, j))!=End(); }


protected:
  SelectedTiles(const SelectedTiles *sel);


private:
  SelectedTiles(const SelectedTiles &);
  SelectedTiles &operator=(const SelectedTiles&);

  set2d tiles_;
};

#endif /* EME__SELECTED_TILES__ */

