/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "mapdlg.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "help.h"
#include "map.h"
#include "globals.h"
#include "selected.h"
#include "gui.h"
#include "commands.h"
#include "lutils.h"

#include "creator.h"

#include <altheme.h>
#include <altheme/internal/theme.h>

#include <allegro.h>
#include <math.h>

class Entry;

#define MIN_SCALE (-32)
#define MID_SCALE (0)
#define MAX_SCALE (+4)


static float odd_array [] = {
  0,   /* ODD_NONE */
  0.5, /* ODD_RIGHT */
 -0.5  /* ODD_LEFT */
};



int MapDialog::MapProc(int msg, DIALOG *d, int c)
{
  static int last_mouse_x = -1;
  static int last_mouse_y = -1;
  MapDialog *mapdlg = CAST_FROM_VOID_PTR(MapDialog*, d->dp);
  int fs = altheme_frame_size;

  switch(msg) {
    case MSG_WANTFOCUS:
    case MSG_LOSTFOCUS:
    return D_WANTFOCUS;

    case MSG_DRAW:
      if(mapdlg && mapdlg->GetMap()) {
        mapdlg->Draw();
      }
      else {
        rectfill(
          screen, d->x+fs, d->y+fs, d->x+d->w-fs-1, d->y+d->h-fs-1, gui_mg_color
        );
      }
      if(d->flags&D_GOTFOCUS) {
        altheme_draw_selected_frame(
          screen, d->x+fs, d->y+fs, d->x+d->w-fs-1, d->y+d->h-fs-1
        );
      }
      altheme_draw_frame(screen, d->x, d->y, d->x+d->w-1, d->y+d->h-1);
    return D_O_K;

    case MSG_LPRESS:
      if(mapdlg && mapdlg->GetMap()) {
        int x = mapdlg->GetTileX(mouse_x, mouse_y);
        int y = mapdlg->GetTileY(mouse_x, mouse_y);
        if(mapdlg->GetMap()->IsIn(x, y)) {
          GUI.Press(x, y, key_shifts);
        }
      }
    return D_O_K;

    case MSG_RPRESS:
      if(mapdlg && mapdlg->GetMap()) {
        int x = mapdlg->GetTileX(mouse_x, mouse_y);
        int y = mapdlg->GetTileY(mouse_x, mouse_y);
        mapdlg->PrintCurrentTileValue(x, y);
        while(mouse_b) {
        }
        mapdlg->Redraw();
      }
    return D_O_K;

    case MSG_LRELEASE:
      if(mapdlg && mapdlg->GetMap()) {
        int x = mapdlg->GetTileX(mouse_x, mouse_y);
        int y = mapdlg->GetTileY(mouse_x, mouse_y);
        if(mapdlg->GetMap()->IsIn(x, y)) {
          GUI.Release(x, y, key_shifts);
        }
      }
    return D_O_K;

    case MSG_CLICK:
      if(mapdlg && mapdlg->GetMap()) {
        int x = mapdlg->GetTileX(mouse_x, mouse_y);
        int y = mapdlg->GetTileY(mouse_x, mouse_y);
        if(mouse_b&1) { /* left click */
          if(mapdlg->GetMap()->IsIn(x, y)) {
            last_mouse_x = x;
            last_mouse_y = y;
            GUI.Click(x, y, key_shifts);
          }
        }
      }
    return D_O_K;

    case MSG_CHAR:
    /* Fall thru */
    case MSG_UCHAR:
      if(mapdlg && mapdlg->GetMap()) {
        if(
          (c>>8) == KEY_PGUP || (c>>8) == KEY_PGDN ||
          (c>>8) == KEY_UP || (c>>8) == KEY_DOWN
        ) {
          if(!(mapdlg->v_sb->flags&D_DISABLED)) {
            mapdlg->v_sb->proc(msg, mapdlg->v_sb, c);
          }
          return D_USED_CHAR | D_REDRAWME;
        }
        if(
          (c>>8) == KEY_HOME || (c>>8) == KEY_END ||
          (c>>8) == KEY_LEFT || (c>>8) == KEY_RIGHT
        ) {
          if(!(mapdlg->h_sb->flags&D_DISABLED)) {
            mapdlg->h_sb->proc(msg, mapdlg->h_sb, c);
          }
          return D_USED_CHAR | D_REDRAWME;
        }
      }
    return D_O_K;

    case MSG_APPLY:
      if(mapdlg && mapdlg->GetMap()) {
        GUI.Apply();
      }
    return D_O_K;

    default:
    return D_O_K;
  }
}


int h_scrollbar_proc(int msg, DIALOG *d, int c)
{
  if(msg == MSG_IDLE) return D_O_K;

  msg = do_help(msg, "Map");

  if(msg == MSG_CHAR || msg == MSG_UCHAR) {
    switch(c>>8) {
      case KEY_HOME:
      return altheme_slider_proc(msg, d, KEY_HOME<<8);
      case KEY_END:
      return altheme_slider_proc(msg, d, KEY_END<<8);
      case KEY_PGUP:
      return D_O_K;
      case KEY_PGDN:
      return D_O_K;
      default:
      return altheme_slider_proc(msg, d, c);
    }
  }
  else {
    return altheme_slider_proc(msg, d, c);
  }
}


int v_scrollbar_proc(int msg, DIALOG *d, int c)
{
  if(msg == MSG_IDLE) return D_O_K;

  msg = do_help(msg, "Map");

  if(msg == MSG_CHAR || msg == MSG_UCHAR) {
    switch(c>>8) {
      case KEY_HOME:
      return D_O_K;
      case KEY_END:
      return D_O_K;
      case KEY_PGUP:
      return altheme_slider_proc(msg, d, KEY_HOME<<8);
      case KEY_PGDN:
      return altheme_slider_proc(msg, d, KEY_END<<8);
      default:
      return altheme_slider_proc(msg, d, c);
    }
  }
  else {
    return altheme_slider_proc(msg, d, c);
  }
}


int h_scrollbar_cb(void *dp3, int /*d2*/)
{
  DIALOG *d = CAST_FROM_VOID_PTR(DIALOG*, dp3); /* Map dialog */
  d->flags |= D_DIRTY;
  return D_O_K;
}


int v_scrollbar_cb(void *dp3, int /*d2*/)
{
  DIALOG *d = CAST_FROM_VOID_PTR(DIALOG*, dp3); /* Map dialog */
  d->flags |= D_DIRTY;
  return D_O_K;
}



MapDialog::MapDialog(int x, int y, int w, int h, DIALOG *d):
  Dialog(x, y, w, h, d)
{
  Init(d);
}


MapDialog::MapDialog(DIALOG *d): Dialog(d)
{
  Init(d);
}


MapDialog::~MapDialog(void)
{
}


void MapDialog::Init(DIALOG *d)
{
  box = &(d[MAP_DIALOG_BOX]);
  box->proc = MapProc;
  box->dp = this;

  h_sb = &(d[MAP_DIALOG_HSLIDER]);
  h_sb->proc = h_scrollbar_proc;
  h_sb->dp2 = CAST_TO_VOID_PTR(h_scrollbar_cb);
  h_sb->dp3 = box;
  h_sb->d1 = 1; /* Map width, cannot be 0 or crash */
  h_sb->d2 = 0; /* Current position */
  h_sb->flags = D_DISABLED;

  v_sb = &(d[MAP_DIALOG_VSLIDER]);
  v_sb->proc = v_scrollbar_proc;
  v_sb->dp2 = CAST_TO_VOID_PTR(v_scrollbar_cb);
  v_sb->dp3 = box;
  v_sb->d1 = 1; /* Map height, cannot be 0 or crash */
  v_sb->d2 = v_sb->d1 - 0; /* Current position */
  v_sb->flags = D_DISABLED;

  SetTileOffset(1, 1, ODD_NONE);

  SetTileShape(TILE_SHAPE_RECT, 1, 1);

  scale = MID_SCALE;
  map_bitmap = NULL;
  map_shape = MAP_SHAPE_RECT;

  DrawGrid(1);
}


void MapDialog::InitMap(void)
{
  if(map_bitmap) {
    destroy_bitmap(map_bitmap);
    map_bitmap=0;
  }
  if(GetMap()) {
    /* Scrollbars */
    UpdateScrollbars();
    h_sb->d2 = 0;
    v_sb->d2 = v_sb->d1 - 0; /* Vertical works upside down */

    /* Bitmap */
    map_bitmap = create_bitmap(box->w, box->h);
    CHECK_POINTER(map_bitmap);
    altheme_fill_frame(map_bitmap, 0, 0, map_bitmap->w, map_bitmap->h, box->bg);
  }
  else {
    h_sb->flags &= ~D_DISABLED;
    v_sb->flags &= ~D_DISABLED;
  }
}


void MapDialog::LazyInit(void)
{
  h_sb->x = box->x;
  h_sb->y = box->y+box->h;
  h_sb->w = box->w;
  h_sb->h = SB_SIZE;

  v_sb->x = box->x+box->w;
  v_sb->y = box->y;
  v_sb->w = SB_SIZE;
  v_sb->h = box->h;

  InitMap();
}


void MapDialog::DrawGrid(int yesno)
{
  draw_grid = yesno;
  Redraw();
}


void MapDialog::SetMapShape(MAP_SHAPE shape)
{
  map_shape = shape;
}


void MapDialog::SetTileShape(TILE_SHAPE shape, int w, int h)
{
  tile_shape = shape;
  tile_w = w;
  tile_h = h;
}


void MapDialog::SetTileOffset(int dx, int dy, ODD odd)
{
  tile_dx = dx;
  tile_dy = dy;
  tile_odd = odd;
}


float MapDialog::GetScale(void) const
{
  if(scale>MID_SCALE) {
    return (float)(1<<(scale-MID_SCALE));
  }
  else if(scale<MID_SCALE) {
    return (float)1.0f / (1<<(MID_SCALE-scale));
  }
  else { /* scale==MID_SCALE */
    return 1.0;
  }
}


int MapDialog::GetTileX(int mousex, int mousey) const
{
  // FIXME: Handle rect with diamond in GetTileX
  int mx = mousex - 2*altheme_frame_size - box->x + GetDX();
  int my = mousey - 2*altheme_frame_size - box->y + GetDY();
  switch(map_shape) {
    case MAP_SHAPE_RECT:
      if (((int)(my / GetScaledTileDy())) & 1) {
        return (int)(
          (mx - odd_array[tile_odd]*GetScaledTileDx()) /
          GetScaledTileDx()
        );
      }
      else {
        return (int)(mx / GetScaledTileDx());
      }

    case MAP_SHAPE_DIAMOND:
      return (int)((mx+2*my)/GetScaledTileDx() - GetMap()->GetHeight()/2);

    default:
      DBG_ASSERT(0);
    return 0;
  }
}


int MapDialog::GetTileY(int mousex, int mousey) const
{
  // FIXME: Handle rect with diamond in GetTileY
  int mx = mousex - 2*altheme_frame_size - box->x + GetDX();
  int my = mousey - 2*altheme_frame_size - box->y + GetDY();
  switch(map_shape) {
    case MAP_SHAPE_RECT:
      return (int)(my / GetScaledTileDy());

    case MAP_SHAPE_DIAMOND:
      return (int)(((-mx+2*my)/GetScaledTileDy()+GetMap()->GetHeight()) / 2);

    default:
      DBG_ASSERT(0);
    return 0;
  }
}


void MapDialog::Rect(int i1, int j1, int i2, int j2)
{
  static int i1_=-1, j1_=-1, i2_=-1, j2_=-1; // FIXME: put it in the class

  switch(map_shape) {
    case MAP_SHAPE_RECT:
      {
        int first_tile_i = h_sb->d2;
        int first_tile_j = v_sb->d1 - v_sb->d2;
        int tw = (int)GetScaledTileWidth();
        int th = (int)GetScaledTileHeight();
        int di = box->x+2*altheme_frame_size - first_tile_i*tw;
        int dj = box->y+2*altheme_frame_size - first_tile_j*th;
        int col = makecol(255, 255, 255);
        if (i2<i1) std::swap(i1, i2);
        if (j2<j1) std::swap(j1, j2);
        xor_mode(TRUE);
        scare_mouse();
        rect(screen, di+i1_*tw, dj+j1_*th, di+(i2_+1)*tw, dj+(j2_+1)*th, col);
        rect(screen, di+i1*tw, dj+j1*th, di+(i2+1)*tw, dj+(j2+1)*th, col);
        unscare_mouse();
        xor_mode(FALSE);
        i1_ = i1;
        j1_ = j1;
        i2_ = i2;
        j2_ = j2;
      }
      break;

    case MAP_SHAPE_DIAMOND: // FIXME: Rect
      break;
  }
}


void MapDialog::Circle(int i1, int j1, int i2, int j2)
{
  static int i_=-1, j_=-1, r_=0; // FIXME: put it in the class

  switch(map_shape) {
    case MAP_SHAPE_RECT:
      {
        int first_tile_i = h_sb->d2;
        int first_tile_j = v_sb->d1 - v_sb->d2;
        int tw = (int)GetScaledTileWidth();
        int th = (int)GetScaledTileHeight();
        int di = box->x+2*altheme_frame_size - first_tile_i*tw;
        int dj = box->y+2*altheme_frame_size - first_tile_j*th;
        int col = makecol(255, 255, 255);
        int r = (int)sqrt((double)((i1-i2)*(i1-i2) + (j1-j2)*(j1-j2)));
        xor_mode(TRUE);
        scare_mouse();
        ellipse(screen, di+i_*tw+tw/2, dj+j_*th+th/2, (r_*tw), (r_*th), col);
        ellipse(screen, di+i1*tw+tw/2, dj+j1*th+th/2, (r*tw), (r*th), col);
        unscare_mouse();
        xor_mode(FALSE);
        i_ = i1;
        j_ = j1;
        r_ = r;
      }
      break;

    case MAP_SHAPE_DIAMOND: // FIXME: Circle
      break;
  }
}



void MapDialog::Unzoom(void)
{
  if(scale!=MID_SCALE) {
    scale=MID_SCALE;
    Redraw();
  }
}


void MapDialog::ZoomIn(void)
{
  if(scale<MAX_SCALE) {
    scale++;
    Redraw();
  }
}


void MapDialog::ZoomOut(void)
{
  if(scale>MIN_SCALE) {
    scale--;
    if(
      ((int)GetScaledTileDx())==0 || ((int)GetScaledTileDy())==0 ||
      ((int)GetScaledTileWidth())==0 || ((int)GetScaledTileHeight())==0
    ) {
      scale++;
    }
    else {
      Redraw();
    }
  }
}


void MapDialog::Redraw(void) const
{
  box->flags |= D_DIRTY;
}


void MapDialog::Draw(void) const
{
  UpdateScrollbars();

  rectfill(map_bitmap, 0, 0, map_bitmap->w, map_bitmap->h, gui_mg_color);
  switch(map_shape) {
    case MAP_SHAPE_RECT:
      DrawRect();
    break;
    case MAP_SHAPE_DIAMOND:
      DrawDiamond();
    break;
  }
  scare_mouse();
  blit(
    map_bitmap, screen, 0, 0,
    box->x+2*altheme_frame_size, box->y+2*altheme_frame_size,
    box->w-4*altheme_frame_size, box->h-4*altheme_frame_size
  );
  unscare_mouse();
}


void MapDialog::PrintCurrentTileValue(int i, int j) const
{
  Map *map = GetMap();

  if(map->IsIn(i, j)) {
    Tiles *layer = map->GetActiveLayer();
    const BaseCreator *creator = layer->creator();
    const BaseProperty *p = layer->get(i, j);
    int fs = altheme_frame_size;
    BITMAP *bmp = create_bitmap(box->w-2*fs, box->h-2*fs);

    altheme_fill_frame(bmp, 0, 0, bmp->w, bmp->h, gui_bg_color);
    int rtm = text_mode(-1);
    textprintf(bmp, font, 0, 0, gui_fg_color, UString("%dx%d"), i, j);
    int nlines = 1;
    if(p) {
      nlines += creator->PrintValue(
        p, bmp, PRINT_VALUE_HINDENT, PRINT_VALUE_VINDENT, gui_fg_color
      );
      creator->UpdateEntry(GUI.GetTileEntry(), p);
    }
    else {
      textprintf(
        bmp, font, PRINT_VALUE_HINDENT, PRINT_VALUE_VINDENT, gui_fg_color,
        Translation("No tile")
      );
      nlines += 1;
    }
    scare_mouse();
    int height = MIN(nlines*PRINT_VALUE_VINDENT, bmp->h);
    blit(bmp, screen, 0, 0, box->x+fs, box->y+fs, bmp->w, height);
    altheme_draw_frame(
      screen, box->x, box->y, box->x+box->w-1, box->y+height+2*fs-1
    );
    unscare_mouse();
    destroy_bitmap(bmp);
    text_mode(rtm);
  }
}


void MapDialog::UpdateScrollbars(void) const
{
  int map_w;
  int map_h;
  int visible_w = (int)((box->w)/GetScaledTileDx());
  int visible_h = (int)((box->h)/GetScaledTileDy());

  switch(map_shape) {
    case MAP_SHAPE_RECT:
      map_w = GetMap()->GetWidth();
      map_h = GetMap()->GetHeight();
    break;

    case MAP_SHAPE_DIAMOND:
      map_w = (GetMap()->GetWidth() + GetMap()->GetHeight()) / 2;
      map_h = (GetMap()->GetWidth() + GetMap()->GetHeight()) / 2;
    break;

    default:
      DBG_ASSERT(0);
    return;
  }

  /* Update the scrollbars lenght */
  h_sb->d1 = map_w - visible_w;
  /* No need to scroll horizontally */
  if(h_sb->d1<=0) {
    h_sb->flags |= D_DISABLED;
    h_sb->d1 = 1; /* Cannot be 0 or crash */
    h_sb->d2 = 0;
  }
  else {
    h_sb->flags &= ~D_DISABLED;
  }

  v_sb->d1 = map_h - visible_h;
  /* No need to scroll vertically */
  if(v_sb->d1<=0) {
    v_sb->flags |= D_DISABLED;
    v_sb->d1 = 1; /* Cannot be 0 or crash */
    v_sb->d2 = v_sb->d1 - 0;
  }
  else {
    v_sb->flags &= ~D_DISABLED;
  }

  /* Make sure the scrollbar position is valid */
  if(h_sb->d2 > h_sb->d1) h_sb->d2 = h_sb->d1;
  if(v_sb->d2 > v_sb->d1) v_sb->d2 = v_sb->d1;

  h_sb->flags |= D_DIRTY;
  v_sb->flags |= D_DIRTY;
}


void (*MapDialog::ChooseDrawDotted(void) const)(BITMAP*,int,int,int,int)
{
  switch(tile_shape) {
    case TILE_SHAPE_RECT:
    return dotted_rect;

    case TILE_SHAPE_DIAMOND:
    return dotted_diamond;

    default:
      DBG_ASSERT(0);
    return 0;
  }
}

void (*MapDialog::ChooseDrawGrid(void) const)(BITMAP*,int,int,int,int)
{
  switch(tile_shape) {
    case TILE_SHAPE_RECT:
    return dotted_grid_rect;

    case TILE_SHAPE_DIAMOND:
    return dotted_grid_diamond;

    default:
      DBG_ASSERT(0);
    return 0;
  }
}

void MapDialog::DrawRect(void) const
{
  Map *map = GetMap();
  DBG_ASSERT(map);

  int actual_tw = (int)GetScaledTileWidth();
  int actual_th = (int)GetScaledTileHeight();
  int actual_dx = (int)GetScaledTileDx();
  int actual_dy = (int)GetScaledTileDy();
  float scale = GetScale();
  int first_tile_i = h_sb->d2;
  int first_tile_j = v_sb->d1 - v_sb->d2;
  int n_tiles_horz = (map_bitmap->w / actual_dx) + 1;
  int n_tiles_vert = (map_bitmap->h / actual_dy) + 1;
  int actual_odd = (int)(odd_array[tile_odd]*actual_dx);

  /* Draw Map */
  LayerUtils::DrawRect(
    map->GetLayers(), &(map->GetViewedLayers()), map_bitmap,
    first_tile_i, first_tile_j, n_tiles_horz, n_tiles_vert,
    actual_dx, actual_dy, odd_array[tile_odd], scale
  );

  /* Draw userdata */
  GUI.DrawUserdata(map->GetUserdata(), map_bitmap, first_tile_i, first_tile_j, scale);

  /* Draw Grid */
  if(IsGridDrawn()) {
    void (*grid_drawer)(BITMAP*,int,int,int,int) = ChooseDrawGrid();;
    int last_i = MIN(first_tile_i+n_tiles_horz, map->GetWidth());
    int last_j = MIN(first_tile_j+n_tiles_vert, map->GetHeight());
    for (int j=first_tile_j, y=0; j < last_j; j++, y+=actual_dy) {
      for (int i=first_tile_i, x=0; i < last_i; i++, x+=actual_dx) {
        int xx = x + (j&1)*actual_odd;
        grid_drawer(map_bitmap, xx, y, xx+actual_tw-1, y+actual_th-1);
      }
    }
  }

  /* Draw Selected tiles */
  void (*draw_dotted)(BITMAP*,int,int,int,int) = ChooseDrawDotted();
  SelectedTiles::iterator id;
  SelectedTiles *current = map->GetSelectedTiles();
  for(id=current->Begin(); id!=current->End(); ++id) {
    int i=current->GetX(id);
    int j=current->GetY(id);
    draw_dotted(
      map_bitmap,
      (i-first_tile_i)*actual_dx + (j&1)*actual_odd,
      (j-first_tile_j)*actual_dy,
      (i-first_tile_i)*actual_dx + (j&1)*actual_odd + actual_tw-1,
      (j-first_tile_j)*actual_dy + actual_th-1
    );
  }

  /* Draw active layer borders */
  Tiles *layer = GetMap()->GetActiveLayer();
  if (layer) {
    int x1 = (layer->begini()-first_tile_i)*actual_tw;
    int y1 = (layer->beginj()-first_tile_j)*actual_th;
    int x2 = (layer->endi()-first_tile_i)*actual_tw;
    int y2 = (layer->endj()-first_tile_j)*actual_th;
    rect(map_bitmap, x1, y1, x2, y2, gui_fg_color);
  }
}


void MapDialog::DrawDiamond(void) const
{
  Map *map = GetMap();
  DBG_ASSERT(map);

  int actual_tw = (int)GetScaledTileWidth();
  int actual_th = (int)GetScaledTileHeight();
  int actual_dx = (int)GetScaledTileDx();
  int actual_dy = (int)GetScaledTileDy();
  float scale = GetScale();
  int first_tile_i = h_sb->d2;
  int first_tile_j = v_sb->d1 - v_sb->d2;
  int n_tiles_horz = (map_bitmap->w / actual_dx) + 1;
  int n_tiles_vert = (map_bitmap->h / actual_dy) + 1;

  /* Draw Map */
  LayerUtils::DrawDiamond(
    map->GetLayers(), &(map->GetViewedLayers()), map_bitmap,
    first_tile_i, first_tile_j, n_tiles_horz, n_tiles_vert,
    actual_dx, actual_dy, odd_array[tile_odd], scale
  );

  /* Draw userdata */
  GUI.DrawUserdata(map->GetUserdata(), map_bitmap, first_tile_i, first_tile_j, scale);

  /* Draw Grid */
  if(IsGridDrawn()) {
    void (*grid_drawer)(BITMAP*,int,int,int,int) = ChooseDrawGrid();;
    int dx = actual_dx * first_tile_i + actual_dx/2;
    int dy = actual_dy * first_tile_j + actual_dy/2;
    int width = map->GetWidth();
    int height = map->GetHeight();
    int bmp_w = map_bitmap->w;
    int bmp_h = map_bitmap->h;
    for(int j=0; j<height; j++) {
      for(int i=0; i<width; i++) {
        int x = -dx + actual_dx * (height+i-j) / 2;
        int y = -dy + actual_dy * (i+j+1) / 2;
        if(x+actual_tw>=0 && x<bmp_w && y+actual_th>=0 && y<bmp_h) {
          grid_drawer(map_bitmap, x, y, x+actual_tw-1, y+actual_th-1);
        }
      }
    }
  }

  /* Draw Selected tiles */
  void (*draw_dotted)(BITMAP*,int,int,int,int) = ChooseDrawDotted();
  int height = map->GetHeight();
  SelectedTiles::iterator id;
  SelectedTiles *current = map->GetSelectedTiles();
  for(id=current->Begin(); id!=current->End(); ++id) {
    int tx = current->GetX(id);
    int ty = current->GetY(id);
    int dx = actual_dx * first_tile_i + actual_dx/2;
    int dy = actual_dy * first_tile_j + actual_dy/2;
    int x = -dx + actual_dx * (height+tx-ty) / 2;
    int y = -dy + actual_dy * (tx+ty+1) / 2;
    if(x+actual_tw>=0 && y+actual_th>=0 && x<map_bitmap->w && y<map_bitmap->h) {
      draw_dotted(map_bitmap, x, y, x+actual_tw-1, y+actual_th-1);
    }
  }

  /* Draw active layer borders */
  Tiles *layer = GetMap()->GetActiveLayer();
  if (layer) {
    // TODO
  }
}


int MapDialog::GetDX(void) const
{
  return (int)(h_sb->d2 * GetScaledTileDx());
}


int MapDialog::GetDY(void) const
{
  return (int)((v_sb->d1 - v_sb->d2) * GetScaledTileDy());
}


const char *MapDialog::HelpText() const
{
  static const Translation i18nHelp("L-click: use tool  R-click: pick value");
  return i18nHelp.string();
}

