/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "map.h"

#include "debug.h"
#include "utils.h"

#include "entry.h"
#include "selected.h"
#include "ccomposi.h"
#include "prop.h"
#include "creator.h"
#include "lutils.h"
#include "gui.h"
#include "globals.h"

#include <allegro.h>


Map::Map(int w, int h, int num_layers):
  width(w), height(h), layers_(),
  active_layer(-1), viewed_layers_(num_layers),
  propcreator(0), active_property(-1),
  userdata_(0), userdata_destroyer_(0),
  selected_tiles_(), commands_manager_()
{
}


Map::Map(int w, int h):
  width(w), height(h), layers_(),
  active_layer(-1), viewed_layers_(0),
  propcreator(0), active_property(-1),
  userdata_(0), userdata_destroyer_(0),
  selected_tiles_(), commands_manager_()
{
}


Map::~Map(void)
{
  SetUserdata(0);
  delete propcreator;
  for (int n=0; n<layers_.depth(); ++n) {
    Tiles *layer = layers_.get(n);
    layers_.clear(n);
    delete layer;
  }
}


int Map::SetLayer(int n, Tiles *layer)
{
  DBG_ASSERT(!layers_.get(n));
  layers_.set(n, layer);
  return 1;
}


Tiles *Map::RemoveLayer(int l)
{
  active_layer = -1;
  Tiles *layer = layers_.get(l);
  layers_.remove(l);
  viewed_layers_.remove(l);
  SetActiveLayer(l>0 ? l-1 : 0);
  return layer;
}


void Map::InsertLayer(int l, Tiles *layer)
{
  active_layer = -1;
  layers_.insert(l, layer);
  viewed_layers_.insert(l);
  viewed_layers_.set(l);
  SetActiveLayer(l);
}


void Map::SetProperties(Composite::Creator *propcreator)
{
  this->propcreator = propcreator;
}


void Map::SetActiveProperty(int prop, Entry *e)
{
  if (!propcreator || propcreator->Count()==0) return;

  /* Change the active property */
  active_property = prop;
  propcreator->Get(active_property)->UpdateEntry(e);
  e->Clear();
  e->Redraw();
}


void Map::UpdateActiveProperty(Entry *e)
{
  if (!propcreator || propcreator->Count()==0) return;

  if (active_property>=0) {
    BaseCreator *active_creator = propcreator->Get(active_property);
    active_creator->UpdateReferenceProperty(e);
  }
}


int Map::FindLayerIndex(const char *name) const
{
  for (int l=0; l<GetNumLayers(); ++l) {
    if (ustrcmp(GetLayerName(l), name)==0) {
      return l;
    }
  }
  return GetNumLayers();
}


const char *Map::GetLayerName(int l) const
{
  return GetLayer(l) ? GetLayer(l)->creator()->GetName()->string() : 0;
}


void Map::SetActiveLayer(int n, Entry *e)
{
  if (GetLayer(n)) {
    active_layer = n;
    GetActiveLayer()->creator()->UpdateEntry(e);
    e->Clear();
    e->Redraw();
  }
}

void Map::SetActiveLayer(int n)
{
  SetActiveLayer(n, GUI.GetTileEntry());
}

void Map::SetViewedLayers(const ViewedLayers *v)
{
  int count = v->count()<viewed_layers_.count() ? v->count() : viewed_layers_.count();
  for (int l=0; l<count; ++l) {
    if (v->at(l)) {
      viewed_layers_.set(l);
    }
    else {
      viewed_layers_.reset(l);
    }
  }
}

void Map::UpdateActiveLayer(Entry *e)
{
  if (GetLayer(active_layer)) {
    GetLayer(active_layer)->creator()->UpdateReferenceProperty(e);
  }
}


const Tiles *Map::GetActiveLayer(void) const
{
  return GetLayer(GetActiveLayerIndex());
}


Tiles *Map::GetActiveLayer(void)
{
  return GetLayer(GetActiveLayerIndex());
}


int Map::GetActiveLayerIndex(void) const
{
  return active_layer;
}


const char *Map::GetActiveLayerName(void) const
{
  return GetLayerName(GetActiveLayerIndex());
}


void Map::IncrActiveLayer(Entry *e)
{
  for (int new_active=active_layer+1; new_active<GetNumLayers(); ++new_active) {
    if (GetLayer(new_active)) {
      SetActiveLayer(new_active, e);
      break;
    }
  }
}


void Map::DecrActiveLayer(Entry *e)
{
  for (int new_active=active_layer-1; new_active>=0; --new_active) {
    if (GetLayer(new_active)) {
      SetActiveLayer(new_active, e);
      break;
    }
  }
}


const BaseProperty *Map::GetLayerReference(int l) const
{
  return GetLayer(l) ?
    GetLayer(l)->creator()->GetReference() :
    0;
}


int Map::GetNumProperties(void) const
{
  return propcreator ? propcreator->Count() : 0;
}


const char *Map::GetPropertyName(int id) const
{
  /* Should not call this function if GetNumProps returns 0 */
  DBG_ASSERT(propcreator && propcreator->Count()!=0);
  return propcreator->Get(id)->GetName()->string();
}


const BaseProperty *Map::GetProperty(int n) const
{
  return propcreator->Get(n)->GetReference();
}

