/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#ifndef EME__CREATOR__
#define EME__CREATOR__
/*----------------------------------------------------------------------------
  Property creator, abstract class

  Creators are used to encapsulate many operations on Properties. For example,
memory management: all Property creation must go trough a Creator, so it'll
be possible in the future that the memory management would be more powerful.

- Constructor/destructor
- Properties
- Entry handlers
- Internal data
- Helper defines
----------------------------------------------------------------------------*/

#include "debug.h"

#include "utils.h"

#include "prop.h"
struct BITMAP;
class BaseProperty;
class Entry;

#include "sttcstr.h"

#ifdef DEBUG
#  define DBG_CREATOR_TYPE char *dbg_type;
#  define DBG_SET_CREATOR_TYPE(name) dbg_type=name
#else
#  define DBG_CREATOR_TYPE
#  define DBG_SET_CREATOR_TYPE(name)
#endif


#define PRINT_VALUE_HINDENT (2*text_length(font, SpaceString()))
#define PRINT_VALUE_VINDENT ((3*text_height(font))/2)


class BaseCreator {

public:
  DBG_CREATOR_TYPE

  /* Constructor/destructor
     ---------------------------------*/
  BaseCreator(StaticString *name, BaseProperty *p);
  virtual ~BaseCreator(void);
  virtual BaseCreator *Clone(void) const = 0;

  /* Returns the creator name */
  StaticString *GetName(void) const;


  /* Properties
     ---------------------------------*/
  /* Returns the prototype property */
  const BaseProperty *GetReference(void) const
    {return reference_;}

  /* Returns a newly created property */
  BaseProperty *Create(void) const;

  /* Returns true (1) if the Property 'p' has the "same type" than the
   * prototype Property. "Same type" can means something else than "same
   * typeid", see for example DatafileCreator: two PropertyDatafile don't have
   * the same type if their datafile name is different. "Same type" means that
   * you can paste one tile in place of another */
  virtual int IsSameType(const BaseProperty *p) const;

  /* Draws the Property 'p' on the bitmap 'bmp' */
  virtual void Draw(
    const BaseProperty *p,  /* Property to display */
    BITMAP *bmp,            /* Where to draw */
    int x, int y,           /* Position (in pixels) on the bitmap */
    int w, int h,           /* Allowed size (in pixels) */
    float scale,            /* Scaling factor */
    int l,                  /* Layer index */
    int i, int j            /* Tile indices */
  ) const;

  /* Prints the Property 'p' value on the bitmap 'bmp' */
  virtual int PrintValue(
    const BaseProperty *p,  /* Property to display */
    BITMAP *bmp,            /* Where to write */
    int x, int y,           /* Position (in pixels) on th bitmap */
    int color               /* Text color */
  ) const;


  /* Entry handlers
     ---------------------------------*/
  /* Sets the Entry 'e' to the currrent value and the DIALOG procedure */
  void UpdateEntry(Entry *e) const;

  /* Sets the Entry 'e' to the Property 'p' value and the DIALOG procedure */
  virtual void UpdateEntry(Entry *e, const BaseProperty *p) const = 0;

  /* Sets the Property 'p' to the value taken from the Entry 'e' */
  virtual void UpdateProperty(BaseProperty *p, const Entry *e) const = 0;

  /* Sets the current property to the value taken from 'e' */
  void UpdateReferenceProperty(const Entry *e) const;

  /* Sets the current property to the value taken from 'p'
   * Usage: creator->UpdateReferenceProperty(Type::Property(data)); */
  void UpdateReferenceProperty(const BaseProperty &p) const;


protected:
  BaseCreator(const BaseCreator *other);

  BaseProperty *GetVariableReference(void)
    {return reference_;}


private:
  BaseCreator(const BaseCreator&);
  BaseCreator &operator=(const BaseCreator&);
  /* Derived classes should define ReferenceProperty(void) to handle the down
   * cast. I.e.
   * Property *ReferenceProperty(void) const
   *   {return dynamic_cast<Property*>(reference_);}
   * This function should be private because it returns a non-const pointer */

  BaseProperty *reference_; /* Prototype property */
  StaticString *name_;    /* Creator's name */
};

#endif /* EME__CREATOR__ */

