/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "cmdmap.h"

#include "debug.h"
#include "utils.h"

#include "command.h"

#include "globals.h"
#include "selected.h"
#include "lutils.h"
#include "tstack.h"
#include "stiles.h"
#include "lviewed.h"
#include "map.h"
#include "cmdlayer.h"

#include "prop.h"
#include "creator.h"

#include <stdlib.h>


/* Inserting a column
   ---------------------------------*/
SelectedTiles *CommandInsertColumn::CreateAfter(const SelectedTiles *prev, int ati, int atj, int count)
{
  SelectedTiles *sel = new SelectedTiles();
  for (SelectedTiles::const_iterator s=prev->Begin(); s!=prev->End(); ++s) {
    int i = prev->GetX(s);
    int j = prev->GetY(s);
    if (i >= ati) i += count;
    sel->Add(i, j);
  }
  return sel;
}


CommandInsertColumn::CommandInsertColumn(
  Map *map, int l, int i, int j, int count
):
  Command(), layer_(map->GetLayer(l)),
  i_(i), j_(j), count_(count)
{
}


CommandInsertColumn::~CommandInsertColumn(void)
{
}


void CommandInsertColumn::UnExecute(void)
{
  layer_->remove_col(i_, count_);
}


void CommandInsertColumn::Execute(void)
{
  layer_->insert_col(i_, count_);
}


int CommandInsertColumn::IsEmpty(void) const
{
  return count_<=0;
}



/* Inserting a row
   ---------------------------------*/
SelectedTiles *CommandInsertRow::CreateAfter(const SelectedTiles *prev, int ati, int atj, int count)
{
  SelectedTiles *sel = new SelectedTiles();
  for (SelectedTiles::const_iterator s=prev->Begin(); s!=prev->End(); ++s) {
    int i = prev->GetX(s);
    int j = prev->GetY(s);
    if (j >= atj) j += count;
    sel->Add(i, j);
  }
  return sel;
}


CommandInsertRow::CommandInsertRow(
  Map *map, int l, int i, int j, int count
):
  Command(), layer_(map->GetLayer(l)),
  i_(i), j_(j), count_(count)
{
}


CommandInsertRow::~CommandInsertRow(void)
{
}


void CommandInsertRow::UnExecute(void)
{
  layer_->remove_row(j_, count_);
}


void CommandInsertRow::Execute(void)
{
  layer_->insert_row(j_, count_);
}


int CommandInsertRow::IsEmpty(void) const
{
  return count_<=0;
}



/* Deleting a column
   ---------------------------------*/
SelectedTiles *CommandDeleteColumn::CreateAfter(const SelectedTiles *prev, int ati, int atj, int count)
{
  SelectedTiles *sel = new SelectedTiles();
  for (SelectedTiles::const_iterator s=prev->Begin(); s!=prev->End(); ++s) {
    int i = prev->GetX(s);
    int j = prev->GetY(s);
    if (i<ati) {
      sel->Add(i, j);
    }
    else if (i>=ati+count) {
      sel->Add(i-count, j);
    }
  }
  return sel;
}


void CommandDeleteColumn::Init()
{
  SelectedTiles rect;
  const int first_j = layer_->beginj();
  const int last_j = layer_->endj();
  for (int j=first_j; j<last_j; ++j) {
    for (int i=i_; i<i_+count_; ++i) {
      rect.Add(i, j);
    }
  }
  saved_tiles_ = LayerUtils::Copy(layer_, &rect);
}


CommandDeleteColumn::CommandDeleteColumn(
  Map *map, int l, int i, int j, int count
):
  Command(), layer_(map->GetLayer(l)),
  saved_tiles_(0), i_(i), j_(layer_->beginj()),
  count_(i+count<layer_->endi() ? count : layer_->endi()-i)
{
  Init();
}


CommandDeleteColumn::~CommandDeleteColumn(void)
{
  delete saved_tiles_;
}


void CommandDeleteColumn::UnExecute(void)
{
  layer_->insert_col(i_, count_);
  LayerUtils::Paste(layer_, i_, j_, saved_tiles_);
}


void CommandDeleteColumn::Execute(void)
{
  layer_->remove_col(i_, count_);
}


int CommandDeleteColumn::IsEmpty(void) const
{
  return count_<=0;
}



/* Deleting a row
   ---------------------------------*/
SelectedTiles *CommandDeleteRow::CreateAfter(const SelectedTiles *prev, int ati, int atj, int count)
{
  SelectedTiles *sel = new SelectedTiles();
  for (SelectedTiles::const_iterator s=prev->Begin(); s!=prev->End(); ++s) {
    int i = prev->GetX(s);
    int j = prev->GetY(s);
    if (j<atj) {
      sel->Add(i, j);
    }
    else if (j>=atj+count) {
      sel->Add(i, j-count);
    }
  }
  return sel;
}


void CommandDeleteRow::Init()
{
  SelectedTiles rect;
  const int first_i = layer_->begini();
  const int last_i = layer_->endi();
  for (int j=j_; j<j_+count_; ++j) {
    for (int i=first_i; i<last_i; ++i) {
      rect.Add(i, j);
    }
  }
  saved_tiles_ = LayerUtils::Copy(layer_, &rect);
}


CommandDeleteRow::CommandDeleteRow(
  Map *map, int l, int i, int j, int count
):
  Command(), layer_(map->GetLayer(l)),
  saved_tiles_(0), i_(layer_->begini()), j_(j),
  count_(j+count<layer_->endj() ? count : layer_->endj()-j)
{
  Init();
}


CommandDeleteRow::~CommandDeleteRow(void)
{
  delete saved_tiles_;
}


void CommandDeleteRow::UnExecute(void)
{
  layer_->insert_row(j_, count_);
  LayerUtils::Paste(layer_, i_, j_, saved_tiles_);
}


void CommandDeleteRow::Execute(void)
{
  layer_->remove_row(j_, count_);
}


int CommandDeleteRow::IsEmpty(void) const
{
  return count_<=0;
}



CommandMoveLayer::CommandMoveLayer(
  Map *map, int l, int di, int dj
):
  layer_(map->GetLayer(l)), di_(di), dj_(dj)
{
}


CommandMoveLayer::~CommandMoveLayer()
{
}


void CommandMoveLayer::Execute()
{
  layer_->move(di_, dj_);
}


void CommandMoveLayer::UnExecute()
{
  layer_->move(-di_, -dj_);
}


int CommandMoveLayer::IsEmpty() const
{
  return di_==0 && dj_==0;
}



CommandResize::CommandResize(
  Map *map, int l, int i, int j, int w, int h
)
{
  Tiles *new_layer = map->GetLayer(l)->clone();
  new_layer->resize(i, j, w, h);
  imp_ = new CommandReplaceLayer(map, new_layer, l);
}


CommandResize::~CommandResize()
{
  delete imp_;
}


void CommandResize::Execute()
{
  imp_->Execute();
}


void CommandResize::UnExecute()
{
  imp_->UnExecute();
}


int CommandResize::IsEmpty() const
{
  return imp_->IsEmpty();
}

