/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "cmdlayer.h"

#include "tstack.h"
#include "tiles.h"
#include "globals.h"
#include "gui.h"
#include "map.h"
#include "cmdgroup.h"
#include "lviewed.h"
#include "stiles.h"
#include "lutils.h"


CommandInsertLayer::CommandInsertLayer(Map *map, Tiles *layer, int l):
  map_(map), layer_(layer), l_(l)
{
}


CommandInsertLayer::~CommandInsertLayer()
{
  if (layer_) delete layer_;
}


void CommandInsertLayer::UnExecute()
{
  layer_ = map_->RemoveLayer(l_);
}


void CommandInsertLayer::Execute()
{
  map_->InsertLayer(l_, layer_);
  layer_ = 0;
}


int CommandInsertLayer::IsEmpty() const
{
  return 0;
}



CommandRemoveLayer::CommandRemoveLayer(Map *map, int l):
  map_(map), layer_(0), l_(l)
{
}


CommandRemoveLayer::~CommandRemoveLayer()
{
  if (layer_) delete layer_;
}


void CommandRemoveLayer::UnExecute()
{
  map_->InsertLayer(l_, layer_);
  layer_ = 0;
}


void CommandRemoveLayer::Execute()
{
  layer_ = map_->RemoveLayer(l_);
}


int CommandRemoveLayer::IsEmpty() const
{
  return 0;
}



CommandSwapLayers::CommandSwapLayers(Map *map, int l1, int l2):
  map_(map), l1_(l1), l2_(l2), group_(new CommandGroup())
{
  Tiles *layer1 = map->GetLayer(l1_)->clone();
  Tiles *layer2 = map->GetLayer(l2_)->clone();
  group_->Add(new CommandRemoveLayer(map, l1));
  group_->Add(new CommandInsertLayer(map, layer2, l1));
  group_->Add(new CommandRemoveLayer(map, l2));
  group_->Add(new CommandInsertLayer(map, layer1, l2));
}


CommandSwapLayers::~CommandSwapLayers()
{
  delete group_;
}


void CommandSwapLayers::UnExecute()
{
  group_->UnExecute();
}


void CommandSwapLayers::Execute()
{
  group_->Execute();
}


int CommandSwapLayers::IsEmpty() const
{
  return group_->IsEmpty();
}



CommandMergeLayers::CommandMergeLayers(
  Map *map, /* Map to work on */
  const ViewedLayers &layers /* Layers that should be merged */
):
  group_(new CommandGroup())
{
  ViewedLayers to_merge(layers); /* Layers that will be merged */
  /* If a layer is unmergeable (i.e. not the same type) it won't be merged */

  int last_layer = map->GetNumLayers();
  /* Find the first selected layer */
  int first_layer = last_layer;
  for (int l=0; l!=last_layer; ++l) {
    if (layers.at(l)) {
      first_layer = l;
      break;
    }
  }
  /* If at least one layer was selected */
  if (first_layer!=last_layer) {
    Tiles *new_layer = map->GetLayer(first_layer)->clone();
    SelectedTiles *all = LayerUtils::SelectAll(new_layer);
    for (int l=first_layer+1; l!=last_layer; ++l) {
      /* if layer is selected copy it on the new layer */
      if (layers.at(l)) {
        Tiles *layer = map->GetLayer(l);
        if (new_layer->creator()->IsSameType(layer->creator()->GetReference())) {
          /* If mergeable, copy the layer */
          SparseTiles *copy = LayerUtils::Copy(layer, all);
          LayerUtils::Paste(new_layer, copy->begini(), copy->beginj(), copy);
          delete copy;
        }
        else {
          /* If not mergeable, don't delete the layer */
          to_merge.reset(l);
        }
      }
    }
    /* Fill the command */
    for (int l=last_layer-1; l!=first_layer; --l) {
      /* If layer was merged */
      if (to_merge.at(l)) {
        group_->Add(new CommandRemoveLayer(map, l));
      }
    }
    group_->Add(new CommandRemoveLayer(map, first_layer));
    group_->Add(new CommandInsertLayer(map, new_layer, first_layer));
  }
}


CommandMergeLayers::~CommandMergeLayers()
{
  delete group_;
}


void CommandMergeLayers::UnExecute()
{
  group_->UnExecute();
}


void CommandMergeLayers::Execute()
{
  group_->Execute();
}


int CommandMergeLayers::IsEmpty() const
{
  return group_->IsEmpty();
}



CommandReplaceLayer::CommandReplaceLayer(
  Map *map,
  Tiles *layer,
  int l
):
  group_(new CommandGroup())
{
  group_->Add(new CommandRemoveLayer(map, l));
  group_->Add(new CommandInsertLayer(map, layer, l));
}


CommandReplaceLayer::~CommandReplaceLayer()
{
  delete group_;
}


void CommandReplaceLayer::UnExecute()
{
  group_->UnExecute();
}


void CommandReplaceLayer::Execute()
{
  group_->Execute();
}


int CommandReplaceLayer::IsEmpty() const
{
  return group_->IsEmpty();
}

