/* eme - a framework for a game map editor
 *
 * Copyright (C) 2002 Annie Testes
 *
 * This code is placed under the GNU General Public License.
 * Please refer to the accompanying file 'copying.txt' for details.
 */
#include "clist.h"

#include "debug.h"
#include "utils.h"
#include "ustring.h"
#include "translat.h"

#include "cstring.h" /* For drawing */

#include <typeinfo>

#include "entry.h"

#include <altheme.h>

#include <allegro.h>



/* Misc
   ---------------------------------*/
int List::display_list_callback(
  BITMAP *dest, int index, int x, int y, int /*w*/, DIALOG *d
)
{
  const Creator *creator = CAST_FROM_VOID_PTR(const Creator*, d->dp3);

  if (index<0) {
    return creator->NumItems();
  }
  else {
    if(dest) {
      int rtm = text_mode(-1);
      textout(dest, font, creator->Item(index), x, y, gui_fg_color);
      text_mode(rtm);
    }
    return text_height(font)+2;
  }
}


/* Property
   ---------------------------------*/
List::Property::Property(char const*const* items, int v):
  BaseProperty(), value_(v), items_(items)
{
  DBG_SET_PROP_TYPE("List");
}


List::Property::~Property(void)
{
}


void List::Property::CopyFrom(const BaseProperty *p)
{
  const Property *pl = ::Cast<List>(p);
  value_ = pl->value_;
}


int List::Property::IsEqualTo(const BaseProperty *p) const
{
  const Property *pl = ::Cast<List>(p);
  return pl->value_ == value_;
}


List::Property *List::Property::Clone(void) const
{
  return new Property(items_, Get());
}


/* Creator
   ---------------------------------*/
List::Creator::Creator(
  StaticString *name, const char *const*strings, int default_value
): BaseCreator(name, new Property(0, default_value))
{
  DBG_SET_CREATOR_TYPE("List");
  Init(strings);
}


void List::Creator::Init(const char *const* strings)
{
  /* Compute the number of items */
  for(num_items_=0; strings[num_items_]; num_items_++) {
  }
  /* Copy the items */
  Malloc(items_, num_items_+1);
  for (int i=0; i<num_items_; i++) {
    items_[i] = ustrdup(strings[i]);
  }
  items_[num_items_] = 0;
  ReferenceProperty()->SetItems(items_);
}


List::Creator *List::Creator::Clone(void) const
{
  return new Creator(this);
}


List::Creator::~Creator(void)
{
  for (int i=0; i<num_items_; i++) {
    free(items_[i]);
  }
  Free(items_);
}


int List::Creator::IsSameType(const BaseProperty *p) const
{
  /* Note that typeid(*current) can be different from List::Property,
     since it is possible to derive from List::Creator
   */
  if(typeid(*GetReference())==typeid(*p)) {
    //const Property *pl = ::Cast<List>(p);
    //return ReferenceProperty()->GetItems() == pl->GetItems();
    // Must check ALL items
    return 1;
  }
  else {
    return 0;
  }
}


void List::Creator::Draw(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int w, int h,
  float scale, int l, int i, int j
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pl = ::Cast<List>(p);
  String::Draw(items_[pl->Get()], bmp, x, y, w, h);
}


int List::Creator::PrintValue(
  const BaseProperty *p, BITMAP *bmp, int x, int y, int color
) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pl = ::Cast<List>(p);
  textprintf(
    bmp, font, x, y, color, Translation("%s: List %s(%d)"),
    GetName()->string(), items_[pl->Get()], pl->Get()
  );
  return 1;
}


void List::Creator::UpdateEntry(Entry *e, const BaseProperty *p) const
{
  DBG_ASSERT(IsSameType(p));
  const Property *pl = ::Cast<List>(p);
  e->SetState(
    GetName()->string(),
    altheme_elist_proc, pl->Get(), 0/*index*/,
    CAST_TO_VOID_PTR(display_list_callback), NULL,
    CAST_TO_VOID_PTR(this), 0
  );
}


void List::Creator::UpdateProperty(BaseProperty *p, const Entry *e) const
{
  DBG_ASSERT(IsSameType(p));
  Property *pl = ::Cast<List>(p);
  pl->Set(e->GetD1State());
}

